"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const cdk = require("../../../core");
const cx_api_1 = require("../../../cx-api");
const cluster_utils_1 = require("./private/cluster-utils");
const task_utils_1 = require("../private/task-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_tasks_EmrCreateClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EmrCreateCluster);
            }
            throw error;
        }
        this.visibleToAllUsers = this.props.visibleToAllUsers ?? true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        (0, task_utils_1.validatePatternSupported)(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = this.props.serviceRole ?? this.createServiceRole();
        this._clusterRole = this.props.clusterRole ?? this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
        if (this.props.releaseLabel !== undefined && !cdk.Token.isUnresolved(this.props.releaseLabel)) {
            this.validateReleaseLabel(this.props.releaseLabel);
        }
        if (this.props.stepConcurrencyLevel !== undefined && !cdk.Token.isUnresolved(this.props.stepConcurrencyLevel)) {
            if (this.props.stepConcurrencyLevel < 1 || this.props.stepConcurrencyLevel > 256) {
                throw new Error(`Step concurrency level must be in range [1, 256], but got ${this.props.stepConcurrencyLevel}.`);
            }
            if (this.props.releaseLabel && this.props.stepConcurrencyLevel !== 1) {
                const [major, minor] = this.props.releaseLabel.slice(4).split('.');
                if (Number(major) < 5 || (Number(major) === 5 && Number(minor) < 28)) {
                    throw new Error(`Step concurrency is only supported in EMR release version 5.28.0 and above but got ${this.props.releaseLabel}.`);
                }
            }
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: (0, task_utils_1.integrationResourceArn)('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: (0, cluster_utils_1.InstancesConfigPropertyToJson)(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation(this._autoScalingRole?.roleName),
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: this.props.ebsRootVolumeSize?.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? (0, cluster_utils_1.KerberosAttributesPropertyToJson)(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation(this.props.scaleDownBehavior?.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                StepConcurrencyLevel: cdk.numberToCloudFormation(this.props.stepConcurrencyLevel),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        if (cdk.FeatureFlags.of(this).isEnabled(cx_api_1.ENABLE_EMR_SERVICE_POLICY_V2)) {
            return new iam.Role(this, 'ServiceRole', {
                assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'aws:RequestTag/for-use-with-amazon-emr-managed-policies': 'true' },
                    },
                }),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEMRServicePolicy_v2')],
            });
        }
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        role.assumeRolePolicy?.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
    /**
     * Validates the release label string is in proper format.
     * Release labels are in the form `emr-x.x.x`. For example, `emr-5.33.0`.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-release-components.html
     */
    validateReleaseLabel(releaseLabel) {
        const prefix = releaseLabel.slice(0, 4);
        const versions = releaseLabel.slice(4).split('.');
        if (prefix !== 'emr-' || versions.length !== 3 || versions.some((e) => isNotANumber(e))) {
            throw new Error(`The release label must be in the format 'emr-x.x.x' but got ${releaseLabel}`);
        }
        return releaseLabel;
        function isNotANumber(value) {
            return value === '' || isNaN(Number(value));
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
EmrCreateCluster[_a] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster", version: "2.74.0" };
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
exports.EmrCreateCluster = EmrCreateCluster;
(function (EmrCreateCluster) {
    /**
     * The Cluster ScaleDownBehavior specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity
     * occurs or an instance group is resized.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html#EMR-RunJobFlow-request-ScaleDownBehavior
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR adds nodes to a deny list and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions
     *
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Spot Allocation Strategies
     *
     * Specifies the strategy to use in launching Spot Instance fleets. For example, "capacity-optimized" launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
     *
     */
    let SpotAllocationStrategy;
    (function (SpotAllocationStrategy) {
        /**
         * Capacity-optimized, which launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.
         */
        SpotAllocationStrategy["CAPACITY_OPTIMIZED"] = "capacity-optimized";
    })(SpotAllocationStrategy = EmrCreateCluster.SpotAllocationStrategy || (EmrCreateCluster.SpotAllocationStrategy = {}));
    /**
     * CloudWatch Alarm Comparison Operators
     *
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market
     *
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
exports.EmrCreateCluster = EmrCreateCluster;
//# sourceMappingURL=data:application/json;base64,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