"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualClusterInput = exports.ReleaseLabel = exports.Classification = exports.EmrContainersStartJobRun = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("../../../aws-iam");
const lambda = require("../../../aws-lambda");
const logs = require("../../../aws-logs");
const s3 = require("../../../aws-s3");
const sfn = require("../../../aws-stepfunctions");
const cdk = require("../../../core");
const cr = require("../../../custom-resources");
const awscli = require("../../../lambda-layer-awscli");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts a job run.
 *
 * A job is a unit of work that you submit to Amazon EMR on EKS for execution.
 * The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query.
 * A job run is an execution of the job on the virtual cluster.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
 */
class EmrContainersStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Render the EMR Containers ConfigurationProperty as JSON
         */
        this.applicationConfigPropertyToJson = (property) => {
            return {
                Classification: cdk.stringToCloudFormation(property.classification.classificationStatement),
                Properties: property.properties ? cdk.objectToCloudFormation(property.properties) : undefined,
                Configurations: property.nestedConfig ? cdk.listMapper(this.applicationConfigPropertyToJson)(property.nestedConfig) : undefined,
            };
        };
        this.assignLogGroup = () => {
            if (this.props.monitoring?.logGroup) {
                return (this.props.monitoring?.logGroup);
            }
            else {
                return (this.props.monitoring?.logging ? new logs.LogGroup(this, 'Monitoring Log Group') : undefined);
            }
        };
        this.assignLogBucket = () => {
            if (this.props.monitoring?.logBucket) {
                return (this.props.monitoring?.logBucket);
            }
            else {
                return (this.props.monitoring?.logging ? new s3.Bucket(this, 'Monitoring Bucket') : undefined);
            }
        };
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_tasks_EmrContainersStartJobRunProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EmrContainersStartJobRun);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.RUN_JOB;
        (0, task_utils_1.validatePatternSupported)(this.integrationPattern, EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.props.applicationConfig) {
            this.validateAppConfig(this.props.applicationConfig);
        }
        if (this.props.jobDriver.sparkSubmitJobDriver) {
            this.validateSparkSubmitJobDriver(props.jobDriver.sparkSubmitJobDriver);
        }
        if (this.props.executionRole === undefined
            && sfn.JsonPath.isEncodedJsonPath(props.virtualCluster.id)) {
            throw new Error('Execution role cannot be undefined when the virtual cluster ID is not a concrete value. Provide an execution role with the correct trust policy');
        }
        this.logGroup = this.assignLogGroup();
        this.logBucket = this.assignLogBucket();
        this.role = this.props.executionRole ?? this.createJobExecutionRole();
        this.grantPrincipal = this.role;
        this.grantMonitoringPolicies();
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: (0, task_utils_1.integrationResourceArn)('emr-containers', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                VirtualClusterId: this.props.virtualCluster.id,
                Name: this.props.jobName,
                ExecutionRoleArn: this.role.roleArn,
                ReleaseLabel: this.props.releaseLabel.label,
                JobDriver: {
                    SparkSubmitJobDriver: {
                        EntryPoint: this.props.jobDriver.sparkSubmitJobDriver?.entryPoint.value,
                        EntryPointArguments: this.props.jobDriver.sparkSubmitJobDriver?.entryPointArguments?.value,
                        SparkSubmitParameters: this.props.jobDriver.sparkSubmitJobDriver?.sparkSubmitParameters,
                    },
                },
                ConfigurationOverrides: {
                    ApplicationConfiguration: cdk.listMapper(this.applicationConfigPropertyToJson)(this.props.applicationConfig),
                    MonitoringConfiguration: {
                        CloudWatchMonitoringConfiguration: this.logGroup ? {
                            LogGroupName: this.logGroup.logGroupName,
                            LogStreamNamePrefix: this.props.monitoring.logStreamNamePrefix,
                        } : undefined,
                        PersistentAppUI: (this.props.monitoring?.persistentAppUI === false)
                            ? 'DISABLED'
                            : 'ENABLED',
                        S3MonitoringConfiguration: this.logBucket ? {
                            LogUri: this.logBucket.s3UrlForObject(),
                        } : undefined,
                    },
                },
                Tags: this.props.tags,
            }),
        };
    }
    validateAppConfigPropertiesLength(appConfig) {
        if (appConfig?.properties === undefined) {
            return;
        }
        else if (Object.keys(appConfig.properties).length > 100) {
            throw new Error(`Application configuration properties must have 100 or fewer entries. Received ${Object.keys(appConfig.properties).length}`);
        }
    }
    validatePropertiesNestedAppConfigBothNotUndefined(appConfig) {
        if (appConfig?.properties === undefined && appConfig?.nestedConfig === undefined) {
            throw new Error('Application configuration must have either properties or nested app configurations defined.');
        }
    }
    validateAppConfig(config) {
        if (config === undefined) {
            return;
        }
        else if (config.length > 100) {
            throw new Error(`Application configuration array must have 100 or fewer entries. Received ${config.length}`);
        }
        else {
            config.forEach(element => this.validateAppConfig(element.nestedConfig));
            config.forEach(element => this.validateAppConfigPropertiesLength(element));
            config.forEach(element => this.validatePropertiesNestedAppConfigBothNotUndefined(element));
        }
    }
    isArrayOfStrings(value) {
        return Array.isArray(value) && value.every(item => typeof item === 'string');
    }
    validateEntryPointArguments(entryPointArguments) {
        if (typeof entryPointArguments.value === 'string') {
            if (!sfn.JsonPath.isEncodedJsonPath(entryPointArguments.value)) {
                throw new Error('Entry point arguments must be a string array or an encoded JSON path, but received a non JSON path string');
            }
        }
        else if (!this.isArrayOfStrings(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or an encoded JSON path but received ${typeof entryPointArguments.value}.`);
        }
    }
    validateEntryPointArgumentsLength(entryPointArguments) {
        if (this.isArrayOfStrings(entryPointArguments.value)
            && (entryPointArguments.value.length > 10280 || entryPointArguments.value.length < 1)) {
            throw new Error(`Entry point arguments must be a string array between 1 and 10280 in length. Received ${entryPointArguments.value.length}.`);
        }
    }
    validateSparkSubmitParametersLength(sparkSubmitParameters) {
        if (sparkSubmitParameters.length > 102400 || sparkSubmitParameters.length < 1) {
            throw new Error(`Spark submit parameters must be between 1 and 102400 characters in length. Received ${sparkSubmitParameters.length}.`);
        }
    }
    validateEntryPoint(entryPoint) {
        if (!sfn.JsonPath.isEncodedJsonPath(entryPoint.value) && (entryPoint.value.length > 256 || entryPoint.value.length < 1)) {
            throw new Error(`Entry point must be between 1 and 256 characters in length. Received ${entryPoint.value.length}.`);
        }
    }
    validateSparkSubmitJobDriver(driver) {
        this.validateEntryPoint(driver.entryPoint);
        if (driver.entryPointArguments) {
            this.validateEntryPointArguments(driver.entryPointArguments);
            this.validateEntryPointArgumentsLength(driver.entryPointArguments);
        }
        if (driver.sparkSubmitParameters) {
            this.validateSparkSubmitParametersLength(driver.sparkSubmitParameters);
        }
    }
    // https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/creating-job-execution-role.html
    createJobExecutionRole() {
        const jobExecutionRole = new iam.Role(this, 'Job-Execution-Role', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('emr-containers.amazonaws.com'), new iam.ServicePrincipal('states.amazonaws.com')),
        });
        this.logBucket?.grantReadWrite(jobExecutionRole);
        this.logGroup?.grantWrite(jobExecutionRole);
        this.logGroup?.grant(jobExecutionRole, 'logs:DescribeLogStreams');
        jobExecutionRole.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: '*',
                }),
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
        this.updateRoleTrustPolicy(jobExecutionRole);
        return jobExecutionRole;
    }
    grantMonitoringPolicies() {
        this.logBucket?.grantReadWrite(this.role);
        this.logGroup?.grantWrite(this.role);
        this.logGroup?.grant(this.role, 'logs:DescribeLogStreams');
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: '*',
                }),
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
    }
    /**
     * If an execution role is not provided by user, the automatically generated job execution role must create a trust relationship
     * between itself and the identity of the EMR managed service account in order to run jobs on the Kubernetes namespace.
     *
     * This cannot occur if the user provided virtualClusterId is within an encoded JSON path.
     *
     * The trust relationship can be created by updating the trust policy of the job execution role.
     *
     * @param role the automatically generated job execution role
     */
    updateRoleTrustPolicy(role) {
        const eksClusterInfo = new cr.AwsCustomResource(this, 'GetEksClusterInfo', {
            onCreate: {
                service: 'EMRcontainers',
                action: 'describeVirtualCluster',
                parameters: {
                    id: this.props.virtualCluster.id,
                },
                outputPaths: ['virtualCluster.containerProvider.info.eksInfo.namespace', 'virtualCluster.containerProvider.id'],
                physicalResourceId: cr.PhysicalResourceId.of('id'),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            // APIs are available in 2.1055.0
            installLatestAwsSdk: false,
        });
        /* We make use of custom resources to call update-roll-trust-policy as this command is only available through
         * aws cli because this is only used during the initial setup and is not available through the sdk.
         * https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/update-role-trust-policy.html
         * Commands available through SDK: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/EMRcontainers.html
         * Commands available through CLI: https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/index.html
        */
        const cliLayer = new awscli.AwsCliLayer(this, 'awsclilayer');
        const shellCliLambda = new lambda.SingletonFunction(this, 'Call Update-Role-Trust-Policy', {
            uuid: '8693BB64-9689-44B6-9AAF-B0CC9EB8757C',
            runtime: lambda.Runtime.PYTHON_3_9,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, 'utils/role-policy')),
            timeout: cdk.Duration.seconds(30),
            memorySize: 256,
            layers: [cliLayer],
        });
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'eks',
                    resource: 'cluster',
                    resourceName: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                }),
            ],
            actions: [
                'eks:DescribeCluster',
            ],
        }));
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [role.roleArn],
            actions: [
                'iam:GetRole',
                'iam:UpdateAssumeRolePolicy',
            ],
        }));
        const provider = new cr.Provider(this, 'CustomResourceProvider', {
            onEventHandler: shellCliLambda,
        });
        new cdk.CustomResource(this, 'Custom Resource', {
            properties: {
                eksNamespace: eksClusterInfo.getResponseField('virtualCluster.containerProvider.info.eksInfo.namespace'),
                eksClusterId: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                roleName: role.roleName,
            },
            serviceToken: provider.serviceToken,
        });
    }
    createPolicyStatements() {
        const policyStatements = [
            new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : this.props.virtualCluster.id, // Need wild card for dynamic start job run https://docs.aws.amazon.com/step-functions/latest/dg/emr-eks-iam.html
                    }),
                ],
                actions: ['emr-containers:StartJobRun'],
                conditions: {
                    StringEquals: {
                        'emr-containers:ExecutionRoleArn': this.role.roleArn,
                    },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : `${this.props.virtualCluster.id}/jobruns/*`, // Need wild card for dynamic start job run https://docs.aws.amazon.com/step-functions/latest/dg/emr-eks-iam.html
                    }),
                ],
                actions: [
                    'emr-containers:DescribeJobRun',
                    'emr-containers:CancelJobRun',
                ],
            }));
        }
        return policyStatements;
    }
}
_a = JSII_RTTI_SYMBOL_1;
EmrContainersStartJobRun[_a] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersStartJobRun", version: "2.74.0" };
EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
exports.EmrContainersStartJobRun = EmrContainersStartJobRun;
/**
 * The classification within a EMR Containers application configuration.
 * Class can be extended to add other classifications.
 * For example, new Classification('xxx-yyy');
 */
class Classification {
    /**
     * Creates a new Classification
     *
     * @param classificationStatement A literal string in case a new EMR classification is released, if not already defined.
     */
    constructor(classificationStatement) {
        this.classificationStatement = classificationStatement;
    }
}
_b = JSII_RTTI_SYMBOL_1;
Classification[_b] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.Classification", version: "2.74.0" };
/**
 * Sets the maximizeResourceAllocation property to true or false.
 * When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.
 *
 * For more info:
 * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
 */
Classification.SPARK = new Classification('spark');
/**
 * Sets values in the spark-defaults.conf file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html
 */
Classification.SPARK_DEFAULTS = new Classification('spark-defaults');
/**
 * Sets values in the spark-env.sh file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html#environment-variables
 */
Classification.SPARK_ENV = new Classification('spark-env');
/**
 * Sets values in the hive-site.xml for Spark.
 */
Classification.SPARK_HIVE_SITE = new Classification('spark-hive-site');
/**
 * Sets values in the log4j.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/log4j.properties.template
 */
Classification.SPARK_LOG4J = new Classification('spark-log4j');
/**
 * Sets values in the metrics.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/metrics.properties.template
 */
Classification.SPARK_METRICS = new Classification('spark-metrics');
exports.Classification = Classification;
/**
 * The Amazon EMR release version to use for the job run.
 *
 * Can be extended to include new EMR releases
 *
 * For example, `new ReleaseLabel('emr-x.xx.x-latest');`
 */
class ReleaseLabel {
    /**
     * Initializes the label string.
     *
     * @param label A literal string that contains the release-version ex. 'emr-x.x.x-latest'
     */
    constructor(label) {
        this.label = label;
    }
}
_c = JSII_RTTI_SYMBOL_1;
ReleaseLabel[_c] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.ReleaseLabel", version: "2.74.0" };
/**
 * EMR Release version 5.32.0
 */
ReleaseLabel.EMR_5_32_0 = new ReleaseLabel('emr-5.32.0-latest');
/**
 * EMR Release version 5.33.0
 */
ReleaseLabel.EMR_5_33_0 = new ReleaseLabel('emr-5.33.0-latest');
/**
 * EMR Release version 6.2.0
 */
ReleaseLabel.EMR_6_2_0 = new ReleaseLabel('emr-6.2.0-latest');
/**
 * EMR Release version 6.3.0
 */
ReleaseLabel.EMR_6_3_0 = new ReleaseLabel('emr-6.3.0-latest');
exports.ReleaseLabel = ReleaseLabel;
/**
 * Class that returns a virtual cluster's id depending on input type
 */
class VirtualClusterInput {
    /**
     * Input for a virtualClusterId from a Task Input
     */
    static fromTaskInput(taskInput) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_TaskInput(taskInput);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTaskInput);
            }
            throw error;
        }
        return new VirtualClusterInput(taskInput.value);
    }
    /**
     * Input for virtualClusterId from a literal string
     */
    static fromVirtualClusterId(virtualClusterId) {
        return new VirtualClusterInput(virtualClusterId);
    }
    /**
     * Initializes the virtual cluster ID.
     *
     * @param id The VirtualCluster Id
     */
    constructor(id) {
        this.id = id;
    }
}
_d = JSII_RTTI_SYMBOL_1;
VirtualClusterInput[_d] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.VirtualClusterInput", version: "2.74.0" };
exports.VirtualClusterInput = VirtualClusterInput;
//# sourceMappingURL=data:application/json;base64,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