"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateEndpointConfig = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const cdk = require("../../../core");
const task_utils_1 = require("../private/task-utils");
/**
 * A Step Functions Task to create a SageMaker endpoint configuration
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 */
class SageMakerCreateEndpointConfig extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_tasks_SageMakerCreateEndpointConfigProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SageMakerCreateEndpointConfig);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        (0, task_utils_1.validatePatternSupported)(this.integrationPattern, SageMakerCreateEndpointConfig.SUPPORTED_INTEGRATION_PATTERNS);
        this.validateProductionVariants();
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: (0, task_utils_1.integrationResourceArn)('sagemaker', 'createEndpointConfig', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            EndpointConfigName: this.props.endpointConfigName,
            Tags: this.props.tags?.value,
            KmsKeyId: this.props.kmsKey?.keyId,
            ProductionVariants: this.props.productionVariants.map((variant) => ({
                InitialInstanceCount: variant.initialInstanceCount ? variant.initialInstanceCount : 1,
                InstanceType: sfn.JsonPath.isEncodedJsonPath(variant.instanceType.toString())
                    ? variant.instanceType.toString() : `ml.${variant.instanceType}`,
                ModelName: variant.modelName,
                VariantName: variant.variantName,
                AcceleratorType: variant.acceleratorType,
                InitialVariantWeight: variant.initialVariantWeight,
            })),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/sagemaker/latest/dg/api-permissions-reference.html
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateEndpointConfig'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'endpoint-config',
                        // If the endpoint configuration name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.endpointConfigName) ? '*' : `${this.props.endpointConfigName.toLowerCase()}`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
        ];
    }
    validateProductionVariants() {
        if (this.props.productionVariants.length < 1 || this.props.productionVariants.length > 10) {
            throw new Error('Must specify from 1 to 10 production variants per endpoint configuration');
        }
        this.props.productionVariants.forEach((variant) => {
            if (variant.initialInstanceCount && variant.initialInstanceCount < 1)
                throw new Error('Must define at least one instance');
            if (variant.initialVariantWeight && variant.initialVariantWeight <= 0) {
                throw new Error('InitialVariantWeight has minimum value of 0');
            }
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateEndpointConfig[_a] = { fqn: "aws-cdk-lib.aws_stepfunctions_tasks.SageMakerCreateEndpointConfig", version: "2.74.0" };
SageMakerCreateEndpointConfig.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
exports.SageMakerCreateEndpointConfig = SageMakerCreateEndpointConfig;
//# sourceMappingURL=data:application/json;base64,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