"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderInExpression = exports.jsonPathFromAny = exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("../../../core");
const intrinstics_1 = require("./intrinstics");
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    constructor(path) {
        this.path = path;
        this.creationStack = (0, core_1.captureStackTrace)();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
        handleResolvable: renderResolvable,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathString(x))) {
                found.add(p);
            }
            return {};
        },
        handleList(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathStringList(x))) {
                found.add(p);
            }
            return {};
        },
        handleNumber(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathNumber(x))) {
                found.add(p);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
        handleResolvable(_key, x) {
            for (const p of findPathsInIntrinsicFunctions(jsonPathFromAny(x))) {
                found.add(p);
            }
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
/**
 * From an expression, return the list of JSON paths referenced in it
 */
function findPathsInIntrinsicFunctions(expression) {
    if (!expression) {
        return [];
    }
    const ret = new Array();
    try {
        const parsed = new intrinstics_1.IntrinsicParser(expression).parseTopLevelIntrinsic();
        recurse(parsed);
        return ret;
    }
    catch {
        // Not sure that our parsing is 100% correct. We don't want to break anyone, so
        // fall back to legacy behavior if we can't parse this string.
        return [expression];
    }
    function recurse(p) {
        switch (p.type) {
            case 'path':
                ret.push(p.path);
                break;
            case 'fncall':
                for (const arg of p.arguments) {
                    recurse(arg);
                }
        }
    }
}
function recurseObject(obj, handlers, visited = []) {
    // If the argument received is not actually an object (string, number, boolean, undefined, ...) or null
    // just return it as is as there's nothing to be rendered. This should only happen in the original call to
    // recurseObject as any recursive calls to it are checking for typeof value === 'object' && value !== null
    if (typeof obj !== 'object' || obj === null) {
        return obj;
    }
    // Avoiding infinite recursion
    if (visited.includes(obj)) {
        return {};
    }
    // Marking current object as visited for the current recursion path
    visited.push(obj);
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers, visited));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            if (core_1.Tokenization.isResolvable(value)) {
                Object.assign(ret, handlers.handleResolvable(key, value));
            }
            else {
                ret[key] = recurseObject(value, handlers, visited);
            }
        }
    }
    // Removing from visited after leaving the current recursion path
    // Allowing it to be visited again if it's not causing a recursion (circular reference)
    visited.pop();
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers, visited = []) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers, visited);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a resolvable
 *
 * If we can extract a Path from it, render as a path string, otherwise as itself (will
 * be resolved later
 */
function renderResolvable(key, value) {
    const path = jsonPathFromAny(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
function jsonPathFromAny(x) {
    if (!x) {
        return undefined;
    }
    if (typeof x === 'string') {
        return jsonPathString(x);
    }
    return pathFromToken(core_1.Tokenization.reverse(x));
}
exports.jsonPathFromAny = jsonPathFromAny;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
/**
 * Render the string or number value in a valid JSON Path expression.
 *
 * If the value is a Tokenized JSON path reference -- return the JSON path reference inside it.
 * If the value is a number -- convert it to string.
 * If the value is a string -- single-quote it.
 * Otherwise, throw errors.
 *
 * Call this function whenever you're building compound JSONPath expressions, in
 * order to avoid having tokens-in-tokens-in-tokens which become very hard to parse.
 */
function renderInExpression(x) {
    const path = jsonPathFromAny(x);
    if (path)
        return path;
    if (typeof x === 'number')
        return x.toString(10);
    if (typeof x === 'string')
        return singleQuotestring(x);
    throw new Error('Unxexpected value.');
}
exports.renderInExpression = renderInExpression;
function singleQuotestring(x) {
    const ret = new Array();
    ret.push("'");
    for (const c of x) {
        if (c === "'") {
            ret.push("\\'");
        }
        else if (c === '\\') {
            ret.push('\\\\');
        }
        else if (c === '\n') {
            ret.push('\\n');
        }
        else {
            ret.push(c);
        }
    }
    ret.push("'");
    return ret.join('');
}
//# sourceMappingURL=data:application/json;base64,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