"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateGraph = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A collection of connected states
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *   be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *   machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched
     * @param graphDescription description of the state machine
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_State(startState);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, StateGraph);
            }
            throw error;
        }
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph
     *
     * Called by State.bindToGraph().
     */
    registerState(state) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_State(state);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerState);
            }
            throw error;
        }
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph
     */
    registerPolicyStatement(statement) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerPolicyStatement);
            }
            throw error;
        }
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph
     *
     * Resource changes will be bubbled up to the given graph.
     */
    registerSuperGraph(graph) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_stepfunctions_StateGraph(graph);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerSuperGraph);
            }
            throw error;
        }
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * Return a string description of this graph
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
StateGraph[_a] = { fqn: "aws-cdk-lib.aws_stepfunctions.StateGraph", version: "2.74.0" };
exports.StateGraph = StateGraph;
//# sourceMappingURL=data:application/json;base64,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