"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core = require("../../core");
const cfn_parse = require("../../core/lib/helpers-internal");
const constructs_1 = require("constructs");
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the `getResource` method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cloudformation_include_CfnIncludeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CfnInclude);
            }
            throw error;
        }
        this.allowCyclicalReferences = props.allowCyclicalReferences ?? false;
        this.logicalIdToPlaceholderMap = new Map();
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = props.preserveLogicalIds ?? true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new constructs_1.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new constructs_1.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new constructs_1.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new constructs_1.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new constructs_1.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks'
     * section of the included CloudFormation template with the given logical ID.
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns a loaded NestedStack with name logicalId.
     * For a nested stack to be returned by this method,
     * it must be specified either in the `CfnIncludeProps.loadNestedStacks` property,
     * or through the `loadNestedStack` method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * Includes a template for a child stack inside of this parent template.
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the `CfnIncludeProps.loadNestedStacks`
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     * @param nestedStackProps the properties of the included child Stack
     * @returns the same `IncludedNestedStack` object that `getNestedStack` returns for this logical ID
     */
    loadNestedStack(logicalId, nestedStackProps) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cloudformation_include_CfnIncludeProps(nestedStackProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.loadNestedStack);
            }
            throw error;
        }
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    return self.resources[elementName] ?? self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    return self.resources[elementName] ?? self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = cfnParser.parseValue(hookAttributes.Properties) ?? {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    return self.resources[elementName] ?? self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getPlaceholderID() {
        return `Placeholder${this.logicalIdToPlaceholderMap.size}`;
    }
    getOrCreateResource(logicalId, cycleChain = []) {
        cycleChain = cycleChain.concat([logicalId]);
        if (cycleChain.length !== new Set(cycleChain).size) {
            if (!this.allowCyclicalReferences) {
                throw new Error(`Found a cycle between resources in the template: ${cycleChain.join(' depends on ')}`);
            }
            //only allow one placeholder per logical id
            if (this.logicalIdToPlaceholderMap.get(logicalId)) {
                return this.resources[this.logicalIdToPlaceholderMap.get(logicalId)];
            }
            let placeholderResourceAttributes = this.template.Resources[logicalId];
            let placeholderId = this.getPlaceholderID();
            this.logicalIdToPlaceholderMap.set(logicalId, placeholderId);
            let placeholderInstance = new core.CfnResource(this, placeholderId, {
                type: placeholderResourceAttributes.Type,
                properties: {},
            });
            placeholderInstance.overrideLogicalId(placeholderId);
            this.resources[placeholderId] = placeholderInstance;
            return placeholderInstance;
        }
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId, cycleChain);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        const resourceAttributes = this.template.Resources[logicalId];
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        /*
        1. remove placeholder version of object created for cycle breaking
        2. override logical id before deletion so references to the placeholder instead reference the original
         */
        if (this.logicalIdToPlaceholderMap.get(logicalId)) {
            let placeholderId = this.logicalIdToPlaceholderMap.get(logicalId);
            this.resources[placeholderId].overrideLogicalId(logicalId);
            this.node.tryRemoveChild(placeholderId);
            delete this.resources[placeholderId];
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        // handle any unknown attributes using overrides
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const [attrName, attrValue] of Object.entries(resourceAttributes)) {
            if (!knownAttributes.includes(attrName)) {
                l1Instance.addOverride(attrName, cfnParser.parseValue(attrValue));
            }
        }
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: cfn_parse.FromCloudFormation.getStringArray(nestedStackProps.NotificationARNs).value,
            timeout: this.timeoutForNestedStack(nestedStackProps.TimeoutInMinutes),
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = this.nestedStacksToInclude[nestedStackId].parameters ?? {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = cfn_parse.FromCloudFormation.getString(parameters[paramName]).value;
            }
        }
        return ret;
    }
    timeoutForNestedStack(value) {
        if (value == null) {
            return undefined;
        }
        return core.Duration.minutes(cfn_parse.FromCloudFormation.getNumber(value).value);
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
CfnInclude[_a] = { fqn: "aws-cdk-lib.cloudformation_include.CfnInclude", version: "2.74.0" };
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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