"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNameOfCloudFormationIntrinsic = exports.minimalCloudFormationJoin = exports.CLOUDFORMATION_TOKEN_RESOLVER = exports.CloudFormationLang = void 0;
const cfn_utils_provider_1 = require("./cfn-utils-provider");
const resolve_1 = require("./resolve");
const yaml_cfn = require("./yaml-cfn");
const lazy_1 = require("../lazy");
const resolvable_1 = require("../resolvable");
const stack_1 = require("../stack");
const token_1 = require("../token");
const type_hints_1 = require("../type-hints");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        return lazy_1.Lazy.uncachedString({
            // We used to do this by hooking into `JSON.stringify()` by adding in objects
            // with custom `toJSON()` functions, but it's ultimately simpler just to
            // reimplement the `stringify()` function from scratch.
            produce: (ctx) => tokenAwareStringify(obj, space ?? 0, ctx),
        });
    }
    /**
     * Turn an arbitrary structure potentially containing Tokens into a YAML string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the YAML representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     */
    static toYAML(obj) {
        return lazy_1.Lazy.uncachedString({
            produce: () => yaml_cfn.serialize(obj),
        });
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && isConcatable(parts[0]) && isConcatable(parts[1])) {
            return `${parts[0]}${parts[1]}`;
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return fnJoinConcat(parts);
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Return a CFN intrinsic mass concatenating any number of CloudFormation expressions
 */
function fnJoinConcat(parts) {
    return { 'Fn::Join': ['', minimalCloudFormationJoin('', parts)] };
}
/**
 * Perform a JSON.stringify()-like operation, except aware of Tokens and CloudFormation intrincics
 *
 * Tokens will be resolved and if any resolve to CloudFormation intrinsics, the intrinsics
 * will be lifted to the top of a giant `{ Fn::Join }` expression.
 *
 * If Tokens resolve to primitive types (for example, by using Lazies), we'll
 * use the primitive type to determine how to encode the value into the JSON.
 *
 * If Tokens resolve to CloudFormation intrinsics, we'll use the type of the encoded
 * value as a type hint to determine how to encode the value into the JSON. The difference
 * is that we add quotes (") around strings, and don't add anything around non-strings.
 *
 * The following structure:
 *
 *    { SomeAttr: resource.someAttr }
 *
 * Will JSONify to either:
 *
 *    '{ "SomeAttr": "' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ '" }'
 * or '{ "SomeAttr": ' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ ' }'
 *
 * Depending on whether `someAttr` is type-hinted to be a string or not.
 *
 * (Where ++ is the CloudFormation string-concat operation (`{ Fn::Join }`).
 *
 * -----------------------
 *
 * This work requires 2 features from the `resolve()` function:
 *
 * - INTRINSICS TYPE HINTS: intrinsics are represented by values like
 *   `{ Ref: 'XYZ' }`. These values can reference either a string or a list/number at
 *   deploy time, and from the value alone there's no way to know which. We need
 *   to know the type to know whether to JSONify this reference to:
 *
 *      '{ "referencedValue": "' ++ { Ref: XYZ } ++ '"}'
 *   or '{ "referencedValue": ' ++ { Ref: XYZ } ++ '}'
 *
 *   I.e., whether or not we need to enclose the reference in quotes or not.
 *
 *   We COULD have done this by resolving one token at a time, and looking at the
 *   type of the encoded token we were resolving to obtain a type hint. However,
 *   the `resolve()` and Token system resist a level-at-a-time resolve
 *   operation: because of the existence of post-processors, we must have done a
 *   complete recursive resolution of a token before we can look at its result
 *   (after which any type information about the sources of nested resolved
 *   values is lost).
 *
 *   To fix this, "type hints" have been added to the `resolve()` function,
 *   giving an idea of the type of the source value for complex result values.
 *   This only works for objects (not strings and numbers) but fortunately
 *   we only care about the types of intrinsics, which are always complex values.
 *
 *   Type hinting could have been added to the `IResolvable` protocol as well,
 *   but for now we just use the type of an encoded value as a type hint. That way
 *   we don't need to annotate anything more at the L1 level--we will use the type
 *   encodings added by construct authors at the L2 levels. L1 users can escape the
 *   default decision of "string" by using `Token.asList()`.
 *
 * - COMPLEX KEYS: since tokens can be string-encoded, we can use string-encoded tokens
 *   as the keys in JavaScript objects. However, after resolution, those string-encoded
 *   tokens could resolve to intrinsics (`{ Ref: ... }`), which CANNOT be stored in
 *   JavaScript objects anymore.
 *
 *   We therefore need a protocol to store the resolved values somewhere in the JavaScript
 *   type model,  which can be returned by `resolve()`, and interpreted by `tokenAwareStringify()`
 *   to produce the correct JSON.
 *
 *   And example will quickly show the point:
 *
 *    User writes:
 *       { [resource.resourceName]: 'SomeValue' }
 *    ------ string actually looks like ------>
 *       { '${Token[1234]}': 'SomeValue' }
 *    ------ resolve ------->
 *       { '$IntrinsicKey$0': [ {Ref: Resource}, 'SomeValue' ] }
 *    ------ tokenAwareStringify ------->
 *       '{ "' ++ { Ref: Resource } ++ '": "SomeValue" }'
 */
function tokenAwareStringify(root, space, ctx) {
    let indent = 0;
    const ret = new Array();
    // First completely resolve the tree, then encode to JSON while respecting the type
    // hints we got for the resolved intrinsics.
    recurse(ctx.resolve(root, { allowIntrinsicKeys: true }));
    switch (ret.length) {
        case 0: return undefined;
        case 1: return renderSegment(ret[0]);
        default:
            return fnJoinConcat(ret.map(renderSegment));
    }
    /**
     * Stringify a JSON element
     */
    function recurse(obj) {
        if (obj === undefined) {
            return;
        }
        if (token_1.Token.isUnresolved(obj)) {
            throw new Error("This shouldn't happen anymore");
        }
        if (Array.isArray(obj)) {
            return renderCollection('[', ']', obj, recurse);
        }
        if (typeof obj === 'object' && obj != null && !(obj instanceof Date)) {
            // Treat as an intrinsic if this LOOKS like a CFN intrinsic (`{ Ref: ... }`)
            // AND it's the result of a token resolution. Otherwise, we just treat this
            // value as a regular old JSON object (that happens to look a lot like an intrinsic).
            if (isIntrinsic(obj) && (0, resolve_1.resolvedTypeHint)(obj)) {
                renderIntrinsic(obj);
                return;
            }
            return renderCollection('{', '}', definedEntries(obj), ([key, value]) => {
                if (key.startsWith(resolve_1.INTRINSIC_KEY_PREFIX)) {
                    [key, value] = value;
                }
                recurse(key);
                pushLiteral(prettyPunctuation(':'));
                recurse(value);
            });
        }
        // Otherwise we have a scalar, defer to JSON.stringify()s serialization
        pushLiteral(JSON.stringify(obj));
    }
    /**
     * Render an object or list
     */
    function renderCollection(pre, post, xs, each) {
        pushLiteral(pre);
        indent += space;
        let atLeastOne = false;
        for (const [comma, item] of sepIter(xs)) {
            if (comma) {
                pushLiteral(',');
            }
            pushLineBreak();
            each(item);
            atLeastOne = true;
        }
        indent -= space;
        if (atLeastOne) {
            pushLineBreak();
        }
        pushLiteral(post);
    }
    function renderIntrinsic(intrinsic) {
        switch ((0, resolve_1.resolvedTypeHint)(intrinsic)) {
            case type_hints_1.ResolutionTypeHint.STRING:
                pushLiteral('"');
                pushIntrinsic(deepQuoteStringLiterals(intrinsic));
                pushLiteral('"');
                return;
            case type_hints_1.ResolutionTypeHint.STRING_LIST:
                // We need this to look like:
                //
                //    '{"listValue":' ++ STRINGIFY(CFN_EVAL({ Ref: MyList })) ++ '}'
                //
                // However, STRINGIFY would need to execute at CloudFormation deployment time, and that doesn't exist.
                //
                // We could *ALMOST* use:
                //
                //   '{"listValue":["' ++ JOIN('","', { Ref: MyList }) ++ '"]}'
                //
                // But that has the unfortunate side effect that if `CFN_EVAL({ Ref: MyList }) == []`, then it would
                // evaluate to `[""]`, which is a different value. Since CloudFormation does not have arbitrary
                // conditionals there's no way to deal with this case properly.
                //
                // Therefore, if we encounter lists we need to defer to a custom resource to handle
                // them properly at deploy time.
                const stack = stack_1.Stack.of(ctx.scope);
                // Because this will be called twice (once during `prepare`, once during `resolve`),
                // we need to make sure to be idempotent, so use a cache.
                const stringifyResponse = stringifyCache.obtain(stack, JSON.stringify(intrinsic), () => cfn_utils_provider_1.CfnUtils.stringify(stack, `CdkJsonStringify${stringifyCounter++}`, intrinsic));
                pushIntrinsic(stringifyResponse);
                return;
            case type_hints_1.ResolutionTypeHint.NUMBER:
                pushIntrinsic(intrinsic);
                return;
        }
        throw new Error(`Unexpected type hint: ${(0, resolve_1.resolvedTypeHint)(intrinsic)}`);
    }
    /**
     * Push a literal onto the current segment if it's also a literal, otherwise open a new Segment
     */
    function pushLiteral(lit) {
        let last = ret[ret.length - 1];
        if (last?.type !== 'literal') {
            last = { type: 'literal', parts: [] };
            ret.push(last);
        }
        last.parts.push(lit);
    }
    /**
     * Add a new intrinsic segment
     */
    function pushIntrinsic(intrinsic) {
        ret.push({ type: 'intrinsic', intrinsic });
    }
    /**
     * Push a line break if we are pretty-printing, otherwise don't
     */
    function pushLineBreak() {
        if (space > 0) {
            pushLiteral(`\n${' '.repeat(indent)}`);
        }
    }
    /**
     * Add a space after the punctuation if we are pretty-printing, no space if not
     */
    function prettyPunctuation(punc) {
        return space > 0 ? `${punc} ` : punc;
    }
}
/**
 * Render a segment
 */
function renderSegment(s) {
    switch (s.type) {
        case 'literal': return s.parts.join('');
        case 'intrinsic': return s.intrinsic;
    }
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    },
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
/**
 * Do an intelligent CloudFormation join on the given values, producing a minimal expression
 */
function minimalCloudFormationJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el['Fn::Join'][1]);
        }
        else if (i > 0 && isConcatable(values[i - 1]) && isConcatable(values[i])) {
            values[i - 1] = `${values[i - 1]}${delimiter}${values[i]}`;
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isSplicableFnJoinIntrinsic(obj) {
        if (!isIntrinsic(obj)) {
            return false;
        }
        if (Object.keys(obj)[0] !== 'Fn::Join') {
            return false;
        }
        const [delim, list] = obj['Fn::Join'];
        if (delim !== delimiter) {
            return false;
        }
        if (token_1.Token.isUnresolved(list)) {
            return false;
        }
        if (!Array.isArray(list)) {
            return false;
        }
        return true;
    }
}
exports.minimalCloudFormationJoin = minimalCloudFormationJoin;
function isConcatable(obj) {
    return ['string', 'number'].includes(typeof obj) && !token_1.Token.isUnresolved(obj);
}
/**
 * Return whether the given value represents a CloudFormation intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== 'object') {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || isNameOfCloudFormationIntrinsic(keys[0]);
}
function isNameOfCloudFormationIntrinsic(name) {
    if (!name.startsWith('Fn::')) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a CFN CodePipeline Action
    return name !== 'Fn::GetArtifactAtt' && name !== 'Fn::GetParam';
}
exports.isNameOfCloudFormationIntrinsic = isNameOfCloudFormationIntrinsic;
/**
 * Separated iterator
 */
function* sepIter(xs) {
    let comma = false;
    for (const item of xs) {
        yield [comma, item];
        comma = true;
    }
}
/**
 * Object.entries() but skipping undefined values
 */
function* definedEntries(xs) {
    for (const [key, value] of Object.entries(xs)) {
        if (value !== undefined) {
            yield [key, value];
        }
    }
}
/**
 * Quote string literals inside an intrinsic
 *
 * Formally, this should only match string literals that will be interpreted as
 * string literals. Fortunately, the strings that should NOT be quoted are
 * Logical IDs and attribute names, which cannot contain quotes anyway. Hence,
 * we can get away not caring about the distinction and just quoting everything.
 */
function deepQuoteStringLiterals(x) {
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringLiterals);
    }
    if (typeof x === 'object' && x != null) {
        const ret = {};
        for (const [key, value] of Object.entries(x)) {
            ret[deepQuoteStringLiterals(key)] = deepQuoteStringLiterals(value);
        }
        return ret;
    }
    if (typeof x === 'string') {
        return quoteString(x);
    }
    return x;
}
/**
 * Quote the characters inside a string, for use inside toJSON
 */
function quoteString(s) {
    s = JSON.stringify(s);
    return s.substring(1, s.length - 1);
}
let stringifyCounter = 1;
/**
 * A cache scoped to object instances, that's maintained externally to the object instances
 */
class ScopedCache {
    constructor() {
        this.cache = new WeakMap();
    }
    obtain(object, key, init) {
        let kvMap = this.cache.get(object);
        if (!kvMap) {
            kvMap = new Map();
            this.cache.set(object, kvMap);
        }
        let ret = kvMap.get(key);
        if (ret === undefined) {
            ret = init();
            kvMap.set(key, ret);
        }
        return ret;
    }
}
const stringifyCache = new ScopedCache();
//# sourceMappingURL=data:application/json;base64,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