"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGeneratedWhenNeededMarker = exports.GeneratedWhenNeededMarker = exports.generatePhysicalName = void 0;
const crypto = require("crypto");
const constructs_1 = require("constructs");
const token_map_1 = require("./token-map");
const names_1 = require("../names");
const stack_1 = require("../stack");
const token_1 = require("../token");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(names_1.Names.nodeUniqueId(resource.node), 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${constructs_1.Node.of(resource).path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${constructs_1.Node.of(resource).path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for('@aws-cdk/core.<private>.GenerateIfNeeded');
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to CloudFormation. Use "this.physicalName" instead');
    }
    toString() {
        return 'PhysicalName.GENERATE_IF_NEEDED';
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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