"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.constructInfoFromStack = exports.constructInfoFromConstruct = void 0;
const app_1 = require("../app");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const ALLOWED_FQN_PREFIXES = [
    // SCOPES
    '@aws-cdk/', '@aws-cdk-containers/', '@aws-solutions-konstruk/', '@aws-solutions-constructs/', '@amzn/', '@cdklabs/',
    // PACKAGES
    'aws-rfdk.', 'aws-cdk-lib.', 'cdk8s.',
];
/**
 * Symbol for accessing jsii runtime information
 *
 * Introduced in jsii 1.19.0, cdk 1.90.0.
 */
const JSII_RUNTIME_SYMBOL = Symbol.for('jsii.rtti');
function constructInfoFromConstruct(construct) {
    const jsiiRuntimeInfo = Object.getPrototypeOf(construct).constructor[JSII_RUNTIME_SYMBOL];
    if (typeof jsiiRuntimeInfo === 'object'
        && jsiiRuntimeInfo !== null
        && typeof jsiiRuntimeInfo.fqn === 'string'
        && typeof jsiiRuntimeInfo.version === 'string') {
        return { fqn: jsiiRuntimeInfo.fqn, version: jsiiRuntimeInfo.version };
    }
    else if (jsiiRuntimeInfo) {
        // There is something defined, but doesn't match our expectations. Fail fast and hard.
        throw new Error(`malformed jsii runtime info for construct: '${construct.node.path}'`);
    }
    return undefined;
}
exports.constructInfoFromConstruct = constructInfoFromConstruct;
/**
 * Add analytics data for any validation plugins that are used.
 * Since validation plugins are not constructs we have to handle them
 * as a special case
 */
function addValidationPluginInfo(stack, allConstructInfos) {
    let stage = stage_1.Stage.of(stack);
    let done = false;
    do {
        if (app_1.App.isApp(stage)) {
            done = true;
        }
        if (stage) {
            allConstructInfos.push(...stage.policyValidationBeta1.map(plugin => {
                return {
                    // the fqn can be in the format of `package.module.construct`
                    // those get pulled out into separate fields
                    fqn: `policyValidation.${plugin.name}`,
                    version: plugin.version ?? '0.0.0',
                };
            }));
            stage = stage_1.Stage.of(stage);
        }
    } while (!done && stage);
}
/**
 * For a given stack, walks the tree and finds the runtime info for all constructs within the tree.
 * Returns the unique list of construct info present in the stack,
 * as long as the construct fully-qualified names match the defined allow list.
 */
function constructInfoFromStack(stack) {
    const isDefined = (value) => value !== undefined;
    const allConstructInfos = constructsInStack(stack)
        .map(construct => constructInfoFromConstruct(construct))
        .filter(isDefined)
        .filter(info => ALLOWED_FQN_PREFIXES.find(prefix => info.fqn.startsWith(prefix)));
    // Adds the jsii runtime as a psuedo construct for reporting purposes.
    allConstructInfos.push({
        fqn: 'jsii-runtime.Runtime',
        version: getJsiiAgentVersion(),
    });
    addValidationPluginInfo(stack, allConstructInfos);
    // Filter out duplicate values
    const uniqKeys = new Set();
    return allConstructInfos.filter(construct => {
        const constructKey = `${construct.fqn}@${construct.version}`;
        const isDuplicate = uniqKeys.has(constructKey);
        uniqKeys.add(constructKey);
        return !isDuplicate;
    });
}
exports.constructInfoFromStack = constructInfoFromStack;
/**
 * Returns all constructs under the parent construct (including the parent),
 * stopping when it reaches a boundary of another stack (e.g., Stack, Stage, NestedStack).
 */
function constructsInStack(construct) {
    const constructs = [construct];
    construct.node.children
        .filter(child => !stage_1.Stage.isStage(child) && !stack_1.Stack.isStack(child))
        .forEach(child => constructs.push(...constructsInStack(child)));
    return constructs;
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    // Sanitize the agent to remove characters which might mess with the downstream
    // prefix encoding & decoding. In particular the .NET jsii agent takes a form like:
    // DotNet/5.0.3/.NETCoreApp,Version=v3.1/1.0.0.0
    // The `,` in the above messes with the prefix decoding when reporting the analytics.
    jsiiAgent = jsiiAgent.replace(/[^a-z0-9.-/=_]/gi, '-');
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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