"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomSynthesis = exports.synthesize = void 0;
const crypto_1 = require("crypto");
const fs = require("fs");
const path = require("path");
const cxapi = require("../../../cx-api");
const metadata_resource_1 = require("./metadata-resource");
const prepare_app_1 = require("./prepare-app");
const tree_metadata_1 = require("./tree-metadata");
const annotations_1 = require("../annotations");
const app_1 = require("../app");
const aspect_1 = require("../aspect");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const construct_tree_1 = require("../validation/private/construct-tree");
const report_1 = require("../validation/private/report");
const POLICY_VALIDATION_FILE_PATH = 'policy-validation-report.json';
const VALIDATION_REPORT_JSON_CONTEXT = '@aws-cdk/core:validationReportJson';
function synthesize(root, options = {}) {
    // add the TreeMetadata resource to the App first
    injectTreeMetadata(root);
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    injectMetadataResources(root);
    // resolve references
    (0, prepare_app_1.prepareApp)(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder, options.validateOnSynthesis);
    const assembly = builder.buildAssembly();
    invokeValidationPlugins(root, builder.outdir, assembly);
    return assembly;
}
exports.synthesize = synthesize;
/**
 * Find all the assemblies in the app, including all levels of nested assemblies
 * and return a map where the assemblyId is the key
 */
function getAssemblies(root, rootAssembly) {
    const assemblies = new Map();
    assemblies.set(root.artifactId, rootAssembly);
    visitAssemblies(root, 'pre', construct => {
        const stage = construct;
        if (stage.parentStage && assemblies.has(stage.parentStage.artifactId)) {
            assemblies.set(stage.artifactId, assemblies.get(stage.parentStage.artifactId).getNestedAssembly(stage.artifactId));
        }
    });
    return assemblies;
}
/**
 * Invoke validation plugins for all stages in an App.
 */
function invokeValidationPlugins(root, outdir, assembly) {
    if (!app_1.App.isApp(root))
        return;
    const hash = computeChecksumOfFolder(outdir);
    const assemblies = getAssemblies(root, assembly);
    const templatePathsByPlugin = new Map();
    visitAssemblies(root, 'post', construct => {
        if (stage_1.Stage.isStage(construct)) {
            for (const plugin of construct.policyValidationBeta1) {
                if (!templatePathsByPlugin.has(plugin)) {
                    templatePathsByPlugin.set(plugin, []);
                }
                let assemblyToUse = assemblies.get(construct.artifactId);
                if (!assemblyToUse)
                    throw new Error(`Validation failed, cannot find cloud assembly for stage ${construct.stageName}`);
                templatePathsByPlugin.get(plugin).push(...assemblyToUse.stacksRecursively.map(stack => stack.templateFullPath));
            }
        }
    });
    const reports = [];
    if (templatePathsByPlugin.size > 0) {
        // eslint-disable-next-line no-console
        console.log('Performing Policy Validations\n');
    }
    for (const [plugin, paths] of templatePathsByPlugin.entries()) {
        try {
            const report = plugin.validate({ templatePaths: paths });
            reports.push({ ...report, pluginName: plugin.name });
        }
        catch (e) {
            reports.push({
                success: false,
                pluginName: plugin.name,
                pluginVersion: plugin.version,
                violations: [],
                metadata: {
                    error: `Validation plugin '${plugin.name}' failed: ${e.message}`,
                },
            });
        }
        if (computeChecksumOfFolder(outdir) !== hash) {
            throw new Error(`Illegal operation: validation plugin '${plugin.name}' modified the cloud assembly`);
        }
    }
    if (reports.length > 0) {
        const tree = new construct_tree_1.ConstructTree(root);
        const formatter = new report_1.PolicyValidationReportFormatter(tree);
        const formatJson = root.node.tryGetContext(VALIDATION_REPORT_JSON_CONTEXT) ?? false;
        const output = formatJson
            ? formatter.formatJson(reports)
            : formatter.formatPrettyPrinted(reports);
        const reportFile = path.join(assembly.directory, POLICY_VALIDATION_FILE_PATH);
        if (formatJson) {
            fs.writeFileSync(reportFile, JSON.stringify(output, undefined, 2));
        }
        else {
            // eslint-disable-next-line no-console
            console.error(output);
        }
        const failed = reports.some(r => !r.success);
        if (failed) {
            const message = formatJson
                ? `Validation failed. See the validation report in '${reportFile}' for details`
                : 'Validation failed. See the validation report above for details';
            // eslint-disable-next-line no-console
            console.log(message);
            process.exitCode = 1;
        }
        else {
            // eslint-disable-next-line no-console
            console.log('Policy Validation Successful!');
        }
    }
}
function computeChecksumOfFolder(folder) {
    const hash = (0, crypto_1.createHash)('sha256');
    const files = fs.readdirSync(folder, { withFileTypes: true });
    for (const file of files) {
        const fullPath = path.join(folder, file.name);
        if (file.isDirectory()) {
            hash.update(computeChecksumOfFolder(fullPath));
        }
        else if (file.isFile()) {
            hash.update(fs.readFileSync(fullPath));
        }
    }
    return hash.digest().toString('hex');
}
const CUSTOM_SYNTHESIS_SYM = Symbol.for('@aws-cdk/core:customSynthesis');
function addCustomSynthesis(construct, synthesis) {
    Object.defineProperty(construct, CUSTOM_SYNTHESIS_SYM, {
        value: synthesis,
        enumerable: false,
    });
}
exports.addCustomSynthesis = addCustomSynthesis;
function getCustomSynthesis(construct) {
    return construct[CUSTOM_SYNTHESIS_SYM];
}
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...inheritedAspects ?? [], ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning('We detected an Aspect was added via another Aspect, and will not be applied');
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Find all stacks and add Metadata Resources to all of them
 *
 * There is no good generic place to do this. Can't do it in the constructor
 * (because adding a child construct makes it impossible to set context on the
 * node), and the generic prepare phase is deprecated.
 *
 * Only do this on [parent] stacks (not nested stacks), don't do this when
 * disabled by the user.
 *
 * Also, only when running via the CLI. If we do it unconditionally,
 * all unit tests everywhere are going to break massively. I've spent a day
 * fixing our own, but downstream users would be affected just as badly.
 *
 * Stop at Assembly boundaries.
 */
function injectMetadataResources(root) {
    visit(root, 'post', construct => {
        if (!stack_1.Stack.isStack(construct) || !construct._versionReportingEnabled) {
            return;
        }
        // Because of https://github.com/aws/aws-cdk/blob/main/packages/assert-internal/lib/synth-utils.ts#L74
        // synthesize() may be called more than once on a stack in unit tests, and the below would break
        // if we execute it a second time. Guard against the constructs already existing.
        const CDKMetadata = 'CDKMetadata';
        if (construct.node.tryFindChild(CDKMetadata)) {
            return;
        }
        new metadata_resource_1.MetadataResource(construct, CDKMetadata);
    });
}
/**
 * Find the root App and add the TreeMetadata resource (if enabled).
 *
 * There is no good generic place to do this. Can't do it in the constructor
 * (because adding a child construct makes it impossible to set context on the
 * node), and the generic prepare phase is deprecated.
 */
function injectTreeMetadata(root) {
    visit(root, 'post', construct => {
        if (!app_1.App.isApp(construct) || !construct._treeMetadata)
            return;
        const CDKTreeMetadata = 'Tree';
        if (construct.node.tryFindChild(CDKTreeMetadata))
            return;
        new tree_metadata_1.TreeMetadata(construct);
    });
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder, validateOnSynth = false) {
    visit(root, 'post', construct => {
        const session = {
            outdir: builder.outdir,
            assembly: builder,
            validateOnSynth,
        };
        if (stack_1.Stack.isStack(construct)) {
            construct.synthesizer.synthesize(session);
        }
        else if (construct instanceof tree_metadata_1.TreeMetadata) {
            construct._synthesizeTree(session);
        }
        else {
            const custom = getCustomSynthesis(construct);
            custom?.onSynthesize(session);
        }
    });
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.node.validate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, only looking at Assemblies
 */
function visitAssemblies(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (!stage_1.Stage.isStage(child)) {
            continue;
        }
        visitAssemblies(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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