"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TreeMetadata = void 0;
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema");
const constructs_1 = require("constructs");
const runtime_info_1 = require("./runtime-info");
const annotations_1 = require("../annotations");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 */
class TreeMetadata extends constructs_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    /**
     * Create tree.json
     * @internal
     */
    _synthesizeTree(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    annotations_1.Annotations.of(this).addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const parent = construct.node.scope;
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                parent: parent && parent.node.path ? {
                    id: parent.node.id,
                    path: parent.node.path,
                    constructInfo: (0, runtime_info_1.constructInfoFromConstruct)(parent),
                } : undefined,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct),
                constructInfo: (0, runtime_info_1.constructInfoFromConstruct)(construct),
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        this._tree = lookup;
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, (key, value) => {
            // we are adding in the `parent` attribute for internal use
            // and it doesn't make much sense to include it in the
            // tree.json
            if (key === 'parent')
                return undefined;
            return value;
        }, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH,
            },
        });
    }
    /**
     * Each node will only have 1 level up (node.parent.parent will always be undefined)
     * so we need to reconstruct the node making sure the parents are set
     */
    getNodeWithParents(node) {
        if (!this._tree) {
            throw new Error(`attempting to get node branch for ${node.path}, but the tree has not been created yet!`);
        }
        let tree = node;
        if (node.parent) {
            tree = {
                ...node,
                parent: this.getNodeWithParents(this._tree[node.parent.path]),
            };
        }
        return tree;
    }
    /**
     * Construct a new tree with only the nodes that we care about.
     * Normally each node can contain many child nodes, but we only care about the
     * tree that leads to a specific construct so drop any nodes not in that path
     *
     * @param node Node the current tree node
     * @param child Node the previous tree node and the current node's child node
     * @returns Node the new tree
     */
    renderTreeWithChildren(node, child) {
        if (node.parent) {
            return this.renderTreeWithChildren(node.parent, node);
        }
        else if (child) {
            return {
                ...node,
                children: {
                    [child.id]: child,
                },
            };
        }
        return node;
    }
    /**
     * This gets a specific "branch" of the tree for a given construct path.
     * It will return the root Node of the tree with non-relevant branches filtered
     * out (i.e. node children that don't traverse to the given construct path)
     *
     * @internal
     */
    _getNodeBranch(constructPath) {
        if (!this._tree) {
            throw new Error(`attempting to get node branch for ${constructPath}, but the tree has not been created yet!`);
        }
        const tree = this._tree[constructPath];
        const treeWithParents = this.getNodeWithParents(tree);
        return this.renderTreeWithChildren(treeWithParents);
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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