"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueResourceName = void 0;
const md5_1 = require("./md5");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = 'Resource';
/**
* Resources with this ID are complete hidden from the logical ID calculation.
*/
const HIDDEN_ID = 'Default';
const PATH_SEP = '/';
const MAX_LEN = 256;
const HASH_LEN = 8;
function makeUniqueResourceName(components, options) {
    const maxLength = options.maxLength ?? 256;
    const separator = options.separator ?? '';
    const prefix = options.prefix ?? '';
    components = components.filter(x => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error('Unable to calculate a unique resource name for an empty set of components');
    }
    // top-level resources will simply use the `name` as-is if the name is also short enough
    // in order to support transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        const topLevelResource = prefix + removeNonAllowedSpecialCharacters(components[0], separator, options.allowedSpecialCharacters);
        if (topLevelResource.length <= maxLength) {
            return topLevelResource;
        }
    }
    // Calculate the hash from the full path, included unresolved tokens so the hash value is always unique
    const hash = pathHash(components);
    if (prefix) {
        components.unshift(prefix);
    }
    const human = removeDupes(components)
        .filter(pathElement => pathElement !== HIDDEN_FROM_HUMAN_ID)
        .map(pathElement => removeNonAllowedSpecialCharacters(pathElement, separator, options.allowedSpecialCharacters))
        .filter(pathElement => pathElement)
        .join(separator)
        .concat(separator);
    const maxhumanLength = maxLength - HASH_LEN;
    return human.length > maxhumanLength ? `${splitInMiddle(human, maxhumanLength)}${hash}` : `${human}${hash}`;
}
exports.makeUniqueResourceName = makeUniqueResourceName;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = (0, md5_1.md5hash)(path.join(PATH_SEP));
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-allowed special characters in a string.
 */
function removeNonAllowedSpecialCharacters(s, _separator, allowedSpecialCharacters) {
    const pattern = allowedSpecialCharacters ? `[^A-Za-z0-9${allowedSpecialCharacters}]` : '[^A-Za-z0-9]';
    const regex = new RegExp(pattern, 'g');
    return s.replace(regex, '');
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
function splitInMiddle(s, maxLength = MAX_LEN - HASH_LEN) {
    const half = maxLength / 2;
    return s.slice(0, half) + s.slice(-half);
}
//# sourceMappingURL=data:application/json;base64,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