"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetManifestBuilder = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxschema = require("../../../cloud-assembly-schema");
const _shared_1 = require("./_shared");
const assets_1 = require("../assets");
/**
 * Build an asset manifest from assets added to a stack
 *
 * This class does not need to be used by app builders; it is only necessary for building Stack Synthesizers.
 */
class AssetManifestBuilder {
    constructor() {
        this.files = {};
        this.dockerImages = {};
    }
    /**
     * Add a file asset to the manifest with default settings
     *
     * Derive the region from the stack, use the asset hash as the key, copy the
     * file extension over, and set the prefix.
     */
    defaultAddFileAsset(stack, asset, target) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_FileAssetSource(asset);
            jsiiDeprecationWarnings.aws_cdk_lib_AssetManifestFileDestination(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.defaultAddFileAsset);
            }
            throw error;
        }
        validateFileAssetSource(asset);
        const extension = asset.fileName != undefined ? path.extname(asset.fileName) : '';
        const objectKey = (target.bucketPrefix ?? '') +
            asset.sourceHash +
            (asset.packaging === assets_1.FileAssetPackaging.ZIP_DIRECTORY
                ? '.zip'
                : extension);
        // Add to manifest
        return this.addFileAsset(stack, asset.sourceHash, {
            path: asset.fileName,
            executable: asset.executable,
            packaging: asset.packaging,
        }, {
            bucketName: target.bucketName,
            objectKey,
            region: (0, _shared_1.resolvedOr)(stack.region, undefined),
            assumeRoleArn: target.role?.assumeRoleArn,
            assumeRoleExternalId: target.role?.assumeRoleExternalId,
        });
    }
    /**
     * Add a docker image asset to the manifest with default settings
     *
     * Derive the region from the stack, use the asset hash as the key, and set the prefix.
     */
    defaultAddDockerImageAsset(stack, asset, target) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_DockerImageAssetSource(asset);
            jsiiDeprecationWarnings.aws_cdk_lib_AssetManifestDockerImageDestination(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.defaultAddDockerImageAsset);
            }
            throw error;
        }
        validateDockerImageAssetSource(asset);
        const imageTag = `${target.dockerTagPrefix ?? ''}${asset.sourceHash}`;
        // Add to manifest
        return this.addDockerImageAsset(stack, asset.sourceHash, {
            executable: asset.executable,
            directory: asset.directoryName,
            dockerBuildArgs: asset.dockerBuildArgs,
            dockerBuildSecrets: asset.dockerBuildSecrets,
            dockerBuildTarget: asset.dockerBuildTarget,
            dockerFile: asset.dockerFile,
            networkMode: asset.networkMode,
            platform: asset.platform,
            dockerOutputs: asset.dockerOutputs,
            cacheFrom: asset.dockerCacheFrom,
            cacheTo: asset.dockerCacheTo,
        }, {
            repositoryName: target.repositoryName,
            imageTag,
            region: (0, _shared_1.resolvedOr)(stack.region, undefined),
            assumeRoleArn: target.role?.assumeRoleArn,
            assumeRoleExternalId: target.role?.assumeRoleExternalId,
        });
    }
    /**
     * Add a file asset source and destination to the manifest
     *
     * sourceHash should be unique for every source.
     */
    addFileAsset(stack, sourceHash, source, dest) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_FileSource(source);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_FileDestination(dest);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFileAsset);
            }
            throw error;
        }
        if (!this.files[sourceHash]) {
            this.files[sourceHash] = {
                source,
                destinations: {},
            };
        }
        this.files[sourceHash].destinations[this.manifestEnvName(stack)] = dest;
        return dest;
    }
    /**
     * Add a docker asset source and destination to the manifest
     *
     * sourceHash should be unique for every source.
     */
    addDockerImageAsset(stack, sourceHash, source, dest) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_DockerImageSource(source);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_DockerImageDestination(dest);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDockerImageAsset);
            }
            throw error;
        }
        if (!this.dockerImages[sourceHash]) {
            this.dockerImages[sourceHash] = {
                source,
                destinations: {},
            };
        }
        this.dockerImages[sourceHash].destinations[this.manifestEnvName(stack)] = dest;
        return dest;
    }
    /**
     * Whether there are any assets registered in the manifest
     */
    get hasAssets() {
        return Object.keys(this.files).length + Object.keys(this.dockerImages).length > 0;
    }
    /**
     * Write the manifest to disk, and add it to the synthesis session
     *
     * Return the artifact id, which should be added to the `additionalDependencies`
     * field of the stack artifact.
     */
    emitManifest(stack, session, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_ISynthesisSession(session);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_AssetManifestOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.emitManifest);
            }
            throw error;
        }
        const artifactId = `${stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        const manifest = {
            version: cxschema.Manifest.version(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        fs.writeFileSync(outPath, JSON.stringify(manifest, undefined, 2));
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
                ...options,
            },
        });
        return artifactId;
    }
    manifestEnvName(stack) {
        return [
            (0, _shared_1.resolvedOr)(stack.account, 'current_account'),
            (0, _shared_1.resolvedOr)(stack.region, 'current_region'),
        ].join('-');
    }
}
_a = JSII_RTTI_SYMBOL_1;
AssetManifestBuilder[_a] = { fqn: "aws-cdk-lib.AssetManifestBuilder", version: "2.74.0" };
exports.AssetManifestBuilder = AssetManifestBuilder;
function validateFileAssetSource(asset) {
    if (!!asset.executable === !!asset.fileName) {
        throw new Error(`Exactly one of 'fileName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    if (!!asset.packaging !== !!asset.fileName) {
        throw new Error(`'packaging' is expected in combination with 'fileName', got: ${JSON.stringify(asset)}`);
    }
}
function validateDockerImageAssetSource(asset) {
    if (!!asset.executable === !!asset.directoryName) {
        throw new Error(`Exactly one of 'directoryName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    check('dockerBuildArgs');
    check('dockerBuildTarget');
    check('dockerOutputs');
    check('dockerFile');
    function check(key) {
        if (asset[key] && !asset.directoryName) {
            throw new Error(`'${key}' is only allowed in combination with 'directoryName', got: ${JSON.stringify(asset)}`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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