"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyStackSynthesizer = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("../../../cloud-assembly-schema");
const cxapi = require("../../../cx-api");
const constructs_1 = require("constructs");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
const cfn_fn_1 = require("../cfn-fn");
const asset_parameters_1 = require("../private/asset-parameters");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the CDK classic way of referencing assets
 *
 * This synthesizer will generate CloudFormation parameters for every referenced
 * asset, and use the CLI's current credentials to deploy the stack.
 *
 * - It does not support cross-account deployment (the CLI must have credentials
 *   to the account you are trying to deploy to).
 * - It cannot be used with **CDK Pipelines**. To deploy using CDK Pipelines,
 *   you must use the `DefaultStackSynthesizer`.
 * - Each asset will take up a CloudFormation Parameter in your template. Keep in
 *   mind that there is a maximum of 200 parameters in a CloudFormation template.
 *   To use deterministic asset locations instead, use `CliCredentialsStackSynthesizer`.
 *
 * Be aware that your CLI credentials must be valid for the duration of the
 * entire deployment. If you are using session credentials, make sure the
 * session lifetime is long enough.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 */
class LegacyStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    constructor() {
        super(...arguments);
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    addFileAsset(asset) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_FileAssetSource(asset);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFileAsset);
            }
            throw error;
        }
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        //
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            const stack = this.boundStack;
            return withoutDeprecationWarnings(() => stack.addFileAsset(asset));
        }
        finally {
            this.cycle = false;
        }
    }
    addDockerImageAsset(asset) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_DockerImageAssetSource(asset);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDockerImageAsset);
            }
            throw error;
        }
        // See `addFileAsset` for explanation.
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            const stack = this.boundStack;
            return withoutDeprecationWarnings(() => stack.addDockerImageAsset(asset));
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Synthesize the associated stack to the session
     */
    synthesize(session) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_ISynthesisSession(session);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.synthesize);
            }
            throw error;
        }
        this.synthesizeTemplate(session);
        // Just do the default stuff, nothing special
        this.emitArtifact(session);
    }
    /**
     * Produce a bound Stack Synthesizer for the given stack.
     *
     * This method may be called more than once on the same object.
     */
    reusableBind(stack) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.reusableBind);
            }
            throw error;
        }
        // Create a copy of the current object and bind that
        const copy = Object.create(this);
        copy.bind(stack);
        return copy;
    }
    doAddDockerImageAsset(asset) {
        // check if we have an override from context
        const repositoryNameOverride = this.boundStack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = asset.repositoryName ?? repositoryNameOverride ?? ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            if (!asset.directoryName) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
                networkMode: asset.networkMode,
                platform: asset.platform,
                outputs: asset.dockerOutputs,
                cacheFrom: asset.dockerCacheFrom,
                cacheTo: asset.dockerCacheTo,
            };
            this.boundStack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.boundStack.account}.dkr.ecr.${this.boundStack.region}.${this.boundStack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            if (!asset.fileName || !asset.packaging) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.boundStack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.boundStack.region}.${this.boundStack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        (0, _shared_1.assertBound)(this.boundStack);
        if (!this._assetParameters) {
            this._assetParameters = new constructs_1.Construct(this.boundStack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
_a = JSII_RTTI_SYMBOL_1;
LegacyStackSynthesizer[_a] = { fqn: "aws-cdk-lib.LegacyStackSynthesizer", version: "2.74.0" };
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
function withoutDeprecationWarnings(block) {
    const orig = process.env.JSII_DEPRECATED;
    process.env.JSII_DEPRECATED = 'quiet';
    try {
        return block();
    }
    finally {
        process.env.JSII_DEPRECATED = orig;
    }
}
//# sourceMappingURL=data:application/json;base64,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