"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackSynthesizer = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxapi = require("../../../cx-api");
const _shared_1 = require("./_shared");
const assets_1 = require("../assets");
const cfn_fn_1 = require("../cfn-fn");
const cfn_parameter_1 = require("../cfn-parameter");
const cfn_rule_1 = require("../cfn-rule");
/**
 * Base class for implementing an IStackSynthesizer
 *
 * This class needs to exist to provide public surface area for external
 * implementations of stack synthesizers. The protected methods give
 * access to functions that are otherwise @_internal to the framework
 * and could not be accessed by external implementors.
 */
class StackSynthesizer {
    /**
     * The qualifier used to bootstrap this stack
     */
    get bootstrapQualifier() {
        return undefined;
    }
    /**
     * Bind to the stack this environment is going to be used on
     *
     * Must be called before any of the other methods are called.
     */
    bind(stack) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (this._boundStack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this._boundStack = stack;
    }
    /**
     * Have the stack write out its template
     *
     * @deprecated Use `synthesizeTemplate` instead
     */
    synthesizeStackTemplate(stack, session) {
        try {
            jsiiDeprecationWarnings.print("aws-cdk-lib.StackSynthesizer#synthesizeStackTemplate", "Use `synthesizeTemplate` instead");
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_ISynthesisSession(session);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.synthesizeStackTemplate);
            }
            throw error;
        }
        stack._synthesizeTemplate(session);
    }
    /**
     * Write the stack template to the given session
     *
     * Return a descriptor that represents the stack template as a file asset
     * source, for adding to an asset manifest (if desired). This can be used to
     * have the asset manifest system (`cdk-assets`) upload the template to S3
     * using the appropriate role, so that afterwards only a CloudFormation
     * deployment is necessary.
     *
     * If the template is uploaded as an asset, the `stackTemplateAssetObjectUrl`
     * property should be set when calling `emitArtifact.`
     *
     * If the template is *NOT* uploaded as an asset first and the template turns
     * out to be >50KB, it will need to be uploaded to S3 anyway. At that point
     * the credentials will be the same identity that is doing the `UpdateStack`
     * call, which may not have the right permissions to write to S3.
     */
    synthesizeTemplate(session, lookupRoleArn) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_ISynthesisSession(session);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.synthesizeTemplate);
            }
            throw error;
        }
        this.boundStack._synthesizeTemplate(session, lookupRoleArn);
        return stackTemplateFileAsset(this.boundStack, session);
    }
    /**
     * Write the stack artifact to the session
     *
     * Use default settings to add a CloudFormationStackArtifact artifact to
     * the given synthesis session.
     *
     * @deprecated Use `emitArtifact` instead
     */
    emitStackArtifact(stack, session, options = {}) {
        try {
            jsiiDeprecationWarnings.print("aws-cdk-lib.StackSynthesizer#emitStackArtifact", "Use `emitArtifact` instead");
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_ISynthesisSession(session);
            jsiiDeprecationWarnings.aws_cdk_lib_SynthesizeStackArtifactOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.emitStackArtifact);
            }
            throw error;
        }
        (0, _shared_1.addStackArtifactToAssembly)(session, stack, options ?? {}, options.additionalDependencies ?? []);
    }
    /**
     * Write the CloudFormation stack artifact to the session
     *
     * Use default settings to add a CloudFormationStackArtifact artifact to
     * the given synthesis session. The Stack artifact will control the settings for the
     * CloudFormation deployment.
     */
    emitArtifact(session, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_ISynthesisSession(session);
            jsiiDeprecationWarnings.aws_cdk_lib_SynthesizeStackArtifactOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.emitArtifact);
            }
            throw error;
        }
        (0, _shared_1.addStackArtifactToAssembly)(session, this.boundStack, options ?? {}, options.additionalDependencies ?? []);
    }
    /**
     * Add a CfnRule to the bound stack that checks whether an SSM parameter exceeds a given version
     *
     * This will modify the template, so must be called before the stack is synthesized.
     */
    addBootstrapVersionRule(requiredVersion, bootstrapStackVersionSsmParameter) {
        addBootstrapVersionRule(this.boundStack, requiredVersion, bootstrapStackVersionSsmParameter);
    }
    /**
     * Retrieve the bound stack
     *
     * Fails if the stack hasn't been bound yet.
     */
    get boundStack() {
        if (!this._boundStack) {
            throw new Error('The StackSynthesizer must be bound to a Stack first before boundStack() can be called');
        }
        return this._boundStack;
    }
    /**
     * Turn a file asset location into a CloudFormation representation of that location
     *
     * If any of the fields contain placeholders, the result will be wrapped in a `Fn.sub`.
     */
    cloudFormationLocationFromFileAsset(location) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_FileDestination(location);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.cloudFormationLocationFromFileAsset);
            }
            throw error;
        }
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.boundStack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${location.bucketName}/${location.objectKey}`);
        const s3ObjectUrlWithPlaceholders = `s3://${location.bucketName}/${location.objectKey}`;
        // Return CFN expression
        //
        // 's3ObjectUrlWithPlaceholders' is intended for the CLI. The CLI ultimately needs a
        // 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL to give to CloudFormation.
        // However, there's no way for us to actually know the URL_SUFFIX in the framework, so
        // we can't construct that URL. Instead, we record the 's3://.../...' form, and the CLI
        // transforms it to the correct 'https://.../' URL before calling CloudFormation.
        return {
            bucketName: cfnify(location.bucketName),
            objectKey: cfnify(location.objectKey),
            httpUrl,
            s3ObjectUrl: cfnify(s3ObjectUrlWithPlaceholders),
            s3ObjectUrlWithPlaceholders,
            s3Url: httpUrl,
        };
    }
    /**
     * Turn a docker asset location into a CloudFormation representation of that location
     *
     * If any of the fields contain placeholders, the result will be wrapped in a `Fn.sub`.
     */
    cloudFormationLocationFromDockerImageAsset(dest) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_DockerImageDestination(dest);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.cloudFormationLocationFromDockerImageAsset);
            }
            throw error;
        }
        const { account, region, urlSuffix } = stackLocationOrInstrinsics(this.boundStack);
        // Return CFN expression
        return {
            repositoryName: cfnify(dest.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${region}.${urlSuffix}/${dest.repositoryName}:${dest.imageTag}`),
            imageTag: cfnify(dest.imageTag),
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
StackSynthesizer[_a] = { fqn: "aws-cdk-lib.StackSynthesizer", version: "2.74.0" };
exports.StackSynthesizer = StackSynthesizer;
function stackTemplateFileAsset(stack, session) {
    const templatePath = path.join(session.assembly.outdir, stack.templateFile);
    if (!fs.existsSync(templatePath)) {
        throw new Error(`Stack template ${stack.stackName} not written yet: ${templatePath}`);
    }
    const template = fs.readFileSync(templatePath, { encoding: 'utf-8' });
    const sourceHash = (0, _shared_1.contentHash)(template);
    return {
        fileName: stack.templateFile,
        packaging: assets_1.FileAssetPackaging.FILE,
        sourceHash,
    };
}
/**
 * Add a CfnRule to the Stack which checks the current version of the bootstrap stack this template is targeting
 *
 * The CLI normally checks this, but in a pipeline the CLI is not involved
 * so we encode this rule into the template in a way that CloudFormation will check it.
 */
function addBootstrapVersionRule(stack, requiredVersion, bootstrapStackVersionSsmParameter) {
    // Because of https://github.com/aws/aws-cdk/blob/main/packages/assert-internal/lib/synth-utils.ts#L74
    // synthesize() may be called more than once on a stack in unit tests, and the below would break
    // if we execute it a second time. Guard against the constructs already existing.
    if (stack.node.tryFindChild('BootstrapVersion')) {
        return;
    }
    const param = new cfn_parameter_1.CfnParameter(stack, 'BootstrapVersion', {
        type: 'AWS::SSM::Parameter::Value<String>',
        description: `Version of the CDK Bootstrap resources in this environment, automatically retrieved from SSM Parameter Store. ${cxapi.SSMPARAM_NO_INVALIDATE}`,
        default: bootstrapStackVersionSsmParameter,
    });
    // There is no >= check in CloudFormation, so we have to check the number
    // is NOT in [1, 2, 3, ... <required> - 1]
    const oldVersions = range(1, requiredVersion).map(n => `${n}`);
    new cfn_rule_1.CfnRule(stack, 'CheckBootstrapVersion', {
        assertions: [
            {
                assert: cfn_fn_1.Fn.conditionNot(cfn_fn_1.Fn.conditionContains(oldVersions, param.valueAsString)),
                assertDescription: `CDK bootstrap stack version ${requiredVersion} required. Please run 'cdk bootstrap' with a recent version of the CDK CLI.`,
            },
        ],
    });
}
function range(startIncl, endExcl) {
    const ret = new Array();
    for (let i = startIncl; i < endExcl; i++) {
        ret.push(i);
    }
    return ret;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: (0, _shared_1.resolvedOr)(stack.account, '${AWS::AccountId}'),
        region: (0, _shared_1.resolvedOr)(stack.region, '${AWS::Region}'),
        urlSuffix: (0, _shared_1.resolvedOr)(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? cfn_fn_1.Fn.sub(s) : s;
}
//# sourceMappingURL=data:application/json;base64,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