"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
const synthesis_1 = require("./private/synthesis");
const STAGE_SYMBOL = Symbol.for('@aws-cdk/core.Stage');
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends constructs_1.Construct {
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     */
    static of(construct) {
        return constructs_1.Node.of(construct).scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     */
    static isStage(x) {
        return x !== null && typeof (x) === 'object' && STAGE_SYMBOL in x;
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * Validation plugins to run during synthesis. If any plugin reports any violation,
         * synthesis will be interrupted and the report displayed to the user.
         *
         * @default - no validation plugins are used
         */
        this.policyValidationBeta1 = [];
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_StageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Stage);
            }
            throw error;
        }
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        Object.defineProperty(this, STAGE_SYMBOL, { value: true });
        this.parentStage = Stage.of(this);
        this.region = props.env?.region ?? this.parentStage?.region;
        this.account = props.env?.account ?? this.parentStage?.account;
        props.permissionsBoundary?._bind(this);
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [this.parentStage?.stageName, props.stageName ?? id].filter(x => x).join('-');
        if (props.policyValidationBeta1) {
            this.policyValidationBeta1 = props.policyValidationBeta1;
        }
    }
    /**
     * The cloud assembly output directory.
     */
    get outdir() {
        return this._assemblyBuilder.outdir;
    }
    /**
     * The cloud assembly asset output directory.
     */
    get assetOutdir() {
        return this._assemblyBuilder.assetOutdir;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_StageSynthesisOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.synth);
            }
            throw error;
        }
        if (!this.assembly || options.force) {
            this.assembly = (0, synthesis_1.synthesize)(this, {
                skipValidation: options.skipValidation,
                validateOnSynthesis: options.validateOnSynthesis,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Stage[_a] = { fqn: "aws-cdk-lib.Stage", version: "2.74.0" };
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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