"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstructTree = void 0;
const app_1 = require("../../app");
const cfn_resource_1 = require("../../cfn-resource");
const stack_1 = require("../../stack");
/**
 * Utility class to help accessing information on constructs in the
 * construct tree. This can be created once and shared between
 * all the validation plugin executions.
 */
class ConstructTree {
    constructor(root) {
        this.root = root;
        /**
         * A cache of the ConstructTrace by node.path. Each construct
         */
        this._traceCache = new Map();
        this._constructByPath = new Map();
        this._constructByTemplatePathAndLogicalId = new Map();
        if (app_1.App.isApp(this.root)) {
            this.treeMetadata = this.root.node.tryFindChild('Tree');
        }
        else {
            this.treeMetadata = app_1.App.of(this.root)?.node.tryFindChild('Tree');
        }
        this._constructByPath.set(this.root.node.path, root);
        // do this once at the start so we don't have to traverse
        // the entire tree everytime we want to find a nested node
        this.root.node.findAll().forEach(child => {
            this._constructByPath.set(child.node.path, child);
            const defaultChild = child.node.defaultChild;
            if (defaultChild && cfn_resource_1.CfnResource.isCfnResource(defaultChild)) {
                const stack = stack_1.Stack.of(defaultChild);
                const logicalId = stack.resolve(defaultChild.logicalId);
                this.setLogicalId(stack, logicalId, child);
            }
        });
        // Another pass to include all the L1s that haven't been added yet
        this.root.node.findAll().forEach(child => {
            if (cfn_resource_1.CfnResource.isCfnResource(child)) {
                const stack = stack_1.Stack.of(child);
                const logicalId = stack_1.Stack.of(child).resolve(child.logicalId);
                this.setLogicalId(stack, logicalId, child);
            }
        });
    }
    setLogicalId(stack, logicalId, child) {
        if (!this._constructByTemplatePathAndLogicalId.has(stack.templateFile)) {
            this._constructByTemplatePathAndLogicalId.set(stack.templateFile, new Map([[logicalId, child]]));
        }
        else {
            this._constructByTemplatePathAndLogicalId.get(stack.templateFile)?.set(logicalId, child);
        }
    }
    /**
     * Get the stack trace from the construct node metadata.
     * The stack trace only gets recorded if the node is a `CfnResource`,
     * but the stack trace will have entries for all types of parent construct
     * scopes
     */
    getTraceMetadata(size, node) {
        if (node) {
            const construct = this.getConstructByPath(node.path);
            if (construct) {
                let trace;
                if (cfn_resource_1.CfnResource.isCfnResource(construct)) {
                    trace = construct.node.metadata.find(meta => !!meta.trace)?.trace ?? [];
                }
                else {
                    trace = construct.node.defaultChild?.node.metadata.find(meta => !!meta.trace)?.trace ?? [];
                }
                // take just the items we need and reverse it since we are
                // displaying the trace bottom up
                return Object.create(trace.slice(0, size));
            }
        }
        return [];
    }
    /**
     * Only the `CfnResource` constructs contain the trace information
     * So we need to go down the tree and find that resource (its always the last one)
     *
     * @param node Node the entire tree where the bottom is the violating resource
     * @return Node the bottom of the tree which will be the violating resource
     */
    getNodeWithTrace(node) {
        if (node.children) {
            return this.getNodeWithTrace(this.getChild(node.children));
        }
        return node;
    }
    /**
     * Get a ConstructTrace from the cache for a given construct
     *
     * Construct the stack trace of constructs. This will start with the
     * root of the tree and go down to the construct that has the violation
     */
    getTrace(node, locations) {
        const trace = this._traceCache.get(node.path);
        if (trace) {
            return trace;
        }
        const size = this.nodeSize(node);
        const nodeWithTrace = this.getNodeWithTrace(node);
        const metadata = (locations ?? this.getTraceMetadata(size, nodeWithTrace));
        const thisLocation = metadata.pop();
        const constructTrace = {
            id: node.id,
            path: node.path,
            // the "child" trace will be the "parent" node
            // since we are going bottom up
            child: node.children
                ? this.getTrace(this.getChild(node.children), metadata)
                : undefined,
            construct: node.constructInfo?.fqn,
            libraryVersion: node.constructInfo?.version,
            location: thisLocation ?? "Run with '--debug' to include location info",
        };
        this._traceCache.set(constructTrace.path, constructTrace);
        return constructTrace;
    }
    /**
     * Each node will only have a single child so just
     * return that
     */
    getChild(children) {
        return Object.values(children)[0];
    }
    /**
     * Get the size of a Node, i.e. how many levels is it
     */
    nodeSize(node) {
        let size = 1;
        if (!node.children) {
            return size;
        }
        let children = this.getChild(node.children);
        do {
            size++;
            children = children.children
                ? this.getChild(children.children)
                : undefined;
        } while (children);
        return size;
    }
    /**
     * Get a specific node in the tree by construct path
     *
     * @param path the construct path of the node to return
     * @returns the TreeMetadata Node
     */
    getTreeNode(path) {
        return this.treeMetadata._getNodeBranch(path);
    }
    /**
     * Get a specific Construct by the node.addr
     *
     * @param path the node.addr of the construct
     * @returns the Construct
     */
    getConstructByPath(path) {
        return this._constructByPath.get(path);
    }
    /**
     * Get a specific Construct by the CfnResource logical ID. This will
     * be the construct.node.defaultChild with the given ID
     *
     * @param logicalId the ID of the CfnResource
     * @returns the Construct
     */
    getConstructByLogicalId(templateFile, logicalId) {
        return this._constructByTemplatePathAndLogicalId.get(templateFile)?.get(logicalId);
    }
}
exports.ConstructTree = ConstructTree;
//# sourceMappingURL=data:application/json;base64,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