"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReportTrace = void 0;
const STARTER_LINE = '└── ';
const VERTICAL_LINE = '│';
/**
 * Utility class to generate the construct stack trace
 * for a report
 */
class ReportTrace {
    constructor(tree) {
        this.tree = tree;
    }
    /**
     * Return a JSON representation of the construct trace
     */
    formatJson(constructPath) {
        return this.trace(constructPath);
    }
    /**
     * This will render something like this:
     *
     *   Creation Stack:
     *     └──  MyStack (MyStack)
     *          │ Library: aws-cdk-lib.Stack
     *          │ Library Version: 2.50.0
     *          │ Location: Object.<anonymous> (/home/hallcor/tmp/cdk-tmp-app/src/main.ts:25:20)
     *          └──  MyCustomL3Construct (MyStack/MyCustomL3Construct)
     *               │ Library: N/A - (Local Construct)
     *               │ Library Version: N/A
     *               │ Location: new MyStack (/home/hallcor/tmp/cdk-tmp-app/src/main.ts:15:20)
     *               └──  Bucket (MyStack/MyCustomL3Construct/Bucket)
     *                    │ Library: aws-cdk-lib/aws-s3.Bucket
     *                    │ Library Version: 2.50.0
     *                    │ Location: new MyCustomL3Construct (/home/hallcor/tmp/cdk-tmp-app/src/main.ts:9:20)/
     */
    formatPrettyPrinted(constructPath) {
        const trace = this.formatJson(constructPath);
        return this.renderPrettyPrintedTraceInfo(trace);
    }
    renderPrettyPrintedTraceInfo(info, indent, start = STARTER_LINE) {
        const notAvailableMessage = '\tConstruct trace not available. Rerun with `--debug` to see trace information';
        if (info) {
            const indentation = indent ?? ' '.repeat(STARTER_LINE.length + 1);
            const result = [
                `${start} ${info?.id} (${info?.path})`,
                `${indentation}${VERTICAL_LINE} Construct: ${info?.construct}`,
                `${indentation}${VERTICAL_LINE} Library Version: ${info?.libraryVersion}`,
                `${indentation}${VERTICAL_LINE} Location: ${info?.location}`,
                ...info?.child ? [this.renderPrettyPrintedTraceInfo(info?.child, ' '.repeat(indentation.length + STARTER_LINE.length + 1), indentation + STARTER_LINE)] : [],
            ];
            return result.join('\n\t');
        }
        return notAvailableMessage;
    }
    trace(constructPath) {
        if (constructPath) {
            const treeNode = this.tree.getTreeNode(constructPath);
            if (treeNode) {
                return this.tree.getTrace(treeNode);
            }
        }
        return;
    }
}
exports.ReportTrace = ReportTrace;
//# sourceMappingURL=data:application/json;base64,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