"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("../../cloud-assembly-schema");
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     *
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cx_api_CloudAssembly(assembly);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_ArtifactManifest(artifact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromManifest);
            }
            throw error;
        }
        // Implementation is defined in a separate file to break cyclic dependencies
        void (assembly), void (id), void (artifact);
        throw new Error('Implementation not overridden yet');
    }
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cx_api_CloudAssembly(assembly);
            jsiiDeprecationWarnings.aws_cdk_lib_cloud_assembly_schema_ArtifactManifest(manifest);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudArtifact);
            }
            throw error;
        }
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
    /**
     * An identifier that shows where this artifact is located in the tree
     * of nested assemblies, based on their manifests. Defaults to the normal
     * id. Should only be used in user interfaces.
     */
    get hierarchicalId() {
        return this.manifest.displayName ?? this.id;
    }
}
_a = JSII_RTTI_SYMBOL_1;
CloudArtifact[_a] = { fqn: "aws-cdk-lib.cx_api.CloudArtifact", version: "2.74.0" };
exports.CloudArtifact = CloudArtifact;
//# sourceMappingURL=data:application/json;base64,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