"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackDeployment = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const url_1 = require("url");
const cxapi = require("../../../cx-api");
const asset_type_1 = require("./asset-type");
const asset_manifest_1 = require("../private/asset-manifest");
const cloud_assembly_internals_1 = require("../private/cloud-assembly-internals");
/**
 * Deployment of a single Stack
 *
 * You don't need to instantiate this class -- it will
 * be automatically instantiated as necessary when you
 * add a `Stage` to a pipeline.
 */
class StackDeployment {
    /**
     * Build a `StackDeployment` from a Stack Artifact in a Cloud Assembly.
     */
    static fromArtifact(stackArtifact) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_cx_api_CloudFormationStackArtifact(stackArtifact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromArtifact);
            }
            throw error;
        }
        const artRegion = stackArtifact.environment.region;
        const region = artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = stackArtifact.environment.account;
        const account = artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        return new StackDeployment({
            account,
            region,
            tags: stackArtifact.tags,
            stackArtifactId: stackArtifact.id,
            constructPath: stackArtifact.hierarchicalId,
            stackName: stackArtifact.stackName,
            absoluteTemplatePath: path.join(stackArtifact.assembly.directory, stackArtifact.templateFile),
            assumeRoleArn: stackArtifact.assumeRoleArn,
            executionRoleArn: stackArtifact.cloudFormationExecutionRoleArn,
            assets: extractStackAssets(stackArtifact),
            templateS3Uri: stackArtifact.stackTemplateAssetObjectUrl,
        });
    }
    constructor(props) {
        /**
         * Other stacks this stack depends on
         */
        this.stackDependencies = [];
        /**
         * Steps that take place before stack is prepared. If your pipeline engine disables 'prepareStep', then this will happen before stack deploys
         */
        this.pre = [];
        /**
         * Steps that take place after stack is prepared but before stack deploys. Your pipeline engine may not disable `prepareStep`.
         */
        this.changeSet = [];
        /**
         * Steps to execute after stack deploys
         */
        this.post = [];
        this.stackArtifactId = props.stackArtifactId;
        this.constructPath = props.constructPath;
        this.account = props.account;
        this.region = props.region;
        this.tags = props.tags ?? {};
        this.assumeRoleArn = props.assumeRoleArn;
        this.executionRoleArn = props.executionRoleArn;
        this.stackName = props.stackName;
        this.absoluteTemplatePath = props.absoluteTemplatePath;
        this.templateUrl = props.templateS3Uri ? s3UrlFromUri(props.templateS3Uri, props.region) : undefined;
        this.assets = new Array();
        for (const asset of props.assets ?? []) {
            if (asset.isTemplate) {
                this.templateAsset = asset;
            }
            else {
                this.assets.push(asset);
            }
        }
    }
    /**
     * Add a dependency on another stack
     */
    addStackDependency(stackDeployment) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_StackDeployment(stackDeployment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addStackDependency);
            }
            throw error;
        }
        this.stackDependencies.push(stackDeployment);
    }
    /**
     * Adds steps to each phase of the stack
     * @param pre steps executed before stack.prepare
     * @param changeSet steps executed after stack.prepare and before stack.deploy
     * @param post steps executed after stack.deploy
     */
    addStackSteps(pre, changeSet, post) {
        this.pre.push(...pre);
        this.changeSet.push(...changeSet);
        this.post.push(...post);
    }
}
_a = JSII_RTTI_SYMBOL_1;
StackDeployment[_a] = { fqn: "aws-cdk-lib.pipelines.StackDeployment", version: "2.74.0" };
exports.StackDeployment = StackDeployment;
function extractStackAssets(stackArtifact) {
    const ret = new Array();
    const assetManifests = stackArtifact.dependencies.filter(cloud_assembly_internals_1.isAssetManifest);
    for (const manifestArtifact of assetManifests) {
        const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
        for (const entry of manifest.entries) {
            let assetType;
            let isTemplate = false;
            if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                assetType = asset_type_1.AssetType.DOCKER_IMAGE;
            }
            else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                isTemplate = entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile;
                assetType = asset_type_1.AssetType.FILE;
            }
            else {
                throw new Error(`Unrecognized asset type: ${entry.type}`);
            }
            ret.push({
                assetManifestPath: manifestArtifact.file,
                assetId: entry.id.assetId,
                assetSelector: entry.id.toString(),
                assetType,
                assetPublishingRoleArn: entry.destination.assumeRoleArn,
                isTemplate,
            });
        }
    }
    return ret;
}
/**
 * Takes an s3://bucket/object-key uri and returns a region-aware https:// url for it
 *
 * @param uri The s3 URI
 * @param region The region (if undefined, we will return the global endpoint)
 * @see https://docs.aws.amazon.com/AmazonS3/latest/userguide/VirtualHosting.html#virtual-hosted-style-access
 */
function s3UrlFromUri(uri, region) {
    const url = (0, url_1.parse)(uri);
    return `https://${url.hostname}.s3.${region ? `${region}.` : ''}amazonaws.com${url.path}`;
}
//# sourceMappingURL=data:application/json;base64,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