"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StageDeployment = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const stack_deployment_1 = require("./stack-deployment");
const cloud_assembly_internals_1 = require("../private/cloud-assembly-internals");
const construct_internals_1 = require("../private/construct-internals");
/**
 * Deployment of a single `Stage`
 *
 * A `Stage` consists of one or more `Stacks`, which will be
 * deployed in dependency order.
 */
class StageDeployment {
    /**
     * Create a new `StageDeployment` from a `Stage`
     *
     * Synthesizes the target stage, and deployes the stacks found inside
     * in dependency order.
     */
    static fromStage(stage, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_StageDeploymentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromStage);
            }
            throw error;
        }
        const assembly = (0, construct_internals_1.pipelineSynth)(stage);
        if (assembly.stacks.length === 0) {
            // If we don't check here, a more puzzling "stage contains no actions"
            // error will be thrown come deployment time.
            throw new Error(`The given Stage construct ('${stage.node.path}') should contain at least one Stack`);
        }
        const stepFromArtifact = new Map();
        for (const artifact of assembly.stacks) {
            const step = stack_deployment_1.StackDeployment.fromArtifact(artifact);
            stepFromArtifact.set(artifact, step);
        }
        if (props.stackSteps) {
            for (const stackstep of props.stackSteps) {
                const stackArtifact = assembly.getStackArtifact(stackstep.stack.artifactId);
                const thisStep = stepFromArtifact.get(stackArtifact);
                if (!thisStep) {
                    throw new Error('Logic error: we just added a step for this artifact but it disappeared.');
                }
                thisStep.addStackSteps(stackstep.pre ?? [], stackstep.changeSet ?? [], stackstep.post ?? []);
            }
        }
        for (const artifact of assembly.stacks) {
            const thisStep = stepFromArtifact.get(artifact);
            if (!thisStep) {
                throw new Error('Logic error: we just added a step for this artifact but it disappeared.');
            }
            const stackDependencies = artifact.dependencies.filter(cloud_assembly_internals_1.isStackArtifact);
            for (const dep of stackDependencies) {
                const depStep = stepFromArtifact.get(dep);
                if (!depStep) {
                    throw new Error(`Stack '${artifact.id}' depends on stack not found in same Stage: '${dep.id}'`);
                }
                thisStep.addStackDependency(depStep);
            }
        }
        return new StageDeployment(Array.from(stepFromArtifact.values()), {
            stageName: stage.stageName,
            ...props,
        });
    }
    constructor(
    /** The stacks deployed in this stage */
    stacks, props = {}) {
        this.stacks = stacks;
        this.stageName = props.stageName ?? '';
        this.pre = props.pre ?? [];
        this.post = props.post ?? [];
        this.stackSteps = props.stackSteps ?? [];
    }
    /**
     * Add an additional step to run before any of the stacks in this stage
     */
    addPre(...steps) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_Step(steps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPre);
            }
            throw error;
        }
        this.pre.push(...steps);
    }
    /**
     * Add an additional step to run after all of the stacks in this stage
     */
    addPost(...steps) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_Step(steps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPost);
            }
            throw error;
        }
        this.post.push(...steps);
    }
}
_a = JSII_RTTI_SYMBOL_1;
StageDeployment[_a] = { fqn: "aws-cdk-lib.pipelines.StageDeployment", version: "2.74.0" };
exports.StageDeployment = StageDeployment;
//# sourceMappingURL=data:application/json;base64,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