"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Step = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../../core");
const step_output_1 = require("../helpers-internal/step-output");
/**
 * A generic Step which can be added to a Pipeline
 *
 * Steps can be used to add Sources, Build Actions and Validations
 * to your pipeline.
 *
 * This class is abstract. See specific subclasses of Step for
 * useful steps to add to your Pipeline
 */
class Step {
    /**
     * Define a sequence of steps to be executed in order.
     *
     * If you need more fine-grained step ordering, use the `addStepDependency()`
     * API. For example, if you want `secondStep` to occur after `firstStep`, call
     * `secondStep.addStepDependency(firstStep)`.
     */
    static sequence(steps) {
        for (let i = 1; i < steps.length; i++) {
            steps[i].addStepDependency(steps[i - 1]);
        }
        return steps;
    }
    constructor(
    /** Identifier for this step */
    id) {
        this.id = id;
        /**
         * The list of FileSets consumed by this Step
         */
        this.dependencyFileSets = [];
        /**
         * Whether or not this is a Source step
         *
         * What it means to be a Source step depends on the engine.
         */
        this.isSource = false;
        this._dependencies = new Set();
        if (core_1.Token.isUnresolved(id)) {
            throw new Error(`Step id cannot be unresolved, got '${id}'`);
        }
    }
    /**
     * Return the steps this step depends on, based on the FileSets it requires
     */
    get dependencies() {
        return Array.from(new Set([
            ...this.dependencyFileSets.map(f => f.producer),
            ...this._dependencies,
        ]));
    }
    /**
     * Return a string representation of this Step
     */
    toString() {
        return `${this.constructor.name}(${this.id})`;
    }
    /**
     * The primary FileSet produced by this Step
     *
     * Not all steps produce an output FileSet--if they do
     * you can substitute the `Step` object for the `FileSet` object.
     */
    get primaryOutput() {
        // Accessor so it can be mutable in children
        return this._primaryOutput;
    }
    /**
     * Add a dependency on another step.
     */
    addStepDependency(step) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_Step(step);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addStepDependency);
            }
            throw error;
        }
        this._dependencies.add(step);
    }
    /**
     * Add an additional FileSet to the set of file sets required by this step
     *
     * This will lead to a dependency on the producer of that file set.
     */
    addDependencyFileSet(fs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_FileSet(fs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDependencyFileSet);
            }
            throw error;
        }
        this.dependencyFileSets.push(fs);
    }
    /**
     * Configure the given FileSet as the primary output of this step
     */
    configurePrimaryOutput(fs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_FileSet(fs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configurePrimaryOutput);
            }
            throw error;
        }
        this._primaryOutput = fs;
    }
    /**
     * Crawl the given structure for references to StepOutputs and add dependencies on all steps found
     *
     * Should be called in the constructor of subclasses based on what the user
     * passes in as construction properties. The format of the structure passed in
     * here does not have to correspond exactly to what gets rendered into the
     * engine, it just needs to contain the same data.
     */
    discoverReferencedOutputs(structure) {
        for (const output of step_output_1.StepOutput.findAll(structure)) {
            this._dependencies.add(output.step);
            step_output_1.StepOutput.recordProducer(output);
        }
    }
    /**
     * StackOutputReferences this step consumes.
     */
    get consumedStackOutputs() {
        return [];
    }
}
_a = JSII_RTTI_SYMBOL_1;
Step[_a] = { fqn: "aws-cdk-lib.pipelines.Step", version: "2.74.0" };
exports.Step = Step;
//# sourceMappingURL=data:application/json;base64,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