"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildStep = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("../../../aws-codebuild");
const buildspecs_1 = require("./private/buildspecs");
const outputs_1 = require("./private/outputs");
const blueprint_1 = require("../blueprint");
/**
 * Run a script as a CodeBuild Project
 *
 * The BuildSpec must be available inline--it cannot reference a file
 * on disk. If your current build instructions are in a file like
 * `buildspec.yml` in your repository, extract them to a script
 * (say, `build.sh`) and invoke that script as part of the build:
 *
 * ```ts
 * new pipelines.CodeBuildStep('Synth', {
 *   commands: ['./build.sh'],
 * });
 * ```
 */
class CodeBuildStep extends blueprint_1.ShellStep {
    constructor(id, props) {
        super(id, props);
        this.exportedVariables = new Set();
        this.exportedVarsRendered = false;
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_CodeBuildStepProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CodeBuildStep);
            }
            throw error;
        }
        this.projectName = props.projectName;
        this.buildEnvironment = props.buildEnvironment;
        this._partialBuildSpec = props.partialBuildSpec;
        this.vpc = props.vpc;
        this.subnetSelection = props.subnetSelection;
        this.cache = props.cache;
        this.role = props.role;
        this.actionRole = props.actionRole;
        this.rolePolicyStatements = props.rolePolicyStatements;
        this.securityGroups = props.securityGroups;
        this.timeout = props.timeout;
        this.fileSystemLocations = props.fileSystemLocations;
    }
    /**
     * CodeBuild Project generated for the pipeline
     *
     * Will only be available after the pipeline has been built.
     */
    get project() {
        if (!this._project) {
            throw new Error('Call pipeline.buildPipeline() before reading this property');
        }
        return this._project;
    }
    /**
     * The CodeBuild Project's principal
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * Additional configuration that can only be configured via BuildSpec
     *
     * Contains exported variables
     *
     * @default - Contains the exported variables
     */
    get partialBuildSpec() {
        this.exportedVarsRendered = true;
        const varsBuildSpec = this.exportedVariables.size > 0 ? codebuild.BuildSpec.fromObject({
            version: '0.2',
            env: {
                'exported-variables': Array.from(this.exportedVariables),
            },
        }) : undefined;
        return (0, buildspecs_1.mergeBuildSpecs)(varsBuildSpec, this._partialBuildSpec);
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuildStep.
     *
     * The variable must be set in the shell of the CodeBuild step when
     * it finishes its `post_build` phase.
     *
     * @param variableName the name of the variable for reference.
     * @example
     * // Access the output of one CodeBuildStep in another CodeBuildStep
     * declare const pipeline: pipelines.CodePipeline;
     *
     * const step1 = new pipelines.CodeBuildStep('Step1', {
     *   commands: ['export MY_VAR=hello'],
     * });
     *
     * const step2 = new pipelines.CodeBuildStep('Step2', {
     *   env: {
     *     IMPORTED_VAR: step1.exportedVariable('MY_VAR'),
     *   },
     *   commands: ['echo $IMPORTED_VAR'],
     * });
     */
    exportedVariable(variableName) {
        if (this.exportedVarsRendered && !this.exportedVariables.has(variableName)) {
            throw new Error('exportVariable(): Pipeline has already been produced, cannot call this function anymore');
        }
        this.exportedVariables.add(variableName);
        return (0, outputs_1.makeCodePipelineOutput)(this, variableName);
    }
    /**
     * Set the internal project value
     *
     * @internal
     */
    _setProject(project) {
        this._project = project;
    }
}
_a = JSII_RTTI_SYMBOL_1;
CodeBuildStep[_a] = { fqn: "aws-cdk-lib.pipelines.CodeBuildStep", version: "2.74.0" };
exports.CodeBuildStep = CodeBuildStep;
//# sourceMappingURL=data:application/json;base64,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