"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineSource = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cp_actions = require("../../../aws-codepipeline-actions");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const outputs_1 = require("./private/outputs");
const blueprint_1 = require("../blueprint");
/**
 * Factory for CodePipeline source steps
 *
 * This class contains a number of factory methods for the different types
 * of sources that CodePipeline supports.
 */
class CodePipelineSource extends blueprint_1.Step {
    constructor() {
        super(...arguments);
        // tells `PipelineGraph` to hoist a "Source" step
        this.isSource = true;
    }
    /**
     * Returns a GitHub source, using OAuth tokens to authenticate with
     * GitHub and a separate webhook to detect changes. This is no longer
     * the recommended method. Please consider using `connection()`
     * instead.
     *
     * Pass in the owner and repository in a single string, like this:
     *
     * ```ts
     * pipelines.CodePipelineSource.gitHub('owner/repo', 'main');
     * ```
     *
     * Authentication will be done by a secret called `github-token` in AWS
     * Secrets Manager (unless specified otherwise).
     *
     * If you rotate the value in the Secret, you must also change at least one property
     * on the Pipeline, to force CloudFormation to re-read the secret.
     *
     * The token should have these permissions:
     *
     * * **repo** - to read the repository
     * * **admin:repo_hook** - if you plan to use webhooks (true by default)
     *
     * If you need access to symlinks or the repository history, use a source of type
     * `connection` instead.
     */
    static gitHub(repoString, branch, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_GitHubSourceOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.gitHub);
            }
            throw error;
        }
        return new GitHubSource(repoString, branch, props);
    }
    /**
     * Returns an S3 source.
     *
     * @param bucket The bucket where the source code is located.
     * @param props The options, which include the key that identifies the source code file and
     * and how the pipeline should be triggered.
     *
     * @example
     * declare const bucket: s3.Bucket;
     * pipelines.CodePipelineSource.s3(bucket, 'path/to/file.zip');
     */
    static s3(bucket, objectKey, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_s3_IBucket(bucket);
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_S3SourceOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.s3);
            }
            throw error;
        }
        return new S3Source(bucket, objectKey, props);
    }
    /**
     * Returns an ECR source.
     *
     * @param repository The repository that will be watched for changes.
     * @param props The options, which include the image tag to be checked for changes.
     *
     * @example
     * declare const repository: ecr.IRepository;
     * pipelines.CodePipelineSource.ecr(repository, {
     *   imageTag: 'latest',
     * });
     */
    static ecr(repository, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_ecr_IRepository(repository);
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_ECRSourceOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ecr);
            }
            throw error;
        }
        return new ECRSource(repository, props);
    }
    /**
     * Returns a CodeStar connection source. A CodeStar connection allows AWS CodePipeline to
     * access external resources, such as repositories in GitHub, GitHub Enterprise or
     * BitBucket.
     *
     * To use this method, you first need to create a CodeStar connection
     * using the AWS console. In the process, you may have to sign in to the external provider
     * -- GitHub, for example -- to authorize AWS to read and modify your repository.
     * Once you have done this, copy the connection ARN and use it to create the source.
     *
     * Example:
     *
     * ```ts
     * pipelines.CodePipelineSource.connection('owner/repo', 'main', {
     *   connectionArn: 'arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41', // Created using the AWS console
     * });
     * ```
     *
     * If you need access to symlinks or the repository history, be sure to set
     * `codeBuildCloneOutput`.
     *
     * @param repoString A string that encodes owner and repository separated by a slash (e.g. 'owner/repo').
     * @param branch The branch to use.
     * @param props The source properties, including the connection ARN.
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/welcome-connections.html
     */
    static connection(repoString, branch, props) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_ConnectionSourceOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.connection);
            }
            throw error;
        }
        return new CodeStarConnectionSource(repoString, branch, props);
    }
    /**
     * Returns a CodeCommit source.
     *
     * If you need access to symlinks or the repository history, be sure to set
     * `codeBuildCloneOutput`.
     *
     *
     * @param repository The CodeCommit repository.
     * @param branch The branch to use.
     * @param props The source properties.
     *
     * @example
     * declare const repository: codecommit.IRepository;
     * pipelines.CodePipelineSource.codeCommit(repository, 'main');
     */
    static codeCommit(repository, branch, props = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codecommit_IRepository(repository);
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_CodeCommitSourceOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.codeCommit);
            }
            throw error;
        }
        return new CodeCommitSource(repository, branch, props);
    }
    produceAction(stage, options) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_codepipeline_IStage(stage);
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_ProduceActionOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.produceAction);
            }
            throw error;
        }
        const output = options.artifacts.toCodePipeline(this.primaryOutput);
        const action = this.getAction(output, options.actionName, options.runOrder, options.variablesNamespace);
        stage.addAction(action);
        return { runOrdersConsumed: 1 };
    }
    /**
     * Return an attribute of the current source revision
     *
     * These values can be passed into the environment variables of pipeline steps,
     * so your steps can access information about the source revision.
     *
     * Pipeline synth step has some source attributes predefined in the environment.
     * If these suffice, you don't need to use this method for the synth step.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-env-vars.html
     *
     * What attributes are available depends on the type of source. These attributes
     * are supported:
     *
     * - GitHub, CodeCommit, and CodeStarSourceConnection
     *   - `AuthorDate`
     *   - `BranchName`
     *   - `CommitId`
     *   - `CommitMessage`
     * - GitHub, CodeCommit and ECR
     *   - `RepositoryName`
     * - GitHub and CodeCommit
     *   - `CommitterDate`
     * - GitHub
     *   - `CommitUrl`
     * - CodeStarSourceConnection
     *   - `FullRepositoryName`
     * - S3
     *   - `ETag`
     *   - `VersionId`
     * - ECR
     *   - `ImageDigest`
     *   - `ImageTag`
     *   - `ImageURI`
     *   - `RegistryId`
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-variables.html#reference-variables-list
     * @example
     * // Access the CommitId of a GitHub source in the synth
     * const source = pipelines.CodePipelineSource.gitHub('owner/repo', 'main');
     *
     * const pipeline = new pipelines.CodePipeline(scope, 'MyPipeline', {
     *   synth: new pipelines.ShellStep('Synth', {
     *     input: source,
     *     commands: [],
     *     env: {
     *       'COMMIT_ID': source.sourceAttribute('CommitId'),
     *     }
     *   })
     * });
     */
    sourceAttribute(name) {
        return (0, outputs_1.makeCodePipelineOutput)(this, name);
    }
}
_a = JSII_RTTI_SYMBOL_1;
CodePipelineSource[_a] = { fqn: "aws-cdk-lib.pipelines.CodePipelineSource", version: "2.74.0" };
exports.CodePipelineSource = CodePipelineSource;
/**
 * Extend CodePipelineSource so we can type-test in the CodePipelineEngine.
 */
class GitHubSource extends CodePipelineSource {
    constructor(repoString, branch, props) {
        super(repoString);
        this.branch = branch;
        this.props = props;
        const parts = repoString.split('/');
        if (core_1.Token.isUnresolved(repoString) || parts.length !== 2) {
            throw new Error(`GitHub repository name should be a resolved string like '<owner>/<repo>', got '${repoString}'`);
        }
        this.owner = parts[0];
        this.repo = parts[1];
        this.authentication = props.authentication ?? core_1.SecretValue.secretsManager('github-token');
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, actionName, runOrder, variablesNamespace) {
        return new cp_actions.GitHubSourceAction({
            output,
            actionName: this.props.actionName ?? actionName,
            runOrder,
            oauthToken: this.authentication,
            owner: this.owner,
            repo: this.repo,
            branch: this.branch,
            trigger: this.props.trigger,
            variablesNamespace,
        });
    }
}
class S3Source extends CodePipelineSource {
    constructor(bucket, objectKey, props) {
        super(constructs_1.Node.of(bucket).addr);
        this.bucket = bucket;
        this.objectKey = objectKey;
        this.props = props;
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, _actionName, runOrder, variablesNamespace) {
        return new cp_actions.S3SourceAction({
            output,
            // Bucket names are guaranteed to conform to ActionName restrictions
            actionName: this.props.actionName ?? this.bucket.bucketName,
            runOrder,
            bucketKey: this.objectKey,
            trigger: this.props.trigger,
            bucket: this.bucket,
            role: this.props.role,
            variablesNamespace,
        });
    }
}
class ECRSource extends CodePipelineSource {
    constructor(repository, props) {
        super(constructs_1.Node.of(repository).addr);
        this.repository = repository;
        this.props = props;
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, _actionName, runOrder, variablesNamespace) {
        // RepositoryName can contain '/' that is not a valid ActionName character, use '_' instead
        const formattedRepositoryName = core_1.Fn.join('_', core_1.Fn.split('/', this.repository.repositoryName));
        return new cp_actions.EcrSourceAction({
            output,
            actionName: this.props.actionName ?? formattedRepositoryName,
            runOrder,
            repository: this.repository,
            imageTag: this.props.imageTag,
            variablesNamespace,
        });
    }
}
class CodeStarConnectionSource extends CodePipelineSource {
    constructor(repoString, branch, props) {
        super(repoString);
        this.branch = branch;
        this.props = props;
        const parts = repoString.split('/');
        if (core_1.Token.isUnresolved(repoString) || parts.length !== 2) {
            throw new Error(`CodeStar repository name should be a resolved string like '<owner>/<repo>', got '${repoString}'`);
        }
        this.owner = parts[0];
        this.repo = parts[1];
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, actionName, runOrder, variablesNamespace) {
        return new cp_actions.CodeStarConnectionsSourceAction({
            output,
            actionName: this.props.actionName ?? actionName,
            runOrder,
            connectionArn: this.props.connectionArn,
            owner: this.owner,
            repo: this.repo,
            branch: this.branch,
            codeBuildCloneOutput: this.props.codeBuildCloneOutput,
            triggerOnPush: this.props.triggerOnPush,
            variablesNamespace,
        });
    }
}
class CodeCommitSource extends CodePipelineSource {
    constructor(repository, branch, props) {
        super(core_1.Token.isUnresolved(repository.repositoryName)
            ? constructs_1.Node.of(repository).addr
            : repository.repositoryName);
        this.repository = repository;
        this.branch = branch;
        this.props = props;
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, _actionName, runOrder, variablesNamespace) {
        return new cp_actions.CodeCommitSourceAction({
            output,
            // Guaranteed to be okay as action name
            actionName: this.props.actionName ?? this.repository.repositoryName,
            runOrder,
            branch: this.branch,
            trigger: this.props.trigger,
            repository: this.repository,
            eventRole: this.props.eventRole,
            codeBuildCloneOutput: this.props.codeBuildCloneOutput,
            variablesNamespace,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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