"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipeline = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cb = require("../../../aws-codebuild");
const cp = require("../../../aws-codepipeline");
const cpa = require("../../../aws-codepipeline-actions");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const cxapi = require("../../../cx-api");
const artifact_map_1 = require("./artifact-map");
const codebuild_step_1 = require("./codebuild-step");
const codebuild_factory_1 = require("./private/codebuild-factory");
const outputs_1 = require("./private/outputs");
const stack_outputs_map_1 = require("./stack-outputs-map");
const blueprint_1 = require("../blueprint");
const docker_credentials_1 = require("../docker-credentials");
const helpers_internal_1 = require("../helpers-internal");
const main_1 = require("../main");
const asset_singleton_role_1 = require("../private/asset-singleton-role");
const cached_fnsub_1 = require("../private/cached-fnsub");
const cli_version_1 = require("../private/cli-version");
const construct_internals_1 = require("../private/construct-internals");
const default_codebuild_image_1 = require("../private/default-codebuild-image");
const fs_1 = require("../private/fs");
const identifiers_1 = require("../private/identifiers");
const javascript_1 = require("../private/javascript");
const template_configuration_1 = require("../private/template-configuration");
/**
 * A CDK Pipeline that uses CodePipeline to deploy CDK apps
 *
 * This is a `Pipeline` with its `engine` property set to
 * `CodePipelineEngine`, and exists for nicer ergonomics for
 * users that don't need to switch out engines.
 */
class CodePipeline extends main_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.artifacts = new artifact_map_1.ArtifactMap();
        this.cachedFnSub = new cached_fnsub_1.CachedFnSub();
        /**
         * Asset roles shared for publishing
         */
        this.assetCodeBuildRoles = new Map();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_pipelines_CodePipelineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CodePipeline);
            }
            throw error;
        }
        this.selfMutationEnabled = props.selfMutation ?? true;
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.singlePublisherPerAssetType = !(props.publishAssetsInParallel ?? true);
        this.cliVersion = props.cliVersion ?? (0, cli_version_1.preferredCliVersion)();
        this.useChangeSets = props.useChangeSets ?? true;
        this.stackOutputs = new stack_outputs_map_1.StackOutputsMap(this);
    }
    /**
     * The CodeBuild project that performs the Synth
     *
     * Only available after the pipeline has been built.
     */
    get synthProject() {
        if (!this._synthProject) {
            throw new Error('Call pipeline.buildPipeline() before reading this property');
        }
        return this._synthProject;
    }
    /**
     * The CodeBuild project that performs the SelfMutation
     *
     * Will throw an error if this is accessed before `buildPipeline()`
     * is called, or if selfMutation has been disabled.
     */
    get selfMutationProject() {
        if (!this._pipeline) {
            throw new Error('Call pipeline.buildPipeline() before reading this property');
        }
        if (!this._selfMutationProject) {
            throw new Error('No selfMutationProject since the selfMutation property was set to false');
        }
        return this._selfMutationProject;
    }
    /**
     * The CodePipeline pipeline that deploys the CDK app
     *
     * Only available after the pipeline has been built.
     */
    get pipeline() {
        if (!this._pipeline) {
            throw new Error('Pipeline not created yet');
        }
        return this._pipeline;
    }
    doBuildPipeline() {
        if (this._pipeline) {
            throw new Error('Pipeline already created');
        }
        this._myCxAsmRoot = path.resolve((0, construct_internals_1.assemblyBuilderOf)((0, construct_internals_1.appOf)(this)).outdir);
        if (this.props.codePipeline) {
            if (this.props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (this.props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (this.props.enableKeyRotation !== undefined) {
                throw new Error('Cannot set \'enableKeyRotation\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (this.props.reuseCrossRegionSupportStacks !== undefined) {
                throw new Error('Cannot set \'reuseCrossRegionSupportStacks\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (this.props.role !== undefined) {
                throw new Error('Cannot set \'role\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (this.props.artifactBucket !== undefined) {
                throw new Error('Cannot set \'artifactBucket\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = this.props.codePipeline;
        }
        else {
            this._pipeline = new cp.Pipeline(this, 'Pipeline', {
                pipelineName: this.props.pipelineName,
                crossAccountKeys: this.props.crossAccountKeys ?? false,
                reuseCrossRegionSupportStacks: this.props.reuseCrossRegionSupportStacks,
                // This is necessary to make self-mutation work (deployments are guaranteed
                // to happen only after the builds of the latest pipeline definition).
                restartExecutionOnUpdate: true,
                role: this.props.role,
                enableKeyRotation: this.props.enableKeyRotation,
                artifactBucket: this.props.artifactBucket,
            });
        }
        const graphFromBp = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: this.selfMutationEnabled,
            singlePublisherPerAssetType: this.singlePublisherPerAssetType,
            prepareStep: this.useChangeSets,
        });
        this._cloudAssemblyFileSet = graphFromBp.cloudAssemblyFileSet;
        this.pipelineStagesAndActionsFromGraph(graphFromBp);
        // Write a dotfile for the pipeline layout
        const dotFile = `${core_1.Names.uniqueId(this)}.dot`;
        fs.writeFileSync(path.join(this.myCxAsmRoot, dotFile), graphFromBp.graph.renderDot().replace(/input\.dot/, dotFile), { encoding: 'utf-8' });
    }
    get myCxAsmRoot() {
        if (!this._myCxAsmRoot) {
            throw new Error('Can\'t read \'myCxAsmRoot\' if build deployment not called yet');
        }
        return this._myCxAsmRoot;
    }
    /**
     * Scope for Assets-related resources.
     *
     * Purely exists for construct tree backwards compatibility with legacy pipelines
     */
    get assetsScope() {
        return (0, construct_internals_1.obtainScope)(this, 'Assets');
    }
    pipelineStagesAndActionsFromGraph(structure) {
        // Translate graph into Pipeline Stages and Actions
        let beforeSelfMutation = this.selfMutationEnabled;
        for (const stageNode of (0, javascript_1.flatten)(structure.graph.sortedChildren())) {
            if (!(0, helpers_internal_1.isGraph)(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            // Group our ordered tranches into blocks of 50.
            // We can map these onto stages without exceeding the capacity of a Stage.
            const chunks = chunkTranches(50, stageNode.sortedLeaves());
            const actionsOverflowStage = chunks.length > 1;
            for (const [i, tranches] of (0, javascript_1.enumerate)(chunks)) {
                const stageName = actionsOverflowStage ? `${stageNode.id}.${i + 1}` : stageNode.id;
                const pipelineStage = this.pipeline.addStage({ stageName });
                const sharedParent = new helpers_internal_1.GraphNodeCollection((0, javascript_1.flatten)(tranches)).commonAncestor();
                let runOrder = 1;
                for (const tranche of tranches) {
                    const runOrdersConsumed = [0];
                    for (const node of tranche) {
                        const factory = this.actionFromNode(node);
                        const nodeType = this.nodeTypeFromNode(node);
                        const name = (0, identifiers_1.actionName)(node, sharedParent);
                        const variablesNamespace = node.data?.type === 'step'
                            ? (0, outputs_1.namespaceStepOutputs)(node.data.step, pipelineStage, name)
                            : undefined;
                        const result = factory.produceAction(pipelineStage, {
                            actionName: name,
                            runOrder,
                            artifacts: this.artifacts,
                            scope: (0, construct_internals_1.obtainScope)(this.pipeline, stageName),
                            fallbackArtifact: this._fallbackArtifact,
                            pipeline: this,
                            // If this step happens to produce a CodeBuild job, set the default options
                            codeBuildDefaults: nodeType ? this.codeBuildDefaultsFor(nodeType) : undefined,
                            beforeSelfMutation,
                            variablesNamespace,
                            stackOutputsMap: this.stackOutputs,
                        });
                        if (node.data?.type === 'self-update') {
                            beforeSelfMutation = false;
                        }
                        this.postProcessNode(node, result);
                        runOrdersConsumed.push(result.runOrdersConsumed);
                    }
                    runOrder += Math.max(...runOrdersConsumed);
                }
            }
        }
    }
    /**
     * Do additional things after the action got added to the pipeline
     *
     * Some minor state manipulation of CodeBuild projects and pipeline
     * artifacts.
     */
    postProcessNode(node, result) {
        const nodeType = this.nodeTypeFromNode(node);
        if (result.project) {
            const dockerUsage = dockerUsageFromCodeBuild(nodeType ?? CodeBuildProjectType.STEP);
            if (dockerUsage) {
                for (const c of this.dockerCredentials) {
                    c.grantRead(result.project, dockerUsage);
                }
            }
            if (nodeType === CodeBuildProjectType.SYNTH) {
                this._synthProject = result.project;
            }
            if (nodeType === CodeBuildProjectType.SELF_MUTATE) {
                this._selfMutationProject = result.project;
            }
        }
        if (node.data?.type === 'step' && node.data.step.primaryOutput?.primaryOutput && !this._fallbackArtifact) {
            this._fallbackArtifact = this.artifacts.toCodePipeline(node.data.step.primaryOutput?.primaryOutput);
        }
    }
    /**
     * Make an action from the given node and/or step
     */
    actionFromNode(node) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`actionFromNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                return this.selfMutateAction();
            case 'publish-assets':
                return this.publishAssetsAction(node, node.data.assets);
            case 'prepare':
                return this.createChangeSetAction(node.data.stack);
            case 'execute':
                return node.data.withoutChangeSet
                    ? this.executeDeploymentAction(node.data.stack, node.data.captureOutputs)
                    : this.executeChangeSetAction(node.data.stack, node.data.captureOutputs);
            case 'step':
                return this.actionFromStep(node, node.data.step);
            default:
                throw new Error(`CodePipeline does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    /**
     * Take a Step and turn it into a CodePipeline Action
     *
     * There are only 3 types of Steps we need to support:
     *
     * - Shell (generic)
     * - ManualApproval (generic)
     * - CodePipelineActionFactory (CodePipeline-specific)
     *
     * The rest is expressed in terms of these 3, or in terms of graph nodes
     * which are handled elsewhere.
     */
    actionFromStep(node, step) {
        const nodeType = this.nodeTypeFromNode(node);
        // CodePipeline-specific steps first -- this includes Sources
        if (isCodePipelineActionFactory(step)) {
            return step;
        }
        // Now built-in steps
        if (step instanceof blueprint_1.ShellStep || step instanceof codebuild_step_1.CodeBuildStep) {
            // The 'CdkBuildProject' will be the construct ID of the CodeBuild project, necessary for backwards compat
            let constructId = nodeType === CodeBuildProjectType.SYNTH
                ? 'CdkBuildProject'
                : step.id;
            return step instanceof codebuild_step_1.CodeBuildStep
                ? codebuild_factory_1.CodeBuildFactory.fromCodeBuildStep(constructId, step)
                : codebuild_factory_1.CodeBuildFactory.fromShellStep(constructId, step);
        }
        if (step instanceof blueprint_1.ManualApprovalStep) {
            return {
                produceAction: (stage, options) => {
                    stage.addAction(new cpa.ManualApprovalAction({
                        actionName: options.actionName,
                        runOrder: options.runOrder,
                        additionalInformation: step.comment,
                    }));
                    return { runOrdersConsumed: 1 };
                },
            };
        }
        throw new Error(`Deployment step '${step}' is not supported for CodePipeline-backed pipelines`);
    }
    createChangeSetAction(stack) {
        const changeSetName = 'PipelineChange';
        const templateArtifact = this.artifacts.toCodePipeline(this._cloudAssemblyFileSet);
        const templateConfigurationPath = this.writeTemplateConfiguration(stack);
        const region = stack.region !== core_1.Stack.of(this).region ? stack.region : undefined;
        const account = stack.account !== core_1.Stack.of(this).account ? stack.account : undefined;
        const relativeTemplatePath = path.relative(this.myCxAsmRoot, stack.absoluteTemplatePath);
        return {
            produceAction: (stage, options) => {
                stage.addAction(new cpa.CloudFormationCreateReplaceChangeSetAction({
                    actionName: options.actionName,
                    runOrder: options.runOrder,
                    changeSetName,
                    stackName: stack.stackName,
                    templatePath: templateArtifact.atPath((0, fs_1.toPosixPath)(relativeTemplatePath)),
                    adminPermissions: true,
                    role: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.assumeRoleArn),
                    deploymentRole: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.executionRoleArn),
                    region: region,
                    templateConfiguration: templateConfigurationPath
                        ? templateArtifact.atPath((0, fs_1.toPosixPath)(templateConfigurationPath))
                        : undefined,
                    cfnCapabilities: [core_1.CfnCapabilities.NAMED_IAM, core_1.CfnCapabilities.AUTO_EXPAND],
                }));
                return { runOrdersConsumed: 1 };
            },
        };
    }
    executeChangeSetAction(stack, captureOutputs) {
        const changeSetName = 'PipelineChange';
        const region = stack.region !== core_1.Stack.of(this).region ? stack.region : undefined;
        const account = stack.account !== core_1.Stack.of(this).account ? stack.account : undefined;
        return {
            produceAction: (stage, options) => {
                stage.addAction(new cpa.CloudFormationExecuteChangeSetAction({
                    actionName: options.actionName,
                    runOrder: options.runOrder,
                    changeSetName,
                    stackName: stack.stackName,
                    role: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.assumeRoleArn),
                    region: region,
                    variablesNamespace: captureOutputs ? (0, identifiers_1.stackVariableNamespace)(stack) : undefined,
                }));
                return { runOrdersConsumed: 1 };
            },
        };
    }
    executeDeploymentAction(stack, captureOutputs) {
        const templateArtifact = this.artifacts.toCodePipeline(this._cloudAssemblyFileSet);
        const templateConfigurationPath = this.writeTemplateConfiguration(stack);
        const region = stack.region !== core_1.Stack.of(this).region ? stack.region : undefined;
        const account = stack.account !== core_1.Stack.of(this).account ? stack.account : undefined;
        const relativeTemplatePath = path.relative(this.myCxAsmRoot, stack.absoluteTemplatePath);
        return {
            produceAction: (stage, options) => {
                stage.addAction(new cpa.CloudFormationCreateUpdateStackAction({
                    actionName: options.actionName,
                    runOrder: options.runOrder,
                    stackName: stack.stackName,
                    templatePath: templateArtifact.atPath((0, fs_1.toPosixPath)(relativeTemplatePath)),
                    adminPermissions: true,
                    role: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.assumeRoleArn),
                    deploymentRole: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.executionRoleArn),
                    region: region,
                    templateConfiguration: templateConfigurationPath
                        ? templateArtifact.atPath((0, fs_1.toPosixPath)(templateConfigurationPath))
                        : undefined,
                    cfnCapabilities: [core_1.CfnCapabilities.NAMED_IAM, core_1.CfnCapabilities.AUTO_EXPAND],
                    variablesNamespace: captureOutputs ? (0, identifiers_1.stackVariableNamespace)(stack) : undefined,
                }));
                return { runOrdersConsumed: 1 };
            },
        };
    }
    selfMutateAction() {
        const installSuffix = this.cliVersion ? `@${this.cliVersion}` : '';
        const pipelineStack = core_1.Stack.of(this.pipeline);
        const pipelineStackIdentifier = pipelineStack.node.path ?? pipelineStack.stackName;
        const step = new codebuild_step_1.CodeBuildStep('SelfMutate', {
            projectName: (0, javascript_1.maybeSuffix)(this.props.pipelineName, '-selfupdate'),
            input: this._cloudAssemblyFileSet,
            installCommands: [
                `npm install -g aws-cdk${installSuffix}`,
            ],
            commands: [
                `cdk -a ${(0, fs_1.toPosixPath)((0, construct_internals_1.embeddedAsmPath)(this.pipeline))} deploy ${pipelineStackIdentifier} --require-approval=never --verbose`,
            ],
            rolePolicyStatements: [
                // allow the self-mutating project permissions to assume the bootstrap Action role
                new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    resources: [`arn:*:iam::${core_1.Stack.of(this.pipeline).account}:role/*`],
                    conditions: {
                        'ForAnyValue:StringEquals': {
                            'iam:ResourceTag/aws-cdk:bootstrap-role': ['image-publishing', 'file-publishing', 'deploy'],
                        },
                    },
                }),
                new iam.PolicyStatement({
                    actions: ['cloudformation:DescribeStacks'],
                    resources: ['*'], // this is needed to check the status of the bootstrap stack when doing `cdk deploy`
                }),
                // S3 checks for the presence of the ListBucket permission
                new iam.PolicyStatement({
                    actions: ['s3:ListBucket'],
                    resources: ['*'],
                }),
            ],
        });
        // Different on purpose -- id needed for backwards compatible LogicalID
        return codebuild_factory_1.CodeBuildFactory.fromCodeBuildStep('SelfMutation', step, {
            additionalConstructLevel: false,
            scope: (0, construct_internals_1.obtainScope)(this, 'UpdatePipeline'),
        });
    }
    publishAssetsAction(node, assets) {
        const installSuffix = this.cliVersion ? `@${this.cliVersion}` : '';
        const commands = assets.map(asset => {
            const relativeAssetManifestPath = path.relative(this.myCxAsmRoot, asset.assetManifestPath);
            return `cdk-assets --path "${(0, fs_1.toPosixPath)(relativeAssetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        });
        const assetType = assets[0].assetType;
        if (assets.some(a => a.assetType !== assetType)) {
            throw new Error('All assets in a single publishing step must be of the same type');
        }
        const role = this.obtainAssetCodeBuildRole(assets[0].assetType);
        for (const roleArn of assets.flatMap(a => a.assetPublishingRoleArn ? [a.assetPublishingRoleArn] : [])) {
            // The ARNs include raw AWS pseudo parameters (e.g., ${AWS::Partition}), which need to be substituted.
            role.addAssumeRole(this.cachedFnSub.fnSub(roleArn));
        }
        ;
        // The base commands that need to be run
        const script = new codebuild_step_1.CodeBuildStep(node.id, {
            commands,
            installCommands: [
                `npm install -g cdk-assets${installSuffix}`,
            ],
            input: this._cloudAssemblyFileSet,
            buildEnvironment: {
                privileged: (assets.some(asset => asset.assetType === blueprint_1.AssetType.DOCKER_IMAGE) ||
                    this.props.codeBuildDefaults?.buildEnvironment?.privileged),
            },
            role,
        });
        // Customizations that are not accessible to regular users
        return codebuild_factory_1.CodeBuildFactory.fromCodeBuildStep(node.id, script, {
            additionalConstructLevel: false,
            // If we use a single publisher, pass buildspec via file otherwise it'll
            // grow too big.
            passBuildSpecViaCloudAssembly: this.singlePublisherPerAssetType,
            scope: this.assetsScope,
        });
    }
    nodeTypeFromNode(node) {
        if (node.data?.type === 'step') {
            return !!node.data?.isBuildStep ? CodeBuildProjectType.SYNTH : CodeBuildProjectType.STEP;
        }
        if (node.data?.type === 'publish-assets') {
            return CodeBuildProjectType.ASSETS;
        }
        if (node.data?.type === 'self-update') {
            return CodeBuildProjectType.SELF_MUTATE;
        }
        return undefined;
    }
    codeBuildDefaultsFor(nodeType) {
        const defaultOptions = {
            buildEnvironment: {
                buildImage: default_codebuild_image_1.CDKP_DEFAULT_CODEBUILD_IMAGE,
                computeType: cb.ComputeType.SMALL,
            },
        };
        const typeBasedCustomizations = {
            [CodeBuildProjectType.SYNTH]: this.props.dockerEnabledForSynth
                ? (0, codebuild_factory_1.mergeCodeBuildOptions)(this.props.synthCodeBuildDefaults, { buildEnvironment: { privileged: true } })
                : this.props.synthCodeBuildDefaults,
            [CodeBuildProjectType.ASSETS]: this.props.assetPublishingCodeBuildDefaults,
            [CodeBuildProjectType.SELF_MUTATE]: this.props.dockerEnabledForSelfMutation
                ? (0, codebuild_factory_1.mergeCodeBuildOptions)(this.props.selfMutationCodeBuildDefaults, { buildEnvironment: { privileged: true } })
                : this.props.selfMutationCodeBuildDefaults,
            [CodeBuildProjectType.STEP]: {},
        };
        const dockerUsage = dockerUsageFromCodeBuild(nodeType);
        const dockerCommands = dockerUsage !== undefined
            ? (0, docker_credentials_1.dockerCredentialsInstallCommands)(dockerUsage, this.dockerCredentials, 'both')
            : [];
        const typeBasedDockerCommands = dockerCommands.length > 0 ? {
            partialBuildSpec: cb.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: dockerCommands,
                    },
                },
            }),
        } : {};
        return (0, codebuild_factory_1.mergeCodeBuildOptions)(defaultOptions, this.props.codeBuildDefaults, typeBasedCustomizations[nodeType], typeBasedDockerCommands);
    }
    roleFromPlaceholderArn(scope, region, account, arn) {
        if (!arn) {
            return undefined;
        }
        // Use placeholder arn as construct ID.
        const id = arn;
        // https://github.com/aws/aws-cdk/issues/7255
        let existingRole = scope.node.tryFindChild(`ImmutableRole${id}`);
        if (existingRole) {
            return existingRole;
        }
        // For when #7255 is fixed.
        existingRole = scope.node.tryFindChild(id);
        if (existingRole) {
            return existingRole;
        }
        const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
            region: region ?? core_1.Aws.REGION,
            accountId: account ?? core_1.Aws.ACCOUNT_ID,
            partition: core_1.Aws.PARTITION,
        });
        return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false, addGrantsToResources: true });
    }
    /**
     * Non-template config files for CodePipeline actions
     *
     * Currently only supports tags.
     */
    writeTemplateConfiguration(stack) {
        if (Object.keys(stack.tags).length === 0) {
            return undefined;
        }
        const absConfigPath = `${stack.absoluteTemplatePath}.config.json`;
        const relativeConfigPath = path.relative(this.myCxAsmRoot, absConfigPath);
        // Write the template configuration file (for parameters into CreateChangeSet call that
        // cannot be configured any other way). They must come from a file, and there's unfortunately
        // no better hook to write this file (`construct.onSynthesize()` would have been the prime candidate
        // but that is being deprecated--and DeployCdkStackAction isn't even a construct).
        (0, template_configuration_1.writeTemplateConfiguration)(absConfigPath, {
            Tags: (0, javascript_1.noUndefined)(stack.tags),
        });
        return relativeConfigPath;
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    obtainAssetCodeBuildRole(assetType) {
        const existing = this.assetCodeBuildRoles.get(assetType);
        if (existing) {
            return existing;
        }
        const stack = core_1.Stack.of(this);
        const rolePrefix = assetType === blueprint_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new asset_singleton_role_1.AssetSingletonRole(this.assetsScope, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(stack.account)),
        });
        // Grant pull access for any ECR registries and secrets that exist
        if (assetType === blueprint_1.AssetType.DOCKER_IMAGE) {
            this.dockerCredentials.forEach(reg => reg.grantRead(assetRole, docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING));
        }
        this.assetCodeBuildRoles.set(assetType, assetRole);
        return assetRole;
    }
}
_a = JSII_RTTI_SYMBOL_1;
CodePipeline[_a] = { fqn: "aws-cdk-lib.pipelines.CodePipeline", version: "2.74.0" };
exports.CodePipeline = CodePipeline;
function dockerUsageFromCodeBuild(cbt) {
    switch (cbt) {
        case CodeBuildProjectType.ASSETS: return docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING;
        case CodeBuildProjectType.SELF_MUTATE: return docker_credentials_1.DockerCredentialUsage.SELF_UPDATE;
        case CodeBuildProjectType.SYNTH: return docker_credentials_1.DockerCredentialUsage.SYNTH;
        case CodeBuildProjectType.STEP: return undefined;
    }
}
var CodeBuildProjectType;
(function (CodeBuildProjectType) {
    CodeBuildProjectType["SYNTH"] = "SYNTH";
    CodeBuildProjectType["ASSETS"] = "ASSETS";
    CodeBuildProjectType["SELF_MUTATE"] = "SELF_MUTATE";
    CodeBuildProjectType["STEP"] = "STEP";
})(CodeBuildProjectType || (CodeBuildProjectType = {}));
/**
 * Take a set of tranches and split them up into groups so
 * that no set of tranches has more than n items total
 */
function chunkTranches(n, xss) {
    const ret = [];
    while (xss.length > 0) {
        const tranches = [];
        let count = 0;
        while (xss.length > 0) {
            const xs = xss[0];
            const spaceRemaining = n - count;
            if (xs.length <= spaceRemaining) {
                tranches.push(xs);
                count += xs.length;
                xss.shift();
            }
            else {
                tranches.push(xs.splice(0, spaceRemaining));
                count = n;
                break;
            }
        }
        ret.push(tranches);
    }
    return ret;
}
function isCodePipelineActionFactory(x) {
    return !!x.produceAction;
}
//# sourceMappingURL=data:application/json;base64,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