"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeCodeBuildOptions = exports.CodeBuildFactory = void 0;
const fs = require("fs");
const path = require("path");
const codebuild = require("../../../../aws-codebuild");
const codepipeline_actions = require("../../../../aws-codepipeline-actions");
const ec2 = require("../../../../aws-ec2");
const iam = require("../../../../aws-iam");
const core_1 = require("../../../../core");
const constructs_1 = require("constructs");
const buildspecs_1 = require("./buildspecs");
const step_output_1 = require("../../helpers-internal/step-output");
const construct_internals_1 = require("../../private/construct-internals");
const identifiers_1 = require("../../private/identifiers");
const javascript_1 = require("../../private/javascript");
/**
 * Produce a CodeBuild project from a ShellStep and some CodeBuild-specific customizations
 *
 * The functionality here is shared between the `CodePipeline` translating a `ShellStep` into
 * a CodeBuild project, as well as the `CodeBuildStep` straight up.
 */
class CodeBuildFactory {
    // eslint-disable-next-line max-len
    static fromShellStep(constructId, shellStep, additional) {
        return new CodeBuildFactory(constructId, {
            commands: shellStep.commands,
            env: shellStep.env,
            envFromCfnOutputs: shellStep.envFromCfnOutputs,
            inputs: shellStep.inputs,
            outputs: shellStep.outputs,
            stepId: shellStep.id,
            installCommands: shellStep.installCommands,
            producedStepOutputs: step_output_1.StepOutput.producedStepOutputs(shellStep),
            ...additional,
        });
    }
    static fromCodeBuildStep(constructId, step, additional) {
        const factory = CodeBuildFactory.fromShellStep(constructId, step, {
            projectName: step.projectName,
            role: step.role,
            actionRole: step.actionRole,
            ...additional,
            projectOptions: mergeCodeBuildOptions(additional?.projectOptions, {
                buildEnvironment: step.buildEnvironment,
                rolePolicy: step.rolePolicyStatements,
                securityGroups: step.securityGroups,
                partialBuildSpec: step.partialBuildSpec,
                vpc: step.vpc,
                subnetSelection: step.subnetSelection,
                cache: step.cache,
                timeout: step.timeout,
                fileSystemLocations: step.fileSystemLocations,
            }),
        });
        return {
            produceAction: (stage, options) => {
                const result = factory.produceAction(stage, options);
                if (result.project) {
                    step._setProject(result.project);
                }
                return result;
            },
        };
    }
    constructor(constructId, props) {
        this.constructId = constructId;
        this.props = props;
        this.stepId = props.stepId ?? constructId;
    }
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after produce() has been called');
        }
        return this._project;
    }
    produceAction(stage, options) {
        const projectOptions = mergeCodeBuildOptions(options.codeBuildDefaults, this.props.projectOptions);
        if ((!projectOptions.buildEnvironment?.privileged || projectOptions.vpc === undefined) &&
            (projectOptions.fileSystemLocations !== undefined && projectOptions.fileSystemLocations.length != 0)) {
            throw new Error('Setting fileSystemLocations requires a vpc to be set and privileged to be set to true.');
        }
        const inputs = this.props.inputs ?? [];
        const outputs = this.props.outputs ?? [];
        const mainInput = inputs.find(x => x.directory === '.');
        const extraInputs = inputs.filter(x => x.directory !== '.');
        const inputArtifact = mainInput
            ? options.artifacts.toCodePipeline(mainInput.fileSet)
            : options.fallbackArtifact;
        const extraInputArtifacts = extraInputs.map(x => options.artifacts.toCodePipeline(x.fileSet));
        const outputArtifacts = outputs.map(x => options.artifacts.toCodePipeline(x.fileSet));
        if (!inputArtifact) {
            // This should actually never happen because CodeBuild projects shouldn't be added before the
            // Source, which always produces at least an artifact.
            throw new Error(`CodeBuild action '${this.stepId}' requires an input (and the pipeline doesn't have a Source to fall back to). Add an input or a pipeline source.`);
        }
        const installCommands = [
            ...generateInputArtifactLinkCommands(options.artifacts, extraInputs),
            ...this.props.installCommands ?? [],
        ];
        const buildSpecHere = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                install: (installCommands.length ?? 0) > 0 ? { commands: installCommands } : undefined,
                build: this.props.commands.length > 0 ? { commands: this.props.commands } : undefined,
            },
            artifacts: (0, javascript_1.noEmptyObject)(renderArtifactsBuildSpec(options.artifacts, this.props.outputs ?? [])),
        });
        // Partition environment variables into environment variables that can go on the project
        // and environment variables that MUST go in the pipeline (those that reference CodePipeline variables)
        const env = (0, javascript_1.noUndefined)(this.props.env ?? {});
        const [actionEnvs, projectEnvs] = (0, javascript_1.partition)(Object.entries(env ?? {}), ([, v]) => containsPipelineVariable(v));
        const environment = mergeBuildEnvironments(projectOptions?.buildEnvironment ?? {}, {
            environmentVariables: (0, javascript_1.noEmptyObject)((0, javascript_1.mapValues)((0, javascript_1.mkdict)(projectEnvs), value => ({ value }))),
        });
        const fullBuildSpec = projectOptions?.partialBuildSpec
            ? codebuild.mergeBuildSpecs(projectOptions.partialBuildSpec, buildSpecHere)
            : buildSpecHere;
        const osFromEnvironment = environment.buildImage && environment.buildImage instanceof codebuild.WindowsBuildImage
            ? ec2.OperatingSystemType.WINDOWS
            : ec2.OperatingSystemType.LINUX;
        const actualBuildSpec = filterBuildSpecCommands(fullBuildSpec, osFromEnvironment);
        const scope = this.props.scope ?? options.scope;
        let projectBuildSpec;
        if (this.props.passBuildSpecViaCloudAssembly) {
            // Write to disk and replace with a reference
            const relativeSpecFile = `buildspec-${constructs_1.Node.of(scope).addr}-${this.constructId}.yaml`;
            const absSpecFile = path.join((0, construct_internals_1.cloudAssemblyBuildSpecDir)(scope), relativeSpecFile);
            // This should resolve to a pure JSON string. If it resolves to an object, it's a CFN
            // expression, and we can't support that yet. Maybe someday if we think really hard about it.
            const fileContents = core_1.Stack.of(scope).resolve(actualBuildSpec.toBuildSpec());
            if (typeof fileContents !== 'string') {
                throw new Error(`This BuildSpec contains CloudFormation references and is supported by publishInParallel=false: ${JSON.stringify(fileContents, undefined, 2)}`);
            }
            fs.writeFileSync(absSpecFile, fileContents, { encoding: 'utf-8' });
            projectBuildSpec = codebuild.BuildSpec.fromSourceFilename(relativeSpecFile);
        }
        else {
            projectBuildSpec = actualBuildSpec;
        }
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = (0, identifiers_1.hash)(core_1.Stack.of(scope).resolve({
            environment: serializeBuildEnvironment(environment),
            buildSpecString: actualBuildSpec.toBuildSpec(),
        }));
        const actionName = options.actionName ?? this.stepId;
        let projectScope = scope;
        if (this.props.additionalConstructLevel ?? true) {
            projectScope = (0, construct_internals_1.obtainScope)(scope, actionName);
        }
        const safePipelineName = core_1.Token.isUnresolved(options.pipeline.pipeline.pipelineName)
            ? `${core_1.Stack.of(options.pipeline).stackName}/${constructs_1.Node.of(options.pipeline.pipeline).id}`
            : options.pipeline.pipeline.pipelineName;
        const project = new codebuild.PipelineProject(projectScope, this.constructId, {
            projectName: this.props.projectName,
            description: `Pipeline step ${safePipelineName}/${stage.stageName}/${actionName}`.substring(0, 255),
            environment,
            vpc: projectOptions.vpc,
            subnetSelection: projectOptions.subnetSelection,
            securityGroups: projectOptions.securityGroups,
            cache: projectOptions.cache,
            buildSpec: projectBuildSpec,
            role: this.props.role,
            timeout: projectOptions.timeout,
            fileSystemLocations: projectOptions.fileSystemLocations,
        });
        if (this.props.additionalDependable) {
            project.node.addDependency(this.props.additionalDependable);
        }
        if (projectOptions.rolePolicy !== undefined) {
            projectOptions.rolePolicy.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        const stackOutputEnv = (0, javascript_1.mapValues)(this.props.envFromCfnOutputs ?? {}, outputRef => options.stackOutputsMap.toCodePipeline(outputRef));
        const configHashEnv = options.beforeSelfMutation
            ? { _PROJECT_CONFIG_HASH: projectConfigHash }
            : {};
        // Start all CodeBuild projects from a single (shared) Action Role, so that we don't have to generate an Action Role for each
        // individual CodeBuild Project and blow out the pipeline policy size (and potentially # of resources in the stack).
        const actionRoleCid = 'CodeBuildActionRole';
        const actionRole = this.props.actionRole
            ?? options.pipeline.node.tryFindChild(actionRoleCid)
            ?? new iam.Role(options.pipeline, actionRoleCid, {
                assumedBy: new iam.PrincipalWithConditions(new iam.AccountRootPrincipal(), {
                    Bool: { 'aws:ViaAWSService': iam.ServicePrincipal.servicePrincipalName('codepipeline.amazonaws.com') },
                }),
            });
        stage.addAction(new codepipeline_actions.CodeBuildAction({
            actionName: actionName,
            input: inputArtifact,
            extraInputs: extraInputArtifacts,
            outputs: outputArtifacts,
            project,
            runOrder: options.runOrder,
            variablesNamespace: options.variablesNamespace,
            role: actionRole,
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: (0, javascript_1.noEmptyObject)(cbEnv({
                ...(0, javascript_1.mkdict)(actionEnvs),
                ...configHashEnv,
                ...stackOutputEnv,
            })),
        }));
        this._project = project;
        return { runOrdersConsumed: 1, project };
    }
}
exports.CodeBuildFactory = CodeBuildFactory;
/**
 * Generate commands to move additional input artifacts into the right place
 */
function generateInputArtifactLinkCommands(artifacts, inputs) {
    return inputs.map(input => {
        const fragments = [];
        fragments.push(`[ ! -d "${input.directory}" ] || { echo 'additionalInputs: "${input.directory}" must not exist yet. If you want to merge multiple artifacts, use a "cp" command.'; exit 1; }`);
        const parentDirectory = path.dirname(input.directory);
        if (!['.', '..'].includes(parentDirectory)) {
            fragments.push(`mkdir -p -- "${parentDirectory}"`);
        }
        const artifact = artifacts.toCodePipeline(input.fileSet);
        fragments.push(`ln -s -- "$CODEBUILD_SRC_DIR_${artifact.artifactName}" "${input.directory}"`);
        return fragments.join(' && ');
    });
}
function renderArtifactsBuildSpec(artifactMap, outputs) {
    // save the generated files in the output artifact
    // This part of the buildspec has to look completely different depending on whether we're
    // using secondary artifacts or not.
    if (outputs.length === 0) {
        return {};
    }
    if (outputs.length === 1) {
        return {
            'base-directory': outputs[0].directory,
            'files': '**/*',
        };
    }
    const secondary = {};
    for (const output of outputs) {
        const art = artifactMap.toCodePipeline(output.fileSet);
        if (!art.artifactName) {
            throw new Error('You must give the output artifact a name');
        }
        secondary[art.artifactName] = {
            'base-directory': output.directory,
            'files': '**/*',
        };
    }
    return { 'secondary-artifacts': secondary };
}
function mergeCodeBuildOptions(...opts) {
    const xs = [{}, ...opts.filter(isDefined)];
    while (xs.length > 1) {
        const [a, b] = xs.splice(xs.length - 2, 2);
        xs.push(merge2(a, b));
    }
    return xs[0];
    function merge2(a, b) {
        return {
            buildEnvironment: mergeBuildEnvironments(a.buildEnvironment, b.buildEnvironment),
            rolePolicy: definedArray([...a.rolePolicy ?? [], ...b.rolePolicy ?? []]),
            securityGroups: definedArray([...a.securityGroups ?? [], ...b.securityGroups ?? []]),
            partialBuildSpec: (0, buildspecs_1.mergeBuildSpecs)(a.partialBuildSpec, b.partialBuildSpec),
            vpc: b.vpc ?? a.vpc,
            subnetSelection: b.subnetSelection ?? a.subnetSelection,
            timeout: b.timeout ?? a.timeout,
            cache: b.cache ?? a.cache,
            fileSystemLocations: definedArray([...a.fileSystemLocations ?? [], ...b.fileSystemLocations ?? []]),
        };
    }
}
exports.mergeCodeBuildOptions = mergeCodeBuildOptions;
function mergeBuildEnvironments(a, b) {
    if (!a || !b) {
        return a ?? b;
    }
    return {
        buildImage: b.buildImage ?? a.buildImage,
        computeType: b.computeType ?? a.computeType,
        environmentVariables: {
            ...a.environmentVariables,
            ...b.environmentVariables,
        },
        privileged: b.privileged ?? a.privileged,
    };
}
function isDefined(x) {
    return x !== undefined;
}
/**
 * Serialize a build environment to data (get rid of constructs & objects), so we can JSON.stringify it
 */
function serializeBuildEnvironment(env) {
    return {
        privileged: env.privileged,
        environmentVariables: env.environmentVariables,
        type: env.buildImage?.type,
        imageId: env.buildImage?.imageId,
        computeType: env.computeType,
        imagePullPrincipalType: env.buildImage?.imagePullPrincipalType,
        secretsManagerArn: env.buildImage?.secretsManagerCredentials?.secretArn,
    };
}
/**
 * Whether the given string contains a reference to a CodePipeline variable
 */
function containsPipelineVariable(s) {
    return !!s.match(/#\{[^}]+\}/) || step_output_1.StepOutput.findAll(s).length > 0;
}
/**
 * Turn a collection into a collection of CodePipeline environment variables
 */
function cbEnv(xs) {
    return (0, javascript_1.mkdict)(Object.entries(xs)
        .filter(([, v]) => v !== undefined)
        .map(([k, v]) => [k, { value: v }]));
}
function definedArray(xs) {
    return xs.length > 0 ? xs : undefined;
}
/**
 * If lines in the buildspec start with '!WINDOWS!' or '!LINUX!', only render them on that platform.
 *
 * Very private protocol for now, but may come in handy in other libraries as well.
 */
function filterBuildSpecCommands(buildSpec, osType) {
    if (!buildSpec.isImmediate) {
        return buildSpec;
    }
    const spec = buildSpec.spec;
    const winTag = '!WINDOWS!';
    const linuxTag = '!LINUX!';
    const expectedTag = osType === ec2.OperatingSystemType.WINDOWS ? winTag : linuxTag;
    return codebuild.BuildSpec.fromObject(recurse(spec));
    function recurse(x) {
        if (Array.isArray(x)) {
            const ret = [];
            for (const el of x) {
                const [tag, payload] = extractTag(el);
                if (tag === undefined || tag === expectedTag) {
                    ret.push(payload);
                }
            }
            return ret;
        }
        if (x && typeof x === 'object') {
            return (0, javascript_1.mapValues)(x, recurse);
        }
        return x;
    }
    function extractTag(x) {
        if (typeof x !== 'string') {
            return [undefined, x];
        }
        for (const tag of [winTag, linuxTag]) {
            if (x.startsWith(tag)) {
                return [tag, x.slice(tag.length)];
            }
        }
        return [undefined, x];
    }
}
//# sourceMappingURL=data:application/json;base64,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