"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGraph = exports.GraphNodeCollection = exports.Graph = exports.DependencyBuilders = exports.DependencyBuilder = exports.GraphNode = void 0;
/**
 * A library for nested graphs
 */
const toposort_1 = require("./toposort");
const javascript_1 = require("../private/javascript");
class GraphNode {
    static of(id, data) {
        return new GraphNode(id, { data });
    }
    constructor(id, props = {}) {
        this.id = id;
        this.dependencies = [];
        this.data = props.data;
    }
    /**
     * A graph-wide unique identifier for this node. Rendered by joining the IDs
     * of all ancestors with hyphens.
     */
    get uniqueId() {
        return this.ancestorPath(this.root).map(x => x.id).join('-');
    }
    /**
     * The union of all dependencies of this node and the dependencies of all
     * parent graphs.
     */
    get allDeps() {
        const fromParent = this.parentGraph?.allDeps ?? [];
        return Array.from(new Set([...this.dependencies, ...fromParent]));
    }
    dependOn(...dependencies) {
        if (dependencies.includes(this)) {
            throw new Error(`Cannot add dependency on self: ${this}`);
        }
        this.dependencies.push(...dependencies.filter(javascript_1.isDefined));
    }
    ancestorPath(upTo) {
        let x = this;
        const ret = [x];
        while (x.parentGraph && x.parentGraph !== upTo) {
            x = x.parentGraph;
            ret.unshift(x);
        }
        return ret;
    }
    rootPath() {
        let x = this;
        const ret = [x];
        while (x.parentGraph) {
            x = x.parentGraph;
            ret.unshift(x);
        }
        return ret;
    }
    get root() {
        let x = this;
        while (x.parentGraph) {
            x = x.parentGraph;
        }
        return x;
    }
    get rootGraph() {
        const root = this.root;
        if (!(root instanceof Graph)) {
            throw new Error(`Expecting a graph as root, got: ${root}`);
        }
        return root;
    }
    get parentGraph() {
        return this._parentGraph;
    }
    /**
     * @internal
     */
    _setParentGraph(parentGraph) {
        if (this._parentGraph) {
            throw new Error('Node already has a parent');
        }
        this._parentGraph = parentGraph;
    }
    toString() {
        return `${this.constructor.name}(${this.id})`;
    }
}
exports.GraphNode = GraphNode;
/**
 * A dependency set that is constructed over time
 *
 * It doesn't matter in what order sources and targets for the dependency
 * relationship(s) get added. This class can serve as a synchronization
 * point if the order in which graph nodes get added to the graph is not
 * well-defined.
 *
 * You can think of a DependencyBuilder as a vertex that doesn't actually exist in the tree:
 *
 *     ┌────┐               ┌────┐
 *     │ P1 │◀─┐         ┌──│ S1 │
 *     └────┘  │   .─.   │  └────┘
 *             ├──( B )◀─┤
 *     ┌────┐  │   `─'   │  ┌────┐
 *     │ P2 │◀─┘         └──│ S2 │
 *     └────┘               └────┘
 *
 * Ultimately leads to: { S1 -> P1, S1 -> P2, S2 -> P1, S2 -> P2 }.
 */
class DependencyBuilder {
    constructor() {
        this._producers = [];
        this._consumers = [];
    }
    /**
     * Add a producer: make all nodes added by 'dependBy' depend on these
     */
    dependOn(...targets) {
        for (const target of targets) {
            for (const source of this._consumers) {
                source.dependOn(target);
            }
            this._producers.push(target);
        }
        return this;
    }
    /**
     * Add a consumer: make these nodes depend on all nodes added by 'dependOn'.
     */
    dependBy(...sources) {
        for (const source of sources) {
            for (const target of this._producers) {
                source.dependOn(target);
            }
            this._consumers.push(source);
        }
        return this;
    }
    /**
     * Whether there are any consumers (nodes added by 'dependBy') but no producers (nodes added by 'dependOn')
     */
    get hasUnsatisfiedConsumers() {
        return this._consumers.length > 0 && this._producers.length === 0;
    }
    get consumers() {
        return this._consumers;
    }
    consumersAsString() {
        return this.consumers.map(c => `${c}`).join(',');
    }
}
exports.DependencyBuilder = DependencyBuilder;
/**
 * A set of dependency builders identified by a given key.
 */
class DependencyBuilders {
    constructor() {
        this.builders = new Map();
    }
    for(key) {
        const b = this.builders.get(key);
        if (b) {
            return b;
        }
        const ret = new DependencyBuilder();
        this.builders.set(key, ret);
        return ret;
    }
    /**
     * @deprecated Use 'for'
     */
    get(key) {
        return this.for(key);
    }
    unsatisfiedBuilders() {
        const ret = new Array();
        for (const [k, builder] of this.builders.entries()) {
            if (builder.hasUnsatisfiedConsumers) {
                ret.push([k, builder]);
            }
        }
        return ret;
    }
}
exports.DependencyBuilders = DependencyBuilders;
class Graph extends GraphNode {
    static of(id, data, nodes) {
        return new Graph(id, { data, nodes });
    }
    constructor(name, props = {}) {
        super(name, props);
        this.children = new Map();
        if (props.nodes) {
            this.add(...props.nodes);
        }
    }
    get nodes() {
        return new Set(this.children.values());
    }
    tryGetChild(name) {
        return this.children.get(name);
    }
    contains(node) {
        return this.nodes.has(node);
    }
    add(...nodes) {
        for (const node of nodes) {
            node._setParentGraph(this);
            if (this.children.has(node.id)) {
                throw new Error(`Node with duplicate id: ${node.id}`);
            }
            this.children.set(node.id, node);
        }
    }
    absorb(other) {
        this.add(...other.nodes);
    }
    /**
     * Return topologically sorted tranches of nodes at this graph level
     */
    sortedChildren(fail = true) {
        // Project dependencies to current children
        const nodes = this.nodes;
        const projectedDependencies = projectDependencies(this.deepDependencies(), (node) => {
            while (!nodes.has(node) && node.parentGraph) {
                node = node.parentGraph;
            }
            return nodes.has(node) ? [node] : [];
        });
        return (0, toposort_1.topoSort)(nodes, projectedDependencies, fail);
    }
    /**
     * Return a topologically sorted list of non-Graph nodes in the entire subgraph
     */
    sortedLeaves() {
        // Project dependencies to leaf nodes
        const descendantsMap = new Map();
        findDescendants(this);
        function findDescendants(node) {
            const ret = [];
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    ret.push(...findDescendants(child));
                }
            }
            else {
                ret.push(node);
            }
            descendantsMap.set(node, ret);
            return ret;
        }
        const projectedDependencies = projectDependencies(this.deepDependencies(), (node) => descendantsMap.get(node) ?? []);
        return (0, toposort_1.topoSort)(new Set(projectedDependencies.keys()), projectedDependencies);
    }
    render() {
        const lines = new Array();
        recurse(this, '', true);
        return lines.join('\n');
        function recurse(x, indent, last) {
            const bullet = last ? '└─' : '├─';
            const follow = last ? '  ' : '│ ';
            lines.push(`${indent} ${bullet} ${x}${depString(x)}`);
            if (x instanceof Graph) {
                let i = 0;
                const sortedNodes = Array.prototype.concat.call([], ...x.sortedChildren(false));
                for (const child of sortedNodes) {
                    recurse(child, `${indent} ${follow} `, i++ == x.nodes.size - 1);
                }
            }
        }
        function depString(node) {
            if (node.dependencies.length > 0) {
                return ` -> ${Array.from(node.dependencies).join(', ')}`;
            }
            return '';
        }
    }
    renderDot() {
        const lines = new Array();
        lines.push('digraph G {');
        lines.push('  # Arrows represent an "unlocks" relationship (opposite of dependency). So chosen');
        lines.push('  # because the layout looks more natural that way.');
        lines.push('  # To represent subgraph dependencies, subgraphs are represented by BEGIN/END nodes.');
        lines.push('  # To render: `dot -Tsvg input.dot > graph.svg`, open in a browser.');
        lines.push('  node [shape="box"];');
        for (const child of this.nodes) {
            recurse(child);
        }
        lines.push('}');
        return lines.join('\n');
        function recurse(node) {
            let dependencySource;
            if (node instanceof Graph) {
                lines.push(`${graphBegin(node)} [shape="cds", style="filled", fillcolor="#b7deff"];`);
                lines.push(`${graphEnd(node)} [shape="cds", style="filled", fillcolor="#b7deff"];`);
                dependencySource = graphBegin(node);
            }
            else {
                dependencySource = nodeLabel(node);
                lines.push(`${nodeLabel(node)};`);
            }
            for (const dep of node.dependencies) {
                const dst = dep instanceof Graph ? graphEnd(dep) : nodeLabel(dep);
                lines.push(`${dst} -> ${dependencySource};`);
            }
            if (node instanceof Graph && node.nodes.size > 0) {
                for (const child of node.nodes) {
                    recurse(child);
                }
                // Add dependency arrows between the "subgraph begin" and the first rank of
                // the children, and the last rank of the children and "subgraph end" nodes.
                const sortedChildren = node.sortedChildren(false);
                for (const first of sortedChildren[0]) {
                    const src = first instanceof Graph ? graphBegin(first) : nodeLabel(first);
                    lines.push(`${graphBegin(node)} -> ${src};`);
                }
                for (const last of sortedChildren[sortedChildren.length - 1]) {
                    const dst = last instanceof Graph ? graphEnd(last) : nodeLabel(last);
                    lines.push(`${dst} -> ${graphEnd(node)};`);
                }
            }
        }
        function id(node) {
            return node.rootPath().slice(1).map(n => n.id).join('.');
        }
        function nodeLabel(node) {
            return `"${id(node)}"`;
        }
        function graphBegin(node) {
            return `"BEGIN ${id(node)}"`;
        }
        function graphEnd(node) {
            return `"END ${id(node)}"`;
        }
    }
    consoleLog(_indent = 0) {
        process.stdout.write(this.render() + '\n');
    }
    /**
     * Return the union of all dependencies of the descendants of this graph
     */
    deepDependencies() {
        const ret = new Map();
        for (const node of this.nodes) {
            recurse(node);
        }
        return ret;
        function recurse(node) {
            let deps = ret.get(node);
            if (!deps) {
                ret.set(node, deps = new Set());
            }
            for (let dep of node.dependencies) {
                deps.add(dep);
            }
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    recurse(child);
                }
            }
        }
    }
    /**
     * Return all non-Graph nodes
     */
    allLeaves() {
        const ret = [];
        recurse(this);
        return new GraphNodeCollection(ret);
        function recurse(node) {
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    recurse(child);
                }
            }
            else {
                ret.push(node);
            }
        }
    }
}
exports.Graph = Graph;
/**
 * A collection of graph nodes
 */
class GraphNodeCollection {
    constructor(nodes) {
        this.nodes = Array.from(nodes);
    }
    /**
     * Add one or more dependencies to all nodes in the collection
     */
    dependOn(...dependencies) {
        for (const node of this.nodes) {
            node.dependOn(...dependencies.filter(javascript_1.isDefined));
        }
    }
    /**
     * Return the topographically first node in the collection
     */
    first() {
        const nodes = new Set(this.nodes);
        const sorted = this.nodes[0].rootGraph.sortedLeaves();
        for (const tranche of sorted) {
            for (const node of tranche) {
                if (nodes.has(node)) {
                    return node;
                }
            }
        }
        throw new Error(`Could not calculate first node between ${this}`);
    }
    /**
    * Returns the graph node that's shared between these nodes
    */
    commonAncestor() {
        const paths = new Array();
        for (const x of this.nodes) {
            paths.push(x.rootPath());
        }
        if (paths.length === 0) {
            throw new Error('Cannot find common ancestor between an empty set of nodes');
        }
        if (paths.length === 1) {
            const path = paths[0];
            if (path.length < 2) {
                throw new Error(`Cannot find ancestor of node without ancestor: ${path[0]}`);
            }
            return path[path.length - 2];
        }
        const originalPaths = [...paths];
        // Remove the first element of every path as long as the 2nd elements are all
        // the same -- this leaves the shared element in first place.
        //
        //   A, B, C, 1, 2    }---> C
        //   A, B, C, 3       }
        while (paths.every(path => paths[0].length >= 2 && path.length >= 2 && path[1] === paths[0][1])) {
            for (const path of paths) {
                path.shift();
            }
        }
        // If any of the paths are left with 1 element, there's no shared parent.
        if (paths.some(path => path.length < 2)) {
            throw new Error(`Could not determine a shared parent between nodes: ${originalPaths.map(nodes => nodes.map(n => n.id).join('/'))}`);
        }
        return paths[0][0];
    }
    toString() {
        return this.nodes.map(n => `${n}`).join(', ');
    }
}
exports.GraphNodeCollection = GraphNodeCollection;
/**
 * Dependency map of nodes in this graph, taking into account dependencies between nodes in subgraphs
 *
 * Guaranteed to return an entry in the map for every node in the current graph.
 */
function projectDependencies(dependencies, project) {
    // Project keys
    for (const node of dependencies.keys()) {
        const projectedNodes = project(node);
        if (projectedNodes.length === 1 && projectedNodes[0] === node) {
            continue;
        } // Nothing to do, just for efficiency
        const deps = (0, javascript_1.extract)(dependencies, node);
        for (const projectedNode of projectedNodes) {
            (0, javascript_1.addAll)(dependencies.get(projectedNode), deps);
        }
    }
    // Project values. Ignore self-dependencies, they were just between nodes that were collapsed into the same node.
    for (const [node, deps] of dependencies.entries()) {
        const depset = new Set((0, javascript_1.flatMap)(deps, project));
        depset.delete(node);
        dependencies.set(node, depset);
    }
    return dependencies;
}
function isGraph(x) {
    return x instanceof Graph;
}
exports.isGraph = isGraph;
//# sourceMappingURL=data:application/json;base64,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