"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineGraph = void 0;
const graph_1 = require("./graph");
const pipeline_queries_1 = require("./pipeline-queries");
const blueprint_1 = require("../blueprint");
/**
 * Logic to turn the deployment blueprint into a graph
 *
 * This code makes all the decisions on how to lay out the CodePipeline
 */
class PipelineGraph {
    constructor(pipeline, props = {}) {
        this.pipeline = pipeline;
        this.graph = graph_1.Graph.of('', { type: 'group' });
        this.added = new Map();
        this.assetNodes = new Map();
        this.assetNodesByType = new Map();
        this.stackOutputDependencies = new graph_1.DependencyBuilders();
        /** Mapping steps to depbuilders, satisfied by the step itself  */
        this.nodeDependencies = new graph_1.DependencyBuilders();
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.publishTemplate = props.publishTemplate ?? false;
        this.prepareStep = props.prepareStep ?? true;
        this.singlePublisher = props.singlePublisherPerAssetType ?? false;
        this.queries = new pipeline_queries_1.PipelineQueries(pipeline);
        if (pipeline.synth instanceof blueprint_1.Step) {
            this.synthNode = this.addBuildStep(pipeline.synth);
            if (this.synthNode?.data?.type === 'step') {
                this.synthNode.data.isBuildStep = true;
            }
        }
        this.lastPreparationNode = this.synthNode;
        const cloudAssembly = pipeline.synth.primaryOutput?.primaryOutput;
        if (!cloudAssembly) {
            throw new Error(`The synth step must produce the cloud assembly artifact, but doesn't: ${pipeline.synth}`);
        }
        this.cloudAssemblyFileSet = cloudAssembly;
        if (props.selfMutation) {
            const stage = graph_1.Graph.of('UpdatePipeline', { type: 'group' });
            this.graph.add(stage);
            this.selfMutateNode = aGraphNode('SelfMutate', { type: 'self-update' });
            stage.add(this.selfMutateNode);
            this.selfMutateNode.dependOn(this.synthNode);
            this.lastPreparationNode = this.selfMutateNode;
        }
        const waves = pipeline.waves.map(w => this.addWave(w));
        // Make sure the waves deploy sequentially
        for (let i = 1; i < waves.length; i++) {
            waves[i].dependOn(waves[i - 1]);
        }
        this.addMissingDependencyNodes();
    }
    isSynthNode(node) {
        return this.synthNode === node;
    }
    addBuildStep(step) {
        return this.addStepNode(step, this.topLevelGraph('Build'));
    }
    addWave(wave) {
        // If the wave only has one Stage in it, don't add an additional Graph around it
        const retGraph = wave.stages.length === 1
            ? this.addStage(wave.stages[0])
            : graph_1.Graph.of(wave.id, { type: 'group' }, wave.stages.map(s => this.addStage(s)));
        this.addPrePost(wave.pre, wave.post, retGraph);
        retGraph.dependOn(this.lastPreparationNode);
        this.graph.add(retGraph);
        return retGraph;
    }
    addStage(stage) {
        const retGraph = graph_1.Graph.of(stage.stageName, { type: 'group' });
        const stackGraphs = new Map();
        for (const stack of stage.stacks) {
            const stackGraph = graph_1.Graph.of(this.simpleStackName(stack.stackName, stage.stageName), { type: 'stack-group', stack });
            const prepareNode = this.prepareStep ? aGraphNode('Prepare', { type: 'prepare', stack }) : undefined;
            const deployNode = aGraphNode('Deploy', {
                type: 'execute',
                stack,
                captureOutputs: this.queries.stackOutputsReferenced(stack).length > 0,
                withoutChangeSet: prepareNode === undefined,
            });
            retGraph.add(stackGraph);
            stackGraph.add(deployNode);
            // node or node collection that represents first point of contact in each stack
            let firstDeployNode;
            if (prepareNode) {
                stackGraph.add(prepareNode);
                deployNode.dependOn(prepareNode);
                firstDeployNode = prepareNode;
            }
            else {
                firstDeployNode = deployNode;
            }
            // add changeset steps at the stack level
            if (stack.changeSet.length > 0) {
                if (prepareNode) {
                    this.addChangeSetNode(stack.changeSet, prepareNode, deployNode, stackGraph);
                }
                else {
                    throw new Error(`Cannot use \'changeSet\' steps for stack \'${stack.stackName}\': the pipeline does not support them or they have been disabled`);
                }
            }
            // add pre and post steps at the stack level
            const preNodes = this.addPrePost(stack.pre, stack.post, stackGraph);
            if (preNodes.nodes.length > 0) {
                firstDeployNode = preNodes;
            }
            stackGraphs.set(stack, stackGraph);
            const cloudAssembly = this.cloudAssemblyFileSet;
            firstDeployNode.dependOn(this.addStepNode(cloudAssembly.producer, retGraph));
            // add the template asset
            if (this.publishTemplate) {
                if (!stack.templateAsset) {
                    throw new Error(`"publishTemplate" is enabled, but stack ${stack.stackArtifactId} does not have a template asset`);
                }
                firstDeployNode.dependOn(this.publishAsset(stack.templateAsset));
            }
            // Depend on Assets
            // FIXME: Custom Cloud Assembly currently doesn't actually help separating
            // out templates from assets!!!
            for (const asset of stack.assets) {
                const assetNode = this.publishAsset(asset);
                firstDeployNode.dependOn(assetNode);
            }
            // Add stack output synchronization point
            if (this.queries.stackOutputsReferenced(stack).length > 0) {
                this.stackOutputDependencies.for(stack).dependOn(deployNode);
            }
        }
        for (const stack of stage.stacks) {
            for (const dep of stack.stackDependencies) {
                const stackNode = stackGraphs.get(stack);
                const depNode = stackGraphs.get(dep);
                if (!stackNode) {
                    throw new Error(`cannot find node for ${stack.stackName}`);
                }
                if (!depNode) {
                    throw new Error(`cannot find node for ${dep.stackName}`);
                }
                stackNode.dependOn(depNode);
            }
        }
        this.addPrePost(stage.pre, stage.post, retGraph);
        return retGraph;
    }
    addChangeSetNode(changeSet, prepareNode, deployNode, graph) {
        for (const c of changeSet) {
            const changeSetNode = this.addStepNode(c, graph);
            changeSetNode?.dependOn(prepareNode);
            deployNode.dependOn(changeSetNode);
        }
    }
    addPrePost(pre, post, parent) {
        const currentNodes = new graph_1.GraphNodeCollection(parent.nodes);
        const preNodes = new graph_1.GraphNodeCollection(new Array());
        for (const p of pre) {
            const preNode = this.addStepNode(p, parent);
            currentNodes.dependOn(preNode);
            preNodes.nodes.push(preNode);
        }
        for (const p of post) {
            const postNode = this.addStepNode(p, parent);
            postNode?.dependOn(...currentNodes.nodes);
        }
        return preNodes;
    }
    topLevelGraph(name) {
        let ret = this.graph.tryGetChild(name);
        if (!ret) {
            ret = new graph_1.Graph(name);
            this.graph.add(ret);
        }
        return ret;
    }
    /**
     * Add a Node to a Graph for a given Step
     *
     * Adds all dependencies for that Node to the same Step as well.
     */
    addStepNode(step, parent) {
        if (step === PipelineGraph.NO_STEP) {
            return undefined;
        }
        const previous = this.added.get(step);
        if (previous) {
            return previous;
        }
        const node = aGraphNode(step.id, { type: 'step', step });
        // If the step is a source step, change the parent to a special "Source" stage
        // (CodePipeline wants it that way)
        if (step.isSource) {
            parent = this.topLevelGraph('Source');
        }
        parent.add(node);
        this.added.set(step, node);
        // This used to recurse -- that's not safe, because it might create nodes in the
        // wrong graph (it would create a dependency node, that might need to be created in
        // a different graph, in the current one). Instead, use DependencyBuilders.
        for (const dep of step.dependencies) {
            this.nodeDependencies.for(dep).dependBy(node);
        }
        this.nodeDependencies.for(step).dependOn(node);
        // Add stack dependencies (by use of the dependency builder this also works
        // if we encounter the Step before the Stack has been properly added yet)
        for (const output of step.consumedStackOutputs) {
            const stack = this.queries.producingStack(output);
            this.stackOutputDependencies.for(stack).dependBy(node);
        }
        return node;
    }
    /**
     * Add dependencies that aren't in the pipeline yet
     *
     * Build steps reference as many sources (or other builds) as they want, which will be added
     * automatically. Do that here. We couldn't do it earlier, because if there were dependencies
     * between steps we didn't want to reparent those unnecessarily.
     */
    addMissingDependencyNodes() {
        // May need to do this more than once to recursively add all missing producers
        let attempts = 20;
        while (attempts-- > 0) {
            const unsatisfied = this.nodeDependencies.unsatisfiedBuilders().filter(([s]) => s !== PipelineGraph.NO_STEP);
            if (unsatisfied.length === 0) {
                return;
            }
            for (const [step, builder] of unsatisfied) {
                // Add a new node for this step to the parent of the "leftmost" consumer.
                const leftMostConsumer = new graph_1.GraphNodeCollection(builder.consumers).first();
                const parent = leftMostConsumer.parentGraph;
                if (!parent) {
                    throw new Error(`Consumer doesn't have a parent graph: ${leftMostConsumer}`);
                }
                this.addStepNode(step, parent);
            }
        }
        const unsatisfied = this.nodeDependencies.unsatisfiedBuilders();
        throw new Error([
            'Recursion depth too large while adding dependency nodes:',
            unsatisfied.map(([step, builder]) => `${builder.consumersAsString()} awaiting ${step}.`),
        ].join(' '));
    }
    publishAsset(stackAsset) {
        const assetsGraph = this.topLevelGraph('Assets');
        let assetNode = this.assetNodes.get(stackAsset.assetId);
        if (assetNode) {
            // If there's already a node publishing this asset, add as a new publishing
            // destination to the same node.
        }
        else if (this.singlePublisher && this.assetNodesByType.has(stackAsset.assetType)) {
            // If we're doing a single node per type, lookup by that
            assetNode = this.assetNodesByType.get(stackAsset.assetType);
        }
        else {
            // Otherwise add a new one
            const id = stackAsset.assetType === blueprint_1.AssetType.FILE
                ? (this.singlePublisher ? 'FileAsset' : `FileAsset${++this._fileAssetCtr}`)
                : (this.singlePublisher ? 'DockerAsset' : `DockerAsset${++this._dockerAssetCtr}`);
            assetNode = aGraphNode(id, { type: 'publish-assets', assets: [] });
            assetsGraph.add(assetNode);
            assetNode.dependOn(this.lastPreparationNode);
            this.assetNodesByType.set(stackAsset.assetType, assetNode);
            this.assetNodes.set(stackAsset.assetId, assetNode);
        }
        const data = assetNode.data;
        if (data?.type !== 'publish-assets') {
            throw new Error(`${assetNode} has the wrong data.type: ${data?.type}`);
        }
        if (!data.assets.some(a => a.assetSelector === stackAsset.assetSelector)) {
            data.assets.push(stackAsset);
        }
        return assetNode;
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simpleStackName(stackName, stageName) {
        return stripPrefix(stackName, `${stageName}-`);
    }
}
/**
 * A Step object that may be used as the producer of FileSets that should not be represented in the graph
 */
PipelineGraph.NO_STEP = new class extends blueprint_1.Step {
}('NO_STEP');
exports.PipelineGraph = PipelineGraph;
function aGraphNode(id, x) {
    return graph_1.GraphNode.of(id, x);
}
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.slice(prefix.length) : s;
}
//# sourceMappingURL=data:application/json;base64,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