"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineQueries = void 0;
/**
 * Answer some questions about a pipeline blueprint
 */
class PipelineQueries {
    constructor(pipeline) {
        this.pipeline = pipeline;
    }
    /**
     * Return the names of all outputs for the given stack that are referenced in this blueprint
     */
    stackOutputsReferenced(stack) {
        const steps = new Array();
        for (const wave of this.pipeline.waves) {
            steps.push(...wave.pre, ...wave.post);
            for (const stage of wave.stages) {
                steps.push(...stage.pre, ...stage.post);
                for (const stackDeployment of stage.stacks) {
                    steps.push(...stackDeployment.pre, ...stackDeployment.changeSet, ...stackDeployment.post);
                }
            }
        }
        const ret = new Array();
        for (const step of steps) {
            for (const outputRef of step.consumedStackOutputs) {
                if (outputRef.isProducedBy(stack)) {
                    ret.push(outputRef.outputName);
                }
            }
        }
        return ret;
    }
    /**
     * Find the stack deployment that is producing the given reference
     */
    producingStack(outputReference) {
        for (const wave of this.pipeline.waves) {
            for (const stage of wave.stages) {
                for (const stack of stage.stacks) {
                    if (outputReference.isProducedBy(stack)) {
                        return stack;
                    }
                }
            }
        }
        throw new Error(`Stack '${outputReference.stackDescription}' (producing output '${outputReference.outputName}') is not in the pipeline; call 'addStage()' to add the stack's Stage to the pipeline`);
    }
    /**
     * All assets referenced in all the Stacks of a StageDeployment
     */
    assetsInStage(stage) {
        const assets = new Map();
        for (const stack of stage.stacks) {
            for (const asset of stack.assets) {
                assets.set(asset.assetSelector, asset);
            }
        }
        return Array.from(assets.values());
    }
}
exports.PipelineQueries = PipelineQueries;
//# sourceMappingURL=data:application/json;base64,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