"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepOutput = void 0;
const core_1 = require("../../../core");
const STEP_OUTPUT_SYM = Symbol.for('@aws-cdk/pipelines.StepOutput');
const PRODUCED_OUTPUTS_SYM = Symbol.for('@aws-cdk/pipelines.outputs');
/**
 * A symbolic reference to a value produced by another step
 *
 * Generating and consuming outputs is engine-specific. Many engines will be
 * able to support a feature like "outputs", but it's not guaranteed that
 * all of them will.
 *
 * Outputs can only be generated by engine-specific steps (CodeBuildStep instead
 * of ShellStep, etc), but can (currently) be consumed anywhere(*). When
 * an engine-specific step generates an Output, it should put a well-known
 * string and arbitrary data that is useful to the engine into the engine-specific
 * fields on the StepOutput.
 *
 * The graph blueprint will take care of dependencies and ordering, the engine
 * is responsible interpreting and rendering StepOutputs. The engine should call
 * `defineResolution()` on all outputs.
 *
 * StepOutputs currently purposely aren't part of the public API because users
 * shouldn't see the innards poking out. So, instead of keeping state on `Step`,
 * we keep side-state here in a WeakMap which can be accessed via static members
 * on `StepOutput`.
 *
 * (*) If we need to restrict this, we add the checking and erroring in the engine.
 */
class StepOutput {
    /**
     * Return true if the given IResolvable is a StepOutput
     */
    static isStepOutput(resolvable) {
        return !!resolvable[STEP_OUTPUT_SYM];
    }
    /**
     * Find all StepOutputs referenced in the given structure
     */
    static findAll(structure) {
        return findAllStepOutputs(structure);
    }
    /**
     * Return the produced outputs for the given step
     */
    static producedStepOutputs(step) {
        return step[PRODUCED_OUTPUTS_SYM] ?? [];
    }
    /**
     * Add produced outputs for the given step
     */
    static recordProducer(...outputs) {
        for (const output of outputs) {
            const step = output.step;
            let list = step[PRODUCED_OUTPUTS_SYM];
            if (!list) {
                list = [];
                step[PRODUCED_OUTPUTS_SYM] = list;
            }
            list.push(...outputs);
        }
    }
    constructor(step, engineName, engineSpecificInformation) {
        this.creationStack = [];
        this.resolution = undefined;
        this.step = step;
        this.engineName = engineName;
        this.engineSpecificInformation = engineSpecificInformation;
        Object.defineProperty(this, STEP_OUTPUT_SYM, { value: true });
    }
    /**
     * Define the resolved value for this StepOutput.
     *
     * Should be called by the engine.
     */
    defineResolution(value) {
        this.resolution = value;
    }
    resolve(_context) {
        if (this.resolution === undefined) {
            throw new Error(`Output for step ${this.step} not configured. Either the step is not in the pipeline, the step implementation did not call 'this.discoverReferencedOutputs()', or this engine does not support Outputs for this step.`);
        }
        return this.resolution;
    }
    toString() {
        return core_1.Token.asString(this);
    }
}
exports.StepOutput = StepOutput;
function findAllStepOutputs(structure) {
    const ret = new Set();
    recurse(structure);
    return Array.from(ret);
    function checkToken(x) {
        if (x && StepOutput.isStepOutput(x)) {
            ret.add(x);
            return true;
        }
        // Return false if it wasn't a Token in the first place (in which case we recurse)
        return x !== undefined;
    }
    function recurse(x) {
        if (!x) {
            return;
        }
        if (core_1.Tokenization.isResolvable(x)) {
            checkToken(x);
            return;
        }
        if (Array.isArray(x)) {
            if (!checkToken(core_1.Tokenization.reverseList(x))) {
                x.forEach(recurse);
            }
            return;
        }
        if (typeof x === 'number') {
            checkToken(core_1.Tokenization.reverseNumber(x));
            return;
        }
        if (typeof x === 'string') {
            core_1.Tokenization.reverseString(x).tokens.forEach(checkToken);
            return;
        }
        if (typeof x === 'object') {
            for (const [k, v] of Object.entries(x)) {
                recurse(k);
                recurse(v);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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