"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topoSort = exports.printDependencyMap = void 0;
function printDependencyMap(dependencies) {
    const lines = ['---'];
    for (const [k, vs] of dependencies.entries()) {
        lines.push(`${k} -> ${Array.from(vs)}`);
    }
    // eslint-disable-next-line no-console
    console.log(lines.join('\n'));
}
exports.printDependencyMap = printDependencyMap;
function topoSort(nodes, dependencies, fail = true) {
    const remaining = new Set(nodes);
    const ret = [];
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => {
            if (!dependencies.has(e)) {
                throw new Error(`No key for ${e}`);
            }
            return dependencies.get(e).size === 0;
        });
        selectable.sort((a, b) => a.id < b.id ? -1 : b.id < a.id ? 1 : 0);
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            const cycle = findCycle(dependencies);
            if (fail) {
                throw new Error(`Dependency cycle in graph: ${cycle.map(n => n.id).join(' => ')}`);
            }
            // If we're trying not to fail, pick one at random from the cycle and treat it
            // as selectable, then continue.
            selectable.push(cycle[0]);
        }
        ret.push(selectable);
        for (const selected of selectable) {
            remaining.delete(selected);
            for (const depSet of dependencies.values()) {
                depSet.delete(selected);
            }
        }
    }
    return ret;
}
exports.topoSort = topoSort;
/**
 * Find cycles in a graph
 *
 * Not the fastest, but effective and should be rare
 */
function findCycle(deps) {
    for (const node of deps.keys()) {
        const cycle = recurse(node, [node]);
        if (cycle) {
            return cycle;
        }
    }
    throw new Error('No cycle found. Assertion failure!');
    function recurse(node, path) {
        for (const dep of deps.get(node) ?? []) {
            if (dep === path[0]) {
                return [...path, dep];
            }
            const cycle = recurse(dep, [...path, dep]);
            if (cycle) {
                return cycle;
            }
        }
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9wb3NvcnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0b3Bvc29ydC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFFQSxTQUFnQixrQkFBa0IsQ0FBSSxZQUFrRDtJQUN0RixNQUFNLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3RCLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsSUFBSSxZQUFZLENBQUMsT0FBTyxFQUFFLEVBQUU7UUFDNUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUN6QztJQUNELHNDQUFzQztJQUN0QyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztBQUNoQyxDQUFDO0FBUEQsZ0RBT0M7QUFFRCxTQUFnQixRQUFRLENBQUksS0FBd0IsRUFBRSxZQUFrRCxFQUFFLElBQUksR0FBQyxJQUFJO0lBQ2pILE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFlLEtBQUssQ0FBQyxDQUFDO0lBRS9DLE1BQU0sR0FBRyxHQUFxQixFQUFFLENBQUM7SUFDakMsT0FBTyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQUMsRUFBRTtRQUN6QiwyREFBMkQ7UUFDM0QsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDM0QsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQ3BDO1lBQ0QsT0FBTyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBRSxDQUFDLElBQUksS0FBSyxDQUFDLENBQUM7UUFDekMsQ0FBQyxDQUFDLENBQUM7UUFDSCxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRWxFLCtDQUErQztRQUMvQyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNCLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUV0QyxJQUFJLElBQUksRUFBRTtnQkFDUixNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDcEY7WUFFRCw4RUFBOEU7WUFDOUUsZ0NBQWdDO1lBQ2hDLFVBQVUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDM0I7UUFFRCxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXJCLEtBQUssTUFBTSxRQUFRLElBQUksVUFBVSxFQUFFO1lBQ2pDLFNBQVMsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDM0IsS0FBSyxNQUFNLE1BQU0sSUFBSSxZQUFZLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQzFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7YUFDekI7U0FDRjtLQUNGO0lBRUQsT0FBTyxHQUFHLENBQUM7QUFDYixDQUFDO0FBdENELDRCQXNDQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLFNBQVMsQ0FBSSxJQUEwQztJQUM5RCxLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRTtRQUM5QixNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNwQyxJQUFJLEtBQUssRUFBRTtZQUFFLE9BQU8sS0FBSyxDQUFDO1NBQUU7S0FDN0I7SUFDRCxNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFFdEQsU0FBUyxPQUFPLENBQUMsSUFBa0IsRUFBRSxJQUFvQjtRQUN2RCxLQUFLLE1BQU0sR0FBRyxJQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFO1lBQ3RDLElBQUksR0FBRyxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFBRSxPQUFPLENBQUMsR0FBRyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7YUFBRTtZQUUvQyxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUMsR0FBRyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUMzQyxJQUFJLEtBQUssRUFBRTtnQkFBRSxPQUFPLEtBQUssQ0FBQzthQUFFO1NBQzdCO1FBRUQsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBHcmFwaE5vZGUgfSBmcm9tICcuL2dyYXBoJztcblxuZXhwb3J0IGZ1bmN0aW9uIHByaW50RGVwZW5kZW5jeU1hcDxBPihkZXBlbmRlbmNpZXM6IE1hcDxHcmFwaE5vZGU8QT4sIFNldDxHcmFwaE5vZGU8QT4+Pikge1xuICBjb25zdCBsaW5lcyA9IFsnLS0tJ107XG4gIGZvciAoY29uc3QgW2ssIHZzXSBvZiBkZXBlbmRlbmNpZXMuZW50cmllcygpKSB7XG4gICAgbGluZXMucHVzaChgJHtrfSAtPiAke0FycmF5LmZyb20odnMpfWApO1xuICB9XG4gIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gIGNvbnNvbGUubG9nKGxpbmVzLmpvaW4oJ1xcbicpKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHRvcG9Tb3J0PEE+KG5vZGVzOiBTZXQ8R3JhcGhOb2RlPEE+PiwgZGVwZW5kZW5jaWVzOiBNYXA8R3JhcGhOb2RlPEE+LCBTZXQ8R3JhcGhOb2RlPEE+Pj4sIGZhaWw9dHJ1ZSk6IEdyYXBoTm9kZTxBPltdW10ge1xuICBjb25zdCByZW1haW5pbmcgPSBuZXcgU2V0PEdyYXBoTm9kZTxBPj4obm9kZXMpO1xuXG4gIGNvbnN0IHJldDogR3JhcGhOb2RlPEE+W11bXSA9IFtdO1xuICB3aGlsZSAocmVtYWluaW5nLnNpemUgPiAwKSB7XG4gICAgLy8gQWxsIGVsZW1lbnRzIHdpdGggbm8gbW9yZSBkZXBzIGluIHRoZSBzZXQgY2FuIGJlIG9yZGVyZWRcbiAgICBjb25zdCBzZWxlY3RhYmxlID0gQXJyYXkuZnJvbShyZW1haW5pbmcudmFsdWVzKCkpLmZpbHRlcihlID0+IHtcbiAgICAgIGlmICghZGVwZW5kZW5jaWVzLmhhcyhlKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYE5vIGtleSBmb3IgJHtlfWApO1xuICAgICAgfVxuICAgICAgcmV0dXJuIGRlcGVuZGVuY2llcy5nZXQoZSkhLnNpemUgPT09IDA7XG4gICAgfSk7XG4gICAgc2VsZWN0YWJsZS5zb3J0KChhLCBiKSA9PiBhLmlkIDwgYi5pZCA/IC0xIDogYi5pZCA8IGEuaWQgPyAxIDogMCk7XG5cbiAgICAvLyBJZiB3ZSBkaWRuJ3QgbWFrZSBhbnkgcHJvZ3Jlc3MsIHdlIGdvdCBzdHVja1xuICAgIGlmIChzZWxlY3RhYmxlLmxlbmd0aCA9PT0gMCkge1xuICAgICAgY29uc3QgY3ljbGUgPSBmaW5kQ3ljbGUoZGVwZW5kZW5jaWVzKTtcblxuICAgICAgaWYgKGZhaWwpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBEZXBlbmRlbmN5IGN5Y2xlIGluIGdyYXBoOiAke2N5Y2xlLm1hcChuID0+IG4uaWQpLmpvaW4oJyA9PiAnKX1gKTtcbiAgICAgIH1cblxuICAgICAgLy8gSWYgd2UncmUgdHJ5aW5nIG5vdCB0byBmYWlsLCBwaWNrIG9uZSBhdCByYW5kb20gZnJvbSB0aGUgY3ljbGUgYW5kIHRyZWF0IGl0XG4gICAgICAvLyBhcyBzZWxlY3RhYmxlLCB0aGVuIGNvbnRpbnVlLlxuICAgICAgc2VsZWN0YWJsZS5wdXNoKGN5Y2xlWzBdKTtcbiAgICB9XG5cbiAgICByZXQucHVzaChzZWxlY3RhYmxlKTtcblxuICAgIGZvciAoY29uc3Qgc2VsZWN0ZWQgb2Ygc2VsZWN0YWJsZSkge1xuICAgICAgcmVtYWluaW5nLmRlbGV0ZShzZWxlY3RlZCk7XG4gICAgICBmb3IgKGNvbnN0IGRlcFNldCBvZiBkZXBlbmRlbmNpZXMudmFsdWVzKCkpIHtcbiAgICAgICAgZGVwU2V0LmRlbGV0ZShzZWxlY3RlZCk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIHJldDtcbn1cblxuLyoqXG4gKiBGaW5kIGN5Y2xlcyBpbiBhIGdyYXBoXG4gKlxuICogTm90IHRoZSBmYXN0ZXN0LCBidXQgZWZmZWN0aXZlIGFuZCBzaG91bGQgYmUgcmFyZVxuICovXG5mdW5jdGlvbiBmaW5kQ3ljbGU8QT4oZGVwczogTWFwPEdyYXBoTm9kZTxBPiwgU2V0PEdyYXBoTm9kZTxBPj4+KTogR3JhcGhOb2RlPEE+W10ge1xuICBmb3IgKGNvbnN0IG5vZGUgb2YgZGVwcy5rZXlzKCkpIHtcbiAgICBjb25zdCBjeWNsZSA9IHJlY3Vyc2Uobm9kZSwgW25vZGVdKTtcbiAgICBpZiAoY3ljbGUpIHsgcmV0dXJuIGN5Y2xlOyB9XG4gIH1cbiAgdGhyb3cgbmV3IEVycm9yKCdObyBjeWNsZSBmb3VuZC4gQXNzZXJ0aW9uIGZhaWx1cmUhJyk7XG5cbiAgZnVuY3Rpb24gcmVjdXJzZShub2RlOiBHcmFwaE5vZGU8QT4sIHBhdGg6IEdyYXBoTm9kZTxBPltdKTogR3JhcGhOb2RlPEE+W10gfCB1bmRlZmluZWQge1xuICAgIGZvciAoY29uc3QgZGVwIG9mIGRlcHMuZ2V0KG5vZGUpID8/IFtdKSB7XG4gICAgICBpZiAoZGVwID09PSBwYXRoWzBdKSB7IHJldHVybiBbLi4ucGF0aCwgZGVwXTsgfVxuXG4gICAgICBjb25zdCBjeWNsZSA9IHJlY3Vyc2UoZGVwLCBbLi4ucGF0aCwgZGVwXSk7XG4gICAgICBpZiAoY3ljbGUpIHsgcmV0dXJuIGN5Y2xlOyB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxufSJdfQ==