"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cpactions = require("../../../../aws-codepipeline-actions");
const iam = require("../../../../aws-iam");
const core_1 = require("../../../../core");
const cxapi = require("../../../../cx-api");
const constructs_1 = require("constructs");
const construct_internals_1 = require("../../private/construct-internals");
const fs_1 = require("../../private/fs");
/**
 * Action to deploy a CDK Stack
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 *
 * @deprecated This class is part of the old API. Use the API based on the `CodePipeline` class instead
 */
class DeployCdkStackAction {
    /**
     * Construct a DeployCdkStackAction from a Stack artifact
     */
    static fromStackArtifact(scope, artifact, options) {
        if (!artifact.assumeRoleArn) {
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = artifact.environment.account;
        const account = artAccount === core_1.Stack.of(scope).account || artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        const actionRole = roleFromPlaceholderArn(scope, region, account, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, region, account, artifact.cloudFormationExecutionRoleArn);
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = (0, construct_internals_1.assemblyBuilderOf)((0, construct_internals_1.appOf)(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        let fullConfigPath;
        if (Object.keys(artifact.tags).length > 0) {
            fullConfigPath = `${fullTemplatePath}.config.json`;
            // Write the template configuration file (for parameters into CreateChangeSet call that
            // cannot be configured any other way). They must come from a file, and there's unfortunately
            // no better hook to write this file (`construct.onSynthesize()` would have been the prime candidate
            // but that is being deprecated--and DeployCdkStackAction isn't even a construct).
            writeTemplateConfiguration(fullConfigPath, {
                Tags: artifact.tags,
            });
        }
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath: (0, fs_1.toPosixPath)(path.relative(appAsmRoot, fullTemplatePath)),
            templateConfigurationPath: fullConfigPath ? (0, fs_1.toPosixPath)(path.relative(appAsmRoot, fullConfigPath)) : undefined,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: options.stackName ?? artifact.stackName,
            ...options,
        });
    }
    constructor(props) {
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = props.dependencyStackArtifactIds ?? [];
        this.prepareRunOrder = props.prepareRunOrder ?? 1;
        this.executeRunOrder = props.executeRunOrder ?? this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = props.baseActionName ?? this.stackName;
        const changeSetName = props.changeSetName ?? 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            cfnCapabilities: [core_1.CfnCapabilities.NAMED_IAM, core_1.CfnCapabilities.AUTO_EXPAND],
            templateConfiguration: props.templateConfigurationPath ? props.cloudAssemblyInput.atPath(props.templateConfigurationPath) : undefined,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
_a = JSII_RTTI_SYMBOL_1;
DeployCdkStackAction[_a] = { fqn: "aws-cdk-lib.pipelines.DeployCdkStackAction", version: "2.74.0" };
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, region, account, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = constructs_1.Node.of(scope).tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = constructs_1.Node.of(scope).tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
        region: region ?? core_1.Aws.REGION,
        accountId: account ?? core_1.Aws.ACCOUNT_ID,
        partition: core_1.Aws.PARTITION,
    });
    return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false, addGrantsToResources: true });
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
/**
 * Write template configuration to the given file
 */
function writeTemplateConfiguration(filename, config) {
    fs.writeFileSync(filename, JSON.stringify(config, undefined, 2), { encoding: 'utf-8' });
}
//# sourceMappingURL=data:application/json;base64,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