"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const codepipeline = require("../../../aws-codepipeline");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const actions_1 = require("./actions");
const stage_1 = require("./stage");
const synths_1 = require("./synths");
const asset_type_1 = require("../blueprint/asset-type");
const docker_credentials_1 = require("../docker-credentials");
const application_security_check_1 = require("../private/application-security-check");
const asset_singleton_role_1 = require("../private/asset-singleton-role");
const cached_fnsub_1 = require("../private/cached-fnsub");
const cli_version_1 = require("../private/cli-version");
const construct_internals_1 = require("../private/construct-internals");
const CODE_BUILD_LENGTH_LIMIT = 100;
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 *
 * @deprecated This class is part of the old API. Use the API based on the `CodePipeline` class instead
 */
class CdkPipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        this.cliVersion = props.cdkCliVersion ?? (0, cli_version_1.preferredCliVersion)();
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this._dockerCredentials = props.dockerCredentials ?? [];
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.enableKeyRotation !== undefined) {
                throw new Error('Cannot set \'enableKeyRotation\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                enableKeyRotation: props.enableKeyRotation,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            if (props.synthAction instanceof synths_1.SimpleSynthAction && this._dockerCredentials.length > 0) {
                props.synthAction._addDockerCredentials(this._dockerCredentials);
            }
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        if (props.selfMutating ?? true) {
            this._pipeline.addStage({
                stageName: 'UpdatePipeline',
                actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                        cloudAssemblyInput: this._cloudAssemblyArtifact,
                        pipelineStackHierarchicalId: pipelineStack.node.path,
                        cdkCliVersion: this.cliVersion,
                        projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        privileged: props.supportDockerAssets,
                        dockerCredentials: this._dockerCredentials,
                        buildSpec: props.selfMutationBuildSpec,
                    })],
            });
        }
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: this.cliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            singlePublisherPerType: props.singlePublisherPerType,
            preInstallCommands: props.assetPreInstallCommands,
            buildSpec: props.assetBuildSpec,
            dockerCredentials: this._dockerCredentials,
        });
        this.node.addValidation({ validate: () => this.validatePipeline() });
    }
    /**
     * The underlying CodePipeline object
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * Access one of the pipeline's stages by stage name
     *
     * You can use this to add more Actions to a stage.
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * Get a cached version of an Application Security Check, which consists of:
     *  - CodeBuild Project to check for security changes in a stage
     *  - Lambda Function that approves the manual approval if no security changes are detected
     *
     * @internal
     */
    _getApplicationSecurityCheck() {
        if (!this._applicationSecurityCheck) {
            this._applicationSecurityCheck = new application_security_check_1.ApplicationSecurityCheck(this, 'PipelineApplicationSecurityCheck', {
                codePipeline: this._pipeline,
            });
        }
        return this._applicationSecurityCheck;
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName, options);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName, options) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
            ...options,
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            const artifactName = `${stack.artifactId}_Outputs`;
            const compactName = artifactName.slice(artifactName.length - Math.min(artifactName.length, CODE_BUILD_LENGTH_LIMIT));
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(compactName);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validatePipeline() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
CdkPipeline[_a] = { fqn: "aws-cdk-lib.pipelines.CdkPipeline", version: "2.74.0" };
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        // CodePipelines has a hard limit of 50 actions per stage. See https://github.com/aws/aws-cdk/issues/9353
        this.MAX_PUBLISHERS_PER_STAGE = 50;
        this.publishers = {};
        this.assetRoles = new Map();
        this.assetAttachedPolicies = {};
        this.cachedFnSub = new cached_fnsub_1.CachedFnSub();
        this.stages = [];
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.myCxAsmRoot = path.resolve((0, construct_internals_1.assemblyBuilderOf)((0, construct_internals_1.appOf)(this)).outdir);
        this.pipeline = this.props.pipeline;
        // Hacks to get access to the innards of Pipeline
        const stages = this.props.pipeline._stages;
        // Any asset publishing stages will be added directly after the last stage that currently exists.
        this.lastStageBeforePublishing = stages.slice(-1)[0];
        this.dockerCredentials = props.dockerCredentials;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        const assetRole = this.generateAssetRole(command.assetType);
        // The ARNs include raw AWS pseudo parameters (e.g., ${AWS::Partition}), which need to be substituted.
        assetRole.addAssumeRole(this.cachedFnSub.fnSub(command.assetPublishingRoleArn));
        const publisherKey = this.props.singlePublisherPerType ? command.assetType.toString() : command.assetId;
        let action = this.publishers[publisherKey];
        if (!action) {
            // Dynamically create new stages as needed, with `MAX_PUBLISHERS_PER_STAGE` assets per stage.
            const stageIndex = this.props.singlePublisherPerType ? 0 :
                Math.floor((this._fileAssetCtr + this._dockerAssetCtr) / this.MAX_PUBLISHERS_PER_STAGE);
            if (!this.props.singlePublisherPerType && stageIndex >= this.stages.length) {
                const previousStage = this.stages.slice(-1)[0] ?? this.lastStageBeforePublishing;
                this.stages.push(this.pipeline.addStage({
                    stageName: `Assets${stageIndex > 0 ? stageIndex + 1 : ''}`,
                    placement: { justAfter: previousStage },
                }));
            }
            else if (this.props.singlePublisherPerType && this.stages.length == 0) {
                this.stages.push(this.pipeline.addStage({
                    stageName: 'Assets',
                    placement: { justAfter: this.lastStageBeforePublishing },
                }));
            }
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = this.props.singlePublisherPerType ?
                command.assetType === asset_type_1.AssetType.FILE ? 'FileAsset' : 'DockerAsset' :
                command.assetType === asset_type_1.AssetType.FILE ? `FileAsset${++this._fileAssetCtr}` : `DockerAsset${++this._dockerAssetCtr}`;
            const credsInstallCommands = (0, docker_credentials_1.dockerCredentialsInstallCommands)(docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING, this.dockerCredentials);
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[publisherKey] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles.get(command.assetType),
                dependable: this.assetAttachedPolicies[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
                buildSpec: this.props.buildSpec,
                createBuildspecFile: this.props.singlePublisherPerType,
                preInstallCommands: [...(this.props.preInstallCommands ?? []), ...credsInstallCommands],
            });
            this.stages[stageIndex].addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        const existing = this.assetRoles.get(assetType);
        if (existing) {
            return existing;
        }
        const rolePrefix = assetType === asset_type_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new asset_singleton_role_1.AssetSingletonRole(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Grant pull access for any ECR registries and secrets that exist
        if (assetType === asset_type_1.AssetType.DOCKER_IMAGE) {
            this.dockerCredentials.forEach(reg => reg.grantRead(assetRole, docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING));
        }
        this.assetRoles.set(assetType, assetRole);
        return assetRole;
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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