"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutput = exports.CdkStage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("../../../aws-codebuild");
const cpactions = require("../../../aws-codepipeline-actions");
const aws_codepipeline_actions_1 = require("../../../aws-codepipeline-actions");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const actions_1 = require("./actions");
const pipeline_1 = require("./pipeline");
const asset_type_1 = require("../blueprint/asset-type");
const application_security_check_1 = require("../private/application-security-check");
const asset_manifest_1 = require("../private/asset-manifest");
const construct_internals_1 = require("../private/construct-internals");
const toposort_1 = require("../private/toposort");
/**
 * Stage in a CdkPipeline
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 *
 * @deprecated This class is part of the old API. Use the API based on the `CodePipeline` class instead
 */
class CdkStage extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._nextSequentialRunOrder = 1; // Must start at 1 eh
        this._manualApprovalCounter = 1;
        this.stacksToDeploy = new Array();
        this._prepared = false;
        if (scope instanceof pipeline_1.CdkPipeline) {
            this.pipeline = scope;
        }
        this.stageName = props.stageName;
        this.pipelineStage = props.pipelineStage;
        this.cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this.host = props.host;
        this.confirmBroadeningPermissions = props.confirmBroadeningPermissions ?? false;
        this.securityNotificationTopic = props.securityNotificationTopic;
        core_1.Aspects.of(this).add({ visit: () => this.prepareStage() });
    }
    /**
     * Add all stacks in the application Stage to this stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplication(appStage, options = {}) {
        const asm = (0, construct_internals_1.pipelineSynth)(appStage);
        const extraRunOrderSpace = options.extraRunOrderSpace ?? 0;
        if (options.confirmBroadeningPermissions ?? this.confirmBroadeningPermissions) {
            this.addSecurityCheck(appStage, options);
        }
        if (asm.stacks.length === 0) {
            // If we don't check here, a more puzzling "stage contains no actions"
            // error will be thrown come deployment time.
            throw new Error(`The given Stage construct ('${appStage.node.path}') should contain at least one Stack`);
        }
        const sortedTranches = (0, toposort_1.topologicalSort)(asm.stacks, stack => stack.id, stack => stack.dependencies.map(d => d.id));
        for (const stacks of sortedTranches) {
            const runOrder = this.nextSequentialRunOrder(extraRunOrderSpace + 2); // 2 actions for Prepare/Execute ChangeSet
            let executeRunOrder = runOrder + extraRunOrderSpace + 1;
            // If we need to insert a manual approval action, then what's the executeRunOrder
            // now is where we add a manual approval step, and we allocate 1 more runOrder
            // for the execute.
            if (options.manualApprovals) {
                this.addManualApprovalAction({ runOrder: runOrder + 1 });
                executeRunOrder = this.nextSequentialRunOrder();
            }
            // These don't have a dependency on each other, so can all be added in parallel
            for (const stack of stacks) {
                this.addStackArtifactDeployment(stack, { runOrder, executeRunOrder });
            }
        }
    }
    /**
     * Get a cached version of an ApplicationSecurityCheck, which consists of:
     *  - CodeBuild Project to check for security changes in a stage
     *  - Lambda Function that approves the manual approval if no security changes are detected
     *
     * The ApplicationSecurityCheck is cached from the pipeline **if** this stage is scoped
     * to a CDK Pipeline. If this stage **is not** scoped to a pipeline, create an ApplicationSecurityCheck
     * scoped to the stage itself.
     *
     * @internal
     */
    getApplicationSecurityCheck() {
        if (this._applicationSecurityCheck) {
            return this._applicationSecurityCheck;
        }
        this._applicationSecurityCheck = this.pipeline
            ? this.pipeline._getApplicationSecurityCheck()
            : new application_security_check_1.ApplicationSecurityCheck(this, 'StageApplicationSecurityCheck', {
                codePipeline: this.pipelineStage.pipeline,
            });
        return this._applicationSecurityCheck;
    }
    /**
     * Add a deployment action based on a stack artifact
     */
    addStackArtifactDeployment(stackArtifact, options = {}) {
        // Get all assets manifests and add the assets in 'em to the asset publishing stage.
        this.publishAssetDependencies(stackArtifact);
        // Remember for later, see 'prepare()'
        // We know that deploying a stack is going to take up 2 runorder slots later on.
        const runOrder = options.runOrder ?? this.nextSequentialRunOrder(2);
        const executeRunOrder = options.executeRunOrder ?? runOrder + 1;
        this.stacksToDeploy.push({
            prepareRunOrder: runOrder,
            executeRunOrder,
            stackArtifact,
        });
        this.advanceRunOrderPast(runOrder);
        this.advanceRunOrderPast(executeRunOrder);
    }
    /**
     * Add a manual approval action
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     */
    addManualApprovalAction(options = {}) {
        let actionName = options.actionName;
        if (!actionName) {
            actionName = `ManualApproval${this._manualApprovalCounter > 1 ? this._manualApprovalCounter : ''}`;
            this._manualApprovalCounter += 1;
        }
        this.addActions(new cpactions.ManualApprovalAction({
            actionName,
            runOrder: options.runOrder ?? this.nextSequentialRunOrder(),
        }));
    }
    /**
     * Add one or more CodePipeline Actions
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     */
    addActions(...actions) {
        for (const action of actions) {
            this.pipelineStage.addAction(action);
        }
    }
    /**
     * Return the runOrder number necessary to run the next Action in sequence with the rest
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    nextSequentialRunOrder(count = 1) {
        const ret = this._nextSequentialRunOrder;
        this._nextSequentialRunOrder += count;
        return ret;
    }
    /**
     * Whether this Stage contains an action to deploy the given stack, identified by its artifact ID
     */
    deploysStack(artifactId) {
        return this.stacksToDeploy.map(s => s.stackArtifact.id).includes(artifactId);
    }
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    prepareStage() {
        // FIXME: Make sure this only gets run once. There seems to be an issue in the reconciliation
        // loop that may trigger this more than once if it throws an error somewhere, and the exception
        // that gets thrown here will then override the actual failure.
        if (this._prepared) {
            return;
        }
        this._prepared = true;
        for (const { prepareRunOrder, stackArtifact, executeRunOrder } of this.stacksToDeploy) {
            const artifact = this.host.stackOutputArtifact(stackArtifact.id);
            this.pipelineStage.addAction(actions_1.DeployCdkStackAction.fromStackArtifact(this, stackArtifact, {
                baseActionName: this.simplifyStackName(stackArtifact.stackName),
                cloudAssemblyInput: this.cloudAssemblyArtifact,
                output: artifact,
                outputFileName: artifact ? 'outputs.json' : undefined,
                prepareRunOrder,
                executeRunOrder,
            }));
        }
    }
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    advanceRunOrderPast(lastUsed) {
        this._nextSequentialRunOrder = Math.max(lastUsed + 1, this._nextSequentialRunOrder);
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simplifyStackName(s) {
        return stripPrefix(s, `${this.stageName}-`);
    }
    /**
     * Add a security check before the prepare/deploy actions of an CDK stage.
     * The security check consists of two actions:
     *  - CodeBuild Action to check for security changes in a stage
     *  - Manual Approval Action that is auto approved via a Lambda if no security changes detected
     */
    addSecurityCheck(appStage, options) {
        const { cdkDiffProject } = this.getApplicationSecurityCheck();
        const notificationTopic = options?.securityNotificationTopic ?? this.securityNotificationTopic;
        notificationTopic?.grantPublish(cdkDiffProject);
        const appStageName = appStage.stageName;
        const approveActionName = `${appStageName}ManualApproval`;
        const diffAction = new aws_codepipeline_actions_1.CodeBuildAction({
            runOrder: this.nextSequentialRunOrder(),
            actionName: `${appStageName}SecurityCheck`,
            input: this.cloudAssemblyArtifact,
            project: cdkDiffProject,
            variablesNamespace: `${appStageName}SecurityCheck`,
            environmentVariables: {
                STAGE_PATH: {
                    value: constructs_1.Node.of(appStage).path,
                    type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                },
                STAGE_NAME: {
                    value: this.stageName,
                    type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                },
                ACTION_NAME: {
                    value: approveActionName,
                    type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                },
                ...notificationTopic ? {
                    NOTIFICATION_ARN: {
                        value: notificationTopic.topicArn,
                        type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    },
                    NOTIFICATION_SUBJECT: {
                        value: `Confirm permission broadening in ${appStageName}`,
                        type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    },
                } : {},
            },
        });
        const approve = new cpactions.ManualApprovalAction({
            actionName: approveActionName,
            runOrder: this.nextSequentialRunOrder(),
            additionalInformation: `#{${appStageName}SecurityCheck.MESSAGE}`,
            externalEntityLink: `#{${appStageName}SecurityCheck.LINK}`,
        });
        this.addActions(diffAction, approve);
    }
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    publishAssetDependencies(stackArtifact) {
        const assetManifests = stackArtifact.dependencies.filter(isAssetManifest);
        for (const manifestArtifact of assetManifests) {
            const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
            for (const entry of manifest.entries) {
                let assetType;
                if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                    assetType = asset_type_1.AssetType.DOCKER_IMAGE;
                }
                else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                    // Don't publish the template for this stack
                    if (entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile) {
                        continue;
                    }
                    assetType = asset_type_1.AssetType.FILE;
                }
                else {
                    throw new Error(`Unrecognized asset type: ${entry.type}`);
                }
                if (!entry.destination.assumeRoleArn) {
                    throw new Error('assumeRoleArn is missing on asset and required');
                }
                this.host.publishAsset({
                    assetManifestPath: manifestArtifact.file,
                    assetId: entry.id.assetId,
                    assetSelector: entry.id.toString(),
                    assetType,
                    assetPublishingRoleArn: entry.destination.assumeRoleArn,
                });
            }
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
CdkStage[_a] = { fqn: "aws-cdk-lib.pipelines.CdkStage", version: "2.74.0" };
exports.CdkStage = CdkStage;
/**
 * A single output of a Stack
 *
 * @deprecated This class is part of the old API. Use the API based on the `CodePipeline` class instead
 */
class StackOutput {
    /**
     * Build a StackOutput from a known artifact and an output name
     */
    constructor(artifactFile, outputName) {
        this.artifactFile = artifactFile;
        this.outputName = outputName;
    }
}
_b = JSII_RTTI_SYMBOL_1;
StackOutput[_b] = { fqn: "aws-cdk-lib.pipelines.StackOutput", version: "2.74.0" };
exports.StackOutput = StackOutput;
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.slice(prefix.length) : s;
}
function isAssetManifest(s) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return s instanceof cxapi.AssetManifestArtifact;
    return s.constructor.name === 'AssetManifestArtifact';
}
//# sourceMappingURL=data:application/json;base64,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