"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const path = require("path");
const codebuild = require("../../../../aws-codebuild");
const codepipeline = require("../../../../aws-codepipeline");
const codepipeline_actions = require("../../../../aws-codepipeline-actions");
const ec2 = require("../../../../aws-ec2");
const core_1 = require("../../../../core");
const _util_1 = require("./_util");
const docker_credentials_1 = require("../../docker-credentials");
const default_codebuild_image_1 = require("../../private/default-codebuild-image");
const fs_1 = require("../../private/fs");
const DEFAULT_OUTPUT_DIR = 'cdk.out';
/**
 * A standard synth with a generated buildspec
 *
 * @deprecated This class is part of the old API. Use the API based on the `CodePipeline` class instead
 */
class SimpleSynthAction {
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options) {
        return new SimpleSynthAction({
            ...options,
            installCommand: options.installCommand ?? 'npm ci',
            synthCommand: options.synthCommand ?? 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
            environment: {
                ...options.environment,
                environmentVariables: {
                    // Need this in case the CDK CLI is not in the 'package.json' of the project,
                    // and 'npx' is going to download it; without this setting, 'npx' will not properly
                    // install the package into the root user's home directory
                    NPM_CONFIG_UNSAFE_PERM: { value: 'true' },
                    ...options.environment?.environmentVariables,
                },
            },
        });
    }
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options) {
        return new SimpleSynthAction({
            ...options,
            installCommand: options.installCommand ?? 'yarn install --frozen-lockfile',
            synthCommand: options.synthCommand ?? 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
            environment: {
                ...options.environment,
                environmentVariables: {
                    // Need this in case the CDK CLI is not in the 'package.json' of the project,
                    // and 'npx' is going to download it; without this setting, 'npx' will not properly
                    // install the package into the root user's home directory
                    NPM_CONFIG_UNSAFE_PERM: { value: 'true' },
                    ...options.environment?.environmentVariables,
                },
            },
        });
    }
    constructor(props) {
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: props.actionName ?? 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...(props.additionalArtifacts ?? []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = props.additionalArtifacts ?? [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Project generated to run the synth command
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after SimpleSynthAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        const buildCommands = this.props.buildCommands ?? [this.props.buildCommand];
        const installCommands = this.props.installCommands ?? [this.props.installCommand];
        const testCommands = this.props.testCommands ?? [];
        const synthCommand = this.props.synthCommand;
        const environment = { buildImage: default_codebuild_image_1.CDKP_DEFAULT_CODEBUILD_IMAGE, ...this.props.environment };
        const osType = (environment.buildImage instanceof codebuild.WindowsBuildImage)
            ? ec2.OperatingSystemType.WINDOWS
            : ec2.OperatingSystemType.LINUX;
        const buildSpec = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: (0, _util_1.filterEmpty)([
                        this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                        ...installCommands,
                        ...(0, docker_credentials_1.dockerCredentialsInstallCommands)(docker_credentials_1.DockerCredentialUsage.SYNTH, this._dockerCredentials, osType),
                    ]),
                },
                build: {
                    commands: (0, _util_1.filterEmpty)([
                        ...buildCommands,
                        ...testCommands,
                        synthCommand,
                    ]),
                },
            },
            artifacts: renderArtifacts(this),
        });
        const environmentVariables = {
            ...(0, _util_1.copyEnvironmentVariables)(...this.props.copyEnvironmentVariables || []),
        };
        const mergedBuildSpec = this.props.buildSpec ? codebuild.mergeBuildSpecs(this.props.buildSpec, buildSpec) : buildSpec;
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = hash(core_1.Stack.of(scope).resolve({
            environment: serializeBuildEnvironment(environment),
            buildSpecString: mergedBuildSpec.toBuildSpec(),
            environmentVariables,
        }));
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: this.props.projectName,
            environment,
            vpc: this.props.vpc,
            subnetSelection: this.props.subnetSelection,
            buildSpec: mergedBuildSpec,
            environmentVariables,
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._project = project;
        this._dockerCredentials?.forEach(reg => reg.grantRead(project.grantPrincipal, docker_credentials_1.DockerCredentialUsage.SYNTH));
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...(this.props.additionalArtifacts ?? []).map(a => a.artifact)],
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: {
                ...this.props.environmentVariables,
                _PROJECT_CONFIG_HASH: { value: projectConfigHash },
            },
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': (0, fs_1.toPosixPath)(path.join(self.props.subdirectory ?? '.', DEFAULT_OUTPUT_DIR)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': (0, fs_1.toPosixPath)(path.join(self.props.subdirectory ?? '.', art.directory)),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * The CodeBuild Project's principal
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
    /**
     * Associate one or more Docker registries and associated credentials with the synth action.
     * This will be used to inject installation commands to set up `cdk-assets`,
     * and grant read access to the credentials.
     * @internal
     */
    _addDockerCredentials(dockerCredentials) {
        this._dockerCredentials = dockerCredentials;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SimpleSynthAction[_a] = { fqn: "aws-cdk-lib.pipelines.SimpleSynthAction", version: "2.74.0" };
exports.SimpleSynthAction = SimpleSynthAction;
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
/**
 * Serialize a build environment to data (get rid of constructs & objects), so we can JSON.stringify it
 */
function serializeBuildEnvironment(env) {
    return {
        privileged: env.privileged,
        environmentVariables: env.environmentVariables,
        type: env.buildImage?.type,
        imageId: env.buildImage?.imageId,
        computeType: env.computeType,
        imagePullPrincipalType: env.buildImage?.imagePullPrincipalType,
        secretsManagerArn: env.buildImage?.secretsManagerCredentials?.secretArn,
    };
}
//# sourceMappingURL=data:application/json;base64,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