"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationSecurityCheck = void 0;
const path = require("path");
const codebuild = require("../../../aws-codebuild");
const iam = require("../../../aws-iam");
const lambda = require("../../../aws-lambda");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const default_codebuild_image_1 = require("./default-codebuild-image");
/**
 * A construct containing both the Lambda and CodeBuild Project
 * needed to conduct a security check on any given application stage.
 *
 * The Lambda acts as an auto approving mechanism that should only be
 * triggered when the CodeBuild Project registers no security changes.
 *
 * The CodeBuild Project runs a security diff on the application stage,
 * and exports the link to the console of the project.
 */
class ApplicationSecurityCheck extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        core_1.Tags.of(props.codePipeline).add('SECURITY_CHECK', 'ALLOW_APPROVE', {
            includeResourceTypes: ['AWS::CodePipeline::Pipeline'],
        });
        this.preApproveLambda = new lambda.Function(this, 'CDKPipelinesAutoApprove', {
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_14_X,
            code: lambda.Code.fromAsset(path.resolve(__dirname, 'approve-lambda')),
            timeout: core_1.Duration.minutes(5),
        });
        this.preApproveLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codepipeline:GetPipelineState', 'codepipeline:PutApprovalResult'],
            conditions: {
                StringEquals: {
                    'aws:ResourceTag/SECURITY_CHECK': 'ALLOW_APPROVE',
                },
            },
            resources: ['*'],
        }));
        const invokeLambda = 'aws lambda invoke' +
            ` --function-name ${this.preApproveLambda.functionName}` +
            ' --invocation-type Event' +
            ' --cli-binary-format raw-in-base64-out' +
            ' --payload "$payload"' +
            ' lambda.out';
        const message = [
            'An upcoming change would broaden security changes in $PIPELINE_NAME.',
            'Review and approve the changes in CodePipeline to proceed with the deployment.',
            '',
            'Review the changes in CodeBuild:',
            '',
            '$LINK',
            '',
            'Approve the changes in CodePipeline (stage $STAGE_NAME, action $ACTION_NAME):',
            '',
            '$PIPELINE_LINK',
        ];
        const publishNotification = 'aws sns publish' +
            ' --topic-arn $NOTIFICATION_ARN' +
            ' --subject "$NOTIFICATION_SUBJECT"' +
            ` --message "${message.join('\n')}"`;
        this.cdkDiffProject = new codebuild.Project(this, 'CDKSecurityCheck', {
            environment: {
                buildImage: default_codebuild_image_1.CDKP_DEFAULT_CODEBUILD_IMAGE,
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: 0.2,
                phases: {
                    build: {
                        commands: [
                            'npm install -g aws-cdk',
                            // $CODEBUILD_INITIATOR will always be Code Pipeline and in the form of:
                            // "codepipeline/example-pipeline-name-Xxx"
                            'export PIPELINE_NAME="$(node -pe \'`${process.env.CODEBUILD_INITIATOR}`.split("/")[1]\')"',
                            'payload="$(node -pe \'JSON.stringify({ "PipelineName": process.env.PIPELINE_NAME, "StageName": process.env.STAGE_NAME, "ActionName": process.env.ACTION_NAME })\' )"',
                            // ARN: "arn:aws:codebuild:$region:$account_id:build/$project_name:$project_execution_id$"
                            'ARN=$CODEBUILD_BUILD_ARN',
                            'REGION="$(node -pe \'`${process.env.ARN}`.split(":")[3]\')"',
                            'ACCOUNT_ID="$(node -pe \'`${process.env.ARN}`.split(":")[4]\')"',
                            'PROJECT_NAME="$(node -pe \'`${process.env.ARN}`.split(":")[5].split("/")[1]\')"',
                            'PROJECT_ID="$(node -pe \'`${process.env.ARN}`.split(":")[6]\')"',
                            // Manual Approval adds 'http/https' to the resolved link
                            'export LINK="https://$REGION.console.aws.amazon.com/codesuite/codebuild/$ACCOUNT_ID/projects/$PROJECT_NAME/build/$PROJECT_NAME:$PROJECT_ID/?region=$REGION"',
                            'export PIPELINE_LINK="https://$REGION.console.aws.amazon.com/codesuite/codepipeline/pipelines/$PIPELINE_NAME/view?region=$REGION"',
                            // Run invoke only if cdk diff passes (returns exit code 0)
                            // 0 -> true, 1 -> false
                            ifElse({
                                condition: 'cdk diff -a . --security-only --fail $STAGE_PATH/\\*',
                                thenStatements: [
                                    invokeLambda,
                                    'export MESSAGE="No security-impacting changes detected."',
                                ],
                                elseStatements: [
                                    `[ -z "\${NOTIFICATION_ARN}" ] || ${publishNotification}`,
                                    'export MESSAGE="Deployment would make security-impacting changes. Click the link below to inspect them, then click Approve if all changes are expected."',
                                ],
                            }),
                        ],
                    },
                },
                env: {
                    'exported-variables': [
                        'LINK',
                        'MESSAGE',
                    ],
                },
            }),
        });
        // this is needed to check the status the stacks when doing `cdk diff`
        this.cdkDiffProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: ['*'],
            conditions: {
                'ForAnyValue:StringEquals': {
                    'iam:ResourceTag/aws-cdk:bootstrap-role': ['deploy'],
                },
            },
        }));
        this.preApproveLambda.grantInvoke(this.cdkDiffProject);
    }
}
exports.ApplicationSecurityCheck = ApplicationSecurityCheck;
const ifElse = ({ condition, thenStatements, elseStatements }) => {
    let statement = thenStatements.reduce((acc, ifTrue) => {
        return `${acc} ${ifTrue};`;
    }, `if ${condition}; then`);
    if (elseStatements) {
        statement = elseStatements.reduce((acc, ifFalse) => {
            return `${acc} ${ifFalse};`;
        }, `${statement} else`);
    }
    return `${statement} fi`;
};
//# sourceMappingURL=data:application/json;base64,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