"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DestinationPattern = exports.DestinationIdentifier = exports.DockerImageManifestEntry = exports.FileManifestEntry = exports.AssetManifestReader = void 0;
// FIXME: copied from `ckd-assets`, because this tool needs to read the asset manifest aswell.
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema");
/**
 * A manifest of assets
 */
class AssetManifestReader {
    /**
     * Load an asset manifest from the given file
     */
    static fromFile(fileName) {
        try {
            const obj = cloud_assembly_schema_1.Manifest.loadAssetManifest(fileName);
            return new AssetManifestReader(path.dirname(fileName), obj);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest '${fileName}': ${e.message}`);
        }
    }
    /**
     * Load an asset manifest from the given file or directory
     *
     * If the argument given is a directoy, the default asset file name will be used.
     */
    static fromPath(filePath) {
        let st;
        try {
            st = fs.statSync(filePath);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest at '${filePath}': ${e.message}`);
        }
        if (st.isDirectory()) {
            return AssetManifestReader.fromFile(path.join(filePath, AssetManifestReader.DEFAULT_FILENAME));
        }
        return AssetManifestReader.fromFile(filePath);
    }
    constructor(directory, manifest) {
        this.manifest = manifest;
        this.directory = directory;
    }
    /**
     * Select a subset of assets and destinations from this manifest.
     *
     * Only assets with at least 1 selected destination are retained.
     *
     * If selection is not given, everything is returned.
     */
    select(selection) {
        if (selection === undefined) {
            return this;
        }
        const ret = { version: this.manifest.version, dockerImages: {}, files: {} };
        for (const assetType of ASSET_TYPES) {
            for (const [assetId, asset] of Object.entries(this.manifest[assetType] || {})) {
                const filteredDestinations = filterDict(asset.destinations, (_, destId) => selection.some(sel => sel.matches(new DestinationIdentifier(assetId, destId))));
                if (Object.keys(filteredDestinations).length > 0) {
                    ret[assetType][assetId] = {
                        ...asset,
                        destinations: filteredDestinations,
                    };
                }
            }
        }
        return new AssetManifestReader(this.directory, ret);
    }
    /**
     * Describe the asset manifest as a list of strings
     */
    list() {
        return [
            ...describeAssets('file', this.manifest.files || {}),
            ...describeAssets('docker-image', this.manifest.dockerImages || {}),
        ];
        function describeAssets(type, assets) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets || {})) {
                ret.push(`${assetId} ${type} ${JSON.stringify(asset.source)}`);
                const destStrings = Object.entries(asset.destinations).map(([destId, dest]) => ` ${assetId}:${destId} ${JSON.stringify(dest)}`);
                ret.push(...prefixTreeChars(destStrings, '  '));
            }
            return ret;
        }
    }
    /**
     * List of assets, splat out to destinations
     */
    get entries() {
        return [
            ...makeEntries(this.manifest.files || {}, FileManifestEntry),
            ...makeEntries(this.manifest.dockerImages || {}, DockerImageManifestEntry),
        ];
        function makeEntries(assets, ctor) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets)) {
                for (const [destId, destination] of Object.entries(asset.destinations)) {
                    ret.push(new ctor(new DestinationIdentifier(assetId, destId), asset.source, destination));
                }
            }
            return ret;
        }
    }
}
/**
 * The default name of the asset manifest in a cdk.out directory
 */
AssetManifestReader.DEFAULT_FILENAME = 'assets.json';
exports.AssetManifestReader = AssetManifestReader;
const ASSET_TYPES = ['files', 'dockerImages'];
/**
 * A manifest entry for a file asset
 */
class FileManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'file';
        this.genericSource = source;
    }
}
exports.FileManifestEntry = FileManifestEntry;
/**
 * A manifest entry for a docker image asset
 */
class DockerImageManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'docker-image';
        this.genericSource = source;
    }
}
exports.DockerImageManifestEntry = DockerImageManifestEntry;
/**
 * Identify an asset destination in an asset manifest
 */
class DestinationIdentifier {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return this.destinationId ? `${this.assetId}:${this.destinationId}` : this.assetId;
    }
}
exports.DestinationIdentifier = DestinationIdentifier;
function filterDict(xs, pred) {
    const ret = {};
    for (const [key, value] of Object.entries(xs)) {
        if (pred(value, key)) {
            ret[key] = value;
        }
    }
    return ret;
}
/**
 * A filter pattern for an destination identifier
 */
class DestinationPattern {
    /**
     * Parse a ':'-separated string into an asset/destination identifier
     */
    static parse(s) {
        if (!s) {
            throw new Error('Empty string is not a valid destination identifier');
        }
        const parts = s.split(':').map(x => x !== '*' ? x : undefined);
        if (parts.length === 1) {
            return new DestinationPattern(parts[0]);
        }
        if (parts.length === 2) {
            return new DestinationPattern(parts[0] || undefined, parts[1] || undefined);
        }
        throw new Error(`Asset identifier must contain at most 2 ':'-separated parts, got '${s}'`);
    }
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Whether or not this pattern matches the given identifier
     */
    matches(id) {
        return (this.assetId === undefined || this.assetId === id.assetId)
            && (this.destinationId === undefined || this.destinationId === id.destinationId);
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return `${this.assetId ?? '*'}:${this.destinationId ?? '*'}`;
    }
}
exports.DestinationPattern = DestinationPattern;
/**
 * Prefix box-drawing characters to make lines look like a hanging tree
 */
function prefixTreeChars(xs, prefix = '') {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        const isLast = i === xs.length - 1;
        const boxChar = isLast ? '└' : '├';
        ret.push(`${prefix}${boxChar}${xs[i]}`);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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