"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetSingletonRole = void 0;
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
/**
 * Role which will be reused across asset jobs
 *
 * Has some '*' resources to save IAM policy space, and will not
 * actually add policies that look like policies that were already added.
 */
class AssetSingletonRole extends iam.Role {
    constructor(scope, id, props) {
        super(scope, id, props);
        this._rejectDuplicates = false;
        // Logging permissions
        this.addToPolicy(new iam.PolicyStatement({
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '/aws/codebuild/*',
                })],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        this.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
                'codebuild:BatchPutCodeCoverages',
            ],
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'report-group',
                    resourceName: '*',
                })],
        }));
        // CodeBuild start/stop
        this.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        this._rejectDuplicates = true;
    }
    addToPrincipalPolicy(statement) {
        const json = statement.toStatementJson();
        const acts = JSON.stringify(json.Action);
        // These have already been added with wildcard resources on creation
        const alreadyAdded = [
            '["logs:CreateLogGroup","logs:CreateLogStream","logs:PutLogEvents"]',
            '["codebuild:CreateReportGroup","codebuild:CreateReport","codebuild:UpdateReport","codebuild:BatchPutTestCases","codebuild:BatchPutCodeCoverages"]',
            '["codebuild:BatchGetBuilds","codebuild:StartBuild","codebuild:StopBuild"]',
        ];
        if (this._rejectDuplicates && alreadyAdded.includes(acts)) {
            // Pretend we did it
            return { statementAdded: true, policyDependable: new class {
                } };
        }
        // These are added in duplicate (specifically these come from
        // Project#bindToCodePipeline) -- the original singleton asset role didn't
        // have these, and they're not necessary either, so in order to not cause
        // unnecessary diffs, recognize and drop them there as well.
        if (acts === '["kms:Decrypt","kms:Encrypt","kms:ReEncrypt*","kms:GenerateDataKey*"]') {
            // Pretend we did it
            return { statementAdded: true, policyDependable: new class {
                } };
        }
        return super.addToPrincipalPolicy(statement);
    }
    /**
     * Make sure the Role has sts:AssumeRole permissions to the given ARN
     *
     * Will add a new PolicyStatement to the Role if necessary, otherwise add resources to the existing
     * PolicyStatement.
     *
     * Normally this would have been many `grantAssume()` calls (which would get deduplicated by the
     * policy minimization logic), but we have to account for old pipelines that don't have policy
     * minimization enabled.
     */
    addAssumeRole(roleArn) {
        if (!this._assumeRoleStatement) {
            this._assumeRoleStatement = new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
            });
            this.addToPrincipalPolicy(this._assumeRoleStatement);
        }
        // Chunk into multiple statements to facilitate overflowing into overflow policies.
        // Ideally we would do one ARN per statement and have policy minimization do its job, but that would make
        // the situation A LOT worse if minimization is not enabled (which it isn't by default). So find a middle
        // ground in pre-minimization chunking: reduce overhead while still allowing splitting.
        const MAX_ARNS_PER_STATEMENT = 10;
        this._assumeRoleStatement.addResources(roleArn);
        if (this._assumeRoleStatement.resources.length >= MAX_ARNS_PER_STATEMENT) {
            // Next call to this function will create a new statement
            this._assumeRoleStatement = undefined;
        }
    }
}
exports.AssetSingletonRole = AssetSingletonRole;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXNzZXQtc2luZ2xldG9uLXJvbGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhc3NldC1zaW5nbGV0b24tcm9sZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBd0M7QUFFeEMsd0NBQWlEO0FBR2pEOzs7OztHQUtHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxHQUFHLENBQUMsSUFBSTtJQUk5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBSmxCLHNCQUFpQixHQUFHLEtBQUssQ0FBQztRQU1oQyxzQkFBc0I7UUFDdEIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdkMsU0FBUyxFQUFFLENBQUMsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ25DLE9BQU8sRUFBRSxNQUFNO29CQUNmLFFBQVEsRUFBRSxXQUFXO29CQUNyQixTQUFTLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUI7b0JBQ3hDLFlBQVksRUFBRSxrQkFBa0I7aUJBQ2pDLENBQUMsQ0FBQztZQUNILE9BQU8sRUFBRSxDQUFDLHFCQUFxQixFQUFFLHNCQUFzQixFQUFFLG1CQUFtQixDQUFDO1NBQzlFLENBQUMsQ0FBQyxDQUFDO1FBRUosMEJBQTBCO1FBQzFCLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3ZDLE9BQU8sRUFBRTtnQkFDUCw2QkFBNkI7Z0JBQzdCLHdCQUF3QjtnQkFDeEIsd0JBQXdCO2dCQUN4Qiw2QkFBNkI7Z0JBQzdCLGlDQUFpQzthQUNsQztZQUNELFNBQVMsRUFBRSxDQUFDLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO29CQUNuQyxPQUFPLEVBQUUsV0FBVztvQkFDcEIsUUFBUSxFQUFFLGNBQWM7b0JBQ3hCLFlBQVksRUFBRSxHQUFHO2lCQUNsQixDQUFDLENBQUM7U0FDSixDQUFDLENBQUMsQ0FBQztRQUVKLHVCQUF1QjtRQUN2QixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN2QyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNQLDBCQUEwQjtnQkFDMUIsc0JBQXNCO2dCQUN0QixxQkFBcUI7YUFDdEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUM7S0FDL0I7SUFFTSxvQkFBb0IsQ0FBQyxTQUEwQjtRQUNwRCxNQUFNLElBQUksR0FBRyxTQUFTLENBQUMsZUFBZSxFQUFFLENBQUM7UUFDekMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFekMsb0VBQW9FO1FBQ3BFLE1BQU0sWUFBWSxHQUFHO1lBQ25CLG9FQUFvRTtZQUNwRSxtSkFBbUo7WUFDbkosMkVBQTJFO1NBQzVFLENBQUM7UUFFRixJQUFJLElBQUksQ0FBQyxpQkFBaUIsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ3pELG9CQUFvQjtZQUNwQixPQUFPLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxJQUFJO2lCQUFnQyxFQUFFLENBQUM7U0FDekY7UUFFRCw2REFBNkQ7UUFDN0QsMEVBQTBFO1FBQzFFLHlFQUF5RTtRQUN6RSw0REFBNEQ7UUFDNUQsSUFBSSxJQUFJLEtBQUssdUVBQXVFLEVBQUU7WUFDcEYsb0JBQW9CO1lBQ3BCLE9BQU8sRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLElBQUk7aUJBQWdDLEVBQUUsQ0FBQztTQUN6RjtRQUVELE9BQU8sS0FBSyxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQzlDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYSxDQUFDLE9BQWU7UUFDbEMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtZQUM5QixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUNsRCxPQUFPLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQzthQUM1QixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7U0FDdEQ7UUFFRCxtRkFBbUY7UUFDbkYseUdBQXlHO1FBQ3pHLHlHQUF5RztRQUN6Ryx1RkFBdUY7UUFDdkYsTUFBTSxzQkFBc0IsR0FBRyxFQUFFLENBQUM7UUFFbEMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNoRCxJQUFJLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsTUFBTSxJQUFJLHNCQUFzQixFQUFFO1lBQ3hFLHlEQUF5RDtZQUN6RCxJQUFJLENBQUMsb0JBQW9CLEdBQUcsU0FBUyxDQUFDO1NBQ3ZDO0tBQ0Y7Q0FDRjtBQTFHRCxnREEwR0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnLi4vLi4vLi4vYXdzLWlhbSc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQgfSBmcm9tICcuLi8uLi8uLi9hd3MtaWFtJztcbmltcG9ydCB7IEFybkZvcm1hdCwgU3RhY2sgfSBmcm9tICcuLi8uLi8uLi9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCwgSURlcGVuZGFibGUgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLyoqXG4gKiBSb2xlIHdoaWNoIHdpbGwgYmUgcmV1c2VkIGFjcm9zcyBhc3NldCBqb2JzXG4gKlxuICogSGFzIHNvbWUgJyonIHJlc291cmNlcyB0byBzYXZlIElBTSBwb2xpY3kgc3BhY2UsIGFuZCB3aWxsIG5vdFxuICogYWN0dWFsbHkgYWRkIHBvbGljaWVzIHRoYXQgbG9vayBsaWtlIHBvbGljaWVzIHRoYXQgd2VyZSBhbHJlYWR5IGFkZGVkLlxuICovXG5leHBvcnQgY2xhc3MgQXNzZXRTaW5nbGV0b25Sb2xlIGV4dGVuZHMgaWFtLlJvbGUge1xuICBwcml2YXRlIF9yZWplY3REdXBsaWNhdGVzID0gZmFsc2U7XG4gIHByaXZhdGUgX2Fzc3VtZVJvbGVTdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQgfCB1bmRlZmluZWQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IGlhbS5Sb2xlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIC8vIExvZ2dpbmcgcGVybWlzc2lvbnNcbiAgICB0aGlzLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogW1N0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgIHNlcnZpY2U6ICdsb2dzJyxcbiAgICAgICAgcmVzb3VyY2U6ICdsb2ctZ3JvdXAnLFxuICAgICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgICByZXNvdXJjZU5hbWU6ICcvYXdzL2NvZGVidWlsZC8qJyxcbiAgICAgIH0pXSxcbiAgICAgIGFjdGlvbnM6IFsnbG9nczpDcmVhdGVMb2dHcm91cCcsICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsICdsb2dzOlB1dExvZ0V2ZW50cyddLFxuICAgIH0pKTtcblxuICAgIC8vIENvZGVCdWlsZCByZXBvcnQgZ3JvdXBzXG4gICAgdGhpcy5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdjb2RlYnVpbGQ6Q3JlYXRlUmVwb3J0R3JvdXAnLFxuICAgICAgICAnY29kZWJ1aWxkOkNyZWF0ZVJlcG9ydCcsXG4gICAgICAgICdjb2RlYnVpbGQ6VXBkYXRlUmVwb3J0JyxcbiAgICAgICAgJ2NvZGVidWlsZDpCYXRjaFB1dFRlc3RDYXNlcycsXG4gICAgICAgICdjb2RlYnVpbGQ6QmF0Y2hQdXRDb2RlQ292ZXJhZ2VzJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnY29kZWJ1aWxkJyxcbiAgICAgICAgcmVzb3VyY2U6ICdyZXBvcnQtZ3JvdXAnLFxuICAgICAgICByZXNvdXJjZU5hbWU6ICcqJyxcbiAgICAgIH0pXSxcbiAgICB9KSk7XG5cbiAgICAvLyBDb2RlQnVpbGQgc3RhcnQvc3RvcFxuICAgIHRoaXMuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2NvZGVidWlsZDpCYXRjaEdldEJ1aWxkcycsXG4gICAgICAgICdjb2RlYnVpbGQ6U3RhcnRCdWlsZCcsXG4gICAgICAgICdjb2RlYnVpbGQ6U3RvcEJ1aWxkJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGhpcy5fcmVqZWN0RHVwbGljYXRlcyA9IHRydWU7XG4gIH1cblxuICBwdWJsaWMgYWRkVG9QcmluY2lwYWxQb2xpY3koc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnQpOiBpYW0uQWRkVG9QcmluY2lwYWxQb2xpY3lSZXN1bHQge1xuICAgIGNvbnN0IGpzb24gPSBzdGF0ZW1lbnQudG9TdGF0ZW1lbnRKc29uKCk7XG4gICAgY29uc3QgYWN0cyA9IEpTT04uc3RyaW5naWZ5KGpzb24uQWN0aW9uKTtcblxuICAgIC8vIFRoZXNlIGhhdmUgYWxyZWFkeSBiZWVuIGFkZGVkIHdpdGggd2lsZGNhcmQgcmVzb3VyY2VzIG9uIGNyZWF0aW9uXG4gICAgY29uc3QgYWxyZWFkeUFkZGVkID0gW1xuICAgICAgJ1tcImxvZ3M6Q3JlYXRlTG9nR3JvdXBcIixcImxvZ3M6Q3JlYXRlTG9nU3RyZWFtXCIsXCJsb2dzOlB1dExvZ0V2ZW50c1wiXScsXG4gICAgICAnW1wiY29kZWJ1aWxkOkNyZWF0ZVJlcG9ydEdyb3VwXCIsXCJjb2RlYnVpbGQ6Q3JlYXRlUmVwb3J0XCIsXCJjb2RlYnVpbGQ6VXBkYXRlUmVwb3J0XCIsXCJjb2RlYnVpbGQ6QmF0Y2hQdXRUZXN0Q2FzZXNcIixcImNvZGVidWlsZDpCYXRjaFB1dENvZGVDb3ZlcmFnZXNcIl0nLFxuICAgICAgJ1tcImNvZGVidWlsZDpCYXRjaEdldEJ1aWxkc1wiLFwiY29kZWJ1aWxkOlN0YXJ0QnVpbGRcIixcImNvZGVidWlsZDpTdG9wQnVpbGRcIl0nLFxuICAgIF07XG5cbiAgICBpZiAodGhpcy5fcmVqZWN0RHVwbGljYXRlcyAmJiBhbHJlYWR5QWRkZWQuaW5jbHVkZXMoYWN0cykpIHtcbiAgICAgIC8vIFByZXRlbmQgd2UgZGlkIGl0XG4gICAgICByZXR1cm4geyBzdGF0ZW1lbnRBZGRlZDogdHJ1ZSwgcG9saWN5RGVwZW5kYWJsZTogbmV3IGNsYXNzIGltcGxlbWVudHMgSURlcGVuZGFibGUgeyB9IH07XG4gICAgfVxuXG4gICAgLy8gVGhlc2UgYXJlIGFkZGVkIGluIGR1cGxpY2F0ZSAoc3BlY2lmaWNhbGx5IHRoZXNlIGNvbWUgZnJvbVxuICAgIC8vIFByb2plY3QjYmluZFRvQ29kZVBpcGVsaW5lKSAtLSB0aGUgb3JpZ2luYWwgc2luZ2xldG9uIGFzc2V0IHJvbGUgZGlkbid0XG4gICAgLy8gaGF2ZSB0aGVzZSwgYW5kIHRoZXkncmUgbm90IG5lY2Vzc2FyeSBlaXRoZXIsIHNvIGluIG9yZGVyIHRvIG5vdCBjYXVzZVxuICAgIC8vIHVubmVjZXNzYXJ5IGRpZmZzLCByZWNvZ25pemUgYW5kIGRyb3AgdGhlbSB0aGVyZSBhcyB3ZWxsLlxuICAgIGlmIChhY3RzID09PSAnW1wia21zOkRlY3J5cHRcIixcImttczpFbmNyeXB0XCIsXCJrbXM6UmVFbmNyeXB0KlwiLFwia21zOkdlbmVyYXRlRGF0YUtleSpcIl0nKSB7XG4gICAgICAvLyBQcmV0ZW5kIHdlIGRpZCBpdFxuICAgICAgcmV0dXJuIHsgc3RhdGVtZW50QWRkZWQ6IHRydWUsIHBvbGljeURlcGVuZGFibGU6IG5ldyBjbGFzcyBpbXBsZW1lbnRzIElEZXBlbmRhYmxlIHsgfSB9O1xuICAgIH1cblxuICAgIHJldHVybiBzdXBlci5hZGRUb1ByaW5jaXBhbFBvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ha2Ugc3VyZSB0aGUgUm9sZSBoYXMgc3RzOkFzc3VtZVJvbGUgcGVybWlzc2lvbnMgdG8gdGhlIGdpdmVuIEFSTlxuICAgKlxuICAgKiBXaWxsIGFkZCBhIG5ldyBQb2xpY3lTdGF0ZW1lbnQgdG8gdGhlIFJvbGUgaWYgbmVjZXNzYXJ5LCBvdGhlcndpc2UgYWRkIHJlc291cmNlcyB0byB0aGUgZXhpc3RpbmdcbiAgICogUG9saWN5U3RhdGVtZW50LlxuICAgKlxuICAgKiBOb3JtYWxseSB0aGlzIHdvdWxkIGhhdmUgYmVlbiBtYW55IGBncmFudEFzc3VtZSgpYCBjYWxscyAod2hpY2ggd291bGQgZ2V0IGRlZHVwbGljYXRlZCBieSB0aGVcbiAgICogcG9saWN5IG1pbmltaXphdGlvbiBsb2dpYyksIGJ1dCB3ZSBoYXZlIHRvIGFjY291bnQgZm9yIG9sZCBwaXBlbGluZXMgdGhhdCBkb24ndCBoYXZlIHBvbGljeVxuICAgKiBtaW5pbWl6YXRpb24gZW5hYmxlZC5cbiAgICovXG4gIHB1YmxpYyBhZGRBc3N1bWVSb2xlKHJvbGVBcm46IHN0cmluZykge1xuICAgIGlmICghdGhpcy5fYXNzdW1lUm9sZVN0YXRlbWVudCkge1xuICAgICAgdGhpcy5fYXNzdW1lUm9sZVN0YXRlbWVudCA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydzdHM6QXNzdW1lUm9sZSddLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYWRkVG9QcmluY2lwYWxQb2xpY3kodGhpcy5fYXNzdW1lUm9sZVN0YXRlbWVudCk7XG4gICAgfVxuXG4gICAgLy8gQ2h1bmsgaW50byBtdWx0aXBsZSBzdGF0ZW1lbnRzIHRvIGZhY2lsaXRhdGUgb3ZlcmZsb3dpbmcgaW50byBvdmVyZmxvdyBwb2xpY2llcy5cbiAgICAvLyBJZGVhbGx5IHdlIHdvdWxkIGRvIG9uZSBBUk4gcGVyIHN0YXRlbWVudCBhbmQgaGF2ZSBwb2xpY3kgbWluaW1pemF0aW9uIGRvIGl0cyBqb2IsIGJ1dCB0aGF0IHdvdWxkIG1ha2VcbiAgICAvLyB0aGUgc2l0dWF0aW9uIEEgTE9UIHdvcnNlIGlmIG1pbmltaXphdGlvbiBpcyBub3QgZW5hYmxlZCAod2hpY2ggaXQgaXNuJ3QgYnkgZGVmYXVsdCkuIFNvIGZpbmQgYSBtaWRkbGVcbiAgICAvLyBncm91bmQgaW4gcHJlLW1pbmltaXphdGlvbiBjaHVua2luZzogcmVkdWNlIG92ZXJoZWFkIHdoaWxlIHN0aWxsIGFsbG93aW5nIHNwbGl0dGluZy5cbiAgICBjb25zdCBNQVhfQVJOU19QRVJfU1RBVEVNRU5UID0gMTA7XG5cbiAgICB0aGlzLl9hc3N1bWVSb2xlU3RhdGVtZW50LmFkZFJlc291cmNlcyhyb2xlQXJuKTtcbiAgICBpZiAodGhpcy5fYXNzdW1lUm9sZVN0YXRlbWVudC5yZXNvdXJjZXMubGVuZ3RoID49IE1BWF9BUk5TX1BFUl9TVEFURU1FTlQpIHtcbiAgICAgIC8vIE5leHQgY2FsbCB0byB0aGlzIGZ1bmN0aW9uIHdpbGwgY3JlYXRlIGEgbmV3IHN0YXRlbWVudFxuICAgICAgdGhpcy5fYXNzdW1lUm9sZVN0YXRlbWVudCA9IHVuZGVmaW5lZDtcbiAgICB9XG4gIH1cbn1cbiJdfQ==