"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.limitIdentifierLength = exports.stackVariableNamespace = exports.actionName = exports.hash = void 0;
const crypto = require("crypto");
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
exports.hash = hash;
function actionName(node, parent) {
    const names = node.ancestorPath(parent).map(n => n.id).map(sanitizeName);
    // Something slightly complicated here:
    //
    // Because we'll have structured action names like:
    //
    //     'VeryLongStackName.Prepare', 'VeryLongStackName.Deploy'
    //
    // it would be shitty if cut and hashed them differently:
    //
    //     'VeryLongSAAAAA.Prepare', 'VeryLonBBBBBme.Deploy'
    //
    // wouldn't sort and comprehend nicely. We will therefore trim each component individually.
    const totalMax = 100; // Max length of everything
    // No need to do anything
    if (names.join('.').length <= totalMax) {
        return names.join('.');
    }
    const componentMin = 15; // Can't really go shorter than this, becomes unreadable
    const dots = names.length - 1;
    const maxLength = Math.max(componentMin, Math.floor((totalMax - dots) / names.length));
    const trimmedNames = names.map(name => limitIdentifierLength(name, maxLength));
    return limitIdentifierLength(trimmedNames.join('.'), totalMax); // Final trim in case we couldn't make it
}
exports.actionName = actionName;
function stackVariableNamespace(stack) {
    return limitIdentifierLength(stack.stackArtifactId, 100);
}
exports.stackVariableNamespace = stackVariableNamespace;
function sanitizeName(x) {
    return x.replace(/[^A-Za-z0-9.@\-_]/g, '_');
}
/**
 * Makes sure the given identifier length does not exceed N characters
 *
 * Replaces characters in the middle (to leave the start and end identifiable) and replaces
 * them with a hash to prevent collissions.
 */
function limitIdentifierLength(s, n) {
    if (s.length <= n) {
        return s;
    }
    const h = hash(s).slice(0, 8);
    const mid = Math.floor((n - h.length) / 2);
    return s.slice(0, mid) + h + s.slice(-mid);
}
exports.limitIdentifierLength = limitIdentifierLength;
//# sourceMappingURL=data:application/json;base64,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