"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topologicalSort = void 0;
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 *
 * Returns tranches of elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => e.dependencies.every(d => !remaining.has(d)));
        selectable.sort((a, b) => a.key < b.key ? -1 : b.key < a.key ? 1 : 0);
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(', ')}`);
        }
        ret.push(selectable.map(s => s.element));
        for (const selected of selectable) {
            remaining.delete(selected.key);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,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