"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.partitionInformation = exports.regionsBefore = exports.before = exports.AWS_SERVICES = exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES = exports.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN = exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = void 0;
/**
 * After this point, S3 website domains look like `s3-website.REGION.s3.amazonaws.com`
 *
 * Before this point, S3 website domains look like `s3-website-REGION.s3.amazonaws.com`.
 */
exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = Symbol('S3_WEBSITE_REGIONAL_SUBDOMAIN');
/**
 * After this point, all regions in the 'aws' partition are opt-in.
 */
exports.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN = Symbol('CLASSIC_PARTITION_BECOMES_OPT_IN');
/**
 * List of AWS region, ordered by launch date (oldest to newest)
 *
 * The significance of this is that standards and conventions change over time.
 * Generally, as rules are changed they only apply to new regions, and old
 * regions are left as-is.
 *
 * We mix the list of regions with a list of rules that were introduced over
 * time (rules are symbols).
 *
 * Therefore, if we want to know if a rule applies to a certain region, we
 * only need to check its position in the list and compare it to when a
 * rule was introduced.
 */
exports.AWS_REGIONS_AND_RULES = [
    'us-east-1',
    'eu-west-1',
    'us-west-1',
    'ap-southeast-1',
    'ap-northeast-1',
    'us-gov-west-1',
    'us-west-2',
    'sa-east-1',
    'ap-southeast-2',
    exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN,
    'cn-north-1',
    'us-iso-east-1',
    'eu-central-1',
    'ap-northeast-2',
    'ap-south-1',
    'us-east-2',
    'ca-central-1',
    'eu-west-2',
    'us-isob-east-1',
    'cn-northwest-1',
    'eu-west-3',
    'ap-northeast-3',
    'us-gov-east-1',
    'eu-north-1',
    exports.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN,
    'ap-east-1',
    'me-south-1',
    'eu-south-1',
    'af-south-1',
    'us-iso-west-1',
    'eu-south-2',
    'ap-southeast-3',
    'me-central-1',
    'ap-south-2',
    'eu-central-2', // Europe (Zurich)
];
/**
 * The names of all (known) AWS regions
 *
 * Not in the list ==> no built-in data for that region.
 */
exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES
    .filter((x) => typeof x === 'string')
    .sort();
/**
 * Possibly non-exhaustive list of all service names, used to locate service principals.
 *
 * Not in the list ==> default service principal mappings.
 */
exports.AWS_SERVICES = [
    'application-autoscaling',
    'autoscaling',
    'codedeploy',
    'ec2',
    'events',
    'lambda',
    'logs',
    's3',
    'ssm',
    'sns',
    'sqs',
    'states',
].sort();
/**
 * Whether or not a region predates a given rule (or region).
 *
 * Unknown region => we have to assume no.
 */
function before(region, ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${String(ruleOrRegion)}`);
    }
    const regionIx = exports.AWS_REGIONS_AND_RULES.indexOf(region);
    return regionIx === -1 ? false : regionIx < ruleIx;
}
exports.before = before;
/**
 * Return all regions before a given rule was introduced (or region)
 */
function regionsBefore(ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${String(ruleOrRegion)}`);
    }
    return exports.AWS_REGIONS_AND_RULES.slice(0, ruleIx)
        .filter((entry) => typeof entry === 'string')
        .sort();
}
exports.regionsBefore = regionsBefore;
const PARTITION_MAP = {
    'default': { partition: 'aws', domainSuffix: 'amazonaws.com' },
    'cn-': { partition: 'aws-cn', domainSuffix: 'amazonaws.com.cn' },
    'us-gov-': { partition: 'aws-us-gov', domainSuffix: 'amazonaws.com' },
    'us-iso-': { partition: 'aws-iso', domainSuffix: 'c2s.ic.gov' },
    'us-isob-': { partition: 'aws-iso-b', domainSuffix: 'sc2s.sgov.gov' },
};
function partitionInformation(region) {
    for (const [prefix, info] of Object.entries(PARTITION_MAP)) {
        if (region.startsWith(prefix)) {
            return info;
        }
    }
    return PARTITION_MAP.default;
}
exports.partitionInformation = partitionInformation;
//# sourceMappingURL=data:application/json;base64,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