'''
# Amazon EC2 Auto Scaling Construct Library

This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.

## Auto Scaling Group

An `AutoScalingGroup` represents a number of instances on which you run your code. You
pick the size of the fleet, the instance type and the OS image:

```python
# vpc: ec2.Vpc


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO),

    # The latest Amazon Linux image of a particular generation
    machine_image=ec2.MachineImage.latest_amazon_linux(
        generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2
    )
)
```

NOTE: AutoScalingGroup has an property called `allowAllOutbound` (allowing the instances to contact the
internet) which is set to `true` by default. Be sure to set this to `false`  if you don't want
your instances to be able to start arbitrary connections. Alternatively, you can specify an existing security
group to attach to the instances that are launched, rather than have the group create a new one.

```python
# vpc: ec2.Vpc


my_security_group = ec2.SecurityGroup(self, "SecurityGroup", vpc=vpc)
autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO),
    machine_image=ec2.MachineImage.latest_amazon_linux(
        generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2
    ),
    security_group=my_security_group
)
```

Alternatively you can create an `AutoScalingGroup` from a `LaunchTemplate`:

```python
# vpc: ec2.Vpc
# launch_template: ec2.LaunchTemplate


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    launch_template=launch_template
)
```

To launch a mixture of Spot and on-demand instances, and/or with multiple instance types, you can create an `AutoScalingGroup` from a MixedInstancesPolicy:

```python
# vpc: ec2.Vpc
# launch_template1: ec2.LaunchTemplate
# launch_template2: ec2.LaunchTemplate


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    mixed_instances_policy=autoscaling.MixedInstancesPolicy(
        instances_distribution=autoscaling.InstancesDistribution(
            on_demand_percentage_above_base_capacity=50
        ),
        launch_template=launch_template1,
        launch_template_overrides=[autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t3.micro")), autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t3a.micro")), autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t4g.micro"), launch_template=launch_template2)]
    )
)
```

## Machine Images (AMIs)

AMIs control the OS that gets launched when you start your EC2 instance. The EC2
library contains constructs to select the AMI you want to use.

Depending on the type of AMI, you select it a different way.

The latest version of Amazon Linux and Microsoft Windows images are
selectable by instantiating one of these classes:

```python
# Pick a Windows edition to use
windows = ec2.WindowsImage(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)

# Pick the right Amazon Linux edition. All arguments shown are optional
# and will default to these values when omitted.
amzn_linux = ec2.AmazonLinuxImage(
    generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
    edition=ec2.AmazonLinuxEdition.STANDARD,
    virtualization=ec2.AmazonLinuxVirt.HVM,
    storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE
)

# For other custom (Linux) images, instantiate a `GenericLinuxImage` with
# a map giving the AMI to in for each region:

linux = ec2.GenericLinuxImage({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})
```

> NOTE: The Amazon Linux images selected will be cached in your `cdk.json`, so that your
> AutoScalingGroups don't automatically change out from under you when you're making unrelated
> changes. To update to the latest version of Amazon Linux, remove the cache entry from the `context`
> section of your `cdk.json`.
>
> We will add command-line options to make this step easier in the future.

## AutoScaling Instance Counts

AutoScalingGroups make it possible to raise and lower the number of instances in the group,
in response to (or in advance of) changes in workload.

When you create your AutoScalingGroup, you specify a `minCapacity` and a
`maxCapacity`. AutoScaling policies that respond to metrics will never go higher
or lower than the indicated capacity (but scheduled scaling actions might, see
below).

There are three ways to scale your capacity:

* **In response to a metric** (also known as step scaling); for example, you
  might want to scale out if the CPU usage across your cluster starts to rise,
  and scale in when it drops again.
* **By trying to keep a certain metric around a given value** (also known as
  target tracking scaling); you might want to automatically scale out and in to
  keep your CPU usage around 50%.
* **On a schedule**; you might want to organize your scaling around traffic
  flows you expect, by scaling out in the morning and scaling in in the
  evening.

The general pattern of autoscaling will look like this:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    min_capacity=5,
    max_capacity=100
)
```

### Step Scaling

This type of scaling scales in and out in deterministics steps that you
configure, in response to metric values. For example, your scaling strategy to
scale in response to a metric that represents your average worker pool usage
might look like this:

```plaintext
 Scaling        -1          (no change)          +1       +3
            │        │                       │        │        │
            ├────────┼───────────────────────┼────────┼────────┤
            │        │                       │        │        │
Worker use  0%      10%                     50%       70%     100%
```

(Note that this is not necessarily a recommended scaling strategy, but it's
a possible one. You will have to determine what thresholds are right for you).

Note that in order to set up this scaling strategy, you will have to emit a
metric representing your worker utilization from your instances. After that,
you would configure the scaling something like this:

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


worker_utilization_metric = cloudwatch.Metric(
    namespace="MyService",
    metric_name="WorkerUtilization"
)

auto_scaling_group.scale_on_metric("ScaleToCPU",
    metric=worker_utilization_metric,
    scaling_steps=[autoscaling.ScalingInterval(upper=10, change=-1), autoscaling.ScalingInterval(lower=50, change=+1), autoscaling.ScalingInterval(lower=70, change=+3)
    ],

    # Change this to AdjustmentType.PERCENT_CHANGE_IN_CAPACITY to interpret the
    # 'change' numbers before as percentages instead of capacity counts.
    adjustment_type=autoscaling.AdjustmentType.CHANGE_IN_CAPACITY
)
```

The AutoScaling construct library will create the required CloudWatch alarms and
AutoScaling policies for you.

### Target Tracking Scaling

This type of scaling scales in and out in order to keep a metric around a value
you prefer. There are four types of predefined metrics you can track, or you can
choose to track a custom metric. If you do choose to track a custom metric,
be aware that the metric has to represent instance utilization in some way
(AutoScaling will scale out if the metric is higher than the target, and scale
in if the metric is lower than the target).

If you configure multiple target tracking policies, AutoScaling will use the
one that yields the highest capacity.

The following example scales to keep the CPU usage of your instances around
50% utilization:

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


auto_scaling_group.scale_on_cpu_utilization("KeepSpareCPU",
    target_utilization_percent=50
)
```

To scale on average network traffic in and out of your instances:

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


auto_scaling_group.scale_on_incoming_bytes("LimitIngressPerInstance",
    target_bytes_per_second=10 * 1024 * 1024
)
auto_scaling_group.scale_on_outgoing_bytes("LimitEgressPerInstance",
    target_bytes_per_second=10 * 1024 * 1024
)
```

To scale on the average request count per instance (only works for
AutoScalingGroups that have been attached to Application Load
Balancers):

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


auto_scaling_group.scale_on_request_count("LimitRPS",
    target_requests_per_second=1000
)
```

### Scheduled Scaling

This type of scaling is used to change capacities based on time. It works by
changing `minCapacity`, `maxCapacity` and `desiredCapacity` of the
AutoScalingGroup, and so can be used for two purposes:

* Scale in and out on a schedule by setting the `minCapacity` high or
  the `maxCapacity` low.
* Still allow the regular scaling actions to do their job, but restrict
  the range they can scale over (by setting both `minCapacity` and
  `maxCapacity` but changing their range over time).

A schedule is expressed as a cron expression. The `Schedule` class has a `cron` method to help build cron expressions.

The following example scales the fleet out in the morning, going back to natural
scaling (all the way down to 1 instance if necessary) at night:

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


auto_scaling_group.scale_on_schedule("PrescaleInTheMorning",
    schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
    min_capacity=20
)

auto_scaling_group.scale_on_schedule("AllowDownscalingAtNight",
    schedule=autoscaling.Schedule.cron(hour="20", minute="0"),
    min_capacity=1
)
```

### Block Devices

This type specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

#### GP3 Volumes

You can only specify the `throughput` on GP3 volumes.

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,
    block_devices=[autoscaling.BlockDevice(
        device_name="gp3-volume",
        volume=autoscaling.BlockDeviceVolume.ebs(15,
            volume_type=autoscaling.EbsDeviceVolumeType.GP3,
            throughput=125
        )
    )
    ]
)
```

## Configuring Instances using CloudFormation Init

It is possible to use the CloudFormation Init mechanism to configure the
instances in the AutoScalingGroup. You can write files to it, run commands,
start services, etc. See the documentation of
[AWS::CloudFormation::Init](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html)
and the documentation of CDK's `aws-ec2` library for more information.

When you specify a CloudFormation Init configuration for an AutoScalingGroup:

* you *must* also specify `signals` to configure how long CloudFormation
  should wait for the instances to successfully configure themselves.
* you *should* also specify an `updatePolicy` to configure how instances
  should be updated when the AutoScalingGroup is updated (for example,
  when the AMI is updated). If you don't specify an update policy, a *rolling
  update* is chosen by default.

Here's an example of using CloudFormation Init to write a file to the
instance hosts on startup:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    init=ec2.CloudFormationInit.from_elements(
        ec2.InitFile.from_string("/etc/my_instance", "This got written during instance startup")),
    signals=autoscaling.Signals.wait_for_all(
        timeout=Duration.minutes(10)
    )
)
```

## Signals

In normal operation, CloudFormation will send a Create or Update command to
an AutoScalingGroup and proceed with the rest of the deployment without waiting
for the *instances in the AutoScalingGroup*.

Configure `signals` to tell CloudFormation to wait for a specific number of
instances in the AutoScalingGroup to have been started (or failed to start)
before moving on. An instance is supposed to execute the
[`cfn-signal`](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-signal.html)
program as part of its startup to indicate whether it was started
successfully or not.

If you use CloudFormation Init support (described in the previous section),
the appropriate call to `cfn-signal` is automatically added to the
AutoScalingGroup's UserData. If you don't use the `signals` directly, you are
responsible for adding such a call yourself.

The following type of `Signals` are available:

* `Signals.waitForAll([options])`: wait for all of `desiredCapacity` amount of instances
  to have started (recommended).
* `Signals.waitForMinCapacity([options])`: wait for a `minCapacity` amount of instances
  to have started (use this if waiting for all instances takes too long and you are happy
  with a minimum count of healthy hosts).
* `Signals.waitForCount(count, [options])`: wait for a specific amount of instances to have
  started.

There are two `options` you can configure:

* `timeout`: maximum time a host startup is allowed to take. If a host does not report
  success within this time, it is considered a failure. Default is 5 minutes.
* `minSuccessPercentage`: percentage of hosts that needs to be healthy in order for the
  update to succeed. If you set this value lower than 100, some percentage of hosts may
  report failure, while still considering the deployment a success. Default is 100%.

## Update Policy

The *update policy* describes what should happen to running instances when the definition
of the AutoScalingGroup is changed. For example, if you add a command to the UserData
of an AutoScalingGroup, do the existing instances get replaced with new instances that
have executed the new UserData? Or do the "old" instances just keep on running?

It is recommended to always use an update policy, otherwise the current state of your
instances also depends the previous state of your instances, rather than just on your
source code. This degrades the reproducibility of your deployments.

The following update policies are available:

* `UpdatePolicy.none()`: leave existing instances alone (not recommended).
* `UpdatePolicy.rollingUpdate([options])`: progressively replace the existing
  instances with new instances, in small batches. At any point in time,
  roughly the same amount of total instances will be running. If the deployment
  needs to be rolled back, the fresh instances will be replaced with the "old"
  configuration again.
* `UpdatePolicy.replacingUpdate([options])`: build a completely fresh copy
  of the new AutoScalingGroup next to the old one. Once the AutoScalingGroup
  has been successfully created (and the instances started, if `signals` is
  configured on the AutoScalingGroup), the old AutoScalingGroup is deleted.
  If the deployment needs to be rolled back, the new AutoScalingGroup is
  deleted and the old one is left unchanged.

## Allowing Connections

See the documentation of the `@aws-cdk/aws-ec2` package for more information
about allowing connections between resources backed by instances.

## Max Instance Lifetime

To enable the max instance lifetime support, specify `maxInstanceLifetime` property
for the `AutoscalingGroup` resource. The value must be between 7 and 365 days(inclusive).
To clear a previously set value, leave this property undefined.

## Instance Monitoring

To disable detailed instance monitoring, specify `instanceMonitoring` property
for the `AutoscalingGroup` resource as `Monitoring.BASIC`. Otherwise detailed monitoring
will be enabled.

## Monitoring Group Metrics

Group metrics are used to monitor group level properties; they describe the group rather than any of its instances (e.g GroupMaxSize, the group maximum size). To enable group metrics monitoring, use the `groupMetrics` property.
All group metrics are reported in a granularity of 1 minute at no additional charge.

See [EC2 docs](https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-monitoring.html#as-group-metrics) for a list of all available group metrics.

To enable group metrics monitoring using the `groupMetrics` property:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


# Enable monitoring of all group metrics
autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    group_metrics=[autoscaling.GroupMetrics.all()]
)

# Enable monitoring for a subset of group metrics
autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    group_metrics=[autoscaling.GroupMetrics(autoscaling.GroupMetric.MIN_SIZE, autoscaling.GroupMetric.MAX_SIZE)]
)
```

## Termination policies

Auto Scaling uses [termination policies](https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html)
to determine which instances it terminates first during scale-in events. You
can specify one or more termination policies with the `terminationPolicies`
property:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    termination_policies=[autoscaling.TerminationPolicy.OLDEST_INSTANCE, autoscaling.TerminationPolicy.DEFAULT
    ]
)
```

## Protecting new instances from being terminated on scale-in

By default, Auto Scaling can terminate an instance at any time after launch when
scaling in an Auto Scaling Group, subject to the group's [termination
policy](https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html).

However, you may wish to protect newly-launched instances from being scaled in
if they are going to run critical applications that should not be prematurely
terminated. EC2 Capacity Providers for Amazon ECS requires this attribute be
set to `true`.

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    new_instances_protected_from_scale_in=True
)
```

## Configuring Capacity Rebalancing

Indicates whether Capacity Rebalancing is enabled. Otherwise, Capacity Rebalancing is disabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. For more information, see [Use Capacity Rebalancing to handle Amazon EC2 Spot Interruptions](https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-capacity-rebalancing.html) in the in the Amazon EC2 Auto Scaling User Guide.

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    capacity_rebalance=True
)
```

## Connecting to your instances using SSM Session Manager

SSM Session Manager makes it possible to connect to your instances from the
AWS Console, without preparing SSH keys.

To do so, you need to:

* Use an image with [SSM agent](https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html) installed
  and configured. [Many images come with SSM Agent
  preinstalled](https://docs.aws.amazon.com/systems-manager/latest/userguide/ami-preinstalled-agent.html), otherwise you
  may need to manually put instructions to [install SSM
  Agent](https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-manual-agent-install.html) into your
  instance's UserData or use EC2 Init).
* Create the AutoScalingGroup with `ssmSessionPermissions: true`.

If these conditions are met, you can connect to the instance from the EC2 Console. Example:

```python
# vpc: ec2.Vpc


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MICRO),

    # Amazon Linux 2 comes with SSM Agent by default
    machine_image=ec2.MachineImage.latest_amazon_linux(
        generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2
    ),

    # Turn on SSM
    ssm_session_permissions=True
)
```

## Configuring Instance Metadata Service (IMDS)

### Toggling IMDSv1

You can configure [EC2 Instance Metadata Service](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html) options to either
allow both IMDSv1 and IMDSv2 or enforce IMDSv2 when interacting with the IMDS.

To do this for a single `AutoScalingGroup`, you can use set the `requireImdsv2` property.
The example below demonstrates IMDSv2 being required on a single `AutoScalingGroup`:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    require_imdsv2=True
)
```

You can also use `AutoScalingGroupRequireImdsv2Aspect` to apply the operation to multiple AutoScalingGroups.
The example below demonstrates the `AutoScalingGroupRequireImdsv2Aspect` being used to require IMDSv2 for all AutoScalingGroups in a stack:

```python
aspect = autoscaling.AutoScalingGroupRequireImdsv2Aspect()

Aspects.of(self).add(aspect)
```

## Warm Pool

Auto Scaling offers [a warm pool](https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html) which gives an ability to decrease latency for applications that have exceptionally long boot times. You can create a warm pool with default parameters as below:

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


auto_scaling_group.add_warm_pool()
```

You can also customize a warm pool by configuring parameters:

```python
# auto_scaling_group: autoscaling.AutoScalingGroup


auto_scaling_group.add_warm_pool(
    min_size=1,
    reuse_on_scale_in=True
)
```

### Default Instance Warming

You can use the default instance warmup feature to improve the Amazon CloudWatch metrics used for dynamic scaling.
When default instance warmup is not enabled, each instance starts contributing usage data to the aggregated metrics
as soon as the instance reaches the InService state. However, if you enable default instance warmup, this lets
your instances finish warming up before they contribute the usage data.

To optimize the performance of scaling policies that scale continuously, such as target tracking and step scaling
policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds.

To set up Default Instance Warming for an autoscaling group, simply pass it in as a prop

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    default_instance_warmup=Duration.seconds(5)
)
```

## Future work

* [ ] CloudWatch Events (impossible to add currently as the AutoScalingGroup ARN is
  necessary to make this rule and this cannot be accessed from CloudFormation).
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnCreationPolicy as _CfnCreationPolicy_d904f690,
    CfnResource as _CfnResource_9df397a6,
    Duration as _Duration_4839e8c3,
    IAspect as _IAspect_118c810a,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    Resource as _Resource_45bc6135,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_cloudwatch import Alarm as _Alarm_9fbab1f1, IMetric as _IMetric_c7fd29de
from ..aws_ec2 import (
    CloudFormationInit as _CloudFormationInit_2bb1d1b2,
    Connections as _Connections_0f31fce8,
    IConnectable as _IConnectable_10015a05,
    ILaunchTemplate as _ILaunchTemplate_f32c0fd7,
    IMachineImage as _IMachineImage_0e8bd50b,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    InstanceType as _InstanceType_f64915b9,
    OperatingSystemType as _OperatingSystemType_9224a1fe,
    SubnetSelection as _SubnetSelection_e57d76df,
    UserData as _UserData_b8b32b5e,
)
from ..aws_elasticloadbalancing import (
    ILoadBalancerTarget as _ILoadBalancerTarget_2e052b5c,
    LoadBalancer as _LoadBalancer_a894d40e,
)
from ..aws_elasticloadbalancingv2 import (
    ApplicationTargetGroup as _ApplicationTargetGroup_906fe365,
    IApplicationLoadBalancerTarget as _IApplicationLoadBalancerTarget_fabf9003,
    IApplicationTargetGroup as _IApplicationTargetGroup_57799827,
    INetworkLoadBalancerTarget as _INetworkLoadBalancerTarget_688b169f,
    INetworkTargetGroup as _INetworkTargetGroup_abca2df7,
    LoadBalancerTargetProps as _LoadBalancerTargetProps_4c30a73c,
)
from ..aws_iam import (
    IGrantable as _IGrantable_71c4f5de,
    IPrincipal as _IPrincipal_539bb2fd,
    IRole as _IRole_235f5d8e,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_sns import ITopic as _ITopic_9eca4852


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.AdjustmentTier",
    jsii_struct_bases=[],
    name_mapping={
        "adjustment": "adjustment",
        "lower_bound": "lowerBound",
        "upper_bound": "upperBound",
    },
)
class AdjustmentTier:
    def __init__(
        self,
        *,
        adjustment: jsii.Number,
        lower_bound: typing.Optional[jsii.Number] = None,
        upper_bound: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''An adjustment.

        :param adjustment: What number to adjust the capacity with. The number is interpeted as an added capacity, a new fixed capacity or an added percentage depending on the AdjustmentType value of the StepScalingPolicy. Can be positive or negative.
        :param lower_bound: Lower bound where this scaling tier applies. The scaling tier applies if the difference between the metric value and its alarm threshold is higher than this value. Default: -Infinity if this is the first tier, otherwise the upperBound of the previous tier
        :param upper_bound: Upper bound where this scaling tier applies. The scaling tier applies if the difference between the metric value and its alarm threshold is lower than this value. Default: +Infinity

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            adjustment_tier = autoscaling.AdjustmentTier(
                adjustment=123,
            
                # the properties below are optional
                lower_bound=123,
                upper_bound=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea44f8895fe46835c1f535a501766e354e1166b5f34e774f4402c55622be2c00)
            check_type(argname="argument adjustment", value=adjustment, expected_type=type_hints["adjustment"])
            check_type(argname="argument lower_bound", value=lower_bound, expected_type=type_hints["lower_bound"])
            check_type(argname="argument upper_bound", value=upper_bound, expected_type=type_hints["upper_bound"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "adjustment": adjustment,
        }
        if lower_bound is not None:
            self._values["lower_bound"] = lower_bound
        if upper_bound is not None:
            self._values["upper_bound"] = upper_bound

    @builtins.property
    def adjustment(self) -> jsii.Number:
        '''What number to adjust the capacity with.

        The number is interpeted as an added capacity, a new fixed capacity or an
        added percentage depending on the AdjustmentType value of the
        StepScalingPolicy.

        Can be positive or negative.
        '''
        result = self._values.get("adjustment")
        assert result is not None, "Required property 'adjustment' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def lower_bound(self) -> typing.Optional[jsii.Number]:
        '''Lower bound where this scaling tier applies.

        The scaling tier applies if the difference between the metric
        value and its alarm threshold is higher than this value.

        :default: -Infinity if this is the first tier, otherwise the upperBound of the previous tier
        '''
        result = self._values.get("lower_bound")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def upper_bound(self) -> typing.Optional[jsii.Number]:
        '''Upper bound where this scaling tier applies.

        The scaling tier applies if the difference between the metric
        value and its alarm threshold is lower than this value.

        :default: +Infinity
        '''
        result = self._values.get("upper_bound")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AdjustmentTier(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.AdjustmentType")
class AdjustmentType(enum.Enum):
    '''How adjustment numbers are interpreted.

    :exampleMetadata: infused

    Example::

        # auto_scaling_group: autoscaling.AutoScalingGroup
        
        
        worker_utilization_metric = cloudwatch.Metric(
            namespace="MyService",
            metric_name="WorkerUtilization"
        )
        
        auto_scaling_group.scale_on_metric("ScaleToCPU",
            metric=worker_utilization_metric,
            scaling_steps=[autoscaling.ScalingInterval(upper=10, change=-1), autoscaling.ScalingInterval(lower=50, change=+1), autoscaling.ScalingInterval(lower=70, change=+3)
            ],
        
            # Change this to AdjustmentType.PERCENT_CHANGE_IN_CAPACITY to interpret the
            # 'change' numbers before as percentages instead of capacity counts.
            adjustment_type=autoscaling.AdjustmentType.CHANGE_IN_CAPACITY
        )
    '''

    CHANGE_IN_CAPACITY = "CHANGE_IN_CAPACITY"
    '''Add the adjustment number to the current capacity.

    A positive number increases capacity, a negative number decreases capacity.
    '''
    PERCENT_CHANGE_IN_CAPACITY = "PERCENT_CHANGE_IN_CAPACITY"
    '''Add this percentage of the current capacity to itself.

    The number must be between -100 and 100; a positive number increases
    capacity and a negative number decreases it.
    '''
    EXACT_CAPACITY = "EXACT_CAPACITY"
    '''Make the capacity equal to the exact number given.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.ApplyCloudFormationInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "include_role": "includeRole",
        "include_url": "includeUrl",
        "print_log": "printLog",
    },
)
class ApplyCloudFormationInitOptions:
    def __init__(
        self,
        *,
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        include_role: typing.Optional[builtins.bool] = None,
        include_url: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for applying CloudFormation init to an instance or instance group.

        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Force instance replacement by embedding a config fingerprint. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes and instances will be replaced (given an UpdatePolicy has been configured on the AutoScalingGroup). If ``false``, no such hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instances. If a config update introduces errors, you will not notice until after the CloudFormation deployment successfully finishes and the next instance fails to launch. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param include_role: Include --role argument when running cfn-init and cfn-signal commands. This will be the IAM instance profile attached to the EC2 instance Default: false
        :param include_url: Include --url argument when running cfn-init and cfn-signal commands. This will be the cloudformation endpoint in the deployed region e.g. https://cloudformation.us-east-1.amazonaws.com Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            apply_cloud_formation_init_options = autoscaling.ApplyCloudFormationInitOptions(
                config_sets=["configSets"],
                embed_fingerprint=False,
                ignore_failures=False,
                include_role=False,
                include_url=False,
                print_log=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f79d27ab1c2e9f43613e0ea3c7c572d5fecc8ecbc5fed3187ac25036e8d9e441)
            check_type(argname="argument config_sets", value=config_sets, expected_type=type_hints["config_sets"])
            check_type(argname="argument embed_fingerprint", value=embed_fingerprint, expected_type=type_hints["embed_fingerprint"])
            check_type(argname="argument ignore_failures", value=ignore_failures, expected_type=type_hints["ignore_failures"])
            check_type(argname="argument include_role", value=include_role, expected_type=type_hints["include_role"])
            check_type(argname="argument include_url", value=include_url, expected_type=type_hints["include_url"])
            check_type(argname="argument print_log", value=print_log, expected_type=type_hints["print_log"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if include_role is not None:
            self._values["include_role"] = include_role
        if include_url is not None:
            self._values["include_url"] = include_url
        if print_log is not None:
            self._values["print_log"] = print_log

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Force instance replacement by embedding a config fingerprint.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes and
        instances will be replaced (given an UpdatePolicy has been configured on
        the AutoScalingGroup).

        If ``false``, no such hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instances. If a
        config update introduces errors, you will not notice until after the
        CloudFormation deployment successfully finishes and the next instance
        fails to launch.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_role(self) -> typing.Optional[builtins.bool]:
        '''Include --role argument when running cfn-init and cfn-signal commands.

        This will be the IAM instance profile attached to the EC2 instance

        :default: false
        '''
        result = self._values.get("include_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_url(self) -> typing.Optional[builtins.bool]:
        '''Include --url argument when running cfn-init and cfn-signal commands.

        This will be the cloudformation endpoint in the deployed region
        e.g. https://cloudformation.us-east-1.amazonaws.com

        :default: false
        '''
        result = self._values.get("include_url")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplyCloudFormationInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IAspect_118c810a)
class AutoScalingGroupRequireImdsv2Aspect(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.AutoScalingGroupRequireImdsv2Aspect",
):
    '''Aspect that makes IMDSv2 required on instances deployed by AutoScalingGroups.

    :exampleMetadata: infused

    Example::

        aspect = autoscaling.AutoScalingGroupRequireImdsv2Aspect()
        
        Aspects.of(self).add(aspect)
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="visit")
    def visit(self, node: _constructs_77d1e7e8.IConstruct) -> None:
        '''All aspects can visit an IConstruct.

        :param node: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fbe079e871bd352381d17fc1d2874dbc0f47144cbadb85ef5427d16815427104)
            check_type(argname="argument node", value=node, expected_type=type_hints["node"])
        return typing.cast(None, jsii.invoke(self, "visit", [node]))

    @jsii.member(jsii_name="warn")
    def _warn(
        self,
        node: _constructs_77d1e7e8.IConstruct,
        message: builtins.str,
    ) -> None:
        '''Adds a warning annotation to a node.

        :param node: The scope to add the warning to.
        :param message: The warning message.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54a78dfb222929608a2bea5f551c7a098f071edae832093abb6c4b784334c92b)
            check_type(argname="argument node", value=node, expected_type=type_hints["node"])
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "warn", [node, message]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BaseTargetTrackingProps",
    jsii_struct_bases=[],
    name_mapping={
        "cooldown": "cooldown",
        "disable_scale_in": "disableScaleIn",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
    },
)
class BaseTargetTrackingProps:
    def __init__(
        self,
        *,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Base interface for target tracking props.

        Contains the attributes that are common to target tracking policies,
        except the ones relating to the metric and to the scalable target.

        This interface is reused by more specific target tracking props objects.

        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            
            base_target_tracking_props = autoscaling.BaseTargetTrackingProps(
                cooldown=cdk.Duration.minutes(30),
                disable_scale_in=False,
                estimated_instance_warmup=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87ecdc767b167c69d3a5c4b6b757c48cd8dd03a0f2ae1a31f296e3efafee3021)
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Period after a scaling completes before another scaling activity can start.

        :default: - The default cooldown configured on the AutoScalingGroup.
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the autoscaling group. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        group.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: - Same as the cooldown.
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseTargetTrackingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BasicLifecycleHookProps",
    jsii_struct_bases=[],
    name_mapping={
        "lifecycle_transition": "lifecycleTransition",
        "default_result": "defaultResult",
        "heartbeat_timeout": "heartbeatTimeout",
        "lifecycle_hook_name": "lifecycleHookName",
        "notification_metadata": "notificationMetadata",
        "notification_target": "notificationTarget",
        "role": "role",
    },
)
class BasicLifecycleHookProps:
    def __init__(
        self,
        *,
        lifecycle_transition: "LifecycleTransition",
        default_result: typing.Optional["DefaultResult"] = None,
        heartbeat_timeout: typing.Optional[_Duration_4839e8c3] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target: typing.Optional["ILifecycleHookTarget"] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> None:
        '''Basic properties for a lifecycle hook.

        :param lifecycle_transition: The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. Default: Continue
        :param heartbeat_timeout: Maximum time between calls to RecordLifecycleActionHeartbeat for the hook. If the lifecycle hook times out, perform the action in DefaultResult. Default: - No heartbeat timeout.
        :param lifecycle_hook_name: Name of the lifecycle hook. Default: - Automatically generated name.
        :param notification_metadata: Additional data to pass to the lifecycle hook target. Default: - No metadata.
        :param notification_target: The target of the lifecycle hook. Default: - No target.
        :param role: The role that allows publishing to the notification target. Default: - A role will be created if a target is provided. Otherwise, no role is created.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_iam as iam
            
            # lifecycle_hook_target: autoscaling.ILifecycleHookTarget
            # role: iam.Role
            
            basic_lifecycle_hook_props = autoscaling.BasicLifecycleHookProps(
                lifecycle_transition=autoscaling.LifecycleTransition.INSTANCE_LAUNCHING,
            
                # the properties below are optional
                default_result=autoscaling.DefaultResult.CONTINUE,
                heartbeat_timeout=cdk.Duration.minutes(30),
                lifecycle_hook_name="lifecycleHookName",
                notification_metadata="notificationMetadata",
                notification_target=lifecycle_hook_target,
                role=role
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f991436277d817beb78dbe6f8ee4371d5ba57465042ba2251e4893d5e9d98782)
            check_type(argname="argument lifecycle_transition", value=lifecycle_transition, expected_type=type_hints["lifecycle_transition"])
            check_type(argname="argument default_result", value=default_result, expected_type=type_hints["default_result"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument lifecycle_hook_name", value=lifecycle_hook_name, expected_type=type_hints["lifecycle_hook_name"])
            check_type(argname="argument notification_metadata", value=notification_metadata, expected_type=type_hints["notification_metadata"])
            check_type(argname="argument notification_target", value=notification_target, expected_type=type_hints["notification_target"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "lifecycle_transition": lifecycle_transition,
        }
        if default_result is not None:
            self._values["default_result"] = default_result
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if lifecycle_hook_name is not None:
            self._values["lifecycle_hook_name"] = lifecycle_hook_name
        if notification_metadata is not None:
            self._values["notification_metadata"] = notification_metadata
        if notification_target is not None:
            self._values["notification_target"] = notification_target
        if role is not None:
            self._values["role"] = role

    @builtins.property
    def lifecycle_transition(self) -> "LifecycleTransition":
        '''The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.'''
        result = self._values.get("lifecycle_transition")
        assert result is not None, "Required property 'lifecycle_transition' is missing"
        return typing.cast("LifecycleTransition", result)

    @builtins.property
    def default_result(self) -> typing.Optional["DefaultResult"]:
        '''The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs.

        :default: Continue
        '''
        result = self._values.get("default_result")
        return typing.cast(typing.Optional["DefaultResult"], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Maximum time between calls to RecordLifecycleActionHeartbeat for the hook.

        If the lifecycle hook times out, perform the action in DefaultResult.

        :default: - No heartbeat timeout.
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def lifecycle_hook_name(self) -> typing.Optional[builtins.str]:
        '''Name of the lifecycle hook.

        :default: - Automatically generated name.
        '''
        result = self._values.get("lifecycle_hook_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_metadata(self) -> typing.Optional[builtins.str]:
        '''Additional data to pass to the lifecycle hook target.

        :default: - No metadata.
        '''
        result = self._values.get("notification_metadata")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_target(self) -> typing.Optional["ILifecycleHookTarget"]:
        '''The target of the lifecycle hook.

        :default: - No target.
        '''
        result = self._values.get("notification_target")
        return typing.cast(typing.Optional["ILifecycleHookTarget"], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that allows publishing to the notification target.

        :default: - A role will be created if a target is provided. Otherwise, no role is created.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasicLifecycleHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BasicScheduledActionProps",
    jsii_struct_bases=[],
    name_mapping={
        "schedule": "schedule",
        "desired_capacity": "desiredCapacity",
        "end_time": "endTime",
        "max_capacity": "maxCapacity",
        "min_capacity": "minCapacity",
        "start_time": "startTime",
        "time_zone": "timeZone",
    },
)
class BasicScheduledActionProps:
    def __init__(
        self,
        *,
        schedule: "Schedule",
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
        time_zone: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for a scheduled scaling action.

        :param schedule: When to perform this action. Supports cron expressions. For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        :param desired_capacity: The new desired capacity. At the scheduled time, set the desired capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new desired capacity.
        :param end_time: When this scheduled action expires. Default: - The rule never expires.
        :param max_capacity: The new maximum capacity. At the scheduled time, set the maximum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new maximum capacity.
        :param min_capacity: The new minimum capacity. At the scheduled time, set the minimum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new minimum capacity.
        :param start_time: When this scheduled action becomes active. Default: - The rule is activate immediately.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti). For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones. Default: - UTC

        :exampleMetadata: infused

        Example::

            # auto_scaling_group: autoscaling.AutoScalingGroup
            
            
            auto_scaling_group.scale_on_schedule("PrescaleInTheMorning",
                schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
                min_capacity=20
            )
            
            auto_scaling_group.scale_on_schedule("AllowDownscalingAtNight",
                schedule=autoscaling.Schedule.cron(hour="20", minute="0"),
                min_capacity=1
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__79c9acc50460f89e855de5b2c3b0541c5a6bba8e2a534460115ae251f49449d5)
            check_type(argname="argument schedule", value=schedule, expected_type=type_hints["schedule"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument end_time", value=end_time, expected_type=type_hints["end_time"])
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            check_type(argname="argument start_time", value=start_time, expected_type=type_hints["start_time"])
            check_type(argname="argument time_zone", value=time_zone, expected_type=type_hints["time_zone"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "schedule": schedule,
        }
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if end_time is not None:
            self._values["end_time"] = end_time
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if start_time is not None:
            self._values["start_time"] = start_time
        if time_zone is not None:
            self._values["time_zone"] = time_zone

    @builtins.property
    def schedule(self) -> "Schedule":
        '''When to perform this action.

        Supports cron expressions.

        For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        '''
        result = self._values.get("schedule")
        assert result is not None, "Required property 'schedule' is missing"
        return typing.cast("Schedule", result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''The new desired capacity.

        At the scheduled time, set the desired capacity to the given capacity.

        At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied.

        :default: - No new desired capacity.
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def end_time(self) -> typing.Optional[datetime.datetime]:
        '''When this scheduled action expires.

        :default: - The rule never expires.
        '''
        result = self._values.get("end_time")
        return typing.cast(typing.Optional[datetime.datetime], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''The new maximum capacity.

        At the scheduled time, set the maximum capacity to the given capacity.

        At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied.

        :default: - No new maximum capacity.
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''The new minimum capacity.

        At the scheduled time, set the minimum capacity to the given capacity.

        At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied.

        :default: - No new minimum capacity.
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def start_time(self) -> typing.Optional[datetime.datetime]:
        '''When this scheduled action becomes active.

        :default: - The rule is activate immediately.
        '''
        result = self._values.get("start_time")
        return typing.cast(typing.Optional[datetime.datetime], result)

    @builtins.property
    def time_zone(self) -> typing.Optional[builtins.str]:
        '''Specifies the time zone for a cron expression.

        If a time zone is not provided, UTC is used by default.

        Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti).

        For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones.

        :default: - UTC
        '''
        result = self._values.get("time_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasicScheduledActionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BasicStepScalingPolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "metric": "metric",
        "scaling_steps": "scalingSteps",
        "adjustment_type": "adjustmentType",
        "cooldown": "cooldown",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "evaluation_periods": "evaluationPeriods",
        "metric_aggregation_type": "metricAggregationType",
        "min_adjustment_magnitude": "minAdjustmentMagnitude",
    },
)
class BasicStepScalingPolicyProps:
    def __init__(
        self,
        *,
        metric: _IMetric_c7fd29de,
        scaling_steps: typing.Sequence[typing.Union["ScalingInterval", typing.Dict[builtins.str, typing.Any]]],
        adjustment_type: typing.Optional[AdjustmentType] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        evaluation_periods: typing.Optional[jsii.Number] = None,
        metric_aggregation_type: typing.Optional["MetricAggregationType"] = None,
        min_adjustment_magnitude: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param metric: Metric to scale on.
        :param scaling_steps: The intervals for scaling. Maps a range of metric values to a particular scaling behavior.
        :param adjustment_type: How the adjustment numbers inside 'intervals' are interpreted. Default: ChangeInCapacity
        :param cooldown: Grace period after scaling activity. Default: Default cooldown period on your AutoScalingGroup
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: Same as the cooldown
        :param evaluation_periods: How many evaluation periods of the metric to wait before triggering a scaling action. Raising this value can be used to smooth out the metric, at the expense of slower response times. Default: 1
        :param metric_aggregation_type: Aggregation to apply to all data points over the evaluation periods. Only has meaning if ``evaluationPeriods != 1``. Default: - The statistic from the metric if applicable (MIN, MAX, AVERAGE), otherwise AVERAGE.
        :param min_adjustment_magnitude: Minimum absolute number to adjust capacity with as result of percentage scaling. Only when using AdjustmentType = PercentChangeInCapacity, this number controls the minimum absolute effect size. Default: No minimum scaling effect

        :exampleMetadata: infused

        Example::

            # auto_scaling_group: autoscaling.AutoScalingGroup
            
            
            worker_utilization_metric = cloudwatch.Metric(
                namespace="MyService",
                metric_name="WorkerUtilization"
            )
            
            auto_scaling_group.scale_on_metric("ScaleToCPU",
                metric=worker_utilization_metric,
                scaling_steps=[autoscaling.ScalingInterval(upper=10, change=-1), autoscaling.ScalingInterval(lower=50, change=+1), autoscaling.ScalingInterval(lower=70, change=+3)
                ],
            
                # Change this to AdjustmentType.PERCENT_CHANGE_IN_CAPACITY to interpret the
                # 'change' numbers before as percentages instead of capacity counts.
                adjustment_type=autoscaling.AdjustmentType.CHANGE_IN_CAPACITY
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8cdae3b81f3c040b82cde443a53388281dab2a51aeefc2c4d0d901151ff7b8fd)
            check_type(argname="argument metric", value=metric, expected_type=type_hints["metric"])
            check_type(argname="argument scaling_steps", value=scaling_steps, expected_type=type_hints["scaling_steps"])
            check_type(argname="argument adjustment_type", value=adjustment_type, expected_type=type_hints["adjustment_type"])
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument evaluation_periods", value=evaluation_periods, expected_type=type_hints["evaluation_periods"])
            check_type(argname="argument metric_aggregation_type", value=metric_aggregation_type, expected_type=type_hints["metric_aggregation_type"])
            check_type(argname="argument min_adjustment_magnitude", value=min_adjustment_magnitude, expected_type=type_hints["min_adjustment_magnitude"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "metric": metric,
            "scaling_steps": scaling_steps,
        }
        if adjustment_type is not None:
            self._values["adjustment_type"] = adjustment_type
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup
        if evaluation_periods is not None:
            self._values["evaluation_periods"] = evaluation_periods
        if metric_aggregation_type is not None:
            self._values["metric_aggregation_type"] = metric_aggregation_type
        if min_adjustment_magnitude is not None:
            self._values["min_adjustment_magnitude"] = min_adjustment_magnitude

    @builtins.property
    def metric(self) -> _IMetric_c7fd29de:
        '''Metric to scale on.'''
        result = self._values.get("metric")
        assert result is not None, "Required property 'metric' is missing"
        return typing.cast(_IMetric_c7fd29de, result)

    @builtins.property
    def scaling_steps(self) -> typing.List["ScalingInterval"]:
        '''The intervals for scaling.

        Maps a range of metric values to a particular scaling behavior.
        '''
        result = self._values.get("scaling_steps")
        assert result is not None, "Required property 'scaling_steps' is missing"
        return typing.cast(typing.List["ScalingInterval"], result)

    @builtins.property
    def adjustment_type(self) -> typing.Optional[AdjustmentType]:
        '''How the adjustment numbers inside 'intervals' are interpreted.

        :default: ChangeInCapacity
        '''
        result = self._values.get("adjustment_type")
        return typing.cast(typing.Optional[AdjustmentType], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Grace period after scaling activity.

        :default: Default cooldown period on your AutoScalingGroup
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: Same as the cooldown
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def evaluation_periods(self) -> typing.Optional[jsii.Number]:
        '''How many evaluation periods of the metric to wait before triggering a scaling action.

        Raising this value can be used to smooth out the metric, at the expense
        of slower response times.

        :default: 1
        '''
        result = self._values.get("evaluation_periods")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def metric_aggregation_type(self) -> typing.Optional["MetricAggregationType"]:
        '''Aggregation to apply to all data points over the evaluation periods.

        Only has meaning if ``evaluationPeriods != 1``.

        :default: - The statistic from the metric if applicable (MIN, MAX, AVERAGE), otherwise AVERAGE.
        '''
        result = self._values.get("metric_aggregation_type")
        return typing.cast(typing.Optional["MetricAggregationType"], result)

    @builtins.property
    def min_adjustment_magnitude(self) -> typing.Optional[jsii.Number]:
        '''Minimum absolute number to adjust capacity with as result of percentage scaling.

        Only when using AdjustmentType = PercentChangeInCapacity, this number controls
        the minimum absolute effect size.

        :default: No minimum scaling effect
        '''
        result = self._values.get("min_adjustment_magnitude")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasicStepScalingPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BasicTargetTrackingScalingPolicyProps",
    jsii_struct_bases=[BaseTargetTrackingProps],
    name_mapping={
        "cooldown": "cooldown",
        "disable_scale_in": "disableScaleIn",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "target_value": "targetValue",
        "custom_metric": "customMetric",
        "predefined_metric": "predefinedMetric",
        "resource_label": "resourceLabel",
    },
)
class BasicTargetTrackingScalingPolicyProps(BaseTargetTrackingProps):
    def __init__(
        self,
        *,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        target_value: jsii.Number,
        custom_metric: typing.Optional[_IMetric_c7fd29de] = None,
        predefined_metric: typing.Optional["PredefinedMetric"] = None,
        resource_label: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for a Target Tracking policy that include the metric but exclude the target.

        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        :param target_value: The target value for the metric.
        :param custom_metric: A custom metric for application autoscaling. The metric must track utilization. Scaling out will happen if the metric is higher than the target value, scaling in will happen in the metric is lower than the target value. Exactly one of customMetric or predefinedMetric must be specified. Default: - No custom metric.
        :param predefined_metric: A predefined metric for application autoscaling. The metric must track utilization. Scaling out will happen if the metric is higher than the target value, scaling in will happen in the metric is lower than the target value. Exactly one of customMetric or predefinedMetric must be specified. Default: - No predefined metric.
        :param resource_label: The resource label associated with the predefined metric. Should be supplied if the predefined metric is ALBRequestCountPerTarget, and the format should be: app///targetgroup// Default: - No resource label.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_cloudwatch as cloudwatch
            
            # metric: cloudwatch.Metric
            
            basic_target_tracking_scaling_policy_props = autoscaling.BasicTargetTrackingScalingPolicyProps(
                target_value=123,
            
                # the properties below are optional
                cooldown=cdk.Duration.minutes(30),
                custom_metric=metric,
                disable_scale_in=False,
                estimated_instance_warmup=cdk.Duration.minutes(30),
                predefined_metric=autoscaling.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
                resource_label="resourceLabel"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c32994c9e0f386560e7512b5d8f97eef97db085bd1a2900db6b87ce79f1b935e)
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument target_value", value=target_value, expected_type=type_hints["target_value"])
            check_type(argname="argument custom_metric", value=custom_metric, expected_type=type_hints["custom_metric"])
            check_type(argname="argument predefined_metric", value=predefined_metric, expected_type=type_hints["predefined_metric"])
            check_type(argname="argument resource_label", value=resource_label, expected_type=type_hints["resource_label"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "target_value": target_value,
        }
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup
        if custom_metric is not None:
            self._values["custom_metric"] = custom_metric
        if predefined_metric is not None:
            self._values["predefined_metric"] = predefined_metric
        if resource_label is not None:
            self._values["resource_label"] = resource_label

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Period after a scaling completes before another scaling activity can start.

        :default: - The default cooldown configured on the AutoScalingGroup.
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the autoscaling group. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        group.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: - Same as the cooldown.
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def target_value(self) -> jsii.Number:
        '''The target value for the metric.'''
        result = self._values.get("target_value")
        assert result is not None, "Required property 'target_value' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def custom_metric(self) -> typing.Optional[_IMetric_c7fd29de]:
        '''A custom metric for application autoscaling.

        The metric must track utilization. Scaling out will happen if the metric is higher than
        the target value, scaling in will happen in the metric is lower than the target value.

        Exactly one of customMetric or predefinedMetric must be specified.

        :default: - No custom metric.
        '''
        result = self._values.get("custom_metric")
        return typing.cast(typing.Optional[_IMetric_c7fd29de], result)

    @builtins.property
    def predefined_metric(self) -> typing.Optional["PredefinedMetric"]:
        '''A predefined metric for application autoscaling.

        The metric must track utilization. Scaling out will happen if the metric is higher than
        the target value, scaling in will happen in the metric is lower than the target value.

        Exactly one of customMetric or predefinedMetric must be specified.

        :default: - No predefined metric.
        '''
        result = self._values.get("predefined_metric")
        return typing.cast(typing.Optional["PredefinedMetric"], result)

    @builtins.property
    def resource_label(self) -> typing.Optional[builtins.str]:
        '''The resource label associated with the predefined metric.

        Should be supplied if the predefined metric is ALBRequestCountPerTarget, and the
        format should be:

        app///targetgroup//

        :default: - No resource label.
        '''
        result = self._values.get("resource_label")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasicTargetTrackingScalingPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BindHookTargetOptions",
    jsii_struct_bases=[],
    name_mapping={"lifecycle_hook": "lifecycleHook", "role": "role"},
)
class BindHookTargetOptions:
    def __init__(
        self,
        *,
        lifecycle_hook: "LifecycleHook",
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> None:
        '''Options needed to bind a target to a lifecycle hook.

        [disable-awslint:ref-via-interface] The lifecycle hook to attach to and an IRole to use

        :param lifecycle_hook: The lifecycle hook to attach to. [disable-awslint:ref-via-interface]
        :param role: The role to use when attaching to the lifecycle hook. [disable-awslint:ref-via-interface] Default: : a role is not created unless the target arn is specified

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_iam as iam
            
            # lifecycle_hook: autoscaling.LifecycleHook
            # role: iam.Role
            
            bind_hook_target_options = autoscaling.BindHookTargetOptions(
                lifecycle_hook=lifecycle_hook,
            
                # the properties below are optional
                role=role
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9c16c9108bca19be619521f14265d509dcdf939cfae6bef31656843832552c6a)
            check_type(argname="argument lifecycle_hook", value=lifecycle_hook, expected_type=type_hints["lifecycle_hook"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "lifecycle_hook": lifecycle_hook,
        }
        if role is not None:
            self._values["role"] = role

    @builtins.property
    def lifecycle_hook(self) -> "LifecycleHook":
        '''The lifecycle hook to attach to.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("lifecycle_hook")
        assert result is not None, "Required property 'lifecycle_hook' is missing"
        return typing.cast("LifecycleHook", result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role to use when attaching to the lifecycle hook.

        [disable-awslint:ref-via-interface]

        :default: : a role is not created unless the target arn is specified
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BindHookTargetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.BlockDevice",
    jsii_struct_bases=[],
    name_mapping={"device_name": "deviceName", "volume": "volume"},
)
class BlockDevice:
    def __init__(
        self,
        *,
        device_name: builtins.str,
        volume: "BlockDeviceVolume",
    ) -> None:
        '''Block device.

        :param device_name: The device name exposed to the EC2 instance. Supply a value like ``/dev/sdh``, ``xvdh``.
        :param volume: Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume. Supply a value like ``BlockDeviceVolume.ebs(15)``, ``BlockDeviceVolume.ephemeral(0)``.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            # block_device_volume: autoscaling.BlockDeviceVolume
            
            block_device = autoscaling.BlockDevice(
                device_name="deviceName",
                volume=block_device_volume
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a9dfa8863842f511ea36189510ce7311be635f9198c0e6dda7ba6df86e3802ce)
            check_type(argname="argument device_name", value=device_name, expected_type=type_hints["device_name"])
            check_type(argname="argument volume", value=volume, expected_type=type_hints["volume"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "device_name": device_name,
            "volume": volume,
        }

    @builtins.property
    def device_name(self) -> builtins.str:
        '''The device name exposed to the EC2 instance.

        Supply a value like ``/dev/sdh``, ``xvdh``.

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html
        '''
        result = self._values.get("device_name")
        assert result is not None, "Required property 'device_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def volume(self) -> "BlockDeviceVolume":
        '''Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume.

        Supply a value like ``BlockDeviceVolume.ebs(15)``, ``BlockDeviceVolume.ephemeral(0)``.
        '''
        result = self._values.get("volume")
        assert result is not None, "Required property 'volume' is missing"
        return typing.cast("BlockDeviceVolume", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BlockDevice(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BlockDeviceVolume(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.BlockDeviceVolume",
):
    '''Describes a block device mapping for an EC2 instance or Auto Scaling group.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        # machine_image: ec2.IMachineImage
        
        
        auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
            block_devices=[autoscaling.BlockDevice(
                device_name="gp3-volume",
                volume=autoscaling.BlockDeviceVolume.ebs(15,
                    volume_type=autoscaling.EbsDeviceVolumeType.GP3,
                    throughput=125
                )
            )
            ]
        )
    '''

    def __init__(
        self,
        ebs_device: typing.Optional[typing.Union["EbsDeviceProps", typing.Dict[builtins.str, typing.Any]]] = None,
        virtual_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param ebs_device: EBS device info.
        :param virtual_name: Virtual device name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f623c1a178e9a7c1559125345b70159a2dcd78b6e119979797954060331d419)
            check_type(argname="argument ebs_device", value=ebs_device, expected_type=type_hints["ebs_device"])
            check_type(argname="argument virtual_name", value=virtual_name, expected_type=type_hints["virtual_name"])
        jsii.create(self.__class__, self, [ebs_device, virtual_name])

    @jsii.member(jsii_name="ebs")
    @builtins.classmethod
    def ebs(
        cls,
        volume_size: jsii.Number,
        *,
        encrypted: typing.Optional[builtins.bool] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        throughput: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device.

        :param volume_size: The volume size, in Gibibytes (GiB).
        :param encrypted: Specifies whether the EBS volume is encrypted. Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption Default: false
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1`` The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for ``EbsDeviceVolumeType.IO1``
        :param throughput: The throughput that the volume supports, in MiB/s Takes a minimum of 125 and maximum of 1000. Default: - 125 MiB/s. Only valid on gp3 volumes.
        :param volume_type: The EBS volume type. Default: ``EbsDeviceVolumeType.GP2``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1285ed0c0610fe02eac319ca1d52bf7e091b1afa9d904036d608a853471ca5bd)
            check_type(argname="argument volume_size", value=volume_size, expected_type=type_hints["volume_size"])
        options = EbsDeviceOptions(
            encrypted=encrypted,
            delete_on_termination=delete_on_termination,
            iops=iops,
            throughput=throughput,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebs", [volume_size, options]))

    @jsii.member(jsii_name="ebsFromSnapshot")
    @builtins.classmethod
    def ebs_from_snapshot(
        cls,
        snapshot_id: builtins.str,
        *,
        volume_size: typing.Optional[jsii.Number] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        throughput: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device from an existing snapshot.

        :param snapshot_id: The snapshot ID of the volume to use.
        :param volume_size: The volume size, in Gibibytes (GiB). If you specify volumeSize, it must be equal or greater than the size of the snapshot. Default: - The snapshot size
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1`` The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for ``EbsDeviceVolumeType.IO1``
        :param throughput: The throughput that the volume supports, in MiB/s Takes a minimum of 125 and maximum of 1000. Default: - 125 MiB/s. Only valid on gp3 volumes.
        :param volume_type: The EBS volume type. Default: ``EbsDeviceVolumeType.GP2``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41abea557528986e2f931c9dc068840019d81ce3ad4978c166925d5b707ebfb2)
            check_type(argname="argument snapshot_id", value=snapshot_id, expected_type=type_hints["snapshot_id"])
        options = EbsDeviceSnapshotOptions(
            volume_size=volume_size,
            delete_on_termination=delete_on_termination,
            iops=iops,
            throughput=throughput,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebsFromSnapshot", [snapshot_id, options]))

    @jsii.member(jsii_name="ephemeral")
    @builtins.classmethod
    def ephemeral(cls, volume_index: jsii.Number) -> "BlockDeviceVolume":
        '''Creates a virtual, ephemeral device.

        The name will be in the form ephemeral{volumeIndex}.

        :param volume_index: the volume index. Must be equal or greater than 0
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c264aeefa613dd790f0a99d130fb7667762ea00d7825956158509255e340e8d)
            check_type(argname="argument volume_index", value=volume_index, expected_type=type_hints["volume_index"])
        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ephemeral", [volume_index]))

    @jsii.member(jsii_name="noDevice")
    @builtins.classmethod
    def no_device(cls) -> "BlockDeviceVolume":
        '''Supresses a volume mapping.'''
        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "noDevice", []))

    @builtins.property
    @jsii.member(jsii_name="ebsDevice")
    def ebs_device(self) -> typing.Optional["EbsDeviceProps"]:
        '''EBS device info.'''
        return typing.cast(typing.Optional["EbsDeviceProps"], jsii.get(self, "ebsDevice"))

    @builtins.property
    @jsii.member(jsii_name="virtualName")
    def virtual_name(self) -> typing.Optional[builtins.str]:
        '''Virtual device name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualName"))


@jsii.implements(_IInspectable_c2943556)
class CfnAutoScalingGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup",
):
    '''A CloudFormation ``AWS::AutoScaling::AutoScalingGroup``.

    The ``AWS::AutoScaling::AutoScalingGroup`` resource defines an Amazon EC2 Auto Scaling group, which is a collection of Amazon EC2 instances that are treated as a logical grouping for the purposes of automatic scaling and management.

    For more information about Amazon EC2 Auto Scaling, see the `Amazon EC2 Auto Scaling User Guide <https://docs.aws.amazon.com/autoscaling/ec2/userguide/what-is-amazon-ec2-auto-scaling.html>`_ .
    .. epigraph::

       Amazon EC2 Auto Scaling configures instances launched as part of an Auto Scaling group using either a `launch template <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html>`_ or a launch configuration. We strongly recommend that you do not use launch configurations. They do not provide full functionality for Amazon EC2 Auto Scaling or Amazon EC2. For more information, see `Launch configurations <https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-configurations.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

    :cloudformationResource: AWS::AutoScaling::AutoScalingGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        cfn_auto_scaling_group = autoscaling.CfnAutoScalingGroup(self, "MyCfnAutoScalingGroup",
            max_size="maxSize",
            min_size="minSize",
        
            # the properties below are optional
            auto_scaling_group_name="autoScalingGroupName",
            availability_zones=["availabilityZones"],
            capacity_rebalance=False,
            context="context",
            cooldown="cooldown",
            default_instance_warmup=123,
            desired_capacity="desiredCapacity",
            desired_capacity_type="desiredCapacityType",
            health_check_grace_period=123,
            health_check_type="healthCheckType",
            instance_id="instanceId",
            launch_configuration_name="launchConfigurationName",
            launch_template=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                version="version",
        
                # the properties below are optional
                launch_template_id="launchTemplateId",
                launch_template_name="launchTemplateName"
            ),
            lifecycle_hook_specification_list=[autoscaling.CfnAutoScalingGroup.LifecycleHookSpecificationProperty(
                lifecycle_hook_name="lifecycleHookName",
                lifecycle_transition="lifecycleTransition",
        
                # the properties below are optional
                default_result="defaultResult",
                heartbeat_timeout=123,
                notification_metadata="notificationMetadata",
                notification_target_arn="notificationTargetArn",
                role_arn="roleArn"
            )],
            load_balancer_names=["loadBalancerNames"],
            max_instance_lifetime=123,
            metrics_collection=[autoscaling.CfnAutoScalingGroup.MetricsCollectionProperty(
                granularity="granularity",
        
                # the properties below are optional
                metrics=["metrics"]
            )],
            mixed_instances_policy=autoscaling.CfnAutoScalingGroup.MixedInstancesPolicyProperty(
                launch_template=autoscaling.CfnAutoScalingGroup.LaunchTemplateProperty(
                    launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                        version="version",
        
                        # the properties below are optional
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName"
                    ),
        
                    # the properties below are optional
                    overrides=[autoscaling.CfnAutoScalingGroup.LaunchTemplateOverridesProperty(
                        instance_requirements=autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty(
                            accelerator_count=autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_manufacturers=["acceleratorManufacturers"],
                            accelerator_names=["acceleratorNames"],
                            accelerator_total_memory_mi_b=autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_types=["acceleratorTypes"],
                            allowed_instance_types=["allowedInstanceTypes"],
                            bare_metal="bareMetal",
                            baseline_ebs_bandwidth_mbps=autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            burstable_performance="burstablePerformance",
                            cpu_manufacturers=["cpuManufacturers"],
                            excluded_instance_types=["excludedInstanceTypes"],
                            instance_generations=["instanceGenerations"],
                            local_storage="localStorage",
                            local_storage_types=["localStorageTypes"],
                            memory_gi_bPer_vCpu=autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                                max=123,
                                min=123
                            ),
                            memory_mi_b=autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_bandwidth_gbps=autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_interface_count=autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            on_demand_max_price_percentage_over_lowest_price=123,
                            require_hibernate_support=False,
                            spot_max_price_percentage_over_lowest_price=123,
                            total_local_storage_gb=autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                                max=123,
                                min=123
                            ),
                            v_cpu_count=autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                                max=123,
                                min=123
                            )
                        ),
                        instance_type="instanceType",
                        launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                            version="version",
        
                            # the properties below are optional
                            launch_template_id="launchTemplateId",
                            launch_template_name="launchTemplateName"
                        ),
                        weighted_capacity="weightedCapacity"
                    )]
                ),
        
                # the properties below are optional
                instances_distribution=autoscaling.CfnAutoScalingGroup.InstancesDistributionProperty(
                    on_demand_allocation_strategy="onDemandAllocationStrategy",
                    on_demand_base_capacity=123,
                    on_demand_percentage_above_base_capacity=123,
                    spot_allocation_strategy="spotAllocationStrategy",
                    spot_instance_pools=123,
                    spot_max_price="spotMaxPrice"
                )
            ),
            new_instances_protected_from_scale_in=False,
            notification_configurations=[autoscaling.CfnAutoScalingGroup.NotificationConfigurationProperty(
                topic_arn="topicArn",
        
                # the properties below are optional
                notification_types=["notificationTypes"]
            )],
            placement_group="placementGroup",
            service_linked_role_arn="serviceLinkedRoleArn",
            tags=[autoscaling.CfnAutoScalingGroup.TagPropertyProperty(
                key="key",
                propagate_at_launch=False,
                value="value"
            )],
            target_group_arns=["targetGroupArns"],
            termination_policies=["terminationPolicies"],
            vpc_zone_identifier=["vpcZoneIdentifier"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        max_size: builtins.str,
        min_size: builtins.str,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        capacity_rebalance: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        context: typing.Optional[builtins.str] = None,
        cooldown: typing.Optional[builtins.str] = None,
        default_instance_warmup: typing.Optional[jsii.Number] = None,
        desired_capacity: typing.Optional[builtins.str] = None,
        desired_capacity_type: typing.Optional[builtins.str] = None,
        health_check_grace_period: typing.Optional[jsii.Number] = None,
        health_check_type: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        launch_configuration_name: typing.Optional[builtins.str] = None,
        launch_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.LaunchTemplateSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        lifecycle_hook_specification_list: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.LifecycleHookSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        load_balancer_names: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_instance_lifetime: typing.Optional[jsii.Number] = None,
        metrics_collection: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.MetricsCollectionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        mixed_instances_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.MixedInstancesPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        new_instances_protected_from_scale_in: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        notification_configurations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.NotificationConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        placement_group: typing.Optional[builtins.str] = None,
        service_linked_role_arn: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union["CfnAutoScalingGroup.TagPropertyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        target_group_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        termination_policies: typing.Optional[typing.Sequence[builtins.str]] = None,
        vpc_zone_identifier: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Create a new ``AWS::AutoScaling::AutoScalingGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param max_size: The maximum size of the group. .. epigraph:: With a mixed instances policy that uses instance weighting, Amazon EC2 Auto Scaling may need to go above ``MaxSize`` to meet your capacity requirements. In this event, Amazon EC2 Auto Scaling will never go above ``MaxSize`` by more than your largest instance weight (weights that define how many units each instance contributes to the desired capacity of the group).
        :param min_size: The minimum size of the group.
        :param auto_scaling_group_name: The name of the Auto Scaling group. This name must be unique per Region per account. The name can contain any ASCII character 33 to 126 including most punctuation characters, digits, and upper and lowercased letters. .. epigraph:: You cannot use a colon (:) in the name.
        :param availability_zones: A list of Availability Zones where instances in the Auto Scaling group can be created. Used for launching into the default VPC subnet in each Availability Zone when not using the ``VPCZoneIdentifier`` property, or for attaching a network interface when an existing network interface ID is specified in a launch template.
        :param capacity_rebalance: Indicates whether Capacity Rebalancing is enabled. Otherwise, Capacity Rebalancing is disabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. For more information, see `Use Capacity Rebalancing to handle Amazon EC2 Spot Interruptions <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-capacity-rebalancing.html>`_ in the in the *Amazon EC2 Auto Scaling User Guide* .
        :param context: Reserved.
        :param cooldown: *Only needed if you use simple scaling policies.*. The amount of time, in seconds, between one scaling activity ending and another one starting due to simple scaling policies. For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Default: ``300`` seconds
        :param default_instance_warmup: The amount of time, in seconds, until a new instance is considered to have finished initializing and resource consumption to become stable after it enters the ``InService`` state. During an instance refresh, Amazon EC2 Auto Scaling waits for the warm-up period after it replaces an instance before it moves on to replacing the next instance. Amazon EC2 Auto Scaling also waits for the warm-up period before aggregating the metrics for new instances with existing instances in the Amazon CloudWatch metrics that are used for scaling, resulting in more reliable usage data. For more information, see `Set the default instance warmup for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . .. epigraph:: To manage various warm-up settings at the group level, we recommend that you set the default instance warmup, *even if it is set to 0 seconds* . To remove a value that you previously set, include the property but specify ``-1`` for the value. However, we strongly recommend keeping the default instance warmup enabled by specifying a value of ``0`` or other nominal value. Default: None
        :param desired_capacity: The desired capacity is the initial capacity of the Auto Scaling group at the time of its creation and the capacity it attempts to maintain. It can scale beyond this capacity if you configure automatic scaling. The number must be greater than or equal to the minimum size of the group and less than or equal to the maximum size of the group. If you do not specify a desired capacity when creating the stack, the default is the minimum size of the group. CloudFormation marks the Auto Scaling group as successful (by setting its status to CREATE_COMPLETE) when the desired capacity is reached. However, if a maximum Spot price is set in the launch template or launch configuration that you specified, then desired capacity is not used as a criteria for success. Whether your request is fulfilled depends on Spot Instance capacity and your maximum price.
        :param desired_capacity_type: The unit of measurement for the value specified for desired capacity. Amazon EC2 Auto Scaling supports ``DesiredCapacityType`` for attribute-based instance type selection only. For more information, see `Creating an Auto Scaling group using attribute-based instance type selection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-instance-type-requirements.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . By default, Amazon EC2 Auto Scaling specifies ``units`` , which translates into number of instances. Valid values: ``units`` | ``vcpu`` | ``memory-mib``
        :param health_check_grace_period: The amount of time, in seconds, that Amazon EC2 Auto Scaling waits before checking the health status of an EC2 instance that has come into service and marking it unhealthy due to a failed health check. This is useful if your instances do not immediately pass their health checks after they enter the ``InService`` state. For more information, see `Set the health check grace period for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/health-check-grace-period.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Default: ``0`` seconds
        :param health_check_type: A comma-separated value string of one or more health check types. The valid values are ``EC2`` , ``ELB`` , and ``VPC_LATTICE`` . ``EC2`` is the default health check and cannot be disabled. For more information, see `Health checks for Auto Scaling instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/healthcheck.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Only specify ``EC2`` if you must clear a value that was previously set.
        :param instance_id: The ID of the instance used to base the launch configuration on. For more information, see `Create an Auto Scaling group using an EC2 instance <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-from-instance.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``LaunchConfigurationName`` , don't specify ``InstanceId`` .
        :param launch_configuration_name: The name of the launch configuration to use to launch instances. Required only if you don't specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``InstanceId`` .
        :param launch_template: Information used to specify the launch template and version to use to launch instances. You can alternatively associate a launch template to the Auto Scaling group by specifying a ``MixedInstancesPolicy`` . For more information about creating launch templates, see `Create a launch template for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-template.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you omit this property, you must specify ``MixedInstancesPolicy`` , ``LaunchConfigurationName`` , or ``InstanceId`` .
        :param lifecycle_hook_specification_list: One or more lifecycle hooks to add to the Auto Scaling group before instances are launched.
        :param load_balancer_names: A list of Classic Load Balancers associated with this Auto Scaling group. For Application Load Balancers, Network Load Balancers, and Gateway Load Balancers, specify the ``TargetGroupARNs`` property instead.
        :param max_instance_lifetime: The maximum amount of time, in seconds, that an instance can be in service. The default is null. If specified, the value must be either 0 or a number equal to or greater than 86,400 seconds (1 day). For more information, see `Replacing Auto Scaling instances based on maximum instance lifetime <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param metrics_collection: Enables the monitoring of group metrics of an Auto Scaling group. By default, these metrics are disabled.
        :param mixed_instances_policy: An embedded object that specifies a mixed instances policy. The policy includes properties that not only define the distribution of On-Demand Instances and Spot Instances, the maximum price to pay for Spot Instances (optional), and how the Auto Scaling group allocates instance types to fulfill On-Demand and Spot capacities, but also the properties that specify the instance configuration information—the launch template and instance types. The policy can also include a weight for each instance type and different launch templates for individual instance types. For more information, see `Auto Scaling groups with multiple instance types and purchase options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param new_instances_protected_from_scale_in: Indicates whether newly launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. For more information about preventing instances from terminating on scale in, see `Using instance scale-in protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-instance-protection.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param notification_configurations: Configures an Auto Scaling group to send notifications when specified events take place.
        :param placement_group: The name of the placement group into which to launch your instances. For more information, see `Placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . .. epigraph:: A *cluster* placement group is a logical grouping of instances within a single Availability Zone. You cannot specify multiple Availability Zones and a cluster placement group.
        :param service_linked_role_arn: The Amazon Resource Name (ARN) of the service-linked role that the Auto Scaling group uses to call other AWS service on your behalf. By default, Amazon EC2 Auto Scaling uses a service-linked role named ``AWSServiceRoleForAutoScaling`` , which it creates if it does not exist. For more information, see `Service-linked roles <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-service-linked-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param tags: One or more tags. You can tag your Auto Scaling group and propagate the tags to the Amazon EC2 instances it launches. Tags are not propagated to Amazon EBS volumes. To add tags to Amazon EBS volumes, specify the tags in a launch template but use caution. If the launch template specifies an instance tag with a key that is also specified for the Auto Scaling group, Amazon EC2 Auto Scaling overrides the value of that instance tag with the value specified by the Auto Scaling group. For more information, see `Tag Auto Scaling groups and instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-tagging.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param target_group_arns: The Amazon Resource Names (ARN) of the Elastic Load Balancing target groups to associate with the Auto Scaling group. Instances are registered as targets with the target groups. The target groups receive incoming traffic and route requests to one or more registered targets. For more information, see `Use Elastic Load Balancing to distribute traffic across the instances in your Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param termination_policies: A policy or a list of policies that are used to select the instance to terminate. These policies are executed in the order that you list them. For more information, see `Work with Amazon EC2 Auto Scaling termination policies <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-termination-policies.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid values: ``Default`` | ``AllocationStrategy`` | ``ClosestToNextInstanceHour`` | ``NewestInstance`` | ``OldestInstance`` | ``OldestLaunchConfiguration`` | ``OldestLaunchTemplate`` | ``arn:aws:lambda:region:account-id:function:my-function:my-alias``
        :param vpc_zone_identifier: A list of subnet IDs for a virtual private cloud (VPC) where instances in the Auto Scaling group can be created. If this resource specifies public subnets and is also in a VPC that is defined in the same stack template, you must use the `DependsOn attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the `VPC-gateway attachment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpc-gateway-attachment.html>`_ . .. epigraph:: When you update ``VPCZoneIdentifier`` , this retains the same Auto Scaling group and replaces old instances with new ones, according to the specified subnets. You can optionally specify how CloudFormation handles these updates by using an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ . Required to launch instances into a nondefault VPC. If you specify ``VPCZoneIdentifier`` with ``AvailabilityZones`` , the subnets that you specify for this property must reside in those Availability Zones.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d8ba2cee6007161ce4ac8e6f271353563746b8194e9da3c4517351b35669b797)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAutoScalingGroupProps(
            max_size=max_size,
            min_size=min_size,
            auto_scaling_group_name=auto_scaling_group_name,
            availability_zones=availability_zones,
            capacity_rebalance=capacity_rebalance,
            context=context,
            cooldown=cooldown,
            default_instance_warmup=default_instance_warmup,
            desired_capacity=desired_capacity,
            desired_capacity_type=desired_capacity_type,
            health_check_grace_period=health_check_grace_period,
            health_check_type=health_check_type,
            instance_id=instance_id,
            launch_configuration_name=launch_configuration_name,
            launch_template=launch_template,
            lifecycle_hook_specification_list=lifecycle_hook_specification_list,
            load_balancer_names=load_balancer_names,
            max_instance_lifetime=max_instance_lifetime,
            metrics_collection=metrics_collection,
            mixed_instances_policy=mixed_instances_policy,
            new_instances_protected_from_scale_in=new_instances_protected_from_scale_in,
            notification_configurations=notification_configurations,
            placement_group=placement_group,
            service_linked_role_arn=service_linked_role_arn,
            tags=tags,
            target_group_arns=target_group_arns,
            termination_policies=termination_policies,
            vpc_zone_identifier=vpc_zone_identifier,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f312dd276d3d70252809e3c5a9da0474186c12a2776c57e7811b571df6d42b9a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47959a7a01dbef79e5bf1d6e6e27303e1c1957b52b2b716af23464bf4a6794ff)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''One or more tags.

        You can tag your Auto Scaling group and propagate the tags to the Amazon EC2 instances it launches. Tags are not propagated to Amazon EBS volumes. To add tags to Amazon EBS volumes, specify the tags in a launch template but use caution. If the launch template specifies an instance tag with a key that is also specified for the Auto Scaling group, Amazon EC2 Auto Scaling overrides the value of that instance tag with the value specified by the Auto Scaling group. For more information, see `Tag Auto Scaling groups and instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-tagging.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-tags
        '''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="maxSize")
    def max_size(self) -> builtins.str:
        '''The maximum size of the group.

        .. epigraph::

           With a mixed instances policy that uses instance weighting, Amazon EC2 Auto Scaling may need to go above ``MaxSize`` to meet your capacity requirements. In this event, Amazon EC2 Auto Scaling will never go above ``MaxSize`` by more than your largest instance weight (weights that define how many units each instance contributes to the desired capacity of the group).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-maxsize
        '''
        return typing.cast(builtins.str, jsii.get(self, "maxSize"))

    @max_size.setter
    def max_size(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f4f63f8bc517d0f73fa0c7f45cd318348c9e8fdc16144be78766518821c5ac0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxSize", value)

    @builtins.property
    @jsii.member(jsii_name="minSize")
    def min_size(self) -> builtins.str:
        '''The minimum size of the group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-minsize
        '''
        return typing.cast(builtins.str, jsii.get(self, "minSize"))

    @min_size.setter
    def min_size(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a3bec99705477b97a5e2263ba2de274457ce880e055ebfa1036a63b6251936d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minSize", value)

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Auto Scaling group. This name must be unique per Region per account.

        The name can contain any ASCII character 33 to 126 including most punctuation characters, digits, and upper and lowercased letters.
        .. epigraph::

           You cannot use a colon (:) in the name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-autoscaling-autoscalinggroup-autoscalinggroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "autoScalingGroupName"))

    @auto_scaling_group_name.setter
    def auto_scaling_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40e8fdc97369d7f8797e7a890acf4c16f1c2a2166b1ca6d1644eaba121cf7435)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoScalingGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="availabilityZones")
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Availability Zones where instances in the Auto Scaling group can be created.

        Used for launching into the default VPC subnet in each Availability Zone when not using the ``VPCZoneIdentifier`` property, or for attaching a network interface when an existing network interface ID is specified in a launch template.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-availabilityzones
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "availabilityZones"))

    @availability_zones.setter
    def availability_zones(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba79ccc6a2a627119d2b42e5f2c3b0c8a81580a2589535c8255f0eb95acbdad6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZones", value)

    @builtins.property
    @jsii.member(jsii_name="capacityRebalance")
    def capacity_rebalance(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether Capacity Rebalancing is enabled.

        Otherwise, Capacity Rebalancing is disabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. For more information, see `Use Capacity Rebalancing to handle Amazon EC2 Spot Interruptions <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-capacity-rebalancing.html>`_ in the in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-capacityrebalance
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "capacityRebalance"))

    @capacity_rebalance.setter
    def capacity_rebalance(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fcd7ade02eab7516c3be5872ef46340937bca4fb38ec4eea0d8119ee5b20247b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "capacityRebalance", value)

    @builtins.property
    @jsii.member(jsii_name="context")
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-context
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "context"))

    @context.setter
    def context(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52553869bb2c8e9fe528514394f739bd1e81992c61e60f7aba37579b8e417255)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "context", value)

    @builtins.property
    @jsii.member(jsii_name="cooldown")
    def cooldown(self) -> typing.Optional[builtins.str]:
        '''*Only needed if you use simple scaling policies.*.

        The amount of time, in seconds, between one scaling activity ending and another one starting due to simple scaling policies. For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Default: ``300`` seconds

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-cooldown
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cooldown"))

    @cooldown.setter
    def cooldown(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00180c96190691ecfc2cce15cbbffd3a93e96f04098cb28287b5b1f2f22a4ab7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cooldown", value)

    @builtins.property
    @jsii.member(jsii_name="defaultInstanceWarmup")
    def default_instance_warmup(self) -> typing.Optional[jsii.Number]:
        '''The amount of time, in seconds, until a new instance is considered to have finished initializing and resource consumption to become stable after it enters the ``InService`` state.

        During an instance refresh, Amazon EC2 Auto Scaling waits for the warm-up period after it replaces an instance before it moves on to replacing the next instance. Amazon EC2 Auto Scaling also waits for the warm-up period before aggregating the metrics for new instances with existing instances in the Amazon CloudWatch metrics that are used for scaling, resulting in more reliable usage data. For more information, see `Set the default instance warmup for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        .. epigraph::

           To manage various warm-up settings at the group level, we recommend that you set the default instance warmup, *even if it is set to 0 seconds* . To remove a value that you previously set, include the property but specify ``-1`` for the value. However, we strongly recommend keeping the default instance warmup enabled by specifying a value of ``0`` or other nominal value.

        Default: None

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-defaultinstancewarmup
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "defaultInstanceWarmup"))

    @default_instance_warmup.setter
    def default_instance_warmup(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c24bc5c20f9d70adfed9b74186ec848f6dd0942c25d9923785d13b0607bd2151)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultInstanceWarmup", value)

    @builtins.property
    @jsii.member(jsii_name="desiredCapacity")
    def desired_capacity(self) -> typing.Optional[builtins.str]:
        '''The desired capacity is the initial capacity of the Auto Scaling group at the time of its creation and the capacity it attempts to maintain.

        It can scale beyond this capacity if you configure automatic scaling.

        The number must be greater than or equal to the minimum size of the group and less than or equal to the maximum size of the group. If you do not specify a desired capacity when creating the stack, the default is the minimum size of the group.

        CloudFormation marks the Auto Scaling group as successful (by setting its status to CREATE_COMPLETE) when the desired capacity is reached. However, if a maximum Spot price is set in the launch template or launch configuration that you specified, then desired capacity is not used as a criteria for success. Whether your request is fulfilled depends on Spot Instance capacity and your maximum price.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "desiredCapacity"))

    @desired_capacity.setter
    def desired_capacity(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a26fca7e1391078df7804e73b80d4d4bf03239a5b160de2f22cdfdb3542dcce5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "desiredCapacity", value)

    @builtins.property
    @jsii.member(jsii_name="desiredCapacityType")
    def desired_capacity_type(self) -> typing.Optional[builtins.str]:
        '''The unit of measurement for the value specified for desired capacity.

        Amazon EC2 Auto Scaling supports ``DesiredCapacityType`` for attribute-based instance type selection only. For more information, see `Creating an Auto Scaling group using attribute-based instance type selection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-instance-type-requirements.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        By default, Amazon EC2 Auto Scaling specifies ``units`` , which translates into number of instances.

        Valid values: ``units`` | ``vcpu`` | ``memory-mib``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacitytype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "desiredCapacityType"))

    @desired_capacity_type.setter
    def desired_capacity_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77e152ae9a6bd62beed66e6599aa26329b2652e03c5b0d07df1c012a46de261b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "desiredCapacityType", value)

    @builtins.property
    @jsii.member(jsii_name="healthCheckGracePeriod")
    def health_check_grace_period(self) -> typing.Optional[jsii.Number]:
        '''The amount of time, in seconds, that Amazon EC2 Auto Scaling waits before checking the health status of an EC2 instance that has come into service and marking it unhealthy due to a failed health check.

        This is useful if your instances do not immediately pass their health checks after they enter the ``InService`` state. For more information, see `Set the health check grace period for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/health-check-grace-period.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Default: ``0`` seconds

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-healthcheckgraceperiod
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "healthCheckGracePeriod"))

    @health_check_grace_period.setter
    def health_check_grace_period(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce99453ae7cd701daae438027248f2dd16b9004241a4861b1b0a49c7871eb464)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "healthCheckGracePeriod", value)

    @builtins.property
    @jsii.member(jsii_name="healthCheckType")
    def health_check_type(self) -> typing.Optional[builtins.str]:
        '''A comma-separated value string of one or more health check types.

        The valid values are ``EC2`` , ``ELB`` , and ``VPC_LATTICE`` . ``EC2`` is the default health check and cannot be disabled. For more information, see `Health checks for Auto Scaling instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/healthcheck.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Only specify ``EC2`` if you must clear a value that was previously set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-healthchecktype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "healthCheckType"))

    @health_check_type.setter
    def health_check_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d89e3a93d75c8339966184a05a5c629f8986c31a8825839aa7fbfc3b063a2a5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "healthCheckType", value)

    @builtins.property
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance used to base the launch configuration on.

        For more information, see `Create an Auto Scaling group using an EC2 instance <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-from-instance.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``LaunchConfigurationName`` , don't specify ``InstanceId`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__807d7a9adb0f9c2f26a711c97bd951df6ae3487de3147ae199953db4f6c229cf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceId", value)

    @builtins.property
    @jsii.member(jsii_name="launchConfigurationName")
    def launch_configuration_name(self) -> typing.Optional[builtins.str]:
        '''The name of the launch configuration to use to launch instances.

        Required only if you don't specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``InstanceId`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-launchconfigurationname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchConfigurationName"))

    @launch_configuration_name.setter
    def launch_configuration_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce1aedc9a7f8ec0ad90dfafceb72240ca56e12323bde67a3ca70fa168bd509ce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "launchConfigurationName", value)

    @builtins.property
    @jsii.member(jsii_name="launchTemplate")
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"]]:
        '''Information used to specify the launch template and version to use to launch instances.

        You can alternatively associate a launch template to the Auto Scaling group by specifying a ``MixedInstancesPolicy`` . For more information about creating launch templates, see `Create a launch template for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-template.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you omit this property, you must specify ``MixedInstancesPolicy`` , ``LaunchConfigurationName`` , or ``InstanceId`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-launchtemplate
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"]], jsii.get(self, "launchTemplate"))

    @launch_template.setter
    def launch_template(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e46a8ef6a1e77f37bd575625d13f34004ddf8d2ce4be91a94dd0bcc391e07a78)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "launchTemplate", value)

    @builtins.property
    @jsii.member(jsii_name="lifecycleHookSpecificationList")
    def lifecycle_hook_specification_list(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LifecycleHookSpecificationProperty"]]]]:
        '''One or more lifecycle hooks to add to the Auto Scaling group before instances are launched.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecificationlist
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LifecycleHookSpecificationProperty"]]]], jsii.get(self, "lifecycleHookSpecificationList"))

    @lifecycle_hook_specification_list.setter
    def lifecycle_hook_specification_list(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LifecycleHookSpecificationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd6e8b42fa69a0df75799ee782ceadb17a838c01878e0eb9ff09ae058b90c9d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "lifecycleHookSpecificationList", value)

    @builtins.property
    @jsii.member(jsii_name="loadBalancerNames")
    def load_balancer_names(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Classic Load Balancers associated with this Auto Scaling group.

        For Application Load Balancers, Network Load Balancers, and Gateway Load Balancers, specify the ``TargetGroupARNs`` property instead.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-loadbalancernames
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "loadBalancerNames"))

    @load_balancer_names.setter
    def load_balancer_names(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a23c0b10a3091bb43ad5e60b04b91b57d605ecd2389d1c36290c46080c492847)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "loadBalancerNames", value)

    @builtins.property
    @jsii.member(jsii_name="maxInstanceLifetime")
    def max_instance_lifetime(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of time, in seconds, that an instance can be in service.

        The default is null. If specified, the value must be either 0 or a number equal to or greater than 86,400 seconds (1 day). For more information, see `Replacing Auto Scaling instances based on maximum instance lifetime <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-maxinstancelifetime
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxInstanceLifetime"))

    @max_instance_lifetime.setter
    def max_instance_lifetime(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c72588d4e6113052b7a3848095a560964c8d8d37e8ba3d61cbbb2e7ba03fea6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxInstanceLifetime", value)

    @builtins.property
    @jsii.member(jsii_name="metricsCollection")
    def metrics_collection(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MetricsCollectionProperty"]]]]:
        '''Enables the monitoring of group metrics of an Auto Scaling group.

        By default, these metrics are disabled.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-metricscollection
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MetricsCollectionProperty"]]]], jsii.get(self, "metricsCollection"))

    @metrics_collection.setter
    def metrics_collection(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MetricsCollectionProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d32a051ae8368000fa35ee4a51495330fedacfcacf365fe8a1e42d21ee86ff2d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "metricsCollection", value)

    @builtins.property
    @jsii.member(jsii_name="mixedInstancesPolicy")
    def mixed_instances_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MixedInstancesPolicyProperty"]]:
        '''An embedded object that specifies a mixed instances policy.

        The policy includes properties that not only define the distribution of On-Demand Instances and Spot Instances, the maximum price to pay for Spot Instances (optional), and how the Auto Scaling group allocates instance types to fulfill On-Demand and Spot capacities, but also the properties that specify the instance configuration information—the launch template and instance types. The policy can also include a weight for each instance type and different launch templates for individual instance types.

        For more information, see `Auto Scaling groups with multiple instance types and purchase options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-mixedinstancespolicy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MixedInstancesPolicyProperty"]], jsii.get(self, "mixedInstancesPolicy"))

    @mixed_instances_policy.setter
    def mixed_instances_policy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MixedInstancesPolicyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2eeb991d1cd003adb697e60e6dec4d4ee0b29f9758fba65cdeda2e67e810d031)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "mixedInstancesPolicy", value)

    @builtins.property
    @jsii.member(jsii_name="newInstancesProtectedFromScaleIn")
    def new_instances_protected_from_scale_in(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether newly launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        For more information about preventing instances from terminating on scale in, see `Using instance scale-in protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-instance-protection.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-newinstancesprotectedfromscalein
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "newInstancesProtectedFromScaleIn"))

    @new_instances_protected_from_scale_in.setter
    def new_instances_protected_from_scale_in(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91b500624d150b98c66fcf098662c0a1fc6935e0d9cd154eff3d3691f9a107e1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "newInstancesProtectedFromScaleIn", value)

    @builtins.property
    @jsii.member(jsii_name="notificationConfigurations")
    def notification_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NotificationConfigurationProperty"]]]]:
        '''Configures an Auto Scaling group to send notifications when specified events take place.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NotificationConfigurationProperty"]]]], jsii.get(self, "notificationConfigurations"))

    @notification_configurations.setter
    def notification_configurations(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NotificationConfigurationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5c3fc9c42371a5f5fe97a5d3fb16d41213e13982291eb878e27d22e968f4e6e8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "notificationConfigurations", value)

    @builtins.property
    @jsii.member(jsii_name="placementGroup")
    def placement_group(self) -> typing.Optional[builtins.str]:
        '''The name of the placement group into which to launch your instances.

        For more information, see `Placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        .. epigraph::

           A *cluster* placement group is a logical grouping of instances within a single Availability Zone. You cannot specify multiple Availability Zones and a cluster placement group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-placementgroup
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "placementGroup"))

    @placement_group.setter
    def placement_group(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a770dfbb658761846c5231237bc7fd8978779e61c7725bccbe14fa238d5b2b12)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "placementGroup", value)

    @builtins.property
    @jsii.member(jsii_name="serviceLinkedRoleArn")
    def service_linked_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the service-linked role that the Auto Scaling group uses to call other AWS service on your behalf.

        By default, Amazon EC2 Auto Scaling uses a service-linked role named ``AWSServiceRoleForAutoScaling`` , which it creates if it does not exist. For more information, see `Service-linked roles <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-service-linked-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-autoscaling-autoscalinggroup-servicelinkedrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "serviceLinkedRoleArn"))

    @service_linked_role_arn.setter
    def service_linked_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed363ed16309e150caf7f0bbc1a1a1fc2c8bd1ddf6ae299fa1540076f7ad0cc9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceLinkedRoleArn", value)

    @builtins.property
    @jsii.member(jsii_name="targetGroupArns")
    def target_group_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon Resource Names (ARN) of the Elastic Load Balancing target groups to associate with the Auto Scaling group.

        Instances are registered as targets with the target groups. The target groups receive incoming traffic and route requests to one or more registered targets. For more information, see `Use Elastic Load Balancing to distribute traffic across the instances in your Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-targetgrouparns
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "targetGroupArns"))

    @target_group_arns.setter
    def target_group_arns(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0bc71ae38a484fac3e58f33ebdec6854c3d29e051ecf69401bbcdde69c0ee80)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetGroupArns", value)

    @builtins.property
    @jsii.member(jsii_name="terminationPolicies")
    def termination_policies(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A policy or a list of policies that are used to select the instance to terminate.

        These policies are executed in the order that you list them. For more information, see `Work with Amazon EC2 Auto Scaling termination policies <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-termination-policies.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Valid values: ``Default`` | ``AllocationStrategy`` | ``ClosestToNextInstanceHour`` | ``NewestInstance`` | ``OldestInstance`` | ``OldestLaunchConfiguration`` | ``OldestLaunchTemplate`` | ``arn:aws:lambda:region:account-id:function:my-function:my-alias``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-termpolicy
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "terminationPolicies"))

    @termination_policies.setter
    def termination_policies(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5daca53bfde511b5f5ae7f0521ed2bd70a9a3cd0feaf10ef8bbf6e9cf4a381f6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "terminationPolicies", value)

    @builtins.property
    @jsii.member(jsii_name="vpcZoneIdentifier")
    def vpc_zone_identifier(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of subnet IDs for a virtual private cloud (VPC) where instances in the Auto Scaling group can be created.

        If this resource specifies public subnets and is also in a VPC that is defined in the same stack template, you must use the `DependsOn attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the `VPC-gateway attachment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpc-gateway-attachment.html>`_ .
        .. epigraph::

           When you update ``VPCZoneIdentifier`` , this retains the same Auto Scaling group and replaces old instances with new ones, according to the specified subnets. You can optionally specify how CloudFormation handles these updates by using an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ .

        Required to launch instances into a nondefault VPC. If you specify ``VPCZoneIdentifier`` with ``AvailabilityZones`` , the subnets that you specify for this property must reside in those Availability Zones.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-vpczoneidentifier
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "vpcZoneIdentifier"))

    @vpc_zone_identifier.setter
    def vpc_zone_identifier(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f647181eceda5a9bfa4bd1a26afd8e4241f99daa32318b27594af04772070ac5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcZoneIdentifier", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``AcceleratorCountRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum number of accelerators for an instance type.

            :param max: The maximum value.
            :param min: The minimum value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-acceleratorcountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                accelerator_count_request_property = autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__642a80bb68b5641f296fa13d9170920dc50ed692bd4e7161d4195ea8d6f7d297)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-acceleratorcountrequest.html#cfn-autoscaling-autoscalinggroup-acceleratorcountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-acceleratorcountrequest.html#cfn-autoscaling-autoscalinggroup-acceleratorcountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorTotalMemoryMiBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``AcceleratorTotalMemoryMiBRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum total memory size for the accelerators for an instance type, in MiB.

            :param max: The memory maximum in MiB.
            :param min: The memory minimum in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-acceleratortotalmemorymibrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                accelerator_total_memory_mi_bRequest_property = autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3e3d46345becc8f13c0ef14e2ac1b4f8c14d13995594052f0a83d66d99f20760)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The memory maximum in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-acceleratortotalmemorymibrequest.html#cfn-autoscaling-autoscalinggroup-acceleratortotalmemorymibrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The memory minimum in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-acceleratortotalmemorymibrequest.html#cfn-autoscaling-autoscalinggroup-acceleratortotalmemorymibrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorTotalMemoryMiBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class BaselineEbsBandwidthMbpsRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``BaselineEbsBandwidthMbpsRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum baseline bandwidth performance for an instance type, in Mbps.

            :param max: The maximum value in Mbps.
            :param min: The minimum value in Mbps.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-baselineebsbandwidthmbpsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                baseline_ebs_bandwidth_mbps_request_property = autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__36b0ab4a571aa009a65070b38b6f4e5ef64fa1f94d861b2083c70c2cae437f53)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum value in Mbps.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-baselineebsbandwidthmbpsrequest.html#cfn-autoscaling-autoscalinggroup-baselineebsbandwidthmbpsrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum value in Mbps.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-baselineebsbandwidthmbpsrequest.html#cfn-autoscaling-autoscalinggroup-baselineebsbandwidthmbpsrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BaselineEbsBandwidthMbpsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "accelerator_count": "acceleratorCount",
            "accelerator_manufacturers": "acceleratorManufacturers",
            "accelerator_names": "acceleratorNames",
            "accelerator_total_memory_mib": "acceleratorTotalMemoryMiB",
            "accelerator_types": "acceleratorTypes",
            "allowed_instance_types": "allowedInstanceTypes",
            "bare_metal": "bareMetal",
            "baseline_ebs_bandwidth_mbps": "baselineEbsBandwidthMbps",
            "burstable_performance": "burstablePerformance",
            "cpu_manufacturers": "cpuManufacturers",
            "excluded_instance_types": "excludedInstanceTypes",
            "instance_generations": "instanceGenerations",
            "local_storage": "localStorage",
            "local_storage_types": "localStorageTypes",
            "memory_gib_per_v_cpu": "memoryGiBPerVCpu",
            "memory_mib": "memoryMiB",
            "network_bandwidth_gbps": "networkBandwidthGbps",
            "network_interface_count": "networkInterfaceCount",
            "on_demand_max_price_percentage_over_lowest_price": "onDemandMaxPricePercentageOverLowestPrice",
            "require_hibernate_support": "requireHibernateSupport",
            "spot_max_price_percentage_over_lowest_price": "spotMaxPricePercentageOverLowestPrice",
            "total_local_storage_gb": "totalLocalStorageGb",
            "v_cpu_count": "vCpuCount",
        },
    )
    class InstanceRequirementsProperty:
        def __init__(
            self,
            *,
            accelerator_count: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.AcceleratorCountRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            accelerator_manufacturers: typing.Optional[typing.Sequence[builtins.str]] = None,
            accelerator_names: typing.Optional[typing.Sequence[builtins.str]] = None,
            accelerator_total_memory_mib: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            accelerator_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            allowed_instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            bare_metal: typing.Optional[builtins.str] = None,
            baseline_ebs_bandwidth_mbps: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            burstable_performance: typing.Optional[builtins.str] = None,
            cpu_manufacturers: typing.Optional[typing.Sequence[builtins.str]] = None,
            excluded_instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_generations: typing.Optional[typing.Sequence[builtins.str]] = None,
            local_storage: typing.Optional[builtins.str] = None,
            local_storage_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            memory_gib_per_v_cpu: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            memory_mib: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.MemoryMiBRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            network_bandwidth_gbps: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            network_interface_count: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            on_demand_max_price_percentage_over_lowest_price: typing.Optional[jsii.Number] = None,
            require_hibernate_support: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            spot_max_price_percentage_over_lowest_price: typing.Optional[jsii.Number] = None,
            total_local_storage_gb: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            v_cpu_count: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.VCpuCountRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The attributes for the instance types for a mixed instances policy.

            Amazon EC2 Auto Scaling uses your specified requirements to identify instance types. Then, it uses your On-Demand and Spot allocation strategies to launch instances from these instance types.

            When you specify multiple attributes, you get instance types that satisfy all of the specified attributes. If you specify multiple values for an attribute, you get instance types that satisfy any of the specified values.

            To limit the list of instance types from which Amazon EC2 Auto Scaling can identify matching instance types, you can use one of the following parameters, but not both in the same request:

            - ``AllowedInstanceTypes`` - The instance types to include in the list. All other instance types are ignored, even if they match your specified attributes.
            - ``ExcludedInstanceTypes`` - The instance types to exclude from the list, even if they match your specified attributes.

            .. epigraph::

               You must specify ``VCpuCount`` and ``MemoryMiB`` . All other attributes are optional. Any unspecified optional attribute is set to its default.

            For an example template, see `Auto scaling template snippets <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/quickref-autoscaling.html>`_ .

            For more information, see `Creating an Auto Scaling group using attribute-based instance type selection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-instance-type-requirements.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . For help determining which instance types match your attributes before you apply them to your Auto Scaling group, see `Preview instance types with specified attributes <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-attribute-based-instance-type-selection.html#ec2fleet-get-instance-types-from-instance-requirements>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            ``InstanceRequirements`` is a property of the ``LaunchTemplateOverrides`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplate <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplate.html>`_ property type.

            :param accelerator_count: The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) for an instance type. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` . Default: No minimum or maximum limits
            :param accelerator_manufacturers: Indicates whether instance types must have accelerators by specific manufacturers. - For instance types with NVIDIA devices, specify ``nvidia`` . - For instance types with AMD devices, specify ``amd`` . - For instance types with AWS devices, specify ``amazon-web-services`` . - For instance types with Xilinx devices, specify ``xilinx`` . Default: Any manufacturer
            :param accelerator_names: Lists the accelerators that must be on an instance type. - For instance types with NVIDIA A100 GPUs, specify ``a100`` . - For instance types with NVIDIA V100 GPUs, specify ``v100`` . - For instance types with NVIDIA K80 GPUs, specify ``k80`` . - For instance types with NVIDIA T4 GPUs, specify ``t4`` . - For instance types with NVIDIA M60 GPUs, specify ``m60`` . - For instance types with AMD Radeon Pro V520 GPUs, specify ``radeon-pro-v520`` . - For instance types with Xilinx VU9P FPGAs, specify ``vu9p`` . Default: Any accelerator
            :param accelerator_total_memory_mib: The minimum and maximum total memory size for the accelerators on an instance type, in MiB. Default: No minimum or maximum limits
            :param accelerator_types: Lists the accelerator types that must be on an instance type. - For instance types with GPU accelerators, specify ``gpu`` . - For instance types with FPGA accelerators, specify ``fpga`` . - For instance types with inference accelerators, specify ``inference`` . Default: Any accelerator type
            :param allowed_instance_types: The instance types to apply your specified attributes against. All other instance types are ignored, even if they match your specified attributes. You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to allow an instance type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` . For example, if you specify ``c5*`` , Amazon EC2 Auto Scaling will allow the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 Auto Scaling will allow all the M5a instance types, but not the M5n instance types. .. epigraph:: If you specify ``AllowedInstanceTypes`` , you can't specify ``ExcludedInstanceTypes`` . Default: All instance types
            :param bare_metal: Indicates whether bare metal instance types are included, excluded, or required. Default: ``excluded``
            :param baseline_ebs_bandwidth_mbps: The minimum and maximum baseline bandwidth performance for an instance type, in Mbps. For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . Default: No minimum or maximum limits
            :param burstable_performance: Indicates whether burstable performance instance types are included, excluded, or required. For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . Default: ``excluded``
            :param cpu_manufacturers: Lists which specific CPU manufacturers to include. - For instance types with Intel CPUs, specify ``intel`` . - For instance types with AMD CPUs, specify ``amd`` . - For instance types with AWS CPUs, specify ``amazon-web-services`` . .. epigraph:: Don't confuse the CPU hardware manufacturer with the CPU hardware architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template. Default: Any manufacturer
            :param excluded_instance_types: The instance types to exclude. You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to exclude an instance family, type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` . For example, if you specify ``c5*`` , you are excluding the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 Auto Scaling will exclude all the M5a instance types, but not the M5n instance types. .. epigraph:: If you specify ``ExcludedInstanceTypes`` , you can't specify ``AllowedInstanceTypes`` . Default: No excluded instance types
            :param instance_generations: Indicates whether current or previous generation instance types are included. - For current generation instance types, specify ``current`` . The current generation includes EC2 instance types currently recommended for use. This typically includes the latest two to three generations in each instance family. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . - For previous generation instance types, specify ``previous`` . Default: Any current or previous generation
            :param local_storage: Indicates whether instance types with instance store volumes are included, excluded, or required. For more information, see `Amazon EC2 instance store <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . Default: ``included``
            :param local_storage_types: Indicates the type of local storage that is required. - For instance types with hard disk drive (HDD) storage, specify ``hdd`` . - For instance types with solid state drive (SSD) storage, specify ``ssd`` . Default: Any local storage type
            :param memory_gib_per_v_cpu: The minimum and maximum amount of memory per vCPU for an instance type, in GiB. Default: No minimum or maximum limits
            :param memory_mib: The minimum and maximum instance memory size for an instance type, in MiB.
            :param network_bandwidth_gbps: The minimum and maximum amount of network bandwidth, in gigabits per second (Gbps). Default: No minimum or maximum limits
            :param network_interface_count: The minimum and maximum number of network interfaces for an instance type. Default: No minimum or maximum limits
            :param on_demand_max_price_percentage_over_lowest_price: The price protection threshold for On-Demand Instances. This is the maximum you’ll pay for an On-Demand Instance, expressed as a percentage higher than the least expensive current generation M, C, or R instance type with your specified attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price is higher than your threshold. The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage. To turn off price protection, specify a high value, such as ``999999`` . If you set ``DesiredCapacityType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per vCPU or per memory price instead of the per instance price. Default: ``20``
            :param require_hibernate_support: Indicates whether instance types must provide On-Demand Instance hibernation support. Default: ``false``
            :param spot_max_price_percentage_over_lowest_price: The price protection threshold for Spot Instances. This is the maximum you’ll pay for a Spot Instance, expressed as a percentage higher than the least expensive current generation M, C, or R instance type with your specified attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price is higher than your threshold. The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage. To turn off price protection, specify a high value, such as ``999999`` . If you set ``DesiredCapacityType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per vCPU or per memory price instead of the per instance price. Default: ``100``
            :param total_local_storage_gb: The minimum and maximum total local storage size for an instance type, in GB. Default: No minimum or maximum limits
            :param v_cpu_count: The minimum and maximum number of vCPUs for an instance type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                instance_requirements_property = autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty(
                    accelerator_count=autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                        max=123,
                        min=123
                    ),
                    accelerator_manufacturers=["acceleratorManufacturers"],
                    accelerator_names=["acceleratorNames"],
                    accelerator_total_memory_mi_b=autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                        max=123,
                        min=123
                    ),
                    accelerator_types=["acceleratorTypes"],
                    allowed_instance_types=["allowedInstanceTypes"],
                    bare_metal="bareMetal",
                    baseline_ebs_bandwidth_mbps=autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                        max=123,
                        min=123
                    ),
                    burstable_performance="burstablePerformance",
                    cpu_manufacturers=["cpuManufacturers"],
                    excluded_instance_types=["excludedInstanceTypes"],
                    instance_generations=["instanceGenerations"],
                    local_storage="localStorage",
                    local_storage_types=["localStorageTypes"],
                    memory_gi_bPer_vCpu=autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                        max=123,
                        min=123
                    ),
                    memory_mi_b=autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                        max=123,
                        min=123
                    ),
                    network_bandwidth_gbps=autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                        max=123,
                        min=123
                    ),
                    network_interface_count=autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                        max=123,
                        min=123
                    ),
                    on_demand_max_price_percentage_over_lowest_price=123,
                    require_hibernate_support=False,
                    spot_max_price_percentage_over_lowest_price=123,
                    total_local_storage_gb=autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                        max=123,
                        min=123
                    ),
                    v_cpu_count=autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                        max=123,
                        min=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e22358c3d5577020d9106a247f13fb06a2cf3668a83ede597e2b41db7bac111f)
                check_type(argname="argument accelerator_count", value=accelerator_count, expected_type=type_hints["accelerator_count"])
                check_type(argname="argument accelerator_manufacturers", value=accelerator_manufacturers, expected_type=type_hints["accelerator_manufacturers"])
                check_type(argname="argument accelerator_names", value=accelerator_names, expected_type=type_hints["accelerator_names"])
                check_type(argname="argument accelerator_total_memory_mib", value=accelerator_total_memory_mib, expected_type=type_hints["accelerator_total_memory_mib"])
                check_type(argname="argument accelerator_types", value=accelerator_types, expected_type=type_hints["accelerator_types"])
                check_type(argname="argument allowed_instance_types", value=allowed_instance_types, expected_type=type_hints["allowed_instance_types"])
                check_type(argname="argument bare_metal", value=bare_metal, expected_type=type_hints["bare_metal"])
                check_type(argname="argument baseline_ebs_bandwidth_mbps", value=baseline_ebs_bandwidth_mbps, expected_type=type_hints["baseline_ebs_bandwidth_mbps"])
                check_type(argname="argument burstable_performance", value=burstable_performance, expected_type=type_hints["burstable_performance"])
                check_type(argname="argument cpu_manufacturers", value=cpu_manufacturers, expected_type=type_hints["cpu_manufacturers"])
                check_type(argname="argument excluded_instance_types", value=excluded_instance_types, expected_type=type_hints["excluded_instance_types"])
                check_type(argname="argument instance_generations", value=instance_generations, expected_type=type_hints["instance_generations"])
                check_type(argname="argument local_storage", value=local_storage, expected_type=type_hints["local_storage"])
                check_type(argname="argument local_storage_types", value=local_storage_types, expected_type=type_hints["local_storage_types"])
                check_type(argname="argument memory_gib_per_v_cpu", value=memory_gib_per_v_cpu, expected_type=type_hints["memory_gib_per_v_cpu"])
                check_type(argname="argument memory_mib", value=memory_mib, expected_type=type_hints["memory_mib"])
                check_type(argname="argument network_bandwidth_gbps", value=network_bandwidth_gbps, expected_type=type_hints["network_bandwidth_gbps"])
                check_type(argname="argument network_interface_count", value=network_interface_count, expected_type=type_hints["network_interface_count"])
                check_type(argname="argument on_demand_max_price_percentage_over_lowest_price", value=on_demand_max_price_percentage_over_lowest_price, expected_type=type_hints["on_demand_max_price_percentage_over_lowest_price"])
                check_type(argname="argument require_hibernate_support", value=require_hibernate_support, expected_type=type_hints["require_hibernate_support"])
                check_type(argname="argument spot_max_price_percentage_over_lowest_price", value=spot_max_price_percentage_over_lowest_price, expected_type=type_hints["spot_max_price_percentage_over_lowest_price"])
                check_type(argname="argument total_local_storage_gb", value=total_local_storage_gb, expected_type=type_hints["total_local_storage_gb"])
                check_type(argname="argument v_cpu_count", value=v_cpu_count, expected_type=type_hints["v_cpu_count"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if accelerator_count is not None:
                self._values["accelerator_count"] = accelerator_count
            if accelerator_manufacturers is not None:
                self._values["accelerator_manufacturers"] = accelerator_manufacturers
            if accelerator_names is not None:
                self._values["accelerator_names"] = accelerator_names
            if accelerator_total_memory_mib is not None:
                self._values["accelerator_total_memory_mib"] = accelerator_total_memory_mib
            if accelerator_types is not None:
                self._values["accelerator_types"] = accelerator_types
            if allowed_instance_types is not None:
                self._values["allowed_instance_types"] = allowed_instance_types
            if bare_metal is not None:
                self._values["bare_metal"] = bare_metal
            if baseline_ebs_bandwidth_mbps is not None:
                self._values["baseline_ebs_bandwidth_mbps"] = baseline_ebs_bandwidth_mbps
            if burstable_performance is not None:
                self._values["burstable_performance"] = burstable_performance
            if cpu_manufacturers is not None:
                self._values["cpu_manufacturers"] = cpu_manufacturers
            if excluded_instance_types is not None:
                self._values["excluded_instance_types"] = excluded_instance_types
            if instance_generations is not None:
                self._values["instance_generations"] = instance_generations
            if local_storage is not None:
                self._values["local_storage"] = local_storage
            if local_storage_types is not None:
                self._values["local_storage_types"] = local_storage_types
            if memory_gib_per_v_cpu is not None:
                self._values["memory_gib_per_v_cpu"] = memory_gib_per_v_cpu
            if memory_mib is not None:
                self._values["memory_mib"] = memory_mib
            if network_bandwidth_gbps is not None:
                self._values["network_bandwidth_gbps"] = network_bandwidth_gbps
            if network_interface_count is not None:
                self._values["network_interface_count"] = network_interface_count
            if on_demand_max_price_percentage_over_lowest_price is not None:
                self._values["on_demand_max_price_percentage_over_lowest_price"] = on_demand_max_price_percentage_over_lowest_price
            if require_hibernate_support is not None:
                self._values["require_hibernate_support"] = require_hibernate_support
            if spot_max_price_percentage_over_lowest_price is not None:
                self._values["spot_max_price_percentage_over_lowest_price"] = spot_max_price_percentage_over_lowest_price
            if total_local_storage_gb is not None:
                self._values["total_local_storage_gb"] = total_local_storage_gb
            if v_cpu_count is not None:
                self._values["v_cpu_count"] = v_cpu_count

        @builtins.property
        def accelerator_count(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.AcceleratorCountRequestProperty"]]:
            '''The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) for an instance type.

            To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-acceleratorcount
            '''
            result = self._values.get("accelerator_count")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.AcceleratorCountRequestProperty"]], result)

        @builtins.property
        def accelerator_manufacturers(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates whether instance types must have accelerators by specific manufacturers.

            - For instance types with NVIDIA devices, specify ``nvidia`` .
            - For instance types with AMD devices, specify ``amd`` .
            - For instance types with AWS devices, specify ``amazon-web-services`` .
            - For instance types with Xilinx devices, specify ``xilinx`` .

            Default: Any manufacturer

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-acceleratormanufacturers
            '''
            result = self._values.get("accelerator_manufacturers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def accelerator_names(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Lists the accelerators that must be on an instance type.

            - For instance types with NVIDIA A100 GPUs, specify ``a100`` .
            - For instance types with NVIDIA V100 GPUs, specify ``v100`` .
            - For instance types with NVIDIA K80 GPUs, specify ``k80`` .
            - For instance types with NVIDIA T4 GPUs, specify ``t4`` .
            - For instance types with NVIDIA M60 GPUs, specify ``m60`` .
            - For instance types with AMD Radeon Pro V520 GPUs, specify ``radeon-pro-v520`` .
            - For instance types with Xilinx VU9P FPGAs, specify ``vu9p`` .

            Default: Any accelerator

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-acceleratornames
            '''
            result = self._values.get("accelerator_names")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def accelerator_total_memory_mib(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty"]]:
            '''The minimum and maximum total memory size for the accelerators on an instance type, in MiB.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-acceleratortotalmemorymib
            '''
            result = self._values.get("accelerator_total_memory_mib")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty"]], result)

        @builtins.property
        def accelerator_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Lists the accelerator types that must be on an instance type.

            - For instance types with GPU accelerators, specify ``gpu`` .
            - For instance types with FPGA accelerators, specify ``fpga`` .
            - For instance types with inference accelerators, specify ``inference`` .

            Default: Any accelerator type

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-acceleratortypes
            '''
            result = self._values.get("accelerator_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allowed_instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance types to apply your specified attributes against.

            All other instance types are ignored, even if they match your specified attributes.

            You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to allow an instance type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` .

            For example, if you specify ``c5*`` , Amazon EC2 Auto Scaling will allow the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 Auto Scaling will allow all the M5a instance types, but not the M5n instance types.
            .. epigraph::

               If you specify ``AllowedInstanceTypes`` , you can't specify ``ExcludedInstanceTypes`` .

            Default: All instance types

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-allowedinstancetypes
            '''
            result = self._values.get("allowed_instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def bare_metal(self) -> typing.Optional[builtins.str]:
            '''Indicates whether bare metal instance types are included, excluded, or required.

            Default: ``excluded``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-baremetal
            '''
            result = self._values.get("bare_metal")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def baseline_ebs_bandwidth_mbps(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty"]]:
            '''The minimum and maximum baseline bandwidth performance for an instance type, in Mbps.

            For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-baselineebsbandwidthmbps
            '''
            result = self._values.get("baseline_ebs_bandwidth_mbps")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty"]], result)

        @builtins.property
        def burstable_performance(self) -> typing.Optional[builtins.str]:
            '''Indicates whether burstable performance instance types are included, excluded, or required.

            For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            Default: ``excluded``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-burstableperformance
            '''
            result = self._values.get("burstable_performance")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cpu_manufacturers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Lists which specific CPU manufacturers to include.

            - For instance types with Intel CPUs, specify ``intel`` .
            - For instance types with AMD CPUs, specify ``amd`` .
            - For instance types with AWS CPUs, specify ``amazon-web-services`` .

            .. epigraph::

               Don't confuse the CPU hardware manufacturer with the CPU hardware architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.

            Default: Any manufacturer

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-cpumanufacturers
            '''
            result = self._values.get("cpu_manufacturers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def excluded_instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance types to exclude.

            You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to exclude an instance family, type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` .

            For example, if you specify ``c5*`` , you are excluding the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 Auto Scaling will exclude all the M5a instance types, but not the M5n instance types.
            .. epigraph::

               If you specify ``ExcludedInstanceTypes`` , you can't specify ``AllowedInstanceTypes`` .

            Default: No excluded instance types

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-excludedinstancetypes
            '''
            result = self._values.get("excluded_instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_generations(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates whether current or previous generation instance types are included.

            - For current generation instance types, specify ``current`` . The current generation includes EC2 instance types currently recommended for use. This typically includes the latest two to three generations in each instance family. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
            - For previous generation instance types, specify ``previous`` .

            Default: Any current or previous generation

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-instancegenerations
            '''
            result = self._values.get("instance_generations")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def local_storage(self) -> typing.Optional[builtins.str]:
            '''Indicates whether instance types with instance store volumes are included, excluded, or required.

            For more information, see `Amazon EC2 instance store <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            Default: ``included``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-localstorage
            '''
            result = self._values.get("local_storage")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def local_storage_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates the type of local storage that is required.

            - For instance types with hard disk drive (HDD) storage, specify ``hdd`` .
            - For instance types with solid state drive (SSD) storage, specify ``ssd`` .

            Default: Any local storage type

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-localstoragetypes
            '''
            result = self._values.get("local_storage_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def memory_gib_per_v_cpu(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty"]]:
            '''The minimum and maximum amount of memory per vCPU for an instance type, in GiB.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-memorygibpervcpu
            '''
            result = self._values.get("memory_gib_per_v_cpu")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty"]], result)

        @builtins.property
        def memory_mib(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MemoryMiBRequestProperty"]]:
            '''The minimum and maximum instance memory size for an instance type, in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-memorymib
            '''
            result = self._values.get("memory_mib")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.MemoryMiBRequestProperty"]], result)

        @builtins.property
        def network_bandwidth_gbps(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty"]]:
            '''The minimum and maximum amount of network bandwidth, in gigabits per second (Gbps).

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-networkbandwidthgbps
            '''
            result = self._values.get("network_bandwidth_gbps")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty"]], result)

        @builtins.property
        def network_interface_count(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty"]]:
            '''The minimum and maximum number of network interfaces for an instance type.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-networkinterfacecount
            '''
            result = self._values.get("network_interface_count")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty"]], result)

        @builtins.property
        def on_demand_max_price_percentage_over_lowest_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''The price protection threshold for On-Demand Instances.

            This is the maximum you’ll pay for an On-Demand Instance, expressed as a percentage higher than the least expensive current generation M, C, or R instance type with your specified attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price is higher than your threshold. The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage. To turn off price protection, specify a high value, such as ``999999`` .

            If you set ``DesiredCapacityType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per vCPU or per memory price instead of the per instance price.

            Default: ``20``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-ondemandmaxpricepercentageoverlowestprice
            '''
            result = self._values.get("on_demand_max_price_percentage_over_lowest_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def require_hibernate_support(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether instance types must provide On-Demand Instance hibernation support.

            Default: ``false``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-requirehibernatesupport
            '''
            result = self._values.get("require_hibernate_support")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def spot_max_price_percentage_over_lowest_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''The price protection threshold for Spot Instances.

            This is the maximum you’ll pay for a Spot Instance, expressed as a percentage higher than the least expensive current generation M, C, or R instance type with your specified attributes. When Amazon EC2 Auto Scaling selects instance types with your attributes, we will exclude instance types whose price is higher than your threshold. The parameter accepts an integer, which Amazon EC2 Auto Scaling interprets as a percentage. To turn off price protection, specify a high value, such as ``999999`` .

            If you set ``DesiredCapacityType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per vCPU or per memory price instead of the per instance price.

            Default: ``100``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-spotmaxpricepercentageoverlowestprice
            '''
            result = self._values.get("spot_max_price_percentage_over_lowest_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def total_local_storage_gb(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty"]]:
            '''The minimum and maximum total local storage size for an instance type, in GB.

            Default: No minimum or maximum limits

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-totallocalstoragegb
            '''
            result = self._values.get("total_local_storage_gb")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty"]], result)

        @builtins.property
        def v_cpu_count(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.VCpuCountRequestProperty"]]:
            '''The minimum and maximum number of vCPUs for an instance type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancerequirements.html#cfn-autoscaling-autoscalinggroup-instancerequirements-vcpucount
            '''
            result = self._values.get("v_cpu_count")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.VCpuCountRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceRequirementsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.InstancesDistributionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "on_demand_allocation_strategy": "onDemandAllocationStrategy",
            "on_demand_base_capacity": "onDemandBaseCapacity",
            "on_demand_percentage_above_base_capacity": "onDemandPercentageAboveBaseCapacity",
            "spot_allocation_strategy": "spotAllocationStrategy",
            "spot_instance_pools": "spotInstancePools",
            "spot_max_price": "spotMaxPrice",
        },
    )
    class InstancesDistributionProperty:
        def __init__(
            self,
            *,
            on_demand_allocation_strategy: typing.Optional[builtins.str] = None,
            on_demand_base_capacity: typing.Optional[jsii.Number] = None,
            on_demand_percentage_above_base_capacity: typing.Optional[jsii.Number] = None,
            spot_allocation_strategy: typing.Optional[builtins.str] = None,
            spot_instance_pools: typing.Optional[jsii.Number] = None,
            spot_max_price: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Use this structure to specify the distribution of On-Demand Instances and Spot Instances and the allocation strategies used to fulfill On-Demand and Spot capacities for a mixed instances policy.

            For more information, see `Auto Scaling groups with multiple instance types and purchase options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            ``InstancesDistribution`` is a property of the `AWS::AutoScaling::AutoScalingGroup MixedInstancesPolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-mixedinstancespolicy.html>`_ property type.

            :param on_demand_allocation_strategy: The allocation strategy to apply to your On-Demand Instances when they are launched. Possible instance types are determined by the launch template overrides that you specify. The following lists the valid values: - **lowest-price** - Uses price to determine which instance types are the highest priority, launching the lowest priced instance types within an Availability Zone first. This is the default value for Auto Scaling groups that specify ``InstanceRequirements`` . - **prioritized** - You set the order of instance types for the launch template overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling launches your highest priority instance types first. If all your On-Demand capacity cannot be fulfilled using your highest priority instance type, then Amazon EC2 Auto Scaling launches the remaining capacity using the second priority instance type, and so on. This is the default value for Auto Scaling groups that don't specify ``InstanceRequirements`` and cannot be used for groups that do.
            :param on_demand_base_capacity: The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances. This base portion is launched first as your group scales. This number has the same unit of measurement as the group's desired capacity. If you change the default unit of measurement (number of instances) by specifying weighted capacity values in your launch template overrides list, or by changing the default desired capacity type setting of the group, you must specify this number using the same unit of measurement. Default: 0 .. epigraph:: An update to this setting means a gradual replacement of instances to adjust the current On-Demand Instance levels. When replacing instances, Amazon EC2 Auto Scaling launches new instances before terminating the previous ones.
            :param on_demand_percentage_above_base_capacity: Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond ``OnDemandBaseCapacity`` . Expressed as a number (for example, 20 specifies 20% On-Demand Instances, 80% Spot Instances). If set to 100, only On-Demand Instances are used. Default: 100 .. epigraph:: An update to this setting means a gradual replacement of instances to adjust the current On-Demand and Spot Instance levels for your additional capacity higher than the base capacity. When replacing instances, Amazon EC2 Auto Scaling launches new instances before terminating the previous ones.
            :param spot_allocation_strategy: The allocation strategy to apply to your Spot Instances when they are launched. Possible instance types are determined by the launch template overrides that you specify. The following lists the valid values: - **capacity-optimized** - Requests Spot Instances using pools that are optimally chosen based on the available Spot capacity. This strategy has the lowest risk of interruption. To give certain instance types a higher chance of launching first, use ``capacity-optimized-prioritized`` . - **capacity-optimized-prioritized** - You set the order of instance types for the launch template overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best effort basis but optimizes for capacity first. Note that if the On-Demand allocation strategy is set to ``prioritized`` , the same priority is applied when fulfilling On-Demand capacity. This is not a valid value for Auto Scaling groups that specify ``InstanceRequirements`` . - **lowest-price** - Requests Spot Instances using the lowest priced pools within an Availability Zone, across the number of Spot pools that you specify for the ``SpotInstancePools`` property. To ensure that your desired capacity is met, you might receive Spot Instances from several pools. This is the default value, but it might lead to high interruption rates because this strategy only considers instance price and not available capacity. - **price-capacity-optimized (recommended)** - The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot Instance pools that are the least likely to be interrupted and have the lowest possible price.
            :param spot_instance_pools: The number of Spot Instance pools across which to allocate your Spot Instances. The Spot pools are determined from the different instance types in the overrides. Valid only when the ``SpotAllocationStrategy`` is ``lowest-price`` . Value must be in the range of 1–20. Default: 2
            :param spot_max_price: The maximum price per unit hour that you are willing to pay for a Spot Instance. If your maximum price is lower than the Spot price for the instance types that you selected, your Spot Instances are not launched. We do not recommend specifying a maximum price because it can lead to increased interruptions. When Spot Instances launch, you pay the current Spot price. To remove a maximum price that you previously set, include the property but specify an empty string ("") for the value. .. epigraph:: If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify one. Valid Range: Minimum value of 0.001

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                instances_distribution_property = autoscaling.CfnAutoScalingGroup.InstancesDistributionProperty(
                    on_demand_allocation_strategy="onDemandAllocationStrategy",
                    on_demand_base_capacity=123,
                    on_demand_percentage_above_base_capacity=123,
                    spot_allocation_strategy="spotAllocationStrategy",
                    spot_instance_pools=123,
                    spot_max_price="spotMaxPrice"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ca530211324712c4b14f117ed8b0d0dc0ab47386d30a812f8497ab80b7bdec40)
                check_type(argname="argument on_demand_allocation_strategy", value=on_demand_allocation_strategy, expected_type=type_hints["on_demand_allocation_strategy"])
                check_type(argname="argument on_demand_base_capacity", value=on_demand_base_capacity, expected_type=type_hints["on_demand_base_capacity"])
                check_type(argname="argument on_demand_percentage_above_base_capacity", value=on_demand_percentage_above_base_capacity, expected_type=type_hints["on_demand_percentage_above_base_capacity"])
                check_type(argname="argument spot_allocation_strategy", value=spot_allocation_strategy, expected_type=type_hints["spot_allocation_strategy"])
                check_type(argname="argument spot_instance_pools", value=spot_instance_pools, expected_type=type_hints["spot_instance_pools"])
                check_type(argname="argument spot_max_price", value=spot_max_price, expected_type=type_hints["spot_max_price"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if on_demand_allocation_strategy is not None:
                self._values["on_demand_allocation_strategy"] = on_demand_allocation_strategy
            if on_demand_base_capacity is not None:
                self._values["on_demand_base_capacity"] = on_demand_base_capacity
            if on_demand_percentage_above_base_capacity is not None:
                self._values["on_demand_percentage_above_base_capacity"] = on_demand_percentage_above_base_capacity
            if spot_allocation_strategy is not None:
                self._values["spot_allocation_strategy"] = spot_allocation_strategy
            if spot_instance_pools is not None:
                self._values["spot_instance_pools"] = spot_instance_pools
            if spot_max_price is not None:
                self._values["spot_max_price"] = spot_max_price

        @builtins.property
        def on_demand_allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''The allocation strategy to apply to your On-Demand Instances when they are launched.

            Possible instance types are determined by the launch template overrides that you specify.

            The following lists the valid values:

            - **lowest-price** - Uses price to determine which instance types are the highest priority, launching the lowest priced instance types within an Availability Zone first. This is the default value for Auto Scaling groups that specify ``InstanceRequirements`` .
            - **prioritized** - You set the order of instance types for the launch template overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling launches your highest priority instance types first. If all your On-Demand capacity cannot be fulfilled using your highest priority instance type, then Amazon EC2 Auto Scaling launches the remaining capacity using the second priority instance type, and so on. This is the default value for Auto Scaling groups that don't specify ``InstanceRequirements`` and cannot be used for groups that do.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html#cfn-autoscaling-autoscalinggroup-instancesdistribution-ondemandallocationstrategy
            '''
            result = self._values.get("on_demand_allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def on_demand_base_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances.

            This base portion is launched first as your group scales.

            This number has the same unit of measurement as the group's desired capacity. If you change the default unit of measurement (number of instances) by specifying weighted capacity values in your launch template overrides list, or by changing the default desired capacity type setting of the group, you must specify this number using the same unit of measurement.

            Default: 0
            .. epigraph::

               An update to this setting means a gradual replacement of instances to adjust the current On-Demand Instance levels. When replacing instances, Amazon EC2 Auto Scaling launches new instances before terminating the previous ones.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html#cfn-autoscaling-autoscalinggroup-instancesdistribution-ondemandbasecapacity
            '''
            result = self._values.get("on_demand_base_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def on_demand_percentage_above_base_capacity(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond ``OnDemandBaseCapacity`` .

            Expressed as a number (for example, 20 specifies 20% On-Demand Instances, 80% Spot Instances). If set to 100, only On-Demand Instances are used.

            Default: 100
            .. epigraph::

               An update to this setting means a gradual replacement of instances to adjust the current On-Demand and Spot Instance levels for your additional capacity higher than the base capacity. When replacing instances, Amazon EC2 Auto Scaling launches new instances before terminating the previous ones.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html#cfn-autoscaling-autoscalinggroup-instancesdistribution-ondemandpercentageabovebasecapacity
            '''
            result = self._values.get("on_demand_percentage_above_base_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spot_allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''The allocation strategy to apply to your Spot Instances when they are launched.

            Possible instance types are determined by the launch template overrides that you specify.

            The following lists the valid values:

            - **capacity-optimized** - Requests Spot Instances using pools that are optimally chosen based on the available Spot capacity. This strategy has the lowest risk of interruption. To give certain instance types a higher chance of launching first, use ``capacity-optimized-prioritized`` .
            - **capacity-optimized-prioritized** - You set the order of instance types for the launch template overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best effort basis but optimizes for capacity first. Note that if the On-Demand allocation strategy is set to ``prioritized`` , the same priority is applied when fulfilling On-Demand capacity. This is not a valid value for Auto Scaling groups that specify ``InstanceRequirements`` .
            - **lowest-price** - Requests Spot Instances using the lowest priced pools within an Availability Zone, across the number of Spot pools that you specify for the ``SpotInstancePools`` property. To ensure that your desired capacity is met, you might receive Spot Instances from several pools. This is the default value, but it might lead to high interruption rates because this strategy only considers instance price and not available capacity.
            - **price-capacity-optimized (recommended)** - The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot Instance pools that are the least likely to be interrupted and have the lowest possible price.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html#cfn-autoscaling-autoscalinggroup-instancesdistribution-spotallocationstrategy
            '''
            result = self._values.get("spot_allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def spot_instance_pools(self) -> typing.Optional[jsii.Number]:
            '''The number of Spot Instance pools across which to allocate your Spot Instances.

            The Spot pools are determined from the different instance types in the overrides. Valid only when the ``SpotAllocationStrategy`` is ``lowest-price`` . Value must be in the range of 1–20.

            Default: 2

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html#cfn-autoscaling-autoscalinggroup-instancesdistribution-spotinstancepools
            '''
            result = self._values.get("spot_instance_pools")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spot_max_price(self) -> typing.Optional[builtins.str]:
            '''The maximum price per unit hour that you are willing to pay for a Spot Instance.

            If your maximum price is lower than the Spot price for the instance types that you selected, your Spot Instances are not launched. We do not recommend specifying a maximum price because it can lead to increased interruptions. When Spot Instances launch, you pay the current Spot price. To remove a maximum price that you previously set, include the property but specify an empty string ("") for the value.
            .. epigraph::

               If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify one.

            Valid Range: Minimum value of 0.001

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-instancesdistribution.html#cfn-autoscaling-autoscalinggroup-instancesdistribution-spotmaxprice
            '''
            result = self._values.get("spot_max_price")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstancesDistributionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.LaunchTemplateOverridesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_requirements": "instanceRequirements",
            "instance_type": "instanceType",
            "launch_template_specification": "launchTemplateSpecification",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class LaunchTemplateOverridesProperty:
        def __init__(
            self,
            *,
            instance_requirements: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.InstanceRequirementsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            instance_type: typing.Optional[builtins.str] = None,
            launch_template_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.LaunchTemplateSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            weighted_capacity: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Use this structure to let Amazon EC2 Auto Scaling do the following when the Auto Scaling group has a mixed instances policy:  - Override the instance type that is specified in the launch template.

            - Use multiple instance types.

            Specify the instance types that you want, or define your instance requirements instead and let Amazon EC2 Auto Scaling provision the available instance types that meet your requirements. This can provide Amazon EC2 Auto Scaling with a larger selection of instance types to choose from when fulfilling Spot and On-Demand capacities. You can view which instance types are matched before you apply the instance requirements to your Auto Scaling group.

            After you define your instance requirements, you don't have to keep updating these settings to get new EC2 instance types automatically. Amazon EC2 Auto Scaling uses the instance requirements of the Auto Scaling group to determine whether a new EC2 instance type can be used.

            ``LaunchTemplateOverrides`` is a property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplate <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplate.html>`_ property type.

            :param instance_requirements: The instance requirements. Amazon EC2 Auto Scaling uses your specified requirements to identify instance types. Then, it uses your On-Demand and Spot allocation strategies to launch instances from these instance types. You can specify up to four separate sets of instance requirements per Auto Scaling group. This is useful for provisioning instances from different Amazon Machine Images (AMIs) in the same Auto Scaling group. To do this, create the AMIs and create a new launch template for each AMI. Then, create a compatible set of instance requirements for each launch template. .. epigraph:: If you specify ``InstanceRequirements`` , you can't specify ``InstanceType`` .
            :param instance_type: The instance type, such as ``m3.xlarge`` . You must specify an instance type that is supported in your requested Region and Availability Zones. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon Elastic Compute Cloud User Guide* . You can specify up to 40 instance types per Auto Scaling group.
            :param launch_template_specification: Provides a launch template for the specified instance type or set of instance requirements. For example, some instance types might require a launch template with a different AMI. If not provided, Amazon EC2 Auto Scaling uses the launch template that's specified in the ``LaunchTemplate`` definition. For more information, see `Specifying a different launch template for an instance type <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups-launch-template-overrides.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . You can specify up to 20 launch templates per Auto Scaling group. The launch templates specified in the overrides and in the ``LaunchTemplate`` definition count towards this limit.
            :param weighted_capacity: If you provide a list of instance types to use, you can specify the number of capacity units provided by each instance type in terms of virtual CPUs, memory, storage, throughput, or other relative performance characteristic. When a Spot or On-Demand Instance is launched, the capacity units count toward the desired capacity. Amazon EC2 Auto Scaling launches instances until the desired capacity is totally fulfilled, even if this results in an overage. For example, if there are two units remaining to fulfill capacity, and Amazon EC2 Auto Scaling can only launch an instance with a ``WeightedCapacity`` of five units, the instance is launched, and the desired capacity is exceeded by three units. For more information, see `Configure instance weighting for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups-instance-weighting.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Value must be in the range of 1-999. If you specify a value for ``WeightedCapacity`` for one instance type, you must specify a value for ``WeightedCapacity`` for all of them. .. epigraph:: Every Auto Scaling group has three size parameters ( ``DesiredCapacity`` , ``MaxSize`` , and ``MinSize`` ). Usually, you set these sizes based on a specific number of instances. However, if you configure a mixed instances policy that defines weights for the instance types, you must specify these sizes with the same units that you use for weighting instances.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplateoverrides.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                launch_template_overrides_property = autoscaling.CfnAutoScalingGroup.LaunchTemplateOverridesProperty(
                    instance_requirements=autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty(
                        accelerator_count=autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_manufacturers=["acceleratorManufacturers"],
                        accelerator_names=["acceleratorNames"],
                        accelerator_total_memory_mi_b=autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_types=["acceleratorTypes"],
                        allowed_instance_types=["allowedInstanceTypes"],
                        bare_metal="bareMetal",
                        baseline_ebs_bandwidth_mbps=autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        burstable_performance="burstablePerformance",
                        cpu_manufacturers=["cpuManufacturers"],
                        excluded_instance_types=["excludedInstanceTypes"],
                        instance_generations=["instanceGenerations"],
                        local_storage="localStorage",
                        local_storage_types=["localStorageTypes"],
                        memory_gi_bPer_vCpu=autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                            max=123,
                            min=123
                        ),
                        memory_mi_b=autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_bandwidth_gbps=autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_interface_count=autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        on_demand_max_price_percentage_over_lowest_price=123,
                        require_hibernate_support=False,
                        spot_max_price_percentage_over_lowest_price=123,
                        total_local_storage_gb=autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                            max=123,
                            min=123
                        ),
                        v_cpu_count=autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                            max=123,
                            min=123
                        )
                    ),
                    instance_type="instanceType",
                    launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                        version="version",
                
                        # the properties below are optional
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName"
                    ),
                    weighted_capacity="weightedCapacity"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__86885fea78f6e876794cae723fc0484932c1c7ba5727a1abf14f45861dbdb3d9)
                check_type(argname="argument instance_requirements", value=instance_requirements, expected_type=type_hints["instance_requirements"])
                check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
                check_type(argname="argument launch_template_specification", value=launch_template_specification, expected_type=type_hints["launch_template_specification"])
                check_type(argname="argument weighted_capacity", value=weighted_capacity, expected_type=type_hints["weighted_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if instance_requirements is not None:
                self._values["instance_requirements"] = instance_requirements
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if launch_template_specification is not None:
                self._values["launch_template_specification"] = launch_template_specification
            if weighted_capacity is not None:
                self._values["weighted_capacity"] = weighted_capacity

        @builtins.property
        def instance_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.InstanceRequirementsProperty"]]:
            '''The instance requirements.

            Amazon EC2 Auto Scaling uses your specified requirements to identify instance types. Then, it uses your On-Demand and Spot allocation strategies to launch instances from these instance types.

            You can specify up to four separate sets of instance requirements per Auto Scaling group. This is useful for provisioning instances from different Amazon Machine Images (AMIs) in the same Auto Scaling group. To do this, create the AMIs and create a new launch template for each AMI. Then, create a compatible set of instance requirements for each launch template.
            .. epigraph::

               If you specify ``InstanceRequirements`` , you can't specify ``InstanceType`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplateoverrides.html#cfn-as-mixedinstancespolicy-instancerequirements
            '''
            result = self._values.get("instance_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.InstanceRequirementsProperty"]], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''The instance type, such as ``m3.xlarge`` . You must specify an instance type that is supported in your requested Region and Availability Zones. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon Elastic Compute Cloud User Guide* .

            You can specify up to 40 instance types per Auto Scaling group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplateoverrides.html#cfn-autoscaling-autoscalinggroup-launchtemplateoverrides-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"]]:
            '''Provides a launch template for the specified instance type or set of instance requirements.

            For example, some instance types might require a launch template with a different AMI. If not provided, Amazon EC2 Auto Scaling uses the launch template that's specified in the ``LaunchTemplate`` definition. For more information, see `Specifying a different launch template for an instance type <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups-launch-template-overrides.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            You can specify up to 20 launch templates per Auto Scaling group. The launch templates specified in the overrides and in the ``LaunchTemplate`` definition count towards this limit.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplateoverrides.html#cfn-autoscaling-autoscalinggroup-launchtemplateoverrides-launchtemplatespecification
            '''
            result = self._values.get("launch_template_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"]], result)

        @builtins.property
        def weighted_capacity(self) -> typing.Optional[builtins.str]:
            '''If you provide a list of instance types to use, you can specify the number of capacity units provided by each instance type in terms of virtual CPUs, memory, storage, throughput, or other relative performance characteristic.

            When a Spot or On-Demand Instance is launched, the capacity units count toward the desired capacity. Amazon EC2 Auto Scaling launches instances until the desired capacity is totally fulfilled, even if this results in an overage. For example, if there are two units remaining to fulfill capacity, and Amazon EC2 Auto Scaling can only launch an instance with a ``WeightedCapacity`` of five units, the instance is launched, and the desired capacity is exceeded by three units. For more information, see `Configure instance weighting for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups-instance-weighting.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Value must be in the range of 1-999.

            If you specify a value for ``WeightedCapacity`` for one instance type, you must specify a value for ``WeightedCapacity`` for all of them.
            .. epigraph::

               Every Auto Scaling group has three size parameters ( ``DesiredCapacity`` , ``MaxSize`` , and ``MinSize`` ). Usually, you set these sizes based on a specific number of instances. However, if you configure a mixed instances policy that defines weights for the instance types, you must specify these sizes with the same units that you use for weighting instances.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplateoverrides.html#cfn-autoscaling-autoscalinggroup-launchtemplateoverrides-weightedcapacity
            '''
            result = self._values.get("weighted_capacity")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateOverridesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.LaunchTemplateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_specification": "launchTemplateSpecification",
            "overrides": "overrides",
        },
    )
    class LaunchTemplateProperty:
        def __init__(
            self,
            *,
            launch_template_specification: typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.LaunchTemplateSpecificationProperty", typing.Dict[builtins.str, typing.Any]]],
            overrides: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.LaunchTemplateOverridesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Use this structure to specify the launch templates and instance types (overrides) for a mixed instances policy.

            ``LaunchTemplate`` is a property of the `AWS::AutoScaling::AutoScalingGroup MixedInstancesPolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-mixedinstancespolicy.html>`_ property type.

            :param launch_template_specification: The launch template.
            :param overrides: Any properties that you specify override the same properties in the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                launch_template_property = autoscaling.CfnAutoScalingGroup.LaunchTemplateProperty(
                    launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                        version="version",
                
                        # the properties below are optional
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName"
                    ),
                
                    # the properties below are optional
                    overrides=[autoscaling.CfnAutoScalingGroup.LaunchTemplateOverridesProperty(
                        instance_requirements=autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty(
                            accelerator_count=autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_manufacturers=["acceleratorManufacturers"],
                            accelerator_names=["acceleratorNames"],
                            accelerator_total_memory_mi_b=autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_types=["acceleratorTypes"],
                            allowed_instance_types=["allowedInstanceTypes"],
                            bare_metal="bareMetal",
                            baseline_ebs_bandwidth_mbps=autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            burstable_performance="burstablePerformance",
                            cpu_manufacturers=["cpuManufacturers"],
                            excluded_instance_types=["excludedInstanceTypes"],
                            instance_generations=["instanceGenerations"],
                            local_storage="localStorage",
                            local_storage_types=["localStorageTypes"],
                            memory_gi_bPer_vCpu=autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                                max=123,
                                min=123
                            ),
                            memory_mi_b=autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_bandwidth_gbps=autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_interface_count=autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            on_demand_max_price_percentage_over_lowest_price=123,
                            require_hibernate_support=False,
                            spot_max_price_percentage_over_lowest_price=123,
                            total_local_storage_gb=autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                                max=123,
                                min=123
                            ),
                            v_cpu_count=autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                                max=123,
                                min=123
                            )
                        ),
                        instance_type="instanceType",
                        launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                            version="version",
                
                            # the properties below are optional
                            launch_template_id="launchTemplateId",
                            launch_template_name="launchTemplateName"
                        ),
                        weighted_capacity="weightedCapacity"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__058746461420956b542c7acb437dc8b0041e30c6486c094342adb80b393afe5d)
                check_type(argname="argument launch_template_specification", value=launch_template_specification, expected_type=type_hints["launch_template_specification"])
                check_type(argname="argument overrides", value=overrides, expected_type=type_hints["overrides"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "launch_template_specification": launch_template_specification,
            }
            if overrides is not None:
                self._values["overrides"] = overrides

        @builtins.property
        def launch_template_specification(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"]:
            '''The launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplate.html#cfn-as-group-launchtemplate
            '''
            result = self._values.get("launch_template_specification")
            assert result is not None, "Required property 'launch_template_specification' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateSpecificationProperty"], result)

        @builtins.property
        def overrides(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateOverridesProperty"]]]]:
            '''Any properties that you specify override the same properties in the launch template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-mixedinstancespolicy-launchtemplate.html#cfn-as-mixedinstancespolicy-overrides
            '''
            result = self._values.get("overrides")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateOverridesProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "version": "version",
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
        },
    )
    class LaunchTemplateSpecificationProperty:
        def __init__(
            self,
            *,
            version: builtins.str,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies a launch template to use when provisioning EC2 instances for an Auto Scaling group.

            You must specify the following:

            - The ID or the name of the launch template, but not both.
            - The version of the launch template.

            ``LaunchTemplateSpecification`` is property of the `AWS::AutoScaling::AutoScalingGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html>`_ resource. It is also a property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplate <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplate.html>`_ and `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property types.

            For information about creating a launch template, see `AWS::EC2::LaunchTemplate <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html>`_ and `Create a launch template for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-template.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            For examples of launch templates, see `Auto scaling template snippets <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/quickref-autoscaling.html>`_ and the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#aws-resource-ec2-launchtemplate--examples>`_ section in the ``AWS::EC2::LaunchTemplate`` resource.

            :param version: The version number of the launch template. Specifying ``$Latest`` or ``$Default`` for the template version number is not supported. However, you can specify ``LatestVersionNumber`` or ``DefaultVersionNumber`` using the ``Fn::GetAtt`` intrinsic function. For more information, see `Fn::GetAtt <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getatt.html>`_ . .. epigraph:: For an example of using the ``Fn::GetAtt`` function, see the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#aws-properties-as-group--examples>`_ section of the ``AWS::AutoScaling::AutoScalingGroup`` resource.
            :param launch_template_id: The ID of the launch template. You must specify the ``LaunchTemplateID`` or the ``LaunchTemplateName`` , but not both.
            :param launch_template_name: The name of the launch template. You must specify the ``LaunchTemplateName`` or the ``LaunchTemplateID`` , but not both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplatespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                launch_template_specification_property = autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                    version="version",
                
                    # the properties below are optional
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5fd486733794768df97cba56f5cfd683f8633b0926591bd68e5e61eb7c80a7a6)
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
                check_type(argname="argument launch_template_id", value=launch_template_id, expected_type=type_hints["launch_template_id"])
                check_type(argname="argument launch_template_name", value=launch_template_name, expected_type=type_hints["launch_template_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "version": version,
            }
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name

        @builtins.property
        def version(self) -> builtins.str:
            '''The version number of the launch template.

            Specifying ``$Latest`` or ``$Default`` for the template version number is not supported. However, you can specify ``LatestVersionNumber`` or ``DefaultVersionNumber`` using the ``Fn::GetAtt`` intrinsic function. For more information, see `Fn::GetAtt <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getatt.html>`_ .
            .. epigraph::

               For an example of using the ``Fn::GetAtt`` function, see the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#aws-properties-as-group--examples>`_ section of the ``AWS::AutoScaling::AutoScalingGroup`` resource.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplatespecification.html#cfn-autoscaling-autoscalinggroup-launchtemplatespecification-version
            '''
            result = self._values.get("version")
            assert result is not None, "Required property 'version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            You must specify the ``LaunchTemplateID`` or the ``LaunchTemplateName`` , but not both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplatespecification.html#cfn-autoscaling-autoscalinggroup-launchtemplatespecification-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            You must specify the ``LaunchTemplateName`` or the ``LaunchTemplateID`` , but not both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplatespecification.html#cfn-autoscaling-autoscalinggroup-launchtemplatespecification-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.LifecycleHookSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "lifecycle_hook_name": "lifecycleHookName",
            "lifecycle_transition": "lifecycleTransition",
            "default_result": "defaultResult",
            "heartbeat_timeout": "heartbeatTimeout",
            "notification_metadata": "notificationMetadata",
            "notification_target_arn": "notificationTargetArn",
            "role_arn": "roleArn",
        },
    )
    class LifecycleHookSpecificationProperty:
        def __init__(
            self,
            *,
            lifecycle_hook_name: builtins.str,
            lifecycle_transition: builtins.str,
            default_result: typing.Optional[builtins.str] = None,
            heartbeat_timeout: typing.Optional[jsii.Number] = None,
            notification_metadata: typing.Optional[builtins.str] = None,
            notification_target_arn: typing.Optional[builtins.str] = None,
            role_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``LifecycleHookSpecification`` specifies a lifecycle hook for the ``LifecycleHookSpecificationList`` property of the `AWS::AutoScaling::AutoScalingGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html>`_ resource. A lifecycle hook specifies actions to perform when Amazon EC2 Auto Scaling launches or terminates instances.

            For more information, see `Amazon EC2 Auto Scaling lifecycle hooks <https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . You can find a sample template snippet in the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-as-lifecyclehook.html#aws-resource-as-lifecyclehook--examples>`_ section of the ``AWS::AutoScaling::LifecycleHook`` resource.

            :param lifecycle_hook_name: The name of the lifecycle hook.
            :param lifecycle_transition: The lifecycle transition. For Auto Scaling groups, there are two major lifecycle transitions. - To create a lifecycle hook for scale-out events, specify ``autoscaling:EC2_INSTANCE_LAUNCHING`` . - To create a lifecycle hook for scale-in events, specify ``autoscaling:EC2_INSTANCE_TERMINATING`` .
            :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. The default value is ``ABANDON`` . Valid values: ``CONTINUE`` | ``ABANDON``
            :param heartbeat_timeout: The maximum time, in seconds, that can elapse before the lifecycle hook times out. The range is from ``30`` to ``7200`` seconds. The default value is ``3600`` seconds (1 hour).
            :param notification_metadata: Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the notification target.
            :param notification_target_arn: The Amazon Resource Name (ARN) of the notification target that Amazon EC2 Auto Scaling sends notifications to when an instance is in a wait state for the lifecycle hook. You can specify an Amazon SNS topic or an Amazon SQS queue.
            :param role_arn: The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target. For information about creating this role, see `Configure a notification target for a lifecycle hook <https://docs.aws.amazon.com/autoscaling/ec2/userguide/prepare-for-lifecycle-notifications.html#lifecycle-hook-notification-target>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid only if the notification target is an Amazon SNS topic or an Amazon SQS queue.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                lifecycle_hook_specification_property = autoscaling.CfnAutoScalingGroup.LifecycleHookSpecificationProperty(
                    lifecycle_hook_name="lifecycleHookName",
                    lifecycle_transition="lifecycleTransition",
                
                    # the properties below are optional
                    default_result="defaultResult",
                    heartbeat_timeout=123,
                    notification_metadata="notificationMetadata",
                    notification_target_arn="notificationTargetArn",
                    role_arn="roleArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dfaa90f01b32f1c94bbd73c5a7ad9f9eb0fbe2c3152e55e2ffab5891022193ef)
                check_type(argname="argument lifecycle_hook_name", value=lifecycle_hook_name, expected_type=type_hints["lifecycle_hook_name"])
                check_type(argname="argument lifecycle_transition", value=lifecycle_transition, expected_type=type_hints["lifecycle_transition"])
                check_type(argname="argument default_result", value=default_result, expected_type=type_hints["default_result"])
                check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
                check_type(argname="argument notification_metadata", value=notification_metadata, expected_type=type_hints["notification_metadata"])
                check_type(argname="argument notification_target_arn", value=notification_target_arn, expected_type=type_hints["notification_target_arn"])
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "lifecycle_hook_name": lifecycle_hook_name,
                "lifecycle_transition": lifecycle_transition,
            }
            if default_result is not None:
                self._values["default_result"] = default_result
            if heartbeat_timeout is not None:
                self._values["heartbeat_timeout"] = heartbeat_timeout
            if notification_metadata is not None:
                self._values["notification_metadata"] = notification_metadata
            if notification_target_arn is not None:
                self._values["notification_target_arn"] = notification_target_arn
            if role_arn is not None:
                self._values["role_arn"] = role_arn

        @builtins.property
        def lifecycle_hook_name(self) -> builtins.str:
            '''The name of the lifecycle hook.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-lifecyclehookname
            '''
            result = self._values.get("lifecycle_hook_name")
            assert result is not None, "Required property 'lifecycle_hook_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def lifecycle_transition(self) -> builtins.str:
            '''The lifecycle transition. For Auto Scaling groups, there are two major lifecycle transitions.

            - To create a lifecycle hook for scale-out events, specify ``autoscaling:EC2_INSTANCE_LAUNCHING`` .
            - To create a lifecycle hook for scale-in events, specify ``autoscaling:EC2_INSTANCE_TERMINATING`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-lifecycletransition
            '''
            result = self._values.get("lifecycle_transition")
            assert result is not None, "Required property 'lifecycle_transition' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def default_result(self) -> typing.Optional[builtins.str]:
            '''The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs.

            The default value is ``ABANDON`` .

            Valid values: ``CONTINUE`` | ``ABANDON``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-defaultresult
            '''
            result = self._values.get("default_result")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def heartbeat_timeout(self) -> typing.Optional[jsii.Number]:
            '''The maximum time, in seconds, that can elapse before the lifecycle hook times out.

            The range is from ``30`` to ``7200`` seconds. The default value is ``3600`` seconds (1 hour).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-heartbeattimeout
            '''
            result = self._values.get("heartbeat_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def notification_metadata(self) -> typing.Optional[builtins.str]:
            '''Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the notification target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-notificationmetadata
            '''
            result = self._values.get("notification_metadata")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def notification_target_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the notification target that Amazon EC2 Auto Scaling sends notifications to when an instance is in a wait state for the lifecycle hook.

            You can specify an Amazon SNS topic or an Amazon SQS queue.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-notificationtargetarn
            '''
            result = self._values.get("notification_target_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def role_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target.

            For information about creating this role, see `Configure a notification target for a lifecycle hook <https://docs.aws.amazon.com/autoscaling/ec2/userguide/prepare-for-lifecycle-notifications.html#lifecycle-hook-notification-target>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            Valid only if the notification target is an Amazon SNS topic or an Amazon SQS queue.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-lifecyclehookspecification.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecification-rolearn
            '''
            result = self._values.get("role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LifecycleHookSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class MemoryGiBPerVCpuRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``MemoryGiBPerVCpuRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum amount of memory per vCPU for an instance type, in GiB.

            :param max: The memory maximum in GiB.
            :param min: The memory minimum in GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-memorygibpervcpurequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                memory_gi_bPer_vCpu_request_property = autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0fa0016fc02beef29b5f52f57c3c8e7274995a201a6ea9623dbe586aa535b1ff)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The memory maximum in GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-memorygibpervcpurequest.html#cfn-autoscaling-autoscalinggroup-memorygibpervcpurequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The memory minimum in GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-memorygibpervcpurequest.html#cfn-autoscaling-autoscalinggroup-memorygibpervcpurequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryGiBPerVCpuRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class MemoryMiBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``MemoryMiBRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum instance memory size for an instance type, in MiB.

            :param max: The memory maximum in MiB.
            :param min: The memory minimum in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-memorymibrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                memory_mi_bRequest_property = autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__95ef1554000b7f543a637e992c7b0cbaa71e219877b4f18b7571da812b4741b8)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The memory maximum in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-memorymibrequest.html#cfn-autoscaling-autoscalinggroup-memorymibrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The memory minimum in MiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-memorymibrequest.html#cfn-autoscaling-autoscalinggroup-memorymibrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryMiBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.MetricsCollectionProperty",
        jsii_struct_bases=[],
        name_mapping={"granularity": "granularity", "metrics": "metrics"},
    )
    class MetricsCollectionProperty:
        def __init__(
            self,
            *,
            granularity: builtins.str,
            metrics: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''``MetricsCollection`` is a property of the `AWS::AutoScaling::AutoScalingGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html>`_ resource that describes the group metrics that an Amazon EC2 Auto Scaling group sends to Amazon CloudWatch. These metrics describe the group rather than any of its instances.

            For more information, see `Monitor CloudWatch metrics for your Auto Scaling groups and instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-monitoring.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . You can find a sample template snippet in the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#aws-properties-as-group--examples>`_ section of the ``AWS::AutoScaling::AutoScalingGroup`` resource.

            :param granularity: The frequency at which Amazon EC2 Auto Scaling sends aggregated data to CloudWatch. The only valid value is ``1Minute`` .
            :param metrics: Identifies the metrics to enable. You can specify one or more of the following metrics: - ``GroupMinSize`` - ``GroupMaxSize`` - ``GroupDesiredCapacity`` - ``GroupInServiceInstances`` - ``GroupPendingInstances`` - ``GroupStandbyInstances`` - ``GroupTerminatingInstances`` - ``GroupTotalInstances`` - ``GroupInServiceCapacity`` - ``GroupPendingCapacity`` - ``GroupStandbyCapacity`` - ``GroupTerminatingCapacity`` - ``GroupTotalCapacity`` - ``WarmPoolDesiredCapacity`` - ``WarmPoolWarmedCapacity`` - ``WarmPoolPendingCapacity`` - ``WarmPoolTerminatingCapacity`` - ``WarmPoolTotalCapacity`` - ``GroupAndWarmPoolDesiredCapacity`` - ``GroupAndWarmPoolTotalCapacity`` If you specify ``Granularity`` and don't specify any metrics, all metrics are enabled. For more information, see `Auto Scaling group metrics <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-cloudwatch-monitoring.html#as-group-metrics>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-metricscollection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                metrics_collection_property = autoscaling.CfnAutoScalingGroup.MetricsCollectionProperty(
                    granularity="granularity",
                
                    # the properties below are optional
                    metrics=["metrics"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__11d67f55b88e32b71a73d39b67ec573e08571c2264ca36d4a9c9e62c7ef1478f)
                check_type(argname="argument granularity", value=granularity, expected_type=type_hints["granularity"])
                check_type(argname="argument metrics", value=metrics, expected_type=type_hints["metrics"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "granularity": granularity,
            }
            if metrics is not None:
                self._values["metrics"] = metrics

        @builtins.property
        def granularity(self) -> builtins.str:
            '''The frequency at which Amazon EC2 Auto Scaling sends aggregated data to CloudWatch.

            The only valid value is ``1Minute`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-metricscollection.html#cfn-as-metricscollection-granularity
            '''
            result = self._values.get("granularity")
            assert result is not None, "Required property 'granularity' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def metrics(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Identifies the metrics to enable.

            You can specify one or more of the following metrics:

            - ``GroupMinSize``
            - ``GroupMaxSize``
            - ``GroupDesiredCapacity``
            - ``GroupInServiceInstances``
            - ``GroupPendingInstances``
            - ``GroupStandbyInstances``
            - ``GroupTerminatingInstances``
            - ``GroupTotalInstances``
            - ``GroupInServiceCapacity``
            - ``GroupPendingCapacity``
            - ``GroupStandbyCapacity``
            - ``GroupTerminatingCapacity``
            - ``GroupTotalCapacity``
            - ``WarmPoolDesiredCapacity``
            - ``WarmPoolWarmedCapacity``
            - ``WarmPoolPendingCapacity``
            - ``WarmPoolTerminatingCapacity``
            - ``WarmPoolTotalCapacity``
            - ``GroupAndWarmPoolDesiredCapacity``
            - ``GroupAndWarmPoolTotalCapacity``

            If you specify ``Granularity`` and don't specify any metrics, all metrics are enabled.

            For more information, see `Auto Scaling group metrics <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-cloudwatch-monitoring.html#as-group-metrics>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-metricscollection.html#cfn-as-metricscollection-metrics
            '''
            result = self._values.get("metrics")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricsCollectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.MixedInstancesPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template": "launchTemplate",
            "instances_distribution": "instancesDistribution",
        },
    )
    class MixedInstancesPolicyProperty:
        def __init__(
            self,
            *,
            launch_template: typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.LaunchTemplateProperty", typing.Dict[builtins.str, typing.Any]]],
            instances_distribution: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAutoScalingGroup.InstancesDistributionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Use this structure to launch multiple instance types and On-Demand Instances and Spot Instances within a single Auto Scaling group.

            A mixed instances policy contains information that Amazon EC2 Auto Scaling can use to launch instances and help optimize your costs. For more information, see `Auto Scaling groups with multiple instance types and purchase options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            You can create a mixed instances policy for new and existing Auto Scaling groups. You must use a launch template to configure the policy. You cannot use a launch configuration.

            There are key differences between Spot Instances and On-Demand Instances:

            - The price for Spot Instances varies based on demand
            - Amazon EC2 can terminate an individual Spot Instance as the availability of, or price for, Spot Instances changes

            When a Spot Instance is terminated, Amazon EC2 Auto Scaling group attempts to launch a replacement instance to maintain the desired capacity for the group.

            ``MixedInstancesPolicy`` is a property of the `AWS::AutoScaling::AutoScalingGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html>`_ resource.

            :param launch_template: One or more launch templates and the instance types (overrides) that are used to launch EC2 instances to fulfill On-Demand and Spot capacities.
            :param instances_distribution: The instances distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-group-mixedinstancespolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                mixed_instances_policy_property = autoscaling.CfnAutoScalingGroup.MixedInstancesPolicyProperty(
                    launch_template=autoscaling.CfnAutoScalingGroup.LaunchTemplateProperty(
                        launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                            version="version",
                
                            # the properties below are optional
                            launch_template_id="launchTemplateId",
                            launch_template_name="launchTemplateName"
                        ),
                
                        # the properties below are optional
                        overrides=[autoscaling.CfnAutoScalingGroup.LaunchTemplateOverridesProperty(
                            instance_requirements=autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty(
                                accelerator_count=autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                accelerator_manufacturers=["acceleratorManufacturers"],
                                accelerator_names=["acceleratorNames"],
                                accelerator_total_memory_mi_b=autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                accelerator_types=["acceleratorTypes"],
                                allowed_instance_types=["allowedInstanceTypes"],
                                bare_metal="bareMetal",
                                baseline_ebs_bandwidth_mbps=autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                burstable_performance="burstablePerformance",
                                cpu_manufacturers=["cpuManufacturers"],
                                excluded_instance_types=["excludedInstanceTypes"],
                                instance_generations=["instanceGenerations"],
                                local_storage="localStorage",
                                local_storage_types=["localStorageTypes"],
                                memory_gi_bPer_vCpu=autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                memory_mi_b=autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                network_bandwidth_gbps=autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                network_interface_count=autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                on_demand_max_price_percentage_over_lowest_price=123,
                                require_hibernate_support=False,
                                spot_max_price_percentage_over_lowest_price=123,
                                total_local_storage_gb=autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                v_cpu_count=autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                                    max=123,
                                    min=123
                                )
                            ),
                            instance_type="instanceType",
                            launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                                version="version",
                
                                # the properties below are optional
                                launch_template_id="launchTemplateId",
                                launch_template_name="launchTemplateName"
                            ),
                            weighted_capacity="weightedCapacity"
                        )]
                    ),
                
                    # the properties below are optional
                    instances_distribution=autoscaling.CfnAutoScalingGroup.InstancesDistributionProperty(
                        on_demand_allocation_strategy="onDemandAllocationStrategy",
                        on_demand_base_capacity=123,
                        on_demand_percentage_above_base_capacity=123,
                        spot_allocation_strategy="spotAllocationStrategy",
                        spot_instance_pools=123,
                        spot_max_price="spotMaxPrice"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a4e1c1acb02ba7d86e2ea0004ef9019d45d3f8ed06dbe7671e059b17f5d38c71)
                check_type(argname="argument launch_template", value=launch_template, expected_type=type_hints["launch_template"])
                check_type(argname="argument instances_distribution", value=instances_distribution, expected_type=type_hints["instances_distribution"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "launch_template": launch_template,
            }
            if instances_distribution is not None:
                self._values["instances_distribution"] = instances_distribution

        @builtins.property
        def launch_template(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateProperty"]:
            '''One or more launch templates and the instance types (overrides) that are used to launch EC2 instances to fulfill On-Demand and Spot capacities.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-group-mixedinstancespolicy.html#cfn-as-mixedinstancespolicy-launchtemplate
            '''
            result = self._values.get("launch_template")
            assert result is not None, "Required property 'launch_template' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.LaunchTemplateProperty"], result)

        @builtins.property
        def instances_distribution(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.InstancesDistributionProperty"]]:
            '''The instances distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-as-group-mixedinstancespolicy.html#cfn-as-mixedinstancespolicy-instancesdistribution
            '''
            result = self._values.get("instances_distribution")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAutoScalingGroup.InstancesDistributionProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MixedInstancesPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class NetworkBandwidthGbpsRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``NetworkBandwidthGbpsRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum network bandwidth for an instance type, in Gbps.

            .. epigraph::

               Setting the minimum bandwidth does not guarantee that your instance will achieve the minimum bandwidth. Amazon EC2 will identify instance types that support the specified minimum bandwidth, but the actual bandwidth of your instance might go below the specified minimum at times. For more information, see `Available instance bandwidth <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html#available-instance-bandwidth>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            :param max: The maximum amount of network bandwidth, in gigabits per second (Gbps).
            :param min: The minimum amount of network bandwidth, in gigabits per second (Gbps).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-networkbandwidthgbpsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                network_bandwidth_gbps_request_property = autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__51067dfafe115e0bef6bda658386efebaab07af5ce9fbf8cb17ca54b8965e1f8)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of network bandwidth, in gigabits per second (Gbps).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-networkbandwidthgbpsrequest.html#cfn-autoscaling-autoscalinggroup-networkbandwidthgbpsrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of network bandwidth, in gigabits per second (Gbps).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-networkbandwidthgbpsrequest.html#cfn-autoscaling-autoscalinggroup-networkbandwidthgbpsrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkBandwidthGbpsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class NetworkInterfaceCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``NetworkInterfaceCountRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum number of network interfaces for an instance type.

            :param max: The maximum number of network interfaces.
            :param min: The minimum number of network interfaces.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-networkinterfacecountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                network_interface_count_request_property = autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0cee3ceb45798c5d9e25da963c1be1070967f06a2f3236c1e31761d602788aab)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of network interfaces.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-networkinterfacecountrequest.html#cfn-autoscaling-autoscalinggroup-networkinterfacecountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of network interfaces.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-networkinterfacecountrequest.html#cfn-autoscaling-autoscalinggroup-networkinterfacecountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkInterfaceCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.NotificationConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "topic_arn": "topicArn",
            "notification_types": "notificationTypes",
        },
    )
    class NotificationConfigurationProperty:
        def __init__(
            self,
            *,
            topic_arn: builtins.str,
            notification_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''A structure that specifies an Amazon SNS notification configuration for the ``NotificationConfigurations`` property of the `AWS::AutoScaling::AutoScalingGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html>`_ resource.

            For an example template snippet, see `Auto scaling template snippets <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/quickref-autoscaling.html>`_ .

            For more information, see `Get Amazon SNS notifications when your Auto Scaling group scales <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ASGettingNotifications.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param topic_arn: The Amazon Resource Name (ARN) of the Amazon SNS topic.
            :param notification_types: A list of event types that send a notification. Event types can include any of the following types. *Allowed values* : - ``autoscaling:EC2_INSTANCE_LAUNCH`` - ``autoscaling:EC2_INSTANCE_LAUNCH_ERROR`` - ``autoscaling:EC2_INSTANCE_TERMINATE`` - ``autoscaling:EC2_INSTANCE_TERMINATE_ERROR`` - ``autoscaling:TEST_NOTIFICATION``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-notificationconfigurations.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                notification_configuration_property = autoscaling.CfnAutoScalingGroup.NotificationConfigurationProperty(
                    topic_arn="topicArn",
                
                    # the properties below are optional
                    notification_types=["notificationTypes"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5d6e36a12e4115639643f5e3b4951b7f3b869c9a2cb0f8a0b3bd14224dfd252a)
                check_type(argname="argument topic_arn", value=topic_arn, expected_type=type_hints["topic_arn"])
                check_type(argname="argument notification_types", value=notification_types, expected_type=type_hints["notification_types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "topic_arn": topic_arn,
            }
            if notification_types is not None:
                self._values["notification_types"] = notification_types

        @builtins.property
        def topic_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Amazon SNS topic.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-notificationconfigurations.html#cfn-autoscaling-autoscalinggroup-notificationconfigurations-topicarn
            '''
            result = self._values.get("topic_arn")
            assert result is not None, "Required property 'topic_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def notification_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of event types that send a notification. Event types can include any of the following types.

            *Allowed values* :

            - ``autoscaling:EC2_INSTANCE_LAUNCH``
            - ``autoscaling:EC2_INSTANCE_LAUNCH_ERROR``
            - ``autoscaling:EC2_INSTANCE_TERMINATE``
            - ``autoscaling:EC2_INSTANCE_TERMINATE_ERROR``
            - ``autoscaling:TEST_NOTIFICATION``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-notificationconfigurations.html#cfn-as-group-notificationconfigurations-notificationtypes
            '''
            result = self._values.get("notification_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NotificationConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.TagPropertyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "key": "key",
            "propagate_at_launch": "propagateAtLaunch",
            "value": "value",
        },
    )
    class TagPropertyProperty:
        def __init__(
            self,
            *,
            key: builtins.str,
            propagate_at_launch: typing.Union[builtins.bool, _IResolvable_da3f097b],
            value: builtins.str,
        ) -> None:
            '''A structure that specifies a tag for the ``Tags`` property of `AWS::AutoScaling::AutoScalingGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html>`_ resource.

            For more information, see `Tag Auto Scaling groups and instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-tagging.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . You can find a sample template snippet in the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#aws-properties-as-group--examples>`_ section of the ``AWS::AutoScaling::AutoScalingGroup`` resource.

            CloudFormation adds the following tags to all Auto Scaling groups and associated instances:

            - aws:cloudformation:stack-name
            - aws:cloudformation:stack-id
            - aws:cloudformation:logical-id

            :param key: The tag key.
            :param propagate_at_launch: Set to ``true`` if you want CloudFormation to copy the tag to EC2 instances that are launched as part of the Auto Scaling group. Set to ``false`` if you want the tag attached only to the Auto Scaling group and not copied to any instances launched as part of the Auto Scaling group.
            :param value: The tag value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-tags.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                tag_property_property = autoscaling.CfnAutoScalingGroup.TagPropertyProperty(
                    key="key",
                    propagate_at_launch=False,
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fbe45b876c171c89a71ab1315093a8b32d7ef922f5c3986e220789bfa73f9633)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument propagate_at_launch", value=propagate_at_launch, expected_type=type_hints["propagate_at_launch"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "propagate_at_launch": propagate_at_launch,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The tag key.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-tags.html#cfn-as-tags-Key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def propagate_at_launch(
            self,
        ) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Set to ``true`` if you want CloudFormation to copy the tag to EC2 instances that are launched as part of the Auto Scaling group.

            Set to ``false`` if you want the tag attached only to the Auto Scaling group and not copied to any instances launched as part of the Auto Scaling group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-tags.html#cfn-as-tags-PropagateAtLaunch
            '''
            result = self._values.get("propagate_at_launch")
            assert result is not None, "Required property 'propagate_at_launch' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The tag value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-tags.html#cfn-as-tags-Value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagPropertyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class TotalLocalStorageGBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``TotalLocalStorageGBRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum total local storage size for an instance type, in GB.

            :param max: The storage maximum in GB.
            :param min: The storage minimum in GB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-totallocalstoragegbrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                total_local_storage_gBRequest_property = autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3ed6e3c63aa6516366b4a7b4feb56e0d620440d9bba7489337d9951d29b96f8a)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The storage maximum in GB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-totallocalstoragegbrequest.html#cfn-autoscaling-autoscalinggroup-totallocalstoragegbrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The storage minimum in GB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-totallocalstoragegbrequest.html#cfn-autoscaling-autoscalinggroup-totallocalstoragegbrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TotalLocalStorageGBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class VCpuCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``VCpuCountRequest`` is a property of the ``InstanceRequirements`` property of the `AWS::AutoScaling::AutoScalingGroup LaunchTemplateOverrides <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-launchtemplateoverrides.html>`_ property type that describes the minimum and maximum number of vCPUs for an instance type.

            :param max: The maximum number of vCPUs.
            :param min: The minimum number of vCPUs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-vcpucountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                v_cpu_count_request_property = autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a9c3111cac11a4fd207e172c8a19e31005ad8e2cfb7a69a76a4191e7dad42224)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of vCPUs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-vcpucountrequest.html#cfn-autoscaling-autoscalinggroup-vcpucountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of vCPUs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-autoscalinggroup-vcpucountrequest.html#cfn-autoscaling-autoscalinggroup-vcpucountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VCpuCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnAutoScalingGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "max_size": "maxSize",
        "min_size": "minSize",
        "auto_scaling_group_name": "autoScalingGroupName",
        "availability_zones": "availabilityZones",
        "capacity_rebalance": "capacityRebalance",
        "context": "context",
        "cooldown": "cooldown",
        "default_instance_warmup": "defaultInstanceWarmup",
        "desired_capacity": "desiredCapacity",
        "desired_capacity_type": "desiredCapacityType",
        "health_check_grace_period": "healthCheckGracePeriod",
        "health_check_type": "healthCheckType",
        "instance_id": "instanceId",
        "launch_configuration_name": "launchConfigurationName",
        "launch_template": "launchTemplate",
        "lifecycle_hook_specification_list": "lifecycleHookSpecificationList",
        "load_balancer_names": "loadBalancerNames",
        "max_instance_lifetime": "maxInstanceLifetime",
        "metrics_collection": "metricsCollection",
        "mixed_instances_policy": "mixedInstancesPolicy",
        "new_instances_protected_from_scale_in": "newInstancesProtectedFromScaleIn",
        "notification_configurations": "notificationConfigurations",
        "placement_group": "placementGroup",
        "service_linked_role_arn": "serviceLinkedRoleArn",
        "tags": "tags",
        "target_group_arns": "targetGroupArns",
        "termination_policies": "terminationPolicies",
        "vpc_zone_identifier": "vpcZoneIdentifier",
    },
)
class CfnAutoScalingGroupProps:
    def __init__(
        self,
        *,
        max_size: builtins.str,
        min_size: builtins.str,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        capacity_rebalance: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        context: typing.Optional[builtins.str] = None,
        cooldown: typing.Optional[builtins.str] = None,
        default_instance_warmup: typing.Optional[jsii.Number] = None,
        desired_capacity: typing.Optional[builtins.str] = None,
        desired_capacity_type: typing.Optional[builtins.str] = None,
        health_check_grace_period: typing.Optional[jsii.Number] = None,
        health_check_type: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        launch_configuration_name: typing.Optional[builtins.str] = None,
        launch_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAutoScalingGroup.LaunchTemplateSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        lifecycle_hook_specification_list: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAutoScalingGroup.LifecycleHookSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        load_balancer_names: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_instance_lifetime: typing.Optional[jsii.Number] = None,
        metrics_collection: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAutoScalingGroup.MetricsCollectionProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        mixed_instances_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAutoScalingGroup.MixedInstancesPolicyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        new_instances_protected_from_scale_in: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        notification_configurations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAutoScalingGroup.NotificationConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        placement_group: typing.Optional[builtins.str] = None,
        service_linked_role_arn: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[CfnAutoScalingGroup.TagPropertyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        target_group_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        termination_policies: typing.Optional[typing.Sequence[builtins.str]] = None,
        vpc_zone_identifier: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAutoScalingGroup``.

        :param max_size: The maximum size of the group. .. epigraph:: With a mixed instances policy that uses instance weighting, Amazon EC2 Auto Scaling may need to go above ``MaxSize`` to meet your capacity requirements. In this event, Amazon EC2 Auto Scaling will never go above ``MaxSize`` by more than your largest instance weight (weights that define how many units each instance contributes to the desired capacity of the group).
        :param min_size: The minimum size of the group.
        :param auto_scaling_group_name: The name of the Auto Scaling group. This name must be unique per Region per account. The name can contain any ASCII character 33 to 126 including most punctuation characters, digits, and upper and lowercased letters. .. epigraph:: You cannot use a colon (:) in the name.
        :param availability_zones: A list of Availability Zones where instances in the Auto Scaling group can be created. Used for launching into the default VPC subnet in each Availability Zone when not using the ``VPCZoneIdentifier`` property, or for attaching a network interface when an existing network interface ID is specified in a launch template.
        :param capacity_rebalance: Indicates whether Capacity Rebalancing is enabled. Otherwise, Capacity Rebalancing is disabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. For more information, see `Use Capacity Rebalancing to handle Amazon EC2 Spot Interruptions <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-capacity-rebalancing.html>`_ in the in the *Amazon EC2 Auto Scaling User Guide* .
        :param context: Reserved.
        :param cooldown: *Only needed if you use simple scaling policies.*. The amount of time, in seconds, between one scaling activity ending and another one starting due to simple scaling policies. For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Default: ``300`` seconds
        :param default_instance_warmup: The amount of time, in seconds, until a new instance is considered to have finished initializing and resource consumption to become stable after it enters the ``InService`` state. During an instance refresh, Amazon EC2 Auto Scaling waits for the warm-up period after it replaces an instance before it moves on to replacing the next instance. Amazon EC2 Auto Scaling also waits for the warm-up period before aggregating the metrics for new instances with existing instances in the Amazon CloudWatch metrics that are used for scaling, resulting in more reliable usage data. For more information, see `Set the default instance warmup for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . .. epigraph:: To manage various warm-up settings at the group level, we recommend that you set the default instance warmup, *even if it is set to 0 seconds* . To remove a value that you previously set, include the property but specify ``-1`` for the value. However, we strongly recommend keeping the default instance warmup enabled by specifying a value of ``0`` or other nominal value. Default: None
        :param desired_capacity: The desired capacity is the initial capacity of the Auto Scaling group at the time of its creation and the capacity it attempts to maintain. It can scale beyond this capacity if you configure automatic scaling. The number must be greater than or equal to the minimum size of the group and less than or equal to the maximum size of the group. If you do not specify a desired capacity when creating the stack, the default is the minimum size of the group. CloudFormation marks the Auto Scaling group as successful (by setting its status to CREATE_COMPLETE) when the desired capacity is reached. However, if a maximum Spot price is set in the launch template or launch configuration that you specified, then desired capacity is not used as a criteria for success. Whether your request is fulfilled depends on Spot Instance capacity and your maximum price.
        :param desired_capacity_type: The unit of measurement for the value specified for desired capacity. Amazon EC2 Auto Scaling supports ``DesiredCapacityType`` for attribute-based instance type selection only. For more information, see `Creating an Auto Scaling group using attribute-based instance type selection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-instance-type-requirements.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . By default, Amazon EC2 Auto Scaling specifies ``units`` , which translates into number of instances. Valid values: ``units`` | ``vcpu`` | ``memory-mib``
        :param health_check_grace_period: The amount of time, in seconds, that Amazon EC2 Auto Scaling waits before checking the health status of an EC2 instance that has come into service and marking it unhealthy due to a failed health check. This is useful if your instances do not immediately pass their health checks after they enter the ``InService`` state. For more information, see `Set the health check grace period for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/health-check-grace-period.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Default: ``0`` seconds
        :param health_check_type: A comma-separated value string of one or more health check types. The valid values are ``EC2`` , ``ELB`` , and ``VPC_LATTICE`` . ``EC2`` is the default health check and cannot be disabled. For more information, see `Health checks for Auto Scaling instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/healthcheck.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Only specify ``EC2`` if you must clear a value that was previously set.
        :param instance_id: The ID of the instance used to base the launch configuration on. For more information, see `Create an Auto Scaling group using an EC2 instance <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-from-instance.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``LaunchConfigurationName`` , don't specify ``InstanceId`` .
        :param launch_configuration_name: The name of the launch configuration to use to launch instances. Required only if you don't specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``InstanceId`` .
        :param launch_template: Information used to specify the launch template and version to use to launch instances. You can alternatively associate a launch template to the Auto Scaling group by specifying a ``MixedInstancesPolicy`` . For more information about creating launch templates, see `Create a launch template for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-template.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you omit this property, you must specify ``MixedInstancesPolicy`` , ``LaunchConfigurationName`` , or ``InstanceId`` .
        :param lifecycle_hook_specification_list: One or more lifecycle hooks to add to the Auto Scaling group before instances are launched.
        :param load_balancer_names: A list of Classic Load Balancers associated with this Auto Scaling group. For Application Load Balancers, Network Load Balancers, and Gateway Load Balancers, specify the ``TargetGroupARNs`` property instead.
        :param max_instance_lifetime: The maximum amount of time, in seconds, that an instance can be in service. The default is null. If specified, the value must be either 0 or a number equal to or greater than 86,400 seconds (1 day). For more information, see `Replacing Auto Scaling instances based on maximum instance lifetime <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param metrics_collection: Enables the monitoring of group metrics of an Auto Scaling group. By default, these metrics are disabled.
        :param mixed_instances_policy: An embedded object that specifies a mixed instances policy. The policy includes properties that not only define the distribution of On-Demand Instances and Spot Instances, the maximum price to pay for Spot Instances (optional), and how the Auto Scaling group allocates instance types to fulfill On-Demand and Spot capacities, but also the properties that specify the instance configuration information—the launch template and instance types. The policy can also include a weight for each instance type and different launch templates for individual instance types. For more information, see `Auto Scaling groups with multiple instance types and purchase options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param new_instances_protected_from_scale_in: Indicates whether newly launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. For more information about preventing instances from terminating on scale in, see `Using instance scale-in protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-instance-protection.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param notification_configurations: Configures an Auto Scaling group to send notifications when specified events take place.
        :param placement_group: The name of the placement group into which to launch your instances. For more information, see `Placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . .. epigraph:: A *cluster* placement group is a logical grouping of instances within a single Availability Zone. You cannot specify multiple Availability Zones and a cluster placement group.
        :param service_linked_role_arn: The Amazon Resource Name (ARN) of the service-linked role that the Auto Scaling group uses to call other AWS service on your behalf. By default, Amazon EC2 Auto Scaling uses a service-linked role named ``AWSServiceRoleForAutoScaling`` , which it creates if it does not exist. For more information, see `Service-linked roles <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-service-linked-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param tags: One or more tags. You can tag your Auto Scaling group and propagate the tags to the Amazon EC2 instances it launches. Tags are not propagated to Amazon EBS volumes. To add tags to Amazon EBS volumes, specify the tags in a launch template but use caution. If the launch template specifies an instance tag with a key that is also specified for the Auto Scaling group, Amazon EC2 Auto Scaling overrides the value of that instance tag with the value specified by the Auto Scaling group. For more information, see `Tag Auto Scaling groups and instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-tagging.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param target_group_arns: The Amazon Resource Names (ARN) of the Elastic Load Balancing target groups to associate with the Auto Scaling group. Instances are registered as targets with the target groups. The target groups receive incoming traffic and route requests to one or more registered targets. For more information, see `Use Elastic Load Balancing to distribute traffic across the instances in your Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param termination_policies: A policy or a list of policies that are used to select the instance to terminate. These policies are executed in the order that you list them. For more information, see `Work with Amazon EC2 Auto Scaling termination policies <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-termination-policies.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid values: ``Default`` | ``AllocationStrategy`` | ``ClosestToNextInstanceHour`` | ``NewestInstance`` | ``OldestInstance`` | ``OldestLaunchConfiguration`` | ``OldestLaunchTemplate`` | ``arn:aws:lambda:region:account-id:function:my-function:my-alias``
        :param vpc_zone_identifier: A list of subnet IDs for a virtual private cloud (VPC) where instances in the Auto Scaling group can be created. If this resource specifies public subnets and is also in a VPC that is defined in the same stack template, you must use the `DependsOn attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the `VPC-gateway attachment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpc-gateway-attachment.html>`_ . .. epigraph:: When you update ``VPCZoneIdentifier`` , this retains the same Auto Scaling group and replaces old instances with new ones, according to the specified subnets. You can optionally specify how CloudFormation handles these updates by using an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ . Required to launch instances into a nondefault VPC. If you specify ``VPCZoneIdentifier`` with ``AvailabilityZones`` , the subnets that you specify for this property must reside in those Availability Zones.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            cfn_auto_scaling_group_props = autoscaling.CfnAutoScalingGroupProps(
                max_size="maxSize",
                min_size="minSize",
            
                # the properties below are optional
                auto_scaling_group_name="autoScalingGroupName",
                availability_zones=["availabilityZones"],
                capacity_rebalance=False,
                context="context",
                cooldown="cooldown",
                default_instance_warmup=123,
                desired_capacity="desiredCapacity",
                desired_capacity_type="desiredCapacityType",
                health_check_grace_period=123,
                health_check_type="healthCheckType",
                instance_id="instanceId",
                launch_configuration_name="launchConfigurationName",
                launch_template=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                    version="version",
            
                    # the properties below are optional
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName"
                ),
                lifecycle_hook_specification_list=[autoscaling.CfnAutoScalingGroup.LifecycleHookSpecificationProperty(
                    lifecycle_hook_name="lifecycleHookName",
                    lifecycle_transition="lifecycleTransition",
            
                    # the properties below are optional
                    default_result="defaultResult",
                    heartbeat_timeout=123,
                    notification_metadata="notificationMetadata",
                    notification_target_arn="notificationTargetArn",
                    role_arn="roleArn"
                )],
                load_balancer_names=["loadBalancerNames"],
                max_instance_lifetime=123,
                metrics_collection=[autoscaling.CfnAutoScalingGroup.MetricsCollectionProperty(
                    granularity="granularity",
            
                    # the properties below are optional
                    metrics=["metrics"]
                )],
                mixed_instances_policy=autoscaling.CfnAutoScalingGroup.MixedInstancesPolicyProperty(
                    launch_template=autoscaling.CfnAutoScalingGroup.LaunchTemplateProperty(
                        launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                            version="version",
            
                            # the properties below are optional
                            launch_template_id="launchTemplateId",
                            launch_template_name="launchTemplateName"
                        ),
            
                        # the properties below are optional
                        overrides=[autoscaling.CfnAutoScalingGroup.LaunchTemplateOverridesProperty(
                            instance_requirements=autoscaling.CfnAutoScalingGroup.InstanceRequirementsProperty(
                                accelerator_count=autoscaling.CfnAutoScalingGroup.AcceleratorCountRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                accelerator_manufacturers=["acceleratorManufacturers"],
                                accelerator_names=["acceleratorNames"],
                                accelerator_total_memory_mi_b=autoscaling.CfnAutoScalingGroup.AcceleratorTotalMemoryMiBRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                accelerator_types=["acceleratorTypes"],
                                allowed_instance_types=["allowedInstanceTypes"],
                                bare_metal="bareMetal",
                                baseline_ebs_bandwidth_mbps=autoscaling.CfnAutoScalingGroup.BaselineEbsBandwidthMbpsRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                burstable_performance="burstablePerformance",
                                cpu_manufacturers=["cpuManufacturers"],
                                excluded_instance_types=["excludedInstanceTypes"],
                                instance_generations=["instanceGenerations"],
                                local_storage="localStorage",
                                local_storage_types=["localStorageTypes"],
                                memory_gi_bPer_vCpu=autoscaling.CfnAutoScalingGroup.MemoryGiBPerVCpuRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                memory_mi_b=autoscaling.CfnAutoScalingGroup.MemoryMiBRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                network_bandwidth_gbps=autoscaling.CfnAutoScalingGroup.NetworkBandwidthGbpsRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                network_interface_count=autoscaling.CfnAutoScalingGroup.NetworkInterfaceCountRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                on_demand_max_price_percentage_over_lowest_price=123,
                                require_hibernate_support=False,
                                spot_max_price_percentage_over_lowest_price=123,
                                total_local_storage_gb=autoscaling.CfnAutoScalingGroup.TotalLocalStorageGBRequestProperty(
                                    max=123,
                                    min=123
                                ),
                                v_cpu_count=autoscaling.CfnAutoScalingGroup.VCpuCountRequestProperty(
                                    max=123,
                                    min=123
                                )
                            ),
                            instance_type="instanceType",
                            launch_template_specification=autoscaling.CfnAutoScalingGroup.LaunchTemplateSpecificationProperty(
                                version="version",
            
                                # the properties below are optional
                                launch_template_id="launchTemplateId",
                                launch_template_name="launchTemplateName"
                            ),
                            weighted_capacity="weightedCapacity"
                        )]
                    ),
            
                    # the properties below are optional
                    instances_distribution=autoscaling.CfnAutoScalingGroup.InstancesDistributionProperty(
                        on_demand_allocation_strategy="onDemandAllocationStrategy",
                        on_demand_base_capacity=123,
                        on_demand_percentage_above_base_capacity=123,
                        spot_allocation_strategy="spotAllocationStrategy",
                        spot_instance_pools=123,
                        spot_max_price="spotMaxPrice"
                    )
                ),
                new_instances_protected_from_scale_in=False,
                notification_configurations=[autoscaling.CfnAutoScalingGroup.NotificationConfigurationProperty(
                    topic_arn="topicArn",
            
                    # the properties below are optional
                    notification_types=["notificationTypes"]
                )],
                placement_group="placementGroup",
                service_linked_role_arn="serviceLinkedRoleArn",
                tags=[autoscaling.CfnAutoScalingGroup.TagPropertyProperty(
                    key="key",
                    propagate_at_launch=False,
                    value="value"
                )],
                target_group_arns=["targetGroupArns"],
                termination_policies=["terminationPolicies"],
                vpc_zone_identifier=["vpcZoneIdentifier"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63de692030eb0bc729681a791501e2edd744cab3141e27cc9c9c8def2c83197a)
            check_type(argname="argument max_size", value=max_size, expected_type=type_hints["max_size"])
            check_type(argname="argument min_size", value=min_size, expected_type=type_hints["min_size"])
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument availability_zones", value=availability_zones, expected_type=type_hints["availability_zones"])
            check_type(argname="argument capacity_rebalance", value=capacity_rebalance, expected_type=type_hints["capacity_rebalance"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument default_instance_warmup", value=default_instance_warmup, expected_type=type_hints["default_instance_warmup"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument desired_capacity_type", value=desired_capacity_type, expected_type=type_hints["desired_capacity_type"])
            check_type(argname="argument health_check_grace_period", value=health_check_grace_period, expected_type=type_hints["health_check_grace_period"])
            check_type(argname="argument health_check_type", value=health_check_type, expected_type=type_hints["health_check_type"])
            check_type(argname="argument instance_id", value=instance_id, expected_type=type_hints["instance_id"])
            check_type(argname="argument launch_configuration_name", value=launch_configuration_name, expected_type=type_hints["launch_configuration_name"])
            check_type(argname="argument launch_template", value=launch_template, expected_type=type_hints["launch_template"])
            check_type(argname="argument lifecycle_hook_specification_list", value=lifecycle_hook_specification_list, expected_type=type_hints["lifecycle_hook_specification_list"])
            check_type(argname="argument load_balancer_names", value=load_balancer_names, expected_type=type_hints["load_balancer_names"])
            check_type(argname="argument max_instance_lifetime", value=max_instance_lifetime, expected_type=type_hints["max_instance_lifetime"])
            check_type(argname="argument metrics_collection", value=metrics_collection, expected_type=type_hints["metrics_collection"])
            check_type(argname="argument mixed_instances_policy", value=mixed_instances_policy, expected_type=type_hints["mixed_instances_policy"])
            check_type(argname="argument new_instances_protected_from_scale_in", value=new_instances_protected_from_scale_in, expected_type=type_hints["new_instances_protected_from_scale_in"])
            check_type(argname="argument notification_configurations", value=notification_configurations, expected_type=type_hints["notification_configurations"])
            check_type(argname="argument placement_group", value=placement_group, expected_type=type_hints["placement_group"])
            check_type(argname="argument service_linked_role_arn", value=service_linked_role_arn, expected_type=type_hints["service_linked_role_arn"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_group_arns", value=target_group_arns, expected_type=type_hints["target_group_arns"])
            check_type(argname="argument termination_policies", value=termination_policies, expected_type=type_hints["termination_policies"])
            check_type(argname="argument vpc_zone_identifier", value=vpc_zone_identifier, expected_type=type_hints["vpc_zone_identifier"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "max_size": max_size,
            "min_size": min_size,
        }
        if auto_scaling_group_name is not None:
            self._values["auto_scaling_group_name"] = auto_scaling_group_name
        if availability_zones is not None:
            self._values["availability_zones"] = availability_zones
        if capacity_rebalance is not None:
            self._values["capacity_rebalance"] = capacity_rebalance
        if context is not None:
            self._values["context"] = context
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if default_instance_warmup is not None:
            self._values["default_instance_warmup"] = default_instance_warmup
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if desired_capacity_type is not None:
            self._values["desired_capacity_type"] = desired_capacity_type
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if health_check_type is not None:
            self._values["health_check_type"] = health_check_type
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if launch_configuration_name is not None:
            self._values["launch_configuration_name"] = launch_configuration_name
        if launch_template is not None:
            self._values["launch_template"] = launch_template
        if lifecycle_hook_specification_list is not None:
            self._values["lifecycle_hook_specification_list"] = lifecycle_hook_specification_list
        if load_balancer_names is not None:
            self._values["load_balancer_names"] = load_balancer_names
        if max_instance_lifetime is not None:
            self._values["max_instance_lifetime"] = max_instance_lifetime
        if metrics_collection is not None:
            self._values["metrics_collection"] = metrics_collection
        if mixed_instances_policy is not None:
            self._values["mixed_instances_policy"] = mixed_instances_policy
        if new_instances_protected_from_scale_in is not None:
            self._values["new_instances_protected_from_scale_in"] = new_instances_protected_from_scale_in
        if notification_configurations is not None:
            self._values["notification_configurations"] = notification_configurations
        if placement_group is not None:
            self._values["placement_group"] = placement_group
        if service_linked_role_arn is not None:
            self._values["service_linked_role_arn"] = service_linked_role_arn
        if tags is not None:
            self._values["tags"] = tags
        if target_group_arns is not None:
            self._values["target_group_arns"] = target_group_arns
        if termination_policies is not None:
            self._values["termination_policies"] = termination_policies
        if vpc_zone_identifier is not None:
            self._values["vpc_zone_identifier"] = vpc_zone_identifier

    @builtins.property
    def max_size(self) -> builtins.str:
        '''The maximum size of the group.

        .. epigraph::

           With a mixed instances policy that uses instance weighting, Amazon EC2 Auto Scaling may need to go above ``MaxSize`` to meet your capacity requirements. In this event, Amazon EC2 Auto Scaling will never go above ``MaxSize`` by more than your largest instance weight (weights that define how many units each instance contributes to the desired capacity of the group).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-maxsize
        '''
        result = self._values.get("max_size")
        assert result is not None, "Required property 'max_size' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def min_size(self) -> builtins.str:
        '''The minimum size of the group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-minsize
        '''
        result = self._values.get("min_size")
        assert result is not None, "Required property 'min_size' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Auto Scaling group. This name must be unique per Region per account.

        The name can contain any ASCII character 33 to 126 including most punctuation characters, digits, and upper and lowercased letters.
        .. epigraph::

           You cannot use a colon (:) in the name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-autoscaling-autoscalinggroup-autoscalinggroupname
        '''
        result = self._values.get("auto_scaling_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Availability Zones where instances in the Auto Scaling group can be created.

        Used for launching into the default VPC subnet in each Availability Zone when not using the ``VPCZoneIdentifier`` property, or for attaching a network interface when an existing network interface ID is specified in a launch template.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-availabilityzones
        '''
        result = self._values.get("availability_zones")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def capacity_rebalance(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether Capacity Rebalancing is enabled.

        Otherwise, Capacity Rebalancing is disabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. For more information, see `Use Capacity Rebalancing to handle Amazon EC2 Spot Interruptions <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-capacity-rebalancing.html>`_ in the in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-capacityrebalance
        '''
        result = self._values.get("capacity_rebalance")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-context
        '''
        result = self._values.get("context")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[builtins.str]:
        '''*Only needed if you use simple scaling policies.*.

        The amount of time, in seconds, between one scaling activity ending and another one starting due to simple scaling policies. For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Default: ``300`` seconds

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-cooldown
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_instance_warmup(self) -> typing.Optional[jsii.Number]:
        '''The amount of time, in seconds, until a new instance is considered to have finished initializing and resource consumption to become stable after it enters the ``InService`` state.

        During an instance refresh, Amazon EC2 Auto Scaling waits for the warm-up period after it replaces an instance before it moves on to replacing the next instance. Amazon EC2 Auto Scaling also waits for the warm-up period before aggregating the metrics for new instances with existing instances in the Amazon CloudWatch metrics that are used for scaling, resulting in more reliable usage data. For more information, see `Set the default instance warmup for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        .. epigraph::

           To manage various warm-up settings at the group level, we recommend that you set the default instance warmup, *even if it is set to 0 seconds* . To remove a value that you previously set, include the property but specify ``-1`` for the value. However, we strongly recommend keeping the default instance warmup enabled by specifying a value of ``0`` or other nominal value.

        Default: None

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-defaultinstancewarmup
        '''
        result = self._values.get("default_instance_warmup")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[builtins.str]:
        '''The desired capacity is the initial capacity of the Auto Scaling group at the time of its creation and the capacity it attempts to maintain.

        It can scale beyond this capacity if you configure automatic scaling.

        The number must be greater than or equal to the minimum size of the group and less than or equal to the maximum size of the group. If you do not specify a desired capacity when creating the stack, the default is the minimum size of the group.

        CloudFormation marks the Auto Scaling group as successful (by setting its status to CREATE_COMPLETE) when the desired capacity is reached. However, if a maximum Spot price is set in the launch template or launch configuration that you specified, then desired capacity is not used as a criteria for success. Whether your request is fulfilled depends on Spot Instance capacity and your maximum price.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def desired_capacity_type(self) -> typing.Optional[builtins.str]:
        '''The unit of measurement for the value specified for desired capacity.

        Amazon EC2 Auto Scaling supports ``DesiredCapacityType`` for attribute-based instance type selection only. For more information, see `Creating an Auto Scaling group using attribute-based instance type selection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-instance-type-requirements.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        By default, Amazon EC2 Auto Scaling specifies ``units`` , which translates into number of instances.

        Valid values: ``units`` | ``vcpu`` | ``memory-mib``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacitytype
        '''
        result = self._values.get("desired_capacity_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[jsii.Number]:
        '''The amount of time, in seconds, that Amazon EC2 Auto Scaling waits before checking the health status of an EC2 instance that has come into service and marking it unhealthy due to a failed health check.

        This is useful if your instances do not immediately pass their health checks after they enter the ``InService`` state. For more information, see `Set the health check grace period for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/health-check-grace-period.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Default: ``0`` seconds

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-healthcheckgraceperiod
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def health_check_type(self) -> typing.Optional[builtins.str]:
        '''A comma-separated value string of one or more health check types.

        The valid values are ``EC2`` , ``ELB`` , and ``VPC_LATTICE`` . ``EC2`` is the default health check and cannot be disabled. For more information, see `Health checks for Auto Scaling instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/healthcheck.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Only specify ``EC2`` if you must clear a value that was previously set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-healthchecktype
        '''
        result = self._values.get("health_check_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance used to base the launch configuration on.

        For more information, see `Create an Auto Scaling group using an EC2 instance <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-asg-from-instance.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``LaunchConfigurationName`` , don't specify ``InstanceId`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_configuration_name(self) -> typing.Optional[builtins.str]:
        '''The name of the launch configuration to use to launch instances.

        Required only if you don't specify ``LaunchTemplate`` , ``MixedInstancesPolicy`` , or ``InstanceId`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-launchconfigurationname
        '''
        result = self._values.get("launch_configuration_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.LaunchTemplateSpecificationProperty]]:
        '''Information used to specify the launch template and version to use to launch instances.

        You can alternatively associate a launch template to the Auto Scaling group by specifying a ``MixedInstancesPolicy`` . For more information about creating launch templates, see `Create a launch template for an Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-template.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you omit this property, you must specify ``MixedInstancesPolicy`` , ``LaunchConfigurationName`` , or ``InstanceId`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-launchtemplate
        '''
        result = self._values.get("launch_template")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.LaunchTemplateSpecificationProperty]], result)

    @builtins.property
    def lifecycle_hook_specification_list(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.LifecycleHookSpecificationProperty]]]]:
        '''One or more lifecycle hooks to add to the Auto Scaling group before instances are launched.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-autoscaling-autoscalinggroup-lifecyclehookspecificationlist
        '''
        result = self._values.get("lifecycle_hook_specification_list")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.LifecycleHookSpecificationProperty]]]], result)

    @builtins.property
    def load_balancer_names(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of Classic Load Balancers associated with this Auto Scaling group.

        For Application Load Balancers, Network Load Balancers, and Gateway Load Balancers, specify the ``TargetGroupARNs`` property instead.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-loadbalancernames
        '''
        result = self._values.get("load_balancer_names")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def max_instance_lifetime(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of time, in seconds, that an instance can be in service.

        The default is null. If specified, the value must be either 0 or a number equal to or greater than 86,400 seconds (1 day). For more information, see `Replacing Auto Scaling instances based on maximum instance lifetime <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-maxinstancelifetime
        '''
        result = self._values.get("max_instance_lifetime")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def metrics_collection(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.MetricsCollectionProperty]]]]:
        '''Enables the monitoring of group metrics of an Auto Scaling group.

        By default, these metrics are disabled.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-metricscollection
        '''
        result = self._values.get("metrics_collection")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.MetricsCollectionProperty]]]], result)

    @builtins.property
    def mixed_instances_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.MixedInstancesPolicyProperty]]:
        '''An embedded object that specifies a mixed instances policy.

        The policy includes properties that not only define the distribution of On-Demand Instances and Spot Instances, the maximum price to pay for Spot Instances (optional), and how the Auto Scaling group allocates instance types to fulfill On-Demand and Spot capacities, but also the properties that specify the instance configuration information—the launch template and instance types. The policy can also include a weight for each instance type and different launch templates for individual instance types.

        For more information, see `Auto Scaling groups with multiple instance types and purchase options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-mixed-instances-groups.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-mixedinstancespolicy
        '''
        result = self._values.get("mixed_instances_policy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.MixedInstancesPolicyProperty]], result)

    @builtins.property
    def new_instances_protected_from_scale_in(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether newly launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        For more information about preventing instances from terminating on scale in, see `Using instance scale-in protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-instance-protection.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-newinstancesprotectedfromscalein
        '''
        result = self._values.get("new_instances_protected_from_scale_in")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def notification_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.NotificationConfigurationProperty]]]]:
        '''Configures an Auto Scaling group to send notifications when specified events take place.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        '''
        result = self._values.get("notification_configurations")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAutoScalingGroup.NotificationConfigurationProperty]]]], result)

    @builtins.property
    def placement_group(self) -> typing.Optional[builtins.str]:
        '''The name of the placement group into which to launch your instances.

        For more information, see `Placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        .. epigraph::

           A *cluster* placement group is a logical grouping of instances within a single Availability Zone. You cannot specify multiple Availability Zones and a cluster placement group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-placementgroup
        '''
        result = self._values.get("placement_group")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_linked_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the service-linked role that the Auto Scaling group uses to call other AWS service on your behalf.

        By default, Amazon EC2 Auto Scaling uses a service-linked role named ``AWSServiceRoleForAutoScaling`` , which it creates if it does not exist. For more information, see `Service-linked roles <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-service-linked-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-autoscaling-autoscalinggroup-servicelinkedrolearn
        '''
        result = self._values.get("service_linked_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(
        self,
    ) -> typing.Optional[typing.List[CfnAutoScalingGroup.TagPropertyProperty]]:
        '''One or more tags.

        You can tag your Auto Scaling group and propagate the tags to the Amazon EC2 instances it launches. Tags are not propagated to Amazon EBS volumes. To add tags to Amazon EBS volumes, specify the tags in a launch template but use caution. If the launch template specifies an instance tag with a key that is also specified for the Auto Scaling group, Amazon EC2 Auto Scaling overrides the value of that instance tag with the value specified by the Auto Scaling group. For more information, see `Tag Auto Scaling groups and instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-tagging.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[CfnAutoScalingGroup.TagPropertyProperty]], result)

    @builtins.property
    def target_group_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon Resource Names (ARN) of the Elastic Load Balancing target groups to associate with the Auto Scaling group.

        Instances are registered as targets with the target groups. The target groups receive incoming traffic and route requests to one or more registered targets. For more information, see `Use Elastic Load Balancing to distribute traffic across the instances in your Auto Scaling group <https://docs.aws.amazon.com/autoscaling/ec2/userguide/autoscaling-load-balancer.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-targetgrouparns
        '''
        result = self._values.get("target_group_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def termination_policies(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A policy or a list of policies that are used to select the instance to terminate.

        These policies are executed in the order that you list them. For more information, see `Work with Amazon EC2 Auto Scaling termination policies <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-termination-policies.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Valid values: ``Default`` | ``AllocationStrategy`` | ``ClosestToNextInstanceHour`` | ``NewestInstance`` | ``OldestInstance`` | ``OldestLaunchConfiguration`` | ``OldestLaunchTemplate`` | ``arn:aws:lambda:region:account-id:function:my-function:my-alias``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-termpolicy
        '''
        result = self._values.get("termination_policies")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def vpc_zone_identifier(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of subnet IDs for a virtual private cloud (VPC) where instances in the Auto Scaling group can be created.

        If this resource specifies public subnets and is also in a VPC that is defined in the same stack template, you must use the `DependsOn attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the `VPC-gateway attachment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpc-gateway-attachment.html>`_ .
        .. epigraph::

           When you update ``VPCZoneIdentifier`` , this retains the same Auto Scaling group and replaces old instances with new ones, according to the specified subnets. You can optionally specify how CloudFormation handles these updates by using an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ .

        Required to launch instances into a nondefault VPC. If you specify ``VPCZoneIdentifier`` with ``AvailabilityZones`` , the subnets that you specify for this property must reside in those Availability Zones.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-vpczoneidentifier
        '''
        result = self._values.get("vpc_zone_identifier")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAutoScalingGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnLaunchConfiguration(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnLaunchConfiguration",
):
    '''A CloudFormation ``AWS::AutoScaling::LaunchConfiguration``.

    The ``AWS::AutoScaling::LaunchConfiguration`` resource specifies the launch configuration that can be used by an Auto Scaling group to configure Amazon EC2 instances.

    When you update the launch configuration for an Auto Scaling group, CloudFormation deletes that resource and creates a new launch configuration with the updated properties and a new name. Existing instances are not affected. To update existing instances when you update the ``AWS::AutoScaling::LaunchConfiguration`` resource, you can specify an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ for the group. You can find sample update policies for rolling updates in `Auto scaling template snippets <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/quickref-autoscaling.html>`_ .
    .. epigraph::

       Amazon EC2 Auto Scaling configures instances launched as part of an Auto Scaling group using either a `launch template <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html>`_ or a launch configuration. We strongly recommend that you do not use launch configurations. They do not provide full functionality for Amazon EC2 Auto Scaling or Amazon EC2. For more information, see `Launch configurations <https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-configurations.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

    :cloudformationResource: AWS::AutoScaling::LaunchConfiguration
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        cfn_launch_configuration = autoscaling.CfnLaunchConfiguration(self, "MyCfnLaunchConfiguration",
            image_id="imageId",
            instance_type="instanceType",
        
            # the properties below are optional
            associate_public_ip_address=False,
            block_device_mappings=[autoscaling.CfnLaunchConfiguration.BlockDeviceMappingProperty(
                device_name="deviceName",
        
                # the properties below are optional
                ebs=autoscaling.CfnLaunchConfiguration.BlockDeviceProperty(
                    delete_on_termination=False,
                    encrypted=False,
                    iops=123,
                    snapshot_id="snapshotId",
                    throughput=123,
                    volume_size=123,
                    volume_type="volumeType"
                ),
                no_device=False,
                virtual_name="virtualName"
            )],
            classic_link_vpc_id="classicLinkVpcId",
            classic_link_vpc_security_groups=["classicLinkVpcSecurityGroups"],
            ebs_optimized=False,
            iam_instance_profile="iamInstanceProfile",
            instance_id="instanceId",
            instance_monitoring=False,
            kernel_id="kernelId",
            key_name="keyName",
            launch_configuration_name="launchConfigurationName",
            metadata_options=autoscaling.CfnLaunchConfiguration.MetadataOptionsProperty(
                http_endpoint="httpEndpoint",
                http_put_response_hop_limit=123,
                http_tokens="httpTokens"
            ),
            placement_tenancy="placementTenancy",
            ram_disk_id="ramDiskId",
            security_groups=["securityGroups"],
            spot_price="spotPrice",
            user_data="userData"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        image_id: builtins.str,
        instance_type: builtins.str,
        associate_public_ip_address: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        block_device_mappings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLaunchConfiguration.BlockDeviceMappingProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        classic_link_vpc_id: typing.Optional[builtins.str] = None,
        classic_link_vpc_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        iam_instance_profile: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        instance_monitoring: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        kernel_id: typing.Optional[builtins.str] = None,
        key_name: typing.Optional[builtins.str] = None,
        launch_configuration_name: typing.Optional[builtins.str] = None,
        metadata_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLaunchConfiguration.MetadataOptionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        placement_tenancy: typing.Optional[builtins.str] = None,
        ram_disk_id: typing.Optional[builtins.str] = None,
        security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        spot_price: typing.Optional[builtins.str] = None,
        user_data: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AutoScaling::LaunchConfiguration``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param image_id: The ID of the Amazon Machine Image (AMI) that was assigned during registration. For more information, see `Finding a Linux AMI <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/finding-an-ami.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . If you specify ``InstanceId`` , an ``ImageId`` is not required.
        :param instance_type: Specifies the instance type of the EC2 instance. For information about available instance types, see `Available instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#AvailableInstanceTypes>`_ in the *Amazon EC2 User Guide for Linux Instances* . If you specify ``InstanceId`` , an ``InstanceType`` is not required.
        :param associate_public_ip_address: Specifies whether to assign a public IPv4 address to the group's instances. If the instance is launched into a default subnet, the default is to assign a public IPv4 address, unless you disabled the option to assign a public IPv4 address on the subnet. If the instance is launched into a nondefault subnet, the default is not to assign a public IPv4 address, unless you enabled the option to assign a public IPv4 address on the subnet. If you specify ``true`` , each instance in the Auto Scaling group receives a unique public IPv4 address. For more information, see `Launching Auto Scaling instances in a VPC <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-in-vpc.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you specify this property, you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group.
        :param block_device_mappings: The block device mapping entries that define the block devices to attach to the instances at launch. By default, the block devices specified in the block device mapping for the AMI are used. For more information, see `Block device mappings <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param classic_link_vpc_id: Available for backward compatibility.
        :param classic_link_vpc_security_groups: Available for backward compatibility.
        :param ebs_optimized: Specifies whether the launch configuration is optimized for EBS I/O ( ``true`` ) or not ( ``false`` ). The optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization is not available with all instance types. Additional fees are incurred when you enable EBS optimization for an instance type that is not EBS-optimized by default. For more information, see `Amazon EBS-optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . The default value is ``false`` .
        :param iam_instance_profile: The name or the Amazon Resource Name (ARN) of the instance profile associated with the IAM role for the instance. The instance profile contains the IAM role. For more information, see `IAM role for applications that run on Amazon EC2 instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/us-iam-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param instance_id: The ID of the Amazon EC2 instance to use to create the launch configuration. When you use an instance to create a launch configuration, all properties are derived from the instance with the exception of ``BlockDeviceMapping`` and ``AssociatePublicIpAddress`` . You can override any properties from the instance by specifying them in the launch configuration.
        :param instance_monitoring: Controls whether instances in this group are launched with detailed ( ``true`` ) or basic ( ``false`` ) monitoring. The default value is ``true`` (enabled). .. epigraph:: When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. For more information, see `Configure Monitoring for Auto Scaling Instances <https://docs.aws.amazon.com/autoscaling/latest/userguide/enable-as-instance-metrics.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param kernel_id: The ID of the kernel associated with the AMI. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param key_name: The name of the key pair. For more information, see `Amazon EC2 key pairs and Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param launch_configuration_name: The name of the launch configuration. This name must be unique per Region per account.
        :param metadata_options: The metadata options for the instances. For more information, see `Configuring the Instance Metadata Options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-config.html#launch-configurations-imds>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param placement_tenancy: The tenancy of the instance, either ``default`` or ``dedicated`` . An instance with ``dedicated`` tenancy runs on isolated, single-tenant hardware and can only be launched into a VPC. To launch dedicated instances into a shared tenancy VPC (a VPC with the instance placement tenancy attribute set to ``default`` ), you must set the value of this property to ``dedicated`` . For more information, see `Configuring instance tenancy with Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/auto-scaling-dedicated-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you specify ``PlacementTenancy`` , you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group. Valid values: ``default`` | ``dedicated``
        :param ram_disk_id: The ID of the RAM disk to select. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param security_groups: A list that contains the security groups to assign to the instances in the Auto Scaling group. The list can contain both the IDs of existing security groups and references to `SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template. For more information, see `Control traffic to resources using security groups <https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_SecurityGroups.html>`_ in the *Amazon Virtual Private Cloud User Guide* .
        :param spot_price: The maximum hourly price to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot price. For more information, see `Request Spot Instances for fault-tolerant and flexible applications <https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-template-spot-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid Range: Minimum value of 0.001 .. epigraph:: When you change your maximum price by creating a new launch configuration, running instances will continue to run as long as the maximum price for those running instances is higher than the current Spot price.
        :param user_data: The Base64-encoded user data to make available to the launched EC2 instances. For more information, see `Instance metadata and user data <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1b7dfa85ece494c51470fc8949ac83f44a43e14933517668759ff3c3d48df65)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLaunchConfigurationProps(
            image_id=image_id,
            instance_type=instance_type,
            associate_public_ip_address=associate_public_ip_address,
            block_device_mappings=block_device_mappings,
            classic_link_vpc_id=classic_link_vpc_id,
            classic_link_vpc_security_groups=classic_link_vpc_security_groups,
            ebs_optimized=ebs_optimized,
            iam_instance_profile=iam_instance_profile,
            instance_id=instance_id,
            instance_monitoring=instance_monitoring,
            kernel_id=kernel_id,
            key_name=key_name,
            launch_configuration_name=launch_configuration_name,
            metadata_options=metadata_options,
            placement_tenancy=placement_tenancy,
            ram_disk_id=ram_disk_id,
            security_groups=security_groups,
            spot_price=spot_price,
            user_data=user_data,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6d1dd362dede4d9e5d99e0867ba1c4a0a20293d42e613770137063dfbe8db7c)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52e42bb737deded4bdd38c8ad43308837b3d11658c0f1aa0e505ea73fa3997c1)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="imageId")
    def image_id(self) -> builtins.str:
        '''The ID of the Amazon Machine Image (AMI) that was assigned during registration.

        For more information, see `Finding a Linux AMI <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/finding-an-ami.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        If you specify ``InstanceId`` , an ``ImageId`` is not required.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-imageid
        '''
        return typing.cast(builtins.str, jsii.get(self, "imageId"))

    @image_id.setter
    def image_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a590f1a11b57ee0507267c1184c4ab1fd25d4c7a2add68efd8567cf12318b59)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "imageId", value)

    @builtins.property
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''Specifies the instance type of the EC2 instance.

        For information about available instance types, see `Available instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#AvailableInstanceTypes>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        If you specify ``InstanceId`` , an ``InstanceType`` is not required.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-instancetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ebc70aaf3294dda2fcb78ecb12433a976f0a814e8b594582e5a4987925a60e0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceType", value)

    @builtins.property
    @jsii.member(jsii_name="associatePublicIpAddress")
    def associate_public_ip_address(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to assign a public IPv4 address to the group's instances.

        If the instance is launched into a default subnet, the default is to assign a public IPv4 address, unless you disabled the option to assign a public IPv4 address on the subnet. If the instance is launched into a nondefault subnet, the default is not to assign a public IPv4 address, unless you enabled the option to assign a public IPv4 address on the subnet.

        If you specify ``true`` , each instance in the Auto Scaling group receives a unique public IPv4 address. For more information, see `Launching Auto Scaling instances in a VPC <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-in-vpc.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you specify this property, you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-associatepublicipaddress
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "associatePublicIpAddress"))

    @associate_public_ip_address.setter
    def associate_public_ip_address(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__674455b1254ecaf087af554d10d231002edb13b7fc7f1388a3fc37f4a56965e8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "associatePublicIpAddress", value)

    @builtins.property
    @jsii.member(jsii_name="blockDeviceMappings")
    def block_device_mappings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.BlockDeviceMappingProperty"]]]]:
        '''The block device mapping entries that define the block devices to attach to the instances at launch.

        By default, the block devices specified in the block device mapping for the AMI are used. For more information, see `Block device mappings <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-blockdevicemappings
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.BlockDeviceMappingProperty"]]]], jsii.get(self, "blockDeviceMappings"))

    @block_device_mappings.setter
    def block_device_mappings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.BlockDeviceMappingProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bdc5f0729f20281d7b5dff9761abbab3b051c02184dbf4917feabace61ee2667)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "blockDeviceMappings", value)

    @builtins.property
    @jsii.member(jsii_name="classicLinkVpcId")
    def classic_link_vpc_id(self) -> typing.Optional[builtins.str]:
        '''Available for backward compatibility.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-classiclinkvpcid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "classicLinkVpcId"))

    @classic_link_vpc_id.setter
    def classic_link_vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd69ee70914f6d11d6f3b9a421b359ab003aa76f5013ffc7ab4640574f50e449)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "classicLinkVpcId", value)

    @builtins.property
    @jsii.member(jsii_name="classicLinkVpcSecurityGroups")
    def classic_link_vpc_security_groups(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''Available for backward compatibility.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-classiclinkvpcsecuritygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "classicLinkVpcSecurityGroups"))

    @classic_link_vpc_security_groups.setter
    def classic_link_vpc_security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8fa4a3aa74644d10d77a099a33b3f970c24ddc0fd7ff782d932f948a20283fc0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "classicLinkVpcSecurityGroups", value)

    @builtins.property
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the launch configuration is optimized for EBS I/O ( ``true`` ) or not ( ``false`` ).

        The optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization is not available with all instance types. Additional fees are incurred when you enable EBS optimization for an instance type that is not EBS-optimized by default. For more information, see `Amazon EBS-optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        The default value is ``false`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-ebsoptimized
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "ebsOptimized"))

    @ebs_optimized.setter
    def ebs_optimized(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f092d7a962971bba208cb655827dc32c385bf989c109ae62f8e956ab07a0c8ce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ebsOptimized", value)

    @builtins.property
    @jsii.member(jsii_name="iamInstanceProfile")
    def iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The name or the Amazon Resource Name (ARN) of the instance profile associated with the IAM role for the instance.

        The instance profile contains the IAM role. For more information, see `IAM role for applications that run on Amazon EC2 instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/us-iam-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-iaminstanceprofile
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "iamInstanceProfile"))

    @iam_instance_profile.setter
    def iam_instance_profile(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__73ae36075c104a56882330fa48b634107a9dcf6b069c6c61b3625de613c66831)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "iamInstanceProfile", value)

    @builtins.property
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the Amazon EC2 instance to use to create the launch configuration.

        When you use an instance to create a launch configuration, all properties are derived from the instance with the exception of ``BlockDeviceMapping`` and ``AssociatePublicIpAddress`` . You can override any properties from the instance by specifying them in the launch configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__18e4df9fb2d7ef59c38687c0726cc07c2e349b85efc5335bcf75ca13e4ea35eb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceId", value)

    @builtins.property
    @jsii.member(jsii_name="instanceMonitoring")
    def instance_monitoring(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Controls whether instances in this group are launched with detailed ( ``true`` ) or basic ( ``false`` ) monitoring.

        The default value is ``true`` (enabled).
        .. epigraph::

           When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. For more information, see `Configure Monitoring for Auto Scaling Instances <https://docs.aws.amazon.com/autoscaling/latest/userguide/enable-as-instance-metrics.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-instancemonitoring
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "instanceMonitoring"))

    @instance_monitoring.setter
    def instance_monitoring(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5ba316c82fe73e99e7f86ec88228cea720950bd4d3d42a8780880ce964b5593)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceMonitoring", value)

    @builtins.property
    @jsii.member(jsii_name="kernelId")
    def kernel_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the kernel associated with the AMI.

        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-kernelid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kernelId"))

    @kernel_id.setter
    def kernel_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a405be9848883655a34f3cb88e0b533539d91ced54ead07763579eacd873a6ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kernelId", value)

    @builtins.property
    @jsii.member(jsii_name="keyName")
    def key_name(self) -> typing.Optional[builtins.str]:
        '''The name of the key pair.

        For more information, see `Amazon EC2 key pairs and Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-keyname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "keyName"))

    @key_name.setter
    def key_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e32ad0180db900e68841993347f7d387928a6646c96dca6d80bb9717cbfd1d1d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "keyName", value)

    @builtins.property
    @jsii.member(jsii_name="launchConfigurationName")
    def launch_configuration_name(self) -> typing.Optional[builtins.str]:
        '''The name of the launch configuration.

        This name must be unique per Region per account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-launchconfigurationname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchConfigurationName"))

    @launch_configuration_name.setter
    def launch_configuration_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f8bba7d570f8c476d61963ecd52300e6f67e1f6d8cac7a67b19fd3c7ad972fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "launchConfigurationName", value)

    @builtins.property
    @jsii.member(jsii_name="metadataOptions")
    def metadata_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.MetadataOptionsProperty"]]:
        '''The metadata options for the instances.

        For more information, see `Configuring the Instance Metadata Options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-config.html#launch-configurations-imds>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-metadataoptions
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.MetadataOptionsProperty"]], jsii.get(self, "metadataOptions"))

    @metadata_options.setter
    def metadata_options(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.MetadataOptionsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f7c6bb8da29baa11401340e2a6d03aef0bf5f521da31f4ea3631d00c07e61ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "metadataOptions", value)

    @builtins.property
    @jsii.member(jsii_name="placementTenancy")
    def placement_tenancy(self) -> typing.Optional[builtins.str]:
        '''The tenancy of the instance, either ``default`` or ``dedicated`` .

        An instance with ``dedicated`` tenancy runs on isolated, single-tenant hardware and can only be launched into a VPC. To launch dedicated instances into a shared tenancy VPC (a VPC with the instance placement tenancy attribute set to ``default`` ), you must set the value of this property to ``dedicated`` . For more information, see `Configuring instance tenancy with Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/auto-scaling-dedicated-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you specify ``PlacementTenancy`` , you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group.

        Valid values: ``default`` | ``dedicated``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-placementtenancy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "placementTenancy"))

    @placement_tenancy.setter
    def placement_tenancy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5fec1cfbab9fa9ea909ebe723dace6e2c257499f88750f54ba6a0ef402bb905b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "placementTenancy", value)

    @builtins.property
    @jsii.member(jsii_name="ramDiskId")
    def ram_disk_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the RAM disk to select.

        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-ramdiskid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ramDiskId"))

    @ram_disk_id.setter
    def ram_disk_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c8b6c0b0981f30e6fda22d72efe839a3ede24080b79389a55877430b1a4b8284)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ramDiskId", value)

    @builtins.property
    @jsii.member(jsii_name="securityGroups")
    def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list that contains the security groups to assign to the instances in the Auto Scaling group.

        The list can contain both the IDs of existing security groups and references to `SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template.

        For more information, see `Control traffic to resources using security groups <https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_SecurityGroups.html>`_ in the *Amazon Virtual Private Cloud User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-securitygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroups"))

    @security_groups.setter
    def security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__290470833418a73ff89f31e864c6abc6e6a0436b8071c33102c6223fcbdc204a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "securityGroups", value)

    @builtins.property
    @jsii.member(jsii_name="spotPrice")
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''The maximum hourly price to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are launched when the price you specify exceeds the current Spot price. For more information, see `Request Spot Instances for fault-tolerant and flexible applications <https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-template-spot-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Valid Range: Minimum value of 0.001
        .. epigraph::

           When you change your maximum price by creating a new launch configuration, running instances will continue to run as long as the maximum price for those running instances is higher than the current Spot price.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-spotprice
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "spotPrice"))

    @spot_price.setter
    def spot_price(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e9651e1cd8f7668bbbdac019ddd62ceacf7e55a6bc5288ab7d2d8a49fd2f395)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spotPrice", value)

    @builtins.property
    @jsii.member(jsii_name="userData")
    def user_data(self) -> typing.Optional[builtins.str]:
        '''The Base64-encoded user data to make available to the launched EC2 instances.

        For more information, see `Instance metadata and user data <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-userdata
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "userData"))

    @user_data.setter
    def user_data(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7568c3d28d0b73eade97e8b9e92d322f8d322e9f9c73d44b4fe5e2ddfebd348c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userData", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnLaunchConfiguration.BlockDeviceMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_name": "deviceName",
            "ebs": "ebs",
            "no_device": "noDevice",
            "virtual_name": "virtualName",
        },
    )
    class BlockDeviceMappingProperty:
        def __init__(
            self,
            *,
            device_name: builtins.str,
            ebs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLaunchConfiguration.BlockDeviceProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            no_device: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            virtual_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``BlockDeviceMapping`` specifies a block device mapping for the ``BlockDeviceMappings`` property of the `AWS::AutoScaling::LaunchConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-launchconfig.html>`_ resource.

            Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched.

            For more information, see `Example block device mapping <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html#block-device-mapping-ex>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            :param device_name: The device name assigned to the volume (for example, ``/dev/sdh`` or ``xvdh`` ). For more information, see `Device naming on Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . .. epigraph:: To define a block device mapping, set the device name and exactly one of the following properties: ``Ebs`` , ``NoDevice`` , or ``VirtualName`` .
            :param ebs: Information to attach an EBS volume to an instance at launch.
            :param no_device: Setting this value to ``true`` prevents a volume that is included in the block device mapping of the AMI from being mapped to the specified device name at launch. If ``NoDevice`` is ``true`` for the root device, instances might fail the EC2 health check. In that case, Amazon EC2 Auto Scaling launches replacement instances.
            :param virtual_name: The name of the instance store volume (virtual device) to attach to an instance at launch. The name must be in the form ephemeral *X* where *X* is a number starting from zero (0), for example, ``ephemeral0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevicemapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                block_device_mapping_property = autoscaling.CfnLaunchConfiguration.BlockDeviceMappingProperty(
                    device_name="deviceName",
                
                    # the properties below are optional
                    ebs=autoscaling.CfnLaunchConfiguration.BlockDeviceProperty(
                        delete_on_termination=False,
                        encrypted=False,
                        iops=123,
                        snapshot_id="snapshotId",
                        throughput=123,
                        volume_size=123,
                        volume_type="volumeType"
                    ),
                    no_device=False,
                    virtual_name="virtualName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__28cc3ae48c0185c32554edba8688366b44d110b2707329683cc7a64163c2b0bf)
                check_type(argname="argument device_name", value=device_name, expected_type=type_hints["device_name"])
                check_type(argname="argument ebs", value=ebs, expected_type=type_hints["ebs"])
                check_type(argname="argument no_device", value=no_device, expected_type=type_hints["no_device"])
                check_type(argname="argument virtual_name", value=virtual_name, expected_type=type_hints["virtual_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "device_name": device_name,
            }
            if ebs is not None:
                self._values["ebs"] = ebs
            if no_device is not None:
                self._values["no_device"] = no_device
            if virtual_name is not None:
                self._values["virtual_name"] = virtual_name

        @builtins.property
        def device_name(self) -> builtins.str:
            '''The device name assigned to the volume (for example, ``/dev/sdh`` or ``xvdh`` ).

            For more information, see `Device naming on Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
            .. epigraph::

               To define a block device mapping, set the device name and exactly one of the following properties: ``Ebs`` , ``NoDevice`` , or ``VirtualName`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevicemapping.html#cfn-autoscaling-launchconfiguration-blockdevicemapping-devicename
            '''
            result = self._values.get("device_name")
            assert result is not None, "Required property 'device_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ebs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.BlockDeviceProperty"]]:
            '''Information to attach an EBS volume to an instance at launch.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevicemapping.html#cfn-autoscaling-launchconfiguration-blockdevicemapping-ebs
            '''
            result = self._values.get("ebs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLaunchConfiguration.BlockDeviceProperty"]], result)

        @builtins.property
        def no_device(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Setting this value to ``true`` prevents a volume that is included in the block device mapping of the AMI from being mapped to the specified device name at launch.

            If ``NoDevice`` is ``true`` for the root device, instances might fail the EC2 health check. In that case, Amazon EC2 Auto Scaling launches replacement instances.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevicemapping.html#cfn-autoscaling-launchconfiguration-blockdevicemapping-nodevice
            '''
            result = self._values.get("no_device")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def virtual_name(self) -> typing.Optional[builtins.str]:
            '''The name of the instance store volume (virtual device) to attach to an instance at launch.

            The name must be in the form ephemeral *X* where *X* is a number starting from zero (0), for example, ``ephemeral0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevicemapping.html#cfn-autoscaling-launchconfiguration-blockdevicemapping-virtualname
            '''
            result = self._values.get("virtual_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnLaunchConfiguration.BlockDeviceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "delete_on_termination": "deleteOnTermination",
            "encrypted": "encrypted",
            "iops": "iops",
            "snapshot_id": "snapshotId",
            "throughput": "throughput",
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
        },
    )
    class BlockDeviceProperty:
        def __init__(
            self,
            *,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            iops: typing.Optional[jsii.Number] = None,
            snapshot_id: typing.Optional[builtins.str] = None,
            throughput: typing.Optional[jsii.Number] = None,
            volume_size: typing.Optional[jsii.Number] = None,
            volume_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``BlockDevice`` is a property of the ``EBS`` property of the `AWS::AutoScaling::LaunchConfiguration BlockDeviceMapping <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-launchconfig-blockdev-mapping.html>`_ property type that describes an Amazon EBS volume.

            :param delete_on_termination: Indicates whether the volume is deleted on instance termination. For Amazon EC2 Auto Scaling, the default value is ``true`` .
            :param encrypted: Specifies whether the volume should be encrypted. Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption. For more information, see `Supported instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances>`_ . If your AMI uses encrypted volumes, you can also only launch it on supported instance types. .. epigraph:: If you are creating a volume from a snapshot, you cannot create an unencrypted volume from an encrypted snapshot. Also, you cannot specify a KMS key ID when using a launch configuration. If you enable encryption by default, the EBS volumes that you create are always encrypted, either using the AWS managed KMS key or a customer-managed KMS key, regardless of whether the snapshot was encrypted. For more information, see `Use AWS KMS keys to encrypt Amazon EBS volumes <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-data-protection.html#encryption>`_ in the *Amazon EC2 Auto Scaling User Guide* .
            :param iops: The number of input/output (I/O) operations per second (IOPS) to provision for the volume. For ``gp3`` and ``io1`` volumes, this represents the number of IOPS that are provisioned for the volume. For ``gp2`` volumes, this represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits for bursting. The following are the supported values for each volume type: - ``gp3`` : 3,000-16,000 IOPS - ``io1`` : 100-64,000 IOPS For ``io1`` volumes, we guarantee 64,000 IOPS only for `Instances built on the Nitro System <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ . Other instance families guarantee performance up to 32,000 IOPS. ``Iops`` is supported when the volume type is ``gp3`` or ``io1`` and required only when the volume type is ``io1`` . (Not used with ``standard`` , ``gp2`` , ``st1`` , or ``sc1`` volumes.)
            :param snapshot_id: The snapshot ID of the volume to use. You must specify either a ``VolumeSize`` or a ``SnapshotId`` .
            :param throughput: The throughput (MiBps) to provision for a ``gp3`` volume.
            :param volume_size: The volume size, in GiBs. The following are the supported volumes sizes for each volume type:. - ``gp2`` and ``gp3`` : 1-16,384 - ``io1`` : 4-16,384 - ``st1`` and ``sc1`` : 125-16,384 - ``standard`` : 1-1,024 You must specify either a ``SnapshotId`` or a ``VolumeSize`` . If you specify both ``SnapshotId`` and ``VolumeSize`` , the volume size must be equal or greater than the size of the snapshot.
            :param volume_type: The volume type. For more information, see `Amazon EBS volume types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . Valid values: ``standard`` | ``io1`` | ``gp2`` | ``st1`` | ``sc1`` | ``gp3``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                block_device_property = autoscaling.CfnLaunchConfiguration.BlockDeviceProperty(
                    delete_on_termination=False,
                    encrypted=False,
                    iops=123,
                    snapshot_id="snapshotId",
                    throughput=123,
                    volume_size=123,
                    volume_type="volumeType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cb46621b8247621ce8684de7d0b8d40deb45a021b4eebaa84336924a1a9f478a)
                check_type(argname="argument delete_on_termination", value=delete_on_termination, expected_type=type_hints["delete_on_termination"])
                check_type(argname="argument encrypted", value=encrypted, expected_type=type_hints["encrypted"])
                check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
                check_type(argname="argument snapshot_id", value=snapshot_id, expected_type=type_hints["snapshot_id"])
                check_type(argname="argument throughput", value=throughput, expected_type=type_hints["throughput"])
                check_type(argname="argument volume_size", value=volume_size, expected_type=type_hints["volume_size"])
                check_type(argname="argument volume_type", value=volume_type, expected_type=type_hints["volume_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if encrypted is not None:
                self._values["encrypted"] = encrypted
            if iops is not None:
                self._values["iops"] = iops
            if snapshot_id is not None:
                self._values["snapshot_id"] = snapshot_id
            if throughput is not None:
                self._values["throughput"] = throughput
            if volume_size is not None:
                self._values["volume_size"] = volume_size
            if volume_type is not None:
                self._values["volume_type"] = volume_type

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether the volume is deleted on instance termination.

            For Amazon EC2 Auto Scaling, the default value is ``true`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether the volume should be encrypted.

            Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption. For more information, see `Supported instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances>`_ . If your AMI uses encrypted volumes, you can also only launch it on supported instance types.
            .. epigraph::

               If you are creating a volume from a snapshot, you cannot create an unencrypted volume from an encrypted snapshot. Also, you cannot specify a KMS key ID when using a launch configuration.

               If you enable encryption by default, the EBS volumes that you create are always encrypted, either using the AWS managed KMS key or a customer-managed KMS key, regardless of whether the snapshot was encrypted.

               For more information, see `Use AWS KMS keys to encrypt Amazon EBS volumes <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-data-protection.html#encryption>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-encrypted
            '''
            result = self._values.get("encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''The number of input/output (I/O) operations per second (IOPS) to provision for the volume.

            For ``gp3`` and ``io1`` volumes, this represents the number of IOPS that are provisioned for the volume. For ``gp2`` volumes, this represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits for bursting.

            The following are the supported values for each volume type:

            - ``gp3`` : 3,000-16,000 IOPS
            - ``io1`` : 100-64,000 IOPS

            For ``io1`` volumes, we guarantee 64,000 IOPS only for `Instances built on the Nitro System <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances>`_ . Other instance families guarantee performance up to 32,000 IOPS.

            ``Iops`` is supported when the volume type is ``gp3`` or ``io1`` and required only when the volume type is ``io1`` . (Not used with ``standard`` , ``gp2`` , ``st1`` , or ``sc1`` volumes.)

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-iops
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def snapshot_id(self) -> typing.Optional[builtins.str]:
            '''The snapshot ID of the volume to use.

            You must specify either a ``VolumeSize`` or a ``SnapshotId`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-snapshotid
            '''
            result = self._values.get("snapshot_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throughput(self) -> typing.Optional[jsii.Number]:
            '''The throughput (MiBps) to provision for a ``gp3`` volume.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-throughput
            '''
            result = self._values.get("throughput")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_size(self) -> typing.Optional[jsii.Number]:
            '''The volume size, in GiBs. The following are the supported volumes sizes for each volume type:.

            - ``gp2`` and ``gp3`` : 1-16,384
            - ``io1`` : 4-16,384
            - ``st1`` and ``sc1`` : 125-16,384
            - ``standard`` : 1-1,024

            You must specify either a ``SnapshotId`` or a ``VolumeSize`` . If you specify both ``SnapshotId`` and ``VolumeSize`` , the volume size must be equal or greater than the size of the snapshot.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-volumesize
            '''
            result = self._values.get("volume_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_type(self) -> typing.Optional[builtins.str]:
            '''The volume type.

            For more information, see `Amazon EBS volume types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            Valid values: ``standard`` | ``io1`` | ``gp2`` | ``st1`` | ``sc1`` | ``gp3``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-blockdevice.html#cfn-autoscaling-launchconfiguration-blockdevice-volumetype
            '''
            result = self._values.get("volume_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnLaunchConfiguration.MetadataOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "http_endpoint": "httpEndpoint",
            "http_put_response_hop_limit": "httpPutResponseHopLimit",
            "http_tokens": "httpTokens",
        },
    )
    class MetadataOptionsProperty:
        def __init__(
            self,
            *,
            http_endpoint: typing.Optional[builtins.str] = None,
            http_put_response_hop_limit: typing.Optional[jsii.Number] = None,
            http_tokens: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``MetadataOptions`` is a property of `AWS::AutoScaling::LaunchConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-launchconfig.html>`_ that describes metadata options for the instances.

            For more information, see `Configure the instance metadata options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-config.html#launch-configurations-imds>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param http_endpoint: This parameter enables or disables the HTTP metadata endpoint on your instances. If the parameter is not specified, the default state is ``enabled`` . .. epigraph:: If you specify a value of ``disabled`` , you will not be able to access your instance metadata.
            :param http_put_response_hop_limit: The desired HTTP PUT response hop limit for instance metadata requests. The larger the number, the further instance metadata requests can travel. Default: 1
            :param http_tokens: The state of token usage for your instance metadata requests. If the parameter is not specified in the request, the default state is ``optional`` . If the state is ``optional`` , you can choose to retrieve instance metadata with or without a signed token header on your request. If you retrieve the IAM role credentials without a token, the version 1.0 role credentials are returned. If you retrieve the IAM role credentials using a valid signed token, the version 2.0 role credentials are returned. If the state is ``required`` , you must send a signed token header with any instance metadata retrieval requests. In this state, retrieving the IAM role credentials always returns the version 2.0 credentials; the version 1.0 credentials are not available.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-metadataoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                metadata_options_property = autoscaling.CfnLaunchConfiguration.MetadataOptionsProperty(
                    http_endpoint="httpEndpoint",
                    http_put_response_hop_limit=123,
                    http_tokens="httpTokens"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5b9184da3be6946dea9c7fd293238b1b50dd5b0051db0d4fda31f6734b79f782)
                check_type(argname="argument http_endpoint", value=http_endpoint, expected_type=type_hints["http_endpoint"])
                check_type(argname="argument http_put_response_hop_limit", value=http_put_response_hop_limit, expected_type=type_hints["http_put_response_hop_limit"])
                check_type(argname="argument http_tokens", value=http_tokens, expected_type=type_hints["http_tokens"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if http_endpoint is not None:
                self._values["http_endpoint"] = http_endpoint
            if http_put_response_hop_limit is not None:
                self._values["http_put_response_hop_limit"] = http_put_response_hop_limit
            if http_tokens is not None:
                self._values["http_tokens"] = http_tokens

        @builtins.property
        def http_endpoint(self) -> typing.Optional[builtins.str]:
            '''This parameter enables or disables the HTTP metadata endpoint on your instances.

            If the parameter is not specified, the default state is ``enabled`` .
            .. epigraph::

               If you specify a value of ``disabled`` , you will not be able to access your instance metadata.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-metadataoptions.html#cfn-autoscaling-launchconfiguration-metadataoptions-httpendpoint
            '''
            result = self._values.get("http_endpoint")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def http_put_response_hop_limit(self) -> typing.Optional[jsii.Number]:
            '''The desired HTTP PUT response hop limit for instance metadata requests.

            The larger the number, the further instance metadata requests can travel.

            Default: 1

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-metadataoptions.html#cfn-autoscaling-launchconfiguration-metadataoptions-httpputresponsehoplimit
            '''
            result = self._values.get("http_put_response_hop_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def http_tokens(self) -> typing.Optional[builtins.str]:
            '''The state of token usage for your instance metadata requests.

            If the parameter is not specified in the request, the default state is ``optional`` .

            If the state is ``optional`` , you can choose to retrieve instance metadata with or without a signed token header on your request. If you retrieve the IAM role credentials without a token, the version 1.0 role credentials are returned. If you retrieve the IAM role credentials using a valid signed token, the version 2.0 role credentials are returned.

            If the state is ``required`` , you must send a signed token header with any instance metadata retrieval requests. In this state, retrieving the IAM role credentials always returns the version 2.0 credentials; the version 1.0 credentials are not available.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-launchconfiguration-metadataoptions.html#cfn-autoscaling-launchconfiguration-metadataoptions-httptokens
            '''
            result = self._values.get("http_tokens")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetadataOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnLaunchConfigurationProps",
    jsii_struct_bases=[],
    name_mapping={
        "image_id": "imageId",
        "instance_type": "instanceType",
        "associate_public_ip_address": "associatePublicIpAddress",
        "block_device_mappings": "blockDeviceMappings",
        "classic_link_vpc_id": "classicLinkVpcId",
        "classic_link_vpc_security_groups": "classicLinkVpcSecurityGroups",
        "ebs_optimized": "ebsOptimized",
        "iam_instance_profile": "iamInstanceProfile",
        "instance_id": "instanceId",
        "instance_monitoring": "instanceMonitoring",
        "kernel_id": "kernelId",
        "key_name": "keyName",
        "launch_configuration_name": "launchConfigurationName",
        "metadata_options": "metadataOptions",
        "placement_tenancy": "placementTenancy",
        "ram_disk_id": "ramDiskId",
        "security_groups": "securityGroups",
        "spot_price": "spotPrice",
        "user_data": "userData",
    },
)
class CfnLaunchConfigurationProps:
    def __init__(
        self,
        *,
        image_id: builtins.str,
        instance_type: builtins.str,
        associate_public_ip_address: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        block_device_mappings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnLaunchConfiguration.BlockDeviceMappingProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        classic_link_vpc_id: typing.Optional[builtins.str] = None,
        classic_link_vpc_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        iam_instance_profile: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        instance_monitoring: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        kernel_id: typing.Optional[builtins.str] = None,
        key_name: typing.Optional[builtins.str] = None,
        launch_configuration_name: typing.Optional[builtins.str] = None,
        metadata_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnLaunchConfiguration.MetadataOptionsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        placement_tenancy: typing.Optional[builtins.str] = None,
        ram_disk_id: typing.Optional[builtins.str] = None,
        security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        spot_price: typing.Optional[builtins.str] = None,
        user_data: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnLaunchConfiguration``.

        :param image_id: The ID of the Amazon Machine Image (AMI) that was assigned during registration. For more information, see `Finding a Linux AMI <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/finding-an-ami.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . If you specify ``InstanceId`` , an ``ImageId`` is not required.
        :param instance_type: Specifies the instance type of the EC2 instance. For information about available instance types, see `Available instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#AvailableInstanceTypes>`_ in the *Amazon EC2 User Guide for Linux Instances* . If you specify ``InstanceId`` , an ``InstanceType`` is not required.
        :param associate_public_ip_address: Specifies whether to assign a public IPv4 address to the group's instances. If the instance is launched into a default subnet, the default is to assign a public IPv4 address, unless you disabled the option to assign a public IPv4 address on the subnet. If the instance is launched into a nondefault subnet, the default is not to assign a public IPv4 address, unless you enabled the option to assign a public IPv4 address on the subnet. If you specify ``true`` , each instance in the Auto Scaling group receives a unique public IPv4 address. For more information, see `Launching Auto Scaling instances in a VPC <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-in-vpc.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you specify this property, you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group.
        :param block_device_mappings: The block device mapping entries that define the block devices to attach to the instances at launch. By default, the block devices specified in the block device mapping for the AMI are used. For more information, see `Block device mappings <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param classic_link_vpc_id: Available for backward compatibility.
        :param classic_link_vpc_security_groups: Available for backward compatibility.
        :param ebs_optimized: Specifies whether the launch configuration is optimized for EBS I/O ( ``true`` ) or not ( ``false`` ). The optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization is not available with all instance types. Additional fees are incurred when you enable EBS optimization for an instance type that is not EBS-optimized by default. For more information, see `Amazon EBS-optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . The default value is ``false`` .
        :param iam_instance_profile: The name or the Amazon Resource Name (ARN) of the instance profile associated with the IAM role for the instance. The instance profile contains the IAM role. For more information, see `IAM role for applications that run on Amazon EC2 instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/us-iam-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param instance_id: The ID of the Amazon EC2 instance to use to create the launch configuration. When you use an instance to create a launch configuration, all properties are derived from the instance with the exception of ``BlockDeviceMapping`` and ``AssociatePublicIpAddress`` . You can override any properties from the instance by specifying them in the launch configuration.
        :param instance_monitoring: Controls whether instances in this group are launched with detailed ( ``true`` ) or basic ( ``false`` ) monitoring. The default value is ``true`` (enabled). .. epigraph:: When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. For more information, see `Configure Monitoring for Auto Scaling Instances <https://docs.aws.amazon.com/autoscaling/latest/userguide/enable-as-instance-metrics.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param kernel_id: The ID of the kernel associated with the AMI. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param key_name: The name of the key pair. For more information, see `Amazon EC2 key pairs and Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param launch_configuration_name: The name of the launch configuration. This name must be unique per Region per account.
        :param metadata_options: The metadata options for the instances. For more information, see `Configuring the Instance Metadata Options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-config.html#launch-configurations-imds>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param placement_tenancy: The tenancy of the instance, either ``default`` or ``dedicated`` . An instance with ``dedicated`` tenancy runs on isolated, single-tenant hardware and can only be launched into a VPC. To launch dedicated instances into a shared tenancy VPC (a VPC with the instance placement tenancy attribute set to ``default`` ), you must set the value of this property to ``dedicated`` . For more information, see `Configuring instance tenancy with Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/auto-scaling-dedicated-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . If you specify ``PlacementTenancy`` , you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group. Valid values: ``default`` | ``dedicated``
        :param ram_disk_id: The ID of the RAM disk to select. .. epigraph:: We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .
        :param security_groups: A list that contains the security groups to assign to the instances in the Auto Scaling group. The list can contain both the IDs of existing security groups and references to `SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template. For more information, see `Control traffic to resources using security groups <https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_SecurityGroups.html>`_ in the *Amazon Virtual Private Cloud User Guide* .
        :param spot_price: The maximum hourly price to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot price. For more information, see `Request Spot Instances for fault-tolerant and flexible applications <https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-template-spot-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid Range: Minimum value of 0.001 .. epigraph:: When you change your maximum price by creating a new launch configuration, running instances will continue to run as long as the maximum price for those running instances is higher than the current Spot price.
        :param user_data: The Base64-encoded user data to make available to the launched EC2 instances. For more information, see `Instance metadata and user data <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            cfn_launch_configuration_props = autoscaling.CfnLaunchConfigurationProps(
                image_id="imageId",
                instance_type="instanceType",
            
                # the properties below are optional
                associate_public_ip_address=False,
                block_device_mappings=[autoscaling.CfnLaunchConfiguration.BlockDeviceMappingProperty(
                    device_name="deviceName",
            
                    # the properties below are optional
                    ebs=autoscaling.CfnLaunchConfiguration.BlockDeviceProperty(
                        delete_on_termination=False,
                        encrypted=False,
                        iops=123,
                        snapshot_id="snapshotId",
                        throughput=123,
                        volume_size=123,
                        volume_type="volumeType"
                    ),
                    no_device=False,
                    virtual_name="virtualName"
                )],
                classic_link_vpc_id="classicLinkVpcId",
                classic_link_vpc_security_groups=["classicLinkVpcSecurityGroups"],
                ebs_optimized=False,
                iam_instance_profile="iamInstanceProfile",
                instance_id="instanceId",
                instance_monitoring=False,
                kernel_id="kernelId",
                key_name="keyName",
                launch_configuration_name="launchConfigurationName",
                metadata_options=autoscaling.CfnLaunchConfiguration.MetadataOptionsProperty(
                    http_endpoint="httpEndpoint",
                    http_put_response_hop_limit=123,
                    http_tokens="httpTokens"
                ),
                placement_tenancy="placementTenancy",
                ram_disk_id="ramDiskId",
                security_groups=["securityGroups"],
                spot_price="spotPrice",
                user_data="userData"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__437aca81a1a333c2908320df15fa11c3acd609bc0ca76c62d631e3efe7b6ddd6)
            check_type(argname="argument image_id", value=image_id, expected_type=type_hints["image_id"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument associate_public_ip_address", value=associate_public_ip_address, expected_type=type_hints["associate_public_ip_address"])
            check_type(argname="argument block_device_mappings", value=block_device_mappings, expected_type=type_hints["block_device_mappings"])
            check_type(argname="argument classic_link_vpc_id", value=classic_link_vpc_id, expected_type=type_hints["classic_link_vpc_id"])
            check_type(argname="argument classic_link_vpc_security_groups", value=classic_link_vpc_security_groups, expected_type=type_hints["classic_link_vpc_security_groups"])
            check_type(argname="argument ebs_optimized", value=ebs_optimized, expected_type=type_hints["ebs_optimized"])
            check_type(argname="argument iam_instance_profile", value=iam_instance_profile, expected_type=type_hints["iam_instance_profile"])
            check_type(argname="argument instance_id", value=instance_id, expected_type=type_hints["instance_id"])
            check_type(argname="argument instance_monitoring", value=instance_monitoring, expected_type=type_hints["instance_monitoring"])
            check_type(argname="argument kernel_id", value=kernel_id, expected_type=type_hints["kernel_id"])
            check_type(argname="argument key_name", value=key_name, expected_type=type_hints["key_name"])
            check_type(argname="argument launch_configuration_name", value=launch_configuration_name, expected_type=type_hints["launch_configuration_name"])
            check_type(argname="argument metadata_options", value=metadata_options, expected_type=type_hints["metadata_options"])
            check_type(argname="argument placement_tenancy", value=placement_tenancy, expected_type=type_hints["placement_tenancy"])
            check_type(argname="argument ram_disk_id", value=ram_disk_id, expected_type=type_hints["ram_disk_id"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument spot_price", value=spot_price, expected_type=type_hints["spot_price"])
            check_type(argname="argument user_data", value=user_data, expected_type=type_hints["user_data"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image_id": image_id,
            "instance_type": instance_type,
        }
        if associate_public_ip_address is not None:
            self._values["associate_public_ip_address"] = associate_public_ip_address
        if block_device_mappings is not None:
            self._values["block_device_mappings"] = block_device_mappings
        if classic_link_vpc_id is not None:
            self._values["classic_link_vpc_id"] = classic_link_vpc_id
        if classic_link_vpc_security_groups is not None:
            self._values["classic_link_vpc_security_groups"] = classic_link_vpc_security_groups
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if iam_instance_profile is not None:
            self._values["iam_instance_profile"] = iam_instance_profile
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if instance_monitoring is not None:
            self._values["instance_monitoring"] = instance_monitoring
        if kernel_id is not None:
            self._values["kernel_id"] = kernel_id
        if key_name is not None:
            self._values["key_name"] = key_name
        if launch_configuration_name is not None:
            self._values["launch_configuration_name"] = launch_configuration_name
        if metadata_options is not None:
            self._values["metadata_options"] = metadata_options
        if placement_tenancy is not None:
            self._values["placement_tenancy"] = placement_tenancy
        if ram_disk_id is not None:
            self._values["ram_disk_id"] = ram_disk_id
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if spot_price is not None:
            self._values["spot_price"] = spot_price
        if user_data is not None:
            self._values["user_data"] = user_data

    @builtins.property
    def image_id(self) -> builtins.str:
        '''The ID of the Amazon Machine Image (AMI) that was assigned during registration.

        For more information, see `Finding a Linux AMI <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/finding-an-ami.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        If you specify ``InstanceId`` , an ``ImageId`` is not required.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-imageid
        '''
        result = self._values.get("image_id")
        assert result is not None, "Required property 'image_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> builtins.str:
        '''Specifies the instance type of the EC2 instance.

        For information about available instance types, see `Available instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#AvailableInstanceTypes>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        If you specify ``InstanceId`` , an ``InstanceType`` is not required.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-instancetype
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def associate_public_ip_address(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to assign a public IPv4 address to the group's instances.

        If the instance is launched into a default subnet, the default is to assign a public IPv4 address, unless you disabled the option to assign a public IPv4 address on the subnet. If the instance is launched into a nondefault subnet, the default is not to assign a public IPv4 address, unless you enabled the option to assign a public IPv4 address on the subnet.

        If you specify ``true`` , each instance in the Auto Scaling group receives a unique public IPv4 address. For more information, see `Launching Auto Scaling instances in a VPC <https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-in-vpc.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you specify this property, you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-associatepublicipaddress
        '''
        result = self._values.get("associate_public_ip_address")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def block_device_mappings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnLaunchConfiguration.BlockDeviceMappingProperty]]]]:
        '''The block device mapping entries that define the block devices to attach to the instances at launch.

        By default, the block devices specified in the block device mapping for the AMI are used. For more information, see `Block device mappings <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-blockdevicemappings
        '''
        result = self._values.get("block_device_mappings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnLaunchConfiguration.BlockDeviceMappingProperty]]]], result)

    @builtins.property
    def classic_link_vpc_id(self) -> typing.Optional[builtins.str]:
        '''Available for backward compatibility.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-classiclinkvpcid
        '''
        result = self._values.get("classic_link_vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def classic_link_vpc_security_groups(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''Available for backward compatibility.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-classiclinkvpcsecuritygroups
        '''
        result = self._values.get("classic_link_vpc_security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the launch configuration is optimized for EBS I/O ( ``true`` ) or not ( ``false`` ).

        The optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization is not available with all instance types. Additional fees are incurred when you enable EBS optimization for an instance type that is not EBS-optimized by default. For more information, see `Amazon EBS-optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        The default value is ``false`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-ebsoptimized
        '''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''The name or the Amazon Resource Name (ARN) of the instance profile associated with the IAM role for the instance.

        The instance profile contains the IAM role. For more information, see `IAM role for applications that run on Amazon EC2 instances <https://docs.aws.amazon.com/autoscaling/ec2/userguide/us-iam-role.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-iaminstanceprofile
        '''
        result = self._values.get("iam_instance_profile")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the Amazon EC2 instance to use to create the launch configuration.

        When you use an instance to create a launch configuration, all properties are derived from the instance with the exception of ``BlockDeviceMapping`` and ``AssociatePublicIpAddress`` . You can override any properties from the instance by specifying them in the launch configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_monitoring(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Controls whether instances in this group are launched with detailed ( ``true`` ) or basic ( ``false`` ) monitoring.

        The default value is ``true`` (enabled).
        .. epigraph::

           When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. For more information, see `Configure Monitoring for Auto Scaling Instances <https://docs.aws.amazon.com/autoscaling/latest/userguide/enable-as-instance-metrics.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-instancemonitoring
        '''
        result = self._values.get("instance_monitoring")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def kernel_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the kernel associated with the AMI.

        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-kernelid
        '''
        result = self._values.get("kernel_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''The name of the key pair.

        For more information, see `Amazon EC2 key pairs and Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-keyname
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_configuration_name(self) -> typing.Optional[builtins.str]:
        '''The name of the launch configuration.

        This name must be unique per Region per account.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-launchconfigurationname
        '''
        result = self._values.get("launch_configuration_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def metadata_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnLaunchConfiguration.MetadataOptionsProperty]]:
        '''The metadata options for the instances.

        For more information, see `Configuring the Instance Metadata Options <https://docs.aws.amazon.com/autoscaling/ec2/userguide/create-launch-config.html#launch-configurations-imds>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-metadataoptions
        '''
        result = self._values.get("metadata_options")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnLaunchConfiguration.MetadataOptionsProperty]], result)

    @builtins.property
    def placement_tenancy(self) -> typing.Optional[builtins.str]:
        '''The tenancy of the instance, either ``default`` or ``dedicated`` .

        An instance with ``dedicated`` tenancy runs on isolated, single-tenant hardware and can only be launched into a VPC. To launch dedicated instances into a shared tenancy VPC (a VPC with the instance placement tenancy attribute set to ``default`` ), you must set the value of this property to ``dedicated`` . For more information, see `Configuring instance tenancy with Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/auto-scaling-dedicated-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        If you specify ``PlacementTenancy`` , you must specify at least one subnet for ``VPCZoneIdentifier`` when you create your group.

        Valid values: ``default`` | ``dedicated``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-placementtenancy
        '''
        result = self._values.get("placement_tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ram_disk_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the RAM disk to select.

        .. epigraph::

           We recommend that you use PV-GRUB instead of kernels and RAM disks. For more information, see `User provided kernels <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/UserProvidedKernels.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-ramdiskid
        '''
        result = self._values.get("ram_disk_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list that contains the security groups to assign to the instances in the Auto Scaling group.

        The list can contain both the IDs of existing security groups and references to `SecurityGroup <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html>`_ resources created in the template.

        For more information, see `Control traffic to resources using security groups <https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_SecurityGroups.html>`_ in the *Amazon Virtual Private Cloud User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-securitygroups
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''The maximum hourly price to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are launched when the price you specify exceeds the current Spot price. For more information, see `Request Spot Instances for fault-tolerant and flexible applications <https://docs.aws.amazon.com/autoscaling/ec2/userguide/launch-template-spot-instances.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Valid Range: Minimum value of 0.001
        .. epigraph::

           When you change your maximum price by creating a new launch configuration, running instances will continue to run as long as the maximum price for those running instances is higher than the current Spot price.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-spotprice
        '''
        result = self._values.get("spot_price")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user_data(self) -> typing.Optional[builtins.str]:
        '''The Base64-encoded user data to make available to the launched EC2 instances.

        For more information, see `Instance metadata and user data <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-launchconfiguration.html#cfn-autoscaling-launchconfiguration-userdata
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLaunchConfigurationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnLifecycleHook(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnLifecycleHook",
):
    '''A CloudFormation ``AWS::AutoScaling::LifecycleHook``.

    The ``AWS::AutoScaling::LifecycleHook`` resource specifies lifecycle hooks for an Auto Scaling group. These hooks let you create solutions that are aware of events in the Auto Scaling instance lifecycle, and then perform a custom action on instances when the corresponding lifecycle event occurs. A lifecycle hook provides a specified amount of time (one hour by default) to wait for the action to complete before the instance transitions to the next state.

    Use lifecycle hooks to prepare new instances for use or to delay them from being registered behind a load balancer before their configuration has been applied completely. You can also use lifecycle hooks to prepare running instances to be terminated by, for example, downloading logs or other data.

    For more information, see `Amazon EC2 Auto Scaling lifecycle hooks <https://docs.aws.amazon.com/autoscaling/ec2/userguide/lifecycle-hooks.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

    :cloudformationResource: AWS::AutoScaling::LifecycleHook
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        cfn_lifecycle_hook = autoscaling.CfnLifecycleHook(self, "MyCfnLifecycleHook",
            auto_scaling_group_name="autoScalingGroupName",
            lifecycle_transition="lifecycleTransition",
        
            # the properties below are optional
            default_result="defaultResult",
            heartbeat_timeout=123,
            lifecycle_hook_name="lifecycleHookName",
            notification_metadata="notificationMetadata",
            notification_target_arn="notificationTargetArn",
            role_arn="roleArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group_name: builtins.str,
        lifecycle_transition: builtins.str,
        default_result: typing.Optional[builtins.str] = None,
        heartbeat_timeout: typing.Optional[jsii.Number] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target_arn: typing.Optional[builtins.str] = None,
        role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AutoScaling::LifecycleHook``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param lifecycle_transition: The lifecycle transition. For Auto Scaling groups, there are two major lifecycle transitions. - To create a lifecycle hook for scale-out events, specify ``autoscaling:EC2_INSTANCE_LAUNCHING`` . - To create a lifecycle hook for scale-in events, specify ``autoscaling:EC2_INSTANCE_TERMINATING`` .
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. The default value is ``ABANDON`` . Valid values: ``CONTINUE`` | ``ABANDON``
        :param heartbeat_timeout: The maximum time, in seconds, that can elapse before the lifecycle hook times out. The range is from ``30`` to ``7200`` seconds. The default value is ``3600`` seconds (1 hour).
        :param lifecycle_hook_name: The name of the lifecycle hook.
        :param notification_metadata: Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the notification target.
        :param notification_target_arn: The Amazon Resource Name (ARN) of the notification target that Amazon EC2 Auto Scaling sends notifications to when an instance is in a wait state for the lifecycle hook. You can specify an Amazon SNS topic or an Amazon SQS queue.
        :param role_arn: The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target. For information about creating this role, see `Configure a notification target for a lifecycle hook <https://docs.aws.amazon.com/autoscaling/ec2/userguide/prepare-for-lifecycle-notifications.html#lifecycle-hook-notification-target>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid only if the notification target is an Amazon SNS topic or an Amazon SQS queue.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dabd61d8a949732e25b3952a106099c905072c82cee5882a0626359068cddbf4)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLifecycleHookProps(
            auto_scaling_group_name=auto_scaling_group_name,
            lifecycle_transition=lifecycle_transition,
            default_result=default_result,
            heartbeat_timeout=heartbeat_timeout,
            lifecycle_hook_name=lifecycle_hook_name,
            notification_metadata=notification_metadata,
            notification_target_arn=notification_target_arn,
            role_arn=role_arn,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dcd6cbadd1cca7e18a403680449778d0e0a8e19fb670a40420c3b3f21fe301b9)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__adfc17ebc81352492f6cbef914cb8fe36f4240d4c671e061748d8587dcfb31fc)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-autoscalinggroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "autoScalingGroupName"))

    @auto_scaling_group_name.setter
    def auto_scaling_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35c060ad9285fb6c54582d235f18ba5215a7979adb3b4fc8a6ff8fb11e6c1457)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoScalingGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="lifecycleTransition")
    def lifecycle_transition(self) -> builtins.str:
        '''The lifecycle transition. For Auto Scaling groups, there are two major lifecycle transitions.

        - To create a lifecycle hook for scale-out events, specify ``autoscaling:EC2_INSTANCE_LAUNCHING`` .
        - To create a lifecycle hook for scale-in events, specify ``autoscaling:EC2_INSTANCE_TERMINATING`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-lifecycletransition
        '''
        return typing.cast(builtins.str, jsii.get(self, "lifecycleTransition"))

    @lifecycle_transition.setter
    def lifecycle_transition(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87c9ac4cd78c5650b361a6ba49286daa88b90741e3e54c32bc18b68ac11fba24)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "lifecycleTransition", value)

    @builtins.property
    @jsii.member(jsii_name="defaultResult")
    def default_result(self) -> typing.Optional[builtins.str]:
        '''The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs.

        The default value is ``ABANDON`` .

        Valid values: ``CONTINUE`` | ``ABANDON``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-defaultresult
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "defaultResult"))

    @default_result.setter
    def default_result(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51517b719f77ce6d6ba6adb7f8c70b2413715e529443ab685154429e8fb59402)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultResult", value)

    @builtins.property
    @jsii.member(jsii_name="heartbeatTimeout")
    def heartbeat_timeout(self) -> typing.Optional[jsii.Number]:
        '''The maximum time, in seconds, that can elapse before the lifecycle hook times out.

        The range is from ``30`` to ``7200`` seconds. The default value is ``3600`` seconds (1 hour).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-heartbeattimeout
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "heartbeatTimeout"))

    @heartbeat_timeout.setter
    def heartbeat_timeout(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38bf2cb9d10d5f01920b8eda34a7f2517db573690c3c85cffda12276b09a3b59)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "heartbeatTimeout", value)

    @builtins.property
    @jsii.member(jsii_name="lifecycleHookName")
    def lifecycle_hook_name(self) -> typing.Optional[builtins.str]:
        '''The name of the lifecycle hook.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-lifecyclehookname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "lifecycleHookName"))

    @lifecycle_hook_name.setter
    def lifecycle_hook_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d5bbfb114220f5d887cc11cd66c4e915d640c5992791aa59c73e07227a04e797)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "lifecycleHookName", value)

    @builtins.property
    @jsii.member(jsii_name="notificationMetadata")
    def notification_metadata(self) -> typing.Optional[builtins.str]:
        '''Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the notification target.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-notificationmetadata
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "notificationMetadata"))

    @notification_metadata.setter
    def notification_metadata(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27918b799a8772981ce11d1d726a74b3feb0bdae57a9890dac50d40416a4b102)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "notificationMetadata", value)

    @builtins.property
    @jsii.member(jsii_name="notificationTargetArn")
    def notification_target_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the notification target that Amazon EC2 Auto Scaling sends notifications to when an instance is in a wait state for the lifecycle hook.

        You can specify an Amazon SNS topic or an Amazon SQS queue.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-notificationtargetarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "notificationTargetArn"))

    @notification_target_arn.setter
    def notification_target_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1be65379a9c211c2e27b903c326bd9b6c233e24d2827c4b0501fdedb5c613e1b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "notificationTargetArn", value)

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target.

        For information about creating this role, see `Configure a notification target for a lifecycle hook <https://docs.aws.amazon.com/autoscaling/ec2/userguide/prepare-for-lifecycle-notifications.html#lifecycle-hook-notification-target>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Valid only if the notification target is an Amazon SNS topic or an Amazon SQS queue.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-rolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3efe05ccb204a2f5b10e050508641230c374dcfa8454a54ed2b9fa3d2823782)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnLifecycleHookProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_name": "autoScalingGroupName",
        "lifecycle_transition": "lifecycleTransition",
        "default_result": "defaultResult",
        "heartbeat_timeout": "heartbeatTimeout",
        "lifecycle_hook_name": "lifecycleHookName",
        "notification_metadata": "notificationMetadata",
        "notification_target_arn": "notificationTargetArn",
        "role_arn": "roleArn",
    },
)
class CfnLifecycleHookProps:
    def __init__(
        self,
        *,
        auto_scaling_group_name: builtins.str,
        lifecycle_transition: builtins.str,
        default_result: typing.Optional[builtins.str] = None,
        heartbeat_timeout: typing.Optional[jsii.Number] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target_arn: typing.Optional[builtins.str] = None,
        role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnLifecycleHook``.

        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param lifecycle_transition: The lifecycle transition. For Auto Scaling groups, there are two major lifecycle transitions. - To create a lifecycle hook for scale-out events, specify ``autoscaling:EC2_INSTANCE_LAUNCHING`` . - To create a lifecycle hook for scale-in events, specify ``autoscaling:EC2_INSTANCE_TERMINATING`` .
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. The default value is ``ABANDON`` . Valid values: ``CONTINUE`` | ``ABANDON``
        :param heartbeat_timeout: The maximum time, in seconds, that can elapse before the lifecycle hook times out. The range is from ``30`` to ``7200`` seconds. The default value is ``3600`` seconds (1 hour).
        :param lifecycle_hook_name: The name of the lifecycle hook.
        :param notification_metadata: Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the notification target.
        :param notification_target_arn: The Amazon Resource Name (ARN) of the notification target that Amazon EC2 Auto Scaling sends notifications to when an instance is in a wait state for the lifecycle hook. You can specify an Amazon SNS topic or an Amazon SQS queue.
        :param role_arn: The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target. For information about creating this role, see `Configure a notification target for a lifecycle hook <https://docs.aws.amazon.com/autoscaling/ec2/userguide/prepare-for-lifecycle-notifications.html#lifecycle-hook-notification-target>`_ in the *Amazon EC2 Auto Scaling User Guide* . Valid only if the notification target is an Amazon SNS topic or an Amazon SQS queue.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            cfn_lifecycle_hook_props = autoscaling.CfnLifecycleHookProps(
                auto_scaling_group_name="autoScalingGroupName",
                lifecycle_transition="lifecycleTransition",
            
                # the properties below are optional
                default_result="defaultResult",
                heartbeat_timeout=123,
                lifecycle_hook_name="lifecycleHookName",
                notification_metadata="notificationMetadata",
                notification_target_arn="notificationTargetArn",
                role_arn="roleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55ab7935babae2422463a7df532373822bd3f50d204a6ecf63bb4ac285894a64)
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument lifecycle_transition", value=lifecycle_transition, expected_type=type_hints["lifecycle_transition"])
            check_type(argname="argument default_result", value=default_result, expected_type=type_hints["default_result"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument lifecycle_hook_name", value=lifecycle_hook_name, expected_type=type_hints["lifecycle_hook_name"])
            check_type(argname="argument notification_metadata", value=notification_metadata, expected_type=type_hints["notification_metadata"])
            check_type(argname="argument notification_target_arn", value=notification_target_arn, expected_type=type_hints["notification_target_arn"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auto_scaling_group_name": auto_scaling_group_name,
            "lifecycle_transition": lifecycle_transition,
        }
        if default_result is not None:
            self._values["default_result"] = default_result
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if lifecycle_hook_name is not None:
            self._values["lifecycle_hook_name"] = lifecycle_hook_name
        if notification_metadata is not None:
            self._values["notification_metadata"] = notification_metadata
        if notification_target_arn is not None:
            self._values["notification_target_arn"] = notification_target_arn
        if role_arn is not None:
            self._values["role_arn"] = role_arn

    @builtins.property
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-autoscalinggroupname
        '''
        result = self._values.get("auto_scaling_group_name")
        assert result is not None, "Required property 'auto_scaling_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def lifecycle_transition(self) -> builtins.str:
        '''The lifecycle transition. For Auto Scaling groups, there are two major lifecycle transitions.

        - To create a lifecycle hook for scale-out events, specify ``autoscaling:EC2_INSTANCE_LAUNCHING`` .
        - To create a lifecycle hook for scale-in events, specify ``autoscaling:EC2_INSTANCE_TERMINATING`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-lifecycletransition
        '''
        result = self._values.get("lifecycle_transition")
        assert result is not None, "Required property 'lifecycle_transition' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def default_result(self) -> typing.Optional[builtins.str]:
        '''The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs.

        The default value is ``ABANDON`` .

        Valid values: ``CONTINUE`` | ``ABANDON``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-defaultresult
        '''
        result = self._values.get("default_result")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[jsii.Number]:
        '''The maximum time, in seconds, that can elapse before the lifecycle hook times out.

        The range is from ``30`` to ``7200`` seconds. The default value is ``3600`` seconds (1 hour).

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-heartbeattimeout
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def lifecycle_hook_name(self) -> typing.Optional[builtins.str]:
        '''The name of the lifecycle hook.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-lifecyclehookname
        '''
        result = self._values.get("lifecycle_hook_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_metadata(self) -> typing.Optional[builtins.str]:
        '''Additional information that you want to include any time Amazon EC2 Auto Scaling sends a message to the notification target.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-notificationmetadata
        '''
        result = self._values.get("notification_metadata")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_target_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the notification target that Amazon EC2 Auto Scaling sends notifications to when an instance is in a wait state for the lifecycle hook.

        You can specify an Amazon SNS topic or an Amazon SQS queue.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-notificationtargetarn
        '''
        result = self._values.get("notification_target_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the IAM role that allows the Auto Scaling group to publish to the specified notification target.

        For information about creating this role, see `Configure a notification target for a lifecycle hook <https://docs.aws.amazon.com/autoscaling/ec2/userguide/prepare-for-lifecycle-notifications.html#lifecycle-hook-notification-target>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Valid only if the notification target is an Amazon SNS topic or an Amazon SQS queue.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-lifecyclehook.html#cfn-autoscaling-lifecyclehook-rolearn
        '''
        result = self._values.get("role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLifecycleHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnScalingPolicy(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy",
):
    '''A CloudFormation ``AWS::AutoScaling::ScalingPolicy``.

    The ``AWS::AutoScaling::ScalingPolicy`` resource specifies an Amazon EC2 Auto Scaling scaling policy so that the Auto Scaling group can scale the number of instances available for your application.

    For more information about using scaling policies to scale your Auto Scaling group automatically, see `Dynamic scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scale-based-on-demand.html>`_ and `Predictive scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-predictive-scaling.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

    :cloudformationResource: AWS::AutoScaling::ScalingPolicy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        cfn_scaling_policy = autoscaling.CfnScalingPolicy(self, "MyCfnScalingPolicy",
            auto_scaling_group_name="autoScalingGroupName",
        
            # the properties below are optional
            adjustment_type="adjustmentType",
            cooldown="cooldown",
            estimated_instance_warmup=123,
            metric_aggregation_type="metricAggregationType",
            min_adjustment_magnitude=123,
            policy_type="policyType",
            predictive_scaling_configuration=autoscaling.CfnScalingPolicy.PredictiveScalingConfigurationProperty(
                metric_specifications=[autoscaling.CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty(
                    target_value=123,
        
                    # the properties below are optional
                    customized_capacity_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty(
                        metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                            id="id",
        
                            # the properties below are optional
                            expression="expression",
                            label="label",
                            metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                    metric_name="metricName",
                                    namespace="namespace",
        
                                    # the properties below are optional
                                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                        name="name",
                                        value="value"
                                    )]
                                ),
                                stat="stat",
        
                                # the properties below are optional
                                unit="unit"
                            ),
                            return_data=False
                        )]
                    ),
                    customized_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty(
                        metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                            id="id",
        
                            # the properties below are optional
                            expression="expression",
                            label="label",
                            metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                    metric_name="metricName",
                                    namespace="namespace",
        
                                    # the properties below are optional
                                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                        name="name",
                                        value="value"
                                    )]
                                ),
                                stat="stat",
        
                                # the properties below are optional
                                unit="unit"
                            ),
                            return_data=False
                        )]
                    ),
                    customized_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty(
                        metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                            id="id",
        
                            # the properties below are optional
                            expression="expression",
                            label="label",
                            metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                    metric_name="metricName",
                                    namespace="namespace",
        
                                    # the properties below are optional
                                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                        name="name",
                                        value="value"
                                    )]
                                ),
                                stat="stat",
        
                                # the properties below are optional
                                unit="unit"
                            ),
                            return_data=False
                        )]
                    ),
                    predefined_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty(
                        predefined_metric_type="predefinedMetricType",
        
                        # the properties below are optional
                        resource_label="resourceLabel"
                    ),
                    predefined_metric_pair_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty(
                        predefined_metric_type="predefinedMetricType",
        
                        # the properties below are optional
                        resource_label="resourceLabel"
                    ),
                    predefined_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty(
                        predefined_metric_type="predefinedMetricType",
        
                        # the properties below are optional
                        resource_label="resourceLabel"
                    )
                )],
        
                # the properties below are optional
                max_capacity_breach_behavior="maxCapacityBreachBehavior",
                max_capacity_buffer=123,
                mode="mode",
                scheduling_buffer_time=123
            ),
            scaling_adjustment=123,
            step_adjustments=[autoscaling.CfnScalingPolicy.StepAdjustmentProperty(
                scaling_adjustment=123,
        
                # the properties below are optional
                metric_interval_lower_bound=123,
                metric_interval_upper_bound=123
            )],
            target_tracking_configuration=autoscaling.CfnScalingPolicy.TargetTrackingConfigurationProperty(
                target_value=123,
        
                # the properties below are optional
                customized_metric_specification=autoscaling.CfnScalingPolicy.CustomizedMetricSpecificationProperty(
                    metric_name="metricName",
                    namespace="namespace",
                    statistic="statistic",
        
                    # the properties below are optional
                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                        name="name",
                        value="value"
                    )],
                    unit="unit"
                ),
                disable_scale_in=False,
                predefined_metric_specification=autoscaling.CfnScalingPolicy.PredefinedMetricSpecificationProperty(
                    predefined_metric_type="predefinedMetricType",
        
                    # the properties below are optional
                    resource_label="resourceLabel"
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group_name: builtins.str,
        adjustment_type: typing.Optional[builtins.str] = None,
        cooldown: typing.Optional[builtins.str] = None,
        estimated_instance_warmup: typing.Optional[jsii.Number] = None,
        metric_aggregation_type: typing.Optional[builtins.str] = None,
        min_adjustment_magnitude: typing.Optional[jsii.Number] = None,
        policy_type: typing.Optional[builtins.str] = None,
        predictive_scaling_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        scaling_adjustment: typing.Optional[jsii.Number] = None,
        step_adjustments: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.StepAdjustmentProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        target_tracking_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.TargetTrackingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::AutoScaling::ScalingPolicy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param adjustment_type: Specifies how the scaling adjustment is interpreted (for example, an absolute number or a percentage). The valid values are ``ChangeInCapacity`` , ``ExactCapacity`` , and ``PercentChangeInCapacity`` . Required if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param cooldown: A cooldown period, in seconds, that applies to a specific simple scaling policy. When a cooldown period is specified here, it overrides the default cooldown. Valid only if the policy type is ``SimpleScaling`` . For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Default: None
        :param estimated_instance_warmup: *Not needed if the default instance warmup is defined for the group.*. The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics. This warm-up period applies to instances launched due to a specific target tracking or step scaling policy. When a warm-up period is specified here, it overrides the default instance warmup. Valid only if the policy type is ``TargetTrackingScaling`` or ``StepScaling`` . .. epigraph:: The default is to use the value for the default instance warmup defined for the group. If default instance warmup is null, then ``EstimatedInstanceWarmup`` falls back to the value of default cooldown.
        :param metric_aggregation_type: The aggregation type for the CloudWatch metrics. The valid values are ``Minimum`` , ``Maximum`` , and ``Average`` . If the aggregation type is null, the value is treated as ``Average`` . Valid only if the policy type is ``StepScaling`` .
        :param min_adjustment_magnitude: The minimum value to scale by when the adjustment type is ``PercentChangeInCapacity`` . For example, suppose that you create a step scaling policy to scale out an Auto Scaling group by 25 percent and you specify a ``MinAdjustmentMagnitude`` of 2. If the group has 4 instances and the scaling policy is performed, 25 percent of 4 is 1. However, because you specified a ``MinAdjustmentMagnitude`` of 2, Amazon EC2 Auto Scaling scales out the group by 2 instances. Valid only if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* . .. epigraph:: Some Auto Scaling groups use instance weights. In this case, set the ``MinAdjustmentMagnitude`` to a value that is at least as large as your largest instance weight.
        :param policy_type: One of the following policy types:. - ``TargetTrackingScaling`` - ``StepScaling`` - ``SimpleScaling`` (default) - ``PredictiveScaling``
        :param predictive_scaling_configuration: A predictive scaling policy. Provides support for predefined and custom metrics. Predefined metrics include CPU utilization, network in/out, and the Application Load Balancer request count. Required if the policy type is ``PredictiveScaling`` .
        :param scaling_adjustment: The amount by which to scale, based on the specified adjustment type. A positive value adds to the current capacity while a negative number removes from the current capacity. For exact capacity, you must specify a positive value. Required if the policy type is ``SimpleScaling`` . (Not used with any other policy type.)
        :param step_adjustments: A set of adjustments that enable you to scale based on the size of the alarm breach. Required if the policy type is ``StepScaling`` . (Not used with any other policy type.)
        :param target_tracking_configuration: A target tracking scaling policy. Provides support for predefined or custom metrics. The following predefined metrics are available: - ``ASGAverageCPUUtilization`` - ``ASGAverageNetworkIn`` - ``ASGAverageNetworkOut`` - ``ALBRequestCountPerTarget`` If you specify ``ALBRequestCountPerTarget`` for the metric, you must specify the ``ResourceLabel`` property with the ``PredefinedMetricSpecification`` . Required if the policy type is ``TargetTrackingScaling`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe327b42a12b9ac1620a18cf03d5ab48d4b5a074b39f9d0b743895a53269b77a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnScalingPolicyProps(
            auto_scaling_group_name=auto_scaling_group_name,
            adjustment_type=adjustment_type,
            cooldown=cooldown,
            estimated_instance_warmup=estimated_instance_warmup,
            metric_aggregation_type=metric_aggregation_type,
            min_adjustment_magnitude=min_adjustment_magnitude,
            policy_type=policy_type,
            predictive_scaling_configuration=predictive_scaling_configuration,
            scaling_adjustment=scaling_adjustment,
            step_adjustments=step_adjustments,
            target_tracking_configuration=target_tracking_configuration,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca9407810c7f1b646f3d1a65c71021ff839d9aaa29cf9dbf8c38fc97672a50e6)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8950f8c8dab087dbd134a08d3f741ac3dc70751ad06a5c9ca2513ec3f434bb5)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrPolicyName")
    def attr_policy_name(self) -> builtins.str:
        '''Returns the name of a scaling policy.

        :cloudformationAttribute: PolicyName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPolicyName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-autoscalinggroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "autoScalingGroupName"))

    @auto_scaling_group_name.setter
    def auto_scaling_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ba0241db3897f1e174d111c17685cbc6fa63a583d3e01b9c33cebdd8f7c60b5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoScalingGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="adjustmentType")
    def adjustment_type(self) -> typing.Optional[builtins.str]:
        '''Specifies how the scaling adjustment is interpreted (for example, an absolute number or a percentage).

        The valid values are ``ChangeInCapacity`` , ``ExactCapacity`` , and ``PercentChangeInCapacity`` .

        Required if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-adjustmenttype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "adjustmentType"))

    @adjustment_type.setter
    def adjustment_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad1e7151ee36a9824de02370c80ec69093a3e830dda8ec76b0ce9cd896ae8b93)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "adjustmentType", value)

    @builtins.property
    @jsii.member(jsii_name="cooldown")
    def cooldown(self) -> typing.Optional[builtins.str]:
        '''A cooldown period, in seconds, that applies to a specific simple scaling policy.

        When a cooldown period is specified here, it overrides the default cooldown.

        Valid only if the policy type is ``SimpleScaling`` . For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Default: None

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-cooldown
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cooldown"))

    @cooldown.setter
    def cooldown(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b447669ce1fb5edee7c3423b0be21b0a3196ef892a3cec7a6362af7d003b988b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cooldown", value)

    @builtins.property
    @jsii.member(jsii_name="estimatedInstanceWarmup")
    def estimated_instance_warmup(self) -> typing.Optional[jsii.Number]:
        '''*Not needed if the default instance warmup is defined for the group.*.

        The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics. This warm-up period applies to instances launched due to a specific target tracking or step scaling policy. When a warm-up period is specified here, it overrides the default instance warmup.

        Valid only if the policy type is ``TargetTrackingScaling`` or ``StepScaling`` .
        .. epigraph::

           The default is to use the value for the default instance warmup defined for the group. If default instance warmup is null, then ``EstimatedInstanceWarmup`` falls back to the value of default cooldown.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-estimatedinstancewarmup
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "estimatedInstanceWarmup"))

    @estimated_instance_warmup.setter
    def estimated_instance_warmup(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c66ff62cee4c0c8f2364bf5bcc5137f8f451284c1fe9b421835285a3a6603379)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "estimatedInstanceWarmup", value)

    @builtins.property
    @jsii.member(jsii_name="metricAggregationType")
    def metric_aggregation_type(self) -> typing.Optional[builtins.str]:
        '''The aggregation type for the CloudWatch metrics.

        The valid values are ``Minimum`` , ``Maximum`` , and ``Average`` . If the aggregation type is null, the value is treated as ``Average`` .

        Valid only if the policy type is ``StepScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-metricaggregationtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "metricAggregationType"))

    @metric_aggregation_type.setter
    def metric_aggregation_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__babf5fc76d4b2a32dff5b20eab1cf2a2e4d55f18c07dfbc24b23dc49e9c0b7ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "metricAggregationType", value)

    @builtins.property
    @jsii.member(jsii_name="minAdjustmentMagnitude")
    def min_adjustment_magnitude(self) -> typing.Optional[jsii.Number]:
        '''The minimum value to scale by when the adjustment type is ``PercentChangeInCapacity`` .

        For example, suppose that you create a step scaling policy to scale out an Auto Scaling group by 25 percent and you specify a ``MinAdjustmentMagnitude`` of 2. If the group has 4 instances and the scaling policy is performed, 25 percent of 4 is 1. However, because you specified a ``MinAdjustmentMagnitude`` of 2, Amazon EC2 Auto Scaling scales out the group by 2 instances.

        Valid only if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        .. epigraph::

           Some Auto Scaling groups use instance weights. In this case, set the ``MinAdjustmentMagnitude`` to a value that is at least as large as your largest instance weight.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-minadjustmentmagnitude
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minAdjustmentMagnitude"))

    @min_adjustment_magnitude.setter
    def min_adjustment_magnitude(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9531ed0d6903bdfa76634c1cdd14be2c72654d1f10e1dbf10b9b5391cb75fc98)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minAdjustmentMagnitude", value)

    @builtins.property
    @jsii.member(jsii_name="policyType")
    def policy_type(self) -> typing.Optional[builtins.str]:
        '''One of the following policy types:.

        - ``TargetTrackingScaling``
        - ``StepScaling``
        - ``SimpleScaling`` (default)
        - ``PredictiveScaling``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-policytype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "policyType"))

    @policy_type.setter
    def policy_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bd4329445b62cbb417ca024062e90be3235f901236445dcfca9b9682478abb33)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "policyType", value)

    @builtins.property
    @jsii.member(jsii_name="predictiveScalingConfiguration")
    def predictive_scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingConfigurationProperty"]]:
        '''A predictive scaling policy. Provides support for predefined and custom metrics.

        Predefined metrics include CPU utilization, network in/out, and the Application Load Balancer request count.

        Required if the policy type is ``PredictiveScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingConfigurationProperty"]], jsii.get(self, "predictiveScalingConfiguration"))

    @predictive_scaling_configuration.setter
    def predictive_scaling_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76cc30a2e0064b6d652232bc49d285ce28467a34c2bd8ec66a130a61970781c6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "predictiveScalingConfiguration", value)

    @builtins.property
    @jsii.member(jsii_name="scalingAdjustment")
    def scaling_adjustment(self) -> typing.Optional[jsii.Number]:
        '''The amount by which to scale, based on the specified adjustment type.

        A positive value adds to the current capacity while a negative number removes from the current capacity. For exact capacity, you must specify a positive value.

        Required if the policy type is ``SimpleScaling`` . (Not used with any other policy type.)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-scalingadjustment
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "scalingAdjustment"))

    @scaling_adjustment.setter
    def scaling_adjustment(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e99a67829ea1191e8cb588be191baca3b10d1c95451c6bab05472cfb30827b1d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scalingAdjustment", value)

    @builtins.property
    @jsii.member(jsii_name="stepAdjustments")
    def step_adjustments(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.StepAdjustmentProperty"]]]]:
        '''A set of adjustments that enable you to scale based on the size of the alarm breach.

        Required if the policy type is ``StepScaling`` . (Not used with any other policy type.)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-stepadjustments
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.StepAdjustmentProperty"]]]], jsii.get(self, "stepAdjustments"))

    @step_adjustments.setter
    def step_adjustments(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.StepAdjustmentProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38dfdb6c2b6aee9f0b1cb3e26e4e9f1c58dab8c3cf6013c4ca258d3702647ac8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stepAdjustments", value)

    @builtins.property
    @jsii.member(jsii_name="targetTrackingConfiguration")
    def target_tracking_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.TargetTrackingConfigurationProperty"]]:
        '''A target tracking scaling policy. Provides support for predefined or custom metrics.

        The following predefined metrics are available:

        - ``ASGAverageCPUUtilization``
        - ``ASGAverageNetworkIn``
        - ``ASGAverageNetworkOut``
        - ``ALBRequestCountPerTarget``

        If you specify ``ALBRequestCountPerTarget`` for the metric, you must specify the ``ResourceLabel`` property with the ``PredefinedMetricSpecification`` .

        Required if the policy type is ``TargetTrackingScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-targettrackingconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.TargetTrackingConfigurationProperty"]], jsii.get(self, "targetTrackingConfiguration"))

    @target_tracking_configuration.setter
    def target_tracking_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.TargetTrackingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7edad0553749ec58702ad72a2bc68104268b76221c11cb19d8dd477ac531d53)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetTrackingConfiguration", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.CustomizedMetricSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "metric_name": "metricName",
            "namespace": "namespace",
            "statistic": "statistic",
            "dimensions": "dimensions",
            "unit": "unit",
        },
    )
    class CustomizedMetricSpecificationProperty:
        def __init__(
            self,
            *,
            metric_name: builtins.str,
            namespace: builtins.str,
            statistic: builtins.str,
            dimensions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricDimensionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            unit: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains customized metric specification information for a target tracking scaling policy for Amazon EC2 Auto Scaling.

            To create your customized metric specification:

            - Add values for each required property from CloudWatch. You can use an existing metric, or a new metric that you create. To use your own metric, you must first publish the metric to CloudWatch. For more information, see `Publish Custom Metrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html>`_ in the *Amazon CloudWatch User Guide* .
            - Choose a metric that changes proportionally with capacity. The value of the metric should increase or decrease in inverse proportion to the number of capacity units. That is, the value of the metric should decrease when capacity increases.

            For more information about CloudWatch, see `Amazon CloudWatch Concepts <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html>`_ .

            ``CustomizedMetricSpecification`` is a property of the `AWS::AutoScaling::ScalingPolicy TargetTrackingConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html>`_ property type.

            :param metric_name: The name of the metric. To get the exact metric name, namespace, and dimensions, inspect the `Metric <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html>`_ object that is returned by a call to `ListMetrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html>`_ .
            :param namespace: The namespace of the metric.
            :param statistic: The statistic of the metric.
            :param dimensions: The dimensions of the metric. Conditional: If you published your metric with dimensions, you must specify the same dimensions in your scaling policy.
            :param unit: The unit of the metric. For a complete list of the units that CloudWatch supports, see the `MetricDatum <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html>`_ data type in the *Amazon CloudWatch API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                customized_metric_specification_property = autoscaling.CfnScalingPolicy.CustomizedMetricSpecificationProperty(
                    metric_name="metricName",
                    namespace="namespace",
                    statistic="statistic",
                
                    # the properties below are optional
                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                        name="name",
                        value="value"
                    )],
                    unit="unit"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8e7031340719d237d3182678de0dc2931e386340d37c2dd60e36be75ada41bb3)
                check_type(argname="argument metric_name", value=metric_name, expected_type=type_hints["metric_name"])
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
                check_type(argname="argument statistic", value=statistic, expected_type=type_hints["statistic"])
                check_type(argname="argument dimensions", value=dimensions, expected_type=type_hints["dimensions"])
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric_name": metric_name,
                "namespace": namespace,
                "statistic": statistic,
            }
            if dimensions is not None:
                self._values["dimensions"] = dimensions
            if unit is not None:
                self._values["unit"] = unit

        @builtins.property
        def metric_name(self) -> builtins.str:
            '''The name of the metric.

            To get the exact metric name, namespace, and dimensions, inspect the `Metric <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html>`_ object that is returned by a call to `ListMetrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html#cfn-autoscaling-scalingpolicy-customizedmetricspecification-metricname
            '''
            result = self._values.get("metric_name")
            assert result is not None, "Required property 'metric_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def namespace(self) -> builtins.str:
            '''The namespace of the metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html#cfn-autoscaling-scalingpolicy-customizedmetricspecification-namespace
            '''
            result = self._values.get("namespace")
            assert result is not None, "Required property 'namespace' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def statistic(self) -> builtins.str:
            '''The statistic of the metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html#cfn-autoscaling-scalingpolicy-customizedmetricspecification-statistic
            '''
            result = self._values.get("statistic")
            assert result is not None, "Required property 'statistic' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def dimensions(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDimensionProperty"]]]]:
            '''The dimensions of the metric.

            Conditional: If you published your metric with dimensions, you must specify the same dimensions in your scaling policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html#cfn-autoscaling-scalingpolicy-customizedmetricspecification-dimensions
            '''
            result = self._values.get("dimensions")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDimensionProperty"]]]], result)

        @builtins.property
        def unit(self) -> typing.Optional[builtins.str]:
            '''The unit of the metric.

            For a complete list of the units that CloudWatch supports, see the `MetricDatum <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html>`_ data type in the *Amazon CloudWatch API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html#cfn-autoscaling-scalingpolicy-customizedmetricspecification-unit
            '''
            result = self._values.get("unit")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomizedMetricSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.MetricDataQueryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "id": "id",
            "expression": "expression",
            "label": "label",
            "metric_stat": "metricStat",
            "return_data": "returnData",
        },
    )
    class MetricDataQueryProperty:
        def __init__(
            self,
            *,
            id: builtins.str,
            expression: typing.Optional[builtins.str] = None,
            label: typing.Optional[builtins.str] = None,
            metric_stat: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricStatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            return_data: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The metric data to return.

            Also defines whether this call is returning data for one metric only, or whether it is performing a math expression on the values of returned metric statistics to create a new time series. A time series is a series of data points, each of which is associated with a timestamp.

            ``MetricDataQuery`` is a property of the following property types:

            - `AWS::AutoScaling::ScalingPolicy PredictiveScalingCustomizedScalingMetric <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedscalingmetric.html>`_
            - `AWS::AutoScaling::ScalingPolicy PredictiveScalingCustomizedLoadMetric <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedloadmetric.html>`_
            - `AWS::AutoScaling::ScalingPolicy PredictiveScalingCustomizedCapacityMetric <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedcapacitymetric.html>`_

            Predictive scaling uses the time series data received from CloudWatch to understand how to schedule capacity based on your historical workload patterns.

            You can call for a single metric or perform math expressions on multiple metrics. Any expressions used in a metric specification must eventually return a single time series.

            For more information and examples, see `Advanced predictive scaling policy configurations using custom metrics <https://docs.aws.amazon.com/autoscaling/ec2/userguide/predictive-scaling-customized-metric-specification.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param id: A short name that identifies the object's results in the response. This name must be unique among all ``MetricDataQuery`` objects specified for a single scaling policy. If you are performing math expressions on this set of data, this name represents that data and can serve as a variable in the mathematical expression. The valid characters are letters, numbers, and underscores. The first character must be a lowercase letter.
            :param expression: The math expression to perform on the returned data, if this object is performing a math expression. This expression can use the ``Id`` of the other metrics to refer to those metrics, and can also use the ``Id`` of other expressions to use the result of those expressions. Conditional: Within each ``MetricDataQuery`` object, you must specify either ``Expression`` or ``MetricStat`` , but not both.
            :param label: A human-readable label for this metric or expression. This is especially useful if this is a math expression, so that you know what the value represents.
            :param metric_stat: Information about the metric data to return. Conditional: Within each ``MetricDataQuery`` object, you must specify either ``Expression`` or ``MetricStat`` , but not both.
            :param return_data: Indicates whether to return the timestamps and raw data values of this metric. If you use any math expressions, specify ``true`` for this value for only the final math expression that the metric specification is based on. You must specify ``false`` for ``ReturnData`` for all the other metrics and expressions used in the metric specification. If you are only retrieving metrics and not performing any math expressions, do not specify anything for ``ReturnData`` . This sets it to its default ( ``true`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                metric_data_query_property = autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                    id="id",
                
                    # the properties below are optional
                    expression="expression",
                    label="label",
                    metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                        metric=autoscaling.CfnScalingPolicy.MetricProperty(
                            metric_name="metricName",
                            namespace="namespace",
                
                            # the properties below are optional
                            dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                name="name",
                                value="value"
                            )]
                        ),
                        stat="stat",
                
                        # the properties below are optional
                        unit="unit"
                    ),
                    return_data=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9e93e050292033868470978d9a82575b6db041f741e699dbbabdfe6b0c0915d0)
                check_type(argname="argument id", value=id, expected_type=type_hints["id"])
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
                check_type(argname="argument label", value=label, expected_type=type_hints["label"])
                check_type(argname="argument metric_stat", value=metric_stat, expected_type=type_hints["metric_stat"])
                check_type(argname="argument return_data", value=return_data, expected_type=type_hints["return_data"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "id": id,
            }
            if expression is not None:
                self._values["expression"] = expression
            if label is not None:
                self._values["label"] = label
            if metric_stat is not None:
                self._values["metric_stat"] = metric_stat
            if return_data is not None:
                self._values["return_data"] = return_data

        @builtins.property
        def id(self) -> builtins.str:
            '''A short name that identifies the object's results in the response.

            This name must be unique among all ``MetricDataQuery`` objects specified for a single scaling policy. If you are performing math expressions on this set of data, this name represents that data and can serve as a variable in the mathematical expression. The valid characters are letters, numbers, and underscores. The first character must be a lowercase letter.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html#cfn-autoscaling-scalingpolicy-metricdataquery-id
            '''
            result = self._values.get("id")
            assert result is not None, "Required property 'id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''The math expression to perform on the returned data, if this object is performing a math expression.

            This expression can use the ``Id`` of the other metrics to refer to those metrics, and can also use the ``Id`` of other expressions to use the result of those expressions.

            Conditional: Within each ``MetricDataQuery`` object, you must specify either ``Expression`` or ``MetricStat`` , but not both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html#cfn-autoscaling-scalingpolicy-metricdataquery-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def label(self) -> typing.Optional[builtins.str]:
            '''A human-readable label for this metric or expression.

            This is especially useful if this is a math expression, so that you know what the value represents.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html#cfn-autoscaling-scalingpolicy-metricdataquery-label
            '''
            result = self._values.get("label")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def metric_stat(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricStatProperty"]]:
            '''Information about the metric data to return.

            Conditional: Within each ``MetricDataQuery`` object, you must specify either ``Expression`` or ``MetricStat`` , but not both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html#cfn-autoscaling-scalingpolicy-metricdataquery-metricstat
            '''
            result = self._values.get("metric_stat")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricStatProperty"]], result)

        @builtins.property
        def return_data(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether to return the timestamps and raw data values of this metric.

            If you use any math expressions, specify ``true`` for this value for only the final math expression that the metric specification is based on. You must specify ``false`` for ``ReturnData`` for all the other metrics and expressions used in the metric specification.

            If you are only retrieving metrics and not performing any math expressions, do not specify anything for ``ReturnData`` . This sets it to its default ( ``true`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html#cfn-autoscaling-scalingpolicy-metricdataquery-returndata
            '''
            result = self._values.get("return_data")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricDataQueryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.MetricDimensionProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class MetricDimensionProperty:
        def __init__(self, *, name: builtins.str, value: builtins.str) -> None:
            '''``MetricDimension`` specifies a name/value pair that is part of the identity of a CloudWatch metric for the ``Dimensions`` property of the `AWS::AutoScaling::ScalingPolicy CustomizedMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-customizedmetricspecification.html>`_ property type. Duplicate dimensions are not allowed.

            :param name: The name of the dimension.
            :param value: The value of the dimension.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdimension.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                metric_dimension_property = autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fe263644851b54bdc57fb10d446cca360e5b21a0a9cc52aadfab88b29cde7666)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "value": value,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the dimension.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdimension.html#cfn-autoscaling-scalingpolicy-metricdimension-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value of the dimension.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdimension.html#cfn-autoscaling-scalingpolicy-metricdimension-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricDimensionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.MetricProperty",
        jsii_struct_bases=[],
        name_mapping={
            "metric_name": "metricName",
            "namespace": "namespace",
            "dimensions": "dimensions",
        },
    )
    class MetricProperty:
        def __init__(
            self,
            *,
            metric_name: builtins.str,
            namespace: builtins.str,
            dimensions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricDimensionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Represents a specific metric.

            ``Metric`` is a property of the `AWS::AutoScaling::ScalingPolicy MetricStat <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricstat.html>`_ property type.

            :param metric_name: The name of the metric.
            :param namespace: The namespace of the metric. For more information, see the table in `AWS services that publish CloudWatch metrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html>`_ in the *Amazon CloudWatch User Guide* .
            :param dimensions: The dimensions for the metric. For the list of available dimensions, see the AWS documentation available from the table in `AWS services that publish CloudWatch metrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html>`_ in the *Amazon CloudWatch User Guide* . Conditional: If you published your metric with dimensions, you must specify the same dimensions in your scaling policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metric.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                metric_property = autoscaling.CfnScalingPolicy.MetricProperty(
                    metric_name="metricName",
                    namespace="namespace",
                
                    # the properties below are optional
                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                        name="name",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d8123e387ec06ce553c8d5d5c99d355f1458989604b38309f3039cb3d1b694cb)
                check_type(argname="argument metric_name", value=metric_name, expected_type=type_hints["metric_name"])
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
                check_type(argname="argument dimensions", value=dimensions, expected_type=type_hints["dimensions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric_name": metric_name,
                "namespace": namespace,
            }
            if dimensions is not None:
                self._values["dimensions"] = dimensions

        @builtins.property
        def metric_name(self) -> builtins.str:
            '''The name of the metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metric.html#cfn-autoscaling-scalingpolicy-metric-metricname
            '''
            result = self._values.get("metric_name")
            assert result is not None, "Required property 'metric_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def namespace(self) -> builtins.str:
            '''The namespace of the metric.

            For more information, see the table in `AWS services that publish CloudWatch metrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html>`_ in the *Amazon CloudWatch User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metric.html#cfn-autoscaling-scalingpolicy-metric-namespace
            '''
            result = self._values.get("namespace")
            assert result is not None, "Required property 'namespace' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def dimensions(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDimensionProperty"]]]]:
            '''The dimensions for the metric.

            For the list of available dimensions, see the AWS documentation available from the table in `AWS services that publish CloudWatch metrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/aws-services-cloudwatch-metrics.html>`_ in the *Amazon CloudWatch User Guide* .

            Conditional: If you published your metric with dimensions, you must specify the same dimensions in your scaling policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metric.html#cfn-autoscaling-scalingpolicy-metric-dimensions
            '''
            result = self._values.get("dimensions")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDimensionProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.MetricStatProperty",
        jsii_struct_bases=[],
        name_mapping={"metric": "metric", "stat": "stat", "unit": "unit"},
    )
    class MetricStatProperty:
        def __init__(
            self,
            *,
            metric: typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricProperty", typing.Dict[builtins.str, typing.Any]]],
            stat: builtins.str,
            unit: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``MetricStat`` is a property of the `AWS::AutoScaling::ScalingPolicy MetricDataQuery <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricdataquery.html>`_ property type.

            This structure defines the CloudWatch metric to return, along with the statistic, period, and unit.

            For more information about the CloudWatch terminology below, see `Amazon CloudWatch concepts <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html>`_ in the *Amazon CloudWatch User Guide* .

            :param metric: The CloudWatch metric to return, including the metric name, namespace, and dimensions. To get the exact metric name, namespace, and dimensions, inspect the `Metric <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html>`_ object that is returned by a call to `ListMetrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html>`_ .
            :param stat: The statistic to return. It can include any CloudWatch statistic or extended statistic. For a list of valid values, see the table in `Statistics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Statistic>`_ in the *Amazon CloudWatch User Guide* . The most commonly used metrics for predictive scaling are ``Average`` and ``Sum`` .
            :param unit: The unit to use for the returned data points. For a complete list of the units that CloudWatch supports, see the `MetricDatum <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html>`_ data type in the *Amazon CloudWatch API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricstat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                metric_stat_property = autoscaling.CfnScalingPolicy.MetricStatProperty(
                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                        metric_name="metricName",
                        namespace="namespace",
                
                        # the properties below are optional
                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                            name="name",
                            value="value"
                        )]
                    ),
                    stat="stat",
                
                    # the properties below are optional
                    unit="unit"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ef713283f2f2a6c305bbd7e03d73709deada07fe9d07055c2a3fee4205c3af39)
                check_type(argname="argument metric", value=metric, expected_type=type_hints["metric"])
                check_type(argname="argument stat", value=stat, expected_type=type_hints["stat"])
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric": metric,
                "stat": stat,
            }
            if unit is not None:
                self._values["unit"] = unit

        @builtins.property
        def metric(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricProperty"]:
            '''The CloudWatch metric to return, including the metric name, namespace, and dimensions.

            To get the exact metric name, namespace, and dimensions, inspect the `Metric <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_Metric.html>`_ object that is returned by a call to `ListMetrics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricstat.html#cfn-autoscaling-scalingpolicy-metricstat-metric
            '''
            result = self._values.get("metric")
            assert result is not None, "Required property 'metric' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricProperty"], result)

        @builtins.property
        def stat(self) -> builtins.str:
            '''The statistic to return.

            It can include any CloudWatch statistic or extended statistic. For a list of valid values, see the table in `Statistics <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Statistic>`_ in the *Amazon CloudWatch User Guide* .

            The most commonly used metrics for predictive scaling are ``Average`` and ``Sum`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricstat.html#cfn-autoscaling-scalingpolicy-metricstat-stat
            '''
            result = self._values.get("stat")
            assert result is not None, "Required property 'stat' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def unit(self) -> typing.Optional[builtins.str]:
            '''The unit to use for the returned data points.

            For a complete list of the units that CloudWatch supports, see the `MetricDatum <https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html>`_ data type in the *Amazon CloudWatch API Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-metricstat.html#cfn-autoscaling-scalingpolicy-metricstat-unit
            '''
            result = self._values.get("unit")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricStatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredefinedMetricSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "predefined_metric_type": "predefinedMetricType",
            "resource_label": "resourceLabel",
        },
    )
    class PredefinedMetricSpecificationProperty:
        def __init__(
            self,
            *,
            predefined_metric_type: builtins.str,
            resource_label: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains predefined metric specification information for a target tracking scaling policy for Amazon EC2 Auto Scaling.

            ``PredefinedMetricSpecification`` is a property of the `AWS::AutoScaling::ScalingPolicy TargetTrackingConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html>`_ property type.

            :param predefined_metric_type: The metric type. The following predefined metrics are available:. - ``ASGAverageCPUUtilization`` - Average CPU utilization of the Auto Scaling group. - ``ASGAverageNetworkIn`` - Average number of bytes received on all network interfaces by the Auto Scaling group. - ``ASGAverageNetworkOut`` - Average number of bytes sent out on all network interfaces by the Auto Scaling group. - ``ALBRequestCountPerTarget`` - Average Application Load Balancer request count per target for your Auto Scaling group.
            :param resource_label: A label that uniquely identifies a specific Application Load Balancer target group from which to determine the average request count served by your Auto Scaling group. You can't specify a resource label unless the target group is attached to the Auto Scaling group. You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is: ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` . Where: - app// is the final portion of the load balancer ARN - targetgroup// is the final portion of the target group ARN. To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predefinedmetricspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predefined_metric_specification_property = autoscaling.CfnScalingPolicy.PredefinedMetricSpecificationProperty(
                    predefined_metric_type="predefinedMetricType",
                
                    # the properties below are optional
                    resource_label="resourceLabel"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b547b62a61eb66aa5b42bed1a05a0a926d9bf446e9c904183bbbaa9cee54f56a)
                check_type(argname="argument predefined_metric_type", value=predefined_metric_type, expected_type=type_hints["predefined_metric_type"])
                check_type(argname="argument resource_label", value=resource_label, expected_type=type_hints["resource_label"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "predefined_metric_type": predefined_metric_type,
            }
            if resource_label is not None:
                self._values["resource_label"] = resource_label

        @builtins.property
        def predefined_metric_type(self) -> builtins.str:
            '''The metric type. The following predefined metrics are available:.

            - ``ASGAverageCPUUtilization`` - Average CPU utilization of the Auto Scaling group.
            - ``ASGAverageNetworkIn`` - Average number of bytes received on all network interfaces by the Auto Scaling group.
            - ``ASGAverageNetworkOut`` - Average number of bytes sent out on all network interfaces by the Auto Scaling group.
            - ``ALBRequestCountPerTarget`` - Average Application Load Balancer request count per target for your Auto Scaling group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predefinedmetricspecification.html#cfn-autoscaling-scalingpolicy-predefinedmetricspecification-predefinedmetrictype
            '''
            result = self._values.get("predefined_metric_type")
            assert result is not None, "Required property 'predefined_metric_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_label(self) -> typing.Optional[builtins.str]:
            '''A label that uniquely identifies a specific Application Load Balancer target group from which to determine the average request count served by your Auto Scaling group.

            You can't specify a resource label unless the target group is attached to the Auto Scaling group.

            You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is:

            ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` .

            Where:

            - app// is the final portion of the load balancer ARN
            - targetgroup// is the final portion of the target group ARN.

            To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predefinedmetricspecification.html#cfn-autoscaling-scalingpolicy-predefinedmetricspecification-resourcelabel
            '''
            result = self._values.get("resource_label")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredefinedMetricSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "metric_specifications": "metricSpecifications",
            "max_capacity_breach_behavior": "maxCapacityBreachBehavior",
            "max_capacity_buffer": "maxCapacityBuffer",
            "mode": "mode",
            "scheduling_buffer_time": "schedulingBufferTime",
        },
    )
    class PredictiveScalingConfigurationProperty:
        def __init__(
            self,
            *,
            metric_specifications: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]]],
            max_capacity_breach_behavior: typing.Optional[builtins.str] = None,
            max_capacity_buffer: typing.Optional[jsii.Number] = None,
            mode: typing.Optional[builtins.str] = None,
            scheduling_buffer_time: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``PredictiveScalingConfiguration`` is a property of the `AWS::AutoScaling::ScalingPolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-policy.html>`_ resource that specifies a predictive scaling policy for Amazon EC2 Auto Scaling.

            For more information, see `Predictive scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-predictive-scaling.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param metric_specifications: This structure includes the metrics and target utilization to use for predictive scaling. This is an array, but we currently only support a single metric specification. That is, you can specify a target value and a single metric pair, or a target value and one scaling metric and one load metric.
            :param max_capacity_breach_behavior: Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum capacity of the Auto Scaling group. Defaults to ``HonorMaxCapacity`` if not specified. The following are possible values: - ``HonorMaxCapacity`` - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum capacity. The maximum capacity is enforced as a hard limit. - ``IncreaseMaxCapacity`` - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is determined by the forecasted capacity and the value for ``MaxCapacityBuffer`` .
            :param max_capacity_buffer: The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum capacity. The value is specified as a percentage relative to the forecast capacity. For example, if the buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum capacity is 40, then the effective maximum capacity is 55. If set to 0, Amazon EC2 Auto Scaling may scale capacity higher than the maximum capacity to equal but not exceed forecast capacity. Required if the ``MaxCapacityBreachBehavior`` property is set to ``IncreaseMaxCapacity`` , and cannot be used otherwise.
            :param mode: The predictive scaling mode. Defaults to ``ForecastOnly`` if not specified.
            :param scheduling_buffer_time: The amount of time, in seconds, by which the instance launch time can be advanced. For example, the forecast says to add capacity at 10:00 AM, and you choose to pre-launch instances by 5 minutes. In that case, the instances will be launched at 9:55 AM. The intention is to give resources time to be provisioned. It can take a few minutes to launch an EC2 instance. The actual amount of time required depends on several factors, such as the size of the instance and whether there are startup scripts to complete. The value must be less than the forecast interval duration of 3600 seconds (60 minutes). Defaults to 300 seconds if not specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_configuration_property = autoscaling.CfnScalingPolicy.PredictiveScalingConfigurationProperty(
                    metric_specifications=[autoscaling.CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty(
                        target_value=123,
                
                        # the properties below are optional
                        customized_capacity_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty(
                            metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                                id="id",
                
                                # the properties below are optional
                                expression="expression",
                                label="label",
                                metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                        metric_name="metricName",
                                        namespace="namespace",
                
                                        # the properties below are optional
                                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                            name="name",
                                            value="value"
                                        )]
                                    ),
                                    stat="stat",
                
                                    # the properties below are optional
                                    unit="unit"
                                ),
                                return_data=False
                            )]
                        ),
                        customized_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty(
                            metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                                id="id",
                
                                # the properties below are optional
                                expression="expression",
                                label="label",
                                metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                        metric_name="metricName",
                                        namespace="namespace",
                
                                        # the properties below are optional
                                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                            name="name",
                                            value="value"
                                        )]
                                    ),
                                    stat="stat",
                
                                    # the properties below are optional
                                    unit="unit"
                                ),
                                return_data=False
                            )]
                        ),
                        customized_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty(
                            metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                                id="id",
                
                                # the properties below are optional
                                expression="expression",
                                label="label",
                                metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                        metric_name="metricName",
                                        namespace="namespace",
                
                                        # the properties below are optional
                                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                            name="name",
                                            value="value"
                                        )]
                                    ),
                                    stat="stat",
                
                                    # the properties below are optional
                                    unit="unit"
                                ),
                                return_data=False
                            )]
                        ),
                        predefined_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty(
                            predefined_metric_type="predefinedMetricType",
                
                            # the properties below are optional
                            resource_label="resourceLabel"
                        ),
                        predefined_metric_pair_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty(
                            predefined_metric_type="predefinedMetricType",
                
                            # the properties below are optional
                            resource_label="resourceLabel"
                        ),
                        predefined_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty(
                            predefined_metric_type="predefinedMetricType",
                
                            # the properties below are optional
                            resource_label="resourceLabel"
                        )
                    )],
                
                    # the properties below are optional
                    max_capacity_breach_behavior="maxCapacityBreachBehavior",
                    max_capacity_buffer=123,
                    mode="mode",
                    scheduling_buffer_time=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c835136b3008eece31e060ed2f94a66aed36a8978c41ac0fc35df358edc49040)
                check_type(argname="argument metric_specifications", value=metric_specifications, expected_type=type_hints["metric_specifications"])
                check_type(argname="argument max_capacity_breach_behavior", value=max_capacity_breach_behavior, expected_type=type_hints["max_capacity_breach_behavior"])
                check_type(argname="argument max_capacity_buffer", value=max_capacity_buffer, expected_type=type_hints["max_capacity_buffer"])
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
                check_type(argname="argument scheduling_buffer_time", value=scheduling_buffer_time, expected_type=type_hints["scheduling_buffer_time"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric_specifications": metric_specifications,
            }
            if max_capacity_breach_behavior is not None:
                self._values["max_capacity_breach_behavior"] = max_capacity_breach_behavior
            if max_capacity_buffer is not None:
                self._values["max_capacity_buffer"] = max_capacity_buffer
            if mode is not None:
                self._values["mode"] = mode
            if scheduling_buffer_time is not None:
                self._values["scheduling_buffer_time"] = scheduling_buffer_time

        @builtins.property
        def metric_specifications(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty"]]]:
            '''This structure includes the metrics and target utilization to use for predictive scaling.

            This is an array, but we currently only support a single metric specification. That is, you can specify a target value and a single metric pair, or a target value and one scaling metric and one load metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration-metricspecifications
            '''
            result = self._values.get("metric_specifications")
            assert result is not None, "Required property 'metric_specifications' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty"]]], result)

        @builtins.property
        def max_capacity_breach_behavior(self) -> typing.Optional[builtins.str]:
            '''Defines the behavior that should be applied if the forecast capacity approaches or exceeds the maximum capacity of the Auto Scaling group.

            Defaults to ``HonorMaxCapacity`` if not specified.

            The following are possible values:

            - ``HonorMaxCapacity`` - Amazon EC2 Auto Scaling cannot scale out capacity higher than the maximum capacity. The maximum capacity is enforced as a hard limit.
            - ``IncreaseMaxCapacity`` - Amazon EC2 Auto Scaling can scale out capacity higher than the maximum capacity when the forecast capacity is close to or exceeds the maximum capacity. The upper limit is determined by the forecasted capacity and the value for ``MaxCapacityBuffer`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration-maxcapacitybreachbehavior
            '''
            result = self._values.get("max_capacity_breach_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_capacity_buffer(self) -> typing.Optional[jsii.Number]:
            '''The size of the capacity buffer to use when the forecast capacity is close to or exceeds the maximum capacity.

            The value is specified as a percentage relative to the forecast capacity. For example, if the buffer is 10, this means a 10 percent buffer, such that if the forecast capacity is 50, and the maximum capacity is 40, then the effective maximum capacity is 55.

            If set to 0, Amazon EC2 Auto Scaling may scale capacity higher than the maximum capacity to equal but not exceed forecast capacity.

            Required if the ``MaxCapacityBreachBehavior`` property is set to ``IncreaseMaxCapacity`` , and cannot be used otherwise.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration-maxcapacitybuffer
            '''
            result = self._values.get("max_capacity_buffer")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def mode(self) -> typing.Optional[builtins.str]:
            '''The predictive scaling mode.

            Defaults to ``ForecastOnly`` if not specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration-mode
            '''
            result = self._values.get("mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def scheduling_buffer_time(self) -> typing.Optional[jsii.Number]:
            '''The amount of time, in seconds, by which the instance launch time can be advanced.

            For example, the forecast says to add capacity at 10:00 AM, and you choose to pre-launch instances by 5 minutes. In that case, the instances will be launched at 9:55 AM. The intention is to give resources time to be provisioned. It can take a few minutes to launch an EC2 instance. The actual amount of time required depends on several factors, such as the size of the instance and whether there are startup scripts to complete.

            The value must be less than the forecast interval duration of 3600 seconds (60 minutes). Defaults to 300 seconds if not specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration-schedulingbuffertime
            '''
            result = self._values.get("scheduling_buffer_time")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty",
        jsii_struct_bases=[],
        name_mapping={"metric_data_queries": "metricDataQueries"},
    )
    class PredictiveScalingCustomizedCapacityMetricProperty:
        def __init__(
            self,
            *,
            metric_data_queries: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricDataQueryProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Contains capacity metric information for the ``CustomizedCapacityMetricSpecification`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html>`_ property type.

            :param metric_data_queries: One or more metric data queries to provide the data points for a capacity metric. Use multiple metric data queries only if you are performing a math expression on returned data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedcapacitymetric.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_customized_capacity_metric_property = autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty(
                    metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                        id="id",
                
                        # the properties below are optional
                        expression="expression",
                        label="label",
                        metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                            metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                metric_name="metricName",
                                namespace="namespace",
                
                                # the properties below are optional
                                dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                    name="name",
                                    value="value"
                                )]
                            ),
                            stat="stat",
                
                            # the properties below are optional
                            unit="unit"
                        ),
                        return_data=False
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6b8c4cdce70a16ebd1a5c2fe31cc7225458c86f82990f0ffe13403efa5cb6d2d)
                check_type(argname="argument metric_data_queries", value=metric_data_queries, expected_type=type_hints["metric_data_queries"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric_data_queries": metric_data_queries,
            }

        @builtins.property
        def metric_data_queries(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDataQueryProperty"]]]:
            '''One or more metric data queries to provide the data points for a capacity metric.

            Use multiple metric data queries only if you are performing a math expression on returned data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedcapacitymetric.html#cfn-autoscaling-scalingpolicy-predictivescalingcustomizedcapacitymetric-metricdataqueries
            '''
            result = self._values.get("metric_data_queries")
            assert result is not None, "Required property 'metric_data_queries' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDataQueryProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingCustomizedCapacityMetricProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty",
        jsii_struct_bases=[],
        name_mapping={"metric_data_queries": "metricDataQueries"},
    )
    class PredictiveScalingCustomizedLoadMetricProperty:
        def __init__(
            self,
            *,
            metric_data_queries: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricDataQueryProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Contains load metric information for the ``CustomizedLoadMetricSpecification`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html>`_ property type.

            :param metric_data_queries: One or more metric data queries to provide the data points for a load metric. Use multiple metric data queries only if you are performing a math expression on returned data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedloadmetric.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_customized_load_metric_property = autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty(
                    metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                        id="id",
                
                        # the properties below are optional
                        expression="expression",
                        label="label",
                        metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                            metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                metric_name="metricName",
                                namespace="namespace",
                
                                # the properties below are optional
                                dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                    name="name",
                                    value="value"
                                )]
                            ),
                            stat="stat",
                
                            # the properties below are optional
                            unit="unit"
                        ),
                        return_data=False
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6f8df7fe95ed69a9fd3f50a28eda7de1e674e48f774cc9150d21b2a0fcba8125)
                check_type(argname="argument metric_data_queries", value=metric_data_queries, expected_type=type_hints["metric_data_queries"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric_data_queries": metric_data_queries,
            }

        @builtins.property
        def metric_data_queries(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDataQueryProperty"]]]:
            '''One or more metric data queries to provide the data points for a load metric.

            Use multiple metric data queries only if you are performing a math expression on returned data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedloadmetric.html#cfn-autoscaling-scalingpolicy-predictivescalingcustomizedloadmetric-metricdataqueries
            '''
            result = self._values.get("metric_data_queries")
            assert result is not None, "Required property 'metric_data_queries' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDataQueryProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingCustomizedLoadMetricProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty",
        jsii_struct_bases=[],
        name_mapping={"metric_data_queries": "metricDataQueries"},
    )
    class PredictiveScalingCustomizedScalingMetricProperty:
        def __init__(
            self,
            *,
            metric_data_queries: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.MetricDataQueryProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Contains scaling metric information for the ``CustomizedScalingMetricSpecification`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html>`_ property type.

            :param metric_data_queries: One or more metric data queries to provide the data points for a scaling metric. Use multiple metric data queries only if you are performing a math expression on returned data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedscalingmetric.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_customized_scaling_metric_property = autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty(
                    metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                        id="id",
                
                        # the properties below are optional
                        expression="expression",
                        label="label",
                        metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                            metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                metric_name="metricName",
                                namespace="namespace",
                
                                # the properties below are optional
                                dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                    name="name",
                                    value="value"
                                )]
                            ),
                            stat="stat",
                
                            # the properties below are optional
                            unit="unit"
                        ),
                        return_data=False
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__458a6ff3cadf721184c4e8bf120a87ac6a614c9bc06677c31d2f914a3d3bf080)
                check_type(argname="argument metric_data_queries", value=metric_data_queries, expected_type=type_hints["metric_data_queries"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "metric_data_queries": metric_data_queries,
            }

        @builtins.property
        def metric_data_queries(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDataQueryProperty"]]]:
            '''One or more metric data queries to provide the data points for a scaling metric.

            Use multiple metric data queries only if you are performing a math expression on returned data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingcustomizedscalingmetric.html#cfn-autoscaling-scalingpolicy-predictivescalingcustomizedscalingmetric-metricdataqueries
            '''
            result = self._values.get("metric_data_queries")
            assert result is not None, "Required property 'metric_data_queries' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.MetricDataQueryProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingCustomizedScalingMetricProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "target_value": "targetValue",
            "customized_capacity_metric_specification": "customizedCapacityMetricSpecification",
            "customized_load_metric_specification": "customizedLoadMetricSpecification",
            "customized_scaling_metric_specification": "customizedScalingMetricSpecification",
            "predefined_load_metric_specification": "predefinedLoadMetricSpecification",
            "predefined_metric_pair_specification": "predefinedMetricPairSpecification",
            "predefined_scaling_metric_specification": "predefinedScalingMetricSpecification",
        },
    )
    class PredictiveScalingMetricSpecificationProperty:
        def __init__(
            self,
            *,
            target_value: jsii.Number,
            customized_capacity_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            customized_load_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            customized_scaling_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            predefined_load_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            predefined_metric_pair_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            predefined_scaling_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A structure that specifies a metric specification for the ``MetricSpecifications`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingconfiguration.html>`_ property type.

            You must specify either a metric pair, or a load metric and a scaling metric individually. Specifying a metric pair instead of individual metrics provides a simpler way to configure metrics for a scaling policy. You choose the metric pair, and the policy automatically knows the correct sum and average statistics to use for the load metric and the scaling metric.

            Example

            - You create a predictive scaling policy and specify ``ALBRequestCount`` as the value for the metric pair and ``1000.0`` as the target value. For this type of metric, you must provide the metric dimension for the corresponding target group, so you also provide a resource label for the Application Load Balancer target group that is attached to your Auto Scaling group.
            - The number of requests the target group receives per minute provides the load metric, and the request count averaged between the members of the target group provides the scaling metric. In CloudWatch, this refers to the ``RequestCount`` and ``RequestCountPerTarget`` metrics, respectively.
            - For optimal use of predictive scaling, you adhere to the best practice of using a dynamic scaling policy to automatically scale between the minimum capacity and maximum capacity in response to real-time changes in resource utilization.
            - Amazon EC2 Auto Scaling consumes data points for the load metric over the last 14 days and creates an hourly load forecast for predictive scaling. (A minimum of 24 hours of data is required.)
            - After creating the load forecast, Amazon EC2 Auto Scaling determines when to reduce or increase the capacity of your Auto Scaling group in each hour of the forecast period so that the average number of requests received by each instance is as close to 1000 requests per minute as possible at all times.

            For information about using custom metrics with predictive scaling, see `Advanced predictive scaling policy configurations using custom metrics <https://docs.aws.amazon.com/autoscaling/ec2/userguide/predictive-scaling-customized-metric-specification.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param target_value: Specifies the target utilization. .. epigraph:: Some metrics are based on a count instead of a percentage, such as the request count for an Application Load Balancer or the number of messages in an SQS queue. If the scaling policy specifies one of these metrics, specify the target utilization as the optimal average request or message count per instance during any one-minute interval.
            :param customized_capacity_metric_specification: The customized capacity metric specification.
            :param customized_load_metric_specification: The customized load metric specification.
            :param customized_scaling_metric_specification: The customized scaling metric specification.
            :param predefined_load_metric_specification: The predefined load metric specification.
            :param predefined_metric_pair_specification: The predefined metric pair specification from which Amazon EC2 Auto Scaling determines the appropriate scaling metric and load metric to use.
            :param predefined_scaling_metric_specification: The predefined scaling metric specification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_metric_specification_property = autoscaling.CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty(
                    target_value=123,
                
                    # the properties below are optional
                    customized_capacity_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty(
                        metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                            id="id",
                
                            # the properties below are optional
                            expression="expression",
                            label="label",
                            metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                    metric_name="metricName",
                                    namespace="namespace",
                
                                    # the properties below are optional
                                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                        name="name",
                                        value="value"
                                    )]
                                ),
                                stat="stat",
                
                                # the properties below are optional
                                unit="unit"
                            ),
                            return_data=False
                        )]
                    ),
                    customized_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty(
                        metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                            id="id",
                
                            # the properties below are optional
                            expression="expression",
                            label="label",
                            metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                    metric_name="metricName",
                                    namespace="namespace",
                
                                    # the properties below are optional
                                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                        name="name",
                                        value="value"
                                    )]
                                ),
                                stat="stat",
                
                                # the properties below are optional
                                unit="unit"
                            ),
                            return_data=False
                        )]
                    ),
                    customized_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty(
                        metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                            id="id",
                
                            # the properties below are optional
                            expression="expression",
                            label="label",
                            metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                    metric_name="metricName",
                                    namespace="namespace",
                
                                    # the properties below are optional
                                    dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                        name="name",
                                        value="value"
                                    )]
                                ),
                                stat="stat",
                
                                # the properties below are optional
                                unit="unit"
                            ),
                            return_data=False
                        )]
                    ),
                    predefined_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty(
                        predefined_metric_type="predefinedMetricType",
                
                        # the properties below are optional
                        resource_label="resourceLabel"
                    ),
                    predefined_metric_pair_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty(
                        predefined_metric_type="predefinedMetricType",
                
                        # the properties below are optional
                        resource_label="resourceLabel"
                    ),
                    predefined_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty(
                        predefined_metric_type="predefinedMetricType",
                
                        # the properties below are optional
                        resource_label="resourceLabel"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__012dc12dc1d9459ee13271aaed0c024f3ab9f083799f0ba98220ebc8754e7f33)
                check_type(argname="argument target_value", value=target_value, expected_type=type_hints["target_value"])
                check_type(argname="argument customized_capacity_metric_specification", value=customized_capacity_metric_specification, expected_type=type_hints["customized_capacity_metric_specification"])
                check_type(argname="argument customized_load_metric_specification", value=customized_load_metric_specification, expected_type=type_hints["customized_load_metric_specification"])
                check_type(argname="argument customized_scaling_metric_specification", value=customized_scaling_metric_specification, expected_type=type_hints["customized_scaling_metric_specification"])
                check_type(argname="argument predefined_load_metric_specification", value=predefined_load_metric_specification, expected_type=type_hints["predefined_load_metric_specification"])
                check_type(argname="argument predefined_metric_pair_specification", value=predefined_metric_pair_specification, expected_type=type_hints["predefined_metric_pair_specification"])
                check_type(argname="argument predefined_scaling_metric_specification", value=predefined_scaling_metric_specification, expected_type=type_hints["predefined_scaling_metric_specification"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target_value": target_value,
            }
            if customized_capacity_metric_specification is not None:
                self._values["customized_capacity_metric_specification"] = customized_capacity_metric_specification
            if customized_load_metric_specification is not None:
                self._values["customized_load_metric_specification"] = customized_load_metric_specification
            if customized_scaling_metric_specification is not None:
                self._values["customized_scaling_metric_specification"] = customized_scaling_metric_specification
            if predefined_load_metric_specification is not None:
                self._values["predefined_load_metric_specification"] = predefined_load_metric_specification
            if predefined_metric_pair_specification is not None:
                self._values["predefined_metric_pair_specification"] = predefined_metric_pair_specification
            if predefined_scaling_metric_specification is not None:
                self._values["predefined_scaling_metric_specification"] = predefined_scaling_metric_specification

        @builtins.property
        def target_value(self) -> jsii.Number:
            '''Specifies the target utilization.

            .. epigraph::

               Some metrics are based on a count instead of a percentage, such as the request count for an Application Load Balancer or the number of messages in an SQS queue. If the scaling policy specifies one of these metrics, specify the target utilization as the optimal average request or message count per instance during any one-minute interval.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-targetvalue
            '''
            result = self._values.get("target_value")
            assert result is not None, "Required property 'target_value' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def customized_capacity_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty"]]:
            '''The customized capacity metric specification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-customizedcapacitymetricspecification
            '''
            result = self._values.get("customized_capacity_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty"]], result)

        @builtins.property
        def customized_load_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty"]]:
            '''The customized load metric specification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-customizedloadmetricspecification
            '''
            result = self._values.get("customized_load_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty"]], result)

        @builtins.property
        def customized_scaling_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty"]]:
            '''The customized scaling metric specification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-customizedscalingmetricspecification
            '''
            result = self._values.get("customized_scaling_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty"]], result)

        @builtins.property
        def predefined_load_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty"]]:
            '''The predefined load metric specification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-predefinedloadmetricspecification
            '''
            result = self._values.get("predefined_load_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty"]], result)

        @builtins.property
        def predefined_metric_pair_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty"]]:
            '''The predefined metric pair specification from which Amazon EC2 Auto Scaling determines the appropriate scaling metric and load metric to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-predefinedmetricpairspecification
            '''
            result = self._values.get("predefined_metric_pair_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty"]], result)

        @builtins.property
        def predefined_scaling_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty"]]:
            '''The predefined scaling metric specification.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html#cfn-autoscaling-scalingpolicy-predictivescalingmetricspecification-predefinedscalingmetricspecification
            '''
            result = self._values.get("predefined_scaling_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingMetricSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty",
        jsii_struct_bases=[],
        name_mapping={
            "predefined_metric_type": "predefinedMetricType",
            "resource_label": "resourceLabel",
        },
    )
    class PredictiveScalingPredefinedLoadMetricProperty:
        def __init__(
            self,
            *,
            predefined_metric_type: builtins.str,
            resource_label: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains load metric information for the ``PredefinedLoadMetricSpecification`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html>`_ property type.

            .. epigraph::

               Does not apply to policies that use a *metric pair* for the metric specification.

            :param predefined_metric_type: The metric type.
            :param resource_label: A label that uniquely identifies a specific Application Load Balancer target group from which to determine the request count served by your Auto Scaling group. You can't specify a resource label unless the target group is attached to the Auto Scaling group. You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is: ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` . Where: - app// is the final portion of the load balancer ARN - targetgroup// is the final portion of the target group ARN. To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedloadmetric.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_predefined_load_metric_property = autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty(
                    predefined_metric_type="predefinedMetricType",
                
                    # the properties below are optional
                    resource_label="resourceLabel"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b892c3276a7b59850f1012ad500f9332d5dc5006d3b054fd94ade070e069bff1)
                check_type(argname="argument predefined_metric_type", value=predefined_metric_type, expected_type=type_hints["predefined_metric_type"])
                check_type(argname="argument resource_label", value=resource_label, expected_type=type_hints["resource_label"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "predefined_metric_type": predefined_metric_type,
            }
            if resource_label is not None:
                self._values["resource_label"] = resource_label

        @builtins.property
        def predefined_metric_type(self) -> builtins.str:
            '''The metric type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedloadmetric.html#cfn-autoscaling-scalingpolicy-predictivescalingpredefinedloadmetric-predefinedmetrictype
            '''
            result = self._values.get("predefined_metric_type")
            assert result is not None, "Required property 'predefined_metric_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_label(self) -> typing.Optional[builtins.str]:
            '''A label that uniquely identifies a specific Application Load Balancer target group from which to determine the request count served by your Auto Scaling group.

            You can't specify a resource label unless the target group is attached to the Auto Scaling group.

            You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is:

            ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` .

            Where:

            - app// is the final portion of the load balancer ARN
            - targetgroup// is the final portion of the target group ARN.

            To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedloadmetric.html#cfn-autoscaling-scalingpolicy-predictivescalingpredefinedloadmetric-resourcelabel
            '''
            result = self._values.get("resource_label")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingPredefinedLoadMetricProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty",
        jsii_struct_bases=[],
        name_mapping={
            "predefined_metric_type": "predefinedMetricType",
            "resource_label": "resourceLabel",
        },
    )
    class PredictiveScalingPredefinedMetricPairProperty:
        def __init__(
            self,
            *,
            predefined_metric_type: builtins.str,
            resource_label: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains metric pair information for the ``PredefinedMetricPairSpecification`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html>`_ property type.

            For more information, see `Predictive scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-predictive-scaling.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param predefined_metric_type: Indicates which metrics to use. There are two different types of metrics for each metric type: one is a load metric and one is a scaling metric. For example, if the metric type is ``ASGCPUUtilization`` , the Auto Scaling group's total CPU metric is used as the load metric, and the average CPU metric is used for the scaling metric.
            :param resource_label: A label that uniquely identifies a specific Application Load Balancer target group from which to determine the total and average request count served by your Auto Scaling group. You can't specify a resource label unless the target group is attached to the Auto Scaling group. You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is: ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` . Where: - app// is the final portion of the load balancer ARN - targetgroup// is the final portion of the target group ARN. To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedmetricpair.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_predefined_metric_pair_property = autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty(
                    predefined_metric_type="predefinedMetricType",
                
                    # the properties below are optional
                    resource_label="resourceLabel"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c785e7f26210bff1ddc4fb3fa18fb6e02212d89551293dfc24bfc8de2cb43693)
                check_type(argname="argument predefined_metric_type", value=predefined_metric_type, expected_type=type_hints["predefined_metric_type"])
                check_type(argname="argument resource_label", value=resource_label, expected_type=type_hints["resource_label"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "predefined_metric_type": predefined_metric_type,
            }
            if resource_label is not None:
                self._values["resource_label"] = resource_label

        @builtins.property
        def predefined_metric_type(self) -> builtins.str:
            '''Indicates which metrics to use.

            There are two different types of metrics for each metric type: one is a load metric and one is a scaling metric. For example, if the metric type is ``ASGCPUUtilization`` , the Auto Scaling group's total CPU metric is used as the load metric, and the average CPU metric is used for the scaling metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedmetricpair.html#cfn-autoscaling-scalingpolicy-predictivescalingpredefinedmetricpair-predefinedmetrictype
            '''
            result = self._values.get("predefined_metric_type")
            assert result is not None, "Required property 'predefined_metric_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_label(self) -> typing.Optional[builtins.str]:
            '''A label that uniquely identifies a specific Application Load Balancer target group from which to determine the total and average request count served by your Auto Scaling group.

            You can't specify a resource label unless the target group is attached to the Auto Scaling group.

            You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is:

            ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` .

            Where:

            - app// is the final portion of the load balancer ARN
            - targetgroup// is the final portion of the target group ARN.

            To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedmetricpair.html#cfn-autoscaling-scalingpolicy-predictivescalingpredefinedmetricpair-resourcelabel
            '''
            result = self._values.get("resource_label")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingPredefinedMetricPairProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty",
        jsii_struct_bases=[],
        name_mapping={
            "predefined_metric_type": "predefinedMetricType",
            "resource_label": "resourceLabel",
        },
    )
    class PredictiveScalingPredefinedScalingMetricProperty:
        def __init__(
            self,
            *,
            predefined_metric_type: builtins.str,
            resource_label: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains scaling metric information for the ``PredefinedScalingMetricSpecification`` property of the `AWS::AutoScaling::ScalingPolicy PredictiveScalingMetricSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingmetricspecification.html>`_ property type.

            .. epigraph::

               Does not apply to policies that use a *metric pair* for the metric specification.

            :param predefined_metric_type: The metric type.
            :param resource_label: A label that uniquely identifies a specific Application Load Balancer target group from which to determine the average request count served by your Auto Scaling group. You can't specify a resource label unless the target group is attached to the Auto Scaling group. You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is: ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` . Where: - app// is the final portion of the load balancer ARN - targetgroup// is the final portion of the target group ARN. To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedscalingmetric.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                predictive_scaling_predefined_scaling_metric_property = autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty(
                    predefined_metric_type="predefinedMetricType",
                
                    # the properties below are optional
                    resource_label="resourceLabel"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5bab617be42e89269ec2d94cc76843c277bbf8ff7acd3fc168f6539883294e2d)
                check_type(argname="argument predefined_metric_type", value=predefined_metric_type, expected_type=type_hints["predefined_metric_type"])
                check_type(argname="argument resource_label", value=resource_label, expected_type=type_hints["resource_label"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "predefined_metric_type": predefined_metric_type,
            }
            if resource_label is not None:
                self._values["resource_label"] = resource_label

        @builtins.property
        def predefined_metric_type(self) -> builtins.str:
            '''The metric type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedscalingmetric.html#cfn-autoscaling-scalingpolicy-predictivescalingpredefinedscalingmetric-predefinedmetrictype
            '''
            result = self._values.get("predefined_metric_type")
            assert result is not None, "Required property 'predefined_metric_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_label(self) -> typing.Optional[builtins.str]:
            '''A label that uniquely identifies a specific Application Load Balancer target group from which to determine the average request count served by your Auto Scaling group.

            You can't specify a resource label unless the target group is attached to the Auto Scaling group.

            You create the resource label by appending the final portion of the load balancer ARN and the final portion of the target group ARN into a single value, separated by a forward slash (/). The format of the resource label is:

            ``app/my-alb/778d41231b141a0f/targetgroup/my-alb-target-group/943f017f100becff`` .

            Where:

            - app// is the final portion of the load balancer ARN
            - targetgroup// is the final portion of the target group ARN.

            To find the ARN for an Application Load Balancer, use the `DescribeLoadBalancers <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeLoadBalancers.html>`_ API operation. To find the ARN for the target group, use the `DescribeTargetGroups <https://docs.aws.amazon.com/elasticloadbalancing/latest/APIReference/API_DescribeTargetGroups.html>`_ API operation.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-predictivescalingpredefinedscalingmetric.html#cfn-autoscaling-scalingpolicy-predictivescalingpredefinedscalingmetric-resourcelabel
            '''
            result = self._values.get("resource_label")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PredictiveScalingPredefinedScalingMetricProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.StepAdjustmentProperty",
        jsii_struct_bases=[],
        name_mapping={
            "scaling_adjustment": "scalingAdjustment",
            "metric_interval_lower_bound": "metricIntervalLowerBound",
            "metric_interval_upper_bound": "metricIntervalUpperBound",
        },
    )
    class StepAdjustmentProperty:
        def __init__(
            self,
            *,
            scaling_adjustment: jsii.Number,
            metric_interval_lower_bound: typing.Optional[jsii.Number] = None,
            metric_interval_upper_bound: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``StepAdjustment`` specifies a step adjustment for the ``StepAdjustments`` property of the `AWS::AutoScaling::ScalingPolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-policy.html>`_ resource.

            For the following examples, suppose that you have an alarm with a breach threshold of 50:

            - To trigger a step adjustment when the metric is greater than or equal to 50 and less than 60, specify a lower bound of 0 and an upper bound of 10.
            - To trigger a step adjustment when the metric is greater than 40 and less than or equal to 50, specify a lower bound of -10 and an upper bound of 0.

            There are a few rules for the step adjustments for your step policy:

            - The ranges of your step adjustments can't overlap or have a gap.
            - At most one step adjustment can have a null lower bound. If one step adjustment has a negative lower bound, then there must be a step adjustment with a null lower bound.
            - At most one step adjustment can have a null upper bound. If one step adjustment has a positive upper bound, then there must be a step adjustment with a null upper bound.
            - The upper and lower bound can't be null in the same step adjustment.

            For more information, see `Step adjustments <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-steps>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            You can find a sample template snippet in the `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-policy.html#aws-properties-as-policy--examples>`_ section of the ``AWS::AutoScaling::ScalingPolicy`` resource.

            :param scaling_adjustment: The amount by which to scale, based on the specified adjustment type. A positive value adds to the current capacity while a negative number removes from the current capacity. The amount by which to scale. The adjustment is based on the value that you specified in the ``AdjustmentType`` property (either an absolute number or a percentage). A positive value adds to the current capacity and a negative number subtracts from the current capacity.
            :param metric_interval_lower_bound: The lower bound for the difference between the alarm threshold and the CloudWatch metric. If the metric value is above the breach threshold, the lower bound is inclusive (the metric must be greater than or equal to the threshold plus the lower bound). Otherwise, it is exclusive (the metric must be greater than the threshold plus the lower bound). A null value indicates negative infinity.
            :param metric_interval_upper_bound: The upper bound for the difference between the alarm threshold and the CloudWatch metric. If the metric value is above the breach threshold, the upper bound is exclusive (the metric must be less than the threshold plus the upper bound). Otherwise, it is inclusive (the metric must be less than or equal to the threshold plus the upper bound). A null value indicates positive infinity. The upper bound must be greater than the lower bound.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-stepadjustment.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                step_adjustment_property = autoscaling.CfnScalingPolicy.StepAdjustmentProperty(
                    scaling_adjustment=123,
                
                    # the properties below are optional
                    metric_interval_lower_bound=123,
                    metric_interval_upper_bound=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7db0ee8d4961a22cdf58fd616c6c93f16a89862d95627b09d3ccbf3f7315e00f)
                check_type(argname="argument scaling_adjustment", value=scaling_adjustment, expected_type=type_hints["scaling_adjustment"])
                check_type(argname="argument metric_interval_lower_bound", value=metric_interval_lower_bound, expected_type=type_hints["metric_interval_lower_bound"])
                check_type(argname="argument metric_interval_upper_bound", value=metric_interval_upper_bound, expected_type=type_hints["metric_interval_upper_bound"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "scaling_adjustment": scaling_adjustment,
            }
            if metric_interval_lower_bound is not None:
                self._values["metric_interval_lower_bound"] = metric_interval_lower_bound
            if metric_interval_upper_bound is not None:
                self._values["metric_interval_upper_bound"] = metric_interval_upper_bound

        @builtins.property
        def scaling_adjustment(self) -> jsii.Number:
            '''The amount by which to scale, based on the specified adjustment type.

            A positive value adds to the current capacity while a negative number removes from the current capacity.

            The amount by which to scale. The adjustment is based on the value that you specified in the ``AdjustmentType`` property (either an absolute number or a percentage). A positive value adds to the current capacity and a negative number subtracts from the current capacity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-stepadjustment.html#cfn-autoscaling-scalingpolicy-stepadjustment-scalingadjustment
            '''
            result = self._values.get("scaling_adjustment")
            assert result is not None, "Required property 'scaling_adjustment' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def metric_interval_lower_bound(self) -> typing.Optional[jsii.Number]:
            '''The lower bound for the difference between the alarm threshold and the CloudWatch metric.

            If the metric value is above the breach threshold, the lower bound is inclusive (the metric must be greater than or equal to the threshold plus the lower bound). Otherwise, it is exclusive (the metric must be greater than the threshold plus the lower bound). A null value indicates negative infinity.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-stepadjustment.html#cfn-autoscaling-scalingpolicy-stepadjustment-metricintervallowerbound
            '''
            result = self._values.get("metric_interval_lower_bound")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def metric_interval_upper_bound(self) -> typing.Optional[jsii.Number]:
            '''The upper bound for the difference between the alarm threshold and the CloudWatch metric.

            If the metric value is above the breach threshold, the upper bound is exclusive (the metric must be less than the threshold plus the upper bound). Otherwise, it is inclusive (the metric must be less than or equal to the threshold plus the upper bound). A null value indicates positive infinity.

            The upper bound must be greater than the lower bound.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-stepadjustment.html#cfn-autoscaling-scalingpolicy-stepadjustment-metricintervalupperbound
            '''
            result = self._values.get("metric_interval_upper_bound")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StepAdjustmentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicy.TargetTrackingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "target_value": "targetValue",
            "customized_metric_specification": "customizedMetricSpecification",
            "disable_scale_in": "disableScaleIn",
            "predefined_metric_specification": "predefinedMetricSpecification",
        },
    )
    class TargetTrackingConfigurationProperty:
        def __init__(
            self,
            *,
            target_value: jsii.Number,
            customized_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.CustomizedMetricSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            disable_scale_in: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            predefined_metric_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnScalingPolicy.PredefinedMetricSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''``TargetTrackingConfiguration`` is a property of the `AWS::AutoScaling::ScalingPolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-policy.html>`_ resource that specifies a target tracking scaling policy configuration for Amazon EC2 Auto Scaling.

            For more information about scaling policies, see `Dynamic scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scale-based-on-demand.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param target_value: The target value for the metric. .. epigraph:: Some metrics are based on a count instead of a percentage, such as the request count for an Application Load Balancer or the number of messages in an SQS queue. If the scaling policy specifies one of these metrics, specify the target utilization as the optimal average request or message count per instance during any one-minute interval.
            :param customized_metric_specification: A customized metric. You must specify either a predefined metric or a customized metric.
            :param disable_scale_in: Indicates whether scaling in by the target tracking scaling policy is disabled. If scaling in is disabled, the target tracking scaling policy doesn't remove instances from the Auto Scaling group. Otherwise, the target tracking scaling policy can remove instances from the Auto Scaling group. The default is ``false`` .
            :param predefined_metric_specification: A predefined metric. You must specify either a predefined metric or a customized metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                target_tracking_configuration_property = autoscaling.CfnScalingPolicy.TargetTrackingConfigurationProperty(
                    target_value=123,
                
                    # the properties below are optional
                    customized_metric_specification=autoscaling.CfnScalingPolicy.CustomizedMetricSpecificationProperty(
                        metric_name="metricName",
                        namespace="namespace",
                        statistic="statistic",
                
                        # the properties below are optional
                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                            name="name",
                            value="value"
                        )],
                        unit="unit"
                    ),
                    disable_scale_in=False,
                    predefined_metric_specification=autoscaling.CfnScalingPolicy.PredefinedMetricSpecificationProperty(
                        predefined_metric_type="predefinedMetricType",
                
                        # the properties below are optional
                        resource_label="resourceLabel"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__406f504322f9bf5761a5c07083390725c370252fcf12bd54230e1f02878b2da1)
                check_type(argname="argument target_value", value=target_value, expected_type=type_hints["target_value"])
                check_type(argname="argument customized_metric_specification", value=customized_metric_specification, expected_type=type_hints["customized_metric_specification"])
                check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
                check_type(argname="argument predefined_metric_specification", value=predefined_metric_specification, expected_type=type_hints["predefined_metric_specification"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target_value": target_value,
            }
            if customized_metric_specification is not None:
                self._values["customized_metric_specification"] = customized_metric_specification
            if disable_scale_in is not None:
                self._values["disable_scale_in"] = disable_scale_in
            if predefined_metric_specification is not None:
                self._values["predefined_metric_specification"] = predefined_metric_specification

        @builtins.property
        def target_value(self) -> jsii.Number:
            '''The target value for the metric.

            .. epigraph::

               Some metrics are based on a count instead of a percentage, such as the request count for an Application Load Balancer or the number of messages in an SQS queue. If the scaling policy specifies one of these metrics, specify the target utilization as the optimal average request or message count per instance during any one-minute interval.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html#cfn-autoscaling-scalingpolicy-targettrackingconfiguration-targetvalue
            '''
            result = self._values.get("target_value")
            assert result is not None, "Required property 'target_value' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def customized_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.CustomizedMetricSpecificationProperty"]]:
            '''A customized metric.

            You must specify either a predefined metric or a customized metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html#cfn-autoscaling-scalingpolicy-targettrackingconfiguration-customizedmetricspecification
            '''
            result = self._values.get("customized_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.CustomizedMetricSpecificationProperty"]], result)

        @builtins.property
        def disable_scale_in(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether scaling in by the target tracking scaling policy is disabled.

            If scaling in is disabled, the target tracking scaling policy doesn't remove instances from the Auto Scaling group. Otherwise, the target tracking scaling policy can remove instances from the Auto Scaling group. The default is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html#cfn-autoscaling-scalingpolicy-targettrackingconfiguration-disablescalein
            '''
            result = self._values.get("disable_scale_in")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def predefined_metric_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredefinedMetricSpecificationProperty"]]:
            '''A predefined metric.

            You must specify either a predefined metric or a customized metric.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-scalingpolicy-targettrackingconfiguration.html#cfn-autoscaling-scalingpolicy-targettrackingconfiguration-predefinedmetricspecification
            '''
            result = self._values.get("predefined_metric_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnScalingPolicy.PredefinedMetricSpecificationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TargetTrackingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnScalingPolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_name": "autoScalingGroupName",
        "adjustment_type": "adjustmentType",
        "cooldown": "cooldown",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "metric_aggregation_type": "metricAggregationType",
        "min_adjustment_magnitude": "minAdjustmentMagnitude",
        "policy_type": "policyType",
        "predictive_scaling_configuration": "predictiveScalingConfiguration",
        "scaling_adjustment": "scalingAdjustment",
        "step_adjustments": "stepAdjustments",
        "target_tracking_configuration": "targetTrackingConfiguration",
    },
)
class CfnScalingPolicyProps:
    def __init__(
        self,
        *,
        auto_scaling_group_name: builtins.str,
        adjustment_type: typing.Optional[builtins.str] = None,
        cooldown: typing.Optional[builtins.str] = None,
        estimated_instance_warmup: typing.Optional[jsii.Number] = None,
        metric_aggregation_type: typing.Optional[builtins.str] = None,
        min_adjustment_magnitude: typing.Optional[jsii.Number] = None,
        policy_type: typing.Optional[builtins.str] = None,
        predictive_scaling_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnScalingPolicy.PredictiveScalingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        scaling_adjustment: typing.Optional[jsii.Number] = None,
        step_adjustments: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnScalingPolicy.StepAdjustmentProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        target_tracking_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnScalingPolicy.TargetTrackingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnScalingPolicy``.

        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param adjustment_type: Specifies how the scaling adjustment is interpreted (for example, an absolute number or a percentage). The valid values are ``ChangeInCapacity`` , ``ExactCapacity`` , and ``PercentChangeInCapacity`` . Required if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        :param cooldown: A cooldown period, in seconds, that applies to a specific simple scaling policy. When a cooldown period is specified here, it overrides the default cooldown. Valid only if the policy type is ``SimpleScaling`` . For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* . Default: None
        :param estimated_instance_warmup: *Not needed if the default instance warmup is defined for the group.*. The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics. This warm-up period applies to instances launched due to a specific target tracking or step scaling policy. When a warm-up period is specified here, it overrides the default instance warmup. Valid only if the policy type is ``TargetTrackingScaling`` or ``StepScaling`` . .. epigraph:: The default is to use the value for the default instance warmup defined for the group. If default instance warmup is null, then ``EstimatedInstanceWarmup`` falls back to the value of default cooldown.
        :param metric_aggregation_type: The aggregation type for the CloudWatch metrics. The valid values are ``Minimum`` , ``Maximum`` , and ``Average`` . If the aggregation type is null, the value is treated as ``Average`` . Valid only if the policy type is ``StepScaling`` .
        :param min_adjustment_magnitude: The minimum value to scale by when the adjustment type is ``PercentChangeInCapacity`` . For example, suppose that you create a step scaling policy to scale out an Auto Scaling group by 25 percent and you specify a ``MinAdjustmentMagnitude`` of 2. If the group has 4 instances and the scaling policy is performed, 25 percent of 4 is 1. However, because you specified a ``MinAdjustmentMagnitude`` of 2, Amazon EC2 Auto Scaling scales out the group by 2 instances. Valid only if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* . .. epigraph:: Some Auto Scaling groups use instance weights. In this case, set the ``MinAdjustmentMagnitude`` to a value that is at least as large as your largest instance weight.
        :param policy_type: One of the following policy types:. - ``TargetTrackingScaling`` - ``StepScaling`` - ``SimpleScaling`` (default) - ``PredictiveScaling``
        :param predictive_scaling_configuration: A predictive scaling policy. Provides support for predefined and custom metrics. Predefined metrics include CPU utilization, network in/out, and the Application Load Balancer request count. Required if the policy type is ``PredictiveScaling`` .
        :param scaling_adjustment: The amount by which to scale, based on the specified adjustment type. A positive value adds to the current capacity while a negative number removes from the current capacity. For exact capacity, you must specify a positive value. Required if the policy type is ``SimpleScaling`` . (Not used with any other policy type.)
        :param step_adjustments: A set of adjustments that enable you to scale based on the size of the alarm breach. Required if the policy type is ``StepScaling`` . (Not used with any other policy type.)
        :param target_tracking_configuration: A target tracking scaling policy. Provides support for predefined or custom metrics. The following predefined metrics are available: - ``ASGAverageCPUUtilization`` - ``ASGAverageNetworkIn`` - ``ASGAverageNetworkOut`` - ``ALBRequestCountPerTarget`` If you specify ``ALBRequestCountPerTarget`` for the metric, you must specify the ``ResourceLabel`` property with the ``PredefinedMetricSpecification`` . Required if the policy type is ``TargetTrackingScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            cfn_scaling_policy_props = autoscaling.CfnScalingPolicyProps(
                auto_scaling_group_name="autoScalingGroupName",
            
                # the properties below are optional
                adjustment_type="adjustmentType",
                cooldown="cooldown",
                estimated_instance_warmup=123,
                metric_aggregation_type="metricAggregationType",
                min_adjustment_magnitude=123,
                policy_type="policyType",
                predictive_scaling_configuration=autoscaling.CfnScalingPolicy.PredictiveScalingConfigurationProperty(
                    metric_specifications=[autoscaling.CfnScalingPolicy.PredictiveScalingMetricSpecificationProperty(
                        target_value=123,
            
                        # the properties below are optional
                        customized_capacity_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedCapacityMetricProperty(
                            metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                                id="id",
            
                                # the properties below are optional
                                expression="expression",
                                label="label",
                                metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                        metric_name="metricName",
                                        namespace="namespace",
            
                                        # the properties below are optional
                                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                            name="name",
                                            value="value"
                                        )]
                                    ),
                                    stat="stat",
            
                                    # the properties below are optional
                                    unit="unit"
                                ),
                                return_data=False
                            )]
                        ),
                        customized_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedLoadMetricProperty(
                            metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                                id="id",
            
                                # the properties below are optional
                                expression="expression",
                                label="label",
                                metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                        metric_name="metricName",
                                        namespace="namespace",
            
                                        # the properties below are optional
                                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                            name="name",
                                            value="value"
                                        )]
                                    ),
                                    stat="stat",
            
                                    # the properties below are optional
                                    unit="unit"
                                ),
                                return_data=False
                            )]
                        ),
                        customized_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingCustomizedScalingMetricProperty(
                            metric_data_queries=[autoscaling.CfnScalingPolicy.MetricDataQueryProperty(
                                id="id",
            
                                # the properties below are optional
                                expression="expression",
                                label="label",
                                metric_stat=autoscaling.CfnScalingPolicy.MetricStatProperty(
                                    metric=autoscaling.CfnScalingPolicy.MetricProperty(
                                        metric_name="metricName",
                                        namespace="namespace",
            
                                        # the properties below are optional
                                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                                            name="name",
                                            value="value"
                                        )]
                                    ),
                                    stat="stat",
            
                                    # the properties below are optional
                                    unit="unit"
                                ),
                                return_data=False
                            )]
                        ),
                        predefined_load_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedLoadMetricProperty(
                            predefined_metric_type="predefinedMetricType",
            
                            # the properties below are optional
                            resource_label="resourceLabel"
                        ),
                        predefined_metric_pair_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedMetricPairProperty(
                            predefined_metric_type="predefinedMetricType",
            
                            # the properties below are optional
                            resource_label="resourceLabel"
                        ),
                        predefined_scaling_metric_specification=autoscaling.CfnScalingPolicy.PredictiveScalingPredefinedScalingMetricProperty(
                            predefined_metric_type="predefinedMetricType",
            
                            # the properties below are optional
                            resource_label="resourceLabel"
                        )
                    )],
            
                    # the properties below are optional
                    max_capacity_breach_behavior="maxCapacityBreachBehavior",
                    max_capacity_buffer=123,
                    mode="mode",
                    scheduling_buffer_time=123
                ),
                scaling_adjustment=123,
                step_adjustments=[autoscaling.CfnScalingPolicy.StepAdjustmentProperty(
                    scaling_adjustment=123,
            
                    # the properties below are optional
                    metric_interval_lower_bound=123,
                    metric_interval_upper_bound=123
                )],
                target_tracking_configuration=autoscaling.CfnScalingPolicy.TargetTrackingConfigurationProperty(
                    target_value=123,
            
                    # the properties below are optional
                    customized_metric_specification=autoscaling.CfnScalingPolicy.CustomizedMetricSpecificationProperty(
                        metric_name="metricName",
                        namespace="namespace",
                        statistic="statistic",
            
                        # the properties below are optional
                        dimensions=[autoscaling.CfnScalingPolicy.MetricDimensionProperty(
                            name="name",
                            value="value"
                        )],
                        unit="unit"
                    ),
                    disable_scale_in=False,
                    predefined_metric_specification=autoscaling.CfnScalingPolicy.PredefinedMetricSpecificationProperty(
                        predefined_metric_type="predefinedMetricType",
            
                        # the properties below are optional
                        resource_label="resourceLabel"
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa1549144f025fb75ca19967d6520b5eaada79209726c0ec131f1146e087fc7d)
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument adjustment_type", value=adjustment_type, expected_type=type_hints["adjustment_type"])
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument metric_aggregation_type", value=metric_aggregation_type, expected_type=type_hints["metric_aggregation_type"])
            check_type(argname="argument min_adjustment_magnitude", value=min_adjustment_magnitude, expected_type=type_hints["min_adjustment_magnitude"])
            check_type(argname="argument policy_type", value=policy_type, expected_type=type_hints["policy_type"])
            check_type(argname="argument predictive_scaling_configuration", value=predictive_scaling_configuration, expected_type=type_hints["predictive_scaling_configuration"])
            check_type(argname="argument scaling_adjustment", value=scaling_adjustment, expected_type=type_hints["scaling_adjustment"])
            check_type(argname="argument step_adjustments", value=step_adjustments, expected_type=type_hints["step_adjustments"])
            check_type(argname="argument target_tracking_configuration", value=target_tracking_configuration, expected_type=type_hints["target_tracking_configuration"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auto_scaling_group_name": auto_scaling_group_name,
        }
        if adjustment_type is not None:
            self._values["adjustment_type"] = adjustment_type
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup
        if metric_aggregation_type is not None:
            self._values["metric_aggregation_type"] = metric_aggregation_type
        if min_adjustment_magnitude is not None:
            self._values["min_adjustment_magnitude"] = min_adjustment_magnitude
        if policy_type is not None:
            self._values["policy_type"] = policy_type
        if predictive_scaling_configuration is not None:
            self._values["predictive_scaling_configuration"] = predictive_scaling_configuration
        if scaling_adjustment is not None:
            self._values["scaling_adjustment"] = scaling_adjustment
        if step_adjustments is not None:
            self._values["step_adjustments"] = step_adjustments
        if target_tracking_configuration is not None:
            self._values["target_tracking_configuration"] = target_tracking_configuration

    @builtins.property
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-autoscalinggroupname
        '''
        result = self._values.get("auto_scaling_group_name")
        assert result is not None, "Required property 'auto_scaling_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def adjustment_type(self) -> typing.Optional[builtins.str]:
        '''Specifies how the scaling adjustment is interpreted (for example, an absolute number or a percentage).

        The valid values are ``ChangeInCapacity`` , ``ExactCapacity`` , and ``PercentChangeInCapacity`` .

        Required if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-adjustmenttype
        '''
        result = self._values.get("adjustment_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[builtins.str]:
        '''A cooldown period, in seconds, that applies to a specific simple scaling policy.

        When a cooldown period is specified here, it overrides the default cooldown.

        Valid only if the policy type is ``SimpleScaling`` . For more information, see `Scaling cooldowns for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/Cooldown.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

        Default: None

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-cooldown
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[jsii.Number]:
        '''*Not needed if the default instance warmup is defined for the group.*.

        The estimated time, in seconds, until a newly launched instance can contribute to the CloudWatch metrics. This warm-up period applies to instances launched due to a specific target tracking or step scaling policy. When a warm-up period is specified here, it overrides the default instance warmup.

        Valid only if the policy type is ``TargetTrackingScaling`` or ``StepScaling`` .
        .. epigraph::

           The default is to use the value for the default instance warmup defined for the group. If default instance warmup is null, then ``EstimatedInstanceWarmup`` falls back to the value of default cooldown.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-estimatedinstancewarmup
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def metric_aggregation_type(self) -> typing.Optional[builtins.str]:
        '''The aggregation type for the CloudWatch metrics.

        The valid values are ``Minimum`` , ``Maximum`` , and ``Average`` . If the aggregation type is null, the value is treated as ``Average`` .

        Valid only if the policy type is ``StepScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-metricaggregationtype
        '''
        result = self._values.get("metric_aggregation_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def min_adjustment_magnitude(self) -> typing.Optional[jsii.Number]:
        '''The minimum value to scale by when the adjustment type is ``PercentChangeInCapacity`` .

        For example, suppose that you create a step scaling policy to scale out an Auto Scaling group by 25 percent and you specify a ``MinAdjustmentMagnitude`` of 2. If the group has 4 instances and the scaling policy is performed, 25 percent of 4 is 1. However, because you specified a ``MinAdjustmentMagnitude`` of 2, Amazon EC2 Auto Scaling scales out the group by 2 instances.

        Valid only if the policy type is ``StepScaling`` or ``SimpleScaling`` . For more information, see `Scaling adjustment types <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-scaling-simple-step.html#as-scaling-adjustment>`_ in the *Amazon EC2 Auto Scaling User Guide* .
        .. epigraph::

           Some Auto Scaling groups use instance weights. In this case, set the ``MinAdjustmentMagnitude`` to a value that is at least as large as your largest instance weight.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-minadjustmentmagnitude
        '''
        result = self._values.get("min_adjustment_magnitude")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def policy_type(self) -> typing.Optional[builtins.str]:
        '''One of the following policy types:.

        - ``TargetTrackingScaling``
        - ``StepScaling``
        - ``SimpleScaling`` (default)
        - ``PredictiveScaling``

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-policytype
        '''
        result = self._values.get("policy_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def predictive_scaling_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnScalingPolicy.PredictiveScalingConfigurationProperty]]:
        '''A predictive scaling policy. Provides support for predefined and custom metrics.

        Predefined metrics include CPU utilization, network in/out, and the Application Load Balancer request count.

        Required if the policy type is ``PredictiveScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-predictivescalingconfiguration
        '''
        result = self._values.get("predictive_scaling_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnScalingPolicy.PredictiveScalingConfigurationProperty]], result)

    @builtins.property
    def scaling_adjustment(self) -> typing.Optional[jsii.Number]:
        '''The amount by which to scale, based on the specified adjustment type.

        A positive value adds to the current capacity while a negative number removes from the current capacity. For exact capacity, you must specify a positive value.

        Required if the policy type is ``SimpleScaling`` . (Not used with any other policy type.)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-scalingadjustment
        '''
        result = self._values.get("scaling_adjustment")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def step_adjustments(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnScalingPolicy.StepAdjustmentProperty]]]]:
        '''A set of adjustments that enable you to scale based on the size of the alarm breach.

        Required if the policy type is ``StepScaling`` . (Not used with any other policy type.)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-stepadjustments
        '''
        result = self._values.get("step_adjustments")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnScalingPolicy.StepAdjustmentProperty]]]], result)

    @builtins.property
    def target_tracking_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnScalingPolicy.TargetTrackingConfigurationProperty]]:
        '''A target tracking scaling policy. Provides support for predefined or custom metrics.

        The following predefined metrics are available:

        - ``ASGAverageCPUUtilization``
        - ``ASGAverageNetworkIn``
        - ``ASGAverageNetworkOut``
        - ``ALBRequestCountPerTarget``

        If you specify ``ALBRequestCountPerTarget`` for the metric, you must specify the ``ResourceLabel`` property with the ``PredefinedMetricSpecification`` .

        Required if the policy type is ``TargetTrackingScaling`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scalingpolicy.html#cfn-autoscaling-scalingpolicy-targettrackingconfiguration
        '''
        result = self._values.get("target_tracking_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnScalingPolicy.TargetTrackingConfigurationProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnScalingPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnScheduledAction(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnScheduledAction",
):
    '''A CloudFormation ``AWS::AutoScaling::ScheduledAction``.

    The ``AWS::AutoScaling::ScheduledAction`` resource specifies an Amazon EC2 Auto Scaling scheduled action so that the Auto Scaling group can change the number of instances available for your application in response to predictable load changes.

    When you update a stack with an Auto Scaling group and scheduled action, CloudFormation always sets the min size, max size, and desired capacity properties of your group to the values that are defined in the ``AWS::AutoScaling::AutoScalingGroup`` section of your template. However, you might not want CloudFormation to do that when you have a scheduled action in effect. You can use an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ to prevent CloudFormation from changing the min size, max size, or desired capacity property values during a stack update unless you modified the individual values in your template. If you have rolling updates enabled, before you can update the Auto Scaling group, you must suspend scheduled actions by specifying an `UpdatePolicy attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html>`_ for the Auto Scaling group. You can find a sample update policy for rolling updates in `Auto scaling template snippets <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/quickref-autoscaling.html>`_ .

    For more information, see `Scheduled scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/schedule_time.html>`_ and `Suspending and resuming scaling processes <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-suspend-resume-processes.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

    :cloudformationResource: AWS::AutoScaling::ScheduledAction
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        cfn_scheduled_action = autoscaling.CfnScheduledAction(self, "MyCfnScheduledAction",
            auto_scaling_group_name="autoScalingGroupName",
        
            # the properties below are optional
            desired_capacity=123,
            end_time="endTime",
            max_size=123,
            min_size=123,
            recurrence="recurrence",
            start_time="startTime",
            time_zone="timeZone"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group_name: builtins.str,
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[builtins.str] = None,
        max_size: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        recurrence: typing.Optional[builtins.str] = None,
        start_time: typing.Optional[builtins.str] = None,
        time_zone: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AutoScaling::ScheduledAction``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param desired_capacity: The desired capacity is the initial capacity of the Auto Scaling group after the scheduled action runs and the capacity it attempts to maintain. It can scale beyond this capacity if you add more scaling conditions. .. epigraph:: You must specify at least one of the following properties: ``MaxSize`` , ``MinSize`` , or ``DesiredCapacity`` .
        :param end_time: The date and time for the recurring schedule to end, in UTC. For example, ``"2021-06-01T00:00:00Z"`` .
        :param max_size: The maximum size of the Auto Scaling group.
        :param min_size: The minimum size of the Auto Scaling group.
        :param recurrence: The recurring schedule for this action. This format consists of five fields separated by white spaces: [Minute] [Hour] [Day_of_Month] [Month_of_Year] [Day_of_Week]. The value must be in quotes (for example, ``"30 0 1 1,6,12 *"`` ). For more information about this format, see `Crontab <https://docs.aws.amazon.com/http://crontab.org>`_ . When ``StartTime`` and ``EndTime`` are specified with ``Recurrence`` , they form the boundaries of when the recurring action starts and stops. Cron expressions use Universal Coordinated Time (UTC) by default.
        :param start_time: The date and time for this action to start, in YYYY-MM-DDThh:mm:ssZ format in UTC/GMT only and in quotes (for example, ``"2021-06-01T00:00:00Z"`` ). If you specify ``Recurrence`` and ``StartTime`` , Amazon EC2 Auto Scaling performs the action at this time, and then performs the action based on the specified recurrence.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as ``Etc/GMT+9`` or ``Pacific/Tahiti`` ). For more information, see `https://en.wikipedia.org/wiki/List_of_tz_database_time_zones <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/List_of_tz_database_time_zones>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74bd532ea628d3cbc35d12b83c73435431c2df43d44c4873521366de3097d9b9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnScheduledActionProps(
            auto_scaling_group_name=auto_scaling_group_name,
            desired_capacity=desired_capacity,
            end_time=end_time,
            max_size=max_size,
            min_size=min_size,
            recurrence=recurrence,
            start_time=start_time,
            time_zone=time_zone,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f00bf537ece7b967f35535b2f323e100a7b8276d49a9a48375e63ea4640bdaa)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__733e03c99ea4656dce9586e10e90b10a5743563d6ad40587ebed3c8695c9456e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrScheduledActionName")
    def attr_scheduled_action_name(self) -> builtins.str:
        '''Returns the name of a scheduled action.

        :cloudformationAttribute: ScheduledActionName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrScheduledActionName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-autoscalinggroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "autoScalingGroupName"))

    @auto_scaling_group_name.setter
    def auto_scaling_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1dd441edc1488cc06dc97c3b0f6e5d68bf2b1c4c01d93f394a9fe376ad37cd23)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoScalingGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="desiredCapacity")
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''The desired capacity is the initial capacity of the Auto Scaling group after the scheduled action runs and the capacity it attempts to maintain.

        It can scale beyond this capacity if you add more scaling conditions.
        .. epigraph::

           You must specify at least one of the following properties: ``MaxSize`` , ``MinSize`` , or ``DesiredCapacity`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-desiredcapacity
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "desiredCapacity"))

    @desired_capacity.setter
    def desired_capacity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c0851483d032bf68ef1bccbf11427891bef8b6d370dbfc2be247824df6bcdae9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "desiredCapacity", value)

    @builtins.property
    @jsii.member(jsii_name="endTime")
    def end_time(self) -> typing.Optional[builtins.str]:
        '''The date and time for the recurring schedule to end, in UTC.

        For example, ``"2021-06-01T00:00:00Z"`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-endtime
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endTime"))

    @end_time.setter
    def end_time(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b4fa3c4bcbd8e2248b3c2ee916128bba3cb67a90832f0a69a6860c209b662c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endTime", value)

    @builtins.property
    @jsii.member(jsii_name="maxSize")
    def max_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum size of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-maxsize
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxSize"))

    @max_size.setter
    def max_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e46adcb734421554106ae9884d4d632394c132e38400a89c5db3680212032e2d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxSize", value)

    @builtins.property
    @jsii.member(jsii_name="minSize")
    def min_size(self) -> typing.Optional[jsii.Number]:
        '''The minimum size of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-minsize
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minSize"))

    @min_size.setter
    def min_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__73e3365646f584b55cee25b1399e6c5dca58fb2b8351e0160b6e5c414247f916)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minSize", value)

    @builtins.property
    @jsii.member(jsii_name="recurrence")
    def recurrence(self) -> typing.Optional[builtins.str]:
        '''The recurring schedule for this action.

        This format consists of five fields separated by white spaces: [Minute] [Hour] [Day_of_Month] [Month_of_Year] [Day_of_Week]. The value must be in quotes (for example, ``"30 0 1 1,6,12 *"`` ). For more information about this format, see `Crontab <https://docs.aws.amazon.com/http://crontab.org>`_ .

        When ``StartTime`` and ``EndTime`` are specified with ``Recurrence`` , they form the boundaries of when the recurring action starts and stops.

        Cron expressions use Universal Coordinated Time (UTC) by default.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-recurrence
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "recurrence"))

    @recurrence.setter
    def recurrence(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ea493c09ea513a5fb9c5a878fef603ce8133cc15aee129c7dbd8bb8d1c83796)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "recurrence", value)

    @builtins.property
    @jsii.member(jsii_name="startTime")
    def start_time(self) -> typing.Optional[builtins.str]:
        '''The date and time for this action to start, in YYYY-MM-DDThh:mm:ssZ format in UTC/GMT only and in quotes (for example, ``"2021-06-01T00:00:00Z"`` ).

        If you specify ``Recurrence`` and ``StartTime`` , Amazon EC2 Auto Scaling performs the action at this time, and then performs the action based on the specified recurrence.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-starttime
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "startTime"))

    @start_time.setter
    def start_time(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ffb28539347fd492bf01e79d453a6972a3045330369eba1e812203800d92462)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "startTime", value)

    @builtins.property
    @jsii.member(jsii_name="timeZone")
    def time_zone(self) -> typing.Optional[builtins.str]:
        '''Specifies the time zone for a cron expression.

        If a time zone is not provided, UTC is used by default.

        Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as ``Etc/GMT+9`` or ``Pacific/Tahiti`` ). For more information, see `https://en.wikipedia.org/wiki/List_of_tz_database_time_zones <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/List_of_tz_database_time_zones>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-timezone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "timeZone"))

    @time_zone.setter
    def time_zone(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a8884960cdb0544ac67e0139227014617c99358bb79d7312ecf1d7ab5a6eb10)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timeZone", value)


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnScheduledActionProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_name": "autoScalingGroupName",
        "desired_capacity": "desiredCapacity",
        "end_time": "endTime",
        "max_size": "maxSize",
        "min_size": "minSize",
        "recurrence": "recurrence",
        "start_time": "startTime",
        "time_zone": "timeZone",
    },
)
class CfnScheduledActionProps:
    def __init__(
        self,
        *,
        auto_scaling_group_name: builtins.str,
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[builtins.str] = None,
        max_size: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        recurrence: typing.Optional[builtins.str] = None,
        start_time: typing.Optional[builtins.str] = None,
        time_zone: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnScheduledAction``.

        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param desired_capacity: The desired capacity is the initial capacity of the Auto Scaling group after the scheduled action runs and the capacity it attempts to maintain. It can scale beyond this capacity if you add more scaling conditions. .. epigraph:: You must specify at least one of the following properties: ``MaxSize`` , ``MinSize`` , or ``DesiredCapacity`` .
        :param end_time: The date and time for the recurring schedule to end, in UTC. For example, ``"2021-06-01T00:00:00Z"`` .
        :param max_size: The maximum size of the Auto Scaling group.
        :param min_size: The minimum size of the Auto Scaling group.
        :param recurrence: The recurring schedule for this action. This format consists of five fields separated by white spaces: [Minute] [Hour] [Day_of_Month] [Month_of_Year] [Day_of_Week]. The value must be in quotes (for example, ``"30 0 1 1,6,12 *"`` ). For more information about this format, see `Crontab <https://docs.aws.amazon.com/http://crontab.org>`_ . When ``StartTime`` and ``EndTime`` are specified with ``Recurrence`` , they form the boundaries of when the recurring action starts and stops. Cron expressions use Universal Coordinated Time (UTC) by default.
        :param start_time: The date and time for this action to start, in YYYY-MM-DDThh:mm:ssZ format in UTC/GMT only and in quotes (for example, ``"2021-06-01T00:00:00Z"`` ). If you specify ``Recurrence`` and ``StartTime`` , Amazon EC2 Auto Scaling performs the action at this time, and then performs the action based on the specified recurrence.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as ``Etc/GMT+9`` or ``Pacific/Tahiti`` ). For more information, see `https://en.wikipedia.org/wiki/List_of_tz_database_time_zones <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/List_of_tz_database_time_zones>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            cfn_scheduled_action_props = autoscaling.CfnScheduledActionProps(
                auto_scaling_group_name="autoScalingGroupName",
            
                # the properties below are optional
                desired_capacity=123,
                end_time="endTime",
                max_size=123,
                min_size=123,
                recurrence="recurrence",
                start_time="startTime",
                time_zone="timeZone"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8337f4190404f5d486535d4fb3527dc3a0756a9c610753786846ddda210ebb1b)
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument end_time", value=end_time, expected_type=type_hints["end_time"])
            check_type(argname="argument max_size", value=max_size, expected_type=type_hints["max_size"])
            check_type(argname="argument min_size", value=min_size, expected_type=type_hints["min_size"])
            check_type(argname="argument recurrence", value=recurrence, expected_type=type_hints["recurrence"])
            check_type(argname="argument start_time", value=start_time, expected_type=type_hints["start_time"])
            check_type(argname="argument time_zone", value=time_zone, expected_type=type_hints["time_zone"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auto_scaling_group_name": auto_scaling_group_name,
        }
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if end_time is not None:
            self._values["end_time"] = end_time
        if max_size is not None:
            self._values["max_size"] = max_size
        if min_size is not None:
            self._values["min_size"] = min_size
        if recurrence is not None:
            self._values["recurrence"] = recurrence
        if start_time is not None:
            self._values["start_time"] = start_time
        if time_zone is not None:
            self._values["time_zone"] = time_zone

    @builtins.property
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-autoscalinggroupname
        '''
        result = self._values.get("auto_scaling_group_name")
        assert result is not None, "Required property 'auto_scaling_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''The desired capacity is the initial capacity of the Auto Scaling group after the scheduled action runs and the capacity it attempts to maintain.

        It can scale beyond this capacity if you add more scaling conditions.
        .. epigraph::

           You must specify at least one of the following properties: ``MaxSize`` , ``MinSize`` , or ``DesiredCapacity`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-desiredcapacity
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def end_time(self) -> typing.Optional[builtins.str]:
        '''The date and time for the recurring schedule to end, in UTC.

        For example, ``"2021-06-01T00:00:00Z"`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-endtime
        '''
        result = self._values.get("end_time")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum size of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-maxsize
        '''
        result = self._values.get("max_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_size(self) -> typing.Optional[jsii.Number]:
        '''The minimum size of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-minsize
        '''
        result = self._values.get("min_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def recurrence(self) -> typing.Optional[builtins.str]:
        '''The recurring schedule for this action.

        This format consists of five fields separated by white spaces: [Minute] [Hour] [Day_of_Month] [Month_of_Year] [Day_of_Week]. The value must be in quotes (for example, ``"30 0 1 1,6,12 *"`` ). For more information about this format, see `Crontab <https://docs.aws.amazon.com/http://crontab.org>`_ .

        When ``StartTime`` and ``EndTime`` are specified with ``Recurrence`` , they form the boundaries of when the recurring action starts and stops.

        Cron expressions use Universal Coordinated Time (UTC) by default.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-recurrence
        '''
        result = self._values.get("recurrence")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def start_time(self) -> typing.Optional[builtins.str]:
        '''The date and time for this action to start, in YYYY-MM-DDThh:mm:ssZ format in UTC/GMT only and in quotes (for example, ``"2021-06-01T00:00:00Z"`` ).

        If you specify ``Recurrence`` and ``StartTime`` , Amazon EC2 Auto Scaling performs the action at this time, and then performs the action based on the specified recurrence.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-starttime
        '''
        result = self._values.get("start_time")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def time_zone(self) -> typing.Optional[builtins.str]:
        '''Specifies the time zone for a cron expression.

        If a time zone is not provided, UTC is used by default.

        Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as ``Etc/GMT+9`` or ``Pacific/Tahiti`` ). For more information, see `https://en.wikipedia.org/wiki/List_of_tz_database_time_zones <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/List_of_tz_database_time_zones>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-scheduledaction.html#cfn-autoscaling-scheduledaction-timezone
        '''
        result = self._values.get("time_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnScheduledActionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnWarmPool(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnWarmPool",
):
    '''A CloudFormation ``AWS::AutoScaling::WarmPool``.

    The ``AWS::AutoScaling::WarmPool`` resource creates a pool of pre-initialized EC2 instances that sits alongside the Auto Scaling group. Whenever your application needs to scale out, the Auto Scaling group can draw on the warm pool to meet its new desired capacity.

    When you create a warm pool, you can define a minimum size. When your Auto Scaling group scales out and the size of the warm pool shrinks, Amazon EC2 Auto Scaling launches new instances into the warm pool to maintain its minimum size.

    For more information, see `Warm pools for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .
    .. epigraph::

       CloudFormation supports the ``UpdatePolicy`` attribute for Auto Scaling groups. During an update, if ``UpdatePolicy`` is set to ``AutoScalingRollingUpdate`` , CloudFormation replaces ``InService`` instances only. Instances in the warm pool are not replaced. The difference in which instances are replaced can potentially result in different instance configurations after the stack update completes. If ``UpdatePolicy`` is set to ``AutoScalingReplacingUpdate`` , you do not encounter this issue because CloudFormation replaces both the Auto Scaling group and the warm pool.

    :cloudformationResource: AWS::AutoScaling::WarmPool
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        cfn_warm_pool = autoscaling.CfnWarmPool(self, "MyCfnWarmPool",
            auto_scaling_group_name="autoScalingGroupName",
        
            # the properties below are optional
            instance_reuse_policy=autoscaling.CfnWarmPool.InstanceReusePolicyProperty(
                reuse_on_scale_in=False
            ),
            max_group_prepared_capacity=123,
            min_size=123,
            pool_state="poolState"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group_name: builtins.str,
        instance_reuse_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnWarmPool.InstanceReusePolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        max_group_prepared_capacity: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        pool_state: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AutoScaling::WarmPool``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param instance_reuse_policy: Indicates whether instances in the Auto Scaling group can be returned to the warm pool on scale in. The default is to terminate instances in the Auto Scaling group when the group scales in.
        :param max_group_prepared_capacity: Specifies the maximum number of instances that are allowed to be in the warm pool or in any state except ``Terminated`` for the Auto Scaling group. This is an optional property. Specify it only if you do not want the warm pool size to be determined by the difference between the group's maximum capacity and its desired capacity. .. epigraph:: If a value for ``MaxGroupPreparedCapacity`` is not specified, Amazon EC2 Auto Scaling launches and maintains the difference between the group's maximum capacity and its desired capacity. If you specify a value for ``MaxGroupPreparedCapacity`` , Amazon EC2 Auto Scaling uses the difference between the ``MaxGroupPreparedCapacity`` and the desired capacity instead. The size of the warm pool is dynamic. Only when ``MaxGroupPreparedCapacity`` and ``MinSize`` are set to the same value does the warm pool have an absolute size. If the desired capacity of the Auto Scaling group is higher than the ``MaxGroupPreparedCapacity`` , the capacity of the warm pool is 0, unless you specify a value for ``MinSize`` . To remove a value that you previously set, include the property but specify -1 for the value.
        :param min_size: Specifies the minimum number of instances to maintain in the warm pool. This helps you to ensure that there is always a certain number of warmed instances available to handle traffic spikes. Defaults to 0 if not specified.
        :param pool_state: Sets the instance state to transition to after the lifecycle actions are complete. Default is ``Stopped`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__11b655705f40e648d013ae07051251b8272b9b017ab829c80340f5640de46e5d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnWarmPoolProps(
            auto_scaling_group_name=auto_scaling_group_name,
            instance_reuse_policy=instance_reuse_policy,
            max_group_prepared_capacity=max_group_prepared_capacity,
            min_size=min_size,
            pool_state=pool_state,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__336e12b82b2e6721e49ed5add717ba97e08d6de95f9768ee47c57cd39f9567ac)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dbbbeb184e238f3d0fd2119f582a6c5a74873218141401d40caedf519a0add22)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-autoscalinggroupname
        '''
        return typing.cast(builtins.str, jsii.get(self, "autoScalingGroupName"))

    @auto_scaling_group_name.setter
    def auto_scaling_group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5fe7145b34e7b316830787a695baac4a9044eb1ab01b142ea8cfa75e799f5133)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoScalingGroupName", value)

    @builtins.property
    @jsii.member(jsii_name="instanceReusePolicy")
    def instance_reuse_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnWarmPool.InstanceReusePolicyProperty"]]:
        '''Indicates whether instances in the Auto Scaling group can be returned to the warm pool on scale in.

        The default is to terminate instances in the Auto Scaling group when the group scales in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-instancereusepolicy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnWarmPool.InstanceReusePolicyProperty"]], jsii.get(self, "instanceReusePolicy"))

    @instance_reuse_policy.setter
    def instance_reuse_policy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnWarmPool.InstanceReusePolicyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e282eb0fff0146c783a0a0cf3c1a6bfd7c1638906a490e7da16a58d31f63e70)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceReusePolicy", value)

    @builtins.property
    @jsii.member(jsii_name="maxGroupPreparedCapacity")
    def max_group_prepared_capacity(self) -> typing.Optional[jsii.Number]:
        '''Specifies the maximum number of instances that are allowed to be in the warm pool or in any state except ``Terminated`` for the Auto Scaling group.

        This is an optional property. Specify it only if you do not want the warm pool size to be determined by the difference between the group's maximum capacity and its desired capacity.
        .. epigraph::

           If a value for ``MaxGroupPreparedCapacity`` is not specified, Amazon EC2 Auto Scaling launches and maintains the difference between the group's maximum capacity and its desired capacity. If you specify a value for ``MaxGroupPreparedCapacity`` , Amazon EC2 Auto Scaling uses the difference between the ``MaxGroupPreparedCapacity`` and the desired capacity instead.

           The size of the warm pool is dynamic. Only when ``MaxGroupPreparedCapacity`` and ``MinSize`` are set to the same value does the warm pool have an absolute size.

        If the desired capacity of the Auto Scaling group is higher than the ``MaxGroupPreparedCapacity`` , the capacity of the warm pool is 0, unless you specify a value for ``MinSize`` . To remove a value that you previously set, include the property but specify -1 for the value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-maxgrouppreparedcapacity
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxGroupPreparedCapacity"))

    @max_group_prepared_capacity.setter
    def max_group_prepared_capacity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b58451a7dfd60fe98c4d41ee8e47ad321ecdf032c681e2efe7d6a509319ca9ef)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxGroupPreparedCapacity", value)

    @builtins.property
    @jsii.member(jsii_name="minSize")
    def min_size(self) -> typing.Optional[jsii.Number]:
        '''Specifies the minimum number of instances to maintain in the warm pool.

        This helps you to ensure that there is always a certain number of warmed instances available to handle traffic spikes. Defaults to 0 if not specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-minsize
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minSize"))

    @min_size.setter
    def min_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__43977a08e37915c5fe1a54890d58173d48bfc0d2e0a878326b384bb4c8ff80d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minSize", value)

    @builtins.property
    @jsii.member(jsii_name="poolState")
    def pool_state(self) -> typing.Optional[builtins.str]:
        '''Sets the instance state to transition to after the lifecycle actions are complete.

        Default is ``Stopped`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-poolstate
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "poolState"))

    @pool_state.setter
    def pool_state(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24bb436b66554232b9c70552368d17246634059370b0dbc27a1029d734d756a1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "poolState", value)

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_autoscaling.CfnWarmPool.InstanceReusePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"reuse_on_scale_in": "reuseOnScaleIn"},
    )
    class InstanceReusePolicyProperty:
        def __init__(
            self,
            *,
            reuse_on_scale_in: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''A structure that specifies an instance reuse policy for the ``InstanceReusePolicy`` property of the `AWS::AutoScaling::WarmPool <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html>`_ resource.

            For more information, see `Warm pools for Amazon EC2 Auto Scaling <https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-warm-pools.html>`_ in the *Amazon EC2 Auto Scaling User Guide* .

            :param reuse_on_scale_in: Specifies whether instances in the Auto Scaling group can be returned to the warm pool on scale in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-warmpool-instancereusepolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_autoscaling as autoscaling
                
                instance_reuse_policy_property = autoscaling.CfnWarmPool.InstanceReusePolicyProperty(
                    reuse_on_scale_in=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__53e0e158b9df1836e1187a74efd57221a6a3d593283d041862ad0fad1f8bbe41)
                check_type(argname="argument reuse_on_scale_in", value=reuse_on_scale_in, expected_type=type_hints["reuse_on_scale_in"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if reuse_on_scale_in is not None:
                self._values["reuse_on_scale_in"] = reuse_on_scale_in

        @builtins.property
        def reuse_on_scale_in(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether instances in the Auto Scaling group can be returned to the warm pool on scale in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-autoscaling-warmpool-instancereusepolicy.html#cfn-autoscaling-warmpool-instancereusepolicy-reuseonscalein
            '''
            result = self._values.get("reuse_on_scale_in")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceReusePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CfnWarmPoolProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_name": "autoScalingGroupName",
        "instance_reuse_policy": "instanceReusePolicy",
        "max_group_prepared_capacity": "maxGroupPreparedCapacity",
        "min_size": "minSize",
        "pool_state": "poolState",
    },
)
class CfnWarmPoolProps:
    def __init__(
        self,
        *,
        auto_scaling_group_name: builtins.str,
        instance_reuse_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnWarmPool.InstanceReusePolicyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        max_group_prepared_capacity: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        pool_state: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnWarmPool``.

        :param auto_scaling_group_name: The name of the Auto Scaling group.
        :param instance_reuse_policy: Indicates whether instances in the Auto Scaling group can be returned to the warm pool on scale in. The default is to terminate instances in the Auto Scaling group when the group scales in.
        :param max_group_prepared_capacity: Specifies the maximum number of instances that are allowed to be in the warm pool or in any state except ``Terminated`` for the Auto Scaling group. This is an optional property. Specify it only if you do not want the warm pool size to be determined by the difference between the group's maximum capacity and its desired capacity. .. epigraph:: If a value for ``MaxGroupPreparedCapacity`` is not specified, Amazon EC2 Auto Scaling launches and maintains the difference between the group's maximum capacity and its desired capacity. If you specify a value for ``MaxGroupPreparedCapacity`` , Amazon EC2 Auto Scaling uses the difference between the ``MaxGroupPreparedCapacity`` and the desired capacity instead. The size of the warm pool is dynamic. Only when ``MaxGroupPreparedCapacity`` and ``MinSize`` are set to the same value does the warm pool have an absolute size. If the desired capacity of the Auto Scaling group is higher than the ``MaxGroupPreparedCapacity`` , the capacity of the warm pool is 0, unless you specify a value for ``MinSize`` . To remove a value that you previously set, include the property but specify -1 for the value.
        :param min_size: Specifies the minimum number of instances to maintain in the warm pool. This helps you to ensure that there is always a certain number of warmed instances available to handle traffic spikes. Defaults to 0 if not specified.
        :param pool_state: Sets the instance state to transition to after the lifecycle actions are complete. Default is ``Stopped`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            cfn_warm_pool_props = autoscaling.CfnWarmPoolProps(
                auto_scaling_group_name="autoScalingGroupName",
            
                # the properties below are optional
                instance_reuse_policy=autoscaling.CfnWarmPool.InstanceReusePolicyProperty(
                    reuse_on_scale_in=False
                ),
                max_group_prepared_capacity=123,
                min_size=123,
                pool_state="poolState"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__648cf8c58fe1c3cd5125c280b61fa42684c7000ce58e8cff5213ddb0d0547be0)
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument instance_reuse_policy", value=instance_reuse_policy, expected_type=type_hints["instance_reuse_policy"])
            check_type(argname="argument max_group_prepared_capacity", value=max_group_prepared_capacity, expected_type=type_hints["max_group_prepared_capacity"])
            check_type(argname="argument min_size", value=min_size, expected_type=type_hints["min_size"])
            check_type(argname="argument pool_state", value=pool_state, expected_type=type_hints["pool_state"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auto_scaling_group_name": auto_scaling_group_name,
        }
        if instance_reuse_policy is not None:
            self._values["instance_reuse_policy"] = instance_reuse_policy
        if max_group_prepared_capacity is not None:
            self._values["max_group_prepared_capacity"] = max_group_prepared_capacity
        if min_size is not None:
            self._values["min_size"] = min_size
        if pool_state is not None:
            self._values["pool_state"] = pool_state

    @builtins.property
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the Auto Scaling group.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-autoscalinggroupname
        '''
        result = self._values.get("auto_scaling_group_name")
        assert result is not None, "Required property 'auto_scaling_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_reuse_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnWarmPool.InstanceReusePolicyProperty]]:
        '''Indicates whether instances in the Auto Scaling group can be returned to the warm pool on scale in.

        The default is to terminate instances in the Auto Scaling group when the group scales in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-instancereusepolicy
        '''
        result = self._values.get("instance_reuse_policy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnWarmPool.InstanceReusePolicyProperty]], result)

    @builtins.property
    def max_group_prepared_capacity(self) -> typing.Optional[jsii.Number]:
        '''Specifies the maximum number of instances that are allowed to be in the warm pool or in any state except ``Terminated`` for the Auto Scaling group.

        This is an optional property. Specify it only if you do not want the warm pool size to be determined by the difference between the group's maximum capacity and its desired capacity.
        .. epigraph::

           If a value for ``MaxGroupPreparedCapacity`` is not specified, Amazon EC2 Auto Scaling launches and maintains the difference between the group's maximum capacity and its desired capacity. If you specify a value for ``MaxGroupPreparedCapacity`` , Amazon EC2 Auto Scaling uses the difference between the ``MaxGroupPreparedCapacity`` and the desired capacity instead.

           The size of the warm pool is dynamic. Only when ``MaxGroupPreparedCapacity`` and ``MinSize`` are set to the same value does the warm pool have an absolute size.

        If the desired capacity of the Auto Scaling group is higher than the ``MaxGroupPreparedCapacity`` , the capacity of the warm pool is 0, unless you specify a value for ``MinSize`` . To remove a value that you previously set, include the property but specify -1 for the value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-maxgrouppreparedcapacity
        '''
        result = self._values.get("max_group_prepared_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_size(self) -> typing.Optional[jsii.Number]:
        '''Specifies the minimum number of instances to maintain in the warm pool.

        This helps you to ensure that there is always a certain number of warmed instances available to handle traffic spikes. Defaults to 0 if not specified.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-minsize
        '''
        result = self._values.get("min_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def pool_state(self) -> typing.Optional[builtins.str]:
        '''Sets the instance state to transition to after the lifecycle actions are complete.

        Default is ``Stopped`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-autoscaling-warmpool.html#cfn-autoscaling-warmpool-poolstate
        '''
        result = self._values.get("pool_state")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnWarmPoolProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CommonAutoScalingGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "allow_all_outbound": "allowAllOutbound",
        "associate_public_ip_address": "associatePublicIpAddress",
        "auto_scaling_group_name": "autoScalingGroupName",
        "block_devices": "blockDevices",
        "capacity_rebalance": "capacityRebalance",
        "cooldown": "cooldown",
        "default_instance_warmup": "defaultInstanceWarmup",
        "desired_capacity": "desiredCapacity",
        "group_metrics": "groupMetrics",
        "health_check": "healthCheck",
        "ignore_unmodified_size_properties": "ignoreUnmodifiedSizeProperties",
        "instance_monitoring": "instanceMonitoring",
        "key_name": "keyName",
        "max_capacity": "maxCapacity",
        "max_instance_lifetime": "maxInstanceLifetime",
        "min_capacity": "minCapacity",
        "new_instances_protected_from_scale_in": "newInstancesProtectedFromScaleIn",
        "notifications": "notifications",
        "signals": "signals",
        "spot_price": "spotPrice",
        "ssm_session_permissions": "ssmSessionPermissions",
        "termination_policies": "terminationPolicies",
        "update_policy": "updatePolicy",
        "vpc_subnets": "vpcSubnets",
    },
)
class CommonAutoScalingGroupProps:
    def __init__(
        self,
        *,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        associate_public_ip_address: typing.Optional[builtins.bool] = None,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence[typing.Union[BlockDevice, typing.Dict[builtins.str, typing.Any]]]] = None,
        capacity_rebalance: typing.Optional[builtins.bool] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        default_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        desired_capacity: typing.Optional[jsii.Number] = None,
        group_metrics: typing.Optional[typing.Sequence["GroupMetrics"]] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        ignore_unmodified_size_properties: typing.Optional[builtins.bool] = None,
        instance_monitoring: typing.Optional["Monitoring"] = None,
        key_name: typing.Optional[builtins.str] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        max_instance_lifetime: typing.Optional[_Duration_4839e8c3] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        new_instances_protected_from_scale_in: typing.Optional[builtins.bool] = None,
        notifications: typing.Optional[typing.Sequence[typing.Union["NotificationConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        signals: typing.Optional["Signals"] = None,
        spot_price: typing.Optional[builtins.str] = None,
        ssm_session_permissions: typing.Optional[builtins.bool] = None,
        termination_policies: typing.Optional[typing.Sequence["TerminationPolicy"]] = None,
        update_policy: typing.Optional["UpdatePolicy"] = None,
        vpc_subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Basic properties of an AutoScalingGroup, except the exact machines to run and where they should run.

        Constructs that want to create AutoScalingGroups can inherit
        this interface and specialize the essential parts in various ways.

        :param allow_all_outbound: Whether the instances can initiate connections to anywhere by default. Default: true
        :param associate_public_ip_address: Whether instances in the Auto Scaling Group should have public IP addresses associated with them. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Use subnet setting.
        :param auto_scaling_group_name: The name of the Auto Scaling group. This name must be unique per Region per account. Default: - Auto generated by CloudFormation
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Uses the block device mapping of the AMI
        :param capacity_rebalance: Indicates whether Capacity Rebalancing is enabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. Default: false
        :param cooldown: Default scaling cooldown for this AutoScalingGroup. Default: Duration.minutes(5)
        :param default_instance_warmup: The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics. This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics, resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource consumption to become stable after an instance reaches the InService state. To optimize the performance of scaling policies that scale continuously, such as target tracking and step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds Default instance warmup will not be added if no value is specified Default: None
        :param desired_capacity: Initial amount of instances in the fleet. If this is set to a number, every deployment will reset the amount of instances to this number. It is recommended to leave this value blank. Default: minCapacity, and leave unchanged during deployment
        :param group_metrics: Enable monitoring for group metrics, these metrics describe the group rather than any of its instances. To report all group metrics use ``GroupMetrics.all()`` Group metrics are reported in a granularity of 1 minute at no additional charge. Default: - no group metrics will be reported
        :param health_check: Configuration for health checks. Default: - HealthCheck.ec2 with no grace period
        :param ignore_unmodified_size_properties: If the ASG has scheduled actions, don't reset unchanged group sizes. Only used if the ASG has scheduled actions (which may scale your ASG up or down regardless of cdk deployments). If true, the size of the group will only be reset if it has been changed in the CDK app. If false, the sizes will always be changed back to what they were in the CDK app on deployment. Default: true
        :param instance_monitoring: Controls whether instances in this group are launched with detailed or basic monitoring. When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Monitoring.DETAILED
        :param key_name: Name of SSH keypair to grant access to instances. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - No SSH access will be possible.
        :param max_capacity: Maximum number of instances in the fleet. Default: desiredCapacity
        :param max_instance_lifetime: The maximum amount of time that an instance can be in service. The maximum duration applies to all current and future instances in the group. As an instance approaches its maximum duration, it is terminated and replaced, and cannot be used again. You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value, leave this property undefined. Default: none
        :param min_capacity: Minimum number of instances in the fleet. Default: 1
        :param new_instances_protected_from_scale_in: Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. By default, Auto Scaling can terminate an instance at any time after launch when scaling in an Auto Scaling Group, subject to the group's termination policy. However, you may wish to protect newly-launched instances from being scaled in if they are going to run critical applications that should not be prematurely terminated. This flag must be enabled if the Auto Scaling Group will be associated with an ECS Capacity Provider with managed termination protection. Default: false
        :param notifications: Configure autoscaling group to send notifications about fleet changes to an SNS topic(s). Default: - No fleet change notifications will be sent.
        :param signals: Configure waiting for signals during deployment. Use this to pause the CloudFormation deployment to wait for the instances in the AutoScalingGroup to report successful startup during creation and updates. The UserData script needs to invoke ``cfn-signal`` with a success or failure code after it is done setting up the instance. Without waiting for signals, the CloudFormation deployment will proceed as soon as the AutoScalingGroup has been created or updated but before the instances in the group have been started. For example, to have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template: https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml Default: - Do not wait for signals
        :param spot_price: The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot market price. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: none
        :param ssm_session_permissions: Add SSM session permissions to the instance role. Setting this to ``true`` adds the necessary permissions to connect to the instance using SSM Session Manager. You can do this from the AWS Console. NOTE: Setting this flag to ``true`` may not be enough by itself. You must also use an AMI that comes with the SSM Agent, or install the SSM Agent yourself. See `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_ in the SSM Developer Guide. Default: false
        :param termination_policies: A policy or a list of policies that are used to select the instances to terminate. The policies are executed in the order that you list them. Default: - ``TerminationPolicy.DEFAULT``
        :param update_policy: What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        :param vpc_subnets: Where to place instances within the VPC. Default: - All Private subnets.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_sns as sns
            
            # block_device_volume: autoscaling.BlockDeviceVolume
            # group_metrics: autoscaling.GroupMetrics
            # health_check: autoscaling.HealthCheck
            # scaling_events: autoscaling.ScalingEvents
            # signals: autoscaling.Signals
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            # topic: sns.Topic
            # update_policy: autoscaling.UpdatePolicy
            
            common_auto_scaling_group_props = autoscaling.CommonAutoScalingGroupProps(
                allow_all_outbound=False,
                associate_public_ip_address=False,
                auto_scaling_group_name="autoScalingGroupName",
                block_devices=[autoscaling.BlockDevice(
                    device_name="deviceName",
                    volume=block_device_volume
                )],
                capacity_rebalance=False,
                cooldown=cdk.Duration.minutes(30),
                default_instance_warmup=cdk.Duration.minutes(30),
                desired_capacity=123,
                group_metrics=[group_metrics],
                health_check=health_check,
                ignore_unmodified_size_properties=False,
                instance_monitoring=autoscaling.Monitoring.BASIC,
                key_name="keyName",
                max_capacity=123,
                max_instance_lifetime=cdk.Duration.minutes(30),
                min_capacity=123,
                new_instances_protected_from_scale_in=False,
                notifications=[autoscaling.NotificationConfiguration(
                    topic=topic,
            
                    # the properties below are optional
                    scaling_events=scaling_events
                )],
                signals=signals,
                spot_price="spotPrice",
                ssm_session_permissions=False,
                termination_policies=[autoscaling.TerminationPolicy.ALLOCATION_STRATEGY],
                update_policy=update_policy,
                vpc_subnets=ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
                )
            )
        '''
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_e57d76df(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7df3cbef8de53a463241b6e60846fac6e519118c7118785f8f4efb6d6cff2c5)
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument associate_public_ip_address", value=associate_public_ip_address, expected_type=type_hints["associate_public_ip_address"])
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument block_devices", value=block_devices, expected_type=type_hints["block_devices"])
            check_type(argname="argument capacity_rebalance", value=capacity_rebalance, expected_type=type_hints["capacity_rebalance"])
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument default_instance_warmup", value=default_instance_warmup, expected_type=type_hints["default_instance_warmup"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument group_metrics", value=group_metrics, expected_type=type_hints["group_metrics"])
            check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
            check_type(argname="argument ignore_unmodified_size_properties", value=ignore_unmodified_size_properties, expected_type=type_hints["ignore_unmodified_size_properties"])
            check_type(argname="argument instance_monitoring", value=instance_monitoring, expected_type=type_hints["instance_monitoring"])
            check_type(argname="argument key_name", value=key_name, expected_type=type_hints["key_name"])
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument max_instance_lifetime", value=max_instance_lifetime, expected_type=type_hints["max_instance_lifetime"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            check_type(argname="argument new_instances_protected_from_scale_in", value=new_instances_protected_from_scale_in, expected_type=type_hints["new_instances_protected_from_scale_in"])
            check_type(argname="argument notifications", value=notifications, expected_type=type_hints["notifications"])
            check_type(argname="argument signals", value=signals, expected_type=type_hints["signals"])
            check_type(argname="argument spot_price", value=spot_price, expected_type=type_hints["spot_price"])
            check_type(argname="argument ssm_session_permissions", value=ssm_session_permissions, expected_type=type_hints["ssm_session_permissions"])
            check_type(argname="argument termination_policies", value=termination_policies, expected_type=type_hints["termination_policies"])
            check_type(argname="argument update_policy", value=update_policy, expected_type=type_hints["update_policy"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if associate_public_ip_address is not None:
            self._values["associate_public_ip_address"] = associate_public_ip_address
        if auto_scaling_group_name is not None:
            self._values["auto_scaling_group_name"] = auto_scaling_group_name
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if capacity_rebalance is not None:
            self._values["capacity_rebalance"] = capacity_rebalance
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if default_instance_warmup is not None:
            self._values["default_instance_warmup"] = default_instance_warmup
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if group_metrics is not None:
            self._values["group_metrics"] = group_metrics
        if health_check is not None:
            self._values["health_check"] = health_check
        if ignore_unmodified_size_properties is not None:
            self._values["ignore_unmodified_size_properties"] = ignore_unmodified_size_properties
        if instance_monitoring is not None:
            self._values["instance_monitoring"] = instance_monitoring
        if key_name is not None:
            self._values["key_name"] = key_name
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if max_instance_lifetime is not None:
            self._values["max_instance_lifetime"] = max_instance_lifetime
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if new_instances_protected_from_scale_in is not None:
            self._values["new_instances_protected_from_scale_in"] = new_instances_protected_from_scale_in
        if notifications is not None:
            self._values["notifications"] = notifications
        if signals is not None:
            self._values["signals"] = signals
        if spot_price is not None:
            self._values["spot_price"] = spot_price
        if ssm_session_permissions is not None:
            self._values["ssm_session_permissions"] = ssm_session_permissions
        if termination_policies is not None:
            self._values["termination_policies"] = termination_policies
        if update_policy is not None:
            self._values["update_policy"] = update_policy
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether the instances can initiate connections to anywhere by default.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def associate_public_ip_address(self) -> typing.Optional[builtins.bool]:
        '''Whether instances in the Auto Scaling Group should have public IP addresses associated with them.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Use subnet setting.
        '''
        result = self._values.get("associate_public_ip_address")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Auto Scaling group.

        This name must be unique per Region per account.

        :default: - Auto generated by CloudFormation
        '''
        result = self._values.get("auto_scaling_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List[BlockDevice]]:
        '''Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List[BlockDevice]], result)

    @builtins.property
    def capacity_rebalance(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether Capacity Rebalancing is enabled.

        When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling
        attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of
        interruption. After launching a new instance, it then terminates an old instance.

        :default: false

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-capacityrebalance
        '''
        result = self._values.get("capacity_rebalance")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Default scaling cooldown for this AutoScalingGroup.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def default_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics.

        This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics,
        resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource
        consumption to become stable after an instance reaches the InService state.

        To optimize the performance of scaling policies that scale continuously, such as target tracking and
        step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds

        Default instance warmup will not be added if no value is specified

        :default: None

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html
        '''
        result = self._values.get("default_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''Initial amount of instances in the fleet.

        If this is set to a number, every deployment will reset the amount of
        instances to this number. It is recommended to leave this value blank.

        :default: minCapacity, and leave unchanged during deployment

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def group_metrics(self) -> typing.Optional[typing.List["GroupMetrics"]]:
        '''Enable monitoring for group metrics, these metrics describe the group rather than any of its instances.

        To report all group metrics use ``GroupMetrics.all()``
        Group metrics are reported in a granularity of 1 minute at no additional charge.

        :default: - no group metrics will be reported
        '''
        result = self._values.get("group_metrics")
        return typing.cast(typing.Optional[typing.List["GroupMetrics"]], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''Configuration for health checks.

        :default: - HealthCheck.ec2 with no grace period
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def ignore_unmodified_size_properties(self) -> typing.Optional[builtins.bool]:
        '''If the ASG has scheduled actions, don't reset unchanged group sizes.

        Only used if the ASG has scheduled actions (which may scale your ASG up
        or down regardless of cdk deployments). If true, the size of the group
        will only be reset if it has been changed in the CDK app. If false, the
        sizes will always be changed back to what they were in the CDK app
        on deployment.

        :default: true
        '''
        result = self._values.get("ignore_unmodified_size_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_monitoring(self) -> typing.Optional["Monitoring"]:
        '''Controls whether instances in this group are launched with detailed or basic monitoring.

        When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account
        is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Monitoring.DETAILED

        :see: https://docs.aws.amazon.com/autoscaling/latest/userguide/as-instance-monitoring.html#enable-as-instance-metrics
        '''
        result = self._values.get("instance_monitoring")
        return typing.cast(typing.Optional["Monitoring"], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''Name of SSH keypair to grant access to instances.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - No SSH access will be possible.
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''Maximum number of instances in the fleet.

        :default: desiredCapacity
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_instance_lifetime(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum amount of time that an instance can be in service.

        The maximum duration applies
        to all current and future instances in the group. As an instance approaches its maximum duration,
        it is terminated and replaced, and cannot be used again.

        You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value,
        leave this property undefined.

        :default: none

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html
        '''
        result = self._values.get("max_instance_lifetime")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''Minimum number of instances in the fleet.

        :default: 1
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def new_instances_protected_from_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        By default, Auto Scaling can terminate an instance at any time after launch
        when scaling in an Auto Scaling Group, subject to the group's termination
        policy. However, you may wish to protect newly-launched instances from
        being scaled in if they are going to run critical applications that should
        not be prematurely terminated.

        This flag must be enabled if the Auto Scaling Group will be associated with
        an ECS Capacity Provider with managed termination protection.

        :default: false
        '''
        result = self._values.get("new_instances_protected_from_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def notifications(
        self,
    ) -> typing.Optional[typing.List["NotificationConfiguration"]]:
        '''Configure autoscaling group to send notifications about fleet changes to an SNS topic(s).

        :default: - No fleet change notifications will be sent.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        '''
        result = self._values.get("notifications")
        return typing.cast(typing.Optional[typing.List["NotificationConfiguration"]], result)

    @builtins.property
    def signals(self) -> typing.Optional["Signals"]:
        '''Configure waiting for signals during deployment.

        Use this to pause the CloudFormation deployment to wait for the instances
        in the AutoScalingGroup to report successful startup during
        creation and updates. The UserData script needs to invoke ``cfn-signal``
        with a success or failure code after it is done setting up the instance.

        Without waiting for signals, the CloudFormation deployment will proceed as
        soon as the AutoScalingGroup has been created or updated but before the
        instances in the group have been started.

        For example, to have instances wait for an Elastic Load Balancing health check before
        they signal success, add a health-check verification by using the
        cfn-init helper script. For an example, see the verify_instance_health
        command in the Auto Scaling rolling updates sample template:

        https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml

        :default: - Do not wait for signals
        '''
        result = self._values.get("signals")
        return typing.cast(typing.Optional["Signals"], result)

    @builtins.property
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are
        launched when the price you specify exceeds the current Spot market price.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: none
        '''
        result = self._values.get("spot_price")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ssm_session_permissions(self) -> typing.Optional[builtins.bool]:
        '''Add SSM session permissions to the instance role.

        Setting this to ``true`` adds the necessary permissions to connect
        to the instance using SSM Session Manager. You can do this
        from the AWS Console.

        NOTE: Setting this flag to ``true`` may not be enough by itself.
        You must also use an AMI that comes with the SSM Agent, or install
        the SSM Agent yourself. See
        `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_
        in the SSM Developer Guide.

        :default: false
        '''
        result = self._values.get("ssm_session_permissions")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def termination_policies(self) -> typing.Optional[typing.List["TerminationPolicy"]]:
        '''A policy or a list of policies that are used to select the instances to terminate.

        The policies are executed in the order that you list them.

        :default: - ``TerminationPolicy.DEFAULT``

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html
        '''
        result = self._values.get("termination_policies")
        return typing.cast(typing.Optional[typing.List["TerminationPolicy"]], result)

    @builtins.property
    def update_policy(self) -> typing.Optional["UpdatePolicy"]:
        '''What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        '''
        result = self._values.get("update_policy")
        return typing.cast(typing.Optional["UpdatePolicy"], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Where to place instances within the VPC.

        :default: - All Private subnets.
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonAutoScalingGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CpuUtilizationScalingProps",
    jsii_struct_bases=[BaseTargetTrackingProps],
    name_mapping={
        "cooldown": "cooldown",
        "disable_scale_in": "disableScaleIn",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "target_utilization_percent": "targetUtilizationPercent",
    },
)
class CpuUtilizationScalingProps(BaseTargetTrackingProps):
    def __init__(
        self,
        *,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        target_utilization_percent: jsii.Number,
    ) -> None:
        '''Properties for enabling scaling based on CPU utilization.

        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        :param target_utilization_percent: Target average CPU utilization across the task.

        :exampleMetadata: infused

        Example::

            # auto_scaling_group: autoscaling.AutoScalingGroup
            
            
            auto_scaling_group.scale_on_cpu_utilization("KeepSpareCPU",
                target_utilization_percent=50
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ab1197c14846ec6b5c20ba6ac3608b3f3b01ef6d827a983ae1e3534837dd92c)
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument target_utilization_percent", value=target_utilization_percent, expected_type=type_hints["target_utilization_percent"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "target_utilization_percent": target_utilization_percent,
        }
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Period after a scaling completes before another scaling activity can start.

        :default: - The default cooldown configured on the AutoScalingGroup.
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the autoscaling group. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        group.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: - Same as the cooldown.
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def target_utilization_percent(self) -> jsii.Number:
        '''Target average CPU utilization across the task.'''
        result = self._values.get("target_utilization_percent")
        assert result is not None, "Required property 'target_utilization_percent' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CpuUtilizationScalingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.CronOptions",
    jsii_struct_bases=[],
    name_mapping={
        "day": "day",
        "hour": "hour",
        "minute": "minute",
        "month": "month",
        "week_day": "weekDay",
    },
)
class CronOptions:
    def __init__(
        self,
        *,
        day: typing.Optional[builtins.str] = None,
        hour: typing.Optional[builtins.str] = None,
        minute: typing.Optional[builtins.str] = None,
        month: typing.Optional[builtins.str] = None,
        week_day: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options to configure a cron expression.

        All fields are strings so you can use complex expressions. Absence of
        a field implies '*' or '?', whichever one is appropriate.

        :param day: The day of the month to run this rule at. Default: - Every day of the month
        :param hour: The hour to run this rule at. Default: - Every hour
        :param minute: The minute to run this rule at. Default: - Every minute
        :param month: The month to run this rule at. Default: - Every month
        :param week_day: The day of the week to run this rule at. Default: - Any day of the week

        :see: http://crontab.org/
        :exampleMetadata: infused

        Example::

            # auto_scaling_group: autoscaling.AutoScalingGroup
            
            
            auto_scaling_group.scale_on_schedule("PrescaleInTheMorning",
                schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
                min_capacity=20
            )
            
            auto_scaling_group.scale_on_schedule("AllowDownscalingAtNight",
                schedule=autoscaling.Schedule.cron(hour="20", minute="0"),
                min_capacity=1
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff2ba980d042c09d15d4b7c8a2897accea72726b7fd7c61c4e35722c413018e2)
            check_type(argname="argument day", value=day, expected_type=type_hints["day"])
            check_type(argname="argument hour", value=hour, expected_type=type_hints["hour"])
            check_type(argname="argument minute", value=minute, expected_type=type_hints["minute"])
            check_type(argname="argument month", value=month, expected_type=type_hints["month"])
            check_type(argname="argument week_day", value=week_day, expected_type=type_hints["week_day"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if day is not None:
            self._values["day"] = day
        if hour is not None:
            self._values["hour"] = hour
        if minute is not None:
            self._values["minute"] = minute
        if month is not None:
            self._values["month"] = month
        if week_day is not None:
            self._values["week_day"] = week_day

    @builtins.property
    def day(self) -> typing.Optional[builtins.str]:
        '''The day of the month to run this rule at.

        :default: - Every day of the month
        '''
        result = self._values.get("day")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def hour(self) -> typing.Optional[builtins.str]:
        '''The hour to run this rule at.

        :default: - Every hour
        '''
        result = self._values.get("hour")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def minute(self) -> typing.Optional[builtins.str]:
        '''The minute to run this rule at.

        :default: - Every minute
        '''
        result = self._values.get("minute")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def month(self) -> typing.Optional[builtins.str]:
        '''The month to run this rule at.

        :default: - Every month
        '''
        result = self._values.get("month")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def week_day(self) -> typing.Optional[builtins.str]:
        '''The day of the week to run this rule at.

        :default: - Any day of the week
        '''
        result = self._values.get("week_day")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CronOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.DefaultResult")
class DefaultResult(enum.Enum):
    CONTINUE = "CONTINUE"
    ABANDON = "ABANDON"


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.EbsDeviceOptionsBase",
    jsii_struct_bases=[],
    name_mapping={
        "delete_on_termination": "deleteOnTermination",
        "iops": "iops",
        "throughput": "throughput",
        "volume_type": "volumeType",
    },
)
class EbsDeviceOptionsBase:
    def __init__(
        self,
        *,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        throughput: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> None:
        '''Base block device options for an EBS volume.

        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1`` The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for ``EbsDeviceVolumeType.IO1``
        :param throughput: The throughput that the volume supports, in MiB/s Takes a minimum of 125 and maximum of 1000. Default: - 125 MiB/s. Only valid on gp3 volumes.
        :param volume_type: The EBS volume type. Default: ``EbsDeviceVolumeType.GP2``

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            ebs_device_options_base = autoscaling.EbsDeviceOptionsBase(
                delete_on_termination=False,
                iops=123,
                throughput=123,
                volume_type=autoscaling.EbsDeviceVolumeType.STANDARD
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__96052136c5a766e5a075c844b44c04d660a7a8b58bc20689cb18dc69ec4f0b04)
            check_type(argname="argument delete_on_termination", value=delete_on_termination, expected_type=type_hints["delete_on_termination"])
            check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            check_type(argname="argument throughput", value=throughput, expected_type=type_hints["throughput"])
            check_type(argname="argument volume_type", value=volume_type, expected_type=type_hints["volume_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if delete_on_termination is not None:
            self._values["delete_on_termination"] = delete_on_termination
        if iops is not None:
            self._values["iops"] = iops
        if throughput is not None:
            self._values["throughput"] = throughput
        if volume_type is not None:
            self._values["volume_type"] = volume_type

    @builtins.property
    def delete_on_termination(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether to delete the volume when the instance is terminated.

        :default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        '''
        result = self._values.get("delete_on_termination")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The number of I/O operations per second (IOPS) to provision for the volume.

        Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1``

        The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS,
        you need at least 100 GiB storage on the volume.

        :default: - none, required for ``EbsDeviceVolumeType.IO1``

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
        '''
        result = self._values.get("iops")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def throughput(self) -> typing.Optional[jsii.Number]:
        '''The throughput that the volume supports, in MiB/s Takes a minimum of 125 and maximum of 1000.

        :default: - 125 MiB/s. Only valid on gp3 volumes.

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
        '''
        result = self._values.get("throughput")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def volume_type(self) -> typing.Optional["EbsDeviceVolumeType"]:
        '''The EBS volume type.

        :default: ``EbsDeviceVolumeType.GP2``

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
        '''
        result = self._values.get("volume_type")
        return typing.cast(typing.Optional["EbsDeviceVolumeType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EbsDeviceOptionsBase(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.EbsDeviceSnapshotOptions",
    jsii_struct_bases=[EbsDeviceOptionsBase],
    name_mapping={
        "delete_on_termination": "deleteOnTermination",
        "iops": "iops",
        "throughput": "throughput",
        "volume_type": "volumeType",
        "volume_size": "volumeSize",
    },
)
class EbsDeviceSnapshotOptions(EbsDeviceOptionsBase):
    def __init__(
        self,
        *,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        throughput: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
        volume_size: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Block device options for an EBS volume created from a snapshot.

        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1`` The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for ``EbsDeviceVolumeType.IO1``
        :param throughput: The throughput that the volume supports, in MiB/s Takes a minimum of 125 and maximum of 1000. Default: - 125 MiB/s. Only valid on gp3 volumes.
        :param volume_type: The EBS volume type. Default: ``EbsDeviceVolumeType.GP2``
        :param volume_size: The volume size, in Gibibytes (GiB). If you specify volumeSize, it must be equal or greater than the size of the snapshot. Default: - The snapshot size

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            ebs_device_snapshot_options = autoscaling.EbsDeviceSnapshotOptions(
                delete_on_termination=False,
                iops=123,
                throughput=123,
                volume_size=123,
                volume_type=autoscaling.EbsDeviceVolumeType.STANDARD
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__19e899c50f45e1e93ed8e6ceab2c1abd8b33cbf64c181c33cde0635f29175f92)
            check_type(argname="argument delete_on_termination", value=delete_on_termination, expected_type=type_hints["delete_on_termination"])
            check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            check_type(argname="argument throughput", value=throughput, expected_type=type_hints["throughput"])
            check_type(argname="argument volume_type", value=volume_type, expected_type=type_hints["volume_type"])
            check_type(argname="argument volume_size", value=volume_size, expected_type=type_hints["volume_size"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if delete_on_termination is not None:
            self._values["delete_on_termination"] = delete_on_termination
        if iops is not None:
            self._values["iops"] = iops
        if throughput is not None:
            self._values["throughput"] = throughput
        if volume_type is not None:
            self._values["volume_type"] = volume_type
        if volume_size is not None:
            self._values["volume_size"] = volume_size

    @builtins.property
    def delete_on_termination(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether to delete the volume when the instance is terminated.

        :default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        '''
        result = self._values.get("delete_on_termination")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def iops(self) -> typing.Optional[jsii.Number]:
        '''The number of I/O operations per second (IOPS) to provision for the volume.

        Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1``

        The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS,
        you need at least 100 GiB storage on the volume.

        :default: - none, required for ``EbsDeviceVolumeType.IO1``

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
        '''
        result = self._values.get("iops")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def throughput(self) -> typing.Optional[jsii.Number]:
        '''The throughput that the volume supports, in MiB/s Takes a minimum of 125 and maximum of 1000.

        :default: - 125 MiB/s. Only valid on gp3 volumes.

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
        '''
        result = self._values.get("throughput")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def volume_type(self) -> typing.Optional["EbsDeviceVolumeType"]:
        '''The EBS volume type.

        :default: ``EbsDeviceVolumeType.GP2``

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
        '''
        result = self._values.get("volume_type")
        return typing.cast(typing.Optional["EbsDeviceVolumeType"], result)

    @builtins.property
    def volume_size(self) -> typing.Optional[jsii.Number]:
        '''The volume size, in Gibibytes (GiB).

        If you specify volumeSize, it must be equal or greater than the size of the snapshot.

        :default: - The snapshot size
        '''
        result = self._values.get("volume_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EbsDeviceSnapshotOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.EbsDeviceVolumeType")
class EbsDeviceVolumeType(enum.Enum):
    '''Supported EBS volume types for blockDevices.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        # machine_image: ec2.IMachineImage
        
        
        auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
            block_devices=[autoscaling.BlockDevice(
                device_name="gp3-volume",
                volume=autoscaling.BlockDeviceVolume.ebs(15,
                    volume_type=autoscaling.EbsDeviceVolumeType.GP3,
                    throughput=125
                )
            )
            ]
        )
    '''

    STANDARD = "STANDARD"
    '''Magnetic.'''
    IO1 = "IO1"
    '''Provisioned IOPS SSD - IO1.'''
    GP2 = "GP2"
    '''General Purpose SSD - GP2.'''
    GP3 = "GP3"
    '''General Purpose SSD - GP3.'''
    ST1 = "ST1"
    '''Throughput Optimized HDD.'''
    SC1 = "SC1"
    '''Cold HDD.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.Ec2HealthCheckOptions",
    jsii_struct_bases=[],
    name_mapping={"grace": "grace"},
)
class Ec2HealthCheckOptions:
    def __init__(self, *, grace: typing.Optional[_Duration_4839e8c3] = None) -> None:
        '''EC2 Heath check options.

        :param grace: Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service. Default: Duration.seconds(0)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            
            ec2_health_check_options = autoscaling.Ec2HealthCheckOptions(
                grace=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ed1c4902bffca62eb69df85375bd1cbc0221879fd972e2805b0341949af215f)
            check_type(argname="argument grace", value=grace, expected_type=type_hints["grace"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if grace is not None:
            self._values["grace"] = grace

    @builtins.property
    def grace(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service.

        :default: Duration.seconds(0)
        '''
        result = self._values.get("grace")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Ec2HealthCheckOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.ElbHealthCheckOptions",
    jsii_struct_bases=[],
    name_mapping={"grace": "grace"},
)
class ElbHealthCheckOptions:
    def __init__(self, *, grace: _Duration_4839e8c3) -> None:
        '''ELB Heath check options.

        :param grace: Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service. This option is required for ELB health checks.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            
            elb_health_check_options = autoscaling.ElbHealthCheckOptions(
                grace=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff6ee0031db4237a2d504a88db32c061b20545da01d54dbae951f9d5b57a3eb3)
            check_type(argname="argument grace", value=grace, expected_type=type_hints["grace"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "grace": grace,
        }

    @builtins.property
    def grace(self) -> _Duration_4839e8c3:
        '''Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service.

        This option is required for ELB health checks.
        '''
        result = self._values.get("grace")
        assert result is not None, "Required property 'grace' is missing"
        return typing.cast(_Duration_4839e8c3, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ElbHealthCheckOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GroupMetric(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.GroupMetric",
):
    '''Group metrics that an Auto Scaling group sends to Amazon CloudWatch.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        # machine_image: ec2.IMachineImage
        
        
        # Enable monitoring of all group metrics
        autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
        
            # ...
        
            group_metrics=[autoscaling.GroupMetrics.all()]
        )
        
        # Enable monitoring for a subset of group metrics
        autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
        
            # ...
        
            group_metrics=[autoscaling.GroupMetrics(autoscaling.GroupMetric.MIN_SIZE, autoscaling.GroupMetric.MAX_SIZE)]
        )
    '''

    def __init__(self, name: builtins.str) -> None:
        '''
        :param name: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d1e1f76d6fee74688fd6dc15de2ae06cba76e8853f9c6bdc4e02f9cfe561cf16)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        jsii.create(self.__class__, self, [name])

    @jsii.python.classproperty
    @jsii.member(jsii_name="DESIRED_CAPACITY")
    def DESIRED_CAPACITY(cls) -> "GroupMetric":
        '''The number of instances that the Auto Scaling group attempts to maintain.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "DESIRED_CAPACITY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IN_SERVICE_INSTANCES")
    def IN_SERVICE_INSTANCES(cls) -> "GroupMetric":
        '''The number of instances that are running as part of the Auto Scaling group This metric does not include instances that are pending or terminating.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "IN_SERVICE_INSTANCES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MAX_SIZE")
    def MAX_SIZE(cls) -> "GroupMetric":
        '''The maximum size of the Auto Scaling group.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "MAX_SIZE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MIN_SIZE")
    def MIN_SIZE(cls) -> "GroupMetric":
        '''The minimum size of the Auto Scaling group.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "MIN_SIZE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PENDING_INSTANCES")
    def PENDING_INSTANCES(cls) -> "GroupMetric":
        '''The number of instances that are pending A pending instance is not yet in service, this metric does not include instances that are in service or terminating.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "PENDING_INSTANCES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STANDBY_INSTANCES")
    def STANDBY_INSTANCES(cls) -> "GroupMetric":
        '''The number of instances that are in a Standby state Instances in this state are still running but are not actively in service.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "STANDBY_INSTANCES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TERMINATING_INSTANCES")
    def TERMINATING_INSTANCES(cls) -> "GroupMetric":
        '''The number of instances that are in the process of terminating This metric does not include instances that are in service or pending.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "TERMINATING_INSTANCES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TOTAL_INSTANCES")
    def TOTAL_INSTANCES(cls) -> "GroupMetric":
        '''The total number of instances in the Auto Scaling group This metric identifies the number of instances that are in service, pending, and terminating.'''
        return typing.cast("GroupMetric", jsii.sget(cls, "TOTAL_INSTANCES"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the group metric.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))


class GroupMetrics(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.GroupMetrics",
):
    '''A set of group metrics.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        # machine_image: ec2.IMachineImage
        
        
        # Enable monitoring of all group metrics
        autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
        
            # ...
        
            group_metrics=[autoscaling.GroupMetrics.all()]
        )
        
        # Enable monitoring for a subset of group metrics
        autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
        
            # ...
        
            group_metrics=[autoscaling.GroupMetrics(autoscaling.GroupMetric.MIN_SIZE, autoscaling.GroupMetric.MAX_SIZE)]
        )
    '''

    def __init__(self, *metrics: GroupMetric) -> None:
        '''
        :param metrics: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4caa94de5310031ab350e06e7ac582fc304ebc075c4f520158c3ff48fbf6bc0)
            check_type(argname="argument metrics", value=metrics, expected_type=typing.Tuple[type_hints["metrics"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        jsii.create(self.__class__, self, [*metrics])

    @jsii.member(jsii_name="all")
    @builtins.classmethod
    def all(cls) -> "GroupMetrics":
        '''Report all group metrics.'''
        return typing.cast("GroupMetrics", jsii.sinvoke(cls, "all", []))


class HealthCheck(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.HealthCheck",
):
    '''Health check settings.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        from aws_cdk import aws_autoscaling as autoscaling
        
        health_check = autoscaling.HealthCheck.ec2(
            grace=cdk.Duration.minutes(30)
        )
    '''

    @jsii.member(jsii_name="ec2")
    @builtins.classmethod
    def ec2(cls, *, grace: typing.Optional[_Duration_4839e8c3] = None) -> "HealthCheck":
        '''Use EC2 for health checks.

        :param grace: Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service. Default: Duration.seconds(0)
        '''
        options = Ec2HealthCheckOptions(grace=grace)

        return typing.cast("HealthCheck", jsii.sinvoke(cls, "ec2", [options]))

    @jsii.member(jsii_name="elb")
    @builtins.classmethod
    def elb(cls, *, grace: _Duration_4839e8c3) -> "HealthCheck":
        '''Use ELB for health checks.

        It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.

        :param grace: Specified the time Auto Scaling waits before checking the health status of an EC2 instance that has come into service. This option is required for ELB health checks.
        '''
        options = ElbHealthCheckOptions(grace=grace)

        return typing.cast("HealthCheck", jsii.sinvoke(cls, "elb", [options]))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @builtins.property
    @jsii.member(jsii_name="gracePeriod")
    def grace_period(self) -> typing.Optional[_Duration_4839e8c3]:
        return typing.cast(typing.Optional[_Duration_4839e8c3], jsii.get(self, "gracePeriod"))


@jsii.interface(jsii_type="aws-cdk-lib.aws_autoscaling.IAutoScalingGroup")
class IAutoScalingGroup(
    _IResource_c80c4260,
    _IGrantable_71c4f5de,
    typing_extensions.Protocol,
):
    '''An AutoScalingGroup.'''

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupArn")
    def auto_scaling_group_arn(self) -> builtins.str:
        '''The arn of the AutoScalingGroup.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the AutoScalingGroup.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="osType")
    def os_type(self) -> _OperatingSystemType_9224a1fe:
        '''The operating system family that the instances in this auto-scaling group belong to.

        Is 'UNKNOWN' for imported ASGs.
        '''
        ...

    @jsii.member(jsii_name="addLifecycleHook")
    def add_lifecycle_hook(
        self,
        id: builtins.str,
        *,
        lifecycle_transition: "LifecycleTransition",
        default_result: typing.Optional[DefaultResult] = None,
        heartbeat_timeout: typing.Optional[_Duration_4839e8c3] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target: typing.Optional["ILifecycleHookTarget"] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> "LifecycleHook":
        '''Send a message to either an SQS queue or SNS topic when instances launch or terminate.

        :param id: -
        :param lifecycle_transition: The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. Default: Continue
        :param heartbeat_timeout: Maximum time between calls to RecordLifecycleActionHeartbeat for the hook. If the lifecycle hook times out, perform the action in DefaultResult. Default: - No heartbeat timeout.
        :param lifecycle_hook_name: Name of the lifecycle hook. Default: - Automatically generated name.
        :param notification_metadata: Additional data to pass to the lifecycle hook target. Default: - No metadata.
        :param notification_target: The target of the lifecycle hook. Default: - No target.
        :param role: The role that allows publishing to the notification target. Default: - A role will be created if a target is provided. Otherwise, no role is created.
        '''
        ...

    @jsii.member(jsii_name="addUserData")
    def add_user_data(self, *commands: builtins.str) -> None:
        '''Add command to the startup script of fleet instances.

        The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
        Does nothing for imported ASGs.

        :param commands: -
        '''
        ...

    @jsii.member(jsii_name="addWarmPool")
    def add_warm_pool(
        self,
        *,
        max_group_prepared_capacity: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        pool_state: typing.Optional["PoolState"] = None,
        reuse_on_scale_in: typing.Optional[builtins.bool] = None,
    ) -> "WarmPool":
        '''Add a pool of pre-initialized EC2 instances that sits alongside an Auto Scaling group.

        :param max_group_prepared_capacity: The maximum number of instances that are allowed to be in the warm pool or in any state except Terminated for the Auto Scaling group. If the value is not specified, Amazon EC2 Auto Scaling launches and maintains the difference between the group's maximum capacity and its desired capacity. Default: - max size of the Auto Scaling group
        :param min_size: The minimum number of instances to maintain in the warm pool. Default: 0
        :param pool_state: The instance state to transition to after the lifecycle actions are complete. Default: PoolState.STOPPED
        :param reuse_on_scale_in: Indicates whether instances in the Auto Scaling group can be returned to the warm pool on scale in. If the value is not specified, instances in the Auto Scaling group will be terminated when the group scales in. Default: false
        '''
        ...

    @jsii.member(jsii_name="scaleOnCpuUtilization")
    def scale_on_cpu_utilization(
        self,
        id: builtins.str,
        *,
        target_utilization_percent: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in to achieve a target CPU utilization.

        :param id: -
        :param target_utilization_percent: Target average CPU utilization across the task.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        ...

    @jsii.member(jsii_name="scaleOnIncomingBytes")
    def scale_on_incoming_bytes(
        self,
        id: builtins.str,
        *,
        target_bytes_per_second: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in to achieve a target network ingress rate.

        :param id: -
        :param target_bytes_per_second: Target average bytes/seconds on each instance.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        ...

    @jsii.member(jsii_name="scaleOnMetric")
    def scale_on_metric(
        self,
        id: builtins.str,
        *,
        metric: _IMetric_c7fd29de,
        scaling_steps: typing.Sequence[typing.Union["ScalingInterval", typing.Dict[builtins.str, typing.Any]]],
        adjustment_type: typing.Optional[AdjustmentType] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        evaluation_periods: typing.Optional[jsii.Number] = None,
        metric_aggregation_type: typing.Optional["MetricAggregationType"] = None,
        min_adjustment_magnitude: typing.Optional[jsii.Number] = None,
    ) -> "StepScalingPolicy":
        '''Scale out or in, in response to a metric.

        :param id: -
        :param metric: Metric to scale on.
        :param scaling_steps: The intervals for scaling. Maps a range of metric values to a particular scaling behavior.
        :param adjustment_type: How the adjustment numbers inside 'intervals' are interpreted. Default: ChangeInCapacity
        :param cooldown: Grace period after scaling activity. Default: Default cooldown period on your AutoScalingGroup
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: Same as the cooldown
        :param evaluation_periods: How many evaluation periods of the metric to wait before triggering a scaling action. Raising this value can be used to smooth out the metric, at the expense of slower response times. Default: 1
        :param metric_aggregation_type: Aggregation to apply to all data points over the evaluation periods. Only has meaning if ``evaluationPeriods != 1``. Default: - The statistic from the metric if applicable (MIN, MAX, AVERAGE), otherwise AVERAGE.
        :param min_adjustment_magnitude: Minimum absolute number to adjust capacity with as result of percentage scaling. Only when using AdjustmentType = PercentChangeInCapacity, this number controls the minimum absolute effect size. Default: No minimum scaling effect
        '''
        ...

    @jsii.member(jsii_name="scaleOnOutgoingBytes")
    def scale_on_outgoing_bytes(
        self,
        id: builtins.str,
        *,
        target_bytes_per_second: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in to achieve a target network egress rate.

        :param id: -
        :param target_bytes_per_second: Target average bytes/seconds on each instance.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        ...

    @jsii.member(jsii_name="scaleOnSchedule")
    def scale_on_schedule(
        self,
        id: builtins.str,
        *,
        schedule: "Schedule",
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
        time_zone: typing.Optional[builtins.str] = None,
    ) -> "ScheduledAction":
        '''Scale out or in based on time.

        :param id: -
        :param schedule: When to perform this action. Supports cron expressions. For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        :param desired_capacity: The new desired capacity. At the scheduled time, set the desired capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new desired capacity.
        :param end_time: When this scheduled action expires. Default: - The rule never expires.
        :param max_capacity: The new maximum capacity. At the scheduled time, set the maximum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new maximum capacity.
        :param min_capacity: The new minimum capacity. At the scheduled time, set the minimum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new minimum capacity.
        :param start_time: When this scheduled action becomes active. Default: - The rule is activate immediately.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti). For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones. Default: - UTC
        '''
        ...

    @jsii.member(jsii_name="scaleToTrackMetric")
    def scale_to_track_metric(
        self,
        id: builtins.str,
        *,
        metric: _IMetric_c7fd29de,
        target_value: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in in order to keep a metric around a target value.

        :param id: -
        :param metric: Metric to track. The metric must represent a utilization, so that if it's higher than the target value, your ASG should scale out, and if it's lower it should scale in.
        :param target_value: Value to keep the metric around.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        ...


class _IAutoScalingGroupProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
    jsii.proxy_for(_IGrantable_71c4f5de), # type: ignore[misc]
):
    '''An AutoScalingGroup.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_autoscaling.IAutoScalingGroup"

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupArn")
    def auto_scaling_group_arn(self) -> builtins.str:
        '''The arn of the AutoScalingGroup.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "autoScalingGroupArn"))

    @builtins.property
    @jsii.member(jsii_name="autoScalingGroupName")
    def auto_scaling_group_name(self) -> builtins.str:
        '''The name of the AutoScalingGroup.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "autoScalingGroupName"))

    @builtins.property
    @jsii.member(jsii_name="osType")
    def os_type(self) -> _OperatingSystemType_9224a1fe:
        '''The operating system family that the instances in this auto-scaling group belong to.

        Is 'UNKNOWN' for imported ASGs.
        '''
        return typing.cast(_OperatingSystemType_9224a1fe, jsii.get(self, "osType"))

    @jsii.member(jsii_name="addLifecycleHook")
    def add_lifecycle_hook(
        self,
        id: builtins.str,
        *,
        lifecycle_transition: "LifecycleTransition",
        default_result: typing.Optional[DefaultResult] = None,
        heartbeat_timeout: typing.Optional[_Duration_4839e8c3] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target: typing.Optional["ILifecycleHookTarget"] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> "LifecycleHook":
        '''Send a message to either an SQS queue or SNS topic when instances launch or terminate.

        :param id: -
        :param lifecycle_transition: The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. Default: Continue
        :param heartbeat_timeout: Maximum time between calls to RecordLifecycleActionHeartbeat for the hook. If the lifecycle hook times out, perform the action in DefaultResult. Default: - No heartbeat timeout.
        :param lifecycle_hook_name: Name of the lifecycle hook. Default: - Automatically generated name.
        :param notification_metadata: Additional data to pass to the lifecycle hook target. Default: - No metadata.
        :param notification_target: The target of the lifecycle hook. Default: - No target.
        :param role: The role that allows publishing to the notification target. Default: - A role will be created if a target is provided. Otherwise, no role is created.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ab9f01a9b271d7397ec179829698e896314a3ff91d8c9925ce60c1a7e1f8152a)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BasicLifecycleHookProps(
            lifecycle_transition=lifecycle_transition,
            default_result=default_result,
            heartbeat_timeout=heartbeat_timeout,
            lifecycle_hook_name=lifecycle_hook_name,
            notification_metadata=notification_metadata,
            notification_target=notification_target,
            role=role,
        )

        return typing.cast("LifecycleHook", jsii.invoke(self, "addLifecycleHook", [id, props]))

    @jsii.member(jsii_name="addUserData")
    def add_user_data(self, *commands: builtins.str) -> None:
        '''Add command to the startup script of fleet instances.

        The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
        Does nothing for imported ASGs.

        :param commands: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a06b5ce71e367f3e231d4f04f4818efa216316eaa294c279886ba7601c8fd8e1)
            check_type(argname="argument commands", value=commands, expected_type=typing.Tuple[type_hints["commands"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(None, jsii.invoke(self, "addUserData", [*commands]))

    @jsii.member(jsii_name="addWarmPool")
    def add_warm_pool(
        self,
        *,
        max_group_prepared_capacity: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        pool_state: typing.Optional["PoolState"] = None,
        reuse_on_scale_in: typing.Optional[builtins.bool] = None,
    ) -> "WarmPool":
        '''Add a pool of pre-initialized EC2 instances that sits alongside an Auto Scaling group.

        :param max_group_prepared_capacity: The maximum number of instances that are allowed to be in the warm pool or in any state except Terminated for the Auto Scaling group. If the value is not specified, Amazon EC2 Auto Scaling launches and maintains the difference between the group's maximum capacity and its desired capacity. Default: - max size of the Auto Scaling group
        :param min_size: The minimum number of instances to maintain in the warm pool. Default: 0
        :param pool_state: The instance state to transition to after the lifecycle actions are complete. Default: PoolState.STOPPED
        :param reuse_on_scale_in: Indicates whether instances in the Auto Scaling group can be returned to the warm pool on scale in. If the value is not specified, instances in the Auto Scaling group will be terminated when the group scales in. Default: false
        '''
        options = WarmPoolOptions(
            max_group_prepared_capacity=max_group_prepared_capacity,
            min_size=min_size,
            pool_state=pool_state,
            reuse_on_scale_in=reuse_on_scale_in,
        )

        return typing.cast("WarmPool", jsii.invoke(self, "addWarmPool", [options]))

    @jsii.member(jsii_name="scaleOnCpuUtilization")
    def scale_on_cpu_utilization(
        self,
        id: builtins.str,
        *,
        target_utilization_percent: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in to achieve a target CPU utilization.

        :param id: -
        :param target_utilization_percent: Target average CPU utilization across the task.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77f448e75ef9c561d8a2e618f3a5836428850eb489fd269f4be00725e79cc782)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CpuUtilizationScalingProps(
            target_utilization_percent=target_utilization_percent,
            cooldown=cooldown,
            disable_scale_in=disable_scale_in,
            estimated_instance_warmup=estimated_instance_warmup,
        )

        return typing.cast("TargetTrackingScalingPolicy", jsii.invoke(self, "scaleOnCpuUtilization", [id, props]))

    @jsii.member(jsii_name="scaleOnIncomingBytes")
    def scale_on_incoming_bytes(
        self,
        id: builtins.str,
        *,
        target_bytes_per_second: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in to achieve a target network ingress rate.

        :param id: -
        :param target_bytes_per_second: Target average bytes/seconds on each instance.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fa88e81790fc7fec6dc0adb813f1c6a5f164cf91d6ae5c166cf580536ceba4f)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = NetworkUtilizationScalingProps(
            target_bytes_per_second=target_bytes_per_second,
            cooldown=cooldown,
            disable_scale_in=disable_scale_in,
            estimated_instance_warmup=estimated_instance_warmup,
        )

        return typing.cast("TargetTrackingScalingPolicy", jsii.invoke(self, "scaleOnIncomingBytes", [id, props]))

    @jsii.member(jsii_name="scaleOnMetric")
    def scale_on_metric(
        self,
        id: builtins.str,
        *,
        metric: _IMetric_c7fd29de,
        scaling_steps: typing.Sequence[typing.Union["ScalingInterval", typing.Dict[builtins.str, typing.Any]]],
        adjustment_type: typing.Optional[AdjustmentType] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        evaluation_periods: typing.Optional[jsii.Number] = None,
        metric_aggregation_type: typing.Optional["MetricAggregationType"] = None,
        min_adjustment_magnitude: typing.Optional[jsii.Number] = None,
    ) -> "StepScalingPolicy":
        '''Scale out or in, in response to a metric.

        :param id: -
        :param metric: Metric to scale on.
        :param scaling_steps: The intervals for scaling. Maps a range of metric values to a particular scaling behavior.
        :param adjustment_type: How the adjustment numbers inside 'intervals' are interpreted. Default: ChangeInCapacity
        :param cooldown: Grace period after scaling activity. Default: Default cooldown period on your AutoScalingGroup
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: Same as the cooldown
        :param evaluation_periods: How many evaluation periods of the metric to wait before triggering a scaling action. Raising this value can be used to smooth out the metric, at the expense of slower response times. Default: 1
        :param metric_aggregation_type: Aggregation to apply to all data points over the evaluation periods. Only has meaning if ``evaluationPeriods != 1``. Default: - The statistic from the metric if applicable (MIN, MAX, AVERAGE), otherwise AVERAGE.
        :param min_adjustment_magnitude: Minimum absolute number to adjust capacity with as result of percentage scaling. Only when using AdjustmentType = PercentChangeInCapacity, this number controls the minimum absolute effect size. Default: No minimum scaling effect
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cfc7f1f455e355ca47e5ecb7405720cfd89dade7e0863a60891f859107bd9c7f)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BasicStepScalingPolicyProps(
            metric=metric,
            scaling_steps=scaling_steps,
            adjustment_type=adjustment_type,
            cooldown=cooldown,
            estimated_instance_warmup=estimated_instance_warmup,
            evaluation_periods=evaluation_periods,
            metric_aggregation_type=metric_aggregation_type,
            min_adjustment_magnitude=min_adjustment_magnitude,
        )

        return typing.cast("StepScalingPolicy", jsii.invoke(self, "scaleOnMetric", [id, props]))

    @jsii.member(jsii_name="scaleOnOutgoingBytes")
    def scale_on_outgoing_bytes(
        self,
        id: builtins.str,
        *,
        target_bytes_per_second: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in to achieve a target network egress rate.

        :param id: -
        :param target_bytes_per_second: Target average bytes/seconds on each instance.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c99749922be06a8e6a6ae56f9ba82fc31a391c7432ef7609a9997fb4b3777be8)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = NetworkUtilizationScalingProps(
            target_bytes_per_second=target_bytes_per_second,
            cooldown=cooldown,
            disable_scale_in=disable_scale_in,
            estimated_instance_warmup=estimated_instance_warmup,
        )

        return typing.cast("TargetTrackingScalingPolicy", jsii.invoke(self, "scaleOnOutgoingBytes", [id, props]))

    @jsii.member(jsii_name="scaleOnSchedule")
    def scale_on_schedule(
        self,
        id: builtins.str,
        *,
        schedule: "Schedule",
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
        time_zone: typing.Optional[builtins.str] = None,
    ) -> "ScheduledAction":
        '''Scale out or in based on time.

        :param id: -
        :param schedule: When to perform this action. Supports cron expressions. For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        :param desired_capacity: The new desired capacity. At the scheduled time, set the desired capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new desired capacity.
        :param end_time: When this scheduled action expires. Default: - The rule never expires.
        :param max_capacity: The new maximum capacity. At the scheduled time, set the maximum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new maximum capacity.
        :param min_capacity: The new minimum capacity. At the scheduled time, set the minimum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new minimum capacity.
        :param start_time: When this scheduled action becomes active. Default: - The rule is activate immediately.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti). For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones. Default: - UTC
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb399691f6facd38db7c6fafc541af1753d52ff948133ce68b6da663ac0cee9d)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BasicScheduledActionProps(
            schedule=schedule,
            desired_capacity=desired_capacity,
            end_time=end_time,
            max_capacity=max_capacity,
            min_capacity=min_capacity,
            start_time=start_time,
            time_zone=time_zone,
        )

        return typing.cast("ScheduledAction", jsii.invoke(self, "scaleOnSchedule", [id, props]))

    @jsii.member(jsii_name="scaleToTrackMetric")
    def scale_to_track_metric(
        self,
        id: builtins.str,
        *,
        metric: _IMetric_c7fd29de,
        target_value: jsii.Number,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "TargetTrackingScalingPolicy":
        '''Scale out or in in order to keep a metric around a target value.

        :param id: -
        :param metric: Metric to track. The metric must represent a utilization, so that if it's higher than the target value, your ASG should scale out, and if it's lower it should scale in.
        :param target_value: Value to keep the metric around.
        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd36a1a3975c7f7d026736d8f092cfe64a3157a6aa6a77d171eec02dc392284c)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = MetricTargetTrackingProps(
            metric=metric,
            target_value=target_value,
            cooldown=cooldown,
            disable_scale_in=disable_scale_in,
            estimated_instance_warmup=estimated_instance_warmup,
        )

        return typing.cast("TargetTrackingScalingPolicy", jsii.invoke(self, "scaleToTrackMetric", [id, props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAutoScalingGroup).__jsii_proxy_class__ = lambda : _IAutoScalingGroupProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_autoscaling.ILifecycleHook")
class ILifecycleHook(_IResource_c80c4260, typing_extensions.Protocol):
    '''A basic lifecycle hook object.'''

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _IRole_235f5d8e:
        '''The role for the lifecycle hook to execute.

        :default:

        - A default role is created if 'notificationTarget' is specified.
        Otherwise, no role is created.
        '''
        ...


class _ILifecycleHookProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''A basic lifecycle hook object.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_autoscaling.ILifecycleHook"

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _IRole_235f5d8e:
        '''The role for the lifecycle hook to execute.

        :default:

        - A default role is created if 'notificationTarget' is specified.
        Otherwise, no role is created.
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "role"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ILifecycleHook).__jsii_proxy_class__ = lambda : _ILifecycleHookProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_autoscaling.ILifecycleHookTarget")
class ILifecycleHookTarget(typing_extensions.Protocol):
    '''Interface for autoscaling lifecycle hook targets.'''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        *,
        lifecycle_hook: "LifecycleHook",
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> "LifecycleHookTargetConfig":
        '''Called when this object is used as the target of a lifecycle hook.

        :param scope: -
        :param lifecycle_hook: The lifecycle hook to attach to. [disable-awslint:ref-via-interface]
        :param role: The role to use when attaching to the lifecycle hook. [disable-awslint:ref-via-interface] Default: : a role is not created unless the target arn is specified
        '''
        ...


class _ILifecycleHookTargetProxy:
    '''Interface for autoscaling lifecycle hook targets.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_autoscaling.ILifecycleHookTarget"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        *,
        lifecycle_hook: "LifecycleHook",
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> "LifecycleHookTargetConfig":
        '''Called when this object is used as the target of a lifecycle hook.

        :param scope: -
        :param lifecycle_hook: The lifecycle hook to attach to. [disable-awslint:ref-via-interface]
        :param role: The role to use when attaching to the lifecycle hook. [disable-awslint:ref-via-interface] Default: : a role is not created unless the target arn is specified
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9770cd60cfc69f2c7d108523545fc5de207bca98daae82701548d55bbefb0c5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        options = BindHookTargetOptions(lifecycle_hook=lifecycle_hook, role=role)

        return typing.cast("LifecycleHookTargetConfig", jsii.invoke(self, "bind", [scope, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ILifecycleHookTarget).__jsii_proxy_class__ = lambda : _ILifecycleHookTargetProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.InstancesDistribution",
    jsii_struct_bases=[],
    name_mapping={
        "on_demand_allocation_strategy": "onDemandAllocationStrategy",
        "on_demand_base_capacity": "onDemandBaseCapacity",
        "on_demand_percentage_above_base_capacity": "onDemandPercentageAboveBaseCapacity",
        "spot_allocation_strategy": "spotAllocationStrategy",
        "spot_instance_pools": "spotInstancePools",
        "spot_max_price": "spotMaxPrice",
    },
)
class InstancesDistribution:
    def __init__(
        self,
        *,
        on_demand_allocation_strategy: typing.Optional["OnDemandAllocationStrategy"] = None,
        on_demand_base_capacity: typing.Optional[jsii.Number] = None,
        on_demand_percentage_above_base_capacity: typing.Optional[jsii.Number] = None,
        spot_allocation_strategy: typing.Optional["SpotAllocationStrategy"] = None,
        spot_instance_pools: typing.Optional[jsii.Number] = None,
        spot_max_price: typing.Optional[builtins.str] = None,
    ) -> None:
        '''InstancesDistribution is a subproperty of MixedInstancesPolicy that describes an instances distribution for an Auto Scaling group.

        The instances distribution specifies the distribution of On-Demand Instances
        and Spot Instances, the maximum price to pay for Spot Instances, and how the Auto Scaling group allocates
        instance types to fulfill On-Demand and Spot capacities.

        For more information and example configurations, see Auto Scaling groups with multiple instance types
        and purchase options in the Amazon EC2 Auto Scaling User Guide:

        https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-purchase-options.html

        :param on_demand_allocation_strategy: Indicates how to allocate instance types to fulfill On-Demand capacity. The only valid value is prioritized, which is also the default value. Default: OnDemandAllocationStrategy.PRIORITIZED
        :param on_demand_base_capacity: The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances. This base portion is provisioned first as your group scales. Defaults to 0 if not specified. If you specify weights for the instance types in the overrides, set the value of OnDemandBaseCapacity in terms of the number of capacity units, and not the number of instances. Default: 0
        :param on_demand_percentage_above_base_capacity: Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond OnDemandBaseCapacity. Expressed as a number (for example, 20 specifies 20% On-Demand Instances, 80% Spot Instances). Defaults to 100 if not specified. If set to 100, only On-Demand Instances are provisioned. Default: 100
        :param spot_allocation_strategy: If the allocation strategy is lowest-price, the Auto Scaling group launches instances using the Spot pools with the lowest price, and evenly allocates your instances across the number of Spot pools that you specify. Defaults to lowest-price if not specified. If the allocation strategy is capacity-optimized (recommended), the Auto Scaling group launches instances using Spot pools that are optimally chosen based on the available Spot capacity. Alternatively, you can use capacity-optimized-prioritized and set the order of instance types in the list of launch template overrides from highest to lowest priority (from first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best-effort basis but optimizes for capacity first. Default: SpotAllocationStrategy.LOWEST_PRICE
        :param spot_instance_pools: The number of Spot Instance pools to use to allocate your Spot capacity. The Spot pools are determined from the different instance types in the overrides. Valid only when the Spot allocation strategy is lowest-price. Value must be in the range of 1 to 20. Defaults to 2 if not specified. Default: 2
        :param spot_max_price: The maximum price per unit hour that you are willing to pay for a Spot Instance. If you leave the value at its default (empty), Amazon EC2 Auto Scaling uses the On-Demand price as the maximum Spot price. To remove a value that you previously set, include the property but specify an empty string ("") for the value. Default: "" - On-Demand price

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            # launch_template1: ec2.LaunchTemplate
            # launch_template2: ec2.LaunchTemplate
            
            
            autoscaling.AutoScalingGroup(self, "ASG",
                vpc=vpc,
                mixed_instances_policy=autoscaling.MixedInstancesPolicy(
                    instances_distribution=autoscaling.InstancesDistribution(
                        on_demand_percentage_above_base_capacity=50
                    ),
                    launch_template=launch_template1,
                    launch_template_overrides=[autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t3.micro")), autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t3a.micro")), autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t4g.micro"), launch_template=launch_template2)]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb58af418fe3e379a2f815054eedffadc45d19358b6956e7eb1f57e0c4b5ac09)
            check_type(argname="argument on_demand_allocation_strategy", value=on_demand_allocation_strategy, expected_type=type_hints["on_demand_allocation_strategy"])
            check_type(argname="argument on_demand_base_capacity", value=on_demand_base_capacity, expected_type=type_hints["on_demand_base_capacity"])
            check_type(argname="argument on_demand_percentage_above_base_capacity", value=on_demand_percentage_above_base_capacity, expected_type=type_hints["on_demand_percentage_above_base_capacity"])
            check_type(argname="argument spot_allocation_strategy", value=spot_allocation_strategy, expected_type=type_hints["spot_allocation_strategy"])
            check_type(argname="argument spot_instance_pools", value=spot_instance_pools, expected_type=type_hints["spot_instance_pools"])
            check_type(argname="argument spot_max_price", value=spot_max_price, expected_type=type_hints["spot_max_price"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if on_demand_allocation_strategy is not None:
            self._values["on_demand_allocation_strategy"] = on_demand_allocation_strategy
        if on_demand_base_capacity is not None:
            self._values["on_demand_base_capacity"] = on_demand_base_capacity
        if on_demand_percentage_above_base_capacity is not None:
            self._values["on_demand_percentage_above_base_capacity"] = on_demand_percentage_above_base_capacity
        if spot_allocation_strategy is not None:
            self._values["spot_allocation_strategy"] = spot_allocation_strategy
        if spot_instance_pools is not None:
            self._values["spot_instance_pools"] = spot_instance_pools
        if spot_max_price is not None:
            self._values["spot_max_price"] = spot_max_price

    @builtins.property
    def on_demand_allocation_strategy(
        self,
    ) -> typing.Optional["OnDemandAllocationStrategy"]:
        '''Indicates how to allocate instance types to fulfill On-Demand capacity.

        The only valid value is prioritized,
        which is also the default value.

        :default: OnDemandAllocationStrategy.PRIORITIZED
        '''
        result = self._values.get("on_demand_allocation_strategy")
        return typing.cast(typing.Optional["OnDemandAllocationStrategy"], result)

    @builtins.property
    def on_demand_base_capacity(self) -> typing.Optional[jsii.Number]:
        '''The minimum amount of the Auto Scaling group's capacity that must be fulfilled by On-Demand Instances.

        This
        base portion is provisioned first as your group scales. Defaults to 0 if not specified. If you specify weights
        for the instance types in the overrides, set the value of OnDemandBaseCapacity in terms of the number of
        capacity units, and not the number of instances.

        :default: 0
        '''
        result = self._values.get("on_demand_base_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_demand_percentage_above_base_capacity(self) -> typing.Optional[jsii.Number]:
        '''Controls the percentages of On-Demand Instances and Spot Instances for your additional capacity beyond OnDemandBaseCapacity.

        Expressed as a number (for example, 20 specifies 20% On-Demand Instances, 80% Spot Instances).
        Defaults to 100 if not specified. If set to 100, only On-Demand Instances are provisioned.

        :default: 100
        '''
        result = self._values.get("on_demand_percentage_above_base_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def spot_allocation_strategy(self) -> typing.Optional["SpotAllocationStrategy"]:
        '''If the allocation strategy is lowest-price, the Auto Scaling group launches instances using the Spot pools with the lowest price, and evenly allocates your instances across the number of Spot pools that you specify.

        Defaults to
        lowest-price if not specified.

        If the allocation strategy is capacity-optimized (recommended), the Auto Scaling group launches instances using Spot
        pools that are optimally chosen based on the available Spot capacity. Alternatively, you can use capacity-optimized-prioritized
        and set the order of instance types in the list of launch template overrides from highest to lowest priority
        (from first to last in the list). Amazon EC2 Auto Scaling honors the instance type priorities on a best-effort basis but
        optimizes for capacity first.

        :default: SpotAllocationStrategy.LOWEST_PRICE
        '''
        result = self._values.get("spot_allocation_strategy")
        return typing.cast(typing.Optional["SpotAllocationStrategy"], result)

    @builtins.property
    def spot_instance_pools(self) -> typing.Optional[jsii.Number]:
        '''The number of Spot Instance pools to use to allocate your Spot capacity.

        The Spot pools are determined from the different instance
        types in the overrides. Valid only when the Spot allocation strategy is lowest-price. Value must be in the range of 1 to 20.
        Defaults to 2 if not specified.

        :default: 2
        '''
        result = self._values.get("spot_instance_pools")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def spot_max_price(self) -> typing.Optional[builtins.str]:
        '''The maximum price per unit hour that you are willing to pay for a Spot Instance.

        If you leave the value at its default (empty),
        Amazon EC2 Auto Scaling uses the On-Demand price as the maximum Spot price. To remove a value that you previously set, include
        the property but specify an empty string ("") for the value.

        :default: "" - On-Demand price
        '''
        result = self._values.get("spot_max_price")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InstancesDistribution(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.LaunchTemplateOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "instance_type": "instanceType",
        "launch_template": "launchTemplate",
        "weighted_capacity": "weightedCapacity",
    },
)
class LaunchTemplateOverrides:
    def __init__(
        self,
        *,
        instance_type: _InstanceType_f64915b9,
        launch_template: typing.Optional[_ILaunchTemplate_f32c0fd7] = None,
        weighted_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''LaunchTemplateOverrides is a subproperty of LaunchTemplate that describes an override for a launch template.

        :param instance_type: The instance type, such as m3.xlarge. You must use an instance type that is supported in your requested Region and Availability Zones. Default: - Do not override instance type
        :param launch_template: Provides the launch template to be used when launching the instance type. For example, some instance types might require a launch template with a different AMI. If not provided, Amazon EC2 Auto Scaling uses the launch template that's defined for your mixed instances policy. Default: - Do not override launch template
        :param weighted_capacity: The number of capacity units provided by the specified instance type in terms of virtual CPUs, memory, storage, throughput, or other relative performance characteristic. When a Spot or On-Demand Instance is provisioned, the capacity units count toward the desired capacity. Amazon EC2 Auto Scaling provisions instances until the desired capacity is totally fulfilled, even if this results in an overage. Value must be in the range of 1 to 999. For example, If there are 2 units remaining to fulfill capacity, and Amazon EC2 Auto Scaling can only provision an instance with a WeightedCapacity of 5 units, the instance is provisioned, and the desired capacity is exceeded by 3 units. Default: - Do not provide weight

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_ec2 as ec2
            
            # instance_type: ec2.InstanceType
            # launch_template: ec2.LaunchTemplate
            
            launch_template_overrides = autoscaling.LaunchTemplateOverrides(
                instance_type=instance_type,
            
                # the properties below are optional
                launch_template=launch_template,
                weighted_capacity=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f9077816541cf5f9b60bf6bf1b1b1cd717f25358a3670bcbe5de6d00eef3687c)
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument launch_template", value=launch_template, expected_type=type_hints["launch_template"])
            check_type(argname="argument weighted_capacity", value=weighted_capacity, expected_type=type_hints["weighted_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "instance_type": instance_type,
        }
        if launch_template is not None:
            self._values["launch_template"] = launch_template
        if weighted_capacity is not None:
            self._values["weighted_capacity"] = weighted_capacity

    @builtins.property
    def instance_type(self) -> _InstanceType_f64915b9:
        '''The instance type, such as m3.xlarge. You must use an instance type that is supported in your requested Region and Availability Zones.

        :default: - Do not override instance type
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(_InstanceType_f64915b9, result)

    @builtins.property
    def launch_template(self) -> typing.Optional[_ILaunchTemplate_f32c0fd7]:
        '''Provides the launch template to be used when launching the instance type.

        For example, some instance types might
        require a launch template with a different AMI. If not provided, Amazon EC2 Auto Scaling uses the launch template
        that's defined for your mixed instances policy.

        :default: - Do not override launch template
        '''
        result = self._values.get("launch_template")
        return typing.cast(typing.Optional[_ILaunchTemplate_f32c0fd7], result)

    @builtins.property
    def weighted_capacity(self) -> typing.Optional[jsii.Number]:
        '''The number of capacity units provided by the specified instance type in terms of virtual CPUs, memory, storage, throughput, or other relative performance characteristic.

        When a Spot or On-Demand Instance is provisioned, the
        capacity units count toward the desired capacity. Amazon EC2 Auto Scaling provisions instances until the desired
        capacity is totally fulfilled, even if this results in an overage. Value must be in the range of 1 to 999.

        For example, If there are 2 units remaining to fulfill capacity, and Amazon EC2 Auto Scaling can only provision
        an instance with a WeightedCapacity of 5 units, the instance is provisioned, and the desired capacity is exceeded
        by 3 units.

        :default: - Do not provide weight

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-weighting.html
        '''
        result = self._values.get("weighted_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LaunchTemplateOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ILifecycleHook)
class LifecycleHook(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.LifecycleHook",
):
    '''Define a life cycle hook.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        from aws_cdk import aws_autoscaling as autoscaling
        from aws_cdk import aws_iam as iam
        
        # auto_scaling_group: autoscaling.AutoScalingGroup
        # lifecycle_hook_target: autoscaling.ILifecycleHookTarget
        # role: iam.Role
        
        lifecycle_hook = autoscaling.LifecycleHook(self, "MyLifecycleHook",
            auto_scaling_group=auto_scaling_group,
            lifecycle_transition=autoscaling.LifecycleTransition.INSTANCE_LAUNCHING,
        
            # the properties below are optional
            default_result=autoscaling.DefaultResult.CONTINUE,
            heartbeat_timeout=cdk.Duration.minutes(30),
            lifecycle_hook_name="lifecycleHookName",
            notification_metadata="notificationMetadata",
            notification_target=lifecycle_hook_target,
            role=role
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group: IAutoScalingGroup,
        lifecycle_transition: "LifecycleTransition",
        default_result: typing.Optional[DefaultResult] = None,
        heartbeat_timeout: typing.Optional[_Duration_4839e8c3] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target: typing.Optional[ILifecycleHookTarget] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param auto_scaling_group: The AutoScalingGroup to add the lifecycle hook to.
        :param lifecycle_transition: The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. Default: Continue
        :param heartbeat_timeout: Maximum time between calls to RecordLifecycleActionHeartbeat for the hook. If the lifecycle hook times out, perform the action in DefaultResult. Default: - No heartbeat timeout.
        :param lifecycle_hook_name: Name of the lifecycle hook. Default: - Automatically generated name.
        :param notification_metadata: Additional data to pass to the lifecycle hook target. Default: - No metadata.
        :param notification_target: The target of the lifecycle hook. Default: - No target.
        :param role: The role that allows publishing to the notification target. Default: - A role will be created if a target is provided. Otherwise, no role is created.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4218ed2a73b3b937967fabbbf8d60e6130673b6215bff8f98266d8721fa7076)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = LifecycleHookProps(
            auto_scaling_group=auto_scaling_group,
            lifecycle_transition=lifecycle_transition,
            default_result=default_result,
            heartbeat_timeout=heartbeat_timeout,
            lifecycle_hook_name=lifecycle_hook_name,
            notification_metadata=notification_metadata,
            notification_target=notification_target,
            role=role,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="lifecycleHookName")
    def lifecycle_hook_name(self) -> builtins.str:
        '''The name of this lifecycle hook.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "lifecycleHookName"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _IRole_235f5d8e:
        '''The role that allows the ASG to publish to the notification target.

        :default:

        - A default role is created if 'notificationTarget' is specified.
        Otherwise, no role is created.
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "role"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.LifecycleHookProps",
    jsii_struct_bases=[BasicLifecycleHookProps],
    name_mapping={
        "lifecycle_transition": "lifecycleTransition",
        "default_result": "defaultResult",
        "heartbeat_timeout": "heartbeatTimeout",
        "lifecycle_hook_name": "lifecycleHookName",
        "notification_metadata": "notificationMetadata",
        "notification_target": "notificationTarget",
        "role": "role",
        "auto_scaling_group": "autoScalingGroup",
    },
)
class LifecycleHookProps(BasicLifecycleHookProps):
    def __init__(
        self,
        *,
        lifecycle_transition: "LifecycleTransition",
        default_result: typing.Optional[DefaultResult] = None,
        heartbeat_timeout: typing.Optional[_Duration_4839e8c3] = None,
        lifecycle_hook_name: typing.Optional[builtins.str] = None,
        notification_metadata: typing.Optional[builtins.str] = None,
        notification_target: typing.Optional[ILifecycleHookTarget] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        auto_scaling_group: IAutoScalingGroup,
    ) -> None:
        '''Properties for a Lifecycle hook.

        :param lifecycle_transition: The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.
        :param default_result: The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs. Default: Continue
        :param heartbeat_timeout: Maximum time between calls to RecordLifecycleActionHeartbeat for the hook. If the lifecycle hook times out, perform the action in DefaultResult. Default: - No heartbeat timeout.
        :param lifecycle_hook_name: Name of the lifecycle hook. Default: - Automatically generated name.
        :param notification_metadata: Additional data to pass to the lifecycle hook target. Default: - No metadata.
        :param notification_target: The target of the lifecycle hook. Default: - No target.
        :param role: The role that allows publishing to the notification target. Default: - A role will be created if a target is provided. Otherwise, no role is created.
        :param auto_scaling_group: The AutoScalingGroup to add the lifecycle hook to.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_iam as iam
            
            # auto_scaling_group: autoscaling.AutoScalingGroup
            # lifecycle_hook_target: autoscaling.ILifecycleHookTarget
            # role: iam.Role
            
            lifecycle_hook_props = autoscaling.LifecycleHookProps(
                auto_scaling_group=auto_scaling_group,
                lifecycle_transition=autoscaling.LifecycleTransition.INSTANCE_LAUNCHING,
            
                # the properties below are optional
                default_result=autoscaling.DefaultResult.CONTINUE,
                heartbeat_timeout=cdk.Duration.minutes(30),
                lifecycle_hook_name="lifecycleHookName",
                notification_metadata="notificationMetadata",
                notification_target=lifecycle_hook_target,
                role=role
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41e05893ecfc67dfb61215da4c71fe4cfc04cbf73d9aab65be153d28a281deb0)
            check_type(argname="argument lifecycle_transition", value=lifecycle_transition, expected_type=type_hints["lifecycle_transition"])
            check_type(argname="argument default_result", value=default_result, expected_type=type_hints["default_result"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument lifecycle_hook_name", value=lifecycle_hook_name, expected_type=type_hints["lifecycle_hook_name"])
            check_type(argname="argument notification_metadata", value=notification_metadata, expected_type=type_hints["notification_metadata"])
            check_type(argname="argument notification_target", value=notification_target, expected_type=type_hints["notification_target"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument auto_scaling_group", value=auto_scaling_group, expected_type=type_hints["auto_scaling_group"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "lifecycle_transition": lifecycle_transition,
            "auto_scaling_group": auto_scaling_group,
        }
        if default_result is not None:
            self._values["default_result"] = default_result
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if lifecycle_hook_name is not None:
            self._values["lifecycle_hook_name"] = lifecycle_hook_name
        if notification_metadata is not None:
            self._values["notification_metadata"] = notification_metadata
        if notification_target is not None:
            self._values["notification_target"] = notification_target
        if role is not None:
            self._values["role"] = role

    @builtins.property
    def lifecycle_transition(self) -> "LifecycleTransition":
        '''The state of the Amazon EC2 instance to which you want to attach the lifecycle hook.'''
        result = self._values.get("lifecycle_transition")
        assert result is not None, "Required property 'lifecycle_transition' is missing"
        return typing.cast("LifecycleTransition", result)

    @builtins.property
    def default_result(self) -> typing.Optional[DefaultResult]:
        '''The action the Auto Scaling group takes when the lifecycle hook timeout elapses or if an unexpected failure occurs.

        :default: Continue
        '''
        result = self._values.get("default_result")
        return typing.cast(typing.Optional[DefaultResult], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Maximum time between calls to RecordLifecycleActionHeartbeat for the hook.

        If the lifecycle hook times out, perform the action in DefaultResult.

        :default: - No heartbeat timeout.
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def lifecycle_hook_name(self) -> typing.Optional[builtins.str]:
        '''Name of the lifecycle hook.

        :default: - Automatically generated name.
        '''
        result = self._values.get("lifecycle_hook_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_metadata(self) -> typing.Optional[builtins.str]:
        '''Additional data to pass to the lifecycle hook target.

        :default: - No metadata.
        '''
        result = self._values.get("notification_metadata")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_target(self) -> typing.Optional[ILifecycleHookTarget]:
        '''The target of the lifecycle hook.

        :default: - No target.
        '''
        result = self._values.get("notification_target")
        return typing.cast(typing.Optional[ILifecycleHookTarget], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that allows publishing to the notification target.

        :default: - A role will be created if a target is provided. Otherwise, no role is created.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def auto_scaling_group(self) -> IAutoScalingGroup:
        '''The AutoScalingGroup to add the lifecycle hook to.'''
        result = self._values.get("auto_scaling_group")
        assert result is not None, "Required property 'auto_scaling_group' is missing"
        return typing.cast(IAutoScalingGroup, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LifecycleHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.LifecycleHookTargetConfig",
    jsii_struct_bases=[],
    name_mapping={
        "created_role": "createdRole",
        "notification_target_arn": "notificationTargetArn",
    },
)
class LifecycleHookTargetConfig:
    def __init__(
        self,
        *,
        created_role: _IRole_235f5d8e,
        notification_target_arn: builtins.str,
    ) -> None:
        '''Result of binding a lifecycle hook to a target.

        :param created_role: The IRole that was used to bind the lifecycle hook to the target.
        :param notification_target_arn: The targetArn that the lifecycle hook was bound to.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_iam as iam
            
            # role: iam.Role
            
            lifecycle_hook_target_config = autoscaling.LifecycleHookTargetConfig(
                created_role=role,
                notification_target_arn="notificationTargetArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6408c457e5d18cb530f3e735fde171aee4fc61344aa2a2d8868d67c1ea89d172)
            check_type(argname="argument created_role", value=created_role, expected_type=type_hints["created_role"])
            check_type(argname="argument notification_target_arn", value=notification_target_arn, expected_type=type_hints["notification_target_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "created_role": created_role,
            "notification_target_arn": notification_target_arn,
        }

    @builtins.property
    def created_role(self) -> _IRole_235f5d8e:
        '''The IRole that was used to bind the lifecycle hook to the target.'''
        result = self._values.get("created_role")
        assert result is not None, "Required property 'created_role' is missing"
        return typing.cast(_IRole_235f5d8e, result)

    @builtins.property
    def notification_target_arn(self) -> builtins.str:
        '''The targetArn that the lifecycle hook was bound to.'''
        result = self._values.get("notification_target_arn")
        assert result is not None, "Required property 'notification_target_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LifecycleHookTargetConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.LifecycleTransition")
class LifecycleTransition(enum.Enum):
    '''What instance transition to attach the hook to.'''

    INSTANCE_LAUNCHING = "INSTANCE_LAUNCHING"
    '''Execute the hook when an instance is about to be added.'''
    INSTANCE_TERMINATING = "INSTANCE_TERMINATING"
    '''Execute the hook when an instance is about to be terminated.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.MetricAggregationType")
class MetricAggregationType(enum.Enum):
    '''How the scaling metric is going to be aggregated.'''

    AVERAGE = "AVERAGE"
    '''Average.'''
    MINIMUM = "MINIMUM"
    '''Minimum.'''
    MAXIMUM = "MAXIMUM"
    '''Maximum.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.MetricTargetTrackingProps",
    jsii_struct_bases=[BaseTargetTrackingProps],
    name_mapping={
        "cooldown": "cooldown",
        "disable_scale_in": "disableScaleIn",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "metric": "metric",
        "target_value": "targetValue",
    },
)
class MetricTargetTrackingProps(BaseTargetTrackingProps):
    def __init__(
        self,
        *,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        metric: _IMetric_c7fd29de,
        target_value: jsii.Number,
    ) -> None:
        '''Properties for enabling tracking of an arbitrary metric.

        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        :param metric: Metric to track. The metric must represent a utilization, so that if it's higher than the target value, your ASG should scale out, and if it's lower it should scale in.
        :param target_value: Value to keep the metric around.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_cloudwatch as cloudwatch
            
            # metric: cloudwatch.Metric
            
            metric_target_tracking_props = autoscaling.MetricTargetTrackingProps(
                metric=metric,
                target_value=123,
            
                # the properties below are optional
                cooldown=cdk.Duration.minutes(30),
                disable_scale_in=False,
                estimated_instance_warmup=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a189761f9bc0664f3982cc0fdb871012e608cd41eee835f94a198fff2550144d)
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument metric", value=metric, expected_type=type_hints["metric"])
            check_type(argname="argument target_value", value=target_value, expected_type=type_hints["target_value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "metric": metric,
            "target_value": target_value,
        }
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Period after a scaling completes before another scaling activity can start.

        :default: - The default cooldown configured on the AutoScalingGroup.
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the autoscaling group. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        group.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: - Same as the cooldown.
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def metric(self) -> _IMetric_c7fd29de:
        '''Metric to track.

        The metric must represent a utilization, so that if it's higher than the
        target value, your ASG should scale out, and if it's lower it should
        scale in.
        '''
        result = self._values.get("metric")
        assert result is not None, "Required property 'metric' is missing"
        return typing.cast(_IMetric_c7fd29de, result)

    @builtins.property
    def target_value(self) -> jsii.Number:
        '''Value to keep the metric around.'''
        result = self._values.get("target_value")
        assert result is not None, "Required property 'target_value' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MetricTargetTrackingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.MixedInstancesPolicy",
    jsii_struct_bases=[],
    name_mapping={
        "launch_template": "launchTemplate",
        "instances_distribution": "instancesDistribution",
        "launch_template_overrides": "launchTemplateOverrides",
    },
)
class MixedInstancesPolicy:
    def __init__(
        self,
        *,
        launch_template: _ILaunchTemplate_f32c0fd7,
        instances_distribution: typing.Optional[typing.Union[InstancesDistribution, typing.Dict[builtins.str, typing.Any]]] = None,
        launch_template_overrides: typing.Optional[typing.Sequence[typing.Union[LaunchTemplateOverrides, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''MixedInstancesPolicy allows you to configure a group that diversifies across On-Demand Instances and Spot Instances of multiple instance types.

        For more information, see Auto Scaling groups with
        multiple instance types and purchase options in the Amazon EC2 Auto Scaling User Guide:

        https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-purchase-options.html

        :param launch_template: Launch template to use.
        :param instances_distribution: InstancesDistribution to use. Default: - The value for each property in it uses a default value.
        :param launch_template_overrides: Launch template overrides. The maximum number of instance types that can be associated with an Auto Scaling group is 40. The maximum number of distinct launch templates you can define for an Auto Scaling group is 20. Default: - Do not provide any overrides

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            # launch_template1: ec2.LaunchTemplate
            # launch_template2: ec2.LaunchTemplate
            
            
            autoscaling.AutoScalingGroup(self, "ASG",
                vpc=vpc,
                mixed_instances_policy=autoscaling.MixedInstancesPolicy(
                    instances_distribution=autoscaling.InstancesDistribution(
                        on_demand_percentage_above_base_capacity=50
                    ),
                    launch_template=launch_template1,
                    launch_template_overrides=[autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t3.micro")), autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t3a.micro")), autoscaling.LaunchTemplateOverrides(instance_type=ec2.InstanceType("t4g.micro"), launch_template=launch_template2)]
                )
            )
        '''
        if isinstance(instances_distribution, dict):
            instances_distribution = InstancesDistribution(**instances_distribution)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d30527dded57fb0bc5326004b541ba9a6346091141cc92b30b0eebf2b8d55c70)
            check_type(argname="argument launch_template", value=launch_template, expected_type=type_hints["launch_template"])
            check_type(argname="argument instances_distribution", value=instances_distribution, expected_type=type_hints["instances_distribution"])
            check_type(argname="argument launch_template_overrides", value=launch_template_overrides, expected_type=type_hints["launch_template_overrides"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "launch_template": launch_template,
        }
        if instances_distribution is not None:
            self._values["instances_distribution"] = instances_distribution
        if launch_template_overrides is not None:
            self._values["launch_template_overrides"] = launch_template_overrides

    @builtins.property
    def launch_template(self) -> _ILaunchTemplate_f32c0fd7:
        '''Launch template to use.'''
        result = self._values.get("launch_template")
        assert result is not None, "Required property 'launch_template' is missing"
        return typing.cast(_ILaunchTemplate_f32c0fd7, result)

    @builtins.property
    def instances_distribution(self) -> typing.Optional[InstancesDistribution]:
        '''InstancesDistribution to use.

        :default: - The value for each property in it uses a default value.
        '''
        result = self._values.get("instances_distribution")
        return typing.cast(typing.Optional[InstancesDistribution], result)

    @builtins.property
    def launch_template_overrides(
        self,
    ) -> typing.Optional[typing.List[LaunchTemplateOverrides]]:
        '''Launch template overrides.

        The maximum number of instance types that can be associated with an Auto Scaling group is 40.

        The maximum number of distinct launch templates you can define for an Auto Scaling group is 20.

        :default: - Do not provide any overrides
        '''
        result = self._values.get("launch_template_overrides")
        return typing.cast(typing.Optional[typing.List[LaunchTemplateOverrides]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MixedInstancesPolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.Monitoring")
class Monitoring(enum.Enum):
    '''The monitoring mode for instances launched in an autoscaling group.'''

    BASIC = "BASIC"
    '''Generates metrics every 5 minutes.'''
    DETAILED = "DETAILED"
    '''Generates metrics every minute.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.NetworkUtilizationScalingProps",
    jsii_struct_bases=[BaseTargetTrackingProps],
    name_mapping={
        "cooldown": "cooldown",
        "disable_scale_in": "disableScaleIn",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "target_bytes_per_second": "targetBytesPerSecond",
    },
)
class NetworkUtilizationScalingProps(BaseTargetTrackingProps):
    def __init__(
        self,
        *,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        target_bytes_per_second: jsii.Number,
    ) -> None:
        '''Properties for enabling scaling based on network utilization.

        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        :param target_bytes_per_second: Target average bytes/seconds on each instance.

        :exampleMetadata: infused

        Example::

            # auto_scaling_group: autoscaling.AutoScalingGroup
            
            
            auto_scaling_group.scale_on_incoming_bytes("LimitIngressPerInstance",
                target_bytes_per_second=10 * 1024 * 1024
            )
            auto_scaling_group.scale_on_outgoing_bytes("LimitEgressPerInstance",
                target_bytes_per_second=10 * 1024 * 1024
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4fb0ff49aadc6c65288ae7dbd1487df187b610d8d237b5fee3b7e0edd2d2995)
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument target_bytes_per_second", value=target_bytes_per_second, expected_type=type_hints["target_bytes_per_second"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "target_bytes_per_second": target_bytes_per_second,
        }
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Period after a scaling completes before another scaling activity can start.

        :default: - The default cooldown configured on the AutoScalingGroup.
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the autoscaling group. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        group.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: - Same as the cooldown.
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def target_bytes_per_second(self) -> jsii.Number:
        '''Target average bytes/seconds on each instance.'''
        result = self._values.get("target_bytes_per_second")
        assert result is not None, "Required property 'target_bytes_per_second' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NetworkUtilizationScalingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.NotificationConfiguration",
    jsii_struct_bases=[],
    name_mapping={"topic": "topic", "scaling_events": "scalingEvents"},
)
class NotificationConfiguration:
    def __init__(
        self,
        *,
        topic: _ITopic_9eca4852,
        scaling_events: typing.Optional["ScalingEvents"] = None,
    ) -> None:
        '''AutoScalingGroup fleet change notifications configurations.

        You can configure AutoScaling to send an SNS notification whenever your Auto Scaling group scales.

        :param topic: SNS topic to send notifications about fleet scaling events.
        :param scaling_events: Which fleet scaling events triggers a notification. Default: ScalingEvents.ALL

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            from aws_cdk import aws_sns as sns
            
            # scaling_events: autoscaling.ScalingEvents
            # topic: sns.Topic
            
            notification_configuration = autoscaling.NotificationConfiguration(
                topic=topic,
            
                # the properties below are optional
                scaling_events=scaling_events
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4737cff733fdd5e81936c884477a23c2ee8c2b2a87eb7c80eb0fcd470532d675)
            check_type(argname="argument topic", value=topic, expected_type=type_hints["topic"])
            check_type(argname="argument scaling_events", value=scaling_events, expected_type=type_hints["scaling_events"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "topic": topic,
        }
        if scaling_events is not None:
            self._values["scaling_events"] = scaling_events

    @builtins.property
    def topic(self) -> _ITopic_9eca4852:
        '''SNS topic to send notifications about fleet scaling events.'''
        result = self._values.get("topic")
        assert result is not None, "Required property 'topic' is missing"
        return typing.cast(_ITopic_9eca4852, result)

    @builtins.property
    def scaling_events(self) -> typing.Optional["ScalingEvents"]:
        '''Which fleet scaling events triggers a notification.

        :default: ScalingEvents.ALL
        '''
        result = self._values.get("scaling_events")
        return typing.cast(typing.Optional["ScalingEvents"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NotificationConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.OnDemandAllocationStrategy")
class OnDemandAllocationStrategy(enum.Enum):
    '''Indicates how to allocate instance types to fulfill On-Demand capacity.'''

    PRIORITIZED = "PRIORITIZED"
    '''This strategy uses the order of instance types in the LaunchTemplateOverrides to define the launch priority of each instance type.

    The first instance type in the array is prioritized higher than the
    last. If all your On-Demand capacity cannot be fulfilled using your highest priority instance, then
    the Auto Scaling group launches the remaining capacity using the second priority instance type, and
    so on.
    '''


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.PoolState")
class PoolState(enum.Enum):
    '''The instance state in the warm pool.'''

    HIBERNATED = "HIBERNATED"
    '''Hibernated.

    To use this state, prerequisites must be in place.

    :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/hibernating-prerequisites.html
    '''
    RUNNING = "RUNNING"
    '''Running.'''
    STOPPED = "STOPPED"
    '''Stopped.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.PredefinedMetric")
class PredefinedMetric(enum.Enum):
    '''One of the predefined autoscaling metrics.'''

    ASG_AVERAGE_CPU_UTILIZATION = "ASG_AVERAGE_CPU_UTILIZATION"
    '''Average CPU utilization of the Auto Scaling group.'''
    ASG_AVERAGE_NETWORK_IN = "ASG_AVERAGE_NETWORK_IN"
    '''Average number of bytes received on all network interfaces by the Auto Scaling group.'''
    ASG_AVERAGE_NETWORK_OUT = "ASG_AVERAGE_NETWORK_OUT"
    '''Average number of bytes sent out on all network interfaces by the Auto Scaling group.'''
    ALB_REQUEST_COUNT_PER_TARGET = "ALB_REQUEST_COUNT_PER_TARGET"
    '''Number of requests completed per target in an Application Load Balancer target group.

    Specify the ALB to look at in the ``resourceLabel`` field.
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.RenderSignalsOptions",
    jsii_struct_bases=[],
    name_mapping={
        "desired_capacity": "desiredCapacity",
        "min_capacity": "minCapacity",
    },
)
class RenderSignalsOptions:
    def __init__(
        self,
        *,
        desired_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Input for Signals.renderCreationPolicy.

        :param desired_capacity: The desiredCapacity of the ASG. Default: - desired capacity not configured
        :param min_capacity: The minSize of the ASG. Default: - minCapacity not configured

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            render_signals_options = autoscaling.RenderSignalsOptions(
                desired_capacity=123,
                min_capacity=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__090c95318bc265c8f26c28b8e4acc8be1056a9012ba59b537db4bc3dd26bc436)
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''The desiredCapacity of the ASG.

        :default: - desired capacity not configured
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''The minSize of the ASG.

        :default: - minCapacity not configured
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RenderSignalsOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.RequestCountScalingProps",
    jsii_struct_bases=[BaseTargetTrackingProps],
    name_mapping={
        "cooldown": "cooldown",
        "disable_scale_in": "disableScaleIn",
        "estimated_instance_warmup": "estimatedInstanceWarmup",
        "target_requests_per_minute": "targetRequestsPerMinute",
    },
)
class RequestCountScalingProps(BaseTargetTrackingProps):
    def __init__(
        self,
        *,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        estimated_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        target_requests_per_minute: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for enabling scaling based on request/second.

        :param cooldown: Period after a scaling completes before another scaling activity can start. Default: - The default cooldown configured on the AutoScalingGroup.
        :param disable_scale_in: Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the autoscaling group. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the group. Default: false
        :param estimated_instance_warmup: Estimated time until a newly launched instance can send metrics to CloudWatch. Default: - Same as the cooldown.
        :param target_requests_per_minute: Target average requests/minute on each instance. Default: - Specify exactly one of 'targetRequestsPerMinute' and 'targetRequestsPerSecond'

        :exampleMetadata: infused

        Example::

            # auto_scaling_group: autoscaling.AutoScalingGroup
            
            
            auto_scaling_group.scale_on_request_count("LimitRPS",
                target_requests_per_second=1000
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__775d27074efc25014f9d47b49e64941b878e6dd1129f5eb7c675a5e5c1b50432)
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument disable_scale_in", value=disable_scale_in, expected_type=type_hints["disable_scale_in"])
            check_type(argname="argument estimated_instance_warmup", value=estimated_instance_warmup, expected_type=type_hints["estimated_instance_warmup"])
            check_type(argname="argument target_requests_per_minute", value=target_requests_per_minute, expected_type=type_hints["target_requests_per_minute"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if estimated_instance_warmup is not None:
            self._values["estimated_instance_warmup"] = estimated_instance_warmup
        if target_requests_per_minute is not None:
            self._values["target_requests_per_minute"] = target_requests_per_minute

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Period after a scaling completes before another scaling activity can start.

        :default: - The default cooldown configured on the AutoScalingGroup.
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the autoscaling group. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        group.

        :default: false
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def estimated_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Estimated time until a newly launched instance can send metrics to CloudWatch.

        :default: - Same as the cooldown.
        '''
        result = self._values.get("estimated_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def target_requests_per_minute(self) -> typing.Optional[jsii.Number]:
        '''Target average requests/minute on each instance.

        :default: - Specify exactly one of 'targetRequestsPerMinute' and 'targetRequestsPerSecond'
        '''
        result = self._values.get("target_requests_per_minute")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RequestCountScalingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.RollingUpdateOptions",
    jsii_struct_bases=[],
    name_mapping={
        "max_batch_size": "maxBatchSize",
        "min_instances_in_service": "minInstancesInService",
        "min_success_percentage": "minSuccessPercentage",
        "pause_time": "pauseTime",
        "suspend_processes": "suspendProcesses",
        "wait_on_resource_signals": "waitOnResourceSignals",
    },
)
class RollingUpdateOptions:
    def __init__(
        self,
        *,
        max_batch_size: typing.Optional[jsii.Number] = None,
        min_instances_in_service: typing.Optional[jsii.Number] = None,
        min_success_percentage: typing.Optional[jsii.Number] = None,
        pause_time: typing.Optional[_Duration_4839e8c3] = None,
        suspend_processes: typing.Optional[typing.Sequence["ScalingProcess"]] = None,
        wait_on_resource_signals: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for customizing the rolling update.

        :param max_batch_size: The maximum number of instances that AWS CloudFormation updates at once. This number affects the speed of the replacement. Default: 1
        :param min_instances_in_service: The minimum number of instances that must be in service before more instances are replaced. This number affects the speed of the replacement. Default: 0
        :param min_success_percentage: The percentage of instances that must signal success for the update to succeed. Default: - The ``minSuccessPercentage`` configured for ``signals`` on the AutoScalingGroup
        :param pause_time: The pause time after making a change to a batch of instances. Default: - The ``timeout`` configured for ``signals`` on the AutoScalingGroup
        :param suspend_processes: Specifies the Auto Scaling processes to suspend during a stack update. Suspending processes prevents Auto Scaling from interfering with a stack update. Default: HealthCheck, ReplaceUnhealthy, AZRebalance, AlarmNotification, ScheduledActions.
        :param wait_on_resource_signals: Specifies whether the Auto Scaling group waits on signals from new instances during an update. Default: true if you configured ``signals`` on the AutoScalingGroup, false otherwise

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_autoscaling as autoscaling
            
            rolling_update_options = autoscaling.RollingUpdateOptions(
                max_batch_size=123,
                min_instances_in_service=123,
                min_success_percentage=123,
                pause_time=cdk.Duration.minutes(30),
                suspend_processes=[autoscaling.ScalingProcess.LAUNCH],
                wait_on_resource_signals=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dba85a906170b46d411e476ef5bc0b166e1b32acff9d41d326761c193522424f)
            check_type(argname="argument max_batch_size", value=max_batch_size, expected_type=type_hints["max_batch_size"])
            check_type(argname="argument min_instances_in_service", value=min_instances_in_service, expected_type=type_hints["min_instances_in_service"])
            check_type(argname="argument min_success_percentage", value=min_success_percentage, expected_type=type_hints["min_success_percentage"])
            check_type(argname="argument pause_time", value=pause_time, expected_type=type_hints["pause_time"])
            check_type(argname="argument suspend_processes", value=suspend_processes, expected_type=type_hints["suspend_processes"])
            check_type(argname="argument wait_on_resource_signals", value=wait_on_resource_signals, expected_type=type_hints["wait_on_resource_signals"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_batch_size is not None:
            self._values["max_batch_size"] = max_batch_size
        if min_instances_in_service is not None:
            self._values["min_instances_in_service"] = min_instances_in_service
        if min_success_percentage is not None:
            self._values["min_success_percentage"] = min_success_percentage
        if pause_time is not None:
            self._values["pause_time"] = pause_time
        if suspend_processes is not None:
            self._values["suspend_processes"] = suspend_processes
        if wait_on_resource_signals is not None:
            self._values["wait_on_resource_signals"] = wait_on_resource_signals

    @builtins.property
    def max_batch_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of instances that AWS CloudFormation updates at once.

        This number affects the speed of the replacement.

        :default: 1
        '''
        result = self._values.get("max_batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_instances_in_service(self) -> typing.Optional[jsii.Number]:
        '''The minimum number of instances that must be in service before more instances are replaced.

        This number affects the speed of the replacement.

        :default: 0
        '''
        result = self._values.get("min_instances_in_service")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_success_percentage(self) -> typing.Optional[jsii.Number]:
        '''The percentage of instances that must signal success for the update to succeed.

        :default: - The ``minSuccessPercentage`` configured for ``signals`` on the AutoScalingGroup
        '''
        result = self._values.get("min_success_percentage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def pause_time(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The pause time after making a change to a batch of instances.

        :default: - The ``timeout`` configured for ``signals`` on the AutoScalingGroup
        '''
        result = self._values.get("pause_time")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def suspend_processes(self) -> typing.Optional[typing.List["ScalingProcess"]]:
        '''Specifies the Auto Scaling processes to suspend during a stack update.

        Suspending processes prevents Auto Scaling from interfering with a stack
        update.

        :default: HealthCheck, ReplaceUnhealthy, AZRebalance, AlarmNotification, ScheduledActions.
        '''
        result = self._values.get("suspend_processes")
        return typing.cast(typing.Optional[typing.List["ScalingProcess"]], result)

    @builtins.property
    def wait_on_resource_signals(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the Auto Scaling group waits on signals from new instances during an update.

        :default: true if you configured ``signals`` on the AutoScalingGroup, false otherwise
        '''
        result = self._values.get("wait_on_resource_signals")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RollingUpdateOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.ScalingEvent")
class ScalingEvent(enum.Enum):
    '''Fleet scaling events.'''

    INSTANCE_LAUNCH = "INSTANCE_LAUNCH"
    '''Notify when an instance was launched.'''
    INSTANCE_TERMINATE = "INSTANCE_TERMINATE"
    '''Notify when an instance was terminated.'''
    INSTANCE_TERMINATE_ERROR = "INSTANCE_TERMINATE_ERROR"
    '''Notify when an instance failed to terminate.'''
    INSTANCE_LAUNCH_ERROR = "INSTANCE_LAUNCH_ERROR"
    '''Notify when an instance failed to launch.'''
    TEST_NOTIFICATION = "TEST_NOTIFICATION"
    '''Send a test notification to the topic.'''


class ScalingEvents(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.ScalingEvents",
):
    '''A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS or create a custom group by instantiating a ``NotificationTypes`` object, e.g: ``new NotificationTypes(``NotificationType.INSTANCE_LAUNCH``)``.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        scaling_events = autoscaling.ScalingEvents.ALL
    '''

    def __init__(self, *types: ScalingEvent) -> None:
        '''
        :param types: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__59a1f6e5c4bdeaea04fe231d617fe9c85e0856a08352d2ea8f913e7ac33d0d84)
            check_type(argname="argument types", value=types, expected_type=typing.Tuple[type_hints["types"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        jsii.create(self.__class__, self, [*types])

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALL")
    def ALL(cls) -> "ScalingEvents":
        '''All fleet scaling events.'''
        return typing.cast("ScalingEvents", jsii.sget(cls, "ALL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ERRORS")
    def ERRORS(cls) -> "ScalingEvents":
        '''Fleet scaling errors.'''
        return typing.cast("ScalingEvents", jsii.sget(cls, "ERRORS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LAUNCH_EVENTS")
    def LAUNCH_EVENTS(cls) -> "ScalingEvents":
        '''Fleet scaling launch events.'''
        return typing.cast("ScalingEvents", jsii.sget(cls, "LAUNCH_EVENTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TERMINATION_EVENTS")
    def TERMINATION_EVENTS(cls) -> "ScalingEvents":
        '''Fleet termination launch events.'''
        return typing.cast("ScalingEvents", jsii.sget(cls, "TERMINATION_EVENTS"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.ScalingInterval",
    jsii_struct_bases=[],
    name_mapping={"change": "change", "lower": "lower", "upper": "upper"},
)
class ScalingInterval:
    def __init__(
        self,
        *,
        change: jsii.Number,
        lower: typing.Optional[jsii.Number] = None,
        upper: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''A range of metric values in which to apply a certain scaling operation.

        :param change: The capacity adjustment to apply in this interval. The number is interpreted differently based on AdjustmentType: - ChangeInCapacity: add the adjustment to the current capacity. The number can be positive or negative. - PercentChangeInCapacity: add or remove the given percentage of the current capacity to itself. The number can be in the range [-100..100]. - ExactCapacity: set the capacity to this number. The number must be positive.
        :param lower: The lower bound of the interval. The scaling adjustment will be applied if the metric is higher than this value. Default: Threshold automatically derived from neighbouring intervals
        :param upper: The upper bound of the interval. The scaling adjustment will be applied if the metric is lower than this value. Default: Threshold automatically derived from neighbouring intervals

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            scaling_interval = autoscaling.ScalingInterval(
                change=123,
            
                # the properties below are optional
                lower=123,
                upper=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b560eae9332d2a2b0774bea52aea5a16c19db15d41968c9f1bcfba58f9345ff1)
            check_type(argname="argument change", value=change, expected_type=type_hints["change"])
            check_type(argname="argument lower", value=lower, expected_type=type_hints["lower"])
            check_type(argname="argument upper", value=upper, expected_type=type_hints["upper"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "change": change,
        }
        if lower is not None:
            self._values["lower"] = lower
        if upper is not None:
            self._values["upper"] = upper

    @builtins.property
    def change(self) -> jsii.Number:
        '''The capacity adjustment to apply in this interval.

        The number is interpreted differently based on AdjustmentType:

        - ChangeInCapacity: add the adjustment to the current capacity.
          The number can be positive or negative.
        - PercentChangeInCapacity: add or remove the given percentage of the current
          capacity to itself. The number can be in the range [-100..100].
        - ExactCapacity: set the capacity to this number. The number must
          be positive.
        '''
        result = self._values.get("change")
        assert result is not None, "Required property 'change' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def lower(self) -> typing.Optional[jsii.Number]:
        '''The lower bound of the interval.

        The scaling adjustment will be applied if the metric is higher than this value.

        :default: Threshold automatically derived from neighbouring intervals
        '''
        result = self._values.get("lower")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def upper(self) -> typing.Optional[jsii.Number]:
        '''The upper bound of the interval.

        The scaling adjustment will be applied if the metric is lower than this value.

        :default: Threshold automatically derived from neighbouring intervals
        '''
        result = self._values.get("upper")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ScalingInterval(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_autoscaling.ScalingProcess")
class ScalingProcess(enum.Enum):
    LAUNCH = "LAUNCH"
    TERMINATE = "TERMINATE"
    HEALTH_CHECK = "HEALTH_CHECK"
    REPLACE_UNHEALTHY = "REPLACE_UNHEALTHY"
    AZ_REBALANCE = "AZ_REBALANCE"
    ALARM_NOTIFICATION = "ALARM_NOTIFICATION"
    SCHEDULED_ACTIONS = "SCHEDULED_ACTIONS"
    ADD_TO_LOAD_BALANCER = "ADD_TO_LOAD_BALANCER"


class Schedule(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_autoscaling.Schedule",
):
    '''Schedule for scheduled scaling actions.

    :exampleMetadata: infused

    Example::

        # auto_scaling_group: autoscaling.AutoScalingGroup
        
        
        auto_scaling_group.scale_on_schedule("PrescaleInTheMorning",
            schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
            min_capacity=20
        )
        
        auto_scaling_group.scale_on_schedule("AllowDownscalingAtNight",
            schedule=autoscaling.Schedule.cron(hour="20", minute="0"),
            min_capacity=1
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="cron")
    @builtins.classmethod
    def cron(
        cls,
        *,
        day: typing.Optional[builtins.str] = None,
        hour: typing.Optional[builtins.str] = None,
        minute: typing.Optional[builtins.str] = None,
        month: typing.Optional[builtins.str] = None,
        week_day: typing.Optional[builtins.str] = None,
    ) -> "Schedule":
        '''Create a schedule from a set of cron fields.

        :param day: The day of the month to run this rule at. Default: - Every day of the month
        :param hour: The hour to run this rule at. Default: - Every hour
        :param minute: The minute to run this rule at. Default: - Every minute
        :param month: The month to run this rule at. Default: - Every month
        :param week_day: The day of the week to run this rule at. Default: - Any day of the week
        '''
        options = CronOptions(
            day=day, hour=hour, minute=minute, month=month, week_day=week_day
        )

        return typing.cast("Schedule", jsii.sinvoke(cls, "cron", [options]))

    @jsii.member(jsii_name="expression")
    @builtins.classmethod
    def expression(cls, expression: builtins.str) -> "Schedule":
        '''Construct a schedule from a literal schedule expression.

        :param expression: The expression to use. Must be in a format that AutoScaling will recognize

        :see: http://crontab.org/
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a472ca2650776afb4c410eee0e25a484de0070b51e7dfc154b5506f87b8a6994)
            check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
        return typing.cast("Schedule", jsii.sinvoke(cls, "expression", [expression]))

    @builtins.property
    @jsii.member(jsii_name="expressionString")
    @abc.abstractmethod
    def expression_string(self) -> builtins.str:
        '''Retrieve the expression for this schedule.'''
        ...


class _ScheduleProxy(Schedule):
    @builtins.property
    @jsii.member(jsii_name="expressionString")
    def expression_string(self) -> builtins.str:
        '''Retrieve the expression for this schedule.'''
        return typing.cast(builtins.str, jsii.get(self, "expressionString"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Schedule).__jsii_proxy_class__ = lambda : _ScheduleProxy


class ScheduledAction(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_autoscaling.ScheduledAction",
):
    '''Define a scheduled scaling action.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_autoscaling as autoscaling
        
        # auto_scaling_group: autoscaling.AutoScalingGroup
        # schedule: autoscaling.Schedule
        
        scheduled_action = autoscaling.ScheduledAction(self, "MyScheduledAction",
            auto_scaling_group=auto_scaling_group,
            schedule=schedule,
        
            # the properties below are optional
            desired_capacity=123,
            end_time=Date(),
            max_capacity=123,
            min_capacity=123,
            start_time=Date(),
            time_zone="timeZone"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auto_scaling_group: IAutoScalingGroup,
        schedule: Schedule,
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
        time_zone: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param auto_scaling_group: The AutoScalingGroup to apply the scheduled actions to.
        :param schedule: When to perform this action. Supports cron expressions. For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        :param desired_capacity: The new desired capacity. At the scheduled time, set the desired capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new desired capacity.
        :param end_time: When this scheduled action expires. Default: - The rule never expires.
        :param max_capacity: The new maximum capacity. At the scheduled time, set the maximum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new maximum capacity.
        :param min_capacity: The new minimum capacity. At the scheduled time, set the minimum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new minimum capacity.
        :param start_time: When this scheduled action becomes active. Default: - The rule is activate immediately.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti). For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones. Default: - UTC
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74b3b8cb09553edb86fdacb4ea37f2427f0041f519ffb8025514989ca3a7ea4c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ScheduledActionProps(
            auto_scaling_group=auto_scaling_group,
            schedule=schedule,
            desired_capacity=desired_capacity,
            end_time=end_time,
            max_capacity=max_capacity,
            min_capacity=min_capacity,
            start_time=start_time,
            time_zone=time_zone,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="scheduledActionName")
    def scheduled_action_name(self) -> builtins.str:
        '''The name of the scheduled action.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "scheduledActionName"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.ScheduledActionProps",
    jsii_struct_bases=[BasicScheduledActionProps],
    name_mapping={
        "schedule": "schedule",
        "desired_capacity": "desiredCapacity",
        "end_time": "endTime",
        "max_capacity": "maxCapacity",
        "min_capacity": "minCapacity",
        "start_time": "startTime",
        "time_zone": "timeZone",
        "auto_scaling_group": "autoScalingGroup",
    },
)
class ScheduledActionProps(BasicScheduledActionProps):
    def __init__(
        self,
        *,
        schedule: Schedule,
        desired_capacity: typing.Optional[jsii.Number] = None,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
        time_zone: typing.Optional[builtins.str] = None,
        auto_scaling_group: IAutoScalingGroup,
    ) -> None:
        '''Properties for a scheduled action on an AutoScalingGroup.

        :param schedule: When to perform this action. Supports cron expressions. For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        :param desired_capacity: The new desired capacity. At the scheduled time, set the desired capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new desired capacity.
        :param end_time: When this scheduled action expires. Default: - The rule never expires.
        :param max_capacity: The new maximum capacity. At the scheduled time, set the maximum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new maximum capacity.
        :param min_capacity: The new minimum capacity. At the scheduled time, set the minimum capacity to the given capacity. At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied. Default: - No new minimum capacity.
        :param start_time: When this scheduled action becomes active. Default: - The rule is activate immediately.
        :param time_zone: Specifies the time zone for a cron expression. If a time zone is not provided, UTC is used by default. Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti). For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones. Default: - UTC
        :param auto_scaling_group: The AutoScalingGroup to apply the scheduled actions to.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_autoscaling as autoscaling
            
            # auto_scaling_group: autoscaling.AutoScalingGroup
            # schedule: autoscaling.Schedule
            
            scheduled_action_props = autoscaling.ScheduledActionProps(
                auto_scaling_group=auto_scaling_group,
                schedule=schedule,
            
                # the properties below are optional
                desired_capacity=123,
                end_time=Date(),
                max_capacity=123,
                min_capacity=123,
                start_time=Date(),
                time_zone="timeZone"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32b96efb9d6e83028fe8f913de37e12991ddcd1c9b905f945ec06666919aecc6)
            check_type(argname="argument schedule", value=schedule, expected_type=type_hints["schedule"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument end_time", value=end_time, expected_type=type_hints["end_time"])
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            check_type(argname="argument start_time", value=start_time, expected_type=type_hints["start_time"])
            check_type(argname="argument time_zone", value=time_zone, expected_type=type_hints["time_zone"])
            check_type(argname="argument auto_scaling_group", value=auto_scaling_group, expected_type=type_hints["auto_scaling_group"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "schedule": schedule,
            "auto_scaling_group": auto_scaling_group,
        }
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if end_time is not None:
            self._values["end_time"] = end_time
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if start_time is not None:
            self._values["start_time"] = start_time
        if time_zone is not None:
            self._values["time_zone"] = time_zone

    @builtins.property
    def schedule(self) -> Schedule:
        '''When to perform this action.

        Supports cron expressions.

        For more information about cron expressions, see https://en.wikipedia.org/wiki/Cron.
        '''
        result = self._values.get("schedule")
        assert result is not None, "Required property 'schedule' is missing"
        return typing.cast(Schedule, result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''The new desired capacity.

        At the scheduled time, set the desired capacity to the given capacity.

        At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied.

        :default: - No new desired capacity.
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def end_time(self) -> typing.Optional[datetime.datetime]:
        '''When this scheduled action expires.

        :default: - The rule never expires.
        '''
        result = self._values.get("end_time")
        return typing.cast(typing.Optional[datetime.datetime], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''The new maximum capacity.

        At the scheduled time, set the maximum capacity to the given capacity.

        At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied.

        :default: - No new maximum capacity.
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''The new minimum capacity.

        At the scheduled time, set the minimum capacity to the given capacity.

        At least one of maxCapacity, minCapacity, or desiredCapacity must be supplied.

        :default: - No new minimum capacity.
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def start_time(self) -> typing.Optional[datetime.datetime]:
        '''When this scheduled action becomes active.

        :default: - The rule is activate immediately.
        '''
        result = self._values.get("start_time")
        return typing.cast(typing.Optional[datetime.datetime], result)

    @builtins.property
    def time_zone(self) -> typing.Optional[builtins.str]:
        '''Specifies the time zone for a cron expression.

        If a time zone is not provided, UTC is used by default.

        Valid values are the canonical names of the IANA time zones, derived from the IANA Time Zone Database (such as Etc/GMT+9 or Pacific/Tahiti).

        For more information, see https://en.wikipedia.org/wiki/List_of_tz_database_time_zones.

        :default: - UTC
        '''
        result = self._values.get("time_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auto_scaling_group(self) -> IAutoScalingGroup:
        '''The AutoScalingGroup to apply the scheduled actions to.'''
        result = self._values.get("auto_scaling_group")
        assert result is not None, "Required property 'auto_scaling_group' is missing"
        return typing.cast(IAutoScalingGroup, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ScheduledActionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Signals(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_autoscaling.Signals",
):
    '''Configure whether the AutoScalingGroup waits for signals.

    If you do configure waiting for signals, you should make sure the instances
    invoke ``cfn-signal`` somewhere in their UserData to signal that they have
    started up (either successfully or unsuccessfully).

    Signals are used both during intial creation and subsequent updates.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        # machine_image: ec2.IMachineImage
        
        
        autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
        
            # ...
        
            init=ec2.CloudFormationInit.from_elements(
                ec2.InitFile.from_string("/etc/my_instance", "This got written during instance startup")),
            signals=autoscaling.Signals.wait_for_all(
                timeout=Duration.minutes(10)
            )
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="waitForAll")
    @builtins.classmethod
    def wait_for_all(
        cls,
        *,
        min_success_percentage: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "Signals":
        '''Wait for the desiredCapacity of the AutoScalingGroup amount of signals to have been received.

        If no desiredCapacity has been configured, wait for minCapacity signals intead.

        This number is used during initial creation and during replacing updates.
        During rolling updates, all updated instances must send a signal.

        :param min_success_percentage: The percentage of signals that need to be successful. If this number is less than 100, a percentage of signals may be failure signals while still succeeding the creation or update in CloudFormation. Default: 100
        :param timeout: How long to wait for the signals to be sent. This should reflect how long it takes your instances to start up (including instance start time and instance initialization time). Default: Duration.minutes(5)
        '''
        options = SignalsOptions(
            min_success_percentage=min_success_percentage, timeout=timeout
        )

        return typing.cast("Signals", jsii.sinvoke(cls, "waitForAll", [options]))

    @jsii.member(jsii_name="waitForCount")
    @builtins.classmethod
    def wait_for_count(
        cls,
        count: jsii.Number,
        *,
        min_success_percentage: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "Signals":
        '''Wait for a specific amount of signals to have been received.

        You should send one signal per instance, so this represents the number of
        instances to wait for.

        This number is used during initial creation and during replacing updates.
        During rolling updates, all updated instances must send a signal.

        :param count: -
        :param min_success_percentage: The percentage of signals that need to be successful. If this number is less than 100, a percentage of signals may be failure signals while still succeeding the creation or update in CloudFormation. Default: 100
        :param timeout: How long to wait for the signals to be sent. This should reflect how long it takes your instances to start up (including instance start time and instance initialization time). Default: Duration.minutes(5)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ae8cd20997fe3f570d309ce263568430957611d2fa82c5b72982de8187e57ee)
            check_type(argname="argument count", value=count, expected_type=type_hints["count"])
        options = SignalsOptions(
            min_success_percentage=min_success_percentage, timeout=timeout
        )

        return typing.cast("Signals", jsii.sinvoke(cls, "waitForCount", [count, options]))

    @jsii.member(jsii_name="waitForMinCapacity")
    @builtins.classmethod
    def wait_for_min_capacity(
        cls,
        *,
        min_success_percentage: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "Signals":
        '''Wait for the minCapacity of the AutoScalingGroup amount of signals to have been received.

        This number is used during initial creation and during replacing updates.
        During rolling updates, all updated instances must send a signal.

        :param min_success_percentage: The percentage of signals that need to be successful. If this number is less than 100, a percentage of signals may be failure signals while still succeeding the creation or update in CloudFormation. Default: 100
        :param timeout: How long to wait for the signals to be sent. This should reflect how long it takes your instances to start up (including instance start time and instance initialization time). Default: Duration.minutes(5)
        '''
        options = SignalsOptions(
            min_success_percentage=min_success_percentage, timeout=timeout
        )

        return typing.cast("Signals", jsii.sinvoke(cls, "waitForMinCapacity", [options]))

    @jsii.member(jsii_name="doRender")
    def _do_render(
        self,
        options: typing.Union["SignalsOptions", typing.Dict[builtins.str, typing.Any]],
        count: typing.Optional[jsii.Number] = None,
    ) -> _CfnCreationPolicy_d904f690:
        '''Helper to render the actual creation policy, as the logic between them is quite similar.

        :param options: -
        :param count: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__396126a6b5062bb7e41dd20b4b29fdf1ea21d824daf9e8e1f51808c34cdcaacb)
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument count", value=count, expected_type=type_hints["count"])
        return typing.cast(_CfnCreationPolicy_d904f690, jsii.invoke(self, "doRender", [options, count]))

    @jsii.member(jsii_name="renderCreationPolicy")
    @abc.abstractmethod
    def render_creation_policy(
        self,
        *,
        desired_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
    ) -> _CfnCreationPolicy_d904f690:
        '''Render the ASG's CreationPolicy.

        :param desired_capacity: The desiredCapacity of the ASG. Default: - desired capacity not configured
        :param min_capacity: The minSize of the ASG. Default: - minCapacity not configured
        '''
        ...


class _SignalsProxy(Signals):
    @jsii.member(jsii_name="renderCreationPolicy")
    def render_creation_policy(
        self,
        *,
        desired_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
    ) -> _CfnCreationPolicy_d904f690:
        '''Render the ASG's CreationPolicy.

        :param desired_capacity: The desiredCapacity of the ASG. Default: - desired capacity not configured
        :param min_capacity: The minSize of the ASG. Default: - minCapacity not configured
        '''
        render_options = RenderSignalsOptions(
            desired_capacity=desired_capacity, min_capacity=min_capacity
        )

        return typing.cast(_CfnCreationPolicy_d904f690, jsii.invoke(self, "renderCreationPolicy", [render_options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Signals).__jsii_proxy_class__ = lambda : _SignalsProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_autoscaling.SignalsOptions",
    jsii_struct_bases=[],
    name_mapping={
        "min_success_percentage": "minSuccessPercentage",
        "timeout": "timeout",
    },
)
class SignalsOptions:
    def __init__(
        self,
        *,
        min_success_percentage: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Customization options for Signal handling.

        :param min_success_percentage: The percentage of signals that need to be successful. If this number is less than 100, a percentage of signals may be failure signals while still succeeding the creation or update in CloudFormation. Default: 100
        :param timeout: How long to wait for the signals to be sent. This should reflect how long it takes your instances to start up (including instance start time and instance initialization time). Default: Duration.minutes(5)

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            # instance_type: ec2.InstanceType
            # machine_image: ec2.IMachineImage
            
            
            autoscaling.AutoScalingGroup(self, "ASG",
                vpc=vpc,
                instance_type=instance_type,
                machine_image=machine_image,
            
                # ...
            
                init=ec2.CloudFormationInit.from_elements(
                    ec2.InitFile.from_string("/etc/my_instance", "This got written during instance startup")),
                signals=autoscaling.Signals.wait_for_all(
                    timeout=Duration.minutes(10)
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7f7a9dc73c47b7dc32e559e38629c11dfe39ea3c3b870aff1c55c507ce4574e)
            check_type(argname="argument min_success_percentage", value=min_success_percentage, expected_type=type_hints["min_success_percentage"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if min_success_percentage is not None:
            self._values["min_success_percentage"] = min_success_percentage
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def min_success_percentage(self) -> typing.Optional[jsii.Number]:
        '''The percent