'''
# Tasks for AWS Step Functions

[AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/welcome.html) is a web service that enables you to coordinate the
components of distributed applications and microservices using visual workflows.
You build applications from individual components that each perform a discrete
function, or task, allowing you to scale and change applications quickly.

A [Task](https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-task-state.html) state represents a single unit of work performed by a state machine.
All work in your state machine is performed by tasks.  This module contains a collection of classes that allow you to call various AWS services
from your Step Functions state machine.

Be sure to familiarize yourself with the [`aws-stepfunctions` module documentation](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions-readme.html) first.

This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.

## Table Of Contents

* [Tasks for AWS Step Functions](#tasks-for-aws-step-functions)

  * [Table Of Contents](#table-of-contents)
  * [Evaluate Expression](#evaluate-expression)
  * [API Gateway](#api-gateway)

    * [Call REST API Endpoint](#call-rest-api-endpoint)
    * [Call HTTP API Endpoint](#call-http-api-endpoint)
  * [AWS SDK](#aws-sdk)
  * [Athena](#athena)

    * [StartQueryExecution](#startqueryexecution)
    * [GetQueryExecution](#getqueryexecution)
    * [GetQueryResults](#getqueryresults)
    * [StopQueryExecution](#stopqueryexecution)
  * [Batch](#batch)

    * [SubmitJob](#submitjob)
  * [CodeBuild](#codebuild)

    * [StartBuild](#startbuild)
  * [DynamoDB](#dynamodb)

    * [GetItem](#getitem)
    * [PutItem](#putitem)
    * [DeleteItem](#deleteitem)
    * [UpdateItem](#updateitem)
  * [ECS](#ecs)

    * [RunTask](#runtask)

      * [EC2](#ec2)
      * [Fargate](#fargate)
  * [EMR](#emr)

    * [Create Cluster](#create-cluster)
    * [Termination Protection](#termination-protection)
    * [Terminate Cluster](#terminate-cluster)
    * [Add Step](#add-step)
    * [Cancel Step](#cancel-step)
    * [Modify Instance Fleet](#modify-instance-fleet)
    * [Modify Instance Group](#modify-instance-group)
  * [EMR on EKS](#emr-on-eks)

    * [Create Virtual Cluster](#create-virtual-cluster)
    * [Delete Virtual Cluster](#delete-virtual-cluster)
    * [Start Job Run](#start-job-run)
  * [EKS](#eks)

    * [Call](#call)
  * [EventBridge](#eventbridge)

    * [Put Events](#put-events)
  * [Glue](#glue)
  * [Glue DataBrew](#glue-databrew)
  * [Lambda](#lambda)
  * [SageMaker](#sagemaker)

    * [Create Training Job](#create-training-job)
    * [Create Transform Job](#create-transform-job)
    * [Create Endpoint](#create-endpoint)
    * [Create Endpoint Config](#create-endpoint-config)
    * [Create Model](#create-model)
    * [Update Endpoint](#update-endpoint)
  * [SNS](#sns)
  * [Step Functions](#step-functions)

    * [Start Execution](#start-execution)
    * [Invoke Activity](#invoke-activity)
  * [SQS](#sqs)

## Paths

Learn more about input and output processing in Step Functions [here](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html)

## Evaluate Expression

Use the `EvaluateExpression` to perform simple operations referencing state paths. The
`expression` referenced in the task will be evaluated in a Lambda function
(`eval()`). This allows you to not have to write Lambda code for simple operations.

Example: convert a wait time from milliseconds to seconds, concat this in a message and wait:

```python
convert_to_seconds = tasks.EvaluateExpression(self, "Convert to seconds",
    expression="$.waitMilliseconds / 1000",
    result_path="$.waitSeconds"
)

create_message = tasks.EvaluateExpression(self, "Create message",
    # Note: this is a string inside a string.
    expression="`Now waiting ${$.waitSeconds} seconds...`",
    runtime=lambda_.Runtime.NODEJS_16_X,
    result_path="$.message"
)

publish_message = tasks.SnsPublish(self, "Publish message",
    topic=sns.Topic(self, "cool-topic"),
    message=sfn.TaskInput.from_json_path_at("$.message"),
    result_path="$.sns"
)

wait = sfn.Wait(self, "Wait",
    time=sfn.WaitTime.seconds_path("$.waitSeconds")
)

sfn.StateMachine(self, "StateMachine",
    definition=convert_to_seconds.next(create_message).next(publish_message).next(wait)
)
```

The `EvaluateExpression` supports a `runtime` prop to specify the Lambda
runtime to use to evaluate the expression. Currently, only runtimes
of the Node.js family are supported.

## API Gateway

Step Functions supports [API Gateway](https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html) through the service integration pattern.

HTTP APIs are designed for low-latency, cost-effective integrations with AWS services, including AWS Lambda, and HTTP endpoints.
HTTP APIs support OIDC and OAuth 2.0 authorization, and come with built-in support for CORS and automatic deployments.
Previous-generation REST APIs currently offer more features. More details can be found [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-vs-rest.html).

### Call REST API Endpoint

The `CallApiGatewayRestApiEndpoint` calls the REST API endpoint.

```python
import aws_cdk.aws_apigateway as apigateway

rest_api = apigateway.RestApi(self, "MyRestApi")

invoke_task = tasks.CallApiGatewayRestApiEndpoint(self, "Call REST API",
    api=rest_api,
    stage_name="prod",
    method=tasks.HttpMethod.GET
)
```

Be aware that the header values must be arrays. When passing the Task Token
in the headers field `WAIT_FOR_TASK_TOKEN` integration, use
`JsonPath.array()` to wrap the token in an array:

```python
import aws_cdk.aws_apigateway as apigateway
# api: apigateway.RestApi


tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
    api=api,
    stage_name="Stage",
    method=tasks.HttpMethod.PUT,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    headers=sfn.TaskInput.from_object({
        "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
    })
)
```

### Call HTTP API Endpoint

The `CallApiGatewayHttpApiEndpoint` calls the HTTP API endpoint.

```python
# Example automatically generated from non-compiling source. May contain errors.
import aws_cdk.aws_apigatewayv2 as apigatewayv2

http_api = apigatewayv2.HttpApi(self, "MyHttpApi")

invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
    api_id=http_api.api_id,
    api_stack=Stack.of(http_api),
    method=tasks.HttpMethod.GET
)
```

### AWS SDK

Step Functions supports calling [AWS service's API actions](https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html)
through the service integration pattern.

You can use Step Functions' AWS SDK integrations to call any of the over two hundred AWS services
directly from your state machine, giving you access to over nine thousand API actions.

```python
# my_bucket: s3.Bucket

get_object = tasks.CallAwsService(self, "GetObject",
    service="s3",
    action="getObject",
    parameters={
        "Bucket": my_bucket.bucket_name,
        "Key": sfn.JsonPath.string_at("$.key")
    },
    iam_resources=[my_bucket.arn_for_objects("*")]
)
```

Use camelCase for actions and PascalCase for parameter names.

The task automatically adds an IAM statement to the state machine role's policy based on the
service and action called. The resources for this statement must be specified in `iamResources`.

Use the `iamAction` prop to manually specify the IAM action name in the case where the IAM
action name does not match with the API service/action name:

```python
list_buckets = tasks.CallAwsService(self, "ListBuckets",
    service="s3",
    action="listBuckets",
    iam_resources=["*"],
    iam_action="s3:ListAllMyBuckets"
)
```

Use the `additionalIamStatements` prop to pass additional IAM statements that will be added to the
state machine role's policy. Use it in the case where the call requires more than a single statement
to be executed:

```python
detect_labels = tasks.CallAwsService(self, "DetectLabels",
    service="rekognition",
    action="detectLabels",
    iam_resources=["*"],
    additional_iam_statements=[
        iam.PolicyStatement(
            actions=["s3:getObject"],
            resources=["arn:aws:s3:::my-bucket/*"]
        )
    ]
)
```

## Athena

Step Functions supports [Athena](https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html) through the service integration pattern.

### StartQueryExecution

The [StartQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html) API runs the SQL query statement.

```python
start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
    query_string=sfn.JsonPath.string_at("$.queryString"),
    query_execution_context=tasks.QueryExecutionContext(
        database_name="mydatabase"
    ),
    result_configuration=tasks.ResultConfiguration(
        encryption_configuration=tasks.EncryptionConfiguration(
            encryption_option=tasks.EncryptionOption.S3_MANAGED
        ),
        output_location=s3.Location(
            bucket_name="query-results-bucket",
            object_key="folder"
        )
    )
)
```

### GetQueryExecution

The [GetQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html) API gets information about a single execution of a query.

```python
get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

### GetQueryResults

The [GetQueryResults](https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html) API that streams the results of a single query execution specified by QueryExecutionId from S3.

```python
get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

### StopQueryExecution

The [StopQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html) API that stops a query execution.

```python
stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

## Batch

Step Functions supports [Batch](https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html) through the service integration pattern.

### SubmitJob

The [SubmitJob](https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html) API submits an AWS Batch job from a job definition.

```python
# Example automatically generated from non-compiling source. May contain errors.
import aws_cdk.aws_batch as batch
# batch_job_definition: batch.JobDefinition
# batch_queue: batch.JobQueue


task = tasks.BatchSubmitJob(self, "Submit Job",
    job_definition_arn=batch_job_definition.job_definition_arn,
    job_name="MyJob",
    job_queue_arn=batch_queue.job_queue_arn
)
```

## CodeBuild

Step Functions supports [CodeBuild](https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html) through the service integration pattern.

### StartBuild

[StartBuild](https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html) starts a CodeBuild Project by Project Name.

```python
import aws_cdk.aws_codebuild as codebuild


codebuild_project = codebuild.Project(self, "Project",
    project_name="MyTestProject",
    build_spec=codebuild.BuildSpec.from_object({
        "version": "0.2",
        "phases": {
            "build": {
                "commands": ["echo \"Hello, CodeBuild!\""
                ]
            }
        }
    })
)

task = tasks.CodeBuildStartBuild(self, "Task",
    project=codebuild_project,
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    environment_variables_override={
        "ZONE": codebuild.BuildEnvironmentVariable(
            type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            value=sfn.JsonPath.string_at("$.envVariables.zone")
        )
    }
)
```

## DynamoDB

You can call DynamoDB APIs from a `Task` state.
Read more about calling DynamoDB APIs [here](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ddb.html)

### GetItem

The [GetItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html) operation returns a set of attributes for the item with the given primary key.

```python
# my_table: dynamodb.Table

tasks.DynamoGetItem(self, "Get Item",
    key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
    table=my_table
)
```

### PutItem

The [PutItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html) operation creates a new item, or replaces an old item with a new item.

```python
# my_table: dynamodb.Table

tasks.DynamoPutItem(self, "PutItem",
    item={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-007"),
        "Text": tasks.DynamoAttributeValue.from_string(sfn.JsonPath.string_at("$.bar")),
        "TotalCount": tasks.DynamoAttributeValue.from_number(10)
    },
    table=my_table
)
```

### DeleteItem

The [DeleteItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html) operation deletes a single item in a table by primary key.

```python
# my_table: dynamodb.Table

tasks.DynamoDeleteItem(self, "DeleteItem",
    key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
    table=my_table,
    result_path=sfn.JsonPath.DISCARD
)
```

### UpdateItem

The [UpdateItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html) operation edits an existing item's attributes, or adds a new item
to the table if it does not already exist.

```python
# my_table: dynamodb.Table

tasks.DynamoUpdateItem(self, "UpdateItem",
    key={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
    },
    table=my_table,
    expression_attribute_values={
        ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
        ":rand": tasks.DynamoAttributeValue.from_number(20)
    },
    update_expression="SET TotalCount = :val + :rand"
)
```

## ECS

Step Functions supports [ECS/Fargate](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html) through the service integration pattern.

### RunTask

[RunTask](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html) starts a new task using the specified task definition.

#### EC2

The EC2 launch type allows you to run your containerized applications on a cluster
of Amazon EC2 instances that you manage.

When a task that uses the EC2 launch type is launched, Amazon ECS must determine where
to place the task based on the requirements specified in the task definition, such as
CPU and memory. Similarly, when you scale down the task count, Amazon ECS must determine
which tasks to terminate. You can apply task placement strategies and constraints to
customize how Amazon ECS places and terminates tasks. Learn more about [task placement](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement.html)

The latest ACTIVE revision of the passed task definition is used for running the task.

The following example runs a job from a task definition on EC2

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)

cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
cluster.add_capacity("DefaultAutoScalingGroup",
    instance_type=ec2.InstanceType("t2.micro"),
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)

task_definition = ecs.TaskDefinition(self, "TD",
    compatibility=ecs.Compatibility.EC2
)

task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "Run",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    launch_target=tasks.EcsEc2LaunchTarget(
        placement_strategies=[
            ecs.PlacementStrategy.spread_across_instances(),
            ecs.PlacementStrategy.packed_by_cpu(),
            ecs.PlacementStrategy.randomly()
        ],
        placement_constraints=[
            ecs.PlacementConstraint.member_of("blieptuut")
        ]
    ),
    propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
)
```

#### Fargate

AWS Fargate is a serverless compute engine for containers that works with Amazon
Elastic Container Service (ECS). Fargate makes it easy for you to focus on building
your applications. Fargate removes the need to provision and manage servers, lets you
specify and pay for resources per application, and improves security through application
isolation by design. Learn more about [Fargate](https://aws.amazon.com/fargate/)

The Fargate launch type allows you to run your containerized applications without the need
to provision and manage the backend infrastructure. Just register your task definition and
Fargate launches the container for you. The latest ACTIVE revision of the passed
task definition is used for running the task. Learn more about
[Fargate Versioning](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html)

The following example runs a job from a task definition on Fargate

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)

cluster = ecs.Cluster(self, "FargateCluster", vpc=vpc)

task_definition = ecs.TaskDefinition(self, "TD",
    memory_mi_b="512",
    cpu="256",
    compatibility=ecs.Compatibility.FARGATE
)

container_definition = task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "RunFargate",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    assign_public_ip=True,
    container_overrides=[tasks.ContainerOverride(
        container_definition=container_definition,
        environment=[tasks.TaskEnvironmentVariable(name="SOME_KEY", value=sfn.JsonPath.string_at("$.SomeKey"))]
    )],
    launch_target=tasks.EcsFargateLaunchTarget(),
    propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
)
```

## EMR

Step Functions supports Amazon EMR through the service integration pattern.
The service integration APIs correspond to Amazon EMR APIs but differ in the
parameters that are used.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html) about the differences when using these service integrations.

### Create Cluster

Creates and starts running a cluster (job flow).
Corresponds to the [`runJobFlow`](https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html) API in EMR.

```python
cluster_role = iam.Role(self, "ClusterRole",
    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
)

service_role = iam.Role(self, "ServiceRole",
    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
)

auto_scaling_role = iam.Role(self, "AutoScalingRole",
    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
)

auto_scaling_role.assume_role_policy.add_statements(
    iam.PolicyStatement(
        effect=iam.Effect.ALLOW,
        principals=[
            iam.ServicePrincipal("application-autoscaling.amazonaws.com")
        ],
        actions=["sts:AssumeRole"
        ]
    ))

tasks.EmrCreateCluster(self, "Create Cluster",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
    cluster_role=cluster_role,
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    service_role=service_role,
    auto_scaling_role=auto_scaling_role
)
```

If you want to run multiple steps in [parallel](https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-concurrent-steps.html),
you can specify the `stepConcurrencyLevel` property. The concurrency range is between 1
and 256 inclusive, where the default concurrency of 1 means no step concurrency is allowed.
`stepConcurrencyLevel` requires the EMR release label to be 5.28.0 or above.

```python
tasks.EmrCreateCluster(self, "Create Cluster",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    step_concurrency_level=10
)
```

### Termination Protection

Locks a cluster (job flow) so the EC2 instances in the cluster cannot be
terminated by user intervention, an API call, or a job-flow error.

Corresponds to the [`setTerminationProtection`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html) API in EMR.

```python
tasks.EmrSetClusterTerminationProtection(self, "Task",
    cluster_id="ClusterId",
    termination_protected=False
)
```

### Terminate Cluster

Shuts down a cluster (job flow).
Corresponds to the [`terminateJobFlows`](https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html) API in EMR.

```python
tasks.EmrTerminateCluster(self, "Task",
    cluster_id="ClusterId"
)
```

### Add Step

Adds a new step to a running cluster.
Corresponds to the [`addJobFlowSteps`](https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html) API in EMR.

```python
tasks.EmrAddStep(self, "Task",
    cluster_id="ClusterId",
    name="StepName",
    jar="Jar",
    action_on_failure=tasks.ActionOnFailure.CONTINUE
)
```

### Cancel Step

Cancels a pending step in a running cluster.
Corresponds to the [`cancelSteps`](https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html) API in EMR.

```python
tasks.EmrCancelStep(self, "Task",
    cluster_id="ClusterId",
    step_id="StepId"
)
```

### Modify Instance Fleet

Modifies the target On-Demand and target Spot capacities for the instance
fleet with the specified InstanceFleetName.

Corresponds to the [`modifyInstanceFleet`](https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html) API in EMR.

```python
tasks.EmrModifyInstanceFleetByName(self, "Task",
    cluster_id="ClusterId",
    instance_fleet_name="InstanceFleetName",
    target_on_demand_capacity=2,
    target_spot_capacity=0
)
```

### Modify Instance Group

Modifies the number of nodes and configuration settings of an instance group.

Corresponds to the [`modifyInstanceGroups`](https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html) API in EMR.

```python
tasks.EmrModifyInstanceGroupByName(self, "Task",
    cluster_id="ClusterId",
    instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
    instance_group=tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
        instance_count=1
    )
)
```

## EMR on EKS

Step Functions supports Amazon EMR on EKS through the service integration pattern.
The service integration APIs correspond to Amazon EMR on EKS APIs, but differ in the parameters that are used.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html) about the differences when using these service integrations.

[Setting up](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html) the EKS cluster is required.

### Create Virtual Cluster

The [CreateVirtualCluster](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_CreateVirtualCluster.html) API creates a single virtual cluster that's mapped to a single Kubernetes namespace.

The EKS cluster containing the Kubernetes namespace where the virtual cluster will be mapped can be passed in from the task input.

```python
tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId"))
)
```

The EKS cluster can also be passed in directly.

```python
import aws_cdk.aws_eks as eks

# eks_cluster: eks.Cluster


tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_cluster(eks_cluster)
)
```

By default, the Kubernetes namespace that a virtual cluster maps to is "default", but a specific namespace within an EKS cluster can be selected.

```python
tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
    eks_namespace="specified-namespace"
)
```

### Delete Virtual Cluster

The [DeleteVirtualCluster](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_DeleteVirtualCluster.html) API deletes a virtual cluster.

```python
tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
    virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
)
```

### Start Job Run

The [StartJobRun](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_StartJobRun.html) API starts a job run. A job is a unit of work that you submit to Amazon EMR on EKS for execution. The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query. A job run is an execution of the job on the virtual cluster.

Required setup:

* If not done already, follow the [steps](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html) to setup EMR on EKS and [create an EKS Cluster](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-eks-readme.html#quick-start).
* Enable [Cluster access](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-cluster-access.html)
* Enable [IAM Role access](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-enable-IAM.html)

The following actions must be performed if the virtual cluster ID is supplied from the task input. Otherwise, if it is supplied statically in the state machine definition, these actions will be done automatically.

* Create an [IAM role](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-iam.Role.html)
* Update the [Role Trust Policy](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-trust-policy.html) of the Job Execution Role.

The job can be configured with spark submit parameters:

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    )
)
```

Configuring the job can also be done via application configuration:

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_name="EMR-Containers-Job",
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
        )
    ),
    application_config=[tasks.ApplicationConfiguration(
        classification=tasks.Classification.SPARK_DEFAULTS,
        properties={
            "spark.executor.instances": "1",
            "spark.executor.memory": "512M"
        }
    )]
)
```

Job monitoring can be enabled if `monitoring.logging` is set true. This automatically generates an S3 bucket and CloudWatch logs.

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    ),
    monitoring=tasks.Monitoring(
        logging=True
    )
)
```

Otherwise, providing monitoring for jobs with existing log groups and log buckets is also available.

```python
import aws_cdk.aws_logs as logs


log_group = logs.LogGroup(self, "Log Group")
log_bucket = s3.Bucket(self, "S3 Bucket")

tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    ),
    monitoring=tasks.Monitoring(
        log_group=log_group,
        log_bucket=log_bucket
    )
)
```

Users can provide their own existing Job Execution Role.

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_task_input(sfn.TaskInput.from_json_path_at("$.VirtualClusterId")),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_name="EMR-Containers-Job",
    execution_role=iam.Role.from_role_arn(self, "Job-Execution-Role", "arn:aws:iam::xxxxxxxxxxxx:role/JobExecutionRole"),
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    )
)
```

## EKS

Step Functions supports Amazon EKS through the service integration pattern.
The service integration APIs correspond to Amazon EKS APIs.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html) about the differences when using these service integrations.

### Call

Read and write Kubernetes resource objects via a Kubernetes API endpoint.
Corresponds to the [`call`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html) API in Step Functions Connector.

The following code snippet includes a Task state that uses eks:call to list the pods.

```python
import aws_cdk.aws_eks as eks


my_eks_cluster = eks.Cluster(self, "my sample cluster",
    version=eks.KubernetesVersion.V1_18,
    cluster_name="myEksCluster"
)

tasks.EksCall(self, "Call a EKS Endpoint",
    cluster=my_eks_cluster,
    http_method=tasks.HttpMethods.GET,
    http_path="/api/v1/namespaces/default/pods"
)
```

## EventBridge

Step Functions supports Amazon EventBridge through the service integration pattern.
The service integration APIs correspond to Amazon EventBridge APIs.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html) about the differences when using these service integrations.

### Put Events

Send events to an EventBridge bus.
Corresponds to the [`put-events`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html) API in Step Functions Connector.

The following code snippet includes a Task state that uses events:putevents to send an event to the default bus.

```python
import aws_cdk.aws_events as events


my_event_bus = events.EventBus(self, "EventBus",
    event_bus_name="MyEventBus1"
)

tasks.EventBridgePutEvents(self, "Send an event to EventBridge",
    entries=[tasks.EventBridgePutEventsEntry(
        detail=sfn.TaskInput.from_object({
            "Message": "Hello from Step Functions!"
        }),
        event_bus=my_event_bus,
        detail_type="MessageFromStepFunctions",
        source="step.functions"
    )]
)
```

## Glue

Step Functions supports [AWS Glue](https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html) through the service integration pattern.

You can call the [`StartJobRun`](https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun) API from a `Task` state.

```python
tasks.GlueStartJobRun(self, "Task",
    glue_job_name="my-glue-job",
    arguments=sfn.TaskInput.from_object({
        "key": "value"
    }),
    task_timeout=sfn.Timeout.duration(Duration.minutes(30)),
    notify_delay_after=Duration.minutes(5)
)
```

## Glue DataBrew

Step Functions supports [AWS Glue DataBrew](https://docs.aws.amazon.com/step-functions/latest/dg/connect-databrew.html) through the service integration pattern.

You can call the [`StartJobRun`](https://docs.aws.amazon.com/databrew/latest/dg/API_StartJobRun.html) API from a `Task` state.

```python
tasks.GlueDataBrewStartJobRun(self, "Task",
    name="databrew-job"
)
```

## Lambda

[Invoke](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html) a Lambda function.

You can specify the input to your Lambda function through the `payload` attribute.
By default, Step Functions invokes Lambda function with the state input (JSON path '$')
as the input.

The following snippet invokes a Lambda Function with the state input as the payload
by referencing the `$` path.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with state input",
    lambda_function=fn
)
```

When a function is invoked, the Lambda service sends  [these response
elements](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseElements)
back.

⚠️ The response from the Lambda function is in an attribute called `Payload`

The following snippet invokes a Lambda Function by referencing the `$.Payload` path
to reference the output of a Lambda executed before it.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with empty object as payload",
    lambda_function=fn,
    payload=sfn.TaskInput.from_object({})
)

# use the output of fn as input
tasks.LambdaInvoke(self, "Invoke with payload field in the state input",
    lambda_function=fn,
    payload=sfn.TaskInput.from_json_path_at("$.Payload")
)
```

The following snippet invokes a Lambda and sets the task output to only include
the Lambda function response.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke and set function response as task output",
    lambda_function=fn,
    output_path="$.Payload"
)
```

If you want to combine the input and the Lambda function response you can use
the `payloadResponseOnly` property and specify the `resultPath`. This will put the
Lambda function ARN directly in the "Resource" string, but it conflicts with the
integrationPattern, invocationType, clientContext, and qualifier properties.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke and combine function response with task input",
    lambda_function=fn,
    payload_response_only=True,
    result_path="$.fn"
)
```

You can have Step Functions pause a task, and wait for an external process to
return a task token. Read more about the [callback pattern](https://docs.aws.amazon.com/step-functions/latest/dg/callback-task-sample-sqs.html#call-back-lambda-example)

To use the callback pattern, set the `token` property on the task. Call the Step
Functions `SendTaskSuccess` or `SendTaskFailure` APIs with the token to
indicate that the task has completed and the state machine should resume execution.

The following snippet invokes a Lambda with the task token as part of the input
to the Lambda.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with callback",
    lambda_function=fn,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    payload=sfn.TaskInput.from_object({
        "token": sfn.JsonPath.task_token,
        "input": sfn.JsonPath.string_at("$.someField")
    })
)
```

⚠️ The task will pause until it receives that task token back with a `SendTaskSuccess` or `SendTaskFailure`
call. Learn more about [Callback with the Task
Token](https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token).

AWS Lambda can occasionally experience transient service errors. In this case, invoking Lambda
results in a 500 error, such as `ServiceException`, `AWSLambdaException`, or `SdkClientException`.
As a best practice, the `LambdaInvoke` task will retry on those errors with an interval of 2 seconds,
a back-off rate of 2 and 6 maximum attempts. Set the `retryOnServiceExceptions` prop to `false` to
disable this behavior.

## SageMaker

Step Functions supports [AWS SageMaker](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html) through the service integration pattern.

If your training job or model uses resources from AWS Marketplace,
[network isolation is required](https://docs.aws.amazon.com/sagemaker/latest/dg/mkt-algo-model-internet-free.html).
To do so, set the `enableNetworkIsolation` property to `true` for `SageMakerCreateModel` or `SageMakerCreateTrainingJob`.

To set environment variables for the Docker container use the `environment` property.

### Create Training Job

You can call the [`CreateTrainingJob`](https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html) API from a `Task` state.

```python
tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
    training_job_name=sfn.JsonPath.string_at("$.JobName"),
    algorithm_specification=tasks.AlgorithmSpecification(
        algorithm_name="BlazingText",
        training_input_mode=tasks.InputMode.FILE
    ),
    input_data_config=[tasks.Channel(
        channel_name="train",
        data_source=tasks.DataSource(
            s3_data_source=tasks.S3DataSource(
                s3_data_type=tasks.S3DataType.S3_PREFIX,
                s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
            )
        )
    )],
    output_data_config=tasks.OutputDataConfig(
        s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
    ),
    resource_config=tasks.ResourceConfig(
        instance_count=1,
        instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
        volume_size=Size.gibibytes(50)
    ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
    stopping_condition=tasks.StoppingCondition(
        max_runtime=Duration.hours(2)
    )
)
```

### Create Transform Job

You can call the [`CreateTransformJob`](https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html) API from a `Task` state.

```python
tasks.SageMakerCreateTransformJob(self, "Batch Inference",
    transform_job_name="MyTransformJob",
    model_name="MyModelName",
    model_client_options=tasks.ModelClientOptions(
        invocations_max_retries=3,  # default is 0
        invocations_timeout=Duration.minutes(5)
    ),
    transform_input=tasks.TransformInput(
        transform_data_source=tasks.TransformDataSource(
            s3_data_source=tasks.TransformS3DataSource(
                s3_uri="s3://inputbucket/train",
                s3_data_type=tasks.S3DataType.S3_PREFIX
            )
        )
    ),
    transform_output=tasks.TransformOutput(
        s3_output_path="s3://outputbucket/TransformJobOutputPath"
    ),
    transform_resources=tasks.TransformResources(
        instance_count=1,
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE)
    )
)
```

### Create Endpoint

You can call the [`CreateEndpoint`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html) API from a `Task` state.

```python
tasks.SageMakerCreateEndpoint(self, "SagemakerEndpoint",
    endpoint_name=sfn.JsonPath.string_at("$.EndpointName"),
    endpoint_config_name=sfn.JsonPath.string_at("$.EndpointConfigName")
)
```

### Create Endpoint Config

You can call the [`CreateEndpointConfig`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpointConfig.html) API from a `Task` state.

```python
tasks.SageMakerCreateEndpointConfig(self, "SagemakerEndpointConfig",
    endpoint_config_name="MyEndpointConfig",
    production_variants=[tasks.ProductionVariant(
        initial_instance_count=2,
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
        model_name="MyModel",
        variant_name="awesome-variant"
    )]
)
```

### Create Model

You can call the [`CreateModel`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModel.html) API from a `Task` state.

```python
tasks.SageMakerCreateModel(self, "Sagemaker",
    model_name="MyModel",
    primary_container=tasks.ContainerDefinition(
        image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
        mode=tasks.Mode.SINGLE_MODEL,
        model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
    )
)
```

### Update Endpoint

You can call the [`UpdateEndpoint`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html) API from a `Task` state.

```python
tasks.SageMakerUpdateEndpoint(self, "SagemakerEndpoint",
    endpoint_name=sfn.JsonPath.string_at("$.Endpoint.Name"),
    endpoint_config_name=sfn.JsonPath.string_at("$.Endpoint.EndpointConfig")
)
```

## SNS

Step Functions supports [Amazon SNS](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sns.html) through the service integration pattern.

You can call the [`Publish`](https://docs.aws.amazon.com/sns/latest/api/API_Publish.html) API from a `Task` state to publish to an SNS topic.

```python
topic = sns.Topic(self, "Topic")

# Use a field from the execution data as message.
task1 = tasks.SnsPublish(self, "Publish1",
    topic=topic,
    integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
    message=sfn.TaskInput.from_data_at("$.state.message"),
    message_attributes={
        "place": tasks.MessageAttribute(
            value=sfn.JsonPath.string_at("$.place")
        ),
        "pic": tasks.MessageAttribute(
            # BINARY must be explicitly set
            data_type=tasks.MessageAttributeDataType.BINARY,
            value=sfn.JsonPath.string_at("$.pic")
        ),
        "people": tasks.MessageAttribute(
            value=4
        ),
        "handles": tasks.MessageAttribute(
            value=["@kslater", "@jjf", null, "@mfanning"]
        )
    }
)

# Combine a field from the execution data with
# a literal object.
task2 = tasks.SnsPublish(self, "Publish2",
    topic=topic,
    message=sfn.TaskInput.from_object({
        "field1": "somedata",
        "field2": sfn.JsonPath.string_at("$.field2")
    })
)
```

## Step Functions

### Start Execution

You can manage [AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html) executions.

AWS Step Functions supports it's own [`StartExecution`](https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html) API as a service integration.

```python
# Define a state machine with one Pass state
child = sfn.StateMachine(self, "ChildStateMachine",
    definition=sfn.Chain.start(sfn.Pass(self, "PassState"))
)

# Include the state machine in a Task state with callback pattern
task = tasks.StepFunctionsStartExecution(self, "ChildTask",
    state_machine=child,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    input=sfn.TaskInput.from_object({
        "token": sfn.JsonPath.task_token,
        "foo": "bar"
    }),
    name="MyExecutionName"
)

# Define a second state machine with the Task state above
sfn.StateMachine(self, "ParentStateMachine",
    definition=task
)
```

You can utilize [Associate Workflow Executions](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-nested-workflows.html#nested-execution-startid)
via the `associateWithParent` property. This allows the Step Functions UI to link child
executions from parent executions, making it easier to trace execution flow across state machines.

```python
# child: sfn.StateMachine

task = tasks.StepFunctionsStartExecution(self, "ChildTask",
    state_machine=child,
    associate_with_parent=True
)
```

This will add the payload `AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID.$: $$.Execution.Id` to the
`input`property for you, which will pass the execution ID from the context object to the
execution input. It requires `input` to be an object or not be set at all.

### Invoke Activity

You can invoke a [Step Functions Activity](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html) which enables you to have
a task in your state machine where the work is performed by a *worker* that can
be hosted on Amazon EC2, Amazon ECS, AWS Lambda, basically anywhere. Activities
are a way to associate code running somewhere (known as an activity worker) with
a specific task in a state machine.

When Step Functions reaches an activity task state, the workflow waits for an
activity worker to poll for a task. An activity worker polls Step Functions by
using GetActivityTask, and sending the ARN for the related activity.

After the activity worker completes its work, it can provide a report of its
success or failure by using `SendTaskSuccess` or `SendTaskFailure`. These two
calls use the taskToken provided by GetActivityTask to associate the result
with that task.

The following example creates an activity and creates a task that invokes the activity.

```python
submit_job_activity = sfn.Activity(self, "SubmitJob")

tasks.StepFunctionsInvokeActivity(self, "Submit Job",
    activity=submit_job_activity
)
```

Use the [Parameters](https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-parameters) field to create a collection of key-value pairs that are passed as input.
The values of each can either be static values that you include in your state machine definition, or selected from either the input or the context object with a path.

```python
submit_job_activity = sfn.Activity(self, "SubmitJob")

tasks.StepFunctionsInvokeActivity(self, "Submit Job",
    activity=submit_job_activity,
    parameters={
        "comment": "Selecting what I care about.",
        "MyDetails": {
            "size": sfn.JsonPath.string_at("$.product.details.size"),
            "exists": sfn.JsonPath.string_at("$.product.availability"),
            "StaticValue": "foo"
        }
    }
)
```

## SQS

Step Functions supports [Amazon SQS](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sqs.html)

You can call the [`SendMessage`](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html) API from a `Task` state
to send a message to an SQS queue.

```python
queue = sqs.Queue(self, "Queue")

# Use a field from the execution data as message.
task1 = tasks.SqsSendMessage(self, "Send1",
    queue=queue,
    message_body=sfn.TaskInput.from_json_path_at("$.message")
)

# Combine a field from the execution data with
# a literal object.
task2 = tasks.SqsSendMessage(self, "Send2",
    queue=queue,
    message_body=sfn.TaskInput.from_object({
        "field1": "somedata",
        "field2": sfn.JsonPath.string_at("$.field2")
    })
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    Duration as _Duration_4839e8c3,
    IgnoreMode as _IgnoreMode_655a98e8,
    Size as _Size_7b441c34,
    Stack as _Stack_2866e57f,
    SymlinkFollowMode as _SymlinkFollowMode_047ec1f6,
)
from ..aws_apigateway import IRestApi as _IRestApi_1f02523d
from ..aws_codebuild import (
    BuildEnvironmentVariable as _BuildEnvironmentVariable_7df4fa0c,
    IProject as _IProject_aafae30a,
)
from ..aws_dynamodb import ITable as _ITable_504fd401
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IConnectable as _IConnectable_10015a05,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    InstanceSize as _InstanceSize_233b2620,
    InstanceType as _InstanceType_f64915b9,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_ecr import IRepository as _IRepository_e6004aa6
from ..aws_ecr_assets import (
    DockerCacheOption as _DockerCacheOption_58ef18ca,
    DockerImageAssetInvalidationOptions as _DockerImageAssetInvalidationOptions_4deb8d45,
    DockerImageAssetProps as _DockerImageAssetProps_6897287d,
    NetworkMode as _NetworkMode_897e5081,
    Platform as _Platform_d16f3cf1,
)
from ..aws_ecs import (
    ContainerDefinition as _ContainerDefinition_8f3b54dc,
    FargatePlatformVersion as _FargatePlatformVersion_55d8be5c,
    ICluster as _ICluster_16cddd09,
    ITaskDefinition as _ITaskDefinition_889ba4d8,
    PlacementConstraint as _PlacementConstraint_11d82a52,
    PlacementStrategy as _PlacementStrategy_2bb6c232,
    PropagatedTagSource as _PropagatedTagSource_ad4e874a,
    TaskDefinition as _TaskDefinition_a541a103,
)
from ..aws_eks import ICluster as _ICluster_6b2b80df
from ..aws_events import IEventBus as _IEventBus_88d13111
from ..aws_iam import (
    IGrantable as _IGrantable_71c4f5de,
    IPrincipal as _IPrincipal_539bb2fd,
    IRole as _IRole_235f5d8e,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_lambda import IFunction as _IFunction_6adb0ab8, Runtime as _Runtime_b4eaa844
from ..aws_logs import ILogGroup as _ILogGroup_3c4fa718
from ..aws_s3 import IBucket as _IBucket_42e086fd, Location as _Location_0948fa7f
from ..aws_sns import ITopic as _ITopic_9eca4852
from ..aws_sqs import IQueue as _IQueue_7ed6f679
from ..aws_stepfunctions import (
    Credentials as _Credentials_2cd64c6b,
    IActivity as _IActivity_3cf6efa2,
    IStateMachine as _IStateMachine_73e8d2b0,
    IntegrationPattern as _IntegrationPattern_949291bc,
    ServiceIntegrationPattern as _ServiceIntegrationPattern_39230c49,
    TaskInput as _TaskInput_91b91b91,
    TaskMetricsConfig as _TaskMetricsConfig_32ea9403,
    TaskStateBase as _TaskStateBase_b5c0a816,
    TaskStateBaseProps as _TaskStateBaseProps_3a62b6d0,
    Timeout as _Timeout_d7c10551,
)


class AcceleratorClass(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AcceleratorClass",
):
    '''The generation of Elastic Inference (EI) instance.

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        accelerator_class = stepfunctions_tasks.AcceleratorClass.of("version")
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, version: builtins.str) -> "AcceleratorClass":
        '''Custom AcceleratorType.

        :param version: - Elastic Inference accelerator generation.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__819c31c0c99426607244dc0c57ea1d116326b5a750cff637f19b64f1a598385c)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AcceleratorClass", jsii.sinvoke(cls, "of", [version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EIA1")
    def EIA1(cls) -> "AcceleratorClass":
        '''Elastic Inference accelerator 1st generation.'''
        return typing.cast("AcceleratorClass", jsii.sget(cls, "EIA1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EIA2")
    def EIA2(cls) -> "AcceleratorClass":
        '''Elastic Inference accelerator 2nd generation.'''
        return typing.cast("AcceleratorClass", jsii.sget(cls, "EIA2"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''- Elastic Inference accelerator generation.'''
        return typing.cast(builtins.str, jsii.get(self, "version"))


class AcceleratorType(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AcceleratorType",
):
    '''The size of the Elastic Inference (EI) instance to use for the production variant.

    EI instances provide on-demand GPU computing for inference

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        accelerator_type = stepfunctions_tasks.AcceleratorType("instanceTypeIdentifier")
    '''

    def __init__(self, instance_type_identifier: builtins.str) -> None:
        '''
        :param instance_type_identifier: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35c8fa35baadd94a7e0608eafaea8bc1776ff404188c0e8e931fe7a326a7bb88)
            check_type(argname="argument instance_type_identifier", value=instance_type_identifier, expected_type=type_hints["instance_type_identifier"])
        jsii.create(self.__class__, self, [instance_type_identifier])

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        accelerator_class: AcceleratorClass,
        instance_size: _InstanceSize_233b2620,
    ) -> "AcceleratorType":
        '''AcceleratorType.

        This class takes a combination of a class and size.

        :param accelerator_class: -
        :param instance_size: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e8b0a3bd440b1d4ddc7b3420d17fb7fbb5662db801beeda1ce3cd59388af374)
            check_type(argname="argument accelerator_class", value=accelerator_class, expected_type=type_hints["accelerator_class"])
            check_type(argname="argument instance_size", value=instance_size, expected_type=type_hints["instance_size"])
        return typing.cast("AcceleratorType", jsii.sinvoke(cls, "of", [accelerator_class, instance_size]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Return the accelerator type as a dotted string.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ActionOnFailure")
class ActionOnFailure(enum.Enum):
    '''The action to take when the cluster step fails.

    :default: CONTINUE

    :see:

    https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html

    Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
    :exampleMetadata: infused

    Example::

        tasks.EmrAddStep(self, "Task",
            cluster_id="ClusterId",
            name="StepName",
            jar="Jar",
            action_on_failure=tasks.ActionOnFailure.CONTINUE
        )
    '''

    TERMINATE_CLUSTER = "TERMINATE_CLUSTER"
    '''Terminate the Cluster on Step Failure.'''
    CANCEL_AND_WAIT = "CANCEL_AND_WAIT"
    '''Cancel Step execution and enter WAITING state.'''
    CONTINUE = "CONTINUE"
    '''Continue to the next Step.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AlgorithmSpecification",
    jsii_struct_bases=[],
    name_mapping={
        "algorithm_name": "algorithmName",
        "metric_definitions": "metricDefinitions",
        "training_image": "trainingImage",
        "training_input_mode": "trainingInputMode",
    },
)
class AlgorithmSpecification:
    def __init__(
        self,
        *,
        algorithm_name: typing.Optional[builtins.str] = None,
        metric_definitions: typing.Optional[typing.Sequence[typing.Union["MetricDefinition", typing.Dict[builtins.str, typing.Any]]]] = None,
        training_image: typing.Optional["DockerImage"] = None,
        training_input_mode: typing.Optional["InputMode"] = None,
    ) -> None:
        '''Specify the training algorithm and algorithm-specific metadata.

        :param algorithm_name: Name of the algorithm resource to use for the training job. This must be an algorithm resource that you created or subscribe to on AWS Marketplace. If you specify a value for this parameter, you can't specify a value for TrainingImage. Default: - No algorithm is specified
        :param metric_definitions: List of metric definition objects. Each object specifies the metric name and regular expressions used to parse algorithm logs. Default: - No metrics
        :param training_image: Registry path of the Docker image that contains the training algorithm. Default: - No Docker image is specified
        :param training_input_mode: Input mode that the algorithm supports. Default: 'File' mode

        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
                training_job_name=sfn.JsonPath.string_at("$.JobName"),
                algorithm_specification=tasks.AlgorithmSpecification(
                    algorithm_name="BlazingText",
                    training_input_mode=tasks.InputMode.FILE
                ),
                input_data_config=[tasks.Channel(
                    channel_name="train",
                    data_source=tasks.DataSource(
                        s3_data_source=tasks.S3DataSource(
                            s3_data_type=tasks.S3DataType.S3_PREFIX,
                            s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                        )
                    )
                )],
                output_data_config=tasks.OutputDataConfig(
                    s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
                ),
                resource_config=tasks.ResourceConfig(
                    instance_count=1,
                    instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                    volume_size=Size.gibibytes(50)
                ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
                stopping_condition=tasks.StoppingCondition(
                    max_runtime=Duration.hours(2)
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e11c280b4cbc931a6ec0d0b6e960ccb7f67456a5fcb77cb1fe1a027cb3c9be4d)
            check_type(argname="argument algorithm_name", value=algorithm_name, expected_type=type_hints["algorithm_name"])
            check_type(argname="argument metric_definitions", value=metric_definitions, expected_type=type_hints["metric_definitions"])
            check_type(argname="argument training_image", value=training_image, expected_type=type_hints["training_image"])
            check_type(argname="argument training_input_mode", value=training_input_mode, expected_type=type_hints["training_input_mode"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if algorithm_name is not None:
            self._values["algorithm_name"] = algorithm_name
        if metric_definitions is not None:
            self._values["metric_definitions"] = metric_definitions
        if training_image is not None:
            self._values["training_image"] = training_image
        if training_input_mode is not None:
            self._values["training_input_mode"] = training_input_mode

    @builtins.property
    def algorithm_name(self) -> typing.Optional[builtins.str]:
        '''Name of the algorithm resource to use for the training job.

        This must be an algorithm resource that you created or subscribe to on AWS Marketplace.
        If you specify a value for this parameter, you can't specify a value for TrainingImage.

        :default: - No algorithm is specified
        '''
        result = self._values.get("algorithm_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def metric_definitions(self) -> typing.Optional[typing.List["MetricDefinition"]]:
        '''List of metric definition objects.

        Each object specifies the metric name and regular expressions used to parse algorithm logs.

        :default: - No metrics
        '''
        result = self._values.get("metric_definitions")
        return typing.cast(typing.Optional[typing.List["MetricDefinition"]], result)

    @builtins.property
    def training_image(self) -> typing.Optional["DockerImage"]:
        '''Registry path of the Docker image that contains the training algorithm.

        :default: - No Docker image is specified
        '''
        result = self._values.get("training_image")
        return typing.cast(typing.Optional["DockerImage"], result)

    @builtins.property
    def training_input_mode(self) -> typing.Optional["InputMode"]:
        '''Input mode that the algorithm supports.

        :default: 'File' mode
        '''
        result = self._values.get("training_input_mode")
        return typing.cast(typing.Optional["InputMode"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AlgorithmSpecification(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ApplicationConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "classification": "classification",
        "nested_config": "nestedConfig",
        "properties": "properties",
    },
)
class ApplicationConfiguration:
    def __init__(
        self,
        *,
        classification: "Classification",
        nested_config: typing.Optional[typing.Sequence[typing.Union["ApplicationConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''A configuration specification to be used when provisioning virtual clusters, which can include configurations for applications and software bundled with Amazon EMR on EKS.

        A configuration consists of a classification, properties, and optional nested configurations.
        A classification refers to an application-specific configuration file.
        Properties are the settings you want to change in that file.

        :param classification: The classification within a configuration. Length Constraints: Minimum length of 1. Maximum length of 1024.
        :param nested_config: A list of additional configurations to apply within a configuration object. Array Members: Maximum number of 100 items. Default: - No other configurations
        :param properties: A set of properties specified within a configuration classification. Map Entries: Maximum number of 100 items. Default: - No properties

        :see: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-configure-apps.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # application_configuration_: stepfunctions_tasks.ApplicationConfiguration
            # classification: stepfunctions_tasks.Classification
            
            application_configuration = stepfunctions_tasks.ApplicationConfiguration(
                classification=classification,
            
                # the properties below are optional
                nested_config=[stepfunctions_tasks.ApplicationConfiguration(
                    classification=classification,
            
                    # the properties below are optional
                    nested_config=[application_configuration_],
                    properties={
                        "properties_key": "properties"
                    }
                )],
                properties={
                    "properties_key": "properties"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2838fe25c75eb2d6ec4a40b92699edb9f4ed7d7743cd75d5155954a10402d192)
            check_type(argname="argument classification", value=classification, expected_type=type_hints["classification"])
            check_type(argname="argument nested_config", value=nested_config, expected_type=type_hints["nested_config"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "classification": classification,
        }
        if nested_config is not None:
            self._values["nested_config"] = nested_config
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def classification(self) -> "Classification":
        '''The classification within a configuration.

        Length Constraints: Minimum length of 1. Maximum length of 1024.
        '''
        result = self._values.get("classification")
        assert result is not None, "Required property 'classification' is missing"
        return typing.cast("Classification", result)

    @builtins.property
    def nested_config(self) -> typing.Optional[typing.List["ApplicationConfiguration"]]:
        '''A list of additional configurations to apply within a configuration object.

        Array Members: Maximum number of 100 items.

        :default: - No other configurations
        '''
        result = self._values.get("nested_config")
        return typing.cast(typing.Optional[typing.List["ApplicationConfiguration"]], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''A set of properties specified within a configuration classification.

        Map Entries: Maximum number of 100 items.

        :default: - No properties
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AssembleWith")
class AssembleWith(enum.Enum):
    '''How to assemble the results of the transform job as a single S3 object.'''

    NONE = "NONE"
    '''Concatenate the results in binary format.'''
    LINE = "LINE"
    '''Add a newline character at the end of every transformed record.'''


class AthenaGetQueryExecution(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryExecution",
):
    '''Get an Athena Query Execution as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f874d4b3d8cca8a1aa5d7809bc14fb2078e4f8cbdcf8a7ade80f98ea33352417)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryExecutionProps(
            query_execution_id=query_execution_id,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaGetQueryExecutionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for getting a Query Execution.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``

        :exampleMetadata: infused

        Example::

            get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__426b683b3fff9459413103d38b8f467a1eef7b493511c26ddef6ccaf20d191b5)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaGetQueryResults(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryResults",
):
    '''Get an Athena Query Results as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fbcf740a1e3e37ea0b4976ce01dbba60ab814c75d0b05f5ea68b86ab3392764)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryResultsProps(
            query_execution_id=query_execution_id,
            max_results=max_results,
            next_token=next_token,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryResultsProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "query_execution_id": "queryExecutionId",
        "max_results": "maxResults",
        "next_token": "nextToken",
    },
)
class AthenaGetQueryResultsProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for getting a Query Results.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token

        :exampleMetadata: infused

        Example::

            get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e62d5ebfcc36e5adfee6cea02c96ccdc0cba73b7dc4169d47fcd62e6b0b398db)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
            check_type(argname="argument max_results", value=max_results, expected_type=type_hints["max_results"])
            check_type(argname="argument next_token", value=next_token, expected_type=type_hints["next_token"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if max_results is not None:
            self._values["max_results"] = max_results
        if next_token is not None:
            self._values["next_token"] = next_token

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def max_results(self) -> typing.Optional[jsii.Number]:
        '''Max number of results.

        :default: 1000
        '''
        result = self._values.get("max_results")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def next_token(self) -> typing.Optional[builtins.str]:
        '''Pagination token.

        :default: - No next token
        '''
        result = self._values.get("next_token")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryResultsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaStartQueryExecution(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecution",
):
    '''Start an Athena Query as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
            query_string=sfn.JsonPath.string_at("$.queryString"),
            query_execution_context=tasks.QueryExecutionContext(
                database_name="mydatabase"
            ),
            result_configuration=tasks.ResultConfiguration(
                encryption_configuration=tasks.EncryptionConfiguration(
                    encryption_option=tasks.EncryptionOption.S3_MANAGED
                ),
                output_location=s3.Location(
                    bucket_name="query-results-bucket",
                    object_key="folder"
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        work_group: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param work_group: Configuration on how and where to save query. Default: - No work group
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74ee21cd0bad6760da4cd18a6c4cd846a24d69d25b7bac8eb004edf6403103e8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStartQueryExecutionProps(
            query_string=query_string,
            client_request_token=client_request_token,
            query_execution_context=query_execution_context,
            result_configuration=result_configuration,
            work_group=work_group,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "query_string": "queryString",
        "client_request_token": "clientRequestToken",
        "query_execution_context": "queryExecutionContext",
        "result_configuration": "resultConfiguration",
        "work_group": "workGroup",
    },
)
class AthenaStartQueryExecutionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        work_group: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for starting a Query Execution.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param work_group: Configuration on how and where to save query. Default: - No work group

        :exampleMetadata: infused

        Example::

            start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
                query_string=sfn.JsonPath.string_at("$.queryString"),
                query_execution_context=tasks.QueryExecutionContext(
                    database_name="mydatabase"
                ),
                result_configuration=tasks.ResultConfiguration(
                    encryption_configuration=tasks.EncryptionConfiguration(
                        encryption_option=tasks.EncryptionOption.S3_MANAGED
                    ),
                    output_location=s3.Location(
                        bucket_name="query-results-bucket",
                        object_key="folder"
                    )
                )
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(query_execution_context, dict):
            query_execution_context = QueryExecutionContext(**query_execution_context)
        if isinstance(result_configuration, dict):
            result_configuration = ResultConfiguration(**result_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2b6c9868cc3485b4ed4a4ef1f89308086ff873bc4d86413aa4b420c9cf24e55)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument query_string", value=query_string, expected_type=type_hints["query_string"])
            check_type(argname="argument client_request_token", value=client_request_token, expected_type=type_hints["client_request_token"])
            check_type(argname="argument query_execution_context", value=query_execution_context, expected_type=type_hints["query_execution_context"])
            check_type(argname="argument result_configuration", value=result_configuration, expected_type=type_hints["result_configuration"])
            check_type(argname="argument work_group", value=work_group, expected_type=type_hints["work_group"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_string": query_string,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if query_execution_context is not None:
            self._values["query_execution_context"] = query_execution_context
        if result_configuration is not None:
            self._values["result_configuration"] = result_configuration
        if work_group is not None:
            self._values["work_group"] = work_group

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def query_string(self) -> builtins.str:
        '''Query that will be started.'''
        result = self._values.get("query_string")
        assert result is not None, "Required property 'query_string' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[builtins.str]:
        '''Unique string string to ensure idempotence.

        :default: - No client request token
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_execution_context(self) -> typing.Optional["QueryExecutionContext"]:
        '''Database within which query executes.

        :default: - No query execution context
        '''
        result = self._values.get("query_execution_context")
        return typing.cast(typing.Optional["QueryExecutionContext"], result)

    @builtins.property
    def result_configuration(self) -> typing.Optional["ResultConfiguration"]:
        '''Configuration on how and where to save query.

        :default: - No result configuration
        '''
        result = self._values.get("result_configuration")
        return typing.cast(typing.Optional["ResultConfiguration"], result)

    @builtins.property
    def work_group(self) -> typing.Optional[builtins.str]:
        '''Configuration on how and where to save query.

        :default: - No work group
        '''
        result = self._values.get("work_group")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStartQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaStopQueryExecution(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStopQueryExecution",
):
    '''Stop an Athena Query Execution as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_execution_id: Query that will be stopped.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d67b0a74228051a1266e72262ec4fc6e739a35f07a82a3e5faf57fbe904ab7b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStopQueryExecutionProps(
            query_execution_id=query_execution_id,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStopQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaStopQueryExecutionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for stoping a Query Execution.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param query_execution_id: Query that will be stopped.

        :exampleMetadata: infused

        Example::

            stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95e47c48916d5954411b46d2ffee40460efc3e552562676549f5de7fe2db36ef)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be stopped.'''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStopQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AuthType")
class AuthType(enum.Enum):
    '''The authentication method used to call the endpoint.'''

    NO_AUTH = "NO_AUTH"
    '''Call the API direclty with no authorization method.'''
    IAM_ROLE = "IAM_ROLE"
    '''Use the IAM role associated with the current state machine for authorization.'''
    RESOURCE_POLICY = "RESOURCE_POLICY"
    '''Use the resource policy of the API for authorization.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchContainerOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "environment": "environment",
        "gpu_count": "gpuCount",
        "instance_type": "instanceType",
        "memory": "memory",
        "vcpus": "vcpus",
    },
)
class BatchContainerOverrides:
    def __init__(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        instance_type: typing.Optional[_InstanceType_f64915b9] = None,
        memory: typing.Optional[_Size_7b441c34] = None,
        vcpus: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The overrides that should be sent to a container.

        :param command: The command to send to the container that overrides the default command from the Docker image or the job definition. Default: - No command overrides
        :param environment: The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the job definition. Default: - No environment overrides
        :param gpu_count: The number of physical GPUs to reserve for the container. The number of GPUs reserved for all containers in a job should not exceed the number of available GPUs on the compute resource that the job is launched on. Default: - No GPU reservation
        :param instance_type: The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container jobs. Default: - No instance type overrides
        :param memory: Memory reserved for the job. Default: - No memory overrides. The memory supplied in the job definition will be used.
        :param vcpus: The number of vCPUs to reserve for the container. This value overrides the value set in the job definition. Default: - No vCPUs overrides

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # instance_type: ec2.InstanceType
            # size: cdk.Size
            
            batch_container_overrides = stepfunctions_tasks.BatchContainerOverrides(
                command=["command"],
                environment={
                    "environment_key": "environment"
                },
                gpu_count=123,
                instance_type=instance_type,
                memory=size,
                vcpus=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d006b47111b6db7647cd90008f27c393312ee16152c31394c4e4f623d7fcc2ed)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument gpu_count", value=gpu_count, expected_type=type_hints["gpu_count"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument vcpus", value=vcpus, expected_type=type_hints["vcpus"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if memory is not None:
            self._values["memory"] = memory
        if vcpus is not None:
            self._values["vcpus"] = vcpus

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command to send to the container that overrides the default command from the Docker image or the job definition.

        :default: - No command overrides
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to send to the container.

        You can add new environment variables, which are added to the container
        at launch, or you can override the existing environment variables from
        the Docker image or the job definition.

        :default: - No environment overrides
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''The number of physical GPUs to reserve for the container.

        The number of GPUs reserved for all containers in a job
        should not exceed the number of available GPUs on the compute
        resource that the job is launched on.

        :default: - No GPU reservation
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[_InstanceType_f64915b9]:
        '''The instance type to use for a multi-node parallel job.

        This parameter is not valid for single-node container jobs.

        :default: - No instance type overrides
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[_InstanceType_f64915b9], result)

    @builtins.property
    def memory(self) -> typing.Optional[_Size_7b441c34]:
        '''Memory reserved for the job.

        :default: - No memory overrides. The memory supplied in the job definition will be used.
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def vcpus(self) -> typing.Optional[jsii.Number]:
        '''The number of vCPUs to reserve for the container.

        This value overrides the value set in the job definition.

        :default: - No vCPUs overrides
        '''
        result = self._values.get("vcpus")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchContainerOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchJobDependency",
    jsii_struct_bases=[],
    name_mapping={"job_id": "jobId", "type": "type"},
)
class BatchJobDependency:
    def __init__(
        self,
        *,
        job_id: typing.Optional[builtins.str] = None,
        type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''An object representing an AWS Batch job dependency.

        :param job_id: The job ID of the AWS Batch job associated with this dependency. Default: - No jobId
        :param type: The type of the job dependency. Default: - No type

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            batch_job_dependency = stepfunctions_tasks.BatchJobDependency(
                job_id="jobId",
                type="type"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f18f114b4b57e22d6d5ccd29edff4424be5b1132c504c1b3ac2ecad9d9cab599)
            check_type(argname="argument job_id", value=job_id, expected_type=type_hints["job_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if job_id is not None:
            self._values["job_id"] = job_id
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def job_id(self) -> typing.Optional[builtins.str]:
        '''The job ID of the AWS Batch job associated with this dependency.

        :default: - No jobId
        '''
        result = self._values.get("job_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The type of the job dependency.

        :default: - No type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchJobDependency(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchStrategy")
class BatchStrategy(enum.Enum):
    '''Specifies the number of records to include in a mini-batch for an HTTP inference request.'''

    MULTI_RECORD = "MULTI_RECORD"
    '''Fits multiple records in a mini-batch.'''
    SINGLE_RECORD = "SINGLE_RECORD"
    '''Use a single record when making an invocation request.'''


class BatchSubmitJob(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchSubmitJob",
):
    '''Task to submits an AWS Batch job from a job definition.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
    :exampleMetadata: infused

    Example::

        # Example automatically generated from non-compiling source. May contain errors.
        import aws_cdk.aws_batch as batch
        # batch_job_definition: batch.JobDefinition
        # batch_queue: batch.JobQueue
        
        
        task = tasks.BatchSubmitJob(self, "Submit Job",
            job_definition_arn=batch_job_definition.job_definition_arn,
            job_name="MyJob",
            job_queue_arn=batch_queue.job_queue_arn
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe5d08d3f91a8bb221c70885cc736bf4f22481309f41eda6cfe2f18298a12423)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BatchSubmitJobProps(
            job_definition_arn=job_definition_arn,
            job_name=job_name,
            job_queue_arn=job_queue_arn,
            array_size=array_size,
            attempts=attempts,
            container_overrides=container_overrides,
            depends_on=depends_on,
            payload=payload,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchSubmitJobProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "job_definition_arn": "jobDefinitionArn",
        "job_name": "jobName",
        "job_queue_arn": "jobQueueArn",
        "array_size": "arraySize",
        "attempts": "attempts",
        "container_overrides": "containerOverrides",
        "depends_on": "dependsOn",
        "payload": "payload",
    },
)
class BatchSubmitJobProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Properties for RunBatchJob.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed

        :exampleMetadata: infused

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            import aws_cdk.aws_batch as batch
            # batch_job_definition: batch.JobDefinition
            # batch_queue: batch.JobQueue
            
            
            task = tasks.BatchSubmitJob(self, "Submit Job",
                job_definition_arn=batch_job_definition.job_definition_arn,
                job_name="MyJob",
                job_queue_arn=batch_queue.job_queue_arn
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(container_overrides, dict):
            container_overrides = BatchContainerOverrides(**container_overrides)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3197d21dad4ba55e29e09cf770b47b7859167ae6b1ae5ef674cca1bedf86a92)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument job_definition_arn", value=job_definition_arn, expected_type=type_hints["job_definition_arn"])
            check_type(argname="argument job_name", value=job_name, expected_type=type_hints["job_name"])
            check_type(argname="argument job_queue_arn", value=job_queue_arn, expected_type=type_hints["job_queue_arn"])
            check_type(argname="argument array_size", value=array_size, expected_type=type_hints["array_size"])
            check_type(argname="argument attempts", value=attempts, expected_type=type_hints["attempts"])
            check_type(argname="argument container_overrides", value=container_overrides, expected_type=type_hints["container_overrides"])
            check_type(argname="argument depends_on", value=depends_on, expected_type=type_hints["depends_on"])
            check_type(argname="argument payload", value=payload, expected_type=type_hints["payload"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "job_definition_arn": job_definition_arn,
            "job_name": job_name,
            "job_queue_arn": job_queue_arn,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if array_size is not None:
            self._values["array_size"] = array_size
        if attempts is not None:
            self._values["attempts"] = attempts
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if depends_on is not None:
            self._values["depends_on"] = depends_on
        if payload is not None:
            self._values["payload"] = payload

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def job_definition_arn(self) -> builtins.str:
        '''The arn of the job definition used by this job.'''
        result = self._values.get("job_definition_arn")
        assert result is not None, "Required property 'job_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_name(self) -> builtins.str:
        '''The name of the job.

        The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase),
        numbers, hyphens, and underscores are allowed.
        '''
        result = self._values.get("job_name")
        assert result is not None, "Required property 'job_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_queue_arn(self) -> builtins.str:
        '''The arn of the job queue into which the job is submitted.'''
        result = self._values.get("job_queue_arn")
        assert result is not None, "Required property 'job_queue_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def array_size(self) -> typing.Optional[jsii.Number]:
        '''The array size can be between 2 and 10,000.

        If you specify array properties for a job, it becomes an array job.
        For more information, see Array Jobs in the AWS Batch User Guide.

        :default: - No array size
        '''
        result = self._values.get("array_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def attempts(self) -> typing.Optional[jsii.Number]:
        '''The number of times to move a job to the RUNNABLE status.

        You may specify between 1 and 10 attempts.
        If the value of attempts is greater than one,
        the job is retried on failure the same number of attempts as the value.

        :default: 1
        '''
        result = self._values.get("attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[BatchContainerOverrides]:
        '''A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive.

        :default: - No container overrides

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-containerOverrides
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[BatchContainerOverrides], result)

    @builtins.property
    def depends_on(self) -> typing.Optional[typing.List[BatchJobDependency]]:
        '''A list of dependencies for the job.

        A job can depend upon a maximum of 20 jobs.

        :default: - No dependencies

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-dependsOn
        '''
        result = self._values.get("depends_on")
        return typing.cast(typing.Optional[typing.List[BatchJobDependency]], result)

    @builtins.property
    def payload(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The payload to be passed as parameters to the batch job.

        :default: - No parameters are passed
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchSubmitJobProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayEndpointBaseProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class CallApiGatewayEndpointBaseProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Base CallApiGatewayEdnpoint Task Props.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # result_selector: Any
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_api_gateway_endpoint_base_props = stepfunctions_tasks.CallApiGatewayEndpointBaseProps(
                method=stepfunctions_tasks.HttpMethod.GET,
            
                # the properties below are optional
                api_path="apiPath",
                auth_type=stepfunctions_tasks.AuthType.NO_AUTH,
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                headers=task_input,
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                query_parameters=task_input,
                request_body=task_input,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e9dbc58cb4f9a28fbab12032a3c39bc929dcf76e35816fe9ba25aab36499c005)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayEndpointBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallApiGatewayHttpApiEndpoint(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpoint",
):
    '''Call HTTP API endpoint as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
    :exampleMetadata: infused

    Example::

        # Example automatically generated from non-compiling source. May contain errors.
        import aws_cdk.aws_apigatewayv2 as apigatewayv2
        
        http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
        
        invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
            api_id=http_api.api_id,
            api_stack=Stack.of(http_api),
            method=tasks.HttpMethod.GET
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e727eac1466f73756eed51efa40866442f05dae5d955ec7e6f15390221c1c6f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayHttpApiEndpointProps(
            api_id=api_id,
            api_stack=api_stack,
            stage_name=stage_name,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="createPolicyStatements")
    def _create_policy_statements(self) -> typing.List[_PolicyStatement_0fe33853]:
        return typing.cast(typing.List[_PolicyStatement_0fe33853], jsii.invoke(self, "createPolicyStatements", []))

    @builtins.property
    @jsii.member(jsii_name="apiEndpoint")
    def _api_endpoint(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "apiEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="arnForExecuteApi")
    def _arn_for_execute_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "arnForExecuteApi"))

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def _stage_name(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpointProps",
    jsii_struct_bases=[CallApiGatewayEndpointBaseProps],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
        "api_id": "apiId",
        "api_stack": "apiStack",
        "stage_name": "stageName",
    },
)
class CallApiGatewayHttpApiEndpointProps(CallApiGatewayEndpointBaseProps):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for calling an HTTP API Endpoint.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'

        :exampleMetadata: infused

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            import aws_cdk.aws_apigatewayv2 as apigatewayv2
            
            http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
            
            invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
                api_id=http_api.api_id,
                api_stack=Stack.of(http_api),
                method=tasks.HttpMethod.GET
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a41fc59378bfacae33639438a7a9b775728e418d0a325893bbe9defe089edfd)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument api_stack", value=api_stack, expected_type=type_hints["api_stack"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
            "api_id": api_id,
            "api_stack": api_stack,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def api_id(self) -> builtins.str:
        '''The Id of the API to call.'''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_stack(self) -> _Stack_2866e57f:
        '''The Stack in which the API is defined.'''
        result = self._values.get("api_stack")
        assert result is not None, "Required property 'api_stack' is missing"
        return typing.cast(_Stack_2866e57f, result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''Name of the stage where the API is deployed to in API Gateway.

        :default: '$default'
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayHttpApiEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallApiGatewayRestApiEndpoint(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpoint",
):
    '''Call REST API endpoint as a Task.

    Be aware that the header values must be arrays. When passing the Task Token
    in the headers field ``WAIT_FOR_TASK_TOKEN`` integration, use
    ``JsonPath.array()`` to wrap the token in an array::

       # Example automatically generated from non-compiling source. May contain errors.
       import aws_cdk.aws_apigateway as apigateway
       # api: apigateway.RestApi


       tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
           api=api,
           stage_name="Stage",
           method=tasks.HttpMethod.PUT,
           integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
           headers=sfn.TaskInput.from_object({
               "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
           })
       )

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_apigateway as apigateway
        # api: apigateway.RestApi
        
        
        tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
            api=api,
            stage_name="Stage",
            method=tasks.HttpMethod.PUT,
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            headers=sfn.TaskInput.from_object({
                "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
            })
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a76886b18651b0cbd9c7ad80e6a462a8540904f212eeb56575f9a2c9ab18e143)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayRestApiEndpointProps(
            api=api,
            stage_name=stage_name,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="createPolicyStatements")
    def _create_policy_statements(self) -> typing.List[_PolicyStatement_0fe33853]:
        return typing.cast(typing.List[_PolicyStatement_0fe33853], jsii.invoke(self, "createPolicyStatements", []))

    @builtins.property
    @jsii.member(jsii_name="apiEndpoint")
    def _api_endpoint(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "apiEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="arnForExecuteApi")
    def _arn_for_execute_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "arnForExecuteApi"))

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def _stage_name(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpointProps",
    jsii_struct_bases=[CallApiGatewayEndpointBaseProps],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
        "api": "api",
        "stage_name": "stageName",
    },
)
class CallApiGatewayRestApiEndpointProps(CallApiGatewayEndpointBaseProps):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
    ) -> None:
        '''Properties for calling an REST API Endpoint.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_apigateway as apigateway
            # api: apigateway.RestApi
            
            
            tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
                api=api,
                stage_name="Stage",
                method=tasks.HttpMethod.PUT,
                integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                headers=sfn.TaskInput.from_object({
                    "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
                })
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5e083701921cac85bdeb9778c84e51785062fe16c2575a0df424f5a5a1dfa22)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
            check_type(argname="argument api", value=api, expected_type=type_hints["api"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
            "api": api,
            "stage_name": stage_name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def api(self) -> _IRestApi_1f02523d:
        '''API to call.'''
        result = self._values.get("api")
        assert result is not None, "Required property 'api' is missing"
        return typing.cast(_IRestApi_1f02523d, result)

    @builtins.property
    def stage_name(self) -> builtins.str:
        '''Name of the stage where the API is deployed to in API Gateway.'''
        result = self._values.get("stage_name")
        assert result is not None, "Required property 'stage_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayRestApiEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallAwsService(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService",
):
    '''A StepFunctions task to call an AWS service API.

    :exampleMetadata: infused

    Example::

        detect_labels = tasks.CallAwsService(self, "DetectLabels",
            service="rekognition",
            action="detectLabels",
            iam_resources=["*"],
            additional_iam_statements=[
                iam.PolicyStatement(
                    actions=["s3:getObject"],
                    resources=["arn:aws:s3:::my-bucket/*"]
                )
            ]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4534f8232f0840d6ef9dc3920130b2d24934f22c4483b9bd14a19e8152f0b6f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceProps(
            action=action,
            iam_resources=iam_resources,
            service=service,
            additional_iam_statements=additional_iam_statements,
            iam_action=iam_action,
            parameters=parameters,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "action": "action",
        "iam_resources": "iamResources",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "iam_action": "iamAction",
        "parameters": "parameters",
    },
)
class CallAwsServiceProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action from your state machine.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        :exampleMetadata: infused

        Example::

            detect_labels = tasks.CallAwsService(self, "DetectLabels",
                service="rekognition",
                action="detectLabels",
                iam_resources=["*"],
                additional_iam_statements=[
                    iam.PolicyStatement(
                        actions=["s3:getObject"],
                        resources=["arn:aws:s3:::my-bucket/*"]
                    )
                ]
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63c577047b5ec17a682dc869c299af67611178c845f52baceb7e79927eee3db3)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.
        '''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call.

        Use PascalCase for the parameter names.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.Channel",
    jsii_struct_bases=[],
    name_mapping={
        "channel_name": "channelName",
        "data_source": "dataSource",
        "compression_type": "compressionType",
        "content_type": "contentType",
        "input_mode": "inputMode",
        "record_wrapper_type": "recordWrapperType",
        "shuffle_config": "shuffleConfig",
    },
)
class Channel:
    def __init__(
        self,
        *,
        channel_name: builtins.str,
        data_source: typing.Union["DataSource", typing.Dict[builtins.str, typing.Any]],
        compression_type: typing.Optional["CompressionType"] = None,
        content_type: typing.Optional[builtins.str] = None,
        input_mode: typing.Optional["InputMode"] = None,
        record_wrapper_type: typing.Optional["RecordWrapperType"] = None,
        shuffle_config: typing.Optional[typing.Union["ShuffleConfig", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Describes the training, validation or test dataset and the Amazon S3 location where it is stored.

        :param channel_name: Name of the channel.
        :param data_source: Location of the channel data.
        :param compression_type: Compression type if training data is compressed. Default: - None
        :param content_type: The MIME type of the data. Default: - None
        :param input_mode: Input mode to use for the data channel in a training job. Default: - None
        :param record_wrapper_type: Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the input data is already in RecordIO format, you don't need to set this attribute. Default: - None
        :param shuffle_config: Shuffle config option for input data in a channel. Default: - None

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # s3_location: stepfunctions_tasks.S3Location
            
            channel = stepfunctions_tasks.Channel(
                channel_name="channelName",
                data_source=stepfunctions_tasks.DataSource(
                    s3_data_source=stepfunctions_tasks.S3DataSource(
                        s3_location=s3_location,
            
                        # the properties below are optional
                        attribute_names=["attributeNames"],
                        s3_data_distribution_type=stepfunctions_tasks.S3DataDistributionType.FULLY_REPLICATED,
                        s3_data_type=stepfunctions_tasks.S3DataType.MANIFEST_FILE
                    )
                ),
            
                # the properties below are optional
                compression_type=stepfunctions_tasks.CompressionType.NONE,
                content_type="contentType",
                input_mode=stepfunctions_tasks.InputMode.PIPE,
                record_wrapper_type=stepfunctions_tasks.RecordWrapperType.NONE,
                shuffle_config=stepfunctions_tasks.ShuffleConfig(
                    seed=123
                )
            )
        '''
        if isinstance(data_source, dict):
            data_source = DataSource(**data_source)
        if isinstance(shuffle_config, dict):
            shuffle_config = ShuffleConfig(**shuffle_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fde65c1648641b68064c5fde3aa4a36cb79bd09822197ea0306c1b558328822e)
            check_type(argname="argument channel_name", value=channel_name, expected_type=type_hints["channel_name"])
            check_type(argname="argument data_source", value=data_source, expected_type=type_hints["data_source"])
            check_type(argname="argument compression_type", value=compression_type, expected_type=type_hints["compression_type"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument input_mode", value=input_mode, expected_type=type_hints["input_mode"])
            check_type(argname="argument record_wrapper_type", value=record_wrapper_type, expected_type=type_hints["record_wrapper_type"])
            check_type(argname="argument shuffle_config", value=shuffle_config, expected_type=type_hints["shuffle_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "channel_name": channel_name,
            "data_source": data_source,
        }
        if compression_type is not None:
            self._values["compression_type"] = compression_type
        if content_type is not None:
            self._values["content_type"] = content_type
        if input_mode is not None:
            self._values["input_mode"] = input_mode
        if record_wrapper_type is not None:
            self._values["record_wrapper_type"] = record_wrapper_type
        if shuffle_config is not None:
            self._values["shuffle_config"] = shuffle_config

    @builtins.property
    def channel_name(self) -> builtins.str:
        '''Name of the channel.'''
        result = self._values.get("channel_name")
        assert result is not None, "Required property 'channel_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def data_source(self) -> "DataSource":
        '''Location of the channel data.'''
        result = self._values.get("data_source")
        assert result is not None, "Required property 'data_source' is missing"
        return typing.cast("DataSource", result)

    @builtins.property
    def compression_type(self) -> typing.Optional["CompressionType"]:
        '''Compression type if training data is compressed.

        :default: - None
        '''
        result = self._values.get("compression_type")
        return typing.cast(typing.Optional["CompressionType"], result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''The MIME type of the data.

        :default: - None
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_mode(self) -> typing.Optional["InputMode"]:
        '''Input mode to use for the data channel in a training job.

        :default: - None
        '''
        result = self._values.get("input_mode")
        return typing.cast(typing.Optional["InputMode"], result)

    @builtins.property
    def record_wrapper_type(self) -> typing.Optional["RecordWrapperType"]:
        '''Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format.

        In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record.
        If the input data is already in RecordIO format, you don't need to set this attribute.

        :default: - None
        '''
        result = self._values.get("record_wrapper_type")
        return typing.cast(typing.Optional["RecordWrapperType"], result)

    @builtins.property
    def shuffle_config(self) -> typing.Optional["ShuffleConfig"]:
        '''Shuffle config option for input data in a channel.

        :default: - None
        '''
        result = self._values.get("shuffle_config")
        return typing.cast(typing.Optional["ShuffleConfig"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Channel(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Classification(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.Classification",
):
    '''The classification within a EMR Containers application configuration.

    Class can be extended to add other classifications.
    For example, new Classification('xxx-yyy');

    :exampleMetadata: infused

    Example::

        tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
            virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
            release_label=tasks.ReleaseLabel.EMR_6_2_0,
            job_name="EMR-Containers-Job",
            job_driver=tasks.JobDriver(
                spark_submit_job_driver=tasks.SparkSubmitJobDriver(
                    entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                )
            ),
            application_config=[tasks.ApplicationConfiguration(
                classification=tasks.Classification.SPARK_DEFAULTS,
                properties={
                    "spark.executor.instances": "1",
                    "spark.executor.memory": "512M"
                }
            )]
        )
    '''

    def __init__(self, classification_statement: builtins.str) -> None:
        '''Creates a new Classification.

        :param classification_statement: A literal string in case a new EMR classification is released, if not already defined.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4aa3125a418732ec7b4985c5191ef4461b33928aff66ff8ba99daa7508d3b9c1)
            check_type(argname="argument classification_statement", value=classification_statement, expected_type=type_hints["classification_statement"])
        jsii.create(self.__class__, self, [classification_statement])

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK")
    def SPARK(cls) -> "Classification":
        '''Sets the maximizeResourceAllocation property to true or false.

        When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.

        For more info:

        :see: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_DEFAULTS")
    def SPARK_DEFAULTS(cls) -> "Classification":
        '''Sets values in the spark-defaults.conf file.

        For more info:

        :see: https://spark.apache.org/docs/latest/configuration.html
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_DEFAULTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_ENV")
    def SPARK_ENV(cls) -> "Classification":
        '''Sets values in the spark-env.sh file.

        For more info:

        :see: https://spark.apache.org/docs/latest/configuration.html#environment-variables
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_ENV"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_HIVE_SITE")
    def SPARK_HIVE_SITE(cls) -> "Classification":
        '''Sets values in the hive-site.xml for Spark.'''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_HIVE_SITE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_LOG4J")
    def SPARK_LOG4_J(cls) -> "Classification":
        '''Sets values in the log4j.properties file.

        For more settings and info:

        :see: https://github.com/apache/spark/blob/master/conf/log4j.properties.template
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_LOG4J"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_METRICS")
    def SPARK_METRICS(cls) -> "Classification":
        '''Sets values in the metrics.properties file.

        For more settings and info:

        :see: https://github.com/apache/spark/blob/master/conf/metrics.properties.template
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_METRICS"))

    @builtins.property
    @jsii.member(jsii_name="classificationStatement")
    def classification_statement(self) -> builtins.str:
        '''A literal string in case a new EMR classification is released, if not already defined.'''
        return typing.cast(builtins.str, jsii.get(self, "classificationStatement"))


class CodeBuildStartBuild(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CodeBuildStartBuild",
):
    '''Start a CodeBuild Build as a task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_codebuild as codebuild
        
        
        codebuild_project = codebuild.Project(self, "Project",
            project_name="MyTestProject",
            build_spec=codebuild.BuildSpec.from_object({
                "version": "0.2",
                "phases": {
                    "build": {
                        "commands": ["echo \"Hello, CodeBuild!\""
                        ]
                    }
                }
            })
        )
        
        task = tasks.CodeBuildStartBuild(self, "Task",
            project=codebuild_project,
            integration_pattern=sfn.IntegrationPattern.RUN_JOB,
            environment_variables_override={
                "ZONE": codebuild.BuildEnvironmentVariable(
                    type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value=sfn.JsonPath.string_at("$.envVariables.zone")
                )
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param project: CodeBuild project to start.
        :param environment_variables_override: A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7420ddc3c5f914d45be1248775c7909c2f866fec2613b9197535be328a08761)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CodeBuildStartBuildProps(
            project=project,
            environment_variables_override=environment_variables_override,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CodeBuildStartBuildProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "project": "project",
        "environment_variables_override": "environmentVariablesOverride",
    },
)
class CodeBuildStartBuildProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for CodeBuildStartBuild.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param project: CodeBuild project to start.
        :param environment_variables_override: A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_codebuild as codebuild
            
            
            codebuild_project = codebuild.Project(self, "Project",
                project_name="MyTestProject",
                build_spec=codebuild.BuildSpec.from_object({
                    "version": "0.2",
                    "phases": {
                        "build": {
                            "commands": ["echo \"Hello, CodeBuild!\""
                            ]
                        }
                    }
                })
            )
            
            task = tasks.CodeBuildStartBuild(self, "Task",
                project=codebuild_project,
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                environment_variables_override={
                    "ZONE": codebuild.BuildEnvironmentVariable(
                        type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                        value=sfn.JsonPath.string_at("$.envVariables.zone")
                    )
                }
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ebd97f3c78209d84d4f3a8d1f76a2b5cb48ba6488d6d752e425bf3de372b16d6)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument project", value=project, expected_type=type_hints["project"])
            check_type(argname="argument environment_variables_override", value=environment_variables_override, expected_type=type_hints["environment_variables_override"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "project": project,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if environment_variables_override is not None:
            self._values["environment_variables_override"] = environment_variables_override

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def project(self) -> _IProject_aafae30a:
        '''CodeBuild project to start.'''
        result = self._values.get("project")
        assert result is not None, "Required property 'project' is missing"
        return typing.cast(_IProject_aafae30a, result)

    @builtins.property
    def environment_variables_override(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, _BuildEnvironmentVariable_7df4fa0c]]:
        '''A set of environment variables to be used for this build only.

        :default: - the latest environment variables already defined in the build project.
        '''
        result = self._values.get("environment_variables_override")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, _BuildEnvironmentVariable_7df4fa0c]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeBuildStartBuildProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CommonEcsRunTaskProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "task_definition": "taskDefinition",
        "container_overrides": "containerOverrides",
        "integration_pattern": "integrationPattern",
    },
)
class CommonEcsRunTaskProps:
    def __init__(
        self,
        *,
        cluster: _ICluster_16cddd09,
        task_definition: _TaskDefinition_a541a103,
        container_overrides: typing.Optional[typing.Sequence[typing.Union["ContainerOverride", typing.Dict[builtins.str, typing.Any]]]] = None,
        integration_pattern: typing.Optional[_ServiceIntegrationPattern_39230c49] = None,
    ) -> None:
        '''Basic properties for ECS Tasks.

        :param cluster: The topic to run the task on.
        :param task_definition: Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions If you want to run a RunTask with an imported task definition, consider using CustomState
        :param container_overrides: Container setting overrides. Key is the name of the container to override, value is the values you want to override. Default: - No overrides
        :param integration_pattern: The service integration pattern indicates different ways to call RunTask in ECS. The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN. Default: FIRE_AND_FORGET

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # task_definition: ecs.TaskDefinition
            
            common_ecs_run_task_props = stepfunctions_tasks.CommonEcsRunTaskProps(
                cluster=cluster,
                task_definition=task_definition,
            
                # the properties below are optional
                container_overrides=[stepfunctions_tasks.ContainerOverride(
                    container_definition=container_definition,
            
                    # the properties below are optional
                    command=["command"],
                    cpu=123,
                    environment=[stepfunctions_tasks.TaskEnvironmentVariable(
                        name="name",
                        value="value"
                    )],
                    memory_limit=123,
                    memory_reservation=123
                )],
                integration_pattern=stepfunctions.ServiceIntegrationPattern.FIRE_AND_FORGET
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8626ddaf1a45c28a9aef828bd240918d820ef568f2f65c0ea555a58b00d42563)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument task_definition", value=task_definition, expected_type=type_hints["task_definition"])
            check_type(argname="argument container_overrides", value=container_overrides, expected_type=type_hints["container_overrides"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "task_definition": task_definition,
        }
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern

    @builtins.property
    def cluster(self) -> _ICluster_16cddd09:
        '''The topic to run the task on.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_16cddd09, result)

    @builtins.property
    def task_definition(self) -> _TaskDefinition_a541a103:
        '''Task Definition used for running tasks in the service.

        Note: this must be TaskDefinition, and not ITaskDefinition,
        as it requires properties that are not known for imported task definitions
        If you want to run a RunTask with an imported task definition,
        consider using CustomState
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(_TaskDefinition_a541a103, result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[typing.List["ContainerOverride"]]:
        '''Container setting overrides.

        Key is the name of the container to override, value is the
        values you want to override.

        :default: - No overrides
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[typing.List["ContainerOverride"]], result)

    @builtins.property
    def integration_pattern(
        self,
    ) -> typing.Optional[_ServiceIntegrationPattern_39230c49]:
        '''The service integration pattern indicates different ways to call RunTask in ECS.

        The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.

        :default: FIRE_AND_FORGET
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_ServiceIntegrationPattern_39230c49], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonEcsRunTaskProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CompressionType")
class CompressionType(enum.Enum):
    '''Compression type of the data.'''

    NONE = "NONE"
    '''None compression type.'''
    GZIP = "GZIP"
    '''Gzip compression type.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ContainerDefinitionConfig",
    jsii_struct_bases=[],
    name_mapping={"parameters": "parameters"},
)
class ContainerDefinitionConfig:
    def __init__(
        self,
        *,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Configuration options for the ContainerDefinition.

        :param parameters: Additional parameters to pass to the base task. Default: - No additional parameters passed

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # parameters: Any
            
            container_definition_config = stepfunctions_tasks.ContainerDefinitionConfig(
                parameters={
                    "parameters_key": parameters
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d3c24d0f9c0c79891b2cc14c81d0089434fb99095dfeb261b0ae5363709003b)
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional parameters to pass to the base task.

        :default: - No additional parameters passed
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDefinitionConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ContainerDefinitionOptions",
    jsii_struct_bases=[],
    name_mapping={
        "container_host_name": "containerHostName",
        "environment_variables": "environmentVariables",
        "image": "image",
        "mode": "mode",
        "model_package_name": "modelPackageName",
        "model_s3_location": "modelS3Location",
    },
)
class ContainerDefinitionOptions:
    def __init__(
        self,
        *,
        container_host_name: typing.Optional[builtins.str] = None,
        environment_variables: typing.Optional[_TaskInput_91b91b91] = None,
        image: typing.Optional["DockerImage"] = None,
        mode: typing.Optional["Mode"] = None,
        model_package_name: typing.Optional[builtins.str] = None,
        model_s3_location: typing.Optional["S3Location"] = None,
    ) -> None:
        '''Properties to define a ContainerDefinition.

        :param container_host_name: This parameter is ignored for models that contain only a PrimaryContainer. When a ContainerDefinition is part of an inference pipeline, the value of the parameter uniquely identifies the container for the purposes of logging and metrics. Default: - None
        :param environment_variables: The environment variables to set in the Docker container. Default: - No variables
        :param image: The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored. Default: - None
        :param mode: Defines how many models the container hosts. Default: - Mode.SINGLE_MODEL
        :param model_package_name: The name or Amazon Resource Name (ARN) of the model package to use to create the model. Default: - None
        :param model_s3_location: The S3 path where the model artifacts, which result from model training, are stored. This path must point to a single gzip compressed tar archive (.tar.gz suffix). The S3 path is required for Amazon SageMaker built-in algorithms, but not if you use your own algorithms. Default: - None

        :see: https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateModel(self, "Sagemaker",
                model_name="MyModel",
                primary_container=tasks.ContainerDefinition(
                    image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
                    mode=tasks.Mode.SINGLE_MODEL,
                    model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b6d7cdf086e4bca11dec53e9606b73de3a3df772c437a98287f48fe6a369295)
            check_type(argname="argument container_host_name", value=container_host_name, expected_type=type_hints["container_host_name"])
            check_type(argname="argument environment_variables", value=environment_variables, expected_type=type_hints["environment_variables"])
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
            check_type(argname="argument model_package_name", value=model_package_name, expected_type=type_hints["model_package_name"])
            check_type(argname="argument model_s3_location", value=model_s3_location, expected_type=type_hints["model_s3_location"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if container_host_name is not None:
            self._values["container_host_name"] = container_host_name
        if environment_variables is not None:
            self._values["environment_variables"] = environment_variables
        if image is not None:
            self._values["image"] = image
        if mode is not None:
            self._values["mode"] = mode
        if model_package_name is not None:
            self._values["model_package_name"] = model_package_name
        if model_s3_location is not None:
            self._values["model_s3_location"] = model_s3_location

    @builtins.property
    def container_host_name(self) -> typing.Optional[builtins.str]:
        '''This parameter is ignored for models that contain only a PrimaryContainer.

        When a ContainerDefinition is part of an inference pipeline,
        the value of the parameter uniquely identifies the container for the purposes of logging and metrics.

        :default: - None
        '''
        result = self._values.get("container_host_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment_variables(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The environment variables to set in the Docker container.

        :default: - No variables
        '''
        result = self._values.get("environment_variables")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def image(self) -> typing.Optional["DockerImage"]:
        '''The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.

        :default: - None
        '''
        result = self._values.get("image")
        return typing.cast(typing.Optional["DockerImage"], result)

    @builtins.property
    def mode(self) -> typing.Optional["Mode"]:
        '''Defines how many models the container hosts.

        :default: - Mode.SINGLE_MODEL
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional["Mode"], result)

    @builtins.property
    def model_package_name(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the model package to use to create the model.

        :default: - None
        '''
        result = self._values.get("model_package_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def model_s3_location(self) -> typing.Optional["S3Location"]:
        '''The S3 path where the model artifacts, which result from model training, are stored.

        This path must point to a single gzip compressed tar archive (.tar.gz suffix).
        The S3 path is required for Amazon SageMaker built-in algorithms, but not if you use your own algorithms.

        :default: - None
        '''
        result = self._values.get("model_s3_location")
        return typing.cast(typing.Optional["S3Location"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDefinitionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ContainerOverride",
    jsii_struct_bases=[],
    name_mapping={
        "container_definition": "containerDefinition",
        "command": "command",
        "cpu": "cpu",
        "environment": "environment",
        "memory_limit": "memoryLimit",
        "memory_reservation": "memoryReservation",
    },
)
class ContainerOverride:
    def __init__(
        self,
        *,
        container_definition: _ContainerDefinition_8f3b54dc,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        cpu: typing.Optional[jsii.Number] = None,
        environment: typing.Optional[typing.Sequence[typing.Union["TaskEnvironmentVariable", typing.Dict[builtins.str, typing.Any]]]] = None,
        memory_limit: typing.Optional[jsii.Number] = None,
        memory_reservation: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''A list of container overrides that specify the name of a container and the overrides it should receive.

        :param container_definition: Name of the container inside the task definition.
        :param command: Command to run inside the container. Default: - Default command from the Docker image or the task definition
        :param cpu: The number of cpu units reserved for the container. Default: - The default value from the task definition.
        :param environment: The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the task definition. Default: - The existing environment variables from the Docker image or the task definition
        :param memory_limit: The hard limit (in MiB) of memory to present to the container. Default: - The default value from the task definition.
        :param memory_reservation: The soft limit (in MiB) of memory to reserve for the container. Default: - The default value from the task definition.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # container_definition: ecs.ContainerDefinition
            
            container_override = stepfunctions_tasks.ContainerOverride(
                container_definition=container_definition,
            
                # the properties below are optional
                command=["command"],
                cpu=123,
                environment=[stepfunctions_tasks.TaskEnvironmentVariable(
                    name="name",
                    value="value"
                )],
                memory_limit=123,
                memory_reservation=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb1cab5a6361b0aceafc1d72f0102de1db4239272cedfb8b0d47f2771ab52ee0)
            check_type(argname="argument container_definition", value=container_definition, expected_type=type_hints["container_definition"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument cpu", value=cpu, expected_type=type_hints["cpu"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument memory_limit", value=memory_limit, expected_type=type_hints["memory_limit"])
            check_type(argname="argument memory_reservation", value=memory_reservation, expected_type=type_hints["memory_reservation"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container_definition": container_definition,
        }
        if command is not None:
            self._values["command"] = command
        if cpu is not None:
            self._values["cpu"] = cpu
        if environment is not None:
            self._values["environment"] = environment
        if memory_limit is not None:
            self._values["memory_limit"] = memory_limit
        if memory_reservation is not None:
            self._values["memory_reservation"] = memory_reservation

    @builtins.property
    def container_definition(self) -> _ContainerDefinition_8f3b54dc:
        '''Name of the container inside the task definition.'''
        result = self._values.get("container_definition")
        assert result is not None, "Required property 'container_definition' is missing"
        return typing.cast(_ContainerDefinition_8f3b54dc, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Command to run inside the container.

        :default: - Default command from the Docker image or the task definition
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''The number of cpu units reserved for the container.

        :default: - The default value from the task definition.
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def environment(self) -> typing.Optional[typing.List["TaskEnvironmentVariable"]]:
        '''The environment variables to send to the container.

        You can add new environment variables, which are added to the container at launch,
        or you can override the existing environment variables from the Docker image or the task definition.

        :default: - The existing environment variables from the Docker image or the task definition
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.List["TaskEnvironmentVariable"]], result)

    @builtins.property
    def memory_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit (in MiB) of memory to present to the container.

        :default: - The default value from the task definition.
        '''
        result = self._values.get("memory_limit")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit (in MiB) of memory to reserve for the container.

        :default: - The default value from the task definition.
        '''
        result = self._values.get("memory_reservation")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerOverride(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ContainerOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "environment": "environment",
        "gpu_count": "gpuCount",
        "instance_type": "instanceType",
        "memory": "memory",
        "vcpus": "vcpus",
    },
)
class ContainerOverrides:
    def __init__(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        instance_type: typing.Optional[_InstanceType_f64915b9] = None,
        memory: typing.Optional[jsii.Number] = None,
        vcpus: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The overrides that should be sent to a container.

        :param command: The command to send to the container that overrides the default command from the Docker image or the job definition. Default: - No command overrides
        :param environment: The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the job definition. Default: - No environment overrides
        :param gpu_count: The number of physical GPUs to reserve for the container. The number of GPUs reserved for all containers in a job should not exceed the number of available GPUs on the compute resource that the job is launched on. Default: - No GPU reservation
        :param instance_type: The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container jobs. Default: - No instance type overrides
        :param memory: The number of MiB of memory reserved for the job. This value overrides the value set in the job definition. Default: - No memory overrides
        :param vcpus: The number of vCPUs to reserve for the container. This value overrides the value set in the job definition. Default: - No vCPUs overrides

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # instance_type: ec2.InstanceType
            
            container_overrides = stepfunctions_tasks.ContainerOverrides(
                command=["command"],
                environment={
                    "environment_key": "environment"
                },
                gpu_count=123,
                instance_type=instance_type,
                memory=123,
                vcpus=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__01041943486aae0759ebb5eb0da770a79ece9051da7b1da79c5ca4956ddf4542)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument gpu_count", value=gpu_count, expected_type=type_hints["gpu_count"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument vcpus", value=vcpus, expected_type=type_hints["vcpus"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if memory is not None:
            self._values["memory"] = memory
        if vcpus is not None:
            self._values["vcpus"] = vcpus

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command to send to the container that overrides the default command from the Docker image or the job definition.

        :default: - No command overrides
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to send to the container.

        You can add new environment variables, which are added to the container
        at launch, or you can override the existing environment variables from
        the Docker image or the job definition.

        :default: - No environment overrides
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''The number of physical GPUs to reserve for the container.

        The number of GPUs reserved for all containers in a job
        should not exceed the number of available GPUs on the compute
        resource that the job is launched on.

        :default: - No GPU reservation
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[_InstanceType_f64915b9]:
        '''The instance type to use for a multi-node parallel job.

        This parameter is not valid for single-node container jobs.

        :default: - No instance type overrides
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[_InstanceType_f64915b9], result)

    @builtins.property
    def memory(self) -> typing.Optional[jsii.Number]:
        '''The number of MiB of memory reserved for the job.

        This value overrides the value set in the job definition.

        :default: - No memory overrides
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def vcpus(self) -> typing.Optional[jsii.Number]:
        '''The number of vCPUs to reserve for the container.

        This value overrides the value set in the job definition.

        :default: - No vCPUs overrides
        '''
        result = self._values.get("vcpus")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DataSource",
    jsii_struct_bases=[],
    name_mapping={"s3_data_source": "s3DataSource"},
)
class DataSource:
    def __init__(
        self,
        *,
        s3_data_source: typing.Union["S3DataSource", typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''Location of the channel data.

        :param s3_data_source: S3 location of the data source that is associated with a channel.

        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
                training_job_name=sfn.JsonPath.string_at("$.JobName"),
                algorithm_specification=tasks.AlgorithmSpecification(
                    algorithm_name="BlazingText",
                    training_input_mode=tasks.InputMode.FILE
                ),
                input_data_config=[tasks.Channel(
                    channel_name="train",
                    data_source=tasks.DataSource(
                        s3_data_source=tasks.S3DataSource(
                            s3_data_type=tasks.S3DataType.S3_PREFIX,
                            s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                        )
                    )
                )],
                output_data_config=tasks.OutputDataConfig(
                    s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
                ),
                resource_config=tasks.ResourceConfig(
                    instance_count=1,
                    instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                    volume_size=Size.gibibytes(50)
                ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
                stopping_condition=tasks.StoppingCondition(
                    max_runtime=Duration.hours(2)
                )
            )
        '''
        if isinstance(s3_data_source, dict):
            s3_data_source = S3DataSource(**s3_data_source)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ae608c3d9d2fb12a77379a82dd6b04cdafe048a3cd431a1c95d472c19c1cbcd4)
            check_type(argname="argument s3_data_source", value=s3_data_source, expected_type=type_hints["s3_data_source"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "s3_data_source": s3_data_source,
        }

    @builtins.property
    def s3_data_source(self) -> "S3DataSource":
        '''S3 location of the data source that is associated with a channel.'''
        result = self._values.get("s3_data_source")
        assert result is not None, "Required property 's3_data_source' is missing"
        return typing.cast("S3DataSource", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DataSource(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DockerImage(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DockerImage",
):
    '''Creates ``IDockerImage`` instances.

    :exampleMetadata: infused

    Example::

        tasks.SageMakerCreateModel(self, "Sagemaker",
            model_name="MyModel",
            primary_container=tasks.ContainerDefinition(
                image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
                mode=tasks.Mode.SINGLE_MODEL,
                model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
            )
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromAsset")
    @builtins.classmethod
    def from_asset(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        directory: builtins.str,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_DockerImageAssetInvalidationOptions_4deb8d45, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_NetworkMode_897e5081] = None,
        outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        platform: typing.Optional[_Platform_d16f3cf1] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> "DockerImage":
        '''Reference a Docker image that is provided as an Asset in the current app.

        :param scope: the scope in which to create the Asset.
        :param id: the ID for the asset in the construct tree.
        :param directory: The directory where the Dockerfile is stored. Any directory inside with a name that matches the CDK output folder (cdk.out by default) will be excluded from the asset
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param build_secrets: Build secrets. Docker BuildKit must be enabled to use build secrets. Default: - no build secrets
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from options are passed to the build command
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to options are passed to the build command
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param outputs: Outputs to pass to the ``docker build`` command. Default: - no outputs are passed to the build command (default outputs are used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param target: Docker target to build to. Default: - no target
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5c9f551a01d287eabffcb602e6704dea79fda61e24da62e8d266fa6d10db6601)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = _DockerImageAssetProps_6897287d(
            directory=directory,
            build_args=build_args,
            build_secrets=build_secrets,
            cache_from=cache_from,
            cache_to=cache_to,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            outputs=outputs,
            platform=platform,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromAsset", [scope, id, props]))

    @jsii.member(jsii_name="fromEcrRepository")
    @builtins.classmethod
    def from_ecr_repository(
        cls,
        repository: _IRepository_e6004aa6,
        tag_or_digest: typing.Optional[builtins.str] = None,
    ) -> "DockerImage":
        '''Reference a Docker image stored in an ECR repository.

        :param repository: the ECR repository where the image is hosted.
        :param tag_or_digest: an optional tag or digest (digests must start with ``sha256:``).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__806ba4031bab68abe2cb9d2e0379ad5b9c15f7b6ab4b9d5046bd93c0c8fe0de7)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument tag_or_digest", value=tag_or_digest, expected_type=type_hints["tag_or_digest"])
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromEcrRepository", [repository, tag_or_digest]))

    @jsii.member(jsii_name="fromJsonExpression")
    @builtins.classmethod
    def from_json_expression(
        cls,
        expression: builtins.str,
        allow_any_ecr_image_pull: typing.Optional[builtins.bool] = None,
    ) -> "DockerImage":
        '''Reference a Docker image which URI is obtained from the task's input.

        :param expression: the JSON path expression with the task input.
        :param allow_any_ecr_image_pull: whether ECR access should be permitted (set to ``false`` if the image will never be in ECR).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d0cf1f7f0b1e4b3fd44b76729d471d12b5756ca0475150e64202bc2847194d6)
            check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
            check_type(argname="argument allow_any_ecr_image_pull", value=allow_any_ecr_image_pull, expected_type=type_hints["allow_any_ecr_image_pull"])
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromJsonExpression", [expression, allow_any_ecr_image_pull]))

    @jsii.member(jsii_name="fromRegistry")
    @builtins.classmethod
    def from_registry(cls, image_uri: builtins.str) -> "DockerImage":
        '''Reference a Docker image by it's URI.

        When referencing ECR images, prefer using ``inEcr``.

        :param image_uri: the URI to the docker image.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d5348c24f84717b5fdf623420c2b684b55fd2c484d57db75a379748e9ca11a47)
            check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromRegistry", [image_uri]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, task: "ISageMakerTask") -> "DockerImageConfig":
        '''Called when the image is used by a SageMaker task.

        :param task: -
        '''
        ...


class _DockerImageProxy(DockerImage):
    @jsii.member(jsii_name="bind")
    def bind(self, task: "ISageMakerTask") -> "DockerImageConfig":
        '''Called when the image is used by a SageMaker task.

        :param task: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cefa72581fe7d523101f805be92e8de90341b2a3276f95707f0e9bddcc1628e8)
            check_type(argname="argument task", value=task, expected_type=type_hints["task"])
        return typing.cast("DockerImageConfig", jsii.invoke(self, "bind", [task]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, DockerImage).__jsii_proxy_class__ = lambda : _DockerImageProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DockerImageConfig",
    jsii_struct_bases=[],
    name_mapping={"image_uri": "imageUri"},
)
class DockerImageConfig:
    def __init__(self, *, image_uri: builtins.str) -> None:
        '''Configuration for a using Docker image.

        :param image_uri: The fully qualified URI of the Docker image.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            docker_image_config = stepfunctions_tasks.DockerImageConfig(
                image_uri="imageUri"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0454997c584be7503756e2b013caa0faebbb42a4f8038ee833eb9d294993d70)
            check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image_uri": image_uri,
        }

    @builtins.property
    def image_uri(self) -> builtins.str:
        '''The fully qualified URI of the Docker image.'''
        result = self._values.get("image_uri")
        assert result is not None, "Required property 'image_uri' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerImageConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DynamoAttributeValue(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoAttributeValue",
):
    '''Represents the data for an attribute.

    Each attribute value is described as a name-value pair.
    The name is the data type, and the value is the data itself.

    :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoDeleteItem(self, "DeleteItem",
            key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
            table=my_table,
            result_path=sfn.JsonPath.DISCARD
        )
    '''

    @jsii.member(jsii_name="booleanFromJsonPath")
    @builtins.classmethod
    def boolean_from_json_path(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''Sets an attribute of type Boolean from state input through Json path.

        For example:  "BOOL": true

        :param value: Json path that specifies state input to be used.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7728f2a14bbf280203295e9fb137a892bd690575f66b9933718c3de3f5c9a02e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "booleanFromJsonPath", [value]))

    @jsii.member(jsii_name="fromBinary")
    @builtins.classmethod
    def from_binary(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''Sets an attribute of type Binary.

        For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"

        :param value: base-64 encoded string.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f6670d52bfa96c0d425b5e97b75eaa2e7a582f39bc52908b2d825a90261ddaa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromBinary", [value]))

    @jsii.member(jsii_name="fromBinarySet")
    @builtins.classmethod
    def from_binary_set(
        cls,
        value: typing.Sequence[builtins.str],
    ) -> "DynamoAttributeValue":
        '''Sets an attribute of type Binary Set.

        For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06d28403260674d92f4637d3f6d600c1d57c6c6fb1e0b617b90daef0ad200ce9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromBinarySet", [value]))

    @jsii.member(jsii_name="fromBoolean")
    @builtins.classmethod
    def from_boolean(cls, value: builtins.bool) -> "DynamoAttributeValue":
        '''Sets an attribute of type Boolean.

        For example:  "BOOL": true

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c59029e7cef9b8ea6ff874e7c89586b1c2a6a73f038a9a77264d1777a2ab7b39)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromBoolean", [value]))

    @jsii.member(jsii_name="fromList")
    @builtins.classmethod
    def from_list(
        cls,
        value: typing.Sequence["DynamoAttributeValue"],
    ) -> "DynamoAttributeValue":
        '''Sets an attribute of type List.

        For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8254000dc56241c6f3d651a5858ac846fbac084a24fcdfb7f26c70429bb2914b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromList", [value]))

    @jsii.member(jsii_name="fromMap")
    @builtins.classmethod
    def from_map(
        cls,
        value: typing.Mapping[builtins.str, "DynamoAttributeValue"],
    ) -> "DynamoAttributeValue":
        '''Sets an attribute of type Map.

        For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6df72932fbe9b157138efeb8887c9d9f779e55f0595ec7f76a2c583744baddd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromMap", [value]))

    @jsii.member(jsii_name="fromNull")
    @builtins.classmethod
    def from_null(cls, value: builtins.bool) -> "DynamoAttributeValue":
        '''Sets an attribute of type Null.

        For example:  "NULL": true

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38b3510fa30973b146932943f7fd6ea8d2c33ae86cb1ee87cb02df8009fe8fb3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromNull", [value]))

    @jsii.member(jsii_name="fromNumber")
    @builtins.classmethod
    def from_number(cls, value: jsii.Number) -> "DynamoAttributeValue":
        '''Sets a literal number.

        For example: 1234
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ee11a96fa2e564225ec629bc118978c444b08fb0b820fb2c2ca2b312b5c5b42)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromNumber", [value]))

    @jsii.member(jsii_name="fromNumberSet")
    @builtins.classmethod
    def from_number_set(
        cls,
        value: typing.Sequence[jsii.Number],
    ) -> "DynamoAttributeValue":
        '''Sets an attribute of type Number Set.

        For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__763273b4406acea894ca34483b37dfdd38b45666fc394231c3671e9fbbc6628c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromNumberSet", [value]))

    @jsii.member(jsii_name="fromString")
    @builtins.classmethod
    def from_string(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''Sets an attribute of type String.

        For example:  "S": "Hello"
        Strings may be literal values or as JsonPath. Example values:

        - ``DynamoAttributeValue.fromString('someValue')``
        - ``DynamoAttributeValue.fromString(JsonPath.stringAt('$.bar'))``

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fbbf8f1e417cbf9aca7dad642f6a69f1d7ef2dfa20206e5d9c05a53cc48a21a3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromString", [value]))

    @jsii.member(jsii_name="fromStringSet")
    @builtins.classmethod
    def from_string_set(
        cls,
        value: typing.Sequence[builtins.str],
    ) -> "DynamoAttributeValue":
        '''Sets an attribute of type String Set.

        For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f61e0bc51e49a847e76b8d727bc615f5819e0ed027f7fb0e11212aa2f4b0a67)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromStringSet", [value]))

    @jsii.member(jsii_name="listFromJsonPath")
    @builtins.classmethod
    def list_from_json_path(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''Sets an attribute of type List.

        For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"S", "Veggies"}]

        :param value: Json path that specifies state input to be used.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66fd5e4f17bedddc1f7e145c925e3b20251cb8ea731826492b3f803968f9a402)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "listFromJsonPath", [value]))

    @jsii.member(jsii_name="mapFromJsonPath")
    @builtins.classmethod
    def map_from_json_path(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''Sets an attribute of type Map.

        For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}

        :param value: Json path that specifies state input to be used.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__691184b4f7c73cd232975b850a0e441f9bef1b4ed62c8d27d65a5293c619cd08)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "mapFromJsonPath", [value]))

    @jsii.member(jsii_name="numberFromString")
    @builtins.classmethod
    def number_from_string(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''Sets an attribute of type Number.

        For example:  "N": "123.45"
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        Numbers may be expressed as literal strings or as JsonPath

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e126fba931eb951bca575ac94d4a71eb9e7ef88ace0d49c7793bd1fe32d9759)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "numberFromString", [value]))

    @jsii.member(jsii_name="numberSetFromStrings")
    @builtins.classmethod
    def number_set_from_strings(
        cls,
        value: typing.Sequence[builtins.str],
    ) -> "DynamoAttributeValue":
        '''Sets an attribute of type Number Set.

        For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        Numbers may be expressed as literal strings or as JsonPath

        :param value: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ba1b7f4d3bfa245902b4bf45d04fd0134d025d060607ad42e42b8181b46a694)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "numberSetFromStrings", [value]))

    @jsii.member(jsii_name="toObject")
    def to_object(self) -> typing.Any:
        '''Returns the DynamoDB attribute value.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toObject", []))

    @builtins.property
    @jsii.member(jsii_name="attributeValue")
    def attribute_value(self) -> typing.Any:
        '''Represents the data for the attribute.

        Data can be
        i.e. "S": "Hello"
        '''
        return typing.cast(typing.Any, jsii.get(self, "attributeValue"))


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoConsumedCapacity")
class DynamoConsumedCapacity(enum.Enum):
    '''Determines the level of detail about provisioned throughput consumption that is returned.'''

    INDEXES = "INDEXES"
    '''The response includes the aggregate ConsumedCapacity for the operation, together with ConsumedCapacity for each table and secondary index that was accessed.'''
    TOTAL = "TOTAL"
    '''The response includes only the aggregate ConsumedCapacity for the operation.'''
    NONE = "NONE"
    '''No ConsumedCapacity details are included in the response.'''


class DynamoDeleteItem(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoDeleteItem",
):
    '''A StepFunctions task to call DynamoDeleteItem.

    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoDeleteItem(self, "DeleteItem",
            key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
            table=my_table,
            result_path=sfn.JsonPath.DISCARD
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional["DynamoItemCollectionMetrics"] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param key: Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: The name of the table containing the requested item.
        :param condition_expression: A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__edb1d36a100179709070b2a353dbc0d0f9dccc17168862b0e912fbc42b5e282e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DynamoDeleteItemProps(
            key=key,
            table=table,
            condition_expression=condition_expression,
            expression_attribute_names=expression_attribute_names,
            expression_attribute_values=expression_attribute_values,
            return_consumed_capacity=return_consumed_capacity,
            return_item_collection_metrics=return_item_collection_metrics,
            return_values=return_values,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoDeleteItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "key": "key",
        "table": "table",
        "condition_expression": "conditionExpression",
        "expression_attribute_names": "expressionAttributeNames",
        "expression_attribute_values": "expressionAttributeValues",
        "return_consumed_capacity": "returnConsumedCapacity",
        "return_item_collection_metrics": "returnItemCollectionMetrics",
        "return_values": "returnValues",
    },
)
class DynamoDeleteItemProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional["DynamoItemCollectionMetrics"] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
    ) -> None:
        '''Properties for DynamoDeleteItem Task.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param key: Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: The name of the table containing the requested item.
        :param condition_expression: A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE

        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoDeleteItem(self, "DeleteItem",
                key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
                table=my_table,
                result_path=sfn.JsonPath.DISCARD
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__510b0be6b21e762c909dde7691f4bf61c85378c52e4d86e80ef8e2ef68cf2ea7)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument table", value=table, expected_type=type_hints["table"])
            check_type(argname="argument condition_expression", value=condition_expression, expected_type=type_hints["condition_expression"])
            check_type(argname="argument expression_attribute_names", value=expression_attribute_names, expected_type=type_hints["expression_attribute_names"])
            check_type(argname="argument expression_attribute_values", value=expression_attribute_values, expected_type=type_hints["expression_attribute_values"])
            check_type(argname="argument return_consumed_capacity", value=return_consumed_capacity, expected_type=type_hints["return_consumed_capacity"])
            check_type(argname="argument return_item_collection_metrics", value=return_item_collection_metrics, expected_type=type_hints["return_item_collection_metrics"])
            check_type(argname="argument return_values", value=return_values, expected_type=type_hints["return_values"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key": key,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if condition_expression is not None:
            self._values["condition_expression"] = condition_expression
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if expression_attribute_values is not None:
            self._values["expression_attribute_values"] = expression_attribute_values
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity
        if return_item_collection_metrics is not None:
            self._values["return_item_collection_metrics"] = return_item_collection_metrics
        if return_values is not None:
            self._values["return_values"] = return_values

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def key(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''Primary key of the item to retrieve.

        For the primary key, you must provide all of the attributes.
        For example, with a simple primary key, you only need to provide a value for the partition key.
        For a composite primary key, you must provide values for both the partition key and the sort key.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_504fd401:
        '''The name of the table containing the requested item.'''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_504fd401, result)

    @builtins.property
    def condition_expression(self) -> typing.Optional[builtins.str]:
        '''A condition that must be satisfied in order for a conditional DeleteItem to succeed.

        :default: - No condition expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ConditionExpression
        '''
        result = self._values.get("condition_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''One or more substitution tokens for attribute names in an expression.

        :default: - No expression attribute names

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ExpressionAttributeNames
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def expression_attribute_values(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]]:
        '''One or more values that can be substituted in an expression.

        :default: - No expression attribute values

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ExpressionAttributeValues
        '''
        result = self._values.get("expression_attribute_values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ReturnConsumedCapacity
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    @builtins.property
    def return_item_collection_metrics(
        self,
    ) -> typing.Optional["DynamoItemCollectionMetrics"]:
        '''Determines whether item collection metrics are returned.

        If set to SIZE, the response includes statistics about item collections, if any,
        that were modified during the operation are returned in the response.
        If set to NONE (the default), no statistics are returned.

        :default: DynamoItemCollectionMetrics.NONE
        '''
        result = self._values.get("return_item_collection_metrics")
        return typing.cast(typing.Optional["DynamoItemCollectionMetrics"], result)

    @builtins.property
    def return_values(self) -> typing.Optional["DynamoReturnValues"]:
        '''Use ReturnValues if you want to get the item attributes as they appeared before they were deleted.

        :default: DynamoReturnValues.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ReturnValues
        '''
        result = self._values.get("return_values")
        return typing.cast(typing.Optional["DynamoReturnValues"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoDeleteItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DynamoGetItem(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoGetItem",
):
    '''A StepFunctions task to call DynamoGetItem.

    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoGetItem(self, "Get Item",
            key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
            table=my_table
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        consistent_read: typing.Optional[builtins.bool] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        projection_expression: typing.Optional[typing.Sequence["DynamoProjectionExpression"]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param key: Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: The name of the table containing the requested item.
        :param consistent_read: Determines the read consistency model: If set to true, then the operation uses strongly consistent reads; otherwise, the operation uses eventually consistent reads. Default: false
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attributes
        :param projection_expression: An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table. These attributes can include scalars, sets, or elements of a JSON document. Default: - No projection expression
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a48aa717b6b76ef93a4795798a823a4c793e85cdb6e1994bad7a54ccc952f4d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DynamoGetItemProps(
            key=key,
            table=table,
            consistent_read=consistent_read,
            expression_attribute_names=expression_attribute_names,
            projection_expression=projection_expression,
            return_consumed_capacity=return_consumed_capacity,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoGetItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "key": "key",
        "table": "table",
        "consistent_read": "consistentRead",
        "expression_attribute_names": "expressionAttributeNames",
        "projection_expression": "projectionExpression",
        "return_consumed_capacity": "returnConsumedCapacity",
    },
)
class DynamoGetItemProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        consistent_read: typing.Optional[builtins.bool] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        projection_expression: typing.Optional[typing.Sequence["DynamoProjectionExpression"]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
    ) -> None:
        '''Properties for DynamoGetItem Task.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param key: Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: The name of the table containing the requested item.
        :param consistent_read: Determines the read consistency model: If set to true, then the operation uses strongly consistent reads; otherwise, the operation uses eventually consistent reads. Default: false
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attributes
        :param projection_expression: An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table. These attributes can include scalars, sets, or elements of a JSON document. Default: - No projection expression
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE

        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoGetItem(self, "Get Item",
                key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
                table=my_table
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3bf052a704a02426d42f908423a0d52a8c6b1252e3a38c30d6e41a412ca3308)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument table", value=table, expected_type=type_hints["table"])
            check_type(argname="argument consistent_read", value=consistent_read, expected_type=type_hints["consistent_read"])
            check_type(argname="argument expression_attribute_names", value=expression_attribute_names, expected_type=type_hints["expression_attribute_names"])
            check_type(argname="argument projection_expression", value=projection_expression, expected_type=type_hints["projection_expression"])
            check_type(argname="argument return_consumed_capacity", value=return_consumed_capacity, expected_type=type_hints["return_consumed_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key": key,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if consistent_read is not None:
            self._values["consistent_read"] = consistent_read
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if projection_expression is not None:
            self._values["projection_expression"] = projection_expression
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def key(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''Primary key of the item to retrieve.

        For the primary key, you must provide all of the attributes.
        For example, with a simple primary key, you only need to provide a value for the partition key.
        For a composite primary key, you must provide values for both the partition key and the sort key.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_504fd401:
        '''The name of the table containing the requested item.'''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_504fd401, result)

    @builtins.property
    def consistent_read(self) -> typing.Optional[builtins.bool]:
        '''Determines the read consistency model: If set to true, then the operation uses strongly consistent reads;

        otherwise, the operation uses eventually consistent reads.

        :default: false
        '''
        result = self._values.get("consistent_read")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''One or more substitution tokens for attribute names in an expression.

        :default: - No expression attributes

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ExpressionAttributeNames
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def projection_expression(
        self,
    ) -> typing.Optional[typing.List["DynamoProjectionExpression"]]:
        '''An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table.

        These attributes can include scalars, sets, or elements of a JSON document.

        :default: - No projection expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ProjectionExpression
        '''
        result = self._values.get("projection_expression")
        return typing.cast(typing.Optional[typing.List["DynamoProjectionExpression"]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ReturnConsumedCapacity
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoGetItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoItemCollectionMetrics")
class DynamoItemCollectionMetrics(enum.Enum):
    '''Determines whether item collection metrics are returned.'''

    SIZE = "SIZE"
    '''If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation.'''
    NONE = "NONE"
    '''If set to NONE, no statistics are returned.'''


class DynamoProjectionExpression(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoProjectionExpression",
):
    '''Class to generate projection expression.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        dynamo_projection_expression = stepfunctions_tasks.DynamoProjectionExpression()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="atIndex")
    def at_index(self, index: jsii.Number) -> "DynamoProjectionExpression":
        '''Adds the array literal access for passed index.

        :param index: array index.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75b74d2d527749cd3e1d152ed6cd344af0e414c1de6ac18da174d6eeb7f8b4ed)
            check_type(argname="argument index", value=index, expected_type=type_hints["index"])
        return typing.cast("DynamoProjectionExpression", jsii.invoke(self, "atIndex", [index]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''converts and return the string expression.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.member(jsii_name="withAttribute")
    def with_attribute(self, attr: builtins.str) -> "DynamoProjectionExpression":
        '''Adds the passed attribute to the chain.

        :param attr: Attribute name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f16909b37071ac5fc90a83c9c96e77d6ec9ad4ee931a670038668dc63d1dd0e)
            check_type(argname="argument attr", value=attr, expected_type=type_hints["attr"])
        return typing.cast("DynamoProjectionExpression", jsii.invoke(self, "withAttribute", [attr]))


class DynamoPutItem(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoPutItem",
):
    '''A StepFunctions task to call DynamoPutItem.

    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoPutItem(self, "PutItem",
            item={
                "MessageId": tasks.DynamoAttributeValue.from_string("message-007"),
                "Text": tasks.DynamoAttributeValue.from_string(sfn.JsonPath.string_at("$.bar")),
                "TotalCount": tasks.DynamoAttributeValue.from_number(10)
            },
            table=my_table
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        item: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param item: A map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
        :param table: The name of the table where the item should be written .
        :param condition_expression: A condition that must be satisfied in order for a conditional PutItem operation to succeed. Default: - No condition expression
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: The item collection metrics to returned in the response. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: Use ReturnValues if you want to get the item attributes as they appeared before they were updated with the PutItem request. Default: DynamoReturnValues.NONE
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__908da2e1959e7dd9a6fb0b8888d346015e5a113821bb480e8aa19402ab2e009f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DynamoPutItemProps(
            item=item,
            table=table,
            condition_expression=condition_expression,
            expression_attribute_names=expression_attribute_names,
            expression_attribute_values=expression_attribute_values,
            return_consumed_capacity=return_consumed_capacity,
            return_item_collection_metrics=return_item_collection_metrics,
            return_values=return_values,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoPutItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "item": "item",
        "table": "table",
        "condition_expression": "conditionExpression",
        "expression_attribute_names": "expressionAttributeNames",
        "expression_attribute_values": "expressionAttributeValues",
        "return_consumed_capacity": "returnConsumedCapacity",
        "return_item_collection_metrics": "returnItemCollectionMetrics",
        "return_values": "returnValues",
    },
)
class DynamoPutItemProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        item: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
    ) -> None:
        '''Properties for DynamoPutItem Task.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param item: A map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
        :param table: The name of the table where the item should be written .
        :param condition_expression: A condition that must be satisfied in order for a conditional PutItem operation to succeed. Default: - No condition expression
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: The item collection metrics to returned in the response. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: Use ReturnValues if you want to get the item attributes as they appeared before they were updated with the PutItem request. Default: DynamoReturnValues.NONE

        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoPutItem(self, "PutItem",
                item={
                    "MessageId": tasks.DynamoAttributeValue.from_string("message-007"),
                    "Text": tasks.DynamoAttributeValue.from_string(sfn.JsonPath.string_at("$.bar")),
                    "TotalCount": tasks.DynamoAttributeValue.from_number(10)
                },
                table=my_table
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__955d8ae263b142df28d2448d874e1df52a64b2f0b68fdf91f649288dfb583845)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument item", value=item, expected_type=type_hints["item"])
            check_type(argname="argument table", value=table, expected_type=type_hints["table"])
            check_type(argname="argument condition_expression", value=condition_expression, expected_type=type_hints["condition_expression"])
            check_type(argname="argument expression_attribute_names", value=expression_attribute_names, expected_type=type_hints["expression_attribute_names"])
            check_type(argname="argument expression_attribute_values", value=expression_attribute_values, expected_type=type_hints["expression_attribute_values"])
            check_type(argname="argument return_consumed_capacity", value=return_consumed_capacity, expected_type=type_hints["return_consumed_capacity"])
            check_type(argname="argument return_item_collection_metrics", value=return_item_collection_metrics, expected_type=type_hints["return_item_collection_metrics"])
            check_type(argname="argument return_values", value=return_values, expected_type=type_hints["return_values"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "item": item,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if condition_expression is not None:
            self._values["condition_expression"] = condition_expression
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if expression_attribute_values is not None:
            self._values["expression_attribute_values"] = expression_attribute_values
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity
        if return_item_collection_metrics is not None:
            self._values["return_item_collection_metrics"] = return_item_collection_metrics
        if return_values is not None:
            self._values["return_values"] = return_values

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def item(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''A map of attribute name/value pairs, one for each attribute.

        Only the primary key attributes are required;
        you can optionally provide other attribute name-value pairs for the item.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-Item
        '''
        result = self._values.get("item")
        assert result is not None, "Required property 'item' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_504fd401:
        '''The name of the table where the item should be written .'''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_504fd401, result)

    @builtins.property
    def condition_expression(self) -> typing.Optional[builtins.str]:
        '''A condition that must be satisfied in order for a conditional PutItem operation to succeed.

        :default: - No condition expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ConditionExpression
        '''
        result = self._values.get("condition_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''One or more substitution tokens for attribute names in an expression.

        :default: - No expression attribute names

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ExpressionAttributeNames
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def expression_attribute_values(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]]:
        '''One or more values that can be substituted in an expression.

        :default: - No expression attribute values

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ExpressionAttributeValues
        '''
        result = self._values.get("expression_attribute_values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ReturnConsumedCapacity
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    @builtins.property
    def return_item_collection_metrics(
        self,
    ) -> typing.Optional[DynamoItemCollectionMetrics]:
        '''The item collection metrics to returned in the response.

        :default: DynamoItemCollectionMetrics.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LSI.html#LSI.ItemCollections
        '''
        result = self._values.get("return_item_collection_metrics")
        return typing.cast(typing.Optional[DynamoItemCollectionMetrics], result)

    @builtins.property
    def return_values(self) -> typing.Optional["DynamoReturnValues"]:
        '''Use ReturnValues if you want to get the item attributes as they appeared before they were updated with the PutItem request.

        :default: DynamoReturnValues.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ReturnValues
        '''
        result = self._values.get("return_values")
        return typing.cast(typing.Optional["DynamoReturnValues"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoPutItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoReturnValues")
class DynamoReturnValues(enum.Enum):
    '''Use ReturnValues if you want to get the item attributes as they appear before or after they are changed.'''

    NONE = "NONE"
    '''Nothing is returned.'''
    ALL_OLD = "ALL_OLD"
    '''Returns all of the attributes of the item.'''
    UPDATED_OLD = "UPDATED_OLD"
    '''Returns only the updated attributes.'''
    ALL_NEW = "ALL_NEW"
    '''Returns all of the attributes of the item.'''
    UPDATED_NEW = "UPDATED_NEW"
    '''Returns only the updated attributes.'''


class DynamoUpdateItem(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoUpdateItem",
):
    '''A StepFunctions task to call DynamoUpdateItem.

    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoUpdateItem(self, "UpdateItem",
            key={
                "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
            },
            table=my_table,
            expression_attribute_values={
                ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
                ":rand": tasks.DynamoAttributeValue.from_number(20)
            },
            update_expression="SET TotalCount = :val + :rand"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional[DynamoReturnValues] = None,
        update_expression: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param key: Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: The name of the table containing the requested item.
        :param condition_expression: A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE
        :param update_expression: An expression that defines one or more attributes to be updated, the action to be performed on them, and new values for them. Default: - No update expression
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f896817c76b8d681471c18eb94a64c54da31608671fa9dd54c2488298bea718d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DynamoUpdateItemProps(
            key=key,
            table=table,
            condition_expression=condition_expression,
            expression_attribute_names=expression_attribute_names,
            expression_attribute_values=expression_attribute_values,
            return_consumed_capacity=return_consumed_capacity,
            return_item_collection_metrics=return_item_collection_metrics,
            return_values=return_values,
            update_expression=update_expression,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.DynamoUpdateItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "key": "key",
        "table": "table",
        "condition_expression": "conditionExpression",
        "expression_attribute_names": "expressionAttributeNames",
        "expression_attribute_values": "expressionAttributeValues",
        "return_consumed_capacity": "returnConsumedCapacity",
        "return_item_collection_metrics": "returnItemCollectionMetrics",
        "return_values": "returnValues",
        "update_expression": "updateExpression",
    },
)
class DynamoUpdateItemProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_504fd401,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional[DynamoReturnValues] = None,
        update_expression: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for DynamoUpdateItem Task.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param key: Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: The name of the table containing the requested item.
        :param condition_expression: A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE
        :param update_expression: An expression that defines one or more attributes to be updated, the action to be performed on them, and new values for them. Default: - No update expression

        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoUpdateItem(self, "UpdateItem",
                key={
                    "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
                },
                table=my_table,
                expression_attribute_values={
                    ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
                    ":rand": tasks.DynamoAttributeValue.from_number(20)
                },
                update_expression="SET TotalCount = :val + :rand"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7aea420152cc3e3e8760f2c58796d00aeae52b0f65e38d68c8a67540106e977)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument table", value=table, expected_type=type_hints["table"])
            check_type(argname="argument condition_expression", value=condition_expression, expected_type=type_hints["condition_expression"])
            check_type(argname="argument expression_attribute_names", value=expression_attribute_names, expected_type=type_hints["expression_attribute_names"])
            check_type(argname="argument expression_attribute_values", value=expression_attribute_values, expected_type=type_hints["expression_attribute_values"])
            check_type(argname="argument return_consumed_capacity", value=return_consumed_capacity, expected_type=type_hints["return_consumed_capacity"])
            check_type(argname="argument return_item_collection_metrics", value=return_item_collection_metrics, expected_type=type_hints["return_item_collection_metrics"])
            check_type(argname="argument return_values", value=return_values, expected_type=type_hints["return_values"])
            check_type(argname="argument update_expression", value=update_expression, expected_type=type_hints["update_expression"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key": key,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if condition_expression is not None:
            self._values["condition_expression"] = condition_expression
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if expression_attribute_values is not None:
            self._values["expression_attribute_values"] = expression_attribute_values
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity
        if return_item_collection_metrics is not None:
            self._values["return_item_collection_metrics"] = return_item_collection_metrics
        if return_values is not None:
            self._values["return_values"] = return_values
        if update_expression is not None:
            self._values["update_expression"] = update_expression

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def key(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''Primary key of the item to retrieve.

        For the primary key, you must provide all of the attributes.
        For example, with a simple primary key, you only need to provide a value for the partition key.
        For a composite primary key, you must provide values for both the partition key and the sort key.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_504fd401:
        '''The name of the table containing the requested item.'''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_504fd401, result)

    @builtins.property
    def condition_expression(self) -> typing.Optional[builtins.str]:
        '''A condition that must be satisfied in order for a conditional DeleteItem to succeed.

        :default: - No condition expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ConditionExpression
        '''
        result = self._values.get("condition_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''One or more substitution tokens for attribute names in an expression.

        :default: - No expression attribute names

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ExpressionAttributeNames
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def expression_attribute_values(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]]:
        '''One or more values that can be substituted in an expression.

        :default: - No expression attribute values

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ExpressionAttributeValues
        '''
        result = self._values.get("expression_attribute_values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ReturnConsumedCapacity
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    @builtins.property
    def return_item_collection_metrics(
        self,
    ) -> typing.Optional[DynamoItemCollectionMetrics]:
        '''Determines whether item collection metrics are returned.

        If set to SIZE, the response includes statistics about item collections, if any,
        that were modified during the operation are returned in the response.
        If set to NONE (the default), no statistics are returned.

        :default: DynamoItemCollectionMetrics.NONE
        '''
        result = self._values.get("return_item_collection_metrics")
        return typing.cast(typing.Optional[DynamoItemCollectionMetrics], result)

    @builtins.property
    def return_values(self) -> typing.Optional[DynamoReturnValues]:
        '''Use ReturnValues if you want to get the item attributes as they appeared before they were deleted.

        :default: DynamoReturnValues.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ReturnValues
        '''
        result = self._values.get("return_values")
        return typing.cast(typing.Optional[DynamoReturnValues], result)

    @builtins.property
    def update_expression(self) -> typing.Optional[builtins.str]:
        '''An expression that defines one or more attributes to be updated, the action to be performed on them, and new values for them.

        :default: - No update expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-UpdateExpression
        '''
        result = self._values.get("update_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoUpdateItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EcsEc2LaunchTargetOptions",
    jsii_struct_bases=[],
    name_mapping={
        "placement_constraints": "placementConstraints",
        "placement_strategies": "placementStrategies",
    },
)
class EcsEc2LaunchTargetOptions:
    def __init__(
        self,
        *,
        placement_constraints: typing.Optional[typing.Sequence[_PlacementConstraint_11d82a52]] = None,
        placement_strategies: typing.Optional[typing.Sequence[_PlacementStrategy_2bb6c232]] = None,
    ) -> None:
        '''Options to run an ECS task on EC2 in StepFunctions and ECS.

        :param placement_constraints: Placement constraints. Default: - None
        :param placement_strategies: Placement strategies. Default: - None

        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc.from_lookup(self, "Vpc",
                is_default=True
            )
            
            cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
            cluster.add_capacity("DefaultAutoScalingGroup",
                instance_type=ec2.InstanceType("t2.micro"),
                vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
            )
            
            task_definition = ecs.TaskDefinition(self, "TD",
                compatibility=ecs.Compatibility.EC2
            )
            
            task_definition.add_container("TheContainer",
                image=ecs.ContainerImage.from_registry("foo/bar"),
                memory_limit_mi_b=256
            )
            
            run_task = tasks.EcsRunTask(self, "Run",
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                cluster=cluster,
                task_definition=task_definition,
                launch_target=tasks.EcsEc2LaunchTarget(
                    placement_strategies=[
                        ecs.PlacementStrategy.spread_across_instances(),
                        ecs.PlacementStrategy.packed_by_cpu(),
                        ecs.PlacementStrategy.randomly()
                    ],
                    placement_constraints=[
                        ecs.PlacementConstraint.member_of("blieptuut")
                    ]
                ),
                propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e21e745270f8e65bd8d333235cc2ca2dc453f19778bf88883dd84ec583fe837)
            check_type(argname="argument placement_constraints", value=placement_constraints, expected_type=type_hints["placement_constraints"])
            check_type(argname="argument placement_strategies", value=placement_strategies, expected_type=type_hints["placement_strategies"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if placement_strategies is not None:
            self._values["placement_strategies"] = placement_strategies

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.List[_PlacementConstraint_11d82a52]]:
        '''Placement constraints.

        :default: - None
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.List[_PlacementConstraint_11d82a52]], result)

    @builtins.property
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.List[_PlacementStrategy_2bb6c232]]:
        '''Placement strategies.

        :default: - None
        '''
        result = self._values.get("placement_strategies")
        return typing.cast(typing.Optional[typing.List[_PlacementStrategy_2bb6c232]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsEc2LaunchTargetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EcsFargateLaunchTargetOptions",
    jsii_struct_bases=[],
    name_mapping={"platform_version": "platformVersion"},
)
class EcsFargateLaunchTargetOptions:
    def __init__(self, *, platform_version: _FargatePlatformVersion_55d8be5c) -> None:
        '''Properties to define an ECS service.

        :param platform_version: Refers to a specific runtime environment for Fargate task infrastructure. Fargate platform version is a combination of the kernel and container runtime versions.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            ecs_fargate_launch_target_options = stepfunctions_tasks.EcsFargateLaunchTargetOptions(
                platform_version=ecs.FargatePlatformVersion.LATEST
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe248c03da98078be0c6af93840fdee2a529790cc7cd32ccb3cc42276913f15c)
            check_type(argname="argument platform_version", value=platform_version, expected_type=type_hints["platform_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "platform_version": platform_version,
        }

    @builtins.property
    def platform_version(self) -> _FargatePlatformVersion_55d8be5c:
        '''Refers to a specific runtime environment for Fargate task infrastructure.

        Fargate platform version is a combination of the kernel and container runtime versions.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
        '''
        result = self._values.get("platform_version")
        assert result is not None, "Required property 'platform_version' is missing"
        return typing.cast(_FargatePlatformVersion_55d8be5c, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsFargateLaunchTargetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EcsLaunchTargetConfig",
    jsii_struct_bases=[],
    name_mapping={"parameters": "parameters"},
)
class EcsLaunchTargetConfig:
    def __init__(
        self,
        *,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Configuration options for the ECS launch type.

        :param parameters: Additional parameters to pass to the base task. Default: - No additional parameters passed

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # parameters: Any
            
            ecs_launch_target_config = stepfunctions_tasks.EcsLaunchTargetConfig(
                parameters={
                    "parameters_key": parameters
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f4b47cb94e1d99fed92e8d358a53d375405bce127a4b6d6c9ce264e058351b45)
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional parameters to pass to the base task.

        :default: - No additional parameters passed
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsLaunchTargetConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IConnectable_10015a05)
class EcsRunTask(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EcsRunTask",
):
    '''Run a Task on ECS or Fargate.

    :exampleMetadata: infused

    Example::

        vpc = ec2.Vpc.from_lookup(self, "Vpc",
            is_default=True
        )
        
        cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
        cluster.add_capacity("DefaultAutoScalingGroup",
            instance_type=ec2.InstanceType("t2.micro"),
            vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
        )
        
        task_definition = ecs.TaskDefinition(self, "TD",
            compatibility=ecs.Compatibility.EC2
        )
        
        task_definition.add_container("TheContainer",
            image=ecs.ContainerImage.from_registry("foo/bar"),
            memory_limit_mi_b=256
        )
        
        run_task = tasks.EcsRunTask(self, "Run",
            integration_pattern=sfn.IntegrationPattern.RUN_JOB,
            cluster=cluster,
            task_definition=task_definition,
            launch_target=tasks.EcsEc2LaunchTarget(
                placement_strategies=[
                    ecs.PlacementStrategy.spread_across_instances(),
                    ecs.PlacementStrategy.packed_by_cpu(),
                    ecs.PlacementStrategy.randomly()
                ],
                placement_constraints=[
                    ecs.PlacementConstraint.member_of("blieptuut")
                ]
            ),
            propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: _ICluster_16cddd09,
        launch_target: "IEcsLaunchTarget",
        task_definition: _TaskDefinition_a541a103,
        assign_public_ip: typing.Optional[builtins.bool] = None,
        container_overrides: typing.Optional[typing.Sequence[typing.Union[ContainerOverride, typing.Dict[builtins.str, typing.Any]]]] = None,
        propagated_tag_source: typing.Optional[_PropagatedTagSource_ad4e874a] = None,
        revision_number: typing.Optional[jsii.Number] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_acf8a799]] = None,
        subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster: The ECS cluster to run the task on.
        :param launch_target: An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.
        :param task_definition: [disable-awslint:ref-via-interface] Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions If you want to run a RunTask with an imported task definition, consider using CustomState
        :param assign_public_ip: Assign public IP addresses to each task. Default: false
        :param container_overrides: Container setting overrides. Specify the container to use and the overrides to apply. Default: - No overrides
        :param propagated_tag_source: Specifies whether to propagate the tags from the task definition to the task. An error will be received if you specify the SERVICE option when running a task. Default: - No tags are propagated.
        :param revision_number: The revision number of ECS task definiton family. Default: - '$latest'
        :param security_groups: Existing security groups to use for the tasks. Default: - A new security group is created
        :param subnets: Subnets to place the task's ENIs. Default: - Public subnets if assignPublicIp is set. Private subnets otherwise.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b9fa1de6876853cd0aa59517079cbcfc7017daf643f17b13556a7a5f1c892e3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EcsRunTaskProps(
            cluster=cluster,
            launch_target=launch_target,
            task_definition=task_definition,
            assign_public_ip=assign_public_ip,
            container_overrides=container_overrides,
            propagated_tag_source=propagated_tag_source,
            revision_number=revision_number,
            security_groups=security_groups,
            subnets=subnets,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="connections")
    def connections(self) -> _Connections_0f31fce8:
        '''Manage allowed network traffic for this service.'''
        return typing.cast(_Connections_0f31fce8, jsii.get(self, "connections"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EcsRunTaskProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster": "cluster",
        "launch_target": "launchTarget",
        "task_definition": "taskDefinition",
        "assign_public_ip": "assignPublicIp",
        "container_overrides": "containerOverrides",
        "propagated_tag_source": "propagatedTagSource",
        "revision_number": "revisionNumber",
        "security_groups": "securityGroups",
        "subnets": "subnets",
    },
)
class EcsRunTaskProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster: _ICluster_16cddd09,
        launch_target: "IEcsLaunchTarget",
        task_definition: _TaskDefinition_a541a103,
        assign_public_ip: typing.Optional[builtins.bool] = None,
        container_overrides: typing.Optional[typing.Sequence[typing.Union[ContainerOverride, typing.Dict[builtins.str, typing.Any]]]] = None,
        propagated_tag_source: typing.Optional[_PropagatedTagSource_ad4e874a] = None,
        revision_number: typing.Optional[jsii.Number] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_acf8a799]] = None,
        subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Properties for ECS Tasks.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster: The ECS cluster to run the task on.
        :param launch_target: An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.
        :param task_definition: [disable-awslint:ref-via-interface] Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions If you want to run a RunTask with an imported task definition, consider using CustomState
        :param assign_public_ip: Assign public IP addresses to each task. Default: false
        :param container_overrides: Container setting overrides. Specify the container to use and the overrides to apply. Default: - No overrides
        :param propagated_tag_source: Specifies whether to propagate the tags from the task definition to the task. An error will be received if you specify the SERVICE option when running a task. Default: - No tags are propagated.
        :param revision_number: The revision number of ECS task definiton family. Default: - '$latest'
        :param security_groups: Existing security groups to use for the tasks. Default: - A new security group is created
        :param subnets: Subnets to place the task's ENIs. Default: - Public subnets if assignPublicIp is set. Private subnets otherwise.

        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc.from_lookup(self, "Vpc",
                is_default=True
            )
            
            cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
            cluster.add_capacity("DefaultAutoScalingGroup",
                instance_type=ec2.InstanceType("t2.micro"),
                vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
            )
            
            task_definition = ecs.TaskDefinition(self, "TD",
                compatibility=ecs.Compatibility.EC2
            )
            
            task_definition.add_container("TheContainer",
                image=ecs.ContainerImage.from_registry("foo/bar"),
                memory_limit_mi_b=256
            )
            
            run_task = tasks.EcsRunTask(self, "Run",
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                cluster=cluster,
                task_definition=task_definition,
                launch_target=tasks.EcsEc2LaunchTarget(
                    placement_strategies=[
                        ecs.PlacementStrategy.spread_across_instances(),
                        ecs.PlacementStrategy.packed_by_cpu(),
                        ecs.PlacementStrategy.randomly()
                    ],
                    placement_constraints=[
                        ecs.PlacementConstraint.member_of("blieptuut")
                    ]
                ),
                propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(subnets, dict):
            subnets = _SubnetSelection_e57d76df(**subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f880997570006602fac387ce93510039132ec3f42612722766b67ec68129da1)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument launch_target", value=launch_target, expected_type=type_hints["launch_target"])
            check_type(argname="argument task_definition", value=task_definition, expected_type=type_hints["task_definition"])
            check_type(argname="argument assign_public_ip", value=assign_public_ip, expected_type=type_hints["assign_public_ip"])
            check_type(argname="argument container_overrides", value=container_overrides, expected_type=type_hints["container_overrides"])
            check_type(argname="argument propagated_tag_source", value=propagated_tag_source, expected_type=type_hints["propagated_tag_source"])
            check_type(argname="argument revision_number", value=revision_number, expected_type=type_hints["revision_number"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "launch_target": launch_target,
            "task_definition": task_definition,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign_public_ip is not None:
            self._values["assign_public_ip"] = assign_public_ip
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if propagated_tag_source is not None:
            self._values["propagated_tag_source"] = propagated_tag_source
        if revision_number is not None:
            self._values["revision_number"] = revision_number
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if subnets is not None:
            self._values["subnets"] = subnets

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster(self) -> _ICluster_16cddd09:
        '''The ECS cluster to run the task on.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_16cddd09, result)

    @builtins.property
    def launch_target(self) -> "IEcsLaunchTarget":
        '''An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html
        '''
        result = self._values.get("launch_target")
        assert result is not None, "Required property 'launch_target' is missing"
        return typing.cast("IEcsLaunchTarget", result)

    @builtins.property
    def task_definition(self) -> _TaskDefinition_a541a103:
        '''[disable-awslint:ref-via-interface] Task Definition used for running tasks in the service.

        Note: this must be TaskDefinition, and not ITaskDefinition,
        as it requires properties that are not known for imported task definitions
        If you want to run a RunTask with an imported task definition,
        consider using CustomState
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(_TaskDefinition_a541a103, result)

    @builtins.property
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''Assign public IP addresses to each task.

        :default: false
        '''
        result = self._values.get("assign_public_ip")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[typing.List[ContainerOverride]]:
        '''Container setting overrides.

        Specify the container to use and the overrides to apply.

        :default: - No overrides
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[typing.List[ContainerOverride]], result)

    @builtins.property
    def propagated_tag_source(self) -> typing.Optional[_PropagatedTagSource_ad4e874a]:
        '''Specifies whether to propagate the tags from the task definition to the task.

        An error will be received if you specify the SERVICE option when running a task.

        :default: - No tags are propagated.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html#ECS-RunTask-request-propagateTags
        '''
        result = self._values.get("propagated_tag_source")
        return typing.cast(typing.Optional[_PropagatedTagSource_ad4e874a], result)

    @builtins.property
    def revision_number(self) -> typing.Optional[jsii.Number]:
        '''The revision number of ECS task definiton family.

        :default: - '$latest'
        '''
        result = self._values.get("revision_number")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[_ISecurityGroup_acf8a799]]:
        '''Existing security groups to use for the tasks.

        :default: - A new security group is created
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_acf8a799]], result)

    @builtins.property
    def subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Subnets to place the task's ENIs.

        :default: - Public subnets if assignPublicIp is set. Private subnets otherwise.
        '''
        result = self._values.get("subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsRunTaskProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EksCall(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EksCall",
):
    '''Call a EKS endpoint as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_eks as eks
        
        
        my_eks_cluster = eks.Cluster(self, "my sample cluster",
            version=eks.KubernetesVersion.V1_18,
            cluster_name="myEksCluster"
        )
        
        tasks.EksCall(self, "Call a EKS Endpoint",
            cluster=my_eks_cluster,
            http_method=tasks.HttpMethods.GET,
            http_path="/api/v1/namespaces/default/pods"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: _ICluster_6b2b80df,
        http_method: "HttpMethods",
        http_path: builtins.str,
        query_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Sequence[builtins.str]]] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster: The EKS cluster.
        :param http_method: HTTP method ("GET", "POST", "PUT", ...) part of HTTP request.
        :param http_path: HTTP path of the Kubernetes REST API operation For example: /api/v1/namespaces/default/pods.
        :param query_parameters: Query Parameters part of HTTP request. Default: - no query parameters
        :param request_body: Request body part of HTTP request. Default: - No request body
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bfd7c8b4f5f0a6a73672aa71209e22351ee76daf9aeecf0866a8edb505fd7407)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EksCallProps(
            cluster=cluster,
            http_method=http_method,
            http_path=http_path,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        '''No policies are required due to eks:call is an Http service integration and does not call and EKS API directly.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html#connect-eks-permissions
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EksCallProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster": "cluster",
        "http_method": "httpMethod",
        "http_path": "httpPath",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class EksCallProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster: _ICluster_6b2b80df,
        http_method: "HttpMethods",
        http_path: builtins.str,
        query_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Sequence[builtins.str]]] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Properties for calling a EKS endpoint with EksCall.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster: The EKS cluster.
        :param http_method: HTTP method ("GET", "POST", "PUT", ...) part of HTTP request.
        :param http_path: HTTP path of the Kubernetes REST API operation For example: /api/v1/namespaces/default/pods.
        :param query_parameters: Query Parameters part of HTTP request. Default: - no query parameters
        :param request_body: Request body part of HTTP request. Default: - No request body

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_eks as eks
            
            
            my_eks_cluster = eks.Cluster(self, "my sample cluster",
                version=eks.KubernetesVersion.V1_18,
                cluster_name="myEksCluster"
            )
            
            tasks.EksCall(self, "Call a EKS Endpoint",
                cluster=my_eks_cluster,
                http_method=tasks.HttpMethods.GET,
                http_path="/api/v1/namespaces/default/pods"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e58be48e4bfa4586fe34b8a7150b2912601a5e10bf73dc98bed0745ccdd19cfc)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
            check_type(argname="argument http_path", value=http_path, expected_type=type_hints["http_path"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "http_method": http_method,
            "http_path": http_path,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster(self) -> _ICluster_6b2b80df:
        '''The EKS cluster.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_6b2b80df, result)

    @builtins.property
    def http_method(self) -> "HttpMethods":
        '''HTTP method ("GET", "POST", "PUT", ...) part of HTTP request.'''
        result = self._values.get("http_method")
        assert result is not None, "Required property 'http_method' is missing"
        return typing.cast("HttpMethods", result)

    @builtins.property
    def http_path(self) -> builtins.str:
        '''HTTP path of the Kubernetes REST API operation For example: /api/v1/namespaces/default/pods.'''
        result = self._values.get("http_path")
        assert result is not None, "Required property 'http_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def query_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.List[builtins.str]]]:
        '''Query Parameters part of HTTP request.

        :default: - no query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.List[builtins.str]]], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Request body part of HTTP request.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EksCallProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EksClusterInput(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EksClusterInput",
):
    '''Class that supports methods which return the EKS cluster name depending on input type.

    :exampleMetadata: infused

    Example::

        tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
            eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
            eks_namespace="specified-namespace"
        )
    '''

    @jsii.member(jsii_name="fromCluster")
    @builtins.classmethod
    def from_cluster(cls, cluster: _ICluster_6b2b80df) -> "EksClusterInput":
        '''Specify an existing EKS Cluster as the name for this Cluster.

        :param cluster: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8bb5e3bb8c3604aafbddebe7f15d858c3cc978f729537b1acfc682ae59aee1c6)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        return typing.cast("EksClusterInput", jsii.sinvoke(cls, "fromCluster", [cluster]))

    @jsii.member(jsii_name="fromTaskInput")
    @builtins.classmethod
    def from_task_input(cls, task_input: _TaskInput_91b91b91) -> "EksClusterInput":
        '''Specify a Task Input as the name for this Cluster.

        :param task_input: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__abfbaed7c398de72d96731383102087383588a40dbbae47aa53312bd808d7863)
            check_type(argname="argument task_input", value=task_input, expected_type=type_hints["task_input"])
        return typing.cast("EksClusterInput", jsii.sinvoke(cls, "fromTaskInput", [task_input]))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of the EKS Cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))


class EmrAddStep(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrAddStep",
):
    '''A Step Functions Task to add a Step to an EMR Cluster.

    The StepConfiguration is defined as Parameters in the state machine definition.

    OUTPUT: the StepId

    :exampleMetadata: infused

    Example::

        tasks.EmrAddStep(self, "Task",
            cluster_id="ClusterId",
            name="StepName",
            jar="Jar",
            action_on_failure=tasks.ActionOnFailure.CONTINUE
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        jar: builtins.str,
        name: builtins.str,
        action_on_failure: typing.Optional[ActionOnFailure] = None,
        args: typing.Optional[typing.Sequence[builtins.str]] = None,
        main_class: typing.Optional[builtins.str] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster_id: The ClusterId to add the Step to.
        :param jar: A path to a JAR file run during the step.
        :param name: The name of the Step.
        :param action_on_failure: The action to take when the cluster step fails. Default: ActionOnFailure.CONTINUE
        :param args: A list of command line arguments passed to the JAR file's main function when executed. Default: - No args
        :param main_class: The name of the main class in the specified Java file. If not specified, the JAR file should specify a Main-Class in its manifest file. Default: - No mainClass
        :param properties: A list of Java properties that are set when the step runs. You can use these properties to pass key value pairs to your main function. Default: - No properties
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3889d4459b3d727348b867bd8b4b4db43cb1a83f081697d2504a788363219b15)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrAddStepProps(
            cluster_id=cluster_id,
            jar=jar,
            name=name,
            action_on_failure=action_on_failure,
            args=args,
            main_class=main_class,
            properties=properties,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrAddStepProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "jar": "jar",
        "name": "name",
        "action_on_failure": "actionOnFailure",
        "args": "args",
        "main_class": "mainClass",
        "properties": "properties",
    },
)
class EmrAddStepProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster_id: builtins.str,
        jar: builtins.str,
        name: builtins.str,
        action_on_failure: typing.Optional[ActionOnFailure] = None,
        args: typing.Optional[typing.Sequence[builtins.str]] = None,
        main_class: typing.Optional[builtins.str] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for EmrAddStep.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster_id: The ClusterId to add the Step to.
        :param jar: A path to a JAR file run during the step.
        :param name: The name of the Step.
        :param action_on_failure: The action to take when the cluster step fails. Default: ActionOnFailure.CONTINUE
        :param args: A list of command line arguments passed to the JAR file's main function when executed. Default: - No args
        :param main_class: The name of the main class in the specified Java file. If not specified, the JAR file should specify a Main-Class in its manifest file. Default: - No mainClass
        :param properties: A list of Java properties that are set when the step runs. You can use these properties to pass key value pairs to your main function. Default: - No properties

        :exampleMetadata: infused

        Example::

            tasks.EmrAddStep(self, "Task",
                cluster_id="ClusterId",
                name="StepName",
                jar="Jar",
                action_on_failure=tasks.ActionOnFailure.CONTINUE
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16e8ce40cbe4d2af2daa80fd98236e2d21a2b602039be22115fc95da93a1d3e1)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster_id", value=cluster_id, expected_type=type_hints["cluster_id"])
            check_type(argname="argument jar", value=jar, expected_type=type_hints["jar"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument action_on_failure", value=action_on_failure, expected_type=type_hints["action_on_failure"])
            check_type(argname="argument args", value=args, expected_type=type_hints["args"])
            check_type(argname="argument main_class", value=main_class, expected_type=type_hints["main_class"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_id": cluster_id,
            "jar": jar,
            "name": name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if action_on_failure is not None:
            self._values["action_on_failure"] = action_on_failure
        if args is not None:
            self._values["args"] = args
        if main_class is not None:
            self._values["main_class"] = main_class
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''The ClusterId to add the Step to.'''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def jar(self) -> builtins.str:
        '''A path to a JAR file run during the step.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        '''
        result = self._values.get("jar")
        assert result is not None, "Required property 'jar' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the Step.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def action_on_failure(self) -> typing.Optional[ActionOnFailure]:
        '''The action to take when the cluster step fails.

        :default: ActionOnFailure.CONTINUE

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
        '''
        result = self._values.get("action_on_failure")
        return typing.cast(typing.Optional[ActionOnFailure], result)

    @builtins.property
    def args(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of command line arguments passed to the JAR file's main function when executed.

        :default: - No args

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        '''
        result = self._values.get("args")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def main_class(self) -> typing.Optional[builtins.str]:
        '''The name of the main class in the specified Java file.

        If not specified, the JAR file should specify a Main-Class in its manifest file.

        :default: - No mainClass

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        '''
        result = self._values.get("main_class")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''A list of Java properties that are set when the step runs.

        You can use these properties to pass key value pairs to your main function.

        :default: - No properties

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrAddStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrCancelStep(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCancelStep",
):
    '''A Step Functions Task to to cancel a Step on an EMR Cluster.

    :exampleMetadata: infused

    Example::

        tasks.EmrCancelStep(self, "Task",
            cluster_id="ClusterId",
            step_id="StepId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        step_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster_id: The ClusterId to update.
        :param step_id: The StepId to cancel.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be7b4f19f262543ce06c3b38b96d7720f5278fbf1216b47537353ebb5e5f3253)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrCancelStepProps(
            cluster_id=cluster_id,
            step_id=step_id,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCancelStepProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "step_id": "stepId",
    },
)
class EmrCancelStepProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster_id: builtins.str,
        step_id: builtins.str,
    ) -> None:
        '''Properties for EmrCancelStep.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster_id: The ClusterId to update.
        :param step_id: The StepId to cancel.

        :exampleMetadata: infused

        Example::

            tasks.EmrCancelStep(self, "Task",
                cluster_id="ClusterId",
                step_id="StepId"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2d4bb2455be55813cd2c251e051cf1070226b61192b46e00299c625b88e3e52)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster_id", value=cluster_id, expected_type=type_hints["cluster_id"])
            check_type(argname="argument step_id", value=step_id, expected_type=type_hints["step_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_id": cluster_id,
            "step_id": step_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''The ClusterId to update.'''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def step_id(self) -> builtins.str:
        '''The StepId to cancel.'''
        result = self._values.get("step_id")
        assert result is not None, "Required property 'step_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrCancelStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrContainersCreateVirtualCluster(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersCreateVirtualCluster",
):
    '''Task that creates an EMR Containers virtual cluster from an EKS cluster.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
            eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
            eks_namespace="specified-namespace"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        eks_cluster: EksClusterInput,
        eks_namespace: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        virtual_cluster_name: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param eks_cluster: EKS Cluster or task input that contains the name of the cluster.
        :param eks_namespace: The namespace of an EKS cluster. Default: - 'default'
        :param tags: The tags assigned to the virtual cluster. Default: {}
        :param virtual_cluster_name: Name of the virtual cluster that will be created. Default: - the name of the state machine execution that runs this task and state name
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2551d81a8c2db12e500c842b415cf280bf3e5262dfb484ab5e30ffb83e8f9ea)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrContainersCreateVirtualClusterProps(
            eks_cluster=eks_cluster,
            eks_namespace=eks_namespace,
            tags=tags,
            virtual_cluster_name=virtual_cluster_name,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersCreateVirtualClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "eks_cluster": "eksCluster",
        "eks_namespace": "eksNamespace",
        "tags": "tags",
        "virtual_cluster_name": "virtualClusterName",
    },
)
class EmrContainersCreateVirtualClusterProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        eks_cluster: EksClusterInput,
        eks_namespace: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        virtual_cluster_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties to define a EMR Containers CreateVirtualCluster Task on an EKS cluster.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param eks_cluster: EKS Cluster or task input that contains the name of the cluster.
        :param eks_namespace: The namespace of an EKS cluster. Default: - 'default'
        :param tags: The tags assigned to the virtual cluster. Default: {}
        :param virtual_cluster_name: Name of the virtual cluster that will be created. Default: - the name of the state machine execution that runs this task and state name

        :exampleMetadata: infused

        Example::

            tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
                eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
                eks_namespace="specified-namespace"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a919131aef8d10da2c45d080cca2469281203956d0996c0891a790eed357255)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument eks_cluster", value=eks_cluster, expected_type=type_hints["eks_cluster"])
            check_type(argname="argument eks_namespace", value=eks_namespace, expected_type=type_hints["eks_namespace"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_cluster_name", value=virtual_cluster_name, expected_type=type_hints["virtual_cluster_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "eks_cluster": eks_cluster,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if eks_namespace is not None:
            self._values["eks_namespace"] = eks_namespace
        if tags is not None:
            self._values["tags"] = tags
        if virtual_cluster_name is not None:
            self._values["virtual_cluster_name"] = virtual_cluster_name

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def eks_cluster(self) -> EksClusterInput:
        '''EKS Cluster or task input that contains the name of the cluster.'''
        result = self._values.get("eks_cluster")
        assert result is not None, "Required property 'eks_cluster' is missing"
        return typing.cast(EksClusterInput, result)

    @builtins.property
    def eks_namespace(self) -> typing.Optional[builtins.str]:
        '''The namespace of an EKS cluster.

        :default: - 'default'
        '''
        result = self._values.get("eks_namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags assigned to the virtual cluster.

        :default: {}
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def virtual_cluster_name(self) -> typing.Optional[builtins.str]:
        '''Name of the virtual cluster that will be created.

        :default: - the name of the state machine execution that runs this task and state name
        '''
        result = self._values.get("virtual_cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrContainersCreateVirtualClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrContainersDeleteVirtualCluster(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersDeleteVirtualCluster",
):
    '''Deletes an EMR Containers virtual cluster as a Task.

    :see: https://docs.amazonaws.cn/en_us/step-functions/latest/dg/connect-emr-eks.html
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
            virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        virtual_cluster_id: _TaskInput_91b91b91,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param virtual_cluster_id: The ID of the virtual cluster that will be deleted.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45d36fc34e9d926d46273dc2bd77731619b155106eabef0bcf97acc6f92bb711)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrContainersDeleteVirtualClusterProps(
            virtual_cluster_id=virtual_cluster_id,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersDeleteVirtualClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "virtual_cluster_id": "virtualClusterId",
    },
)
class EmrContainersDeleteVirtualClusterProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        virtual_cluster_id: _TaskInput_91b91b91,
    ) -> None:
        '''Properties to define a EMR Containers DeleteVirtualCluster Task.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param virtual_cluster_id: The ID of the virtual cluster that will be deleted.

        :exampleMetadata: infused

        Example::

            tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
                virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9195cd27c10930f3e13b47dd7dc37795ebf5a270019a8ad2374cbcf76dd48c0)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument virtual_cluster_id", value=virtual_cluster_id, expected_type=type_hints["virtual_cluster_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "virtual_cluster_id": virtual_cluster_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def virtual_cluster_id(self) -> _TaskInput_91b91b91:
        '''The ID of the virtual cluster that will be deleted.'''
        result = self._values.get("virtual_cluster_id")
        assert result is not None, "Required property 'virtual_cluster_id' is missing"
        return typing.cast(_TaskInput_91b91b91, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrContainersDeleteVirtualClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IGrantable_71c4f5de)
class EmrContainersStartJobRun(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersStartJobRun",
):
    '''Starts a job run.

    A job is a unit of work that you submit to Amazon EMR on EKS for execution.
    The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query.
    A job run is an execution of the job on the virtual cluster.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
            virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
            release_label=tasks.ReleaseLabel.EMR_6_2_0,
            job_name="EMR-Containers-Job",
            job_driver=tasks.JobDriver(
                spark_submit_job_driver=tasks.SparkSubmitJobDriver(
                    entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                )
            ),
            application_config=[tasks.ApplicationConfiguration(
                classification=tasks.Classification.SPARK_DEFAULTS,
                properties={
                    "spark.executor.instances": "1",
                    "spark.executor.memory": "512M"
                }
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        job_driver: typing.Union["JobDriver", typing.Dict[builtins.str, typing.Any]],
        release_label: "ReleaseLabel",
        virtual_cluster: "VirtualClusterInput",
        application_config: typing.Optional[typing.Sequence[typing.Union[ApplicationConfiguration, typing.Dict[builtins.str, typing.Any]]]] = None,
        execution_role: typing.Optional[_IRole_235f5d8e] = None,
        job_name: typing.Optional[builtins.str] = None,
        monitoring: typing.Optional[typing.Union["Monitoring", typing.Dict[builtins.str, typing.Any]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param job_driver: The job driver for the job run.
        :param release_label: The Amazon EMR release version to use for the job run.
        :param virtual_cluster: The ID of the virtual cluster where the job will be run.
        :param application_config: The configurations for the application running in the job run. Maximum of 100 items Default: - No application config
        :param execution_role: The execution role for the job run. If ``virtualClusterId`` is from a JSON input path, an execution role must be provided. If an execution role is provided, follow the documentation to update the role trust policy. Default: - Automatically generated only when the provided ``virtualClusterId`` is not an encoded JSON path
        :param job_name: The name of the job run. Default: - No job run name
        :param monitoring: Configuration for monitoring the job run. Default: - logging enabled and resources automatically generated if ``monitoring.logging`` is set to ``true``
        :param tags: The tags assigned to job runs. Default: - None
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e81261774db571ee755e76dd1bfc8ac82bba1efd6875d4a031b2df44c817639e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrContainersStartJobRunProps(
            job_driver=job_driver,
            release_label=release_label,
            virtual_cluster=virtual_cluster,
            application_config=application_config,
            execution_role=execution_role,
            job_name=job_name,
            monitoring=monitoring,
            tags=tags,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="grantPrincipal")
    def grant_principal(self) -> _IPrincipal_539bb2fd:
        '''The principal to grant permissions to.'''
        return typing.cast(_IPrincipal_539bb2fd, jsii.get(self, "grantPrincipal"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrContainersStartJobRunProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "job_driver": "jobDriver",
        "release_label": "releaseLabel",
        "virtual_cluster": "virtualCluster",
        "application_config": "applicationConfig",
        "execution_role": "executionRole",
        "job_name": "jobName",
        "monitoring": "monitoring",
        "tags": "tags",
    },
)
class EmrContainersStartJobRunProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        job_driver: typing.Union["JobDriver", typing.Dict[builtins.str, typing.Any]],
        release_label: "ReleaseLabel",
        virtual_cluster: "VirtualClusterInput",
        application_config: typing.Optional[typing.Sequence[typing.Union[ApplicationConfiguration, typing.Dict[builtins.str, typing.Any]]]] = None,
        execution_role: typing.Optional[_IRole_235f5d8e] = None,
        job_name: typing.Optional[builtins.str] = None,
        monitoring: typing.Optional[typing.Union["Monitoring", typing.Dict[builtins.str, typing.Any]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''The props for a EMR Containers StartJobRun Task.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param job_driver: The job driver for the job run.
        :param release_label: The Amazon EMR release version to use for the job run.
        :param virtual_cluster: The ID of the virtual cluster where the job will be run.
        :param application_config: The configurations for the application running in the job run. Maximum of 100 items Default: - No application config
        :param execution_role: The execution role for the job run. If ``virtualClusterId`` is from a JSON input path, an execution role must be provided. If an execution role is provided, follow the documentation to update the role trust policy. Default: - Automatically generated only when the provided ``virtualClusterId`` is not an encoded JSON path
        :param job_name: The name of the job run. Default: - No job run name
        :param monitoring: Configuration for monitoring the job run. Default: - logging enabled and resources automatically generated if ``monitoring.logging`` is set to ``true``
        :param tags: The tags assigned to job runs. Default: - None

        :exampleMetadata: infused

        Example::

            tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
                virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
                release_label=tasks.ReleaseLabel.EMR_6_2_0,
                job_name="EMR-Containers-Job",
                job_driver=tasks.JobDriver(
                    spark_submit_job_driver=tasks.SparkSubmitJobDriver(
                        entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                    )
                ),
                application_config=[tasks.ApplicationConfiguration(
                    classification=tasks.Classification.SPARK_DEFAULTS,
                    properties={
                        "spark.executor.instances": "1",
                        "spark.executor.memory": "512M"
                    }
                )]
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(job_driver, dict):
            job_driver = JobDriver(**job_driver)
        if isinstance(monitoring, dict):
            monitoring = Monitoring(**monitoring)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e94f69fc1284cb9be57f4899558468aff459289a87ff8dec0791e5bf733359c1)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument job_driver", value=job_driver, expected_type=type_hints["job_driver"])
            check_type(argname="argument release_label", value=release_label, expected_type=type_hints["release_label"])
            check_type(argname="argument virtual_cluster", value=virtual_cluster, expected_type=type_hints["virtual_cluster"])
            check_type(argname="argument application_config", value=application_config, expected_type=type_hints["application_config"])
            check_type(argname="argument execution_role", value=execution_role, expected_type=type_hints["execution_role"])
            check_type(argname="argument job_name", value=job_name, expected_type=type_hints["job_name"])
            check_type(argname="argument monitoring", value=monitoring, expected_type=type_hints["monitoring"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "job_driver": job_driver,
            "release_label": release_label,
            "virtual_cluster": virtual_cluster,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if application_config is not None:
            self._values["application_config"] = application_config
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if job_name is not None:
            self._values["job_name"] = job_name
        if monitoring is not None:
            self._values["monitoring"] = monitoring
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def job_driver(self) -> "JobDriver":
        '''The job driver for the job run.

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_JobDriver.html
        '''
        result = self._values.get("job_driver")
        assert result is not None, "Required property 'job_driver' is missing"
        return typing.cast("JobDriver", result)

    @builtins.property
    def release_label(self) -> "ReleaseLabel":
        '''The Amazon EMR release version to use for the job run.'''
        result = self._values.get("release_label")
        assert result is not None, "Required property 'release_label' is missing"
        return typing.cast("ReleaseLabel", result)

    @builtins.property
    def virtual_cluster(self) -> "VirtualClusterInput":
        '''The ID of the virtual cluster where the job will be run.'''
        result = self._values.get("virtual_cluster")
        assert result is not None, "Required property 'virtual_cluster' is missing"
        return typing.cast("VirtualClusterInput", result)

    @builtins.property
    def application_config(
        self,
    ) -> typing.Optional[typing.List[ApplicationConfiguration]]:
        '''The configurations for the application running in the job run.

        Maximum of 100 items

        :default: - No application config

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_Configuration.html
        '''
        result = self._values.get("application_config")
        return typing.cast(typing.Optional[typing.List[ApplicationConfiguration]], result)

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The execution role for the job run.

        If ``virtualClusterId`` is from a JSON input path, an execution role must be provided.
        If an execution role is provided, follow the documentation to update the role trust policy.

        :default: - Automatically generated only when the provided ``virtualClusterId`` is not an encoded JSON path

        :see: https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-trust-policy.html
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def job_name(self) -> typing.Optional[builtins.str]:
        '''The name of the job run.

        :default: - No job run name
        '''
        result = self._values.get("job_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def monitoring(self) -> typing.Optional["Monitoring"]:
        '''Configuration for monitoring the job run.

        :default: - logging enabled and resources automatically generated if ``monitoring.logging`` is set to ``true``

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_MonitoringConfiguration.html
        '''
        result = self._values.get("monitoring")
        return typing.cast(typing.Optional["Monitoring"], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags assigned to job runs.

        :default: - None
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrContainersStartJobRunProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrCreateCluster(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster",
):
    '''A Step Functions Task to create an EMR Cluster.

    The ClusterConfiguration is defined as Parameters in the state machine definition.

    OUTPUT: the ClusterId.

    :exampleMetadata: infused

    Example::

        cluster_role = iam.Role(self, "ClusterRole",
            assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
        )
        
        service_role = iam.Role(self, "ServiceRole",
            assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
        )
        
        auto_scaling_role = iam.Role(self, "AutoScalingRole",
            assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
        )
        
        auto_scaling_role.assume_role_policy.add_statements(
            iam.PolicyStatement(
                effect=iam.Effect.ALLOW,
                principals=[
                    iam.ServicePrincipal("application-autoscaling.amazonaws.com")
                ],
                actions=["sts:AssumeRole"
                ]
            ))
        
        tasks.EmrCreateCluster(self, "Create Cluster",
            instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
            cluster_role=cluster_role,
            name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
            service_role=service_role,
            auto_scaling_role=auto_scaling_role
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        instances: typing.Union["EmrCreateCluster.InstancesConfigProperty", typing.Dict[builtins.str, typing.Any]],
        name: builtins.str,
        additional_info: typing.Optional[builtins.str] = None,
        applications: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.ApplicationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        auto_scaling_role: typing.Optional[_IRole_235f5d8e] = None,
        bootstrap_actions: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.BootstrapActionConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        cluster_role: typing.Optional[_IRole_235f5d8e] = None,
        configurations: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        custom_ami_id: typing.Optional[builtins.str] = None,
        ebs_root_volume_size: typing.Optional[_Size_7b441c34] = None,
        kerberos_attributes: typing.Optional[typing.Union["EmrCreateCluster.KerberosAttributesProperty", typing.Dict[builtins.str, typing.Any]]] = None,
        log_uri: typing.Optional[builtins.str] = None,
        release_label: typing.Optional[builtins.str] = None,
        scale_down_behavior: typing.Optional["EmrCreateCluster.EmrClusterScaleDownBehavior"] = None,
        security_configuration: typing.Optional[builtins.str] = None,
        service_role: typing.Optional[_IRole_235f5d8e] = None,
        step_concurrency_level: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        visible_to_all_users: typing.Optional[builtins.bool] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param instances: A specification of the number and type of Amazon EC2 instances.
        :param name: The Name of the Cluster.
        :param additional_info: A JSON string for selecting additional features. Default: - None
        :param applications: A case-insensitive list of applications for Amazon EMR to install and configure when launching the cluster. Default: - EMR selected default
        :param auto_scaling_role: An IAM role for automatic scaling policies. Default: - A role will be created.
        :param bootstrap_actions: A list of bootstrap actions to run before Hadoop starts on the cluster nodes. Default: - None
        :param cluster_role: Also called instance profile and EC2 role. An IAM role for an EMR cluster. The EC2 instances of the cluster assume this role. This attribute has been renamed from jobFlowRole to clusterRole to align with other ERM/StepFunction integration parameters. Default: - - A Role will be created
        :param configurations: The list of configurations supplied for the EMR cluster you are creating. Default: - None
        :param custom_ami_id: The ID of a custom Amazon EBS-backed Linux AMI. Default: - None
        :param ebs_root_volume_size: The size of the EBS root device volume of the Linux AMI that is used for each EC2 instance. Default: - EMR selected default
        :param kerberos_attributes: Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration. Default: - None
        :param log_uri: The location in Amazon S3 to write the log files of the job flow. Default: - None
        :param release_label: The Amazon EMR release label, which determines the version of open-source application packages installed on the cluster. Default: - EMR selected default
        :param scale_down_behavior: Specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized. Default: - EMR selected default
        :param security_configuration: The name of a security configuration to apply to the cluster. Default: - None
        :param service_role: The IAM role that will be assumed by the Amazon EMR service to access AWS resources on your behalf. Default: - A role will be created that Amazon EMR service can assume.
        :param step_concurrency_level: Specifies the step concurrency level to allow multiple steps to run in parallel. Requires EMR release label 5.28.0 or above. Must be in range [1, 256]. Default: 1 - no step concurrency allowed
        :param tags: A list of tags to associate with a cluster and propagate to Amazon EC2 instances. Default: - None
        :param visible_to_all_users: A value of true indicates that all IAM users in the AWS account can perform cluster actions if they have the proper IAM policy permissions. Default: true
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cea19fa2105afe952446f5bd0f605c16f7727beccef02df2d7846fc03d43ef98)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrCreateClusterProps(
            instances=instances,
            name=name,
            additional_info=additional_info,
            applications=applications,
            auto_scaling_role=auto_scaling_role,
            bootstrap_actions=bootstrap_actions,
            cluster_role=cluster_role,
            configurations=configurations,
            custom_ami_id=custom_ami_id,
            ebs_root_volume_size=ebs_root_volume_size,
            kerberos_attributes=kerberos_attributes,
            log_uri=log_uri,
            release_label=release_label,
            scale_down_behavior=scale_down_behavior,
            security_configuration=security_configuration,
            service_role=service_role,
            step_concurrency_level=step_concurrency_level,
            tags=tags,
            visible_to_all_users=visible_to_all_users,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="autoScalingRole")
    def auto_scaling_role(self) -> _IRole_235f5d8e:
        '''The autoscaling role for the EMR Cluster.

        Only available after task has been added to a state machine.
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "autoScalingRole"))

    @builtins.property
    @jsii.member(jsii_name="clusterRole")
    def cluster_role(self) -> _IRole_235f5d8e:
        '''The instance role for the EMR Cluster.

        Only available after task has been added to a state machine.
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "clusterRole"))

    @builtins.property
    @jsii.member(jsii_name="serviceRole")
    def service_role(self) -> _IRole_235f5d8e:
        '''The service role for the EMR Cluster.

        Only available after task has been added to a state machine.
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "serviceRole"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ApplicationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "additional_info": "additionalInfo",
            "args": "args",
            "version": "version",
        },
    )
    class ApplicationConfigProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            additional_info: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
            args: typing.Optional[typing.Sequence[builtins.str]] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Properties for the EMR Cluster Applications.

            Applies to Amazon EMR releases 4.0 and later. A case-insensitive list of applications for Amazon EMR to install and configure when launching
            the cluster.

            See the RunJobFlow API for complete documentation on input parameters

            :param name: The name of the application.
            :param additional_info: This option is for advanced users only. This is meta information about third-party applications that third-party vendors use for testing purposes. Default: No additionalInfo
            :param args: Arguments for Amazon EMR to pass to the application. Default: No args
            :param version: The version of the application. Default: No version

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_Application.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                application_config_property = stepfunctions_tasks.EmrCreateCluster.ApplicationConfigProperty(
                    name="name",
                
                    # the properties below are optional
                    additional_info={
                        "additional_info_key": "additionalInfo"
                    },
                    args=["args"],
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8c9bcc0f986947c22ead4ec28a172844b6b6a7f5df5eb856689d0a3c27291547)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument additional_info", value=additional_info, expected_type=type_hints["additional_info"])
                check_type(argname="argument args", value=args, expected_type=type_hints["args"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if additional_info is not None:
                self._values["additional_info"] = additional_info
            if args is not None:
                self._values["args"] = args
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the application.'''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def additional_info(
            self,
        ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
            '''This option is for advanced users only.

            This is meta information about third-party applications that third-party vendors use
            for testing purposes.

            :default: No additionalInfo
            '''
            result = self._values.get("additional_info")
            return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

        @builtins.property
        def args(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Arguments for Amazon EMR to pass to the application.

            :default: No args
            '''
            result = self._values.get("args")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''The version of the application.

            :default: No version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ApplicationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.AutoScalingPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"constraints": "constraints", "rules": "rules"},
    )
    class AutoScalingPolicyProperty:
        def __init__(
            self,
            *,
            constraints: typing.Union["EmrCreateCluster.ScalingConstraintsProperty", typing.Dict[builtins.str, typing.Any]],
            rules: typing.Sequence[typing.Union["EmrCreateCluster.ScalingRuleProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.

            :param constraints: The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity will not cause an instance group to grow above or below these limits.
            :param rules: The scale-in and scale-out rules that comprise the automatic scaling policy.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_AutoScalingPolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                auto_scaling_policy_property = stepfunctions_tasks.EmrCreateCluster.AutoScalingPolicyProperty(
                    constraints=stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty(
                        max_capacity=123,
                        min_capacity=123
                    ),
                    rules=[stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty(
                        action=stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                            simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                                scaling_adjustment=123,
                
                                # the properties below are optional
                                adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                cool_down=123
                            ),
                
                            # the properties below are optional
                            market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                        ),
                        name="name",
                        trigger=stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                            cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                                comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                                metric_name="metricName",
                                period=cdk.Duration.minutes(30),
                
                                # the properties below are optional
                                dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                                    key="key",
                                    value="value"
                                )],
                                evaluation_periods=123,
                                namespace="namespace",
                                statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                                threshold=123,
                                unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                            )
                        ),
                
                        # the properties below are optional
                        description="description"
                    )]
                )
            '''
            if isinstance(constraints, dict):
                constraints = EmrCreateCluster.ScalingConstraintsProperty(**constraints)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__531181a287f9a80f443b52345f0fab10075ddb8943da51958dc6aae81c7d0042)
                check_type(argname="argument constraints", value=constraints, expected_type=type_hints["constraints"])
                check_type(argname="argument rules", value=rules, expected_type=type_hints["rules"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "constraints": constraints,
                "rules": rules,
            }

        @builtins.property
        def constraints(self) -> "EmrCreateCluster.ScalingConstraintsProperty":
            '''The upper and lower EC2 instance limits for an automatic scaling policy.

            Automatic scaling activity will not cause an instance
            group to grow above or below these limits.
            '''
            result = self._values.get("constraints")
            assert result is not None, "Required property 'constraints' is missing"
            return typing.cast("EmrCreateCluster.ScalingConstraintsProperty", result)

        @builtins.property
        def rules(self) -> typing.List["EmrCreateCluster.ScalingRuleProperty"]:
            '''The scale-in and scale-out rules that comprise the automatic scaling policy.'''
            result = self._values.get("rules")
            assert result is not None, "Required property 'rules' is missing"
            return typing.cast(typing.List["EmrCreateCluster.ScalingRuleProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AutoScalingPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.BootstrapActionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "script_bootstrap_action": "scriptBootstrapAction",
        },
    )
    class BootstrapActionConfigProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            script_bootstrap_action: typing.Union["EmrCreateCluster.ScriptBootstrapActionConfigProperty", typing.Dict[builtins.str, typing.Any]],
        ) -> None:
            '''Configuration of a bootstrap action.

            See the RunJobFlow API for complete documentation on input parameters

            :param name: The name of the bootstrap action.
            :param script_bootstrap_action: The script run by the bootstrap action.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_BootstrapActionConfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                bootstrap_action_config_property = stepfunctions_tasks.EmrCreateCluster.BootstrapActionConfigProperty(
                    name="name",
                    script_bootstrap_action=stepfunctions_tasks.EmrCreateCluster.ScriptBootstrapActionConfigProperty(
                        path="path",
                
                        # the properties below are optional
                        args=["args"]
                    )
                )
            '''
            if isinstance(script_bootstrap_action, dict):
                script_bootstrap_action = EmrCreateCluster.ScriptBootstrapActionConfigProperty(**script_bootstrap_action)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f0c2f0e0d59e9f09410fbb85838da9f96bdcd50c13c8ab8ae5819ee1e5799adf)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument script_bootstrap_action", value=script_bootstrap_action, expected_type=type_hints["script_bootstrap_action"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "script_bootstrap_action": script_bootstrap_action,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the bootstrap action.'''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def script_bootstrap_action(
            self,
        ) -> "EmrCreateCluster.ScriptBootstrapActionConfigProperty":
            '''The script run by the bootstrap action.'''
            result = self._values.get("script_bootstrap_action")
            assert result is not None, "Required property 'script_bootstrap_action' is missing"
            return typing.cast("EmrCreateCluster.ScriptBootstrapActionConfigProperty", result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BootstrapActionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator"
    )
    class CloudWatchAlarmComparisonOperator(enum.Enum):
        '''CloudWatch Alarm Comparison Operators.'''

        GREATER_THAN_OR_EQUAL = "GREATER_THAN_OR_EQUAL"
        '''GREATER_THAN_OR_EQUAL.'''
        GREATER_THAN = "GREATER_THAN"
        '''GREATER_THAN.'''
        LESS_THAN = "LESS_THAN"
        '''LESS_THAN.'''
        LESS_THAN_OR_EQUAL = "LESS_THAN_OR_EQUAL"
        '''LESS_THAN_OR_EQUAL.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "comparison_operator": "comparisonOperator",
            "metric_name": "metricName",
            "period": "period",
            "dimensions": "dimensions",
            "evaluation_periods": "evaluationPeriods",
            "namespace": "namespace",
            "statistic": "statistic",
            "threshold": "threshold",
            "unit": "unit",
        },
    )
    class CloudWatchAlarmDefinitionProperty:
        def __init__(
            self,
            *,
            comparison_operator: "EmrCreateCluster.CloudWatchAlarmComparisonOperator",
            metric_name: builtins.str,
            period: _Duration_4839e8c3,
            dimensions: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.MetricDimensionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            evaluation_periods: typing.Optional[jsii.Number] = None,
            namespace: typing.Optional[builtins.str] = None,
            statistic: typing.Optional["EmrCreateCluster.CloudWatchAlarmStatistic"] = None,
            threshold: typing.Optional[jsii.Number] = None,
            unit: typing.Optional["EmrCreateCluster.CloudWatchAlarmUnit"] = None,
        ) -> None:
            '''The definition of a CloudWatch metric alarm, which determines when an automatic scaling activity is triggered.

            When the defined alarm conditions
            are satisfied, scaling activity begins.

            :param comparison_operator: Determines how the metric specified by MetricName is compared to the value specified by Threshold.
            :param metric_name: The name of the CloudWatch metric that is watched to determine an alarm condition.
            :param period: The period, in seconds, over which the statistic is applied. EMR CloudWatch metrics are emitted every five minutes (300 seconds), so if an EMR CloudWatch metric is specified, specify 300.
            :param dimensions: A CloudWatch metric dimension. Default: - No dimensions
            :param evaluation_periods: The number of periods, in five-minute increments, during which the alarm condition must exist before the alarm triggers automatic scaling activity. Default: 1
            :param namespace: The namespace for the CloudWatch metric. Default: 'AWS/ElasticMapReduce'
            :param statistic: The statistic to apply to the metric associated with the alarm. Default: CloudWatchAlarmStatistic.AVERAGE
            :param threshold: The value against which the specified statistic is compared. Default: - None
            :param unit: The unit of measure associated with the CloudWatch metric being watched. The value specified for Unit must correspond to the units specified in the CloudWatch metric. Default: CloudWatchAlarmUnit.NONE

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_CloudWatchAlarmDefinition.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                cloud_watch_alarm_definition_property = stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                    comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                    metric_name="metricName",
                    period=cdk.Duration.minutes(30),
                
                    # the properties below are optional
                    dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                        key="key",
                        value="value"
                    )],
                    evaluation_periods=123,
                    namespace="namespace",
                    statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                    threshold=123,
                    unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__308d83fa08feba364914876d740ad555be0e63ef868f4b1674e396ec10f544d5)
                check_type(argname="argument comparison_operator", value=comparison_operator, expected_type=type_hints["comparison_operator"])
                check_type(argname="argument metric_name", value=metric_name, expected_type=type_hints["metric_name"])
                check_type(argname="argument period", value=period, expected_type=type_hints["period"])
                check_type(argname="argument dimensions", value=dimensions, expected_type=type_hints["dimensions"])
                check_type(argname="argument evaluation_periods", value=evaluation_periods, expected_type=type_hints["evaluation_periods"])
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
                check_type(argname="argument statistic", value=statistic, expected_type=type_hints["statistic"])
                check_type(argname="argument threshold", value=threshold, expected_type=type_hints["threshold"])
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "comparison_operator": comparison_operator,
                "metric_name": metric_name,
                "period": period,
            }
            if dimensions is not None:
                self._values["dimensions"] = dimensions
            if evaluation_periods is not None:
                self._values["evaluation_periods"] = evaluation_periods
            if namespace is not None:
                self._values["namespace"] = namespace
            if statistic is not None:
                self._values["statistic"] = statistic
            if threshold is not None:
                self._values["threshold"] = threshold
            if unit is not None:
                self._values["unit"] = unit

        @builtins.property
        def comparison_operator(
            self,
        ) -> "EmrCreateCluster.CloudWatchAlarmComparisonOperator":
            '''Determines how the metric specified by MetricName is compared to the value specified by Threshold.'''
            result = self._values.get("comparison_operator")
            assert result is not None, "Required property 'comparison_operator' is missing"
            return typing.cast("EmrCreateCluster.CloudWatchAlarmComparisonOperator", result)

        @builtins.property
        def metric_name(self) -> builtins.str:
            '''The name of the CloudWatch metric that is watched to determine an alarm condition.'''
            result = self._values.get("metric_name")
            assert result is not None, "Required property 'metric_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def period(self) -> _Duration_4839e8c3:
            '''The period, in seconds, over which the statistic is applied.

            EMR CloudWatch metrics are emitted every five minutes (300 seconds), so if
            an EMR CloudWatch metric is specified, specify 300.
            '''
            result = self._values.get("period")
            assert result is not None, "Required property 'period' is missing"
            return typing.cast(_Duration_4839e8c3, result)

        @builtins.property
        def dimensions(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.MetricDimensionProperty"]]:
            '''A CloudWatch metric dimension.

            :default: - No dimensions
            '''
            result = self._values.get("dimensions")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.MetricDimensionProperty"]], result)

        @builtins.property
        def evaluation_periods(self) -> typing.Optional[jsii.Number]:
            '''The number of periods, in five-minute increments, during which the alarm condition must exist before the alarm triggers automatic scaling activity.

            :default: 1
            '''
            result = self._values.get("evaluation_periods")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''The namespace for the CloudWatch metric.

            :default: 'AWS/ElasticMapReduce'
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def statistic(
            self,
        ) -> typing.Optional["EmrCreateCluster.CloudWatchAlarmStatistic"]:
            '''The statistic to apply to the metric associated with the alarm.

            :default: CloudWatchAlarmStatistic.AVERAGE
            '''
            result = self._values.get("statistic")
            return typing.cast(typing.Optional["EmrCreateCluster.CloudWatchAlarmStatistic"], result)

        @builtins.property
        def threshold(self) -> typing.Optional[jsii.Number]:
            '''The value against which the specified statistic is compared.

            :default: - None
            '''
            result = self._values.get("threshold")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def unit(self) -> typing.Optional["EmrCreateCluster.CloudWatchAlarmUnit"]:
            '''The unit of measure associated with the CloudWatch metric being watched.

            The value specified for Unit must correspond to the units
            specified in the CloudWatch metric.

            :default: CloudWatchAlarmUnit.NONE
            '''
            result = self._values.get("unit")
            return typing.cast(typing.Optional["EmrCreateCluster.CloudWatchAlarmUnit"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CloudWatchAlarmDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic"
    )
    class CloudWatchAlarmStatistic(enum.Enum):
        '''CloudWatch Alarm Statistics.'''

        SAMPLE_COUNT = "SAMPLE_COUNT"
        '''SAMPLE_COUNT.'''
        AVERAGE = "AVERAGE"
        '''AVERAGE.'''
        SUM = "SUM"
        '''SUM.'''
        MINIMUM = "MINIMUM"
        '''MINIMUM.'''
        MAXIMUM = "MAXIMUM"
        '''MAXIMUM.'''

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit"
    )
    class CloudWatchAlarmUnit(enum.Enum):
        '''CloudWatch Alarm Units.'''

        NONE = "NONE"
        '''NONE.'''
        SECONDS = "SECONDS"
        '''SECONDS.'''
        MICRO_SECONDS = "MICRO_SECONDS"
        '''MICRO_SECONDS.'''
        MILLI_SECONDS = "MILLI_SECONDS"
        '''MILLI_SECONDS.'''
        BYTES = "BYTES"
        '''BYTES.'''
        KILO_BYTES = "KILO_BYTES"
        '''KILO_BYTES.'''
        MEGA_BYTES = "MEGA_BYTES"
        '''MEGA_BYTES.'''
        GIGA_BYTES = "GIGA_BYTES"
        '''GIGA_BYTES.'''
        TERA_BYTES = "TERA_BYTES"
        '''TERA_BYTES.'''
        BITS = "BITS"
        '''BITS.'''
        KILO_BITS = "KILO_BITS"
        '''KILO_BITS.'''
        MEGA_BITS = "MEGA_BITS"
        '''MEGA_BITS.'''
        GIGA_BITS = "GIGA_BITS"
        '''GIGA_BITS.'''
        TERA_BITS = "TERA_BITS"
        '''TERA_BITS.'''
        PERCENT = "PERCENT"
        '''PERCENT.'''
        COUNT = "COUNT"
        '''COUNT.'''
        BYTES_PER_SECOND = "BYTES_PER_SECOND"
        '''BYTES_PER_SECOND.'''
        KILO_BYTES_PER_SECOND = "KILO_BYTES_PER_SECOND"
        '''KILO_BYTES_PER_SECOND.'''
        MEGA_BYTES_PER_SECOND = "MEGA_BYTES_PER_SECOND"
        '''MEGA_BYTES_PER_SECOND.'''
        GIGA_BYTES_PER_SECOND = "GIGA_BYTES_PER_SECOND"
        '''GIGA_BYTES_PER_SECOND.'''
        TERA_BYTES_PER_SECOND = "TERA_BYTES_PER_SECOND"
        '''TERA_BYTES_PER_SECOND.'''
        BITS_PER_SECOND = "BITS_PER_SECOND"
        '''BITS_PER_SECOND.'''
        KILO_BITS_PER_SECOND = "KILO_BITS_PER_SECOND"
        '''KILO_BITS_PER_SECOND.'''
        MEGA_BITS_PER_SECOND = "MEGA_BITS_PER_SECOND"
        '''MEGA_BITS_PER_SECOND.'''
        GIGA_BITS_PER_SECOND = "GIGA_BITS_PER_SECOND"
        '''GIGA_BITS_PER_SECOND.'''
        TERA_BITS_PER_SECOND = "TERA_BITS_PER_SECOND"
        '''TERA_BITS_PER_SECOND.'''
        COUNT_PER_SECOND = "COUNT_PER_SECOND"
        '''COUNT_PER_SECOND.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "classification": "classification",
            "configurations": "configurations",
            "properties": "properties",
        },
    )
    class ConfigurationProperty:
        def __init__(
            self,
            *,
            classification: typing.Optional[builtins.str] = None,
            configurations: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        ) -> None:
            '''An optional configuration specification to be used when provisioning cluster instances, which can include configurations for applications and software bundled with Amazon EMR.

            See the RunJobFlow API for complete documentation on input parameters

            :param classification: The classification within a configuration. Default: No classification
            :param configurations: A list of additional configurations to apply within a configuration object. Default: No configurations
            :param properties: A set of properties specified within a configuration classification. Default: No properties

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_Configuration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                
                configuration_property = stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                    classification="classification",
                    configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                        classification="classification",
                        configurations=[configuration_property_],
                        properties={
                            "properties_key": "properties"
                        }
                    )],
                    properties={
                        "properties_key": "properties"
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b8d9274e41a4fbb5e204acb98157f80259e5c167b1b884c3cb298c49404c5732)
                check_type(argname="argument classification", value=classification, expected_type=type_hints["classification"])
                check_type(argname="argument configurations", value=configurations, expected_type=type_hints["configurations"])
                check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if classification is not None:
                self._values["classification"] = classification
            if configurations is not None:
                self._values["configurations"] = configurations
            if properties is not None:
                self._values["properties"] = properties

        @builtins.property
        def classification(self) -> typing.Optional[builtins.str]:
            '''The classification within a configuration.

            :default: No classification
            '''
            result = self._values.get("classification")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]]:
            '''A list of additional configurations to apply within a configuration object.

            :default: No configurations
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]], result)

        @builtins.property
        def properties(
            self,
        ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
            '''A set of properties specified within a configuration classification.

            :default: No properties
            '''
            result = self._values.get("properties")
            return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "volume_specification": "volumeSpecification",
            "volumes_per_instance": "volumesPerInstance",
        },
    )
    class EbsBlockDeviceConfigProperty:
        def __init__(
            self,
            *,
            volume_specification: typing.Union["EmrCreateCluster.VolumeSpecificationProperty", typing.Dict[builtins.str, typing.Any]],
            volumes_per_instance: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Configuration of requested EBS block device associated with the instance group with count of volumes that will be associated to every instance.

            :param volume_specification: EBS volume specifications such as volume type, IOPS, and size (GiB) that will be requested for the EBS volume attached to an EC2 instance in the cluster.
            :param volumes_per_instance: Number of EBS volumes with a specific volume configuration that will be associated with every instance in the instance group. Default: EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_EbsBlockDeviceConfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # size: cdk.Size
                
                ebs_block_device_config_property = stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                    volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                        volume_size=size,
                        volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                        # the properties below are optional
                        iops=123
                    ),
                
                    # the properties below are optional
                    volumes_per_instance=123
                )
            '''
            if isinstance(volume_specification, dict):
                volume_specification = EmrCreateCluster.VolumeSpecificationProperty(**volume_specification)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__993e68971c6e40f7857ca8483deec1bcc102e62906c0c6edb0cad6833ddf9d38)
                check_type(argname="argument volume_specification", value=volume_specification, expected_type=type_hints["volume_specification"])
                check_type(argname="argument volumes_per_instance", value=volumes_per_instance, expected_type=type_hints["volumes_per_instance"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "volume_specification": volume_specification,
            }
            if volumes_per_instance is not None:
                self._values["volumes_per_instance"] = volumes_per_instance

        @builtins.property
        def volume_specification(
            self,
        ) -> "EmrCreateCluster.VolumeSpecificationProperty":
            '''EBS volume specifications such as volume type, IOPS, and size (GiB) that will be requested for the EBS volume attached to an EC2 instance in the cluster.'''
            result = self._values.get("volume_specification")
            assert result is not None, "Required property 'volume_specification' is missing"
            return typing.cast("EmrCreateCluster.VolumeSpecificationProperty", result)

        @builtins.property
        def volumes_per_instance(self) -> typing.Optional[jsii.Number]:
            '''Number of EBS volumes with a specific volume configuration that will be associated with every instance in the instance group.

            :default: EMR selected default
            '''
            result = self._values.get("volumes_per_instance")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsBlockDeviceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType"
    )
    class EbsBlockDeviceVolumeType(enum.Enum):
        '''EBS Volume Types.'''

        GP2 = "GP2"
        '''gp2 Volume Type.'''
        IO1 = "IO1"
        '''io1 Volume Type.'''
        STANDARD = "STANDARD"
        '''Standard Volume Type.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ebs_block_device_configs": "ebsBlockDeviceConfigs",
            "ebs_optimized": "ebsOptimized",
        },
    )
    class EbsConfigurationProperty:
        def __init__(
            self,
            *,
            ebs_block_device_configs: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.EbsBlockDeviceConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            ebs_optimized: typing.Optional[builtins.bool] = None,
        ) -> None:
            '''The Amazon EBS configuration of a cluster instance.

            :param ebs_block_device_configs: An array of Amazon EBS volume specifications attached to a cluster instance. Default: - None
            :param ebs_optimized: Indicates whether an Amazon EBS volume is EBS-optimized. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_EbsConfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # size: cdk.Size
                
                ebs_configuration_property = stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                    ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                        volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                            volume_size=size,
                            volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                            # the properties below are optional
                            iops=123
                        ),
                
                        # the properties below are optional
                        volumes_per_instance=123
                    )],
                    ebs_optimized=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9a81720bcfce5beac72fe4fa2fe331e463cd39f50c9fbd68c2407390cb729646)
                check_type(argname="argument ebs_block_device_configs", value=ebs_block_device_configs, expected_type=type_hints["ebs_block_device_configs"])
                check_type(argname="argument ebs_optimized", value=ebs_optimized, expected_type=type_hints["ebs_optimized"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ebs_block_device_configs is not None:
                self._values["ebs_block_device_configs"] = ebs_block_device_configs
            if ebs_optimized is not None:
                self._values["ebs_optimized"] = ebs_optimized

        @builtins.property
        def ebs_block_device_configs(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.EbsBlockDeviceConfigProperty"]]:
            '''An array of Amazon EBS volume specifications attached to a cluster instance.

            :default: - None
            '''
            result = self._values.get("ebs_block_device_configs")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.EbsBlockDeviceConfigProperty"]], result)

        @builtins.property
        def ebs_optimized(self) -> typing.Optional[builtins.bool]:
            '''Indicates whether an Amazon EBS volume is EBS-optimized.

            :default: - EMR selected default
            '''
            result = self._values.get("ebs_optimized")
            return typing.cast(typing.Optional[builtins.bool], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.EmrClusterScaleDownBehavior"
    )
    class EmrClusterScaleDownBehavior(enum.Enum):
        '''The Cluster ScaleDownBehavior specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html#EMR-RunJobFlow-request-ScaleDownBehavior
        '''

        TERMINATE_AT_INSTANCE_HOUR = "TERMINATE_AT_INSTANCE_HOUR"
        '''Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was submitted.

        This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
        '''
        TERMINATE_AT_TASK_COMPLETION = "TERMINATE_AT_TASK_COMPLETION"
        '''Indicates that Amazon EMR adds nodes to a deny list and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the instance-hour boundary.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstanceFleetConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_fleet_type": "instanceFleetType",
            "instance_type_configs": "instanceTypeConfigs",
            "launch_specifications": "launchSpecifications",
            "name": "name",
            "target_on_demand_capacity": "targetOnDemandCapacity",
            "target_spot_capacity": "targetSpotCapacity",
        },
    )
    class InstanceFleetConfigProperty:
        def __init__(
            self,
            *,
            instance_fleet_type: "EmrCreateCluster.InstanceRoleType",
            instance_type_configs: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.InstanceTypeConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            launch_specifications: typing.Optional[typing.Union["EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty", typing.Dict[builtins.str, typing.Any]]] = None,
            name: typing.Optional[builtins.str] = None,
            target_on_demand_capacity: typing.Optional[jsii.Number] = None,
            target_spot_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The configuration that defines an instance fleet.

            :param instance_fleet_type: The node type that the instance fleet hosts. Valid values are MASTER,CORE,and TASK.
            :param instance_type_configs: The instance type configurations that define the EC2 instances in the instance fleet. Default: No instanceTpeConfigs
            :param launch_specifications: The launch specification for the instance fleet. Default: No launchSpecifications
            :param name: The friendly name of the instance fleet. Default: No name
            :param target_on_demand_capacity: The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand instances to provision. Default: No targetOnDemandCapacity
            :param target_spot_capacity: The target capacity of Spot units for the instance fleet, which determines how many Spot instances to provision. Default: No targetSpotCapacity

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetConfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                # size: cdk.Size
                
                instance_fleet_config_property = stepfunctions_tasks.EmrCreateCluster.InstanceFleetConfigProperty(
                    instance_fleet_type=stepfunctions_tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                
                    # the properties below are optional
                    instance_type_configs=[stepfunctions_tasks.EmrCreateCluster.InstanceTypeConfigProperty(
                        instance_type="instanceType",
                
                        # the properties below are optional
                        bid_price="bidPrice",
                        bid_price_as_percentage_of_on_demand_price=123,
                        configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                            classification="classification",
                            configurations=[configuration_property_],
                            properties={
                                "properties_key": "properties"
                            }
                        )],
                        ebs_configuration=stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                            ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                                volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                                    volume_size=size,
                                    volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                                    # the properties below are optional
                                    iops=123
                                ),
                
                                # the properties below are optional
                                volumes_per_instance=123
                            )],
                            ebs_optimized=False
                        ),
                        weighted_capacity=123
                    )],
                    launch_specifications=stepfunctions_tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(
                        spot_specification=stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                            timeout_action=stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction.SWITCH_TO_ON_DEMAND,
                            timeout_duration_minutes=123,
                
                            # the properties below are optional
                            allocation_strategy=stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                            block_duration_minutes=123
                        )
                    ),
                    name="name",
                    target_on_demand_capacity=123,
                    target_spot_capacity=123
                )
            '''
            if isinstance(launch_specifications, dict):
                launch_specifications = EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(**launch_specifications)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__28f711175b544963e45f41e6432a10f405c3d502da592254bc7cc983677da3c7)
                check_type(argname="argument instance_fleet_type", value=instance_fleet_type, expected_type=type_hints["instance_fleet_type"])
                check_type(argname="argument instance_type_configs", value=instance_type_configs, expected_type=type_hints["instance_type_configs"])
                check_type(argname="argument launch_specifications", value=launch_specifications, expected_type=type_hints["launch_specifications"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument target_on_demand_capacity", value=target_on_demand_capacity, expected_type=type_hints["target_on_demand_capacity"])
                check_type(argname="argument target_spot_capacity", value=target_spot_capacity, expected_type=type_hints["target_spot_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "instance_fleet_type": instance_fleet_type,
            }
            if instance_type_configs is not None:
                self._values["instance_type_configs"] = instance_type_configs
            if launch_specifications is not None:
                self._values["launch_specifications"] = launch_specifications
            if name is not None:
                self._values["name"] = name
            if target_on_demand_capacity is not None:
                self._values["target_on_demand_capacity"] = target_on_demand_capacity
            if target_spot_capacity is not None:
                self._values["target_spot_capacity"] = target_spot_capacity

        @builtins.property
        def instance_fleet_type(self) -> "EmrCreateCluster.InstanceRoleType":
            '''The node type that the instance fleet hosts.

            Valid values are MASTER,CORE,and TASK.
            '''
            result = self._values.get("instance_fleet_type")
            assert result is not None, "Required property 'instance_fleet_type' is missing"
            return typing.cast("EmrCreateCluster.InstanceRoleType", result)

        @builtins.property
        def instance_type_configs(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.InstanceTypeConfigProperty"]]:
            '''The instance type configurations that define the EC2 instances in the instance fleet.

            :default: No instanceTpeConfigs
            '''
            result = self._values.get("instance_type_configs")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.InstanceTypeConfigProperty"]], result)

        @builtins.property
        def launch_specifications(
            self,
        ) -> typing.Optional["EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty"]:
            '''The launch specification for the instance fleet.

            :default: No launchSpecifications
            '''
            result = self._values.get("launch_specifications")
            return typing.cast(typing.Optional["EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty"], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The friendly name of the instance fleet.

            :default: No name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_on_demand_capacity(self) -> typing.Optional[jsii.Number]:
            '''The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand instances to provision.

            :default: No targetOnDemandCapacity
            '''
            result = self._values.get("target_on_demand_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def target_spot_capacity(self) -> typing.Optional[jsii.Number]:
            '''The target capacity of Spot units for the instance fleet, which determines how many Spot instances to provision.

            :default: No targetSpotCapacity
            '''
            result = self._values.get("target_spot_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceFleetConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty",
        jsii_struct_bases=[],
        name_mapping={"spot_specification": "spotSpecification"},
    )
    class InstanceFleetProvisioningSpecificationsProperty:
        def __init__(
            self,
            *,
            spot_specification: typing.Union["EmrCreateCluster.SpotProvisioningSpecificationProperty", typing.Dict[builtins.str, typing.Any]],
        ) -> None:
            '''The launch specification for Spot instances in the fleet, which determines the defined duration and provisioning timeout behavior.

            :param spot_specification: The launch specification for Spot instances in the fleet, which determines the defined duration and provisioning timeout behavior.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetProvisioningSpecifications.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                instance_fleet_provisioning_specifications_property = stepfunctions_tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(
                    spot_specification=stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                        timeout_action=stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction.SWITCH_TO_ON_DEMAND,
                        timeout_duration_minutes=123,
                
                        # the properties below are optional
                        allocation_strategy=stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                        block_duration_minutes=123
                    )
                )
            '''
            if isinstance(spot_specification, dict):
                spot_specification = EmrCreateCluster.SpotProvisioningSpecificationProperty(**spot_specification)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c27d6c374d4f11f4124618f0ce49e71f2efc7cabef4d24230f643a6e96da589c)
                check_type(argname="argument spot_specification", value=spot_specification, expected_type=type_hints["spot_specification"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "spot_specification": spot_specification,
            }

        @builtins.property
        def spot_specification(
            self,
        ) -> "EmrCreateCluster.SpotProvisioningSpecificationProperty":
            '''The launch specification for Spot instances in the fleet, which determines the defined duration and provisioning timeout behavior.'''
            result = self._values.get("spot_specification")
            assert result is not None, "Required property 'spot_specification' is missing"
            return typing.cast("EmrCreateCluster.SpotProvisioningSpecificationProperty", result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceFleetProvisioningSpecificationsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstanceGroupConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_count": "instanceCount",
            "instance_role": "instanceRole",
            "instance_type": "instanceType",
            "auto_scaling_policy": "autoScalingPolicy",
            "bid_price": "bidPrice",
            "configurations": "configurations",
            "ebs_configuration": "ebsConfiguration",
            "market": "market",
            "name": "name",
        },
    )
    class InstanceGroupConfigProperty:
        def __init__(
            self,
            *,
            instance_count: jsii.Number,
            instance_role: "EmrCreateCluster.InstanceRoleType",
            instance_type: builtins.str,
            auto_scaling_policy: typing.Optional[typing.Union["EmrCreateCluster.AutoScalingPolicyProperty", typing.Dict[builtins.str, typing.Any]]] = None,
            bid_price: typing.Optional[builtins.str] = None,
            configurations: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            ebs_configuration: typing.Optional[typing.Union["EmrCreateCluster.EbsConfigurationProperty", typing.Dict[builtins.str, typing.Any]]] = None,
            market: typing.Optional["EmrCreateCluster.InstanceMarket"] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration defining a new instance group.

            :param instance_count: Target number of instances for the instance group.
            :param instance_role: The role of the instance group in the cluster.
            :param instance_type: The EC2 instance type for all instances in the instance group.
            :param auto_scaling_policy: An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. Default: - None
            :param bid_price: The bid price for each EC2 Spot instance type as defined by InstanceType. Expressed in USD. Default: - None
            :param configurations: The list of configurations supplied for an EMR cluster instance group. Default: - None
            :param ebs_configuration: EBS configurations that will be attached to each EC2 instance in the instance group. Default: - None
            :param market: Market type of the EC2 instances used to create a cluster node. Default: - EMR selected default
            :param name: Friendly name given to the instance group. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceGroupConfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                # size: cdk.Size
                
                instance_group_config_property = stepfunctions_tasks.EmrCreateCluster.InstanceGroupConfigProperty(
                    instance_count=123,
                    instance_role=stepfunctions_tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                    instance_type="instanceType",
                
                    # the properties below are optional
                    auto_scaling_policy=stepfunctions_tasks.EmrCreateCluster.AutoScalingPolicyProperty(
                        constraints=stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty(
                            max_capacity=123,
                            min_capacity=123
                        ),
                        rules=[stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty(
                            action=stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                                simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                                    scaling_adjustment=123,
                
                                    # the properties below are optional
                                    adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                    cool_down=123
                                ),
                
                                # the properties below are optional
                                market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                            ),
                            name="name",
                            trigger=stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                                cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                                    comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                                    metric_name="metricName",
                                    period=cdk.Duration.minutes(30),
                
                                    # the properties below are optional
                                    dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    evaluation_periods=123,
                                    namespace="namespace",
                                    statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                                    threshold=123,
                                    unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                                )
                            ),
                
                            # the properties below are optional
                            description="description"
                        )]
                    ),
                    bid_price="bidPrice",
                    configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                        classification="classification",
                        configurations=[configuration_property_],
                        properties={
                            "properties_key": "properties"
                        }
                    )],
                    ebs_configuration=stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                        ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                            volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                                volume_size=size,
                                volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                                # the properties below are optional
                                iops=123
                            ),
                
                            # the properties below are optional
                            volumes_per_instance=123
                        )],
                        ebs_optimized=False
                    ),
                    market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                    name="name"
                )
            '''
            if isinstance(auto_scaling_policy, dict):
                auto_scaling_policy = EmrCreateCluster.AutoScalingPolicyProperty(**auto_scaling_policy)
            if isinstance(ebs_configuration, dict):
                ebs_configuration = EmrCreateCluster.EbsConfigurationProperty(**ebs_configuration)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__df620c4c473d84d03e3e47c7050c893b8551770ecc1face587c2a0d9f8cebf02)
                check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
                check_type(argname="argument instance_role", value=instance_role, expected_type=type_hints["instance_role"])
                check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
                check_type(argname="argument auto_scaling_policy", value=auto_scaling_policy, expected_type=type_hints["auto_scaling_policy"])
                check_type(argname="argument bid_price", value=bid_price, expected_type=type_hints["bid_price"])
                check_type(argname="argument configurations", value=configurations, expected_type=type_hints["configurations"])
                check_type(argname="argument ebs_configuration", value=ebs_configuration, expected_type=type_hints["ebs_configuration"])
                check_type(argname="argument market", value=market, expected_type=type_hints["market"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "instance_count": instance_count,
                "instance_role": instance_role,
                "instance_type": instance_type,
            }
            if auto_scaling_policy is not None:
                self._values["auto_scaling_policy"] = auto_scaling_policy
            if bid_price is not None:
                self._values["bid_price"] = bid_price
            if configurations is not None:
                self._values["configurations"] = configurations
            if ebs_configuration is not None:
                self._values["ebs_configuration"] = ebs_configuration
            if market is not None:
                self._values["market"] = market
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def instance_count(self) -> jsii.Number:
            '''Target number of instances for the instance group.'''
            result = self._values.get("instance_count")
            assert result is not None, "Required property 'instance_count' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def instance_role(self) -> "EmrCreateCluster.InstanceRoleType":
            '''The role of the instance group in the cluster.'''
            result = self._values.get("instance_role")
            assert result is not None, "Required property 'instance_role' is missing"
            return typing.cast("EmrCreateCluster.InstanceRoleType", result)

        @builtins.property
        def instance_type(self) -> builtins.str:
            '''The EC2 instance type for all instances in the instance group.'''
            result = self._values.get("instance_type")
            assert result is not None, "Required property 'instance_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def auto_scaling_policy(
            self,
        ) -> typing.Optional["EmrCreateCluster.AutoScalingPolicyProperty"]:
            '''An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.

            :default: - None
            '''
            result = self._values.get("auto_scaling_policy")
            return typing.cast(typing.Optional["EmrCreateCluster.AutoScalingPolicyProperty"], result)

        @builtins.property
        def bid_price(self) -> typing.Optional[builtins.str]:
            '''The bid price for each EC2 Spot instance type as defined by InstanceType.

            Expressed in USD.

            :default: - None
            '''
            result = self._values.get("bid_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]]:
            '''The list of configurations supplied for an EMR cluster instance group.

            :default: - None
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]], result)

        @builtins.property
        def ebs_configuration(
            self,
        ) -> typing.Optional["EmrCreateCluster.EbsConfigurationProperty"]:
            '''EBS configurations that will be attached to each EC2 instance in the instance group.

            :default: - None
            '''
            result = self._values.get("ebs_configuration")
            return typing.cast(typing.Optional["EmrCreateCluster.EbsConfigurationProperty"], result)

        @builtins.property
        def market(self) -> typing.Optional["EmrCreateCluster.InstanceMarket"]:
            '''Market type of the EC2 instances used to create a cluster node.

            :default: - EMR selected default
            '''
            result = self._values.get("market")
            return typing.cast(typing.Optional["EmrCreateCluster.InstanceMarket"], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''Friendly name given to the instance group.

            :default: - None
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceGroupConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstanceMarket"
    )
    class InstanceMarket(enum.Enum):
        '''EC2 Instance Market.'''

        ON_DEMAND = "ON_DEMAND"
        '''On Demand Instance.'''
        SPOT = "SPOT"
        '''Spot Instance.'''

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstanceRoleType"
    )
    class InstanceRoleType(enum.Enum):
        '''Instance Role Types.'''

        MASTER = "MASTER"
        '''Master Node.'''
        CORE = "CORE"
        '''Core Node.'''
        TASK = "TASK"
        '''Task Node.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstanceTypeConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_type": "instanceType",
            "bid_price": "bidPrice",
            "bid_price_as_percentage_of_on_demand_price": "bidPriceAsPercentageOfOnDemandPrice",
            "configurations": "configurations",
            "ebs_configuration": "ebsConfiguration",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class InstanceTypeConfigProperty:
        def __init__(
            self,
            *,
            instance_type: builtins.str,
            bid_price: typing.Optional[builtins.str] = None,
            bid_price_as_percentage_of_on_demand_price: typing.Optional[jsii.Number] = None,
            configurations: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            ebs_configuration: typing.Optional[typing.Union["EmrCreateCluster.EbsConfigurationProperty", typing.Dict[builtins.str, typing.Any]]] = None,
            weighted_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An instance type configuration for each instance type in an instance fleet, which determines the EC2 instances Amazon EMR attempts to provision to fulfill On-Demand and Spot target capacities.

            :param instance_type: An EC2 instance type.
            :param bid_price: The bid price for each EC2 Spot instance type as defined by InstanceType. Expressed in USD. Default: - None
            :param bid_price_as_percentage_of_on_demand_price: The bid price, as a percentage of On-Demand price. Default: - None
            :param configurations: A configuration classification that applies when provisioning cluster instances, which can include configurations for applications and software that run on the cluster. Default: - None
            :param ebs_configuration: The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by InstanceType. Default: - None
            :param weighted_capacity: The number of units that a provisioned instance of this type provides toward fulfilling the target capacities defined in the InstanceFleetConfig. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceTypeConfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                # size: cdk.Size
                
                instance_type_config_property = stepfunctions_tasks.EmrCreateCluster.InstanceTypeConfigProperty(
                    instance_type="instanceType",
                
                    # the properties below are optional
                    bid_price="bidPrice",
                    bid_price_as_percentage_of_on_demand_price=123,
                    configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                        classification="classification",
                        configurations=[configuration_property_],
                        properties={
                            "properties_key": "properties"
                        }
                    )],
                    ebs_configuration=stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                        ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                            volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                                volume_size=size,
                                volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                                # the properties below are optional
                                iops=123
                            ),
                
                            # the properties below are optional
                            volumes_per_instance=123
                        )],
                        ebs_optimized=False
                    ),
                    weighted_capacity=123
                )
            '''
            if isinstance(ebs_configuration, dict):
                ebs_configuration = EmrCreateCluster.EbsConfigurationProperty(**ebs_configuration)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1a0b989143eb079fc2e825545709ff3d7f6dc6d67aa065d6c1eadf3ee7a0725c)
                check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
                check_type(argname="argument bid_price", value=bid_price, expected_type=type_hints["bid_price"])
                check_type(argname="argument bid_price_as_percentage_of_on_demand_price", value=bid_price_as_percentage_of_on_demand_price, expected_type=type_hints["bid_price_as_percentage_of_on_demand_price"])
                check_type(argname="argument configurations", value=configurations, expected_type=type_hints["configurations"])
                check_type(argname="argument ebs_configuration", value=ebs_configuration, expected_type=type_hints["ebs_configuration"])
                check_type(argname="argument weighted_capacity", value=weighted_capacity, expected_type=type_hints["weighted_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "instance_type": instance_type,
            }
            if bid_price is not None:
                self._values["bid_price"] = bid_price
            if bid_price_as_percentage_of_on_demand_price is not None:
                self._values["bid_price_as_percentage_of_on_demand_price"] = bid_price_as_percentage_of_on_demand_price
            if configurations is not None:
                self._values["configurations"] = configurations
            if ebs_configuration is not None:
                self._values["ebs_configuration"] = ebs_configuration
            if weighted_capacity is not None:
                self._values["weighted_capacity"] = weighted_capacity

        @builtins.property
        def instance_type(self) -> builtins.str:
            '''An EC2 instance type.'''
            result = self._values.get("instance_type")
            assert result is not None, "Required property 'instance_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def bid_price(self) -> typing.Optional[builtins.str]:
            '''The bid price for each EC2 Spot instance type as defined by InstanceType.

            Expressed in USD.

            :default: - None
            '''
            result = self._values.get("bid_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def bid_price_as_percentage_of_on_demand_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''The bid price, as a percentage of On-Demand price.

            :default: - None
            '''
            result = self._values.get("bid_price_as_percentage_of_on_demand_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]]:
            '''A configuration classification that applies when provisioning cluster instances, which can include configurations for applications and software that run on the cluster.

            :default: - None
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]], result)

        @builtins.property
        def ebs_configuration(
            self,
        ) -> typing.Optional["EmrCreateCluster.EbsConfigurationProperty"]:
            '''The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by InstanceType.

            :default: - None
            '''
            result = self._values.get("ebs_configuration")
            return typing.cast(typing.Optional["EmrCreateCluster.EbsConfigurationProperty"], result)

        @builtins.property
        def weighted_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of units that a provisioned instance of this type provides toward fulfilling the target capacities defined in the InstanceFleetConfig.

            :default: - None
            '''
            result = self._values.get("weighted_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceTypeConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "additional_master_security_groups": "additionalMasterSecurityGroups",
            "additional_slave_security_groups": "additionalSlaveSecurityGroups",
            "ec2_key_name": "ec2KeyName",
            "ec2_subnet_id": "ec2SubnetId",
            "ec2_subnet_ids": "ec2SubnetIds",
            "emr_managed_master_security_group": "emrManagedMasterSecurityGroup",
            "emr_managed_slave_security_group": "emrManagedSlaveSecurityGroup",
            "hadoop_version": "hadoopVersion",
            "instance_count": "instanceCount",
            "instance_fleets": "instanceFleets",
            "instance_groups": "instanceGroups",
            "master_instance_type": "masterInstanceType",
            "placement": "placement",
            "service_access_security_group": "serviceAccessSecurityGroup",
            "slave_instance_type": "slaveInstanceType",
            "termination_protected": "terminationProtected",
        },
    )
    class InstancesConfigProperty:
        def __init__(
            self,
            *,
            additional_master_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            additional_slave_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            ec2_key_name: typing.Optional[builtins.str] = None,
            ec2_subnet_id: typing.Optional[builtins.str] = None,
            ec2_subnet_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
            emr_managed_master_security_group: typing.Optional[builtins.str] = None,
            emr_managed_slave_security_group: typing.Optional[builtins.str] = None,
            hadoop_version: typing.Optional[builtins.str] = None,
            instance_count: typing.Optional[jsii.Number] = None,
            instance_fleets: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.InstanceFleetConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            instance_groups: typing.Optional[typing.Sequence[typing.Union["EmrCreateCluster.InstanceGroupConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            master_instance_type: typing.Optional[builtins.str] = None,
            placement: typing.Optional[typing.Union["EmrCreateCluster.PlacementTypeProperty", typing.Dict[builtins.str, typing.Any]]] = None,
            service_access_security_group: typing.Optional[builtins.str] = None,
            slave_instance_type: typing.Optional[builtins.str] = None,
            termination_protected: typing.Optional[builtins.bool] = None,
        ) -> None:
            '''A specification of the number and type of Amazon EC2 instances.

            See the RunJobFlow API for complete documentation on input parameters

            :param additional_master_security_groups: A list of additional Amazon EC2 security group IDs for the master node. Default: - None
            :param additional_slave_security_groups: A list of additional Amazon EC2 security group IDs for the core and task nodes. Default: - None
            :param ec2_key_name: The name of the EC2 key pair that can be used to ssh to the master node as the user called "hadoop.". Default: - None
            :param ec2_subnet_id: Applies to clusters that use the uniform instance group configuration. To launch the cluster in Amazon Virtual Private Cloud (Amazon VPC), set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch. Default: EMR selected default
            :param ec2_subnet_ids: Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are specified, Amazon EMR evaluates them and launches instances in the optimal subnet. Default: EMR selected default
            :param emr_managed_master_security_group: The identifier of the Amazon EC2 security group for the master node. Default: - None
            :param emr_managed_slave_security_group: The identifier of the Amazon EC2 security group for the core and task nodes. Default: - None
            :param hadoop_version: Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster. Default: - 0.18 if the AmiVersion parameter is not set. If AmiVersion is set, the version of Hadoop for that AMI version is used.
            :param instance_count: The number of EC2 instances in the cluster. Default: 0
            :param instance_fleets: Describes the EC2 instances and instance configurations for clusters that use the instance fleet configuration. The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions. Default: - None
            :param instance_groups: Configuration for the instance groups in a cluster. Default: - None
            :param master_instance_type: The EC2 instance type of the master node. Default: - None
            :param placement: The Availability Zone in which the cluster runs. Default: - EMR selected default
            :param service_access_security_group: The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private subnets. Default: - None
            :param slave_instance_type: The EC2 instance type of the core and task nodes. Default: - None
            :param termination_protected: Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API call, user intervention, or in the event of a job-flow error. Default: false

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_JobFlowInstancesConfig.html
            :exampleMetadata: infused

            Example::

                cluster_role = iam.Role(self, "ClusterRole",
                    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
                )
                
                service_role = iam.Role(self, "ServiceRole",
                    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
                )
                
                auto_scaling_role = iam.Role(self, "AutoScalingRole",
                    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
                )
                
                auto_scaling_role.assume_role_policy.add_statements(
                    iam.PolicyStatement(
                        effect=iam.Effect.ALLOW,
                        principals=[
                            iam.ServicePrincipal("application-autoscaling.amazonaws.com")
                        ],
                        actions=["sts:AssumeRole"
                        ]
                    ))
                
                tasks.EmrCreateCluster(self, "Create Cluster",
                    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
                    cluster_role=cluster_role,
                    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
                    service_role=service_role,
                    auto_scaling_role=auto_scaling_role
                )
            '''
            if isinstance(placement, dict):
                placement = EmrCreateCluster.PlacementTypeProperty(**placement)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3517676b54e4658a197a1cfe143f00684b6c674a2f98f787be2714a2f8f058d1)
                check_type(argname="argument additional_master_security_groups", value=additional_master_security_groups, expected_type=type_hints["additional_master_security_groups"])
                check_type(argname="argument additional_slave_security_groups", value=additional_slave_security_groups, expected_type=type_hints["additional_slave_security_groups"])
                check_type(argname="argument ec2_key_name", value=ec2_key_name, expected_type=type_hints["ec2_key_name"])
                check_type(argname="argument ec2_subnet_id", value=ec2_subnet_id, expected_type=type_hints["ec2_subnet_id"])
                check_type(argname="argument ec2_subnet_ids", value=ec2_subnet_ids, expected_type=type_hints["ec2_subnet_ids"])
                check_type(argname="argument emr_managed_master_security_group", value=emr_managed_master_security_group, expected_type=type_hints["emr_managed_master_security_group"])
                check_type(argname="argument emr_managed_slave_security_group", value=emr_managed_slave_security_group, expected_type=type_hints["emr_managed_slave_security_group"])
                check_type(argname="argument hadoop_version", value=hadoop_version, expected_type=type_hints["hadoop_version"])
                check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
                check_type(argname="argument instance_fleets", value=instance_fleets, expected_type=type_hints["instance_fleets"])
                check_type(argname="argument instance_groups", value=instance_groups, expected_type=type_hints["instance_groups"])
                check_type(argname="argument master_instance_type", value=master_instance_type, expected_type=type_hints["master_instance_type"])
                check_type(argname="argument placement", value=placement, expected_type=type_hints["placement"])
                check_type(argname="argument service_access_security_group", value=service_access_security_group, expected_type=type_hints["service_access_security_group"])
                check_type(argname="argument slave_instance_type", value=slave_instance_type, expected_type=type_hints["slave_instance_type"])
                check_type(argname="argument termination_protected", value=termination_protected, expected_type=type_hints["termination_protected"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if additional_master_security_groups is not None:
                self._values["additional_master_security_groups"] = additional_master_security_groups
            if additional_slave_security_groups is not None:
                self._values["additional_slave_security_groups"] = additional_slave_security_groups
            if ec2_key_name is not None:
                self._values["ec2_key_name"] = ec2_key_name
            if ec2_subnet_id is not None:
                self._values["ec2_subnet_id"] = ec2_subnet_id
            if ec2_subnet_ids is not None:
                self._values["ec2_subnet_ids"] = ec2_subnet_ids
            if emr_managed_master_security_group is not None:
                self._values["emr_managed_master_security_group"] = emr_managed_master_security_group
            if emr_managed_slave_security_group is not None:
                self._values["emr_managed_slave_security_group"] = emr_managed_slave_security_group
            if hadoop_version is not None:
                self._values["hadoop_version"] = hadoop_version
            if instance_count is not None:
                self._values["instance_count"] = instance_count
            if instance_fleets is not None:
                self._values["instance_fleets"] = instance_fleets
            if instance_groups is not None:
                self._values["instance_groups"] = instance_groups
            if master_instance_type is not None:
                self._values["master_instance_type"] = master_instance_type
            if placement is not None:
                self._values["placement"] = placement
            if service_access_security_group is not None:
                self._values["service_access_security_group"] = service_access_security_group
            if slave_instance_type is not None:
                self._values["slave_instance_type"] = slave_instance_type
            if termination_protected is not None:
                self._values["termination_protected"] = termination_protected

        @builtins.property
        def additional_master_security_groups(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of additional Amazon EC2 security group IDs for the master node.

            :default: - None
            '''
            result = self._values.get("additional_master_security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def additional_slave_security_groups(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of additional Amazon EC2 security group IDs for the core and task nodes.

            :default: - None
            '''
            result = self._values.get("additional_slave_security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def ec2_key_name(self) -> typing.Optional[builtins.str]:
            '''The name of the EC2 key pair that can be used to ssh to the master node as the user called "hadoop.".

            :default: - None
            '''
            result = self._values.get("ec2_key_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_subnet_id(self) -> typing.Optional[builtins.str]:
            '''Applies to clusters that use the uniform instance group configuration.

            To launch the cluster in Amazon Virtual Private Cloud (Amazon VPC),
            set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch.

            :default: EMR selected default
            '''
            result = self._values.get("ec2_subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_subnet_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Applies to clusters that use the instance fleet configuration.

            When multiple EC2 subnet IDs are specified, Amazon EMR evaluates them and
            launches instances in the optimal subnet.

            :default: EMR selected default
            '''
            result = self._values.get("ec2_subnet_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def emr_managed_master_security_group(self) -> typing.Optional[builtins.str]:
            '''The identifier of the Amazon EC2 security group for the master node.

            :default: - None
            '''
            result = self._values.get("emr_managed_master_security_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def emr_managed_slave_security_group(self) -> typing.Optional[builtins.str]:
            '''The identifier of the Amazon EC2 security group for the core and task nodes.

            :default: - None
            '''
            result = self._values.get("emr_managed_slave_security_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def hadoop_version(self) -> typing.Optional[builtins.str]:
            '''Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster.

            :default: - 0.18 if the AmiVersion parameter is not set. If AmiVersion is set, the version of Hadoop for that AMI version is used.
            '''
            result = self._values.get("hadoop_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_count(self) -> typing.Optional[jsii.Number]:
            '''The number of EC2 instances in the cluster.

            :default: 0
            '''
            result = self._values.get("instance_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def instance_fleets(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.InstanceFleetConfigProperty"]]:
            '''Describes the EC2 instances and instance configurations for clusters that use the instance fleet configuration.

            The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions.

            :default: - None
            '''
            result = self._values.get("instance_fleets")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.InstanceFleetConfigProperty"]], result)

        @builtins.property
        def instance_groups(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.InstanceGroupConfigProperty"]]:
            '''Configuration for the instance groups in a cluster.

            :default: - None
            '''
            result = self._values.get("instance_groups")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.InstanceGroupConfigProperty"]], result)

        @builtins.property
        def master_instance_type(self) -> typing.Optional[builtins.str]:
            '''The EC2 instance type of the master node.

            :default: - None
            '''
            result = self._values.get("master_instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def placement(
            self,
        ) -> typing.Optional["EmrCreateCluster.PlacementTypeProperty"]:
            '''The Availability Zone in which the cluster runs.

            :default: - EMR selected default
            '''
            result = self._values.get("placement")
            return typing.cast(typing.Optional["EmrCreateCluster.PlacementTypeProperty"], result)

        @builtins.property
        def service_access_security_group(self) -> typing.Optional[builtins.str]:
            '''The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private subnets.

            :default: - None
            '''
            result = self._values.get("service_access_security_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def slave_instance_type(self) -> typing.Optional[builtins.str]:
            '''The EC2 instance type of the core and task nodes.

            :default: - None
            '''
            result = self._values.get("slave_instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def termination_protected(self) -> typing.Optional[builtins.bool]:
            '''Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API call, user intervention, or in the event of a job-flow error.

            :default: false
            '''
            result = self._values.get("termination_protected")
            return typing.cast(typing.Optional[builtins.bool], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstancesConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.KerberosAttributesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "realm": "realm",
            "ad_domain_join_password": "adDomainJoinPassword",
            "ad_domain_join_user": "adDomainJoinUser",
            "cross_realm_trust_principal_password": "crossRealmTrustPrincipalPassword",
            "kdc_admin_password": "kdcAdminPassword",
        },
    )
    class KerberosAttributesProperty:
        def __init__(
            self,
            *,
            realm: builtins.str,
            ad_domain_join_password: typing.Optional[builtins.str] = None,
            ad_domain_join_user: typing.Optional[builtins.str] = None,
            cross_realm_trust_principal_password: typing.Optional[builtins.str] = None,
            kdc_admin_password: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration.

            See the RunJobFlow API for complete documentation on input parameters

            :param realm: The name of the Kerberos realm to which all nodes in a cluster belong. For example, EC2.INTERNAL.
            :param ad_domain_join_password: The Active Directory password for ADDomainJoinUser. Default: No adDomainJoinPassword
            :param ad_domain_join_user: Required only when establishing a cross-realm trust with an Active Directory domain. A user with sufficient privileges to join resources to the domain. Default: No adDomainJoinUser
            :param cross_realm_trust_principal_password: Required only when establishing a cross-realm trust with a KDC in a different realm. The cross-realm principal password, which must be identical across realms. Default: No crossRealmTrustPrincipalPassword
            :param kdc_admin_password: The password used within the cluster for the kadmin service on the cluster-dedicated KDC, which maintains Kerberos principals, password policies, and keytabs for the cluster. Default: No kdcAdminPassword

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_KerberosAttributes.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                kerberos_attributes_property = stepfunctions_tasks.EmrCreateCluster.KerberosAttributesProperty(
                    realm="realm",
                
                    # the properties below are optional
                    ad_domain_join_password="adDomainJoinPassword",
                    ad_domain_join_user="adDomainJoinUser",
                    cross_realm_trust_principal_password="crossRealmTrustPrincipalPassword",
                    kdc_admin_password="kdcAdminPassword"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__55fd4ff6bee4f87932c9408a964df71df75a5355df79438f571de69d16e6ab8d)
                check_type(argname="argument realm", value=realm, expected_type=type_hints["realm"])
                check_type(argname="argument ad_domain_join_password", value=ad_domain_join_password, expected_type=type_hints["ad_domain_join_password"])
                check_type(argname="argument ad_domain_join_user", value=ad_domain_join_user, expected_type=type_hints["ad_domain_join_user"])
                check_type(argname="argument cross_realm_trust_principal_password", value=cross_realm_trust_principal_password, expected_type=type_hints["cross_realm_trust_principal_password"])
                check_type(argname="argument kdc_admin_password", value=kdc_admin_password, expected_type=type_hints["kdc_admin_password"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "realm": realm,
            }
            if ad_domain_join_password is not None:
                self._values["ad_domain_join_password"] = ad_domain_join_password
            if ad_domain_join_user is not None:
                self._values["ad_domain_join_user"] = ad_domain_join_user
            if cross_realm_trust_principal_password is not None:
                self._values["cross_realm_trust_principal_password"] = cross_realm_trust_principal_password
            if kdc_admin_password is not None:
                self._values["kdc_admin_password"] = kdc_admin_password

        @builtins.property
        def realm(self) -> builtins.str:
            '''The name of the Kerberos realm to which all nodes in a cluster belong.

            For example, EC2.INTERNAL.
            '''
            result = self._values.get("realm")
            assert result is not None, "Required property 'realm' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ad_domain_join_password(self) -> typing.Optional[builtins.str]:
            '''The Active Directory password for ADDomainJoinUser.

            :default: No adDomainJoinPassword
            '''
            result = self._values.get("ad_domain_join_password")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ad_domain_join_user(self) -> typing.Optional[builtins.str]:
            '''Required only when establishing a cross-realm trust with an Active Directory domain.

            A user with sufficient privileges to join
            resources to the domain.

            :default: No adDomainJoinUser
            '''
            result = self._values.get("ad_domain_join_user")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cross_realm_trust_principal_password(self) -> typing.Optional[builtins.str]:
            '''Required only when establishing a cross-realm trust with a KDC in a different realm.

            The cross-realm principal password, which
            must be identical across realms.

            :default: No crossRealmTrustPrincipalPassword
            '''
            result = self._values.get("cross_realm_trust_principal_password")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def kdc_admin_password(self) -> typing.Optional[builtins.str]:
            '''The password used within the cluster for the kadmin service on the cluster-dedicated KDC, which maintains Kerberos principals, password policies, and keytabs for the cluster.

            :default: No kdcAdminPassword
            '''
            result = self._values.get("kdc_admin_password")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KerberosAttributesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class MetricDimensionProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''A CloudWatch dimension, which is specified using a Key (known as a Name in CloudWatch), Value pair.

            By default, Amazon EMR uses
            one dimension whose Key is JobFlowID and Value is a variable representing the cluster ID, which is ${emr.clusterId}. This enables
            the rule to bootstrap when the cluster ID becomes available

            :param key: The dimension name.
            :param value: The dimension value.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_MetricDimension.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                metric_dimension_property = stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b08b745951eed0d937bba525b556337b2cbf7ac04b7cf244aa893e39122e1abc)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The dimension name.'''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The dimension value.'''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricDimensionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.PlacementTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "availability_zones": "availabilityZones",
        },
    )
    class PlacementTypeProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The Amazon EC2 Availability Zone configuration of the cluster (job flow).

            :param availability_zone: The Amazon EC2 Availability Zone for the cluster. AvailabilityZone is used for uniform instance groups, while AvailabilityZones (plural) is used for instance fleets. Default: - EMR selected default
            :param availability_zones: When multiple Availability Zones are specified, Amazon EMR evaluates them and launches instances in the optimal Availability Zone. AvailabilityZones is used for instance fleets, while AvailabilityZone (singular) is used for uniform instance groups. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_PlacementType.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                placement_type_property = stepfunctions_tasks.EmrCreateCluster.PlacementTypeProperty(
                    availability_zone="availabilityZone",
                    availability_zones=["availabilityZones"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__358af855c95eafd968c1ce2be342c56b08c58e04170ddc323a344e96f471ba2d)
                check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
                check_type(argname="argument availability_zones", value=availability_zones, expected_type=type_hints["availability_zones"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if availability_zones is not None:
                self._values["availability_zones"] = availability_zones

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Amazon EC2 Availability Zone for the cluster.

            AvailabilityZone is used for uniform instance groups, while AvailabilityZones
            (plural) is used for instance fleets.

            :default: - EMR selected default
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
            '''When multiple Availability Zones are specified, Amazon EMR evaluates them and launches instances in the optimal Availability Zone.

            AvailabilityZones is used for instance fleets, while AvailabilityZone (singular) is used for uniform instance groups.

            :default: - EMR selected default
            '''
            result = self._values.get("availability_zones")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "simple_scaling_policy_configuration": "simpleScalingPolicyConfiguration",
            "market": "market",
        },
    )
    class ScalingActionProperty:
        def __init__(
            self,
            *,
            simple_scaling_policy_configuration: typing.Union["EmrCreateCluster.SimpleScalingPolicyConfigurationProperty", typing.Dict[builtins.str, typing.Any]],
            market: typing.Optional["EmrCreateCluster.InstanceMarket"] = None,
        ) -> None:
            '''The type of adjustment the automatic scaling activity makes when triggered, and the periodicity of the adjustment.

            And an automatic scaling configuration, which describes how the policy adds or removes instances, the cooldown period,
            and the number of EC2 instances that will be added each time the CloudWatch metric alarm condition is satisfied.

            :param simple_scaling_policy_configuration: The type of adjustment the automatic scaling activity makes when triggered, and the periodicity of the adjustment.
            :param market: Not available for instance groups. Instance groups use the market type specified for the group. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingAction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                scaling_action_property = stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                    simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                        scaling_adjustment=123,
                
                        # the properties below are optional
                        adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                        cool_down=123
                    ),
                
                    # the properties below are optional
                    market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                )
            '''
            if isinstance(simple_scaling_policy_configuration, dict):
                simple_scaling_policy_configuration = EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(**simple_scaling_policy_configuration)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5639e6b6057a46c714d477b85c15f8fbcd4d84dc88d1718c4557ee8ce0a6bead)
                check_type(argname="argument simple_scaling_policy_configuration", value=simple_scaling_policy_configuration, expected_type=type_hints["simple_scaling_policy_configuration"])
                check_type(argname="argument market", value=market, expected_type=type_hints["market"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "simple_scaling_policy_configuration": simple_scaling_policy_configuration,
            }
            if market is not None:
                self._values["market"] = market

        @builtins.property
        def simple_scaling_policy_configuration(
            self,
        ) -> "EmrCreateCluster.SimpleScalingPolicyConfigurationProperty":
            '''The type of adjustment the automatic scaling activity makes when triggered, and the periodicity of the adjustment.'''
            result = self._values.get("simple_scaling_policy_configuration")
            assert result is not None, "Required property 'simple_scaling_policy_configuration' is missing"
            return typing.cast("EmrCreateCluster.SimpleScalingPolicyConfigurationProperty", result)

        @builtins.property
        def market(self) -> typing.Optional["EmrCreateCluster.InstanceMarket"]:
            '''Not available for instance groups.

            Instance groups use the market type specified for the group.

            :default: - EMR selected default
            '''
            result = self._values.get("market")
            return typing.cast(typing.Optional["EmrCreateCluster.InstanceMarket"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType"
    )
    class ScalingAdjustmentType(enum.Enum):
        '''AutoScaling Adjustment Type.'''

        CHANGE_IN_CAPACITY = "CHANGE_IN_CAPACITY"
        '''CHANGE_IN_CAPACITY.'''
        PERCENT_CHANGE_IN_CAPACITY = "PERCENT_CHANGE_IN_CAPACITY"
        '''PERCENT_CHANGE_IN_CAPACITY.'''
        EXACT_CAPACITY = "EXACT_CAPACITY"
        '''EXACT_CAPACITY.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty",
        jsii_struct_bases=[],
        name_mapping={"max_capacity": "maxCapacity", "min_capacity": "minCapacity"},
    )
    class ScalingConstraintsProperty:
        def __init__(
            self,
            *,
            max_capacity: jsii.Number,
            min_capacity: jsii.Number,
        ) -> None:
            '''The upper and lower EC2 instance limits for an automatic scaling policy.

            Automatic scaling activities triggered by automatic scaling
            rules will not cause an instance group to grow above or below these limits.

            :param max_capacity: The upper boundary of EC2 instances in an instance group beyond which scaling activities are not allowed to grow. Scale-out activities will not add instances beyond this boundary.
            :param min_capacity: The lower boundary of EC2 instances in an instance group below which scaling activities are not allowed to shrink. Scale-in activities will not terminate instances below this boundary.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingConstraints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                scaling_constraints_property = stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty(
                    max_capacity=123,
                    min_capacity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a4f9e6c7c59b3b9439bf2e928e07dc6424fe05344634726850be046a24e3f467)
                check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
                check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_capacity": max_capacity,
                "min_capacity": min_capacity,
            }

        @builtins.property
        def max_capacity(self) -> jsii.Number:
            '''The upper boundary of EC2 instances in an instance group beyond which scaling activities are not allowed to grow.

            Scale-out
            activities will not add instances beyond this boundary.
            '''
            result = self._values.get("max_capacity")
            assert result is not None, "Required property 'max_capacity' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def min_capacity(self) -> jsii.Number:
            '''The lower boundary of EC2 instances in an instance group below which scaling activities are not allowed to shrink.

            Scale-in
            activities will not terminate instances below this boundary.
            '''
            result = self._values.get("min_capacity")
            assert result is not None, "Required property 'min_capacity' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConstraintsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "name": "name",
            "trigger": "trigger",
            "description": "description",
        },
    )
    class ScalingRuleProperty:
        def __init__(
            self,
            *,
            action: typing.Union["EmrCreateCluster.ScalingActionProperty", typing.Dict[builtins.str, typing.Any]],
            name: builtins.str,
            trigger: typing.Union["EmrCreateCluster.ScalingTriggerProperty", typing.Dict[builtins.str, typing.Any]],
            description: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A scale-in or scale-out rule that defines scaling activity, including the CloudWatch metric alarm that triggers activity, how EC2 instances are added or removed, and the periodicity of adjustments.

            :param action: The conditions that trigger an automatic scaling activity.
            :param name: The name used to identify an automatic scaling rule. Rule names must be unique within a scaling policy.
            :param trigger: The CloudWatch alarm definition that determines when automatic scaling activity is triggered.
            :param description: A friendly, more verbose description of the automatic scaling rule. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingRule.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                scaling_rule_property = stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty(
                    action=stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                        simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                            scaling_adjustment=123,
                
                            # the properties below are optional
                            adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                            cool_down=123
                        ),
                
                        # the properties below are optional
                        market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                    ),
                    name="name",
                    trigger=stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                        cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                            comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                            metric_name="metricName",
                            period=cdk.Duration.minutes(30),
                
                            # the properties below are optional
                            dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                                key="key",
                                value="value"
                            )],
                            evaluation_periods=123,
                            namespace="namespace",
                            statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                            threshold=123,
                            unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                        )
                    ),
                
                    # the properties below are optional
                    description="description"
                )
            '''
            if isinstance(action, dict):
                action = EmrCreateCluster.ScalingActionProperty(**action)
            if isinstance(trigger, dict):
                trigger = EmrCreateCluster.ScalingTriggerProperty(**trigger)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0da905669b09231159da7c1e071fd64c7b86d17c68313da9df0113e676fc523a)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument trigger", value=trigger, expected_type=type_hints["trigger"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "name": name,
                "trigger": trigger,
            }
            if description is not None:
                self._values["description"] = description

        @builtins.property
        def action(self) -> "EmrCreateCluster.ScalingActionProperty":
            '''The conditions that trigger an automatic scaling activity.'''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast("EmrCreateCluster.ScalingActionProperty", result)

        @builtins.property
        def name(self) -> builtins.str:
            '''The name used to identify an automatic scaling rule.

            Rule names must be unique within a scaling policy.
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def trigger(self) -> "EmrCreateCluster.ScalingTriggerProperty":
            '''The CloudWatch alarm definition that determines when automatic scaling activity is triggered.'''
            result = self._values.get("trigger")
            assert result is not None, "Required property 'trigger' is missing"
            return typing.cast("EmrCreateCluster.ScalingTriggerProperty", result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A friendly, more verbose description of the automatic scaling rule.

            :default: - None
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingRuleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty",
        jsii_struct_bases=[],
        name_mapping={"cloud_watch_alarm_definition": "cloudWatchAlarmDefinition"},
    )
    class ScalingTriggerProperty:
        def __init__(
            self,
            *,
            cloud_watch_alarm_definition: typing.Union["EmrCreateCluster.CloudWatchAlarmDefinitionProperty", typing.Dict[builtins.str, typing.Any]],
        ) -> None:
            '''The conditions that trigger an automatic scaling activity and the definition of a CloudWatch metric alarm.

            When the defined alarm conditions are met along with other trigger parameters, scaling activity begins.

            :param cloud_watch_alarm_definition: The definition of a CloudWatch metric alarm. When the defined alarm conditions are met along with other trigger parameters, scaling activity begins.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingTrigger.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                scaling_trigger_property = stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                    cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                        comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                        metric_name="metricName",
                        period=cdk.Duration.minutes(30),
                
                        # the properties below are optional
                        dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                            key="key",
                            value="value"
                        )],
                        evaluation_periods=123,
                        namespace="namespace",
                        statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                        threshold=123,
                        unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                    )
                )
            '''
            if isinstance(cloud_watch_alarm_definition, dict):
                cloud_watch_alarm_definition = EmrCreateCluster.CloudWatchAlarmDefinitionProperty(**cloud_watch_alarm_definition)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a71e7ac46b26b98f557d877037da24ec5bcd0e09957a9c133b42362d07b63f25)
                check_type(argname="argument cloud_watch_alarm_definition", value=cloud_watch_alarm_definition, expected_type=type_hints["cloud_watch_alarm_definition"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cloud_watch_alarm_definition": cloud_watch_alarm_definition,
            }

        @builtins.property
        def cloud_watch_alarm_definition(
            self,
        ) -> "EmrCreateCluster.CloudWatchAlarmDefinitionProperty":
            '''The definition of a CloudWatch metric alarm.

            When the defined alarm conditions are met along with other trigger parameters,
            scaling activity begins.
            '''
            result = self._values.get("cloud_watch_alarm_definition")
            assert result is not None, "Required property 'cloud_watch_alarm_definition' is missing"
            return typing.cast("EmrCreateCluster.CloudWatchAlarmDefinitionProperty", result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingTriggerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.ScriptBootstrapActionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path", "args": "args"},
    )
    class ScriptBootstrapActionConfigProperty:
        def __init__(
            self,
            *,
            path: builtins.str,
            args: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Configuration of the script to run during a bootstrap action.

            :param path: Location of the script to run during a bootstrap action. Can be either a location in Amazon S3 or on a local file system.
            :param args: A list of command line arguments to pass to the bootstrap action script. Default: No args

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScriptBootstrapActionConfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                script_bootstrap_action_config_property = stepfunctions_tasks.EmrCreateCluster.ScriptBootstrapActionConfigProperty(
                    path="path",
                
                    # the properties below are optional
                    args=["args"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__da21a0de45bdd650726ace0a0a5f0f054f8bd6719d4455899f826bfa18226152)
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument args", value=args, expected_type=type_hints["args"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "path": path,
            }
            if args is not None:
                self._values["args"] = args

        @builtins.property
        def path(self) -> builtins.str:
            '''Location of the script to run during a bootstrap action.

            Can be either a location in Amazon S3 or on a local file system.
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def args(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of command line arguments to pass to the bootstrap action script.

            :default: No args
            '''
            result = self._values.get("args")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScriptBootstrapActionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "scaling_adjustment": "scalingAdjustment",
            "adjustment_type": "adjustmentType",
            "cool_down": "coolDown",
        },
    )
    class SimpleScalingPolicyConfigurationProperty:
        def __init__(
            self,
            *,
            scaling_adjustment: jsii.Number,
            adjustment_type: typing.Optional["EmrCreateCluster.ScalingAdjustmentType"] = None,
            cool_down: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An automatic scaling configuration, which describes how the policy adds or removes instances, the cooldown period, and the number of EC2 instances that will be added each time the CloudWatch metric alarm condition is satisfied.

            :param scaling_adjustment: The amount by which to scale in or scale out, based on the specified AdjustmentType. A positive value adds to the instance group's EC2 instance count while a negative number removes instances. If AdjustmentType is set to EXACT_CAPACITY, the number should only be a positive integer.
            :param adjustment_type: The way in which EC2 instances are added (if ScalingAdjustment is a positive number) or terminated (if ScalingAdjustment is a negative number) each time the scaling activity is triggered. Default: - None
            :param cool_down: The amount of time, in seconds, after a scaling activity completes before any further trigger-related scaling activities can start. Default: 0

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_SimpleScalingPolicyConfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                simple_scaling_policy_configuration_property = stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                    scaling_adjustment=123,
                
                    # the properties below are optional
                    adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                    cool_down=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cdaa7a075b4efdc50702fd6877f9b79e13b0eb9f1d360c4243fbf89fdd418506)
                check_type(argname="argument scaling_adjustment", value=scaling_adjustment, expected_type=type_hints["scaling_adjustment"])
                check_type(argname="argument adjustment_type", value=adjustment_type, expected_type=type_hints["adjustment_type"])
                check_type(argname="argument cool_down", value=cool_down, expected_type=type_hints["cool_down"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "scaling_adjustment": scaling_adjustment,
            }
            if adjustment_type is not None:
                self._values["adjustment_type"] = adjustment_type
            if cool_down is not None:
                self._values["cool_down"] = cool_down

        @builtins.property
        def scaling_adjustment(self) -> jsii.Number:
            '''The amount by which to scale in or scale out, based on the specified AdjustmentType.

            A positive value adds to the instance group's
            EC2 instance count while a negative number removes instances. If AdjustmentType is set to EXACT_CAPACITY, the number should only be
            a positive integer.
            '''
            result = self._values.get("scaling_adjustment")
            assert result is not None, "Required property 'scaling_adjustment' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def adjustment_type(
            self,
        ) -> typing.Optional["EmrCreateCluster.ScalingAdjustmentType"]:
            '''The way in which EC2 instances are added (if ScalingAdjustment is a positive number) or terminated (if ScalingAdjustment is a negative number) each time the scaling activity is triggered.

            :default: - None
            '''
            result = self._values.get("adjustment_type")
            return typing.cast(typing.Optional["EmrCreateCluster.ScalingAdjustmentType"], result)

        @builtins.property
        def cool_down(self) -> typing.Optional[jsii.Number]:
            '''The amount of time, in seconds, after a scaling activity completes before any further trigger-related scaling activities can start.

            :default: 0
            '''
            result = self._values.get("cool_down")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SimpleScalingPolicyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy"
    )
    class SpotAllocationStrategy(enum.Enum):
        '''Spot Allocation Strategies.

        Specifies the strategy to use in launching Spot Instance fleets. For example, "capacity-optimized" launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
        '''

        CAPACITY_OPTIMIZED = "CAPACITY_OPTIMIZED"
        '''Capacity-optimized, which launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "timeout_action": "timeoutAction",
            "timeout_duration_minutes": "timeoutDurationMinutes",
            "allocation_strategy": "allocationStrategy",
            "block_duration_minutes": "blockDurationMinutes",
        },
    )
    class SpotProvisioningSpecificationProperty:
        def __init__(
            self,
            *,
            timeout_action: "EmrCreateCluster.SpotTimeoutAction",
            timeout_duration_minutes: jsii.Number,
            allocation_strategy: typing.Optional["EmrCreateCluster.SpotAllocationStrategy"] = None,
            block_duration_minutes: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The launch specification for Spot instances in the instance fleet, which determines the defined duration and provisioning timeout behavior.

            :param timeout_action: The action to take when TargetSpotCapacity has not been fulfilled when the TimeoutDurationMinutes has expired.
            :param timeout_duration_minutes: The spot provisioning timeout period in minutes.
            :param allocation_strategy: Specifies the strategy to use in launching Spot Instance fleets. Default: - No allocation strategy, i.e. spot instance type will be chosen based on current price only
            :param block_duration_minutes: The defined duration for Spot instances (also known as Spot blocks) in minutes. Default: - No blockDurationMinutes

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                spot_provisioning_specification_property = stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                    timeout_action=stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction.SWITCH_TO_ON_DEMAND,
                    timeout_duration_minutes=123,
                
                    # the properties below are optional
                    allocation_strategy=stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                    block_duration_minutes=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dafa3d17787fdab36cd1f4f58ea87a372f8918f012ead69328993cd9b112eef0)
                check_type(argname="argument timeout_action", value=timeout_action, expected_type=type_hints["timeout_action"])
                check_type(argname="argument timeout_duration_minutes", value=timeout_duration_minutes, expected_type=type_hints["timeout_duration_minutes"])
                check_type(argname="argument allocation_strategy", value=allocation_strategy, expected_type=type_hints["allocation_strategy"])
                check_type(argname="argument block_duration_minutes", value=block_duration_minutes, expected_type=type_hints["block_duration_minutes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "timeout_action": timeout_action,
                "timeout_duration_minutes": timeout_duration_minutes,
            }
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if block_duration_minutes is not None:
                self._values["block_duration_minutes"] = block_duration_minutes

        @builtins.property
        def timeout_action(self) -> "EmrCreateCluster.SpotTimeoutAction":
            '''The action to take when TargetSpotCapacity has not been fulfilled when the TimeoutDurationMinutes has expired.'''
            result = self._values.get("timeout_action")
            assert result is not None, "Required property 'timeout_action' is missing"
            return typing.cast("EmrCreateCluster.SpotTimeoutAction", result)

        @builtins.property
        def timeout_duration_minutes(self) -> jsii.Number:
            '''The spot provisioning timeout period in minutes.'''
            result = self._values.get("timeout_duration_minutes")
            assert result is not None, "Required property 'timeout_duration_minutes' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def allocation_strategy(
            self,
        ) -> typing.Optional["EmrCreateCluster.SpotAllocationStrategy"]:
            '''Specifies the strategy to use in launching Spot Instance fleets.

            :default: - No allocation strategy, i.e. spot instance type will be chosen based on current price only
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional["EmrCreateCluster.SpotAllocationStrategy"], result)

        @builtins.property
        def block_duration_minutes(self) -> typing.Optional[jsii.Number]:
            '''The defined duration for Spot instances (also known as Spot blocks) in minutes.

            :default: - No blockDurationMinutes
            '''
            result = self._values.get("block_duration_minutes")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SpotProvisioningSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction"
    )
    class SpotTimeoutAction(enum.Enum):
        '''Spot Timeout Actions.'''

        SWITCH_TO_ON_DEMAND = "SWITCH_TO_ON_DEMAND"
        '''SWITCH_TO_ON_DEMAND.'''
        TERMINATE_CLUSTER = "TERMINATE_CLUSTER"
        '''TERMINATE_CLUSTER.'''

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
            "iops": "iops",
        },
    )
    class VolumeSpecificationProperty:
        def __init__(
            self,
            *,
            volume_size: _Size_7b441c34,
            volume_type: "EmrCreateCluster.EbsBlockDeviceVolumeType",
            iops: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''EBS volume specifications such as volume type, IOPS, and size (GiB) that will be requested for the EBS volume attached to an EC2 instance in the cluster.

            :param volume_size: The volume size. If the volume type is EBS-optimized, the minimum value is 10GiB. Maximum size is 1TiB
            :param volume_type: The volume type. Volume types supported are gp2, io1, standard.
            :param iops: The number of I/O operations per second (IOPS) that the volume supports. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_VolumeSpecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # size: cdk.Size
                
                volume_specification_property = stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                    volume_size=size,
                    volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                    # the properties below are optional
                    iops=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__27d7ac896986899baaa90e73872a210db22470deca535c19426cbbc58d2da449)
                check_type(argname="argument volume_size", value=volume_size, expected_type=type_hints["volume_size"])
                check_type(argname="argument volume_type", value=volume_type, expected_type=type_hints["volume_type"])
                check_type(argname="argument iops", value=iops, expected_type=type_hints["iops"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "volume_size": volume_size,
                "volume_type": volume_type,
            }
            if iops is not None:
                self._values["iops"] = iops

        @builtins.property
        def volume_size(self) -> _Size_7b441c34:
            '''The volume size.

            If the volume type is EBS-optimized, the minimum value is 10GiB.
            Maximum size is 1TiB
            '''
            result = self._values.get("volume_size")
            assert result is not None, "Required property 'volume_size' is missing"
            return typing.cast(_Size_7b441c34, result)

        @builtins.property
        def volume_type(self) -> "EmrCreateCluster.EbsBlockDeviceVolumeType":
            '''The volume type.

            Volume types supported are gp2, io1, standard.
            '''
            result = self._values.get("volume_type")
            assert result is not None, "Required property 'volume_type' is missing"
            return typing.cast("EmrCreateCluster.EbsBlockDeviceVolumeType", result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''The number of I/O operations per second (IOPS) that the volume supports.

            :default: - EMR selected default
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrCreateClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "instances": "instances",
        "name": "name",
        "additional_info": "additionalInfo",
        "applications": "applications",
        "auto_scaling_role": "autoScalingRole",
        "bootstrap_actions": "bootstrapActions",
        "cluster_role": "clusterRole",
        "configurations": "configurations",
        "custom_ami_id": "customAmiId",
        "ebs_root_volume_size": "ebsRootVolumeSize",
        "kerberos_attributes": "kerberosAttributes",
        "log_uri": "logUri",
        "release_label": "releaseLabel",
        "scale_down_behavior": "scaleDownBehavior",
        "security_configuration": "securityConfiguration",
        "service_role": "serviceRole",
        "step_concurrency_level": "stepConcurrencyLevel",
        "tags": "tags",
        "visible_to_all_users": "visibleToAllUsers",
    },
)
class EmrCreateClusterProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        instances: typing.Union[EmrCreateCluster.InstancesConfigProperty, typing.Dict[builtins.str, typing.Any]],
        name: builtins.str,
        additional_info: typing.Optional[builtins.str] = None,
        applications: typing.Optional[typing.Sequence[typing.Union[EmrCreateCluster.ApplicationConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        auto_scaling_role: typing.Optional[_IRole_235f5d8e] = None,
        bootstrap_actions: typing.Optional[typing.Sequence[typing.Union[EmrCreateCluster.BootstrapActionConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        cluster_role: typing.Optional[_IRole_235f5d8e] = None,
        configurations: typing.Optional[typing.Sequence[typing.Union[EmrCreateCluster.ConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        custom_ami_id: typing.Optional[builtins.str] = None,
        ebs_root_volume_size: typing.Optional[_Size_7b441c34] = None,
        kerberos_attributes: typing.Optional[typing.Union[EmrCreateCluster.KerberosAttributesProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        log_uri: typing.Optional[builtins.str] = None,
        release_label: typing.Optional[builtins.str] = None,
        scale_down_behavior: typing.Optional[EmrCreateCluster.EmrClusterScaleDownBehavior] = None,
        security_configuration: typing.Optional[builtins.str] = None,
        service_role: typing.Optional[_IRole_235f5d8e] = None,
        step_concurrency_level: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        visible_to_all_users: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for EmrCreateCluster.

        See the RunJobFlow API for complete documentation on input parameters

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param instances: A specification of the number and type of Amazon EC2 instances.
        :param name: The Name of the Cluster.
        :param additional_info: A JSON string for selecting additional features. Default: - None
        :param applications: A case-insensitive list of applications for Amazon EMR to install and configure when launching the cluster. Default: - EMR selected default
        :param auto_scaling_role: An IAM role for automatic scaling policies. Default: - A role will be created.
        :param bootstrap_actions: A list of bootstrap actions to run before Hadoop starts on the cluster nodes. Default: - None
        :param cluster_role: Also called instance profile and EC2 role. An IAM role for an EMR cluster. The EC2 instances of the cluster assume this role. This attribute has been renamed from jobFlowRole to clusterRole to align with other ERM/StepFunction integration parameters. Default: - - A Role will be created
        :param configurations: The list of configurations supplied for the EMR cluster you are creating. Default: - None
        :param custom_ami_id: The ID of a custom Amazon EBS-backed Linux AMI. Default: - None
        :param ebs_root_volume_size: The size of the EBS root device volume of the Linux AMI that is used for each EC2 instance. Default: - EMR selected default
        :param kerberos_attributes: Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration. Default: - None
        :param log_uri: The location in Amazon S3 to write the log files of the job flow. Default: - None
        :param release_label: The Amazon EMR release label, which determines the version of open-source application packages installed on the cluster. Default: - EMR selected default
        :param scale_down_behavior: Specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized. Default: - EMR selected default
        :param security_configuration: The name of a security configuration to apply to the cluster. Default: - None
        :param service_role: The IAM role that will be assumed by the Amazon EMR service to access AWS resources on your behalf. Default: - A role will be created that Amazon EMR service can assume.
        :param step_concurrency_level: Specifies the step concurrency level to allow multiple steps to run in parallel. Requires EMR release label 5.28.0 or above. Must be in range [1, 256]. Default: 1 - no step concurrency allowed
        :param tags: A list of tags to associate with a cluster and propagate to Amazon EC2 instances. Default: - None
        :param visible_to_all_users: A value of true indicates that all IAM users in the AWS account can perform cluster actions if they have the proper IAM policy permissions. Default: true

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html
        :exampleMetadata: infused

        Example::

            cluster_role = iam.Role(self, "ClusterRole",
                assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
            )
            
            service_role = iam.Role(self, "ServiceRole",
                assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
            )
            
            auto_scaling_role = iam.Role(self, "AutoScalingRole",
                assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
            )
            
            auto_scaling_role.assume_role_policy.add_statements(
                iam.PolicyStatement(
                    effect=iam.Effect.ALLOW,
                    principals=[
                        iam.ServicePrincipal("application-autoscaling.amazonaws.com")
                    ],
                    actions=["sts:AssumeRole"
                    ]
                ))
            
            tasks.EmrCreateCluster(self, "Create Cluster",
                instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
                cluster_role=cluster_role,
                name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
                service_role=service_role,
                auto_scaling_role=auto_scaling_role
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(instances, dict):
            instances = EmrCreateCluster.InstancesConfigProperty(**instances)
        if isinstance(kerberos_attributes, dict):
            kerberos_attributes = EmrCreateCluster.KerberosAttributesProperty(**kerberos_attributes)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47264d4efff2249070476b1432ca851665096bb89026c6d6564ff3c77327b86c)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument instances", value=instances, expected_type=type_hints["instances"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument additional_info", value=additional_info, expected_type=type_hints["additional_info"])
            check_type(argname="argument applications", value=applications, expected_type=type_hints["applications"])
            check_type(argname="argument auto_scaling_role", value=auto_scaling_role, expected_type=type_hints["auto_scaling_role"])
            check_type(argname="argument bootstrap_actions", value=bootstrap_actions, expected_type=type_hints["bootstrap_actions"])
            check_type(argname="argument cluster_role", value=cluster_role, expected_type=type_hints["cluster_role"])
            check_type(argname="argument configurations", value=configurations, expected_type=type_hints["configurations"])
            check_type(argname="argument custom_ami_id", value=custom_ami_id, expected_type=type_hints["custom_ami_id"])
            check_type(argname="argument ebs_root_volume_size", value=ebs_root_volume_size, expected_type=type_hints["ebs_root_volume_size"])
            check_type(argname="argument kerberos_attributes", value=kerberos_attributes, expected_type=type_hints["kerberos_attributes"])
            check_type(argname="argument log_uri", value=log_uri, expected_type=type_hints["log_uri"])
            check_type(argname="argument release_label", value=release_label, expected_type=type_hints["release_label"])
            check_type(argname="argument scale_down_behavior", value=scale_down_behavior, expected_type=type_hints["scale_down_behavior"])
            check_type(argname="argument security_configuration", value=security_configuration, expected_type=type_hints["security_configuration"])
            check_type(argname="argument service_role", value=service_role, expected_type=type_hints["service_role"])
            check_type(argname="argument step_concurrency_level", value=step_concurrency_level, expected_type=type_hints["step_concurrency_level"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument visible_to_all_users", value=visible_to_all_users, expected_type=type_hints["visible_to_all_users"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "instances": instances,
            "name": name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if applications is not None:
            self._values["applications"] = applications
        if auto_scaling_role is not None:
            self._values["auto_scaling_role"] = auto_scaling_role
        if bootstrap_actions is not None:
            self._values["bootstrap_actions"] = bootstrap_actions
        if cluster_role is not None:
            self._values["cluster_role"] = cluster_role
        if configurations is not None:
            self._values["configurations"] = configurations
        if custom_ami_id is not None:
            self._values["custom_ami_id"] = custom_ami_id
        if ebs_root_volume_size is not None:
            self._values["ebs_root_volume_size"] = ebs_root_volume_size
        if kerberos_attributes is not None:
            self._values["kerberos_attributes"] = kerberos_attributes
        if log_uri is not None:
            self._values["log_uri"] = log_uri
        if release_label is not None:
            self._values["release_label"] = release_label
        if scale_down_behavior is not None:
            self._values["scale_down_behavior"] = scale_down_behavior
        if security_configuration is not None:
            self._values["security_configuration"] = security_configuration
        if service_role is not None:
            self._values["service_role"] = service_role
        if step_concurrency_level is not None:
            self._values["step_concurrency_level"] = step_concurrency_level
        if tags is not None:
            self._values["tags"] = tags
        if visible_to_all_users is not None:
            self._values["visible_to_all_users"] = visible_to_all_users

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def instances(self) -> EmrCreateCluster.InstancesConfigProperty:
        '''A specification of the number and type of Amazon EC2 instances.'''
        result = self._values.get("instances")
        assert result is not None, "Required property 'instances' is missing"
        return typing.cast(EmrCreateCluster.InstancesConfigProperty, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The Name of the Cluster.'''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_info(self) -> typing.Optional[builtins.str]:
        '''A JSON string for selecting additional features.

        :default: - None
        '''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def applications(
        self,
    ) -> typing.Optional[typing.List[EmrCreateCluster.ApplicationConfigProperty]]:
        '''A case-insensitive list of applications for Amazon EMR to install and configure when launching the cluster.

        :default: - EMR selected default
        '''
        result = self._values.get("applications")
        return typing.cast(typing.Optional[typing.List[EmrCreateCluster.ApplicationConfigProperty]], result)

    @builtins.property
    def auto_scaling_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role for automatic scaling policies.

        :default: - A role will be created.
        '''
        result = self._values.get("auto_scaling_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def bootstrap_actions(
        self,
    ) -> typing.Optional[typing.List[EmrCreateCluster.BootstrapActionConfigProperty]]:
        '''A list of bootstrap actions to run before Hadoop starts on the cluster nodes.

        :default: - None
        '''
        result = self._values.get("bootstrap_actions")
        return typing.cast(typing.Optional[typing.List[EmrCreateCluster.BootstrapActionConfigProperty]], result)

    @builtins.property
    def cluster_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''Also called instance profile and EC2 role.

        An IAM role for an EMR cluster. The EC2 instances of the cluster assume this role.

        This attribute has been renamed from jobFlowRole to clusterRole to align with other ERM/StepFunction integration parameters.

        :default:

        -
        - A Role will be created
        '''
        result = self._values.get("cluster_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def configurations(
        self,
    ) -> typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]]:
        '''The list of configurations supplied for the EMR cluster you are creating.

        :default: - None
        '''
        result = self._values.get("configurations")
        return typing.cast(typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]], result)

    @builtins.property
    def custom_ami_id(self) -> typing.Optional[builtins.str]:
        '''The ID of a custom Amazon EBS-backed Linux AMI.

        :default: - None
        '''
        result = self._values.get("custom_ami_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ebs_root_volume_size(self) -> typing.Optional[_Size_7b441c34]:
        '''The size of the EBS root device volume of the Linux AMI that is used for each EC2 instance.

        :default: - EMR selected default
        '''
        result = self._values.get("ebs_root_volume_size")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def kerberos_attributes(
        self,
    ) -> typing.Optional[EmrCreateCluster.KerberosAttributesProperty]:
        '''Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration.

        :default: - None
        '''
        result = self._values.get("kerberos_attributes")
        return typing.cast(typing.Optional[EmrCreateCluster.KerberosAttributesProperty], result)

    @builtins.property
    def log_uri(self) -> typing.Optional[builtins.str]:
        '''The location in Amazon S3 to write the log files of the job flow.

        :default: - None
        '''
        result = self._values.get("log_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def release_label(self) -> typing.Optional[builtins.str]:
        '''The Amazon EMR release label, which determines the version of open-source application packages installed on the cluster.

        :default: - EMR selected default
        '''
        result = self._values.get("release_label")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale_down_behavior(
        self,
    ) -> typing.Optional[EmrCreateCluster.EmrClusterScaleDownBehavior]:
        '''Specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized.

        :default: - EMR selected default
        '''
        result = self._values.get("scale_down_behavior")
        return typing.cast(typing.Optional[EmrCreateCluster.EmrClusterScaleDownBehavior], result)

    @builtins.property
    def security_configuration(self) -> typing.Optional[builtins.str]:
        '''The name of a security configuration to apply to the cluster.

        :default: - None
        '''
        result = self._values.get("security_configuration")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role that will be assumed by the Amazon EMR service to access AWS resources on your behalf.

        :default: - A role will be created that Amazon EMR service can assume.
        '''
        result = self._values.get("service_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def step_concurrency_level(self) -> typing.Optional[jsii.Number]:
        '''Specifies the step concurrency level to allow multiple steps to run in parallel.

        Requires EMR release label 5.28.0 or above.
        Must be in range [1, 256].

        :default: 1 - no step concurrency allowed
        '''
        result = self._values.get("step_concurrency_level")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''A list of tags to associate with a cluster and propagate to Amazon EC2 instances.

        :default: - None
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def visible_to_all_users(self) -> typing.Optional[builtins.bool]:
        '''A value of true indicates that all IAM users in the AWS account can perform cluster actions if they have the proper IAM policy permissions.

        :default: true
        '''
        result = self._values.get("visible_to_all_users")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrCreateClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrModifyInstanceFleetByName(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceFleetByName",
):
    '''A Step Functions Task to to modify an InstanceFleet on an EMR Cluster.

    :exampleMetadata: infused

    Example::

        tasks.EmrModifyInstanceFleetByName(self, "Task",
            cluster_id="ClusterId",
            instance_fleet_name="InstanceFleetName",
            target_on_demand_capacity=2,
            target_spot_capacity=0
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        instance_fleet_name: builtins.str,
        target_on_demand_capacity: jsii.Number,
        target_spot_capacity: jsii.Number,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster_id: The ClusterId to update.
        :param instance_fleet_name: The InstanceFleetName to update.
        :param target_on_demand_capacity: The target capacity of On-Demand units for the instance fleet. Default: - None
        :param target_spot_capacity: The target capacity of Spot units for the instance fleet. Default: - None
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__44c29c883ea8c3f9dafe68d628e20bea269edf8d0742e0637ea23b88d0353665)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrModifyInstanceFleetByNameProps(
            cluster_id=cluster_id,
            instance_fleet_name=instance_fleet_name,
            target_on_demand_capacity=target_on_demand_capacity,
            target_spot_capacity=target_spot_capacity,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceFleetByNameProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "instance_fleet_name": "instanceFleetName",
        "target_on_demand_capacity": "targetOnDemandCapacity",
        "target_spot_capacity": "targetSpotCapacity",
    },
)
class EmrModifyInstanceFleetByNameProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster_id: builtins.str,
        instance_fleet_name: builtins.str,
        target_on_demand_capacity: jsii.Number,
        target_spot_capacity: jsii.Number,
    ) -> None:
        '''Properties for EmrModifyInstanceFleetByName.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster_id: The ClusterId to update.
        :param instance_fleet_name: The InstanceFleetName to update.
        :param target_on_demand_capacity: The target capacity of On-Demand units for the instance fleet. Default: - None
        :param target_spot_capacity: The target capacity of Spot units for the instance fleet. Default: - None

        :exampleMetadata: infused

        Example::

            tasks.EmrModifyInstanceFleetByName(self, "Task",
                cluster_id="ClusterId",
                instance_fleet_name="InstanceFleetName",
                target_on_demand_capacity=2,
                target_spot_capacity=0
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d43fbaef8b6dee665530d57b843a8c0b590ccbe7d76a5babea20fd55e6f66600)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster_id", value=cluster_id, expected_type=type_hints["cluster_id"])
            check_type(argname="argument instance_fleet_name", value=instance_fleet_name, expected_type=type_hints["instance_fleet_name"])
            check_type(argname="argument target_on_demand_capacity", value=target_on_demand_capacity, expected_type=type_hints["target_on_demand_capacity"])
            check_type(argname="argument target_spot_capacity", value=target_spot_capacity, expected_type=type_hints["target_spot_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_id": cluster_id,
            "instance_fleet_name": instance_fleet_name,
            "target_on_demand_capacity": target_on_demand_capacity,
            "target_spot_capacity": target_spot_capacity,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''The ClusterId to update.'''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_fleet_name(self) -> builtins.str:
        '''The InstanceFleetName to update.'''
        result = self._values.get("instance_fleet_name")
        assert result is not None, "Required property 'instance_fleet_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_on_demand_capacity(self) -> jsii.Number:
        '''The target capacity of On-Demand units for the instance fleet.

        :default: - None

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetModifyConfig.html
        '''
        result = self._values.get("target_on_demand_capacity")
        assert result is not None, "Required property 'target_on_demand_capacity' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def target_spot_capacity(self) -> jsii.Number:
        '''The target capacity of Spot units for the instance fleet.

        :default: - None

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetModifyConfig.html
        '''
        result = self._values.get("target_spot_capacity")
        assert result is not None, "Required property 'target_spot_capacity' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrModifyInstanceFleetByNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrModifyInstanceGroupByName(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName",
):
    '''A Step Functions Task to to modify an InstanceGroup on an EMR Cluster.

    :exampleMetadata: infused

    Example::

        tasks.EmrModifyInstanceGroupByName(self, "Task",
            cluster_id="ClusterId",
            instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
            instance_group=tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
                instance_count=1
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        instance_group: typing.Union["EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty", typing.Dict[builtins.str, typing.Any]],
        instance_group_name: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster_id: The ClusterId to update.
        :param instance_group: The JSON that you want to provide to your ModifyInstanceGroup call as input. This uses the same syntax as the ModifyInstanceGroups API.
        :param instance_group_name: The InstanceGroupName to update.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1827c77327dd89e8130c07793fdfbf5a53b4078f3db4417c751d28c129ffee23)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrModifyInstanceGroupByNameProps(
            cluster_id=cluster_id,
            instance_group=instance_group,
            instance_group_name=instance_group_name,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "configurations": "configurations",
            "e_c2_instance_ids_to_terminate": "eC2InstanceIdsToTerminate",
            "instance_count": "instanceCount",
            "shrink_policy": "shrinkPolicy",
        },
    )
    class InstanceGroupModifyConfigProperty:
        def __init__(
            self,
            *,
            configurations: typing.Optional[typing.Sequence[typing.Union[EmrCreateCluster.ConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
            e_c2_instance_ids_to_terminate: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_count: typing.Optional[jsii.Number] = None,
            shrink_policy: typing.Optional[typing.Union["EmrModifyInstanceGroupByName.ShrinkPolicyProperty", typing.Dict[builtins.str, typing.Any]]] = None,
        ) -> None:
            '''Modify the size or configurations of an instance group.

            :param configurations: A list of new or modified configurations to apply for an instance group. Default: - None
            :param e_c2_instance_ids_to_terminate: The EC2 InstanceIds to terminate. After you terminate the instances, the instance group will not return to its original requested size. Default: - None
            :param instance_count: Target size for the instance group. Default: - None
            :param shrink_policy: Policy for customizing shrink operations. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceGroupModifyConfig.html
            :exampleMetadata: infused

            Example::

                tasks.EmrModifyInstanceGroupByName(self, "Task",
                    cluster_id="ClusterId",
                    instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
                    instance_group=tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
                        instance_count=1
                    )
                )
            '''
            if isinstance(shrink_policy, dict):
                shrink_policy = EmrModifyInstanceGroupByName.ShrinkPolicyProperty(**shrink_policy)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a8056f2617e88a3638aeacdd442e5a0e107e12ce12b5dee24bbb38bf8c255446)
                check_type(argname="argument configurations", value=configurations, expected_type=type_hints["configurations"])
                check_type(argname="argument e_c2_instance_ids_to_terminate", value=e_c2_instance_ids_to_terminate, expected_type=type_hints["e_c2_instance_ids_to_terminate"])
                check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
                check_type(argname="argument shrink_policy", value=shrink_policy, expected_type=type_hints["shrink_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if configurations is not None:
                self._values["configurations"] = configurations
            if e_c2_instance_ids_to_terminate is not None:
                self._values["e_c2_instance_ids_to_terminate"] = e_c2_instance_ids_to_terminate
            if instance_count is not None:
                self._values["instance_count"] = instance_count
            if shrink_policy is not None:
                self._values["shrink_policy"] = shrink_policy

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]]:
            '''A list of new or modified configurations to apply for an instance group.

            :default: - None
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]], result)

        @builtins.property
        def e_c2_instance_ids_to_terminate(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''The EC2 InstanceIds to terminate.

            After you terminate the instances, the instance group will not return to its original requested size.

            :default: - None
            '''
            result = self._values.get("e_c2_instance_ids_to_terminate")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_count(self) -> typing.Optional[jsii.Number]:
            '''Target size for the instance group.

            :default: - None
            '''
            result = self._values.get("instance_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def shrink_policy(
            self,
        ) -> typing.Optional["EmrModifyInstanceGroupByName.ShrinkPolicyProperty"]:
            '''Policy for customizing shrink operations.

            :default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ShrinkPolicy.html
            '''
            result = self._values.get("shrink_policy")
            return typing.cast(typing.Optional["EmrModifyInstanceGroupByName.ShrinkPolicyProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceGroupModifyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceResizePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instances_to_protect": "instancesToProtect",
            "instances_to_terminate": "instancesToTerminate",
            "instance_termination_timeout": "instanceTerminationTimeout",
        },
    )
    class InstanceResizePolicyProperty:
        def __init__(
            self,
            *,
            instances_to_protect: typing.Optional[typing.Sequence[builtins.str]] = None,
            instances_to_terminate: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_termination_timeout: typing.Optional[_Duration_4839e8c3] = None,
        ) -> None:
            '''Custom policy for requesting termination protection or termination of specific instances when shrinking an instance group.

            :param instances_to_protect: Specific list of instances to be protected when shrinking an instance group. Default: - No instances will be protected when shrinking an instance group
            :param instances_to_terminate: Specific list of instances to be terminated when shrinking an instance group. Default: - No instances will be terminated when shrinking an instance group.
            :param instance_termination_timeout: Decommissioning timeout override for the specific list of instances to be terminated. Default: cdk.Duration.seconds

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceResizePolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                instance_resize_policy_property = stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceResizePolicyProperty(
                    instances_to_protect=["instancesToProtect"],
                    instances_to_terminate=["instancesToTerminate"],
                    instance_termination_timeout=cdk.Duration.minutes(30)
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__778ecacc32665d0ab8666358980dba4f6e4692bd6ad3fdac5756778824350571)
                check_type(argname="argument instances_to_protect", value=instances_to_protect, expected_type=type_hints["instances_to_protect"])
                check_type(argname="argument instances_to_terminate", value=instances_to_terminate, expected_type=type_hints["instances_to_terminate"])
                check_type(argname="argument instance_termination_timeout", value=instance_termination_timeout, expected_type=type_hints["instance_termination_timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if instances_to_protect is not None:
                self._values["instances_to_protect"] = instances_to_protect
            if instances_to_terminate is not None:
                self._values["instances_to_terminate"] = instances_to_terminate
            if instance_termination_timeout is not None:
                self._values["instance_termination_timeout"] = instance_termination_timeout

        @builtins.property
        def instances_to_protect(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specific list of instances to be protected when shrinking an instance group.

            :default: - No instances will be protected when shrinking an instance group
            '''
            result = self._values.get("instances_to_protect")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instances_to_terminate(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specific list of instances to be terminated when shrinking an instance group.

            :default: - No instances will be terminated when shrinking an instance group.
            '''
            result = self._values.get("instances_to_terminate")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_termination_timeout(self) -> typing.Optional[_Duration_4839e8c3]:
            '''Decommissioning timeout override for the specific list of instances to be terminated.

            :default: cdk.Duration.seconds
            '''
            result = self._values.get("instance_termination_timeout")
            return typing.cast(typing.Optional[_Duration_4839e8c3], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceResizePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.ShrinkPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "decommission_timeout": "decommissionTimeout",
            "instance_resize_policy": "instanceResizePolicy",
        },
    )
    class ShrinkPolicyProperty:
        def __init__(
            self,
            *,
            decommission_timeout: typing.Optional[_Duration_4839e8c3] = None,
            instance_resize_policy: typing.Optional[typing.Union["EmrModifyInstanceGroupByName.InstanceResizePolicyProperty", typing.Dict[builtins.str, typing.Any]]] = None,
        ) -> None:
            '''Policy for customizing shrink operations.

            Allows configuration of decommissioning timeout and targeted instance shrinking.

            :param decommission_timeout: The desired timeout for decommissioning an instance. Overrides the default YARN decommissioning timeout. Default: - EMR selected default
            :param instance_resize_policy: Custom policy for requesting termination protection or termination of specific instances when shrinking an instance group. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ShrinkPolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk as cdk
                from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                shrink_policy_property = stepfunctions_tasks.EmrModifyInstanceGroupByName.ShrinkPolicyProperty(
                    decommission_timeout=cdk.Duration.minutes(30),
                    instance_resize_policy=stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceResizePolicyProperty(
                        instances_to_protect=["instancesToProtect"],
                        instances_to_terminate=["instancesToTerminate"],
                        instance_termination_timeout=cdk.Duration.minutes(30)
                    )
                )
            '''
            if isinstance(instance_resize_policy, dict):
                instance_resize_policy = EmrModifyInstanceGroupByName.InstanceResizePolicyProperty(**instance_resize_policy)
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__86c726b15b0acf49d38041a7d08b5207cf1aabed2dd8fc2bd227d1b5c0bc01da)
                check_type(argname="argument decommission_timeout", value=decommission_timeout, expected_type=type_hints["decommission_timeout"])
                check_type(argname="argument instance_resize_policy", value=instance_resize_policy, expected_type=type_hints["instance_resize_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if decommission_timeout is not None:
                self._values["decommission_timeout"] = decommission_timeout
            if instance_resize_policy is not None:
                self._values["instance_resize_policy"] = instance_resize_policy

        @builtins.property
        def decommission_timeout(self) -> typing.Optional[_Duration_4839e8c3]:
            '''The desired timeout for decommissioning an instance.

            Overrides the default YARN decommissioning timeout.

            :default: - EMR selected default
            '''
            result = self._values.get("decommission_timeout")
            return typing.cast(typing.Optional[_Duration_4839e8c3], result)

        @builtins.property
        def instance_resize_policy(
            self,
        ) -> typing.Optional["EmrModifyInstanceGroupByName.InstanceResizePolicyProperty"]:
            '''Custom policy for requesting termination protection or termination of specific instances when shrinking an instance group.

            :default: - None
            '''
            result = self._values.get("instance_resize_policy")
            return typing.cast(typing.Optional["EmrModifyInstanceGroupByName.InstanceResizePolicyProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ShrinkPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrModifyInstanceGroupByNameProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "instance_group": "instanceGroup",
        "instance_group_name": "instanceGroupName",
    },
)
class EmrModifyInstanceGroupByNameProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster_id: builtins.str,
        instance_group: typing.Union[EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty, typing.Dict[builtins.str, typing.Any]],
        instance_group_name: builtins.str,
    ) -> None:
        '''Properties for EmrModifyInstanceGroupByName.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster_id: The ClusterId to update.
        :param instance_group: The JSON that you want to provide to your ModifyInstanceGroup call as input. This uses the same syntax as the ModifyInstanceGroups API.
        :param instance_group_name: The InstanceGroupName to update.

        :exampleMetadata: infused

        Example::

            tasks.EmrModifyInstanceGroupByName(self, "Task",
                cluster_id="ClusterId",
                instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
                instance_group=tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
                    instance_count=1
                )
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(instance_group, dict):
            instance_group = EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(**instance_group)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9c62909c53696cbd5e1b50294401de7e37a1a51225d9bd25c6cf98c6c0f556b8)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster_id", value=cluster_id, expected_type=type_hints["cluster_id"])
            check_type(argname="argument instance_group", value=instance_group, expected_type=type_hints["instance_group"])
            check_type(argname="argument instance_group_name", value=instance_group_name, expected_type=type_hints["instance_group_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_id": cluster_id,
            "instance_group": instance_group,
            "instance_group_name": instance_group_name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''The ClusterId to update.'''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_group(
        self,
    ) -> EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty:
        '''The JSON that you want to provide to your ModifyInstanceGroup call as input.

        This uses the same syntax as the ModifyInstanceGroups API.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html
        '''
        result = self._values.get("instance_group")
        assert result is not None, "Required property 'instance_group' is missing"
        return typing.cast(EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty, result)

    @builtins.property
    def instance_group_name(self) -> builtins.str:
        '''The InstanceGroupName to update.'''
        result = self._values.get("instance_group_name")
        assert result is not None, "Required property 'instance_group_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrModifyInstanceGroupByNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrSetClusterTerminationProtection(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrSetClusterTerminationProtection",
):
    '''A Step Functions Task to to set Termination Protection on an EMR Cluster.

    :exampleMetadata: infused

    Example::

        tasks.EmrSetClusterTerminationProtection(self, "Task",
            cluster_id="ClusterId",
            termination_protected=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        termination_protected: builtins.bool,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster_id: The ClusterId to update.
        :param termination_protected: Termination protection indicator.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__152b964688ba0a54d4029e6c9be6f9b58c7872d51ce6c52c872bbf5b5f961b62)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrSetClusterTerminationProtectionProps(
            cluster_id=cluster_id,
            termination_protected=termination_protected,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrSetClusterTerminationProtectionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "termination_protected": "terminationProtected",
    },
)
class EmrSetClusterTerminationProtectionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster_id: builtins.str,
        termination_protected: builtins.bool,
    ) -> None:
        '''Properties for EmrSetClusterTerminationProtection.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster_id: The ClusterId to update.
        :param termination_protected: Termination protection indicator.

        :exampleMetadata: infused

        Example::

            tasks.EmrSetClusterTerminationProtection(self, "Task",
                cluster_id="ClusterId",
                termination_protected=False
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__43f523884966182c76c7e1e2fef4a4b49988e980e5a676cef0ca41c1875a0f51)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster_id", value=cluster_id, expected_type=type_hints["cluster_id"])
            check_type(argname="argument termination_protected", value=termination_protected, expected_type=type_hints["termination_protected"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_id": cluster_id,
            "termination_protected": termination_protected,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''The ClusterId to update.'''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def termination_protected(self) -> builtins.bool:
        '''Termination protection indicator.'''
        result = self._values.get("termination_protected")
        assert result is not None, "Required property 'termination_protected' is missing"
        return typing.cast(builtins.bool, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrSetClusterTerminationProtectionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrTerminateCluster(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrTerminateCluster",
):
    '''A Step Functions Task to terminate an EMR Cluster.

    :exampleMetadata: infused

    Example::

        tasks.EmrTerminateCluster(self, "Task",
            cluster_id="ClusterId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param cluster_id: The ClusterId to terminate.
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40203f9a6659336dbad4533d895e62225d438ca7ba10b27e16a78a5f22c1ea05)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EmrTerminateClusterProps(
            cluster_id=cluster_id,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EmrTerminateClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "cluster_id": "clusterId",
    },
)
class EmrTerminateClusterProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        cluster_id: builtins.str,
    ) -> None:
        '''Properties for EmrTerminateCluster.

        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param cluster_id: The ClusterId to terminate.

        :exampleMetadata: infused

        Example::

            tasks.EmrTerminateCluster(self, "Task",
                cluster_id="ClusterId"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c80e906432a50136c0f9e04017864ce1c2e4248cf7a80bc25f1be91c45190d1a)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument cluster_id", value=cluster_id, expected_type=type_hints["cluster_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_id": cluster_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''An optional description for this state.

        :default: - No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''The ClusterId to terminate.'''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrTerminateClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EncryptionConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "encryption_option": "encryptionOption",
        "encryption_key": "encryptionKey",
    },
)
class EncryptionConfiguration:
    def __init__(
        self,
        *,
        encryption_option: "EncryptionOption",
        encryption_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> None:
        '''Encryption Configuration of the S3 bucket.

        :param encryption_option: Type of S3 server-side encryption enabled. Default: EncryptionOption.S3_MANAGED
        :param encryption_key: KMS key ARN or ID. Default: - No KMS key for Encryption Option SSE_S3 and default master key for Encryption Option SSE_KMS and CSE_KMS

        :see: https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html
        :exampleMetadata: infused

        Example::

            start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
                query_string=sfn.JsonPath.string_at("$.queryString"),
                query_execution_context=tasks.QueryExecutionContext(
                    database_name="mydatabase"
                ),
                result_configuration=tasks.ResultConfiguration(
                    encryption_configuration=tasks.EncryptionConfiguration(
                        encryption_option=tasks.EncryptionOption.S3_MANAGED
                    ),
                    output_location=s3.Location(
                        bucket_name="query-results-bucket",
                        object_key="folder"
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__313b7b0e1a834568a08e21e304144a11ad51f72b18d4c52f214e39ca8c2e7e15)
            check_type(argname="argument encryption_option", value=encryption_option, expected_type=type_hints["encryption_option"])
            check_type(argname="argument encryption_key", value=encryption_key, expected_type=type_hints["encryption_key"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "encryption_option": encryption_option,
        }
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key

    @builtins.property
    def encryption_option(self) -> "EncryptionOption":
        '''Type of S3 server-side encryption enabled.

        :default: EncryptionOption.S3_MANAGED
        '''
        result = self._values.get("encryption_option")
        assert result is not None, "Required property 'encryption_option' is missing"
        return typing.cast("EncryptionOption", result)

    @builtins.property
    def encryption_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''KMS key ARN or ID.

        :default: - No KMS key for Encryption Option SSE_S3 and default master key for Encryption Option SSE_KMS and CSE_KMS
        '''
        result = self._values.get("encryption_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EncryptionConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EncryptionOption")
class EncryptionOption(enum.Enum):
    '''Encryption Options of the S3 bucket.

    :see: https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html#athena-Type-EncryptionConfiguration-EncryptionOption
    :exampleMetadata: infused

    Example::

        start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
            query_string=sfn.JsonPath.string_at("$.queryString"),
            query_execution_context=tasks.QueryExecutionContext(
                database_name="mydatabase"
            ),
            result_configuration=tasks.ResultConfiguration(
                encryption_configuration=tasks.EncryptionConfiguration(
                    encryption_option=tasks.EncryptionOption.S3_MANAGED
                ),
                output_location=s3.Location(
                    bucket_name="query-results-bucket",
                    object_key="folder"
                )
            )
        )
    '''

    S3_MANAGED = "S3_MANAGED"
    '''Server side encryption (SSE) with an Amazon S3-managed key.

    :see: https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
    '''
    KMS = "KMS"
    '''Server-side encryption (SSE) with an AWS KMS key managed by the account owner.

    :see: https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
    '''
    CLIENT_SIDE_KMS = "CLIENT_SIDE_KMS"
    '''Client-side encryption (CSE) with an AWS KMS key managed by the account owner.

    :see: https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
    '''


class EvaluateExpression(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EvaluateExpression",
):
    '''A Step Functions Task to evaluate an expression.

    OUTPUT: the output of this task is the evaluated expression.

    :exampleMetadata: infused

    Example::

        convert_to_seconds = tasks.EvaluateExpression(self, "Convert to seconds",
            expression="$.waitMilliseconds / 1000",
            result_path="$.waitSeconds"
        )
        
        create_message = tasks.EvaluateExpression(self, "Create message",
            # Note: this is a string inside a string.
            expression="`Now waiting ${$.waitSeconds} seconds...`",
            runtime=lambda_.Runtime.NODEJS_16_X,
            result_path="$.message"
        )
        
        publish_message = tasks.SnsPublish(self, "Publish message",
            topic=sns.Topic(self, "cool-topic"),
            message=sfn.TaskInput.from_json_path_at("$.message"),
            result_path="$.sns"
        )
        
        wait = sfn.Wait(self, "Wait",
            time=sfn.WaitTime.seconds_path("$.waitSeconds")
        )
        
        sfn.StateMachine(self, "StateMachine",
            definition=convert_to_seconds.next(create_message).next(publish_message).next(wait)
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        expression: builtins.str,
        runtime: typing.Optional[_Runtime_b4eaa844] = None,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param expression: The expression to evaluate. The expression may contain state paths. Example value: ``'$.a + $.b'``
        :param runtime: The runtime language to use to evaluate the expression. Default: lambda.Runtime.NODEJS_14_X
        :param comment: An optional description for this state. Default: - No comment
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param output_path: JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7da4306c848323ad0fd19f879d1e450efab7eb83fb1ef1cb0d748933f38bdb35)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EvaluateExpressionProps(
            expression=expression,
            runtime=runtime,
            comment=comment,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            task_timeout=task_timeout,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.EvaluateExpressionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "expression": "expression",
        "runtime": "runtime",
    },
)
class EvaluateExpressionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        expression: builtins.str,
        runtime: typing.Optional[_Runtime_b4eaa844] = None,
