"""Define tests for the Vaughan model."""

from typing import Any

import numpy as np
import pandas as pd
import pytest
from eemilib.emission_data.data_matrix import DataMatrix
from eemilib.emission_data.emission_yield import EmissionYield
from eemilib.model.vaughan import Vaughan, VaughanImplementation
from numpy.testing import assert_array_almost_equal
from numpy.typing import NDArray
from pytest import approx


@pytest.fixture
def vaughan_model() -> Vaughan:
    """Create a default instance of :class:`.Vaughan` model."""
    return Vaughan()


class MockDataMatrix(DataMatrix):
    """Mock a data matrix with only a TEEY."""

    def __init__(self, emission_data):
        """Set emission yield for 'all' population."""
        self.data_matrix = [
            [None, None, None],
            [None, None, None],
            [None, None, None],
            [emission_data, None, None],
        ]

    def has_all_mandatory_files(self, *args, **kwargs) -> bool:
        """Skip this check."""
        return True


def test_initial_parameters(vaughan_model: Vaughan) -> None:
    """Check that the mandatory parameters are defined."""
    expected_parameters = {
        "E_0",
        "E_max",
        "delta_E_transition",
        "teey_low",
        "teey_max",
        "k_s",
        "k_se",
        "E_c1",
    }
    assert set(vaughan_model.initial_parameters.keys()) == expected_parameters


def test_teey_output_shape(vaughan_model: Vaughan) -> None:
    """Check that TEEY array has proper shape."""
    energy = np.linspace(0, 100, 5, dtype=np.float64)
    theta = np.linspace(0, 90, 3, dtype=np.float64)
    result = vaughan_model.teey(energy, theta)
    assert isinstance(result, pd.DataFrame)
    assert result.shape == (5, 4)  # 3 theta columns + 1 energy column


@pytest.mark.parametrize(
    "emission_yield,expected",
    [
        pytest.param(
            EmissionYield(
                population="all",
                data=pd.DataFrame(
                    {
                        # fmt: off
                "Energy [eV]": [0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100, 110, 120, 130, 140, 150, 160, 170, 180, 190, 200, 210, 220, 230, 240, 250, 260, 270, 280, 290, 300, 310, 320, 330, 340, 350, 360, 370, 380, 390, 400, 410, 420, 430, 440, 450, 460, 470, 480, 490, 500, 510, 520, 530, 540, 550, 560, 570, 580, 590, 600, 610, 620, 630, 640, 650, 660, 670, 680, 690, 700, 710, 720, 730, 740, 750, 760, 770, 780, 790, 800, 810, 820, 830, 840, 850, 860, 870, 880, 890, 900, 910, 920, 930, 940, 950, 960, 970, 980, 990, 1000],
                "0.0 [deg]": [0.814, 0.574, 0.632, 0.677, 0.722, 0.768, 0.825, 0.879, 0.922, 0.98, 1.019, 1.068, 1.122, 1.151, 1.187, 1.224, 1.254, 1.281, 1.304, 1.326, 1.343, 1.365, 1.379, 1.39, 1.405, 1.415, 1.426, 1.435, 1.441, 1.449, 1.458, 1.464, 1.47, 1.476, 1.485, 1.484, 1.485, 1.493, 1.494, 1.502, 1.502, 1.506, 1.508, 1.511, 1.51, 1.513, 1.518, 1.52, 1.522, 1.521, 1.52, 1.523, 1.522, 1.521, 1.521, 1.526, 1.525, 1.523, 1.524, 1.521, 1.523, 1.52, 1.518, 1.515, 1.519, 1.52, 1.516, 1.514, 1.509, 1.511, 1.508, 1.504, 1.501, 1.502, 1.504, 1.506, 1.499, 1.497, 1.495, 1.496, 1.494, 1.489, 1.489, 1.484, 1.482, 1.477, 1.479, 1.473, 1.475, 1.469, 1.469, 1.467, 1.464, 1.461, 1.459, 1.457, 1.45, 1.451, 1.451, 1.443, 1.43,],
                "20.0 [deg]": [0.781, 0.553, 0.594, 0.628, 0.673, 0.72, 0.771, 0.823, 0.872, 0.92, 0.972, 1.011, 1.054, 1.09, 1.119, 1.155, 1.189, 1.21, 1.232, 1.259, 1.277, 1.294, 1.311, 1.325, 1.342, 1.355, 1.361, 1.367, 1.377, 1.381, 1.394, 1.402, 1.409, 1.417, 1.423, 1.424, 1.428, 1.435, 1.442, 1.448, 1.446, 1.454, 1.457, 1.463, 1.465, 1.465, 1.47, 1.473, 1.473, 1.474, 1.476, 1.478, 1.477, 1.479, 1.481, 1.483, 1.484, 1.485, 1.482, 1.484, 1.484, 1.484, 1.486, 1.487, 1.49, 1.485, 1.482, 1.484, 1.484, 1.481, 1.485, 1.479, 1.479, 1.479, 1.482, 1.48, 1.477, 1.476, 1.475, 1.47, 1.468, 1.467, 1.469, 1.462, 1.463, 1.463, 1.46, 1.46, 1.457, 1.455, 1.455, 1.451, 1.448, 1.45, 1.447, 1.442, 1.442, 1.44, 1.441, 1.436, 1.435],
                "40.0 [deg]": [0.866, 0.637, 0.671, 0.715, 0.76, 0.808, 0.868, 0.924, 0.979, 1.03, 1.086, 1.131, 1.176, 1.213, 1.254, 1.293, 1.317, 1.342, 1.372, 1.389, 1.411, 1.424, 1.445, 1.461, 1.48, 1.488, 1.502, 1.513, 1.524, 1.534, 1.546, 1.553, 1.564, 1.569, 1.574, 1.583, 1.59, 1.594, 1.602, 1.604, 1.612, 1.614, 1.621, 1.628, 1.629, 1.635, 1.637, 1.642, 1.64, 1.643, 1.647, 1.65, 1.654, 1.654, 1.655, 1.657, 1.655, 1.656, 1.657, 1.661, 1.661, 1.66, 1.659, 1.661, 1.665, 1.665, 1.663, 1.658, 1.659, 1.66, 1.659, 1.66, 1.655, 1.656, 1.656, 1.654, 1.652, 1.653, 1.65, 1.65, 1.648, 1.65, 1.648, 1.646, 1.644, 1.643, 1.646, 1.641, 1.638, 1.639, 1.633, 1.635, 1.631, 1.627, 1.627, 1.624, 1.621, 1.622, 1.621, 1.618, 1.616],
                "60.0 [deg]": [0.918, 0.803, 0.817, 0.859, 0.926, 0.995, 1.061, 1.127, 1.195, 1.257, 1.32, 1.375, 1.428, 1.466, 1.51, 1.552, 1.584, 1.616, 1.644, 1.67, 1.694, 1.718, 1.738, 1.759, 1.781, 1.797, 1.817, 1.824, 1.836, 1.852, 1.87, 1.881, 1.893, 1.896, 1.905, 1.917, 1.92, 1.937, 1.94, 1.952, 1.956, 1.966, 1.97, 1.971, 1.976, 1.981, 1.99, 1.993, 1.993, 1.996, 2.001, 2.005, 2.009, 2.008, 2.013, 2.012, 2.016, 2.018, 2.023, 2.024, 2.018, 2.027, 2.019, 2.028, 2.025, 2.027, 2.034, 2.035, 2.027, 2.028, 2.035, 2.033, 2.034, 2.031, 2.031, 2.025, 2.025, 2.033, 2.03, 2.022, 2.023, 2.025, 2.022, 2.019, 2.02, 2.012, 2.018, 2.017, 2.009, 2.018, 2.018, 2.009, 2.008, 2.013, 2.003, 2.009, 2.001, 2.005, 2.002, 1.997, 2.001],
                        # fmt: on
                    }
                ),
            ),
            {
                "E_0": 12.5,
                "E_max": 550.5505505505506,
                "delta_E_transition": 1.0,
                "teey_low": 0.5,
                "teey_max": 1.525944944944945,
                "k_s": 1.0,
                "k_se": 1.0,
                "E_c1": 95.0950950950951,
            },
            id="Cu 1 eroded",
        ),
        pytest.param(
            EmissionYield(
                population="all",
                data=pd.DataFrame(
                    {
                        # fmt: off
                "Energy [eV]": [10, 30, 50, 70, 90, 110, 130, 150, 170, 190, 210, 230, 250, 270, 290, 310, 330, 350, 370, 390, 410, 430, 450, 470, 490, 510, 530, 550, 570, 590, 610, 630, 650, 670, 690, 710, 730, 750, 770, 790, 810, 830, 850, 870, 890, 910, 930, 950, 970, 990],
                "0.0 [deg]": [0.696, 1.1, 1.364, 1.605, 1.754, 1.895, 1.979, 2.021, 2.115, 2.153, 2.185, 2.221, 2.237, 2.234, 2.221, 2.208, 2.201, 2.193, 2.17, 2.199, 2.181, 2.182, 2.145, 2.15, 2.121, 2.097, 2.102, 2.077, 2.075, 2.039, 2.047, 2.032, 2.027, 2.007, 1.998, 2.001, 1.982, 1.969, 1.936, 1.94, 1.931, 1.918, 1.921, 1.909, 1.89, 1.884, 1.88, 1.856, 1.875, 1.838],
                "20.0 [deg]": [0.704, 1.075, 1.357, 1.58, 1.716, 1.876, 1.988, 2.092, 2.17, 2.217, 2.207, 2.274, 2.281, 2.326, 2.3, 2.298, 2.288, 2.281, 2.262, 2.27, 2.271, 2.235, 2.227, 2.199, 2.201, 2.156, 2.188, 2.165, 2.136, 2.107, 2.13, 2.109, 2.114, 2.121, 2.078, 2.044, 2.064, 2.033, 2.022, 2.009, 2.023, 2.018, 1.972, 1.978, 1.994, 1.974, 1.941, 1.951, 1.915, 1.91],
                "40.0 [deg]": [0.784, 1.086, 1.37, 1.609, 1.777, 1.957, 2.082, 2.141, 2.235, 2.301, 2.328, 2.372, 2.373, 2.386, 2.363, 2.388, 2.402, 2.399, 2.389, 2.378, 2.373, 2.396, 2.404, 2.358, 2.374, 2.383, 2.344, 2.38, 2.334, 2.324, 2.289, 2.274, 2.268, 2.236, 2.263, 2.225, 2.235, 2.228, 2.219, 2.191, 2.179, 2.163, 2.155, 2.149, 2.18, 2.153, 2.13, 2.12, 2.104, 2.099],
                "60.0 [deg]": [0.744, 1.063, 1.334, 1.597, 1.739, 1.928, 2.169, 2.183, 2.331, 2.375, 2.374, 2.519, 2.519, 2.506, 2.528, 2.558, 2.583, 2.586, 2.616, 2.605, 2.576, 2.56, 2.582, 2.608, 2.606, 2.582, 2.618, 2.585, 2.546, 2.542, 2.534, 2.556, 2.543, 2.53, 2.509, 2.578, 2.488, 2.511, 2.479, 2.475, 2.541, 2.459, 2.46, 2.503, 2.441, 2.478, 2.444, 2.427, 2.433, 2.411],
                        # fmt: on
                    }
                ),
            ),
            {
                "E_0": 12.5,
                "E_max": 250.34034034034033,
                "delta_E_transition": 1.0,
                "teey_low": 0.5,
                "teey_max": 2.236948948948949,
                "k_s": 1.0,
                "k_se": 1.0,
                "E_c1": 24.714714714714717,
            },
            marks=pytest.mark.smoke,
            id="Cu 2 as received",
        ),
        pytest.param(
            EmissionYield(
                population="all",
                data=pd.DataFrame(
                    {
                        # fmt: off
                "Energy [eV]": [10, 30, 50, 70, 90, 110, 130, 150, 170, 190, 210, 230, 250, 270, 290, 310, 330, 350, 370, 390, 410, 430, 450, 470, 490, 510, 530, 550, 570, 590, 610, 630, 650, 670, 690, 710, 730, 750, 770, 790, 810, 830, 850, 870, 890, 910, 930, 950, 970, 990],
                "0.0 [deg]": [0.569, 0.871, 1.039, 1.189, 1.303, 1.388, 1.465, 1.528, 1.574, 1.601, 1.644, 1.659, 1.677, 1.688, 1.674, 1.675, 1.675, 1.68, 1.689, 1.696, 1.68, 1.665, 1.659, 1.652, 1.642, 1.642, 1.623, 1.614, 1.622, 1.599, 1.614, 1.568, 1.583, 1.567, 1.557, 1.548, 1.549, 1.54, 1.525, 1.517, 1.518, 1.51, 1.486, 1.477, 1.471, 1.467, 1.465, 1.442, 1.45, 1.436],
                "20.0 [deg]": [0.584, 0.851, 1.063, 1.202, 1.355, 1.433, 1.526, 1.57, 1.612, 1.666, 1.694, 1.711, 1.717, 1.735, 1.75, 1.734, 1.744, 1.743, 1.745, 1.739, 1.724, 1.72, 1.712, 1.709, 1.698, 1.699, 1.681, 1.655, 1.66, 1.659, 1.642, 1.644, 1.644, 1.62, 1.615, 1.614, 1.616, 1.599, 1.586, 1.579, 1.568, 1.549, 1.548, 1.535, 1.536, 1.516, 1.514, 1.496, 1.509, 1.493],
                "40.0 [deg]": [0.688, 0.861, 1.146, 1.304, 1.459, 1.582, 1.632, 1.74, 1.779, 1.807, 1.852, 1.885, 1.887, 1.915, 1.942, 1.937, 1.965, 1.961, 1.971, 1.951, 1.947, 1.944, 1.918, 1.919, 1.943, 1.914, 1.9, 1.905, 1.891, 1.89, 1.86, 1.878, 1.848, 1.865, 1.842, 1.827, 1.823, 1.825, 1.789, 1.808, 1.798, 1.767, 1.769, 1.758, 1.751, 1.725, 1.738, 1.721, 1.715, 1.704],
                "60.0 [deg]": [0.777, 0.867, 1.058, 1.208, 1.347, 1.464, 1.557, 1.611, 1.694, 1.737, 1.798, 1.822, 1.847, 1.853, 1.876, 1.87, 1.886, 1.918, 1.919, 1.923, 1.91, 1.909, 1.895, 1.926, 1.904, 1.914, 1.924, 1.906, 1.904, 1.897, 1.898, 1.916, 1.917, 1.905, 1.905, 1.886, 1.879, 1.886, 1.881, 1.878, 1.867, 1.851, 1.854, 1.852, 1.855, 1.868, 1.844, 1.839, 1.851, 1.842],
                        # fmt: on
                    }
                ),
            ),
            {
                "E_0": 12.5,
                "E_max": 389.63963963963965,
                "delta_E_transition": 1.0,
                "teey_low": 0.5,
                "teey_max": 1.695873873873874,
                "k_s": 1.0,
                "k_se": 1.0,
                "E_c1": 45.31531531531532,
            },
            id="Cu 2 heated",
        ),
    ],
)
def test_find_optimal_parameters(
    vaughan_model: Vaughan,
    emission_yield: Any,
    expected: dict[str, float],
) -> None:
    """Test on several samples that the fit gives expected results."""
    mock_data_matrix = MockDataMatrix(emission_yield)
    vaughan_model.find_optimal_parameters(mock_data_matrix)
    found_parameters = {
        name: val.value for name, val in vaughan_model.parameters.items()
    }
    assert expected == approx(found_parameters)


@pytest.mark.parametrize(
    "vaughan_parameters,vaughan_implementation,energy,theta,expected",
    [
        pytest.param(
            {
                "E_c1": 17,
                "E_max": 276,
                "teey_low": 0.8,
                "teey_max": 2.92,
                "E_0": 8,
            },
            "SPARK3D",
            np.linspace(0, 1000, 501),
            np.zeros(1),
            # fmt: off
            np.array([[0, 0, 0, 0, 0, 0.729686, 0.815298, 0.893233, 0.965107, 1.03202, 1.09475, 1.15389, 1.20991, 1.26315, 1.31392, 1.36245, 1.40895, 1.45358, 1.49649, 1.53782, 1.57767, 1.61614, 1.65331, 1.68927, 1.72408, 1.7578, 1.7905, 1.82222, 1.85301, 1.88291, 1.91196, 1.9402, 1.96767, 1.99439, 2.02039, 2.0457, 2.07035, 2.09436, 2.11775, 2.14055, 2.16276, 2.18442, 2.20554, 2.22614, 2.24622, 2.26581, 2.28493, 2.30358, 2.32177, 2.33953, 2.35686, 2.37377, 2.39028, 2.40639, 2.42211, 2.43746, 2.45244, 2.46706, 2.48133, 2.49526, 2.50886, 2.52213, 2.53508, 2.54772, 2.56005, 2.57208, 2.58383, 2.59528, 2.60646, 2.61736, 2.62799, 2.63835, 2.64846, 2.65832, 2.66792, 2.67729, 2.68641, 2.6953, 2.70396, 2.71239, 2.7206, 2.7286, 2.73638, 2.74395, 2.75131, 2.75847, 2.76544, 2.7722, 2.77878, 2.78517, 2.79137, 2.79739, 2.80323, 2.80889, 2.81438, 2.8197, 2.82485, 2.82984, 2.83467, 2.83934, 2.84385, 2.8482, 2.8524, 2.85646, 2.86037, 2.86413, 2.86775, 2.87123, 2.87457, 2.87777, 2.88085, 2.88379, 2.8866, 2.88928, 2.89184, 2.89427, 2.89658, 2.89877, 2.90085, 2.9028, 2.90465, 2.90637, 2.90799, 2.9095, 2.9109, 2.91219, 2.91338, 2.91447, 2.91545, 2.91633, 2.91712, 2.9178, 2.91839, 2.91889, 2.91929, 2.9196, 2.91983, 2.91996, 2.92, 2.91998, 2.91992, 2.91983, 2.9197, 2.91953, 2.91932, 2.91908, 2.91881, 2.9185, 2.91816, 2.91778, 2.91737, 2.91693, 2.91645, 2.91594, 2.91541, 2.91484, 2.91424, 2.91361, 2.91295, 2.91226, 2.91155, 2.9108, 2.91003, 2.90923, 2.9084, 2.90754, 2.90666, 2.90575, 2.90482, 2.90386, 2.90287, 2.90186, 2.90082, 2.89977, 2.89868, 2.89758, 2.89644, 2.89529, 2.89412, 2.89292, 2.8917, 2.89045, 2.88919, 2.8879, 2.88659, 2.88527, 2.88392, 2.88255, 2.88116, 2.87975, 2.87832, 2.87688, 2.87541, 2.87392, 2.87242, 2.8709, 2.86935, 2.8678, 2.86622, 2.86463, 2.86301, 2.86139, 2.85974, 2.85808, 2.8564, 2.8547, 2.85299, 2.85127, 2.84952, 2.84777, 2.84599, 2.8442, 2.8424, 2.84058, 2.83875, 2.8369, 2.83504, 2.83317, 2.83128, 2.82938, 2.82746, 2.82553, 2.82359, 2.82163, 2.81966, 2.81768, 2.81569, 2.81368, 2.81166, 2.80963, 2.80759, 2.80553, 2.80346, 2.80139, 2.7993, 2.7972, 2.79508, 2.79296, 2.79083, 2.78868, 2.78653, 2.78436, 2.78218, 2.78, 2.7778, 2.7756, 2.77338, 2.77115, 2.76892, 2.76667, 2.76442, 2.76215, 2.75988, 2.7576, 2.75531, 2.75301, 2.7507, 2.74838, 2.74606, 2.74372, 2.74138, 2.73903, 2.73667, 2.73431, 2.73193, 2.72955, 2.72716, 2.72476, 2.72236, 2.71995, 2.71753, 2.7151, 2.71267, 2.71023, 2.70778, 2.70532, 2.70286, 2.70039, 2.69792, 2.69544, 2.69295, 2.69046, 2.68796, 2.68545, 2.68294, 2.68042, 2.6779, 2.67537, 2.67283, 2.67029, 2.66774, 2.66519, 2.66263, 2.66007, 2.6575, 2.65493, 2.65235, 2.64976, 2.64717, 2.64458, 2.64198, 2.63938, 2.63677, 2.63415, 2.63154, 2.62891, 2.62629, 2.62366, 2.62102, 2.61838, 2.61574, 2.61309, 2.61044, 2.60778, 2.60512, 2.60245, 2.59979, 2.59711, 2.59444, 2.59176, 2.58908, 2.58639, 2.5837, 2.58101, 2.57831, 2.57561, 2.57291, 2.5702, 2.56749, 2.56478, 2.56206, 2.55934, 2.55662, 2.55389, 2.55117, 2.54844, 2.5457, 2.54297, 2.54023, 2.53749, 2.53474, 2.532, 2.52925, 2.5265, 2.52374, 2.52099, 2.51823, 2.51547, 2.5127, 2.50994, 2.50717, 2.5044, 2.50163, 2.49886, 2.49608, 2.4933, 2.49053, 2.48774, 2.48496, 2.48218, 2.47939, 2.4766, 2.47381, 2.47102, 2.46823, 2.46544, 2.46264, 2.45984, 2.45705, 2.45425, 2.45144, 2.44864, 2.44584, 2.44303, 2.44023, 2.43742, 2.43461, 2.4318, 2.42899, 2.42618, 2.42337, 2.42056, 2.41774, 2.41493, 2.41211, 2.40929, 2.40648, 2.40366, 2.40084, 2.39802, 2.3952, 2.39238, 2.38956, 2.38673, 2.38391, 2.38109, 2.37826, 2.37544, 2.37261, 2.36979, 2.36696, 2.36414, 2.36131, 2.35849, 2.35566, 2.35283, 2.35001, 2.34718, 2.34435, 2.34152, 2.3387, 2.33587, 2.33304, 2.33021, 2.32739, 2.32456, 2.32173, 2.3189, 2.31608, 2.31325, 2.31042, 2.3076, 2.30477, 2.30194, 2.29912, 2.29629, 2.29346, 2.29064, 2.28781, 2.28499, 2.28216, 2.27934, 2.27652, 2.27369, 2.27087, 2.26805, 2.26523, 2.26241, 2.25959, 2.25677, 2.25395, 2.25113, 2.24831, 2.24549, 2.24268, 2.23986, 2.23704, 2.23423, 2.23142, 2.2286, 2.22579, 2.22298, 2.22017, 2.21736, 2.21455, 2.21174, 2.20893, 2.20613, 2.20332, 2.20052, 2.19771, 2.19491, 2.19211, 2.18931, 2.18651, 2.18371, 2.18091, 2.17812, 2.17532, 2.17253, 2.16973, 2.16694, 2.16415, 2.16136, 2.15857, 2.15578, 2.153, 2.15021, 2.14743, 2.14465, 2.14187, 2.13909, 2.13631, 2.13353, 2.13075, 2.12798, 2.12521, 2.12243, 2.11966, 2.11689, 2.11413, 2.11136, 2.1086, 2.10583, 2.10307, 2.10031, 2.09693, 2.09546, 2.09398, 2.09251, 2.09104]]),
            # fmt: on
            id="Compare EEmiLib with manual SPARK3D export on Aluminium (ECSS-2020)",
        ),
        pytest.param(
            {"E_c1": 19, "E_max": 232, "teey_low": 0.8, "teey_max": 2.48},
            "SPARK3D",
            np.linspace(0, 1000, 501),
            np.zeros(1),
            # fmt: off
            np.array([[0, 0, 0, 0, 0, 0.696006, 0.773736, 0.844535, 0.909806, 0.970519, 1.02737, 1.0809, 1.1315, 1.17952, 1.22521, 1.26881, 1.31048, 1.3504, 1.38869, 1.42547, 1.46086, 1.49493, 1.52778, 1.55946, 1.59006, 1.61962, 1.64819, 1.67584, 1.7026, 1.72851, 1.75361, 1.77794, 1.80152, 1.82439, 1.84658, 1.8681, 1.889, 1.90928, 1.92897, 1.94809, 1.96666, 1.9847, 2.00222, 2.01924, 2.03577, 2.05184, 2.06745, 2.08262, 2.09735, 2.11167, 2.12558, 2.1391, 2.15223, 2.16498, 2.17737, 2.18941, 2.20109, 2.21244, 2.22345, 2.23415, 2.24452, 2.25459, 2.26436, 2.27384, 2.28303, 2.29194, 2.30057, 2.30894, 2.31704, 2.32489, 2.33249, 2.33984, 2.34695, 2.35382, 2.36046, 2.36688, 2.37308, 2.37905, 2.38482, 2.39038, 2.39573, 2.40088, 2.40584, 2.4106, 2.41517, 2.41956, 2.42377, 2.4278, 2.43165, 2.43533, 2.43885, 2.44219, 2.44538, 2.44841, 2.45127, 2.45399, 2.45656, 2.45897, 2.46125, 2.46337, 2.46536, 2.46721, 2.46893, 2.47051, 2.47196, 2.47329, 2.47449, 2.47556, 2.47651, 2.47735, 2.47806, 2.47866, 2.47915, 2.47952, 2.47979, 2.47995, 2.48, 2.47998, 2.47991, 2.47979, 2.47964, 2.47944, 2.47919, 2.47891, 2.47858, 2.47821, 2.47781, 2.47736, 2.47687, 2.47635, 2.47579, 2.47519, 2.47456, 2.47389, 2.47319, 2.47245, 2.47168, 2.47087, 2.47004, 2.46917, 2.46826, 2.46733, 2.46637, 2.46537, 2.46435, 2.46329, 2.46221, 2.4611, 2.45996, 2.45879, 2.45759, 2.45637, 2.45512, 2.45385, 2.45255, 2.45122, 2.44987, 2.4485, 2.4471, 2.44568, 2.44423, 2.44276, 2.44127, 2.43976, 2.43822, 2.43666, 2.43508, 2.43348, 2.43186, 2.43021, 2.42855, 2.42687, 2.42517, 2.42344, 2.4217, 2.41994, 2.41817, 2.41637, 2.41455, 2.41272, 2.41087, 2.409, 2.40712, 2.40522, 2.4033, 2.40137, 2.39942, 2.39745, 2.39547, 2.39347, 2.39146, 2.38944, 2.38739, 2.38534, 2.38327, 2.38118, 2.37909, 2.37697, 2.37485, 2.37271, 2.37056, 2.36839, 2.36622, 2.36403, 2.36182, 2.35961, 2.35738, 2.35514, 2.35289, 2.35063, 2.34836, 2.34608, 2.34378, 2.34148, 2.33916, 2.33683, 2.33449, 2.33215, 2.32979, 2.32742, 2.32504, 2.32266, 2.32026, 2.31786, 2.31544, 2.31302, 2.31058, 2.30814, 2.30569, 2.30323, 2.30077, 2.29829, 2.29581, 2.29332, 2.29082, 2.28831, 2.28579, 2.28327, 2.28074, 2.2782, 2.27566, 2.27311, 2.27055, 2.26799, 2.26541, 2.26284, 2.26025, 2.25766, 2.25506, 2.25246, 2.24985, 2.24723, 2.24461, 2.24198, 2.23935, 2.23671, 2.23407, 2.23142, 2.22876, 2.2261, 2.22343, 2.22076, 2.21809, 2.21541, 2.21272, 2.21003, 2.20734, 2.20464, 2.20194, 2.19923, 2.19652, 2.1938, 2.19108, 2.18835, 2.18563, 2.18289, 2.18016, 2.17742, 2.17467, 2.17193, 2.16918, 2.16642, 2.16366, 2.1609, 2.15814, 2.15537, 2.1526, 2.14983, 2.14705, 2.14427, 2.14149, 2.13871, 2.13592, 2.13313, 2.13034, 2.12754, 2.12474, 2.12194, 2.11914, 2.11633, 2.11353, 2.11072, 2.10791, 2.10509, 2.10228, 2.09946, 2.09664, 2.09382, 2.091, 2.08818, 2.08535, 2.08252, 2.07969, 2.07686, 2.07403, 2.0712, 2.06836, 2.06553, 2.06269, 2.05985, 2.05701, 2.05417, 2.05133, 2.04848, 2.04564, 2.04279, 2.03995, 2.0371, 2.03425, 2.03141, 2.02856, 2.02571, 2.02286, 2.02001, 2.01716, 2.0143, 2.01145, 2.0086, 2.00575, 2.00289, 2.00004, 1.99719, 1.99433, 1.99148, 1.98862, 1.98577, 1.98291, 1.98006, 1.9772, 1.97435, 1.97149, 1.96864, 1.96578, 1.96293, 1.96008, 1.95722, 1.95437, 1.95152, 1.94866, 1.94581, 1.94296, 1.94011, 1.93725, 1.9344, 1.93155, 1.9287, 1.92586, 1.92301, 1.92016, 1.91731, 1.91446, 1.91162, 1.90877, 1.90593, 1.90309, 1.90024, 1.8974, 1.89456, 1.89172, 1.88888, 1.88604, 1.88321, 1.88037, 1.87754, 1.8747, 1.87187, 1.86904, 1.86621, 1.86338, 1.86055, 1.85772, 1.8549, 1.85207, 1.84925, 1.84643, 1.84361, 1.84079, 1.83797, 1.83515, 1.83234, 1.82953, 1.82671, 1.8239, 1.82109, 1.81829, 1.81548, 1.81268, 1.80987, 1.80707, 1.80427, 1.80147, 1.79868, 1.79588, 1.79309, 1.7903, 1.78751, 1.78472, 1.7812, 1.77971, 1.77822, 1.77673, 1.77526, 1.77378, 1.77231, 1.77084, 1.76938, 1.76793, 1.76648, 1.76503, 1.76359, 1.76215, 1.76071, 1.75929, 1.75786, 1.75644, 1.75502, 1.75361, 1.75221, 1.7508, 1.74941, 1.74801, 1.74662, 1.74524, 1.74386, 1.74248, 1.74111, 1.73974, 1.73837, 1.73701, 1.73566, 1.7343, 1.73296, 1.73161, 1.73027, 1.72894, 1.7276, 1.72627, 1.72495, 1.72363, 1.72231, 1.721, 1.71969, 1.71839, 1.71709, 1.71579, 1.7145, 1.71321, 1.71192, 1.71064, 1.70936, 1.70808, 1.70681, 1.70554, 1.70428, 1.70302, 1.70176, 1.70051, 1.69926, 1.69801, 1.69677, 1.69553, 1.6943, 1.69306, 1.69184, 1.69061, 1.68939, 1.68817, 1.68695, 1.68574, 1.68453, 1.68333, 1.68213, 1.68093, 1.67973, 1.67854, 1.67735, 1.67617, 1.67499, 1.67381, 1.67263, 1.67146]]),
            # fmt: on
            id="Compare EEmiLib with manual SPARK3D export on Copper (ECSS-2020)",
        ),
        pytest.param(
            {"E_c1": 20, "E_max": 315, "teey_low": 0.8, "teey_max": 2.34},
            "SPARK3D",
            np.linspace(0, 1000, 501),
            np.zeros(1),
            # fmt: off
            np.array([[0, 0, 0, 0, 0, 0.798212, 0.842623, 0.884734, 0.924818, 0.963093, 0.999738, 1.0349, 1.06872, 1.10129, 1.13271, 1.16306, 1.19242, 1.22084, 1.24839, 1.27511, 1.30105, 1.32626, 1.35076, 1.37459, 1.39779, 1.42038, 1.44239, 1.46384, 1.48475, 1.50516, 1.52507, 1.5445, 1.56348, 1.58201, 1.60012, 1.61782, 1.63511, 1.65202, 1.66855, 1.68472, 1.70054, 1.71601, 1.73115, 1.74597, 1.76047, 1.77466, 1.78855, 1.80216, 1.81547, 1.82852, 1.84129, 1.8538, 1.86604, 1.87804, 1.88979, 1.90131, 1.91258, 1.92363, 1.93445, 1.94505, 1.95543, 1.96561, 1.97557, 1.98534, 1.9949, 2.00427, 2.01345, 2.02244, 2.03124, 2.03987, 2.04832, 2.05659, 2.06469, 2.07263, 2.0804, 2.08801, 2.09546, 2.10275, 2.10989, 2.11688, 2.12372, 2.13041, 2.13696, 2.14337, 2.14964, 2.15577, 2.16177, 2.16763, 2.17337, 2.17898, 2.18446, 2.18981, 2.19505, 2.20016, 2.20515, 2.21003, 2.2148, 2.21944, 2.22398, 2.22841, 2.23273, 2.23694, 2.24105, 2.24505, 2.24895, 2.25275, 2.25645, 2.26005, 2.26355, 2.26696, 2.27028, 2.2735, 2.27663, 2.27967, 2.28263, 2.28549, 2.28827, 2.29096, 2.29357, 2.29609, 2.29854, 2.3009, 2.30318, 2.30538, 2.30751, 2.30956, 2.31153, 2.31343, 2.31526, 2.31701, 2.31869, 2.3203, 2.32184, 2.32331, 2.32471, 2.32605, 2.32731, 2.32852, 2.32966, 2.33073, 2.33174, 2.33269, 2.33358, 2.3344, 2.33517, 2.33588, 2.33652, 2.33711, 2.33765, 2.33812, 2.33855, 2.33891, 2.33922, 2.33948, 2.33969, 2.33984, 2.33994, 2.33999, 2.34, 2.33997, 2.33993, 2.33986, 2.33978, 2.33967, 2.33954, 2.33939, 2.33922, 2.33903, 2.33882, 2.33859, 2.33834, 2.33807, 2.33778, 2.33747, 2.33715, 2.33681, 2.33644, 2.33606, 2.33567, 2.33525, 2.33482, 2.33437, 2.3339, 2.33342, 2.33292, 2.33241, 2.33187, 2.33133, 2.33076, 2.33018, 2.32959, 2.32898, 2.32835, 2.32771, 2.32706, 2.32639, 2.3257, 2.32501, 2.32429, 2.32357, 2.32283, 2.32207, 2.3213, 2.32052, 2.31973, 2.31892, 2.3181, 2.31727, 2.31642, 2.31556, 2.31469, 2.31381, 2.31292, 2.31201, 2.31109, 2.31016, 2.30922, 2.30826, 2.3073, 2.30632, 2.30533, 2.30433, 2.30332, 2.3023, 2.30127, 2.30023, 2.29918, 2.29812, 2.29704, 2.29596, 2.29487, 2.29376, 2.29265, 2.29153, 2.2904, 2.28926, 2.2881, 2.28694, 2.28577, 2.2846, 2.28341, 2.28221, 2.28101, 2.27979, 2.27857, 2.27734, 2.2761, 2.27485, 2.27359, 2.27233, 2.27105, 2.26977, 2.26848, 2.26718, 2.26588, 2.26457, 2.26325, 2.26192, 2.26058, 2.25924, 2.25789, 2.25653, 2.25517, 2.2538, 2.25242, 2.25103, 2.24964, 2.24824, 2.24683, 2.24542, 2.244, 2.24257, 2.24114, 2.2397, 2.23825, 2.2368, 2.23535, 2.23388, 2.23241, 2.23093, 2.22945, 2.22796, 2.22647, 2.22497, 2.22346, 2.22195, 2.22044, 2.21891, 2.21739, 2.21585, 2.21431, 2.21277, 2.21122, 2.20967, 2.20811, 2.20654, 2.20497, 2.2034, 2.20182, 2.20023, 2.19864, 2.19705, 2.19545, 2.19385, 2.19224, 2.19062, 2.18901, 2.18738, 2.18576, 2.18413, 2.18249, 2.18085, 2.17921, 2.17756, 2.17591, 2.17425, 2.17259, 2.17093, 2.16926, 2.16759, 2.16591, 2.16423, 2.16255, 2.16086, 2.15917, 2.15747, 2.15578, 2.15407, 2.15237, 2.15066, 2.14895, 2.14723, 2.14551, 2.14379, 2.14206, 2.14033, 2.1386, 2.13686, 2.13512, 2.13338, 2.13164, 2.12989, 2.12814, 2.12638, 2.12462, 2.12286, 2.1211, 2.11933, 2.11757, 2.11579, 2.11402, 2.11224, 2.11046, 2.10868, 2.10689, 2.10511, 2.10332, 2.10152, 2.09973, 2.09793, 2.09613, 2.09433, 2.09252, 2.09072, 2.08891, 2.08709, 2.08528, 2.08346, 2.08164, 2.07982, 2.078, 2.07618, 2.07435, 2.07252, 2.07069, 2.06886, 2.06702, 2.06518, 2.06334, 2.0615, 2.05966, 2.05782, 2.05597, 2.05412, 2.05227, 2.05042, 2.04857, 2.04671, 2.04485, 2.043, 2.04114, 2.03927, 2.03741, 2.03555, 2.03368, 2.03181, 2.02994, 2.02807, 2.0262, 2.02433, 2.02245, 2.02058, 2.0187, 2.01682, 2.01494, 2.01306, 2.01118, 2.00929, 2.00741, 2.00552, 2.00363, 2.00175, 1.99986, 1.99796, 1.99607, 1.99418, 1.99229, 1.99039, 1.9885, 1.9866, 1.9847, 1.9828, 1.9809, 1.979, 1.9771, 1.9752, 1.97329, 1.97139, 1.96949, 1.96758, 1.96567, 1.96377, 1.96186, 1.95995, 1.95804, 1.95613, 1.95422, 1.95231, 1.9504, 1.94848, 1.94657, 1.94466, 1.94274, 1.94083, 1.93891, 1.937, 1.93508, 1.93316, 1.93124, 1.92933, 1.92741, 1.92549, 1.92357, 1.92165, 1.91973, 1.91781, 1.91589, 1.91397, 1.91205, 1.91013, 1.9082, 1.90628, 1.90436, 1.90244, 1.90051, 1.89859, 1.89667, 1.89474, 1.89282, 1.8909, 1.88897, 1.88705, 1.88512, 1.8832, 1.88127, 1.87935, 1.87742, 1.8755, 1.87358, 1.87165, 1.86973, 1.8678, 1.86588, 1.86395, 1.86203, 1.8601, 1.85818, 1.85625, 1.85433, 1.8524, 1.85048, 1.84855, 1.84663, 1.84471, 1.84278, 1.84086, 1.83893, 1.83701, 1.83509, 1.83316, 1.83124]]),
            # fmt: on
            id="Compare EEmiLib with manual SPARK3D export on Silver (ECSS-2020)",
        ),
    ],
)
def test_teey(
    vaughan_parameters: dict[str, float],
    vaughan_implementation: VaughanImplementation,
    energy: NDArray[np.float64],
    theta: NDArray[np.float64],
    expected: NDArray[np.float64],
) -> None:
    """Check the returned values of the TEEY function.

    Test that the TEEY function works as expected
      1. Give in the parameters that are already defined, compare them to
         hard-written TEEY file
      2. Give in the parameters that are already defined to vaughan_cst,
         compare them to hard-written exported TEEY files

    """
    model = Vaughan(
        implementation=vaughan_implementation,
        parameters_values=vaughan_parameters,
    )

    calculated = model.teey(energy, theta).to_numpy().transpose()[:-1]
    assert_array_almost_equal(calculated, expected, decimal=3)


@pytest.fixture
def fil_technical_ag() -> MockDataMatrix:
    """Instantiate technical Ag from :cite:`Fil2016a,Fil2020`."""
    emission_yield = EmissionYield(
        population="all",
        data=pd.DataFrame(
            {
                # fmt: off
        "Energy [eV]": [0, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 30, 36, 49, 70, 75, 90, 100, 150, 200, 250, 300, 350, 400, 450, 500, 600, 700, 800, 900, 1000, 1200, 1400, 1600, 1800, 2000, 3000, 4000, 5000, 6000, 7000, 8000, 10000, 12000, 14000, 16000, 18000, 20000, 22000],
        "0.0 [deg]": [0, 0.75, 0.78, 0.79, 0.82, 0.83, 0.85, 0.87, 0.9, 0.94, 0.97, 1, 1.03, 1.1, 1.17, 1.31, 1.55, 1.61, 1.725, 1.77, 1.975, 2.119, 2.173, 2.17, 2.16, 2.152, 2.137, 2.106, 2.041, 1.975, 1.907, 1.847, 1.777, 1.67, 1.582, 1.5, 1.42, 1.34, 1.07, 0.95, 0.87, 0.795, 0.75, 0.72, 0.68, 0.635, 0.62, 0.6, 0.58, 0.57, 0.56],
                # fmt: on
            }
        ),
    )
    data_matrix = MockDataMatrix(emission_yield)
    return data_matrix


@pytest.mark.xfail
def test_error_ec1(
    vaughan_model: Vaughan, fil_technical_ag: MockDataMatrix
) -> None:
    """Check that we retrieve N. Fil results :cite:`Fil2016a,Fil2020`.

    We use the same technical Ag as he did.

    """
    vaughan_model.find_optimal_parameters(fil_technical_ag)
    returned = vaughan_model._error_ec1(fil_technical_ag.teey)
    expected = 23.4
    assert returned == approx(expected)


@pytest.mark.xfail
def test_error_teey(
    vaughan_model: Vaughan, fil_technical_ag: MockDataMatrix
) -> None:
    """Check that we retrieve N. Fil results :cite:`Fil2016a,Fil2020`.

    We use the same technical Ag as he did.

    """
    vaughan_model.find_optimal_parameters(fil_technical_ag)
    returned = vaughan_model._error_teey(fil_technical_ag.teey)
    expected = 3.1
    assert returned == approx(expected)
