"""
Test enhanced string and datetime accessors with pandas compatibility.

All tests compare polarpandas output against actual pandas output
to ensure 100% compatibility.
"""

from datetime import datetime

import pandas as pd
import pytest

import polarpandas as ppd


class TestStringAccessorEnhanced:
    """Test enhanced string accessor methods with pandas compatibility."""

    def setup_method(self):
        """Create test data in both pandas and polarpandas."""
        self.data = ["hello world", "test string", "another example", "final test"]
        # Don't create Series here to avoid state pollution
        # Each test method will create fresh Series

    def test_split_basic(self):
        """Test split method."""
        pd_result = pd.Series(self.data).str.split()
        ppd_result = ppd.Series(self.data).str.split()
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_split_with_separator(self):
        """Test split with specific separator."""
        pd_result = pd.Series(self.data).str.split(" ")
        ppd_result = ppd.Series(self.data).str.split(" ")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_split_expand(self):
        """Test split with expand=True."""
        pd_result = pd.Series(self.data).str.split(" ", expand=True)
        ppd_result = ppd.Series(self.data).str.split(" ", expand=True)
        pd.testing.assert_frame_equal(ppd_result.to_pandas(), pd_result)

    def test_extract_basic(self):
        """Test extract method."""
        pd_result = pd.Series(self.data).str.extract(r"(\d+)")
        ppd_result = ppd.Series(self.data).str.extract(r"(\d+)")
        pd.testing.assert_frame_equal(ppd_result.to_pandas(), pd_result)

    def test_slice_basic(self):
        """Test slice method."""
        pd_result = pd.Series(self.data).str.slice(0, 5)
        ppd_result = ppd.Series(self.data).str.slice(0, 5)
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_slice_with_step(self):
        """Test slice with step."""
        pd_result = pd.Series(self.data).str.slice(0, 10, 2)
        ppd_result = ppd.Series(self.data).str.slice(0, 10, 2)
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_contains_basic(self):
        """Test contains method."""
        pd_result = pd.Series(self.data).str.contains("test")
        ppd_result = ppd.Series(self.data).str.contains("test")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_startswith_basic(self):
        """Test startswith method."""
        pd_result = pd.Series(self.data).str.startswith("h")
        ppd_result = ppd.Series(self.data).str.startswith("h")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_endswith_basic(self):
        """Test endswith method."""
        pd_result = pd.Series(self.data).str.endswith("d")
        ppd_result = ppd.Series(self.data).str.endswith("d")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_string_methods_with_nulls(self):
        """Test string methods with null values."""
        # Test contains with nulls
        pd_result = pd.Series(self.data).str.contains("o")
        ppd_result = ppd.Series(self.data).str.contains("o")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_string_methods_empty_series(self):
        """Test string methods with empty Series."""
        pd_empty = pd.Series([], dtype=str)
        ppd_empty = ppd.Series([], dtype=str)

        # Test contains with empty series
        pd_result = pd_empty.str.contains("test")
        ppd_result = ppd_empty.str.contains("test")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )


class TestDatetimeAccessorEnhanced:
    """Test enhanced datetime accessor methods with pandas compatibility."""

    def setup_method(self):
        """Create test data in both pandas and polarpandas."""
        self.data = [
            datetime(2023, 1, 15, 10, 30, 45),
            datetime(2023, 6, 20, 14, 15, 30),
            datetime(2023, 12, 31, 23, 59, 59),
        ]
        # Don't create Series here to avoid state pollution
        # Each test method will create fresh Series

    @pytest.mark.skip(
        reason="Polars returns datetime64[ms] with time 00:00:00 instead of date objects - fundamental limitation"
    )
    def test_date_property(self):
        """Test date property."""
        pd_result = pd.Series(self.data).dt.date
        ppd_result = ppd.Series(self.data).dt.date
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_time_property(self):
        """Test time property."""
        pd_result = pd.Series(self.data).dt.time
        ppd_result = ppd.Series(self.data).dt.time
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_dayofweek_property(self):
        """Test dayofweek property."""
        pd_result = pd.Series(self.data).dt.dayofweek
        ppd_result = ppd.Series(self.data).dt.dayofweek
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_dayofyear_property(self):
        """Test dayofyear property."""
        pd_result = pd.Series(self.data).dt.dayofyear
        ppd_result = ppd.Series(self.data).dt.dayofyear
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_quarter_property(self):
        """Test quarter property."""
        pd_result = pd.Series(self.data).dt.quarter
        ppd_result = ppd.Series(self.data).dt.quarter
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_is_month_start_property(self):
        """Test is_month_start property."""
        pd_result = pd.Series(self.data).dt.is_month_start
        ppd_result = ppd.Series(self.data).dt.is_month_start
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_is_month_end_property(self):
        """Test is_month_end property."""
        pd_result = pd.Series(self.data).dt.is_month_end
        ppd_result = ppd.Series(self.data).dt.is_month_end
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_is_quarter_start_property(self):
        """Test is_quarter_start property."""
        pd_result = pd.Series(self.data).dt.is_quarter_start
        ppd_result = ppd.Series(self.data).dt.is_quarter_start
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_is_quarter_end_property(self):
        """Test is_quarter_end property."""
        pd_result = pd.Series(self.data).dt.is_quarter_end
        ppd_result = ppd.Series(self.data).dt.is_quarter_end
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_is_year_start_property(self):
        """Test is_year_start property."""
        pd_result = pd.Series(self.data).dt.is_year_start
        ppd_result = ppd.Series(self.data).dt.is_year_start
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_is_year_end_property(self):
        """Test is_year_end property."""
        pd_result = pd.Series(self.data).dt.is_year_end
        ppd_result = ppd.Series(self.data).dt.is_year_end
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_floor_method(self):
        """Test floor method."""
        pd_result = pd.Series(self.data).dt.floor("D")
        ppd_result = ppd.Series(self.data).dt.floor("D")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_ceil_method(self):
        """Test ceil method."""
        pd_result = pd.Series(self.data).dt.ceil("D")
        ppd_result = ppd.Series(self.data).dt.ceil("D")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_round_method(self):
        """Test round method."""
        pd_result = pd.Series(self.data).dt.round("h")
        ppd_result = ppd.Series(self.data).dt.round("h")
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_datetime_methods_with_nulls(self):
        """Test datetime methods with null values."""
        # Test year with nulls
        pd_result = pd.Series(self.data).dt.year
        ppd_result = ppd.Series(self.data).dt.year
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    @pytest.mark.skip(
        reason="Polars datetime accessor with empty series differs from pandas. See KNOWN_LIMITATIONS.md - permanent limitation"
    )
    def test_datetime_methods_empty_series(self):
        """Test datetime methods with empty Series."""
        pd_empty = pd.Series([], dtype="datetime64[ns]")
        ppd_empty = ppd.Series([], dtype="datetime64[ns]")

        # Test year with empty series
        pd_result = pd_empty.dt.year
        ppd_result = ppd_empty.dt.year
        pd.testing.assert_series_equal(
            ppd_result.to_pandas(), pd_result, check_dtype=False, check_names=False
        )

    def test_datetime_methods_return_types(self):
        """Test that datetime methods return correct types."""
        # Test properties
        result = ppd.Series(self.data).dt.year
        assert isinstance(result, ppd.Series)

        result = ppd.Series(self.data).dt.month
        assert isinstance(result, ppd.Series)

        result = ppd.Series(self.data).dt.day
        assert isinstance(result, ppd.Series)

        # Test methods
        result = ppd.Series(self.data).dt.floor("D")
        assert isinstance(result, ppd.Series)

        result = ppd.Series(self.data).dt.ceil("D")
        assert isinstance(result, ppd.Series)

        result = ppd.Series(self.data).dt.round("H")
        assert isinstance(result, ppd.Series)

    def test_datetime_methods_preserve_original(self):
        """Test that datetime methods don't modify original Series."""
        original_pd = pd.Series(self.data).copy()
        original_ppd = ppd.Series(self.data).copy()

        # Perform operations
        _ = pd.Series(self.data).dt.year
        _ = ppd.Series(self.data).dt.year
        _ = pd.Series(self.data).dt.month
        _ = ppd.Series(self.data).dt.month

        # Original should be unchanged
        pd.testing.assert_series_equal(original_pd, pd.Series(self.data))
        pd.testing.assert_series_equal(
            original_ppd.to_pandas(), ppd.Series(self.data).to_pandas()
        )
