#!/usr/bin/env python3

"""Simple HTTP Server With Upload.

This module builds on BaseHTTPServer by implementing the standard GET
and HEAD requests in a fairly straightforward manner.

see: https://gist.github.com/UniIsland/3346170

Original Author: bones7456

2020-01-08
Ver 0.1.1 Improved by Frank Xu
"""
__version__ = "0.1.1"
__all__ = ["SimpleHTTPRequestHandler"]
__author__ = "Frank Xu"

# Security Configuration
MAX_FILE_SIZE = 10 * 1024 * 1024  # 10MB limit
REQUIRE_AUTH = True  # Set to True to require authentication
ALLOWED_EXTENSIONS = {'.txt', '.jpg', '.jpeg', '.png', '.gif', '.pdf', '.doc', '.docx'}
BIND_LOCALHOST_ONLY = True  # Set to True to bind only to localhost

import html
import errno
import mimetypes
import os
import re
import posixpath
import shutil
import socket
import sys
import urllib.request
import urllib.parse
import urllib.error
from io import BytesIO
from http.server import HTTPServer, SimpleHTTPRequestHandler, BaseHTTPRequestHandler


class SimpleHTTPRequestHandler(BaseHTTPRequestHandler):
    """Simple HTTP request handler with GET/HEAD/POST commands.

    This serves files from the current directory and any of its
    subdirectories.  The MIME type for files is determined by
    calling the .guess_type() method. And can receive file uploaded
    by client.

    The GET/HEAD/POST requests are identical except that the HEAD
    request omits the actual contents of the file.

    """
    server_version = "SimpleHTTPWithUpload/" + __version__

    def do_GET(self):
        """Serve a GET request."""
        f = self.send_head()
        if f:
            self.copy_file(f, self.wfile)
            f.close()

    def do_HEAD(self):
        """Serve a HEAD request."""
        f = self.send_head()
        if f:
            f.close()

    def do_POST(self):
        """Serve a POST request."""
        # Security: Check authentication if required
        if REQUIRE_AUTH and not self._check_auth():
            self.send_response(401)
            self.send_header('WWW-Authenticate', 'Basic realm="File Upload"')
            self.end_headers()
            self.wfile.write(b'Authentication required')
            return
        
        r, info = self.deal_post_data()
        print((r, info, "by: ", self.client_address))
        f = BytesIO()
        f.write(b'<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">')
        f.write(b"<html>\n<title>Upload Result Page</title>\n")
        f.write(b"<body>\n<h2>Upload Result Page</h2>\n")
        f.write(b"<hr>\n")
        if r:
            f.write(b"<strong>Success:</strong>")
        else:
            f.write(b"<strong>Failed:</strong>")
        f.write(info.encode())
        f.write(("<br><a href=\"%s\">back</a>" % self.headers['referer']).encode())
        f.write(b"</body>\n</html>\n")
        length = f.tell()
        f.seek(0)
        self.send_response(200)
        self.send_header("Content-type", "text/html")
        self.send_header("Content-Length", str(length))
        self.end_headers()
        if f:
            self.copy_file(f, self.wfile)
            f.close()
    
    def _is_safe_filename(self, filename):
        """Validate filename to prevent security vulnerabilities."""
        import re
        
        # Check for empty or None filename
        if not filename or not filename.strip():
            return False
        
        # Check for path traversal attempts
        if '..' in filename or '/' in filename or '\\' in filename:
            return False
        
        # Check for absolute paths
        if filename.startswith('/') or (len(filename) > 1 and filename[1] == ':'):
            return False
        
        # Check for dangerous characters
        dangerous_chars = ['<', '>', ':', '"', '|', '?', '*', '\x00']
        if any(char in filename for char in dangerous_chars):
            return False
        
        # Check for Windows reserved names
        reserved_names = [
            'CON', 'PRN', 'AUX', 'NUL',
            'COM1', 'COM2', 'COM3', 'COM4', 'COM5', 'COM6', 'COM7', 'COM8', 'COM9',
            'LPT1', 'LPT2', 'LPT3', 'LPT4', 'LPT5', 'LPT6', 'LPT7', 'LPT8', 'LPT9'
        ]
        base_name = filename.split('.')[0].upper()
        if base_name in reserved_names:
            return False
        
        # Check filename length
        if len(filename) > 255:
            return False
        
        # Check for control characters
        if any(ord(char) < 32 and char not in ['\t'] for char in filename):
            return False
        
        # Check file extension
        import os.path
        _, ext = os.path.splitext(filename.lower())
        if ext not in ALLOWED_EXTENSIONS:
            return False
        
        return True
    
    def _check_auth(self):
        """Check HTTP Basic Authentication."""
        import base64
        
        auth_header = self.headers.get('Authorization')
        if not auth_header:
            return False
        
        try:
            auth_type, auth_string = auth_header.split(' ', 1)
            if auth_type.lower() != 'basic':
                return False
            
            decoded = base64.b64decode(auth_string).decode('utf-8')
            username, password = decoded.split(':', 1)
            
            # Simple authentication - in production use proper authentication
            # For now, accept any non-empty username/password
            return len(username) > 0 and len(password) > 0
        except Exception:
            return False

    def deal_post_data(self):
        content_type = self.headers['content-type']
        if not content_type:
            return False, "Content-Type header doesn't contain boundary"
        boundary = content_type.split("=")[1].encode()
        remain_bytes = int(self.headers['content-length'])
        
        # Security: Check file size limit
        if remain_bytes > MAX_FILE_SIZE:
            return False, f"File too large. Maximum size is {MAX_FILE_SIZE} bytes"
        line = self.rfile.readline()
        remain_bytes -= len(line)
        if boundary not in line:
            return False, "Content NOT begin with boundary"
        line = self.rfile.readline()
        remain_bytes -= len(line)
        fn = re.findall(r'Content-Disposition.*name="file"; filename="(.*)"', line.decode())
        if not fn:
            return False, "Can't find out file name..."
        
        # Security: Validate filename to prevent path traversal attacks
        filename = fn[0]
        if not self._is_safe_filename(filename):
            return False, "Invalid or unsafe filename"
        
        path = self.translate_path(self.path)
        fn = os.path.join(path, filename)
        line = self.rfile.readline()
        remain_bytes -= len(line)
        line = self.rfile.readline()
        remain_bytes -= len(line)
        try:
            out = open(fn, 'wb')
        except IOError:
            return False, "Can't create file to write, do you have permission to write?"

        pre_line = self.rfile.readline()
        remain_bytes -= len(pre_line)
        while remain_bytes > 0:
            line = self.rfile.readline()
            remain_bytes -= len(line)
            if boundary in line:
                pre_line = pre_line[0:-1]
                if pre_line.endswith(b'\r'):
                    pre_line = pre_line[0:-1]
                out.write(pre_line)
                out.close()
                return True, "File '%s' upload success!" % fn
            else:
                out.write(pre_line)
                pre_line = line
        return False, "Unexpect Ends of data."

    def send_head(self):
        """Common code for GET and HEAD commands.

        This sends the response code and MIME headers.

        Return value is either a file object (which has to be copied
        to the outputfile by the caller unless the command was HEAD,
        and must be closed by the caller under all circumstances), or
        None, in which case the caller has nothing further to do.

        """
        path = self.translate_path(self.path)
        # f = None
        if os.path.isdir(path):
            if not self.path.endswith('/'):
                # redirect browser - doing basically what apache does
                self.send_response(301)
                self.send_header("Location", self.path + "/")
                self.end_headers()
                return None
            for index in "index.html", "index.htm":
                index = os.path.join(path, index)
                if os.path.exists(index):
                    path = index
                    break
            else:
                return self.list_directory(path)
        ctype = self.guess_type(path)
        try:
            # Always read in binary mode. Opening files in text mode may cause
            # newline translations, making the actual size of the content
            # transmitted *less* than the content-length!
            f = open(path, 'rb')
        except IOError:
            self.send_error(404, "File not found")
            return None
        self.send_response(200)
        self.send_header("Content-type", ctype)
        fs = os.fstat(f.fileno())
        self.send_header("Content-Length", str(fs[6]))
        self.send_header("Last-Modified", self.date_time_string(fs.st_mtime))
        self.end_headers()
        return f

    def list_directory(self, path):
        """Helper to produce a directory listing (absent index.html).

        Return value is either a file object, or None (indicating an
        error).  In either case, the headers are sent, making the
        interface the same as for send_head().

        """
        try:
            lst = os.listdir(path)
        except os.error:
            self.send_error(404, "No permission to list directory")
            return None
        lst.sort(key=lambda a: a.lower())
        f = BytesIO()
        display_path = html.escape(urllib.parse.unquote(self.path))
        f.write(b'<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">')
        f.write(("<html>\n<title>Directory listing for %s</title>\n" % display_path).encode())
        f.write(("<body>\n<h2>Directory listing for %s</h2>\n" % display_path).encode())
        f.write(b"<hr>\n")
        f.write(b"<form ENCTYPE=\"multipart/form-data\" method=\"post\">")
        f.write(b"<input name=\"file\" type=\"file\"/>")
        f.write(b"<input type=\"submit\" value=\"upload\"/></form>\n")
        f.write(b"<hr>\n<ul>\n")
        for name in lst:
            fullname = os.path.join(path, name)
            display_name = link_name = name
            # Append / for directories or @ for symbolic links
            if os.path.isdir(fullname):
                display_name = name + "/"
                link_name = name + "/"
            if os.path.islink(fullname):
                display_name = name + "@"
                # Note: a link to a directory displays with @ and links with /
            f.write(('<li><a href="%s">%s</a>\n'
                    % (urllib.parse.quote(link_name), html.escape(display_name))).encode())
        f.write(b"</ul>\n<hr>\n</body>\n</html>\n")
        length = f.tell()
        f.seek(0)
        self.send_response(200)
        self.send_header("Content-type", "text/html")
        self.send_header("Content-Length", str(length))
        self.end_headers()
        return f

    def translate_path(self, path):
        """Translate a /-separated PATH to the local filename syntax.

        Components that mean special things to the local file system
        (e.g. drive or directory names) are ignored.  (XXX They should
        probably be diagnosed.)

        """
        # abandon query parameters
        path = path.split('?', 1)[0]
        path = path.split('#', 1)[0]
        path = posixpath.normpath(urllib.parse.unquote(path))
        words = path.split('/')
        words = [_f for _f in words if _f]
        path = os.getcwd()
        for word in words:
            drive, word = os.path.splitdrive(word)
            head, word = os.path.split(word)
            if word in (os.curdir, os.pardir):
                continue
            path = os.path.join(path, word)
        return path

    def copy_file(self, source, output_file):
        """Copy all data between two file objects.

        The SOURCE argument is a file object open for reading
        (or anything with a read() method) and the DESTINATION
        argument is a file object open for writing (or
        anything with a write() method).

        The only reason for overriding this would be to change
        the block size or perhaps to replace newlines by CRLF
        -- note however that this the default server uses this
        to copy binary data as well.

        """
        shutil.copyfileobj(source, output_file)

    def guess_type(self, path):
        """Guess the type of a file.

        Argument is a PATH (a filename).

        Return value is a string of the form type/subtype,
        usable for a MIME Content-type header.

        The default implementation looks the file's extension
        up in the table self.extensions_map, using application/octet-stream
        as a default; however it would be permissible (if
        slow) to look inside the data to make a better guess.

        """

        base, ext = posixpath.splitext(path)
        if ext in self.extensions_map:
            return self.extensions_map[ext]
        ext = ext.lower()
        if ext in self.extensions_map:
            return self.extensions_map[ext]
        else:
            return self.extensions_map['']

    if not mimetypes.inited:
        mimetypes.init()  # try to read system mime.types
    extensions_map = mimetypes.types_map.copy()
    extensions_map.update({
        '': 'application/octet-stream',  # Default
        '.py': 'text/plain',
        '.c': 'text/plain',
        '.h': 'text/plain',
        })


def get_lan_ip():
    """
    https://www.w3resource.com/python-exercises/python-basic-exercise-55.php
    """
    # Attempt to get IP from host name
    hostname = socket.gethostname()
    ips_from_hostname = socket.gethostbyname_ex(hostname)[2]
    non_local_ips_from_hostname = [ip for ip in ips_from_hostname if not ip.startswith("127.")]
    # If there are non-local IPs from the hostname, return the first one
    if non_local_ips_from_hostname:
        return non_local_ips_from_hostname[0]
    # If there are no non-local IPs from the hostname, try another method
    with socket.socket(socket.AF_INET, socket.SOCK_DGRAM) as s:
        s.connect(('8.8.8.8', 53))
        ip_from_socket = s.getsockname()[0]
    return ip_from_socket


def main():
    # Security: Bind to localhost only if security is enabled
    if BIND_LOCALHOST_ONLY:
        ip = '127.0.0.1'
        print("Security mode: Binding to localhost only")
    else:
        ip = get_lan_ip()
        print("Warning: Binding to all interfaces - consider security implications")
    
    port = 8000
    while port < 8050:
        try:
            httpd = HTTPServer((ip, port), SimpleHTTPRequestHandler)
            print("Running http server at {0}:{1}".format(ip, port))
            if REQUIRE_AUTH:
                print("Authentication required for uploads")
            print(f"Maximum file size: {MAX_FILE_SIZE} bytes")
            print(f"Allowed extensions: {ALLOWED_EXTENSIONS}")
            httpd.serve_forever()
        except socket.error as e:
            if e.errno == errno.EADDRINUSE:
                inuse_port = port
                port += 1
                msg = f"Port '{inuse_port}' is already in use. Try new port '{port}'"
                print(msg)
                continue
            else:
                print(e)
                break
        except KeyboardInterrupt:
            print('Interrupted By User')
            try:
                sys.exit(0)
            except SystemExit:
                sys.exit(0)


if __name__ == '__main__':
    main()
