"""Detect available ONNX Runtime execution providers and provide helpers."""
from typing import Dict, List
import platform

try:
    import onnxruntime as ort
except Exception:
    ort = None


def detect_providers() -> Dict:
    """Return available execution providers and basic system info.

    Example:
        {"providers": [...], "platform": "Windows"}
    """
    providers = []
    if ort is not None:
        try:
            providers = ort.get_available_providers()
        except Exception:
            providers = []

    return {"providers": providers, "platform": platform.system()}


def prefers_dml_for_amd(providers: List[str]) -> bool:
    """Heuristic: on Windows, prefer DirectML (DmlExecutionProvider) for integrated AMD GPUs."""
    return "DmlExecutionProvider" in providers


def has_gpu_provider(providers: List[str]) -> bool:
    return any(p for p in providers if p.lower() not in ("cpuexecutionprovider",))
