"""Simple scheduler that picks a backend based on user preference and available providers."""
from typing import List

from .device_manager import detect_providers, prefers_dml_for_amd
from .backends import CPUBackend, GPUBackend, NPUBackend


class BackendScheduler:
    def __init__(self, preference: List[str]):
        self.preference = preference
        info = detect_providers()
        self.providers = info.get("providers", [])
        self.platform = info.get("platform", "")

    def load_backend(self, model_path: str):
        # preference e.g. ['npu','dml','cpu']
        for pref in self.preference:
            p = pref.lower()
            if p == "npu":
                # Placeholder - skip since we don't have an implementation
                continue
            if p in ("dml", "gpu"):
                # prefer DirectML on Windows for AMD iGPU when available
                if "DmlExecutionProvider" in self.providers:
                    return GPUBackend(model_path, provider="DmlExecutionProvider")
                # try CUDA/ROCm
                if "CUDAExecutionProvider" in self.providers:
                    return GPUBackend(model_path, provider="CUDAExecutionProvider")
                if "ROCMExecutionProvider" in self.providers:
                    return GPUBackend(model_path, provider="ROCMExecutionProvider")
            if p == "cpu":
                return CPUBackend(model_path)
        # fallback to CPU
        return CPUBackend(model_path)
