## Version 4.26.0: Transits Time Range

### Overview
- Introduces the new `TransitsTimeRangeFactory` class for calculating transit events across a specified time range.
- Compares ephemeris data points with a subject’s natal chart to identify significant astrological aspects.
- Supports user-defined active points, aspects.

### Key Features
- Produces a `TransitsTimeRangeModel` that consolidates transit dates, the natal chart, and all computed transit events.
- Each ephemeris entry is mapped to a `TransitMomentModel`, providing detailed aspect analysis through `SynastryAspects`.

### Example Usage
```python
from datetime import datetime, timedelta
from kerykeion import (
    TransitsTimeRangeFactory,
    EphemerisDataFactory,
    AstrologicalSubject,
)

# Create a natal chart for the subject
person = AstrologicalSubject(
    "Johnny Depp", 1963, 6, 9, 20, 15, "Owensboro", "US"
)

# Define the time period for transit calculation
start_date = datetime.now()
end_date = datetime.now() + timedelta(days=30)

# Create ephemeris data for the specified time period
ephemeris_factory = EphemerisDataFactory(
    start_datetime=start_date,lg
    end_datetime=end_date,
    step_type="days",
    step=1,
    lat=person.lat,
    lng=person.lng,
    tz_str=person.tz_str,
)

ephemeris_data_points = ephemeris_factory.get_ephemeris_data_as_astrological_subjects()

# Calculate transits for the subject
transit_factory = TransitsTimeRangeFactory(
    natal_chart=person,
    ephemeris_data_points=ephemeris_data_points,
)

transit_results = transit_factory.get_transit_moments()

# Print example data
print(transit_results.model_dump()["dates"][2])
print(transit_results.model_dump()["transits"][2]['date'])
print(transit_results.model_dump()["transits"][2]['aspects'][0])
```

### Benefits
- Streamlines bulk analysis of transit events for comprehensive astrological research.
- Integrates seamlessly with existing Kerykeion modules for more robust synastry or predictive insights.

### Additional Enhancement
- The main class in `ephemeris_data.py` now includes a new `get_ephemeris_data_as_astrological_subjects` method, which returns ephemeris data as fully structured astrological subject models.
- The `natal_aspects` and `synastry_aspects` modules now add a `p*_owner` field for each planet, indicating which astrological subject owns that planet in the aspect data.

### Notes
- This release strengthens time-based astrological features, providing users a structured way to retrieve and examine future or past transit data in detail.
