import warnings
from typing import Optional, Type, TypeVar

T = TypeVar("T", bound="Singleton")


class Singleton:
    """
    Plain vanilla singleton class.
    """
    _instance: Optional["Singleton"] = None

    def __new__(cls: Type[T], *args, **kwargs) -> T:
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance

    def __init__(self, *args, **kwargs):
        if getattr(self, "_initialized", False):
            return

        # Set attributes from kwargs
        for k, v in kwargs.items():
            if not hasattr(self, k):
                setattr(self, k, v)
            else:
                warnings.warn(
                    f"Cannot re-set singleton attribute '{k}'. Skipping...")

        self._initialized = True  # Mark as initialized

    @property
    def is_initialized(self) -> bool:
        """Check if the instance is initialized."""
        return getattr(self, "_initialized", False)

    def let_die(self):
        """Reset the singleton instance and initialization state."""
        self._initialized = False
        type(self)._instance = None
